! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

! The following is the copyright for sodex

! Copyright (c) 2004, Ernst Hairer

! Redistribution and use in source and binary forms, with or without 
! modification, are permitted provided that the following conditions are 
! met:

! - Redistributions of source code must retain the above copyright 
! notice, this list of conditions and the following disclaimer.

! - Redistributions in binary form must reproduce the above copyright 
! notice, this list of conditions and the following disclaimer in the 
! documentation and/or other materials provided with the distribution.

! THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS 
! IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
! TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
! PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR 
! CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
! EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
! PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
! PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
! LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
! NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
! SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


      module mod_sodex
      use mod_dc_decsol
      use utils_lib
      use const_def, only: dp
      
      
      contains

      subroutine do_sodex(
     >      n,fcn,ifcn,x,y,xend,
     >      h,max_step_size,max_steps,
     >      rtol,atol,itol,
     >      jac,ijac,sjac,nzmax,isparse,mljac,mujac,dfx,idfx,
     >      mas,imas,mlmas,mumas,
     >      solout,iout,
     >      decsol,decsols,
     >      lrd,rpar_decsol,lid,ipar_decsol, 
     >      work,lwork,iwork,liwork,
     >      lrpar,rpar,lipar,ipar,
     >      lout,idid)
! ----------------------------------------------------------
!     numerical solution of a stiff (or differential algebraic)
!     system of first order ordinary differential equations  my'=f(x,y).
!     this is an extrapolation-algorithm, based on the
!     linearly implicit mid-point rule, due to bader-deuflhard
!     (with step size control and order selection).
!     c.f. section iv.9 
!
!     authors: e. hairer and g. wanner
!              universite de geneve, dept. de mathematiques
!              ch-1211 geneve 24, switzerland 
!              e-mail:  ernst.hairer@math.unige.ch
!                       gerhard.wanner@math.unige.ch
!     
!     this code is part of the book:
!         e. hairer and g. wanner, solving ordinary differential
!         equations ii. stiff and differential-algebraic problems.
!         springer series in computational mathematics,
!         springer-verlag (1990)               
!      
!     version of november 17, 1992
!         small corrections on june 11, 1999
!
         implicit real*8 (a-h,o-z)
         integer, intent(in) :: n ! the dimension of the system
         interface ! subroutine for computing the value of f(x,y)
            include "num_fcn.dek"
         end interface
         integer, intent(in) :: ifcn ! gives information on fcn:
         real(dp), intent(inout) :: x 
         real(dp), intent(inout) :: y(n) 
         real(dp), intent(in) :: xend ! desired final x value (positive or negative)
         real(dp), intent(inout) :: h 
         real(dp), intent(in) :: max_step_size
         integer, intent(in) :: max_steps
         real(dp), intent(in) :: rtol(*) ! relative error tolerance(s)
         real(dp), intent(in) :: atol(*) ! absolute error tolerance(s)
         integer, intent(in) :: itol ! switch for rtol and atol
         interface ! subroutine for computing the jacobian
            include "num_jac.dek"
            include "num_sjac.dek" ! for sparse matrix
         end interface
         integer, intent(in) :: ijac, nzmax, isparse, mljac, mujac
         interface ! subroutine for computing the partials of f(x,y) wrt x
            include "num_dfx.dek"
         end interface
         integer, intent(in) :: idfx ! switch for the computation of df/dx:
         interface ! subroutine for computing the mass matrix
            include "num_mas.dek"
         end interface
         integer, intent(in) :: imas ! gives information on the mass-matrix:
         integer, intent(in) :: mlmas
         integer, intent(in) :: mumas
         interface ! subroutine called after each successful step
            include "num_solout.dek"
         end interface
         integer, intent(in)  :: iout ! switch for calling the subroutine solout:
         interface
            include "mtx_decsol.dek"
            include "mtx_decsols.dek"
         end interface
         integer, intent(inout), target :: ipar_decsol(lid)
         real(dp), intent(inout), target :: rpar_decsol(lrd)
         real(dp), pointer :: work(:) ! (lwork)
         integer, intent(in) :: lwork
         integer, pointer :: iwork(:) ! (liwork)
         integer, intent(in) :: liwork
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(lrpar)
         integer, intent(inout) :: ipar(lipar)
         integer, intent(in)  :: lout
         integer, intent(out)  :: idid

      logical autnms,implct,arret,jband
! *** *** *** *** *** *** ***
!        setting the parameters 
! *** *** *** *** *** *** ***
       nfcn=0
       njac=0
       nstep=0
       naccpt=0
       nrejct=0
       ndec=0
       nsol=0
      arret=.false.
      nhess=0
! -------- nmax , the maximal number of steps -----
      if(max_steps.eq.0)then
         nmax=100000
      else
         nmax=max_steps
         if(nmax.le.0)then
            if (lout > 0) write(lout,*)' wrong input iwork(2)=',max_steps
            arret=.true.
         end if
      end if
! -------- km     maximum number of columns in the extrapolation 
      if(iwork(3).eq.0)then
         km=6
      else
         km=iwork(3)
         if(km.le.2)then
            if (lout > 0) write(lout,*)' curious input iwork(3)=',iwork(3)
            arret=.true.
         end if
      end if
! -------- nsequ     sequence numbers 
      nsequ=iwork(4)
      if(iwork(4).eq.0) nsequ=1
      if(nsequ.le.0.or.nsequ.ge.2)then
         if (lout > 0) write(lout,*)' curious input iwork(4)=',iwork(4)
         arret=.true.
      end if
! -------- uround   smallest number satisfying 1.d0+uround>1.d0  
      if(work(1).eq.0.d0)then
         uround=1.d-16
      else
         uround=work(1)
         if(uround.le.1.d-19.or.uround.ge.1.d0)then
            if (lout > 0) write(lout,*)' coefficients have 20 digits, uround=',work(1)
            arret=.true.
         end if
      end if
! -------- maximal step size
      if(work(2).eq.0.d0)then
         hmax=xend-x
      else
         hmax=work(2)
      end if
! ------ thet     decides whether the jacobian should be recomputed;
      if(work(3).eq.0.d0)then
         thet=rtol(1)
      else
         thet=work(3)
      end if
! -------  fac1,fac2     parameters for step size selection
      if(work(4).eq.0.d0)then
         fac1=0.1d0
      else
         fac1=work(4)
      end if
      if(work(5).eq.0.d0)then
         fac2=4.0d0
      else
         fac2=work(5)
      end if
! -------  fac3, fac4   parameters for the order selection
      if(work(6).eq.0.d0)then
         fac3=0.9d0
      else
         fac3=work(6)
      end if
      if(work(7).eq.0.d0)then
         fac4=0.9d0
      else
         fac4=work(7)
      end if
! ------- safe1, safe2 safety factors for step size prediction
      if(work(8).eq.0.d0)then
         safe1=0.8d0
      else
         safe1=work(8)
      end if
      if(work(9).eq.0.d0)then
         safe2=0.93d0
      else
         safe2=work(9)
      end if
! ------- wkfcn,wkjac,wkdec,wksol  estimated work for  fcn,jac,dec,sol
      if(work(10).eq.0.d0)then
         wkfcn=1.d0
      else
         wkfcn=work(10)
      end if
      if(work(11).eq.0.d0)then
         wkjac=5.d0
      else
         wkjac=work(11)
      end if
      if(work(12).eq.0.d0)then
         wkdec=1.d0
      else
         wkdec=work(12)
      end if
      if(work(13).eq.0.d0)then
         wksol=1.d0
      else
         wksol=work(13)
      end if
      wkrow=wkfcn+wksol
! --------- check if tolerances are o.k.
      if (itol.eq.0) then
          if (atol(1).le.0.d0.or.rtol(1).le.10.d0*uround) then
              write (6,*) ' tolerances are too small'
              arret=.true.
          end if
      else
          do 15 i=1,n
          if (atol(i).le.0.d0.or.rtol(i).le.10.d0*uround) then
              write (6,*) ' tolerances(',i,') are too small'
              arret=.true.
          end if
  15      continue
      end if
! *** *** *** *** *** *** *** *** *** *** *** *** ***
!         computation of array entries
! *** *** *** *** *** *** *** *** *** *** *** *** ***
! ---- autonomous, implicit, banded or not ?
      arret=.false.
      autnms=ifcn.eq.0
      implct=imas.ne.0
      jband=mljac.ne.n .and. nzmax == 0
      if ((nzmax > 0) .and. (jband .or. implct .or. ijac==0)) then
         if (lout > 0) write(lout,'(a)') 'nzmax > 0 requires ijac=1, imas=0 -- unlike the newer codes, ' //
     >      'sodex does not support non-identity mass matrix when using sparse linear algebra'         
         arret=.true.
      end if
! -------- computation of the row-dimensions of the 2-arrays ---
! -- jacobian 
      if(jband)then
         ldjac=mljac+mujac+1
      else
         ldjac=n
      end if
! -- matrix e for linear algebra
      if(jband)then
         lde=2*mljac+mujac+1
      else
         lde=n
      end if
! -- mass matrix
      if (implct) then
          if (mlmas.ne.n) then
              ldmas=mlmas+mumas+1
          else
              ldmas=n
          end if
! ------ bandwith of "mas" not larger than bandwith of "jac"
          if (mlmas.gt.mljac.or.mumas.gt.mujac) then
              write (6,*) 'bandwith of "mas" not larger than bandwith of "jac"'
            arret=.true.
          end if
      else
          ldmas=0
      end if
      ldmas2=max(1,ldmas)
! ------- prepare the entry-points for the arrays in work -----
      ieyh=14
      iedy=ieyh+n
      iefx=iedy+n
      ieyhh=iefx+n
      iedyh=ieyhh+n
      iedel=iedyh+n
      iefxh=iedel+n
      iewh =iefxh+n
      iescal=iewh+n
      iehh =iescal+n
      iew  =iehh+km
      iea  =iew+km
      iejac=iea+km
      iee  =iejac+n*ldjac
      iemas=iee+n*lde
      iet=iemas+n*ldmas
      iesj=iet+n*km
      iesa=iesj+nzmax
! ------ total storage requirement -----------
      istore=iesa+nzmax-1
      if(istore.gt.lwork)then
         ierr = 0
         call realloc_double(work,istore,ierr)
         if (ierr /= 0) then
            write(lout,*)
     >         ' sodex insufficient storage for work, min. lwork=',istore
            arret=.true.
         end if
      end if
! ------- entry points for integer workspace -----
      ieip=21
      ienj=ieip+n
      ieiph=ienj+km
      ieia=ieiph+n
      ieja=ieia+n+1
! --------- total requirement ---------------
      istore=ieja+nzmax-1
      if(istore.gt.liwork)then
         ierr = 0
         call realloc_integer(iwork,istore,ierr)
         if (ierr /= 0) then
            write(lout,*)
     >         ' sodex insufficient storage for iwork, min. liwork=',istore
            arret=.true.
         end if
      end if
! ------ when a fail has occured, we return with idid=-1
      if (arret) then
         idid=-1
         return
      end if
! -------- call to core integrator ------------
      call sodcor(n,fcn,x,y,xend,hmax,h,km,rtol,atol,itol,jac,ijac,sjac,nzmax,isparse,
     &   mljac,mujac,dfx,idfx,mas,mlmas,mumas,solout,iout,idid,
     &   nmax,uround,nsequ,autnms,implct,jband,ldjac,lde,ldmas2,
     &   work(ieyh:lwork),work(iedy:lwork),work(iefx:lwork),work(ieyhh:lwork),work(iedyh:lwork),
     &   work(iedel:lwork),work(iefxh:lwork),work(iewh:lwork),work(iescal:lwork),work(iehh:lwork),
     &   work(iew:lwork),work(iea:lwork),work(iejac:lwork),work(iee:lwork),work(iemas:lwork),
     &   work(iet:lwork),iwork(ieip:liwork),iwork(ienj:liwork),iwork(ieiph:liwork),fac1,fac2,fac3,
     &   fac4,thet,safe1,safe2,wkjac,wkdec,wkrow,
     &   nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &   decsol,decsols,iwork(ieia:liwork),iwork(ieja:liwork),work(iesj:lwork),work(iesa:lwork),
     &   lrd,rpar_decsol,lid,ipar_decsol,lrpar,rpar,lipar,ipar)
! ----------- return -----------
      iwork(14)=nfcn
      iwork(15)=njac
      iwork(16)=nstep
      iwork(17)=naccpt
      iwork(18)=nrejct
      iwork(19)=ndec
      iwork(20)=nsol
      return
      end subroutine do_sodex


!
!  ----- ... and here is the core integrator  ----------
!
      subroutine sodcor(n,fcn,x,y,xend,hmax,h,km,rtol,atol,itol,
     &  jac,ijac,sjac,nzmax,isparse,mljac,mujac,dfx,idfx,mas,mlb,mub,
     &  solout,iout,idid,
     &  nmax,uround,nsequ,autnms,implct,banded,lfjac,le,
     &  ldmas,yh,dy,fx,yhh,dyh,del,fxh,wh,scal,hh,w,a,fjac,e,fmas,t,ip,
     &  nj,iphes,fac1,fac2,fac3,fac4,thet,safe1,safe2,wkjac,wkdec,wkrow,
     &  nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &  decsol,decsols,ia,ja,sparse_jac,sa,
     &  lrd,rpar_decsol,lid,ipar_decsol,lrpar,rpar,lipar,ipar)
! ----------------------------------------------------------
!     core integrator for sodex
!     parameters same as in sodex with workspace added 
! ---------------------------------------------------------- 
!         declarations 
! ---------------------------------------------------------- 
       implicit real*8 (a-h,o-z)
       interface
         include "num_solout.dek"
         include "num_mas.dek"
         include "num_fcn.dek"
         include "num_jac.dek"
         include "num_sjac.dek"
         include "mtx_decsol.dek"
         include "mtx_decsols.dek"
       end interface
       integer, intent(in) :: nzmax, lrpar, lipar, lrd, lid
       integer, intent(out) :: ia(n+1), ja(nzmax)
       real(dp), intent(out) :: sparse_jac(nzmax), sa(nzmax)
       dimension y(n),yh(n),dy(n),fx(n),yhh(n),dyh(n),del(n),fxh(n)
       dimension wh(n),scal(n),hh(km),w(km),a(km),fjac(lfjac,n),e(le,n)
       dimension fmas(ldmas,n),t(km,n),ip(n),nj(km),rtol(1),atol(1)
       dimension iphes(n),rwork(0),iwork(0)
       dimension rpar(lrpar),ipar(lipar),rpar_decsol(lrd),ipar_decsol(lid)
       logical reject,last,atov,caljac,autnms,implct,banded
       !common/stat/nfcn,njac,nstep,naccpt,nrejct,ndec,nsol
! ------- compute mass matrix for implicit case ----------
      if (implct) then
        call mas(n,fmas,ldmas,lrpar,rpar,lipar,ipar)
        mbdiag=mub+1
      end if
! *** *** *** *** *** *** ***
!  initialisations
! *** *** *** *** *** *** ***
       ierr=0
       if (banded) then
          mle=mljac
          mue=mujac
          mbjac=mljac+mujac+1
          mbb=mlb+mub+1
          mdiag=mle+mue+1
          mdiff=mle+mue-mub
       end if
! --- define the step size sequence
       if (nsequ.eq.1) then
           nj(1)=2
           nj(2)=6
           if (km.ge.3) nj(3)=10
           if (km.ge.4) nj(4)=14
           if (km.ge.5) nj(5)=22
           if (km.ge.6) nj(6)=34
           if (km.ge.7) nj(7)=50
           if (km.ge.8) then
              do 1 i=8,km
   1          nj(i)=2*nj(i-2)+nj(1) 
           end if
       end if
       a(1)=wkjac+(nj(1)+1)*wkrow+wkdec
       do 4 i=2,km
   4   a(i)=a(i-1)+nj(i)*wkrow+wkdec
       posneg=sign(1.d0,xend-x) 
       k=max0(2,min0(km-1,int(-dlog10(rtol(1)+atol(1))*.6d0+1.5d0))) 
       hmaxn=min(abs(hmax),abs(xend-x)) 
       h=max(abs(h),1.d-12) 
       h=posneg*min(h,hmaxn)
       theta=2*abs(thet)
       if (iout.ne.0) then
          irtrn=1
          nrsol=1
          xsol=x 
          xold=x
          nsolu=n
          call solout(nrsol,xold,xsol,nsolu,y,rwork,iwork,interp_y,lrpar,rpar,lipar,ipar,irtrn)
          yh=y
          if (irtrn.lt.0) goto 111
       end if
       err=0.d0
       w(1)=1.d30  
       do 8 i=1,n
       if (itol.eq.0) then
         scal(i)=atol(1)+rtol(1)*dabs(y(i))
       else
         scal(i)=atol(i)+rtol(i)*dabs(y(i))
       end if
   8   continue
       caljac=.false.
       reject=.false.
       last=.false.
  10   continue
       if (reject) theta=2*abs(thet)
       atov=.false.
! *** *** *** *** *** *** ***
! --- is xend reached in the next step?
! *** *** *** *** *** *** ***
       if (0.1d0*abs(xend-x).le.abs(x)*uround) go to 110
       hopt=h
       h=posneg*min(abs(h),abs(xend-x),hmaxn)
       if ((x+1.01d0*h-xend)*posneg.gt.0.d0) then
          h=xend-x 
          last=.true.
       end if
       if (ijac == 0 .or. .not.(theta.gt.thet.and..not.caljac)) then
          call fcn(n,x,y,dy,lrpar,rpar,lipar,ipar,ierr)
          if (ierr /= 0) goto 120
          nfcn=nfcn+1
       end if
       if (theta.gt.thet.and..not.caljac) then
! *** *** *** *** *** *** ***
!  computation of the jacobian
! *** *** *** *** *** *** ***
          njac=njac+1
          if (ijac.eq.0) then
! --- compute jacobian matrix numerically
              if (banded) then
! --- jacobian is banded
                  mujacp=mujac+1
                  md=min(mbjac,n)
                  do 16 kk=1,md
                  j=kk
 12               yhh(j)=y(j)
                  del(j)=dsqrt(uround*max(1.d-5,abs(y(j))))
                  y(j)=y(j)+del(j)
                  j=j+md
                  if (j.le.n) goto 12 
                  call fcn(n,x,y,yh,lrpar,rpar,lipar,ipar,ierr)
                  if (ierr /= 0) goto 120
                  j=kk
                  lbeg=max(1,j-mujac)
 14               lend=min(n,j+mljac)
                  y(j)=yhh(j)
                  mujacj=mujacp-j
                  do 15 l=lbeg,lend
 15               fjac(l+mujacj,j)=(yh(l)-dy(l))/del(j) 
                  j=j+md
                  lbeg=lend+1
                  if (j.le.n) goto 14
 16               continue
              else
! --- jacobian is full
                  do i=1,n
                     ysafe=y(i)
                     delt=dsqrt(uround*max(1.d-5,abs(ysafe)))
                     y(i)=ysafe+delt
                     call fcn(n,x,y,yh,lrpar,rpar,lipar,ipar,ierr)
                     if (ierr /= 0) goto 120
                     do j=1,n
                        fjac(j,i)=(yh(j)-dy(j))/delt
                     end do
                     y(i)=ysafe
                  end do
                  mljac=n
              end if
          else
! --- compute jacobian matrix analytically
               if (nzmax == 0) then
                  call jac(n,x,y,dy,fjac,lfjac,lrpar,rpar,lipar,ipar,ierr)
               else
                  call sjac(n,x,y,dy,nzmax,ia,ja,sparse_jac,lrpar,rpar,lipar,ipar,ierr)
               end if
              if (ierr /= 0) goto 120
          end if
          caljac=.true.
       end if
       if (.not.autnms.and..not.reject) then
           if (idfx.eq.0) then
! --- compute numerically the derivative with respect to x
               delt=dsqrt(uround*max(1.d-5,abs(x)))
               xdelt=x+delt
               call fcn(n,xdelt,y,yh,lrpar,rpar,lipar,ipar,ierr)
               if (ierr /= 0) goto 120
               do 19 j=1,n
  19           fx(j)=(yh(j)-dy(j))/delt
           else
! --- compute analytically the derivative with respect to x
               call dfx(n,x,y,fx)
           end if
       end if
! *** *** *** *** *** *** ***
! --- the first and last step 
! *** *** *** *** *** *** ***
       if (nstep.eq.0.or.last) then 
          nstep=nstep+1 
          do 20 j=1,k
          kc=j
       call simex(j,n,fcn,x,y,dy,fx,fjac,lfjac,fmas,ldmas,e,le,ip,h,km,
     &              hmaxn,t,scal,nj,hh,w,a,yhh,dyh,del,fxh,wh,err,safe1,
     &              fac,fac1,fac2,safe2,theta,mljac,mujac,mbjac,
     &              mlb,mub,mbb,mbdiag,mdiff,mle,mue,mdiag,errold,iphes,
     &              autnms,implct,banded,reject,atov,
     &              nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &              decsol,decsols,nzmax,isparse,ia,ja,sparse_jac,sa,
     &              lrd,rpar_decsol,lid,ipar_decsol,
     &              lrpar,rpar,lipar,ipar,ierr)
          if (ierr /= 0) goto 120
          if (atov) goto 10
          if (j.gt.1.and.err.le.1.d0) go to 60
  20      continue
          go to 55
       end if
! --- basic integration step  
  30   continue
       nstep=nstep+1
       if (nstep.ge.nmax) go to 120 
       kc=k-1
       do 40 j=1,kc
         call simex(j,n,fcn,x,y,dy,fx,fjac,lfjac,fmas,ldmas,e,le,ip,h,km,
     &              hmaxn,t,scal,nj,hh,w,a,yhh,dyh,del,fxh,wh,err,safe1,
     &              fac,fac1,fac2,safe2,theta,mljac,mujac,mbjac,
     &              mlb,mub,mbb,mbdiag,mdiff,mle,mue,mdiag,errold,iphes,
     &              autnms,implct,banded,reject,atov,
     &              nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &              decsol,decsols,nzmax,isparse,ia,ja,sparse_jac,sa,
     &              lrd,rpar_decsol,lid,ipar_decsol,
     &              lrpar,rpar,lipar,ipar,ierr)
          if (ierr /= 0) goto 120
          if (atov) goto 10
  40   continue
! *** *** *** *** *** *** ***
! --- convergence monitor
! *** *** *** *** *** *** ***
       if (k.eq.2.or.reject) go to 50
       if (err.le.1.d0) go to 60
       if (err.gt.(dble(nj(k+1)*nj(k))/4.d0)**2) go to 100  
 50    call simex(k,n,fcn,x,y,dy,fx,fjac,lfjac,fmas,ldmas,e,le,ip,h,km,
     &              hmaxn,t,scal,nj,hh,w,a,yhh,dyh,del,fxh,wh,err,safe1,
     &              fac,fac1,fac2,safe2,theta,mljac,mujac,mbjac,
     &              mlb,mub,mbb,mbdiag,mdiff,mle,mue,mdiag,errold,iphes,
     &              autnms,implct,banded,reject,atov,
     &              nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &              decsol,decsols,nzmax,isparse,ia,ja,sparse_jac,sa,
     &              lrd,rpar_decsol,lid,ipar_decsol,
     &              lrpar,rpar,lipar,ipar,ierr)
       if (ierr /= 0) goto 120
       if (atov) goto 10
       kc=k 
       if (err.le.1.d0) go to 60
! --- hope for convergence in line k+1
  55   if (err.gt.(dble(nj(k+1))/2.d0)**2) go to 100  
       kc=k+1
       call simex(kc,n,fcn,x,y,dy,fx,fjac,lfjac,fmas,ldmas,e,le,ip,h,km,
     &              hmaxn,t,scal,nj,hh,w,a,yhh,dyh,del,fxh,wh,err,safe1,
     &              fac,fac1,fac2,safe2,theta,mljac,mujac,mbjac,
     &              mlb,mub,mbb,mbdiag,mdiff,mle,mue,mdiag,errold,iphes,
     &              autnms,implct,banded,reject,atov,
     &              nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &              decsol,decsols,nzmax,isparse,ia,ja,sparse_jac,sa,
     &              lrd,rpar_decsol,lid,ipar_decsol,
     &              lrpar,rpar,lipar,ipar,ierr)
       if (ierr /= 0) goto 120
       if (atov) goto 10
       if (err.gt.1.d0) go to 100
! *** *** *** *** *** *** ***
! --- step is accepted  
! *** *** *** *** *** *** ***
  60   xold=x
       x=x+h
       do 70 i=1,n
          t1i=t(1,i)
          if (itol.eq.0) then
            scal(i)=atol(1)+rtol(1)*dabs(t1i)
          else
            scal(i)=atol(i)+rtol(i)*dabs(t1i)
          end if
  70      y(i)=t1i
       naccpt=naccpt+1  
       caljac=.false.
       if (iout.ne.0) then
          irtrn=1
          nrsol=naccpt+1
          xsol=x
          nsolu=n
          call solout(nrsol,xold,xsol,nsolu,y,rwork,iwork,interp_y,lrpar,rpar,lipar,ipar,irtrn)
          if (irtrn.lt.0) goto 111
          do 75 i=1,n
  75      yh(i)=y(i)
       end if
! --- compute optimal order
       if (kc.eq.2) then
          kopt=3  
          if (reject) kopt=2  
          go to 80
       end if
       if (kc.le.k) then
          kopt=kc 
          if (w(kc-1).lt.w(kc)*fac3) kopt=kc-1  
          if (w(kc).lt.w(kc-1)*fac4) kopt=min0(kc+1,km-1)
       else 
          kopt=kc-1
          if (kc.gt.3.and.w(kc-2).lt.w(kc-1)*fac3) kopt=kc-2
          if (w(kc).lt.w(kopt)*fac4) kopt=min0(kc,km-1)
       end if
! --- after a rejected step
  80   if (reject) then 
          k=min0(kopt,kc)
          h=posneg*min(abs(h),abs(hh(k)))
          reject=.false.
          go to 10
       end if
! --- compute step size for next step
       if (kopt.le.kc) then
          h=hh(kopt)
       else 
          if (kc.lt.k.and.w(kc).lt.w(kc-1)*fac4) then 
             h=hh(kc)*a(kopt+1)/a(kc)
          else
             h=hh(kc)*a(kopt)/a(kc) 
          end if  
       end if
       k=kopt
       h=posneg*abs(h)
       go to 10
! *** *** *** *** *** *** ***
! --- step is rejected  
! *** *** *** *** *** *** ***
 100   k=min(k,kc)
       if (k.gt.2.and.w(k-1).lt.w(k)*fac3) k=k-1
       nrejct=nrejct+1  
       h=posneg*hh(k)
       last=.false.
       reject=.true.
       if (caljac) goto 30
       go to 10
! --- solution exit
 110   continue
       idid=1
       return
! --- solout says stop now exit
 111  continue
      idid=2
      return
! --- fail exit
 120   idid=-1
       return
       end subroutine sodcor


! *** *** *** *** *** *** ***
!     s u b r o u t i n e    s i m e x
! *** *** *** *** *** *** ***
!
      subroutine simex(jj,n,fcn,x,y,dy,fx,fjac,lfjac,fmas,ldmas,e,le,ip,
     &          h,km,hmaxn,t,scal,nj,hh,w,a,yh,dyh,del,fxh,wh,err,safe1,
     &          fac,fac1,fac2,safe2,theta,mljac,mujac,mbjac,
     &          mlb,mub,mbb,mbdiag,mdiff,mle,mue,mdiag,errold,iphes,
     &          autnms,implct,banded,reject,atov,
     &          nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &          decsol,decsols,nzmax,isparse,ia,ja,sparse_jac,sa,
     &          lrd,rpar_decsol,lid,ipar_decsol,
     &          lrpar,rpar,lipar,ipar,ierr)
! --- this subroutine computes the j-th line of the
! --- extrapolation table and provides an estimate  
! --- of the optimal step size 
       use utils_lib,only:is_bad_num
       implicit real*8 (a-h,o-z)
       interface
         include "mtx_decsol.dek"
         include "mtx_decsols.dek"
       end interface
       integer, intent(in) :: nzmax, lrpar, lipar, lrd, lid
       integer, intent(inout) :: ia(n+1), ja(nzmax)
       integer, intent(out) :: ierr
       real(dp), intent(inout) :: sparse_jac(nzmax), sa(nzmax)
       dimension y(n),yh(n),dy(n),fx(n),dyh(n),del(n),fxh(n)
       dimension wh(n),scal(n),hh(km),w(km),a(km),fjac(lfjac,n),e(le,n)
       dimension fmas(ldmas,n),t(km,n),ip(n),nj(km),iphes(n)
       dimension rpar(lrpar),ipar(lipar),rpar_decsol(lrd),ipar_decsol(lid)
       logical atov,reject,autnms,implct,banded
       !common/stat/nfcn,njac,nstep,naccpt,nrejct,ndec,nsol
       ierr=0
! *** *** *** *** *** *** ***
!  compute the matrix e and its decomposition
! *** *** *** *** *** *** ***
      hj=h/nj(jj)
      hji=1.d0/hj
      if (implct) then
          if (banded) then
! --- the matrix e (b is a banded matrix, jacobian a banded matrix)
              do 127 j=1,n
              i1j=max0(1,mujac+2-j)
              i2j=min(mbjac,mujac+1-j+n)
              do 125 i=i1j,i2j
 125          e(i+mle,j)=-fjac(i,j)
              i1b=max0(1,mub+2-j)
              i2b=min0(mbb,mub+1-j+n)
              do 126 i=i1b,i2b
              ib=i+mdiff
 126          e(ib,j)=e(ib,j)+hji*fmas(i,j)
 127          continue
              call decsol(0,n,le,e,mle,mue,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
              if (ier.ne.0) goto 79
          else
              if (mlb.ne.n) then
c --- the matrix e (b is a banded matrix, jacobian a full matrix)
                  do 225 j=1,n
                  do 225 i=1,n
 225              e(i,j)=-fjac(i,j)
                  do 226 j=1,n
                  i1=max0(1,j-mub)
                  i2=min0(n,j+mlb)
                  do 226 i=i1,i2
 226              e(i,j)=e(i,j)+hji*fmas(i-j+mbdiag,j)
                  call decsol(0,n,le,e,n,n,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
                  if (ier.ne.0) goto 79
              else
! --- the matrix e (b is a full matrix, jacobian a full matrix)
                  if (mljac.eq.n) then
                      do 324 j=1,n
                      do 324 i=1,n
 324                  e(i,j)=fmas(i,j)*hji-fjac(i,j)
                      call decsol(0,n,le,e,n,n,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
                      if (ier.ne.0) goto 79
                  else
! --- the matrix e (b is a full matrix, jacobian a banded matrix)
                      madd=mujac+1
                      do 405 j=1,n
                      do 405 i=1,n
  405                 e(i,j)=fmas(i,j)*hji
                      do 406 j=1,n
                      i1=max0(1,j-mujac)
                      i2=min0(n,j+mljac)
                      do 406 i=i1,i2
  406                 e(i,j)=e(i,j)-fjac(i-j+madd,j)
                      call decsol(0,n,le,e,n,n,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
                      if (ier.ne.0) goto 79
                  end if
              end if
          end if
      else
          if (banded) then
! --- the matrix e (b=identity, jacobian a banded matrix)
              do 427 j=1,n
              i1j=max0(1,mujac+2-j)
              i2j=min(mbjac,mujac+1-j+n)
              do 425 i=i1j,i2j
 425          e(i+mle,j)=-fjac(i,j)
 427          e(mdiag,j)=e(mdiag,j)+hji
              call decsol(0,n,le,e,mle,mue,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
              if (ier.ne.0) goto 79
          else if (nzmax > 0) then
! --- the matrix e (b=identity, jacobian a sparse matrix)
               sa(1:nzmax) = -sparse_jac(1:nzmax)
               do j=1,n
                  do k=ia(j),ia(j+1)-1
                     i = ja(k)
                     if (i == j) then
                        sa(k) = sa(k) + hji
                        exit
                     end if
                  end do
               end do
               call decsols(0,n,nzmax,ia,ja,sa,dy,lrd,rpar_decsol,lid,ipar_decsol,ier)
               if (ier.ne.0) goto 79 
          else
! --- the matrix e (b=identity, jacobian a full matrix)
               do 526 j=1,n
               do 525 i=1,n
 525           e(i,j)=-fjac(i,j)
 526           e(j,j)=e(j,j)+hji
               call decsol(0,n,le,e,n,n,dy,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
               if (ier.ne.0) goto 79 
          end if
      end if
      ndec=ndec+1
! *** *** *** *** *** *** ***
! --- starting procedure
! *** *** *** *** *** *** ***
       do 20 i=1,n
       yh(i)=y(i)
  20   del(i)=dy(i)
       if (.not.autnms) then
           do 634 i=1,n
 634       del(i)=del(i)+hj*fx(i)
       end if
       if (banded) then
           call decsol(1,n,le,e,mle,mue,del,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
           if (ier.ne.0) goto 79
       else if (nzmax > 0) then
            call decsols(1,n,nzmax,ia,ja,sa,del,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       else
            call decsol(1,n,le,e,n,n,del,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       end if
       nsol=nsol+1
       m=nj(jj)
! *** *** *** *** *** *** ***
! --- semi-implicit mid-point rule
! *** *** *** *** *** *** ***
       do 30 mm=1,m
       do 23 i=1,n
  23   yh(i)=yh(i)+del(i)
       call fcn(n,x+hj*mm,yh,dyh,lrpar,rpar,lipar,ipar,ierr)
       if (ierr /= 0) goto 79
       nfcn=nfcn+1
       if (implct) then
           if (mlb.eq.n) then
               do 361 i=1,n
               sum=0.d0
               do 360 j=1,n
 360           sum=sum+fmas(i,j)*del(j)
 361           wh(i)=sum
           else
               do 363 i=1,n
               sum=0.d0
               j1b=max0(1,i-mlb)
               j2b=min0(n,i+mub)
               do 362 j=j1b,j2b
 362           sum=sum+fmas(i-j+mbdiag,j)*del(j)
 363           wh(i)=sum
           end if
           if (mm.eq.m) goto 30
           do 24 i=1,n
  24       dyh(i)=2.d0*(dyh(i)-hji*wh(i))
       else
           if (mm.eq.m) goto 30
           do 28 i=1,n
  28       dyh(i)=2.d0*(dyh(i)-hji*del(i))
       end if
       if (banded) then
           call decsol(1,n,le,e,mle,mue,dyh,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
           if (ier.ne.0) goto 79
       else if (nzmax > 0) then
            call decsols(1,n,nzmax,ia,ja,sa,dyh,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       else
            call decsol(1,n,le,e,n,n,dyh,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       end if
       nsol=nsol+1
       if (mm.eq.1.and.jj.le.2) then
! --- stability check
          del1=0.d0
          do 21 i=1,n
  21      del1=del1+(del(i)/scal(i))**2
          del1=dsqrt(del1)
          del2=0.d0
          do 26 i=1,n
  26      del2=del2+(dyh(i)/scal(i))**2
          del2=.5d0*dsqrt(del2)
          theta=del2/max(1.d0,del1)
          if (theta.gt.1.d0) goto 79
       end if
       do 25 i=1,n
  25   del(i)=del(i)+dyh(i)
  30   continue
  
! --- final step (due to bader)
       if (implct) then
           do 34 i=1,n
  34       dyh(i)=dyh(i)-hji*wh(i)
       else
           do 36 i=1,n
  36       dyh(i)=dyh(i)-hji*del(i)
       end if
       if (banded) then
           call decsol(1,n,le,e,mle,mue,dyh,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
           if (ier.ne.0) goto 79
       else if (nzmax > 0) then
            call decsols(1,n,nzmax,ia,ja,sa,dyh,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       else
            call decsol(1,n,le,e,n,n,dyh,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
            if (ier.ne.0) goto 79
       end if
       nsol=nsol+1
       t(jj,1:n) = yh(1:n)+dyh(1:n)
! *** *** *** *** *** *** ***
! --- polynomial extrapolation
! *** *** *** *** *** *** ***
       if (jj.eq.1) goto 80
       do 60 l=jj,2,-1
       fac=(dble(nj(jj))/dble(nj(l-1)))**2-1.d0
       do 60 i=1,n
       t(l-1,i)=t(l,i)+(t(l,i)-t(l-1,i))/fac
  60   continue
       err=0.d0
       do 65 i=1,n
  65   err=err+min(abs((t(1,i)-t(2,i)))/scal(i),1.d10)**2  
       if (is_bad_num(err)) goto 79
       if (err.ge.1.d20) goto 79
       err=dsqrt(err/dble(n))
       if (jj.gt.2.and.err.ge.errold) goto 79
       errold=dmax1(4*err,1.d0)
! --- compute optimal step sizes
       expo=1.d0/(2*jj-2)
       facmin=fac1**expo
       fac=min(fac2/facmin,max(facmin,(err/safe1)**expo/safe2))
       fac=1.d0/fac
       hh(jj)=min(abs(h)*fac,hmaxn)
       w(jj)=a(jj)/hh(jj)  
       goto 80
  79   atov=.true.
       h=h*0.5d0
       reject=.true.
  80   continue
       call decsol(2,n,le,e,n,n,dyh,ip,lrd,rpar_decsol,lid,ipar_decsol,ier)
       if (ier.ne.0) then
         atov=.true.
         h=h*0.5d0
         reject=.true.
       end if
       return
       end subroutine simex


      real(dp) function interp_y(i,s,rwork,iwork,ierr)
         use alert_lib,only:alert
         integer, intent(in) :: i
         real(dp), intent(in) :: s
         real(dp), intent(inout), target :: rwork(*)
         integer, intent(inout), target :: iwork(*)
         integer, intent(out) :: ierr
         ierr = -1 ! no dense output
         call alert(ierr,'no dense output for sodex')
         interp_y = 0
      end function interp_y


      end module mod_sodex