! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 - 1307 USA
!
! ***********************************************************************

! the following is the copyright for radau5

! copyright (c) 2004, ernst hairer

! redistribution and use in source and binary forms, with or without 
! modification, are permitted provided that the following conditions are 
! met:

!  -  redistributions of source code must retain the above copyright 
! notice, this list of conditions and the following disclaimer.

!  -  redistributions in binary form must reproduce the above copyright 
! notice, this list of conditions and the following disclaimer in the 
! documentation and/or other materials provided with the distribution.

! this software is provided by the copyright holders and contributors “as 
! is” and any express or implied warranties, including, but not limited 
! to, the implied warranties of merchantability and fitness for a 
! particular purpose are disclaimed. in no event shall the regents or 
! contributors be liable for any direct, indirect, incidental, special, 
! exemplary, or consequential damages (including, but not limited to, 
! procurement of substitute goods or services; loss of use, data, or 
! profits; or business interruption) however caused and on any theory of 
! liability, whether in contract, strict liability, or tort (including 
! negligence or otherwise) arising in any way out of the use of this 
! software, even if advised of the possibility of such damage.


      module mod_radau5
      use mod_dc_decsol
      use utils_lib
      use const_def, only: dp
            
      integer, parameter :: in_progress = 0
      integer, parameter :: newton_iterations_finished = 1
      integer, parameter :: start_with_old_jacobian = 2
      integer, parameter :: start_with_new_jacobian = 3
      integer, parameter :: keep_going_with_same_time_step = 4
      integer, parameter :: retry_with_smaller_step = 5
      integer, parameter :: finished = 6
      
      logical, parameter :: dbg = .false.


      contains


      subroutine do_radau5(
     >      n,fcn,x,y,xend,
     >      h,max_step_size,max_steps,
     >      rtol,atol,itol,
     >      jac,ijac,sjac,nzmax,isparse,
     >      mljac_in,mujac_in,
     >      mas,imas,mlmas,mumas_in,
     >      solout,iout,
     >      radau5_debug_routine,call_debug_routine,
     >      decsol, decsols,  
     >      lrd, rpar_decsol, lid, ipar_decsol, 
     >      decsolc, decsolcs, 
     >      lcd, cpar_decsol, lrdc, rpar_decsolc, lidc, ipar_decsolc, 
     >      work,lwork,iwork,liwork,
     >      lrpar,rpar,lipar,ipar,
     >      lout,idid)
! *** *** *** *** *** *** *** *** *** *** *** *** ***
!          declarations 
! *** *** *** *** *** *** *** *** *** *** *** *** ***
         implicit none
         !implicit real(dp) (a - h,o - z)
         integer, intent(in) :: n ! the dimension of the system
         interface ! subroutine for computing the value of f(x,y)
            include "num_fcn.dek"
         end interface
         real(dp), intent(inout) :: x 
         real(dp), intent(inout) :: y(n) 
         real(dp), intent(in) :: xend ! desired final x value (positive or negative)
         real(dp), intent(inout) :: h 
         real(dp), intent(in) :: max_step_size
         integer, intent(in) :: max_steps
         real(dp), intent(inout) :: rtol(*) ! relative error tolerance(s)
         real(dp), intent(inout) :: atol(*) ! absolute error tolerance(s)
         integer, intent(in) :: itol ! switch for rtol and atol
         interface ! subroutines for computing the jacobian
            include "num_jac.dek" ! for dense or banded matrix
            include "num_sjac.dek" ! for sparse matrix
         end interface
         integer, intent(in) :: ijac, nzmax, isparse, mljac_in, mujac_in
         interface ! subroutine for computing the mass matrix
            include "num_mas.dek"
         end interface
         integer, intent(in) :: imas ! gives information on the mass - matrix:
         integer, intent(in) :: mlmas
         integer, intent(in) :: mumas_in
         interface ! subroutine called after each successful step
            include "num_solout.dek"
         end interface
         integer, intent(in)  :: iout ! switch for calling the subroutine solout
         interface ! subroutine for debugging
            include "num_radau5_debug_routine.dek"
         end interface
         logical, intent(in)  :: call_debug_routine
         
         interface
            include "mtx_decsol.dek"
            include "mtx_decsols.dek"
         end interface
         ! work arrays for decsol and decsols
         integer, intent(in) :: lrd, lid
         real(dp), intent(inout), target :: rpar_decsol(lrd)
         integer, intent(inout), target :: ipar_decsol(lid)
         interface
            include "mtx_decsolc.dek"
            include "mtx_decsolcs.dek"
         end interface
         ! work arrays for decsolc and decsolcs
         integer, intent(in) :: lcd, lrdc, lidc
         complex*16, intent(inout), target :: cpar_decsol(lcd)
         real(dp), intent(inout), target :: rpar_decsolc(lrdc)
         integer, intent(inout), target :: ipar_decsolc(lidc)

         integer, intent(in) :: lwork, liwork
         real(dp), pointer :: work(:) ! (lwork)
         integer, pointer :: iwork(:) ! (liwork)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(lrpar)
         integer, intent(inout) :: ipar(lipar)
         integer, intent(in)  :: lout
         integer, intent(out)  :: idid
         
      ! LOCALS
      logical :: implct, jband, arret, startn, pred, okay_to_reuse_jac
      integer :: nfcn, njac, nstep, naccpt, nrejct, ndec, nsol, i
      integer :: nmax, nit, nind1, nind2, nind3, m1, m2, n_minus_m1, nerror
      integer :: ldjac, lde1, ldmas, ijob, ldmas2, ldmax, iez1, iez2, iez3
      integer :: iey0, iescal, ief1, ief2, ief3, iecon, iejac, iemas
      integer :: iee1, iee2r, iee2i, iesj, iesa, iesar, iesai
      integer :: ieip1, ieip2, ieiph, ieia, ieja, ierr
      integer :: mljac, mujac, mumas, needed_lwork, needed_liwork
      real(dp) :: uround, expm, quot, quot1, quot2, safe, thet, tolst
      real(dp) :: fnewt, hmax, facl, facr
      
      mljac = mljac_in; mujac = mujac_in; mumas = mumas_in
      
! *** *** *** *** *** *** ***
!        setting the parameters 
! *** *** *** *** *** *** ***
       nfcn = 0
       njac = 0
       nstep = 0
       naccpt = 0
       nrejct = 0
       ndec = 0
       nsol = 0
       arret = .false.
!  --------  uround   smallest number satisfying 1d0 + uround>1d0  
      if (work(1) == 0d0) then
         uround = 1.0d-16
      else
         uround = work(1)
         if (uround <= 1.0d-19 .or. uround >= 1d0) then
            if (lout > 0) write(lout,*)' coefficients have 20 digits, uround = ',work(1)
            arret = .true.
         end if
      end if
!  --------  check and change the tolerances
      expm = 2d0/3d0
      if (itol == 0) then
          if (atol(1) <= 0d0 .or. rtol(1) <= 10d0*uround) then
              write (6,*) ' tolerances are too small'
              arret = .true.
          else
              quot = atol(1)/rtol(1)
              rtol(1) = 0.1d0*rtol(1)**expm
              atol(1) = rtol(1)*quot
          end if
      else
          do i = 1,n
          if (atol(i) <= 0d0 .or. rtol(i) <= 10d0*uround) then
              write (6,*) ' tolerances(',i,') are too small'
              arret = .true.
          else
              quot = atol(i)/rtol(i)
              rtol(i) = 0.1d0*rtol(i)**expm
              atol(i) = rtol(i)*quot
          end if
          end do
      end if
!  --------  nmax , the maximal number of steps  ----- 
      if (max_steps == 0) then
         nmax = 100000
      else
         nmax = max_steps
         if (nmax <= 0) then
            if (lout > 0) write(lout,*)' wrong input max_steps = ',max_steps
            arret = .true.
         end if
      end if
!  -------- reuse jacobian or remake before factor
      okay_to_reuse_jac = (iwork(2) == 0)
!  --------  nit    maximal number of newton iterations
      if (iwork(3) == 0) then
         nit = 7
      else
         nit = iwork(3)
         if (nit <= 0) then
            if (lout > 0) write(lout,*)' curious input iwork(3) = ',iwork(3)
            arret = .true.
         end if
      end if
!  --------  startn  switch for starting values of newton iterations
      if(iwork(4) == 0)then
         startn = .false.
      else
         startn = .true.
      end if
!  --------  parameter for differential - algebraic components
      nind1 = iwork(5)
      nind2 = iwork(6)
      nind3 = iwork(7)
      if (nind1 == 0) nind1 = n
      if (nind1 + nind2 + nind3 /= n) then
       if (lout > 0) write(lout,*)' curious input for iwork(5,6,7) = ',nind1,nind2,nind3
       arret = .true.
      end if
!  --------  pred   step size control
      if(iwork(8) <= 1)then
         pred = .true.
      else
         pred = .false.
      end if
!  --------  parameter for second order equations
      m1 = iwork(9)
      m2 = iwork(10)
      n_minus_m1 = n - m1
      if (m1 == 0) m2 = n
      if (m2 == 0) m2 = m1
      if (m1 < 0 .or. m2 < 0 .or. m1 + m2 > n) then
       if (lout > 0) write(lout,*)' curious input for iwork(9,10) = ',m1,m2
       arret = .true.
      end if
      nerror = iwork(11) ! number of variables to use for tolerances
      if (nerror == 0) nerror = n
!  ---------  safe     safety factor in step size prediction
      if (work(2) == 0d0) then
         safe = 0.9d0
      else
         safe = work(2)
         if (safe <= 0.001d0 .or. safe >= 1d0) then
            if (lout > 0) write(lout,*)' curious input for work(2) = ',work(2)
            arret = .true.
         end if
      end if
!  ------  thet     decides whether the jacobian should be recomputed;
      if (work(3) == 0d0) then
         thet = 0.001d0
      else
         thet = work(3)
         if (thet >= 1d0) then
            if (lout > 0) write(lout,*)' curious input for work(3) = ',work(3)
            arret = .true.
         end if
      end if
!  ---  fnewt   stopping criterion for newton's method, usually chosen <1.
      tolst = rtol(1)
      if (work(4) == 0d0) then
         fnewt = max(10*uround/tolst,min(0.03d0,tolst**0.5d0))
      else
         fnewt = work(4)
         if (fnewt <= uround/tolst) then
            if (lout > 0) write(lout,*)' curious input for work(4) = ',work(4)
            arret = .true.
         end if
      end if
!  ---  quot1 and quot2: if quot1 < hnew/hold < quot2, step size = const.
      if (work(5) == 0d0) then
         quot1 = 1d0
      else
         quot1 = work(5)
      end if
      if (work(6) == 0d0) then
         quot2 = 1.2d0
      else
         quot2 = work(6)
      end if
      if (quot1 > 1d0 .or. quot2 < 1d0) then
         if (lout > 0) write(lout,*)' curious input for work(5,6) = ',quot1,quot2
         arret = .true.
      end if
!  --------  maximal step size
      if (max_step_size == 0d0) then
         hmax = xend - x
      else
         hmax = max_step_size
      end if 
!  -------   facl,facr     parameters for step size selection
      if(work(8) == 0d0)then
         facl = 5d0
      else
         facl = 1d0/work(8)
      end if
      if(work(9) == 0d0)then
         facr = 1d0/8d0
      else
         facr = 1d0/work(9)
      end if
      if (facl < 1d0 .or. facr > 1d0) then
            if (lout > 0) write(lout,*)' curious input work(8,9) = ',work(8),work(9)
            arret = .true.
         end if
! *** *** *** *** *** *** *** *** *** *** *** *** ***
!         computation of array entries
! *** *** *** *** *** *** *** *** *** *** *** *** ***
!  ----  implicit, banded or not ?
      implct = imas /= 0
      jband = mljac < n_minus_m1
!  --------  computation of the row - dimensions of the 2 - arrays  --- 
!  --  jacobian  and  matrices e1, e2
      if (jband) then
         ldjac = mljac + mujac + 1
         lde1 = mljac + ldjac
      else
         mljac = n_minus_m1
         mujac = n_minus_m1
         ldjac = n_minus_m1
         lde1 = n_minus_m1
      end if
!  --  mass matrix
      if (implct) then
          if (mlmas /= n_minus_m1) then
              ldmas = mlmas + mumas + 1
              if (nzmax > 0) then ! sparse jacobian
                 ijob = 9
              else if (jband) then
                 ijob = 4
              else
                 ijob = 3
              end if
          else
              mumas = n_minus_m1
              ldmas = n_minus_m1
              ijob = 5
          end if
!  ------  bandwith of "mas" not smaller than bandwith of "jac"
          if (mlmas > mljac .or. mumas > mujac) then
             if (lout > 0) write(lout,*) 'bandwith of "mas" not smaller than bandwith of "jac"'
            arret = .true.
          end if
      else
          ldmas = 0
          if (nzmax > 0) then ! sparse jacobian
             ijob = 8
          else if (jband) then
             ijob = 2
          else
             ijob = 1
             if (n > 2 .and. iwork(1) /= 0) then
               ijob = 7
               if (lout > 0) write(lout,*) 'radau5: hessenberg option not supported in mesa'
               arret = .true.
             end if
          end if
      end if
      ldmas2 = max(1,ldmas)

      call calculate_work_sizes(
     >      n, ldjac, ldmas, ldmax, lde1, n_minus_m1, nzmax,
     >      iez1, iez2, iez3, iey0, iescal, ief1, ief2, ief3, iecon, 
     >      iejac, iemas, iee1, iee2r, iee2i, iesj, iesa, iesar, iesai, 
     >      needed_lwork, ieip1, ieip2, ieiph, ieia, ieja, needed_liwork)
     
      if(needed_lwork > lwork)then
         ierr = 0
         call realloc_double(work,needed_lwork,ierr)
         if (ierr /= 0) then
            write(lout,*)
     >         ' insufficient storage for work, min. lwork = ',needed_lwork
            arret = .true.
         end if
      end if
      
      if(needed_liwork > liwork)then
         ierr = 0
         call realloc_integer(iwork,needed_liwork,ierr)
         if (ierr /= 0) then
            write(lout,*)
     >         ' insufficient storage for iwork, min. liwork = ',needed_liwork
            arret = .true.
         end if
      end if   
      
!  ------  when a fail has occured, we return with idid = -1
      if (arret) then
         idid = -1
         return
      end if
!  --------  call to core integrator  ------------ 
      call radcor(
     &   n,fcn,x,y,xend,hmax,h,rtol,atol,itol,okay_to_reuse_jac,
     &   jac,ijac,sjac,nzmax,isparse,mljac,mujac,mas,mlmas,mumas,solout,iout,
     &   radau5_debug_routine, call_debug_routine,  
     &   decsol, decsols,  
     &   lrd, rpar_decsol, lid, ipar_decsol, 
     &   decsolc, decsolcs, 
     &   lcd, cpar_decsol, lrdc, rpar_decsolc, lidc, ipar_decsolc, 
     &   iwork(ieia:liwork),iwork(ieja:liwork),work(iesj:lwork),
     &   work(iesa:lwork),work(iesar:lwork),work(iesai:lwork),
     &   idid,nmax,uround,safe,thet,fnewt,quot1,quot2,nit,ijob,startn,
     &   nind1,nind2,nind3,pred,facl,facr,m1,m2,n_minus_m1,nerror,
     &   implct,jband,ldjac,lde1,ldmas2,work(iez1:lwork),work(iez2:lwork),
     &   work(iez3:lwork),work(iey0:lwork),work(iescal:lwork),work(ief1:lwork),work(ief2:lwork),
     &   work(ief3:lwork),work(iejac:lwork),work(iee1:lwork),work(iee2r:lwork),work(iee2i:lwork),
     &   work(iemas:lwork),iwork(ieip1:liwork),iwork(ieip2:liwork),iwork(ieiph:liwork),
     &   work(iecon:lwork),nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &   lrpar,rpar,lipar,ipar,lout)
      iwork(14) = nfcn
      iwork(15) = njac
      iwork(16) = nstep
      iwork(17) = naccpt
      iwork(18) = nrejct
      iwork(19) = ndec
      iwork(20) = nsol
!  --------  restore tolerances
      expm = 1d0/expm
      if (itol == 0) then
              quot = atol(1)/rtol(1)
              rtol(1) = (10d0*rtol(1))**expm
              atol(1) = rtol(1)*quot
      else
          do i = 1,n
              quot = atol(i)/rtol(i)
              rtol(i) = (10d0*rtol(i))**expm
              atol(i) = rtol(i)*quot
          end do
      end if
!  -----------  return  ----------- 
      return
      end subroutine do_radau5


      subroutine calculate_work_sizes(
     >      n, ldjac, ldmas, ldmax, lde1, n_minus_m1, nzmax,
     >      iez1, iez2, iez3, iey0, iescal, ief1, ief2, ief3, iecon, 
     >      iejac, iemas, iee1, iee2r, iee2i, iesj, iesa, iesar, iesai, 
     >      lwork, ieip1, ieip2, ieiph, ieia, ieja, liwork)
         implicit none
         integer, intent(in) :: n, ldjac, ldmas, ldmax, lde1, n_minus_m1, nzmax
         integer, intent(out) :: 
     >      iez1, iez2, iez3, iey0, iescal, ief1, ief2, ief3, iecon, 
     >      iejac, iemas, iee1, iee2r, iee2i, iesj, iesa, iesar, iesai, 
     >      lwork, ieip1, ieip2, ieiph, ieia, ieja, liwork
     
         iez1 = 21
         iez2 = iez1 + n
         iez3 = iez2 + n*2
         iey0 = iez3 + n
         iescal = iey0 + n
         ief1 = iescal + n
         ief2 = ief1 + n
         ief3 = ief2 + n
         iecon = ief3 + n
         iejac = iecon + 4*n + 4
         iemas = iejac + n*ldjac
         iee1 = iemas + n_minus_m1*ldmas
         iee2r = iee1 + n_minus_m1*lde1
         iee2i = iee2r + n_minus_m1*lde1
         iesj = iee2i + n_minus_m1*lde1
         iesa = iesj + nzmax
         iesar = iesa + nzmax
         iesai = iesar + nzmax
         lwork = iesai + nzmax - 1  
         
         ieip1 = 21
         ieip2 = ieip1 + n_minus_m1
         ieiph = ieip2 + n_minus_m1
         ieia = ieiph + n_minus_m1
         ieja = ieia + n + 1
         liwork = ieja + nzmax - 1
         
      end subroutine calculate_work_sizes

!
! ***********************************************************
!
      subroutine radcor(
     &   n,fcn,x,y,xend,hmax,h,rtol,atol,itol,okay_to_reuse_jac,
     &   jac,ijac,sjac,nzmax,isparse,mljac,mujac,mas,mlmas,mumas,solout,iout,
     &   radau5_debug_routine, call_debug_routine,  
     &   decsol, decsols,  
     &   lrd, rpar_decsol, lid, ipar_decsol, 
     &   decsolc, decsolcs, 
     &   lcd, cpar_decsol, lrdc, rpar_decsolc, lidc, ipar_decsolc, 
     &   ia,ja,sparse_jac,sa,sar,sai,
     &   idid,nmax,uround,safe,thet,fnewt,quot1,quot2,nit,ijob,startn,
     &   nind1,nind2,nind3,pred,facl,facr,m1,m2,n_minus_m1,nerror,
     &   implct,banded,ldjac,lde1,ldmas,z1,z2,z3,
     &   y0,scal,f1,f2,f3,fjac,e1,e2r,e2i,fmas,ip1,ip2,iphes,
     &   rwork,nfcn,njac,nstep,naccpt,nrejct,ndec,nsol,
     &   lrpar,rpar,lipar,ipar,lout)
!  ---------------------------------------------------------- 
!     core integrator for radau5
!     parameters same as in radau5 with workspace added 
!  ----------------------------------------------------------  
!         declarations 
!  ----------------------------------------------------------  
      implicit none
      interface
         include "num_solout.dek"
         include "num_mas.dek"
         include "num_fcn.dek"
         include "num_jac.dek"
         include "num_sjac.dek"
         include "num_radau5_debug_routine.dek"
         include "mtx_decsol.dek"
         include "mtx_decsols.dek"
         include "mtx_decsolc.dek"
         include "mtx_decsolcs.dek"
      end interface
      integer, intent(in) :: 
     >      n, 
     >      nzmax, 
     >      lrpar, 
     >      lipar, 
     >      lcd, 
     >      lrdc, 
     >      lidc, 
     >      lrd, 
     >      lid, 
     >      itol, 
     >      lout
      integer, intent(out) :: 
     >      ia(n + 1), 
     >      ja(nzmax), 
     >      nfcn, 
     >      njac, 
     >      nstep, 
     >      naccpt, 
     >      nrejct, 
     >      ndec, 
     >      nsol
      real(dp), intent(out) :: 
     >      sparse_jac(nzmax), 
     >      sa(nzmax), 
     >      sar(nzmax), 
     >      sai(nzmax)
      complex*16, target :: cpar_decsol(lcd)
      real*8, target :: rpar_decsol(lrd)
      integer, target :: ipar_decsol(lid)
      real*8, target :: rpar_decsolc(lrd)
      integer, target :: ipar_decsolc(lid)
      real(dp), intent(inout), target :: rpar(lrpar)
      integer, intent(inout), target :: ipar(lipar)
      real(dp), target :: rwork(4 + 4*n)
      real(dp) :: atol(*), rtol(*)
      integer :: 
     >      ijac, 
     >      isparse, 
     >      mljac, 
     >      mujac, 
     >      mlmas, 
     >      mumas, 
     >      iout, 
     >      idid, 
     >      nmax, 
     >      nerror ! variables 1 to nerror are used in estimating local errors
      integer :: 
     >      ip1(n_minus_m1), 
     >      ip2(n_minus_m1), 
     >      iphes(n_minus_m1), 
     >      nind1, ! dimension of the index 1 variables
     >      nind2, ! dimension of the index 2 variables
     >      nind3, ! dimension of the index 3 variables
     >      m1, ! special structure
     >      m2, ! special structure
     >      n_minus_m1 ! n - 1
      integer :: 
     >      nit, ! maximal number of newton iterations
     >      ijob, ! specifies form of matrices for decomr, decomc, slvrad, estrad
     >      ldjac, 
     >      lde1, 
     >      ldmas
      real(dp) :: 
     >      fjac(ldjac,n), ! jacobian matrix (dense or banded)
     >      fmas(ldmas,n_minus_m1), ! mass matrix (dense or banded)
     >      quot1, ! if quot1 < hnew/hold < quot2, step size = const  --  work 5
     >      quot2,  ! if quot1 < hnew/hold < quot2, step size = const  --  work 6
     >      facl, ! parameter for step size selection  --  work 8
     >      facr, ! parameter for step size selection  --  work 9
     >      y(n), 
     >      z1(n), 
     >      z2(2*n), 
     >      z3(n), 
     >      y0(n), 
     >      scal(n), 
     >      f1(n), 
     >      f2(n), 
     >      f3(n),
     >      e1(lde1,n_minus_m1), ! fac1*M - fjac, then factored
     >      e2r(lde1,n_minus_m1), ! alphn*M - fjac, then factored
     >      e2i(lde1,n_minus_m1) ! betan*M, then factored
      logical :: 
     >      call_debug_routine, 
     >      okay_to_reuse_jac,
     >      implct, 
     >      banded, 
     >      startn, ! if true then start from 0; else extrapolate from previous step
     >      pred ! switch for predictive controller
      real(dp) :: 
     >      x, ! current
     >      xend, ! end
     >      hmax, ! max stepsize
     >      h, ! step size
     >      uround, ! smallest number s.t. not dropped by roundoff when add to 1.0
     >      safe, ! safety factor in step size prediction  --  work 2
     >      thet, ! decides whether the jacobian should be recomputed  --  work 3
     >      fnewt ! stopping criterion for newton's method  --  work 4
      
!     LOCALS
      real(dp), pointer :: cont(:)
      integer :: 
     >      n2, 
     >      n3, 
     >      nn, 
     >      nn2, 
     >      nn3, 
     >      lrc, 
     >      ierr_cnt, 
     >      irtrn, 
     >      nrsol, 
     >      i, 
     >      nsolu, 
     >      j, 
     >      mle, 
     >      mue,
     >      iwork(3), 
     >      mbjac, 
     >      mbb, 
     >      mdiag, 
     >      mbdiag, 
     >      mdiff, 
     >      ierr, 
     >      mujacp, 
     >      md, 
     >      mm, 
     >      k, 
     >      j1, 
     >      lbeg, 
     >      lend,
     >      mujacj, 
     >      l, 
     >      result,
     >      newt ! current number of iterations for newton on this step
      real(dp) :: 
     >      sq6, 
     >      c1, 
     >      c2, 
     >      c1m1, 
     >      c2m1, 
     >      c1mc2, 
     >      dd1, 
     >      dd2, 
     >      dd3, 
     >      alph, 
     >      beta, 
     >      cno,
     >      u1, 
     >      t11, 
     >      t12, 
     >      t13, 
     >      t21, 
     >      t22, 
     >      t23, 
     >      t31, 
     >      t32, 
     >      t33,
     >      ti11, 
     >      ti12, 
     >      ti13, 
     >      ti21, 
     >      ti22, 
     >      ti23, 
     >      ti31, 
     >      ti32, 
     >      ti33,
     >      posneg, 
     >      hmaxn, 
     >      hold, 
     >      hopt, 
     >      faccon, 
     >      cfac, 
     >      xold, 
     >      xosol, 
     >      xsol, 
     >      hsol,
     >      hhfac, 
     >      ysafe, 
     >      delt, 
     >      fac1, 
     >      alphn, 
     >      betan, 
     >      xph, ! x+h, the end of the current step
     >      x0, ! starting value of x
     >      c1q, 
     >      c2q, 
     >      c3q,
     >      ak1, 
     >      ak2, 
     >      ak3, 
     >      z1i, 
     >      z2i, 
     >      z3i, 
     >      theta, 
     >      a1, 
     >      a2, 
     >      a3,
     >      dyno, 
     >      denom, 
     >      thq, 
     >      dynold, 
     >      thqold, 
     >      dyth, 
     >      qnewt, 
     >      f1i, 
     >      f2i, 
     >      f3i, 
     >      err,
     >      fac, 
     >      quot, 
     >      hnew, 
     >      facgus, 
     >      hacc, 
     >      erracc, 
     >      ak, 
     >      acont3, 
     >      qt
      logical :: 
     >      retry,
     >      use_current_jacobian,
     >      reject, ! true if reject the step
     >      first, ! true if this is the first step
     >      last, ! true if the current h takes us to xend
     >      caljac, ! true if we just calculated the jacobian at start of this step
     >      calhes, ! not used
     >      index1, 
     >      index2, 
     >      index3, 
     >      need_free_r, 
     >      need_free_c

      cont => rwork(1:4*n)
      
      call initializatons(ierr)
      if (ierr /= 0) then
         idid = -6
         return
      end if
      if (irtrn < 0) then ! solout says to return now (idid = 2)
         idid = 2
         return
      end if

      use_current_jacobian = .false.
      retry = .false.
      
      if (lout /= 0) write(lout,*)
      
      step_loop: do
      
         if (nstep >= nmax) then ! reached max steps
            if (lout /= 0) write(lout,'(a)') 'quit -- reached max steps'
            idid = -2
            exit step_loop
         end if

         if (retry) then ! retry with a smaller step size
            if (ierr /= 0) then
               idid = -4
               exit step_loop
            end if
            call prepare_for_retry
            if (caljac) then ! do 1 retry using just created jac
               use_current_jacobian = okay_to_reuse_jac
            else ! force new when retry with old
               use_current_jacobian = .false.
            end if
         end if
      
         if (.not. use_current_jacobian) then
            if (lout /= 0) write(lout,'(a)') 'compute new jacobian'
            call create_jacobian(ierr)
            if (ierr /= 0) then
               if (lout /= 0) write(lout,'(a,i3)') 'quit -- ierr /= 0 from create_jacobian'
               idid = -5
               exit step_loop
            end if
         else if (lout /= 0) then
            write(lout,'(a)') 'use previous jacobian'
         end if
      
         call free_prev_factors(ierr)
         if (ierr /= 0) then
            if (lout /= 0) write(lout,'(a,i3)') 'quit -- ierr /= 0 from free_prev_factors'
            idid = -7
            exit step_loop
         end if      
      
         ! need to factor even if use same jacobian
         ! since factors depend on time step too.
         call factor(ierr)
         if (ierr /= 0) then
            if (lout /= 0) write(lout,'(a,i3)') 'retry -- ierr /= 0 from factor'
            idid = -4
            exit step_loop
         end if
      
         newton_loop: do
            ! within this loop, size of timestep is constant, so can reuse factored matrix            
            nstep = nstep + 1
            if (nstep > nmax) then ! reached max steps
               if (lout /= 0) write(lout,'(a)') 'quit -- reached max steps'
               idid = -2
               exit step_loop
            end if
            if (lout /= 0) write(lout,'(5(a,i5,3x),2(a,1pe10.3,3x))') 
     >            '*nstep', nstep, 'naccpt', naccpt, 'nrejct', nrejct, 'njac', njac, 
     >            'ndec', ndec, 'h/(xend-x0)', h/(xend-x0), '(x-x0)/(xend-x0)', (x-x0)/(xend-x0)
            if (0.1d0*abs(h) <= abs(x)*uround) then ! step size too small.  give up.
               if (lout /= 0) write(lout,'(a,e10.3)') 'quit -- step size too small', h
               idid = -3
               exit step_loop
            end if
      
            result = do_newton_iterations(ierr)
            if (ierr /= 0) then
               idid = -4
               exit step_loop
            end if
            if (result /= newton_iterations_finished) then
               select case (result)
                  case (start_with_new_jacobian)
                     use_current_jacobian = .false.
                  case (start_with_old_jacobian)
                     use_current_jacobian = okay_to_reuse_jac
                  case (retry_with_smaller_step)
                  case default
                     stop 'bad value for newt_result in radau5'
               end select
               retry = .true.
               cycle step_loop
            end if

            result = accept_or_reject(ierr)
            if (ierr /= 0) then
               idid = -4
               exit step_loop
            end if
            if (irtrn < 0) then ! solout says to return now
               if (lout /= 0) write(lout,'(a,i3)') 'quit -- solout irtrn', irtrn
               idid = 2
               exit step_loop
            end if
            select case (result)
               case (start_with_new_jacobian)
                  use_current_jacobian = .false.
                  cycle step_loop
               case (start_with_old_jacobian)
                  use_current_jacobian = okay_to_reuse_jac
                  cycle step_loop
               case (keep_going_with_same_time_step)
                  cycle newton_loop
               case (finished)
                  if (lout /= 0) write(lout,'(a)') 'finished'
                  exit step_loop
               case default
                  stop 'bad result from accept_or_reject'
            end select
      
         end do newton_loop

      end do step_loop
      
      call free_prev_factors(ierr)
      
 979  format(' exit of radau5 at x = ',e18.4) 
      
      contains
      
      
      subroutine free_prev_factors(ierr)
         integer, intent(out) :: ierr
         integer :: ierr1
         ierr = 0; ierr1 = 0
         if (need_free_r) then
            call decsol_done(n,fjac,ldjac,fmas,ldmas,mlmas,mumas,
     &            m1,m2,n_minus_m1,fac1,e1,lde1,ip1,y0,ierr1,ijob,calhes,iphes,
     &            mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,decsol,decsols,
     &            sparse_jac,nzmax,isparse,ia,ja,sa,lrd,rpar_decsol,lid,ipar_decsol)
         end if
         if (need_free_c) then
            call decsolc_done(n,fjac,ldjac,fmas,ldmas,mlmas,mumas,
     &            m1,m2,n_minus_m1,alphn,betan,e2r,e2i,lde1,ip2,z1,z3,ierr,ijob,
     &            mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,
     &            decsolc,decsolcs,sparse_jac,nzmax,isparse,ia,ja,sar,sai,
     &            lcd,cpar_decsol,lrdc,rpar_decsolc,lidc,ipar_decsolc)
         end if
         if (ierr1 /= 0) ierr = ierr1
      end subroutine free_prev_factors
      
      
      subroutine prepare_for_newton(ierr)
         integer, intent(out) :: ierr
         if (index2) then
            do i = nind1 + 1,nind1 + nind2
               scal(i) = scal(i)/hhfac
            end do
         end if
         if (index3) then
            do i = nind1 + nind2 + 1,nind1 + nind2 + nind3
               scal(i) = scal(i)/(hhfac*hhfac)
            end do
         end if
         xph = x + h ! the end of this step
         if (first .or. startn) then
            z1(1:n) = 0
            z2(1:n) = 0
            z3(1:n) = 0
            f1(1:n) = 0
            f2(1:n) = 0
            f3(1:n) = 0
         else ! extrapolate using previous solutions
            c3q = h/hold
            c1q = c1*c3q
            c2q = c2*c3q
            do i = 1,n
               ak1 = cont(i + n)
               ak2 = cont(i + n2)
               ak3 = cont(i + n3)
               z1i = c1q*(ak1 + (c1q - c2m1)*(ak2 + (c1q - c1m1)*ak3))
               z2i = c2q*(ak1 + (c2q - c2m1)*(ak2 + (c2q - c1m1)*ak3))
               z3i = c3q*(ak1 + (c3q - c2m1)*(ak2 + (c3q - c1m1)*ak3))
               z1(i) = z1i
               z2(i) = z2i
               z3(i) = z3i
               ! f is A^-1*z
               f1(i) = ti11*z1i + ti12*z2i + ti13*z3i
               f2(i) = ti21*z1i + ti22*z2i + ti23*z3i
               f3(i) = ti31*z1i + ti32*z2i + ti33*z3i
            end do
         end if
         faccon = max(faccon,uround)**0.8d0
         theta = abs(thet)
         if (call_debug_routine) call debug('prepare_for_newton', 0, ierr)
      end subroutine prepare_for_newton
      
      
      subroutine factor(ierr)
         ! compute the matrices e1 and e2 and their decompositions
         integer, intent(out) :: ierr
         ierr = 0
         fac1 = u1/h
         alphn = alph/h
         betan = beta/h
         call decomr(n,fjac,ldjac,fmas,ldmas,mlmas,mumas,
     &            m1,m2,n_minus_m1,fac1,e1,lde1,ip1,y0,ierr,ijob,calhes,iphes,
     &            mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,decsol,decsols,
     &            sparse_jac,nzmax,isparse,ia,ja,sa,lrd,rpar_decsol,lid,ipar_decsol)
         if (ierr /= 0) return
         need_free_r = .true.
         call decomc(n,fjac,ldjac,fmas,ldmas,mlmas,mumas,
     &            m1,m2,n_minus_m1,alphn,betan,e2r,e2i,lde1,ip2,z1,z3,ierr,ijob,
     &            mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,
     &            decsolc,decsolcs,sparse_jac,nzmax,isparse,ia,ja,sar,sai,
     &            lcd,cpar_decsol,lrdc,rpar_decsolc,lidc,ipar_decsolc)
         if (ierr /= 0) return
         need_free_c = .true.
         ndec = ndec + 1 ! keep count of number of factorings
         if (call_debug_routine) call debug('factor', 0, ierr)
      end subroutine factor


      subroutine initializatons(ierr)
         integer, intent(out) :: ierr
         ierr = 0
         need_free_r = .false.
         need_free_c = .false.
         nn = n
         nn2 = 2*n
         nn3 = 3*n 
         lrc = 4*n
         !  --------  check the index of the problem  -----  
         index1 = nind1 /= 0
         index2 = nind2 /= 0
         index3 = nind3 /= 0
         !  -------  compute mass matrix for implicit case  ---------- 
         if (implct) call mas(n_minus_m1,fmas,ldmas,lrpar,rpar,lipar,ipar)
         !  ----------  constants  --------- 
         sq6 = dsqrt(6d0)
         c1 = (4d0 - sq6)/10d0
         c2 = (4d0 + sq6)/10d0
         c1m1 = c1 - 1d0
         c2m1 = c2 - 1d0
         c1mc2 = c1 - c2
         dd1 = -(13d0 + 7d0*sq6)/3d0
         dd2 = (-13d0 + 7d0*sq6)/3d0
         dd3 = -1d0/3d0
         u1 = (6d0 + 81d0**(1d0/3d0) - 9d0**(1d0/3d0))/30d0
         alph = (12d0 - 81d0**(1d0/3d0) + 9d0**(1d0/3d0))/60d0
         beta = (81d0**(1d0/3d0) + 9d0**(1d0/3d0))*dsqrt(3d0)/60d0
         cno = alph**2 + beta**2
         u1 = 1d0/u1
         alph = alph/cno
         beta = beta/cno
         t11 = 9.1232394870892942792d - 02
         t12 = -0.14125529502095420843d0
         t13 = -3.0029194105147424492d - 02
         t21 = 0.24171793270710701896d0
         t22 = 0.20412935229379993199d0
         t23 = 0.38294211275726193779d0
         t31 = 0.96604818261509293619d0
         ti11 = 4.3255798900631553510d0
         ti12 = 0.33919925181580986954d0
         ti13 = 0.54177053993587487119d0
         ti21 = -4.1787185915519047273d0
         ti22 = -0.32768282076106238708d0
         ti23 = 0.47662355450055045196d0
         ti31 = -0.50287263494578687595d0
         ti32 = 2.5719269498556054292d0
         ti33 = -0.59603920482822492497d0
         if (m1 > 0) ijob = ijob + 10
       
         x0 = x
         posneg = sign(1d0,xend - x)
         hmaxn = min(abs(hmax),abs(xend - x)) 
         if (abs(h) <= 10d0*uround) h = 1.0d - 6
         h = min(abs(h),hmaxn)
         h = sign(h,posneg)
         hold = h
         reject = .false.
         first = .true.
         last = .false.
         if ((x + h*1.0001d0 - xend)*posneg >= 0d0) then
            h = xend - x
            last = .true.
         end if
         hopt = h
         faccon = 1d0
         cfac = safe*(1 + 2*nit)
         ierr_cnt = 0
         xold = x
         irtrn = 1
         if (iout /= 0) then
            call do_solout
            if (irtrn < 0) return
         end if
         mle = mljac
         mue = mujac
         mbjac = mljac + mujac + 1
         mbb = mlmas + mumas + 1
         mdiag = mle + mue + 1
         mdiff = mle + mue - mumas
         mbdiag = mumas + 1
         n2 = 2*n
         n3 = 3*n
         if (itol == 0) then
             do i = 1,n
                scal(i) = atol(1) + rtol(1)*abs(y(i))
             end do
         else
             do i = 1,n
                scal(i) = atol(i) + rtol(i)*abs(y(i))
             end do
         end if
         hhfac = h
         ierr = 0
         if (ijac == 0) then ! initial call for numerical differencing to create jacobian
            call fcn(n,x,y,y0,lrpar,rpar,lipar,ipar,ierr)
            if (ierr /= 0) return
            nfcn = nfcn + 1
         end if
         if (call_debug_routine) call debug('initializatons', 0, ierr)
      end subroutine initializatons

      
      subroutine create_jacobian(ierr)
         integer, intent(out) :: ierr
         ierr = 0
         njac = njac + 1
         if (ijac == 0) then ! compute jacobian matrix numerically
            call create_numerical_jacobian(ierr)
            if (ierr /= 0) return
         else ! compute jacobian matrix analytically
            ! change from Hairer -- use x+h instead of x
            ! so jac has time at end of this step instead of beginning
            if (nzmax == 0) then
               call jac(n,x+h,y,y0,fjac,ldjac,lrpar,rpar,lipar,ipar,ierr)
            else
               call sjac(n,x+h,y,y0,nzmax,ia,ja,sparse_jac,lrpar,rpar,lipar,ipar,ierr)
            end if
            if (ierr /= 0) return
            if (call_debug_routine) then
               call debug('create_jacobian', 0, ierr)
               if (ierr /= 0) return
            end if
         end if
         caljac = .true.
         calhes = .true.
      end subroutine create_jacobian
            
      
      integer function do_newton_iterations(ierr) result(result)
         integer, intent(out) :: ierr
         character (len = 32) :: msg
         result = in_progress
         call prepare_for_newton(ierr)
         if (ierr /= 0) then
            if (lout /= 0) write(lout,'(10x,i3,3x,a)') newt, 'ierr from prepare_for_newton'
            return
         end if
         newt = 0
         do while (newt < nit)
            newt = newt + 1
            call get_rhs(ierr)
            if (ierr /= 0) then
               if (lout /= 0) write(lout,'(10x,i3,3x,a)') newt, 'ierr from get_rhs'
               return
            end if
            call solve(ierr)
            if (ierr /= 0) then
               if (lout /= 0) write(lout,'(10x,i3,3x,a)') newt, 'ierr from solve'
               return
            end if
            call check_convergence(result, ierr)
            if (ierr /= 0) then
               if (lout /= 0) write(lout,'(10x,i3,3x,a)') newt, 'ierr from check_convergence'
               return
            end if
            if (result /= in_progress) then ! give up
               if (lout /= 0) write(lout,'(10x,i3,3x,a)') 
     >               newt, 'retry -- poor convergence'
               return
            end if
            call update_solution(ierr)
            if (ierr /= 0) then
               if (lout /= 0) write(lout,'(10x,i3,3x,a)') newt, 'ierr from update_solution'
               return
            end if
            if (faccon*dyno <= fnewt) then
               if (lout /= 0) write(*,'(14x,a,i3,3x,4(a,1pe10.3,3x),a)') 
     >               'newt', newt,'theta', theta, 'faccon', faccon, 'rms dx/scale', dyno,
     >               'convergence', faccon*dyno/fnewt, 'done'
               exit ! converged before reaching max number of iterations
            end if
            if (lout /= 0) write(lout,'(14x,a,i3,3x,4(a,1pe10.3,3x))') 
     >               'newt', newt, 'theta', theta, 'faccon', faccon, 'rms dx/scale', dyno,
     >               'convergence', faccon*dyno/fnewt
         end do
         result = newton_iterations_finished
      end function do_newton_iterations
      
      
      subroutine check_convergence(result, ierr)
         integer, intent(inout) :: result, ierr
         real(dp) :: dyno1, dyno2, dyno3
         ierr = 0
         dyno1 = 0; dyno2 = 0; dyno3 = 0
         do i = 1,n
            denom = scal(i)
            dyno1 = dyno1 + (z1(i)/denom)**2
            dyno2 = dyno2 + (z2(i)/denom)**2
            dyno3 = dyno3 + (z3(i)/denom)**2
         end do
         dyno = dsqrt((dyno1+dyno2+dyno3)/n3) ! rms of relative change to solution
         if (lout /= 0) write(lout,'(a,3x,i3,9e18.8)') 'dyno', newt, dyno, dyno1, dyno2, dyno3
         if (newt <= 1 .or. newt >= nit) then
            return
         end if
         ! check rate of convergence
         thq = dyno/dynold
         if (newt == 2) then
            theta = thq
         else
            theta = sqrt(thq*thqold)
         end if
         thqold = thq
         if (call_debug_routine) then
            call debug('check_convergence', 0, ierr)
            if (ierr /= 0) return
         end if
         if (theta >= 0.99d0) then
            if (lout /= 0) 
     >         write(lout,'(a,3x,i3,9e18.8)') 'check_convergence: theta >= 0.99d0', 
     >           newt, theta, dyno, dyno1, dyno2, dyno3
            !stop 'check_convergence'
            result = retry_with_smaller_step
            return
         end if
         faccon = theta/(1d0 - theta)
         dyth = faccon*dyno*theta**(nit - 1 - newt)/fnewt
         if (dyth < 1) return ! convergence rate is okay
         qnewt = dmax1(1.0d - 4,dmin1(20d0,dyth))
         hhfac = .8d0*qnewt**(-1d0/(4d0 + nit - 1 - newt))
         h = hhfac*h
         reject = .true.
         last = .false.
         if (caljac) then
            result = start_with_old_jacobian
         else
            result = start_with_new_jacobian
         end if
      end subroutine check_convergence
      
      
      subroutine update_solution(ierr)
         integer, intent(out) :: ierr
         dynold = max(dyno,uround)
         ! increment f = T^-1*y
         ! multiply z by A to return to the untransformed variables
         do i = 1,n
            f1i = f1(i) + z1(i)
            f2i = f2(i) + z2(i)
            f3i = f3(i) + z3(i)
            f1(i) = f1i
            f2(i) = f2i
            f3(i) = f3i
            z1(i) = t11*f1i + t12*f2i + t13*f3i
            z2(i) = t21*f1i + t22*f2i + t23*f3i
            z3(i) = t31*f1i +     f2i
         end do
         if (call_debug_routine) call debug('update_solution', 0, ierr)
      end subroutine update_solution


      subroutine solve(ierr)
         integer, intent(out) :: ierr
         ierr = 0
         ! multiply rhs by T^-1 before solve
         do i = 1,n
           a1 = z1(i)
           a2 = z2(i)
           a3 = z3(i)
           z1(i) = ti11*a1 + ti12*a2 + ti13*a3
           z2(i) = ti21*a1 + ti22*a2 + ti23*a3
           z3(i) = ti31*a1 + ti32*a2 + ti33*a3
         end do
         call slvrad(n,fjac,ldjac,mljac,mujac,fmas,ldmas,mlmas,mumas,
     &          m1,m2,n_minus_m1,fac1,alphn,betan,e1,e2r,e2i,lde1,z1,z2,z3,
     &          f1,f2,f3,cont,ip1,ip2,iphes,ierr,ijob,
     &          mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,
     &          decsol,decsols,decsolc,decsolcs,nzmax,isparse,ia,ja,sa,sar,sai,
     &          lcd,cpar_decsol,lrdc,rpar_decsolc,lidc,ipar_decsolc)
         if (ierr /= 0 .and. lout /= 0) write(lout,'(a,/)') 'ierr from slvrad'
         nsol = nsol + 1
         if (call_debug_routine) call debug('solve', 0, ierr)
      end subroutine solve
      
      
      subroutine get_rhs(ierr) ! store rhs in (z1,z2,z3)
         integer, intent(out) :: ierr
         do i = 1,n
            cont(i) = y(i) + z1(i)
         end do
         ierr = 0
         call fcn(n,x + c1*h,cont,z1,lrpar,rpar,lipar,ipar,ierr) 
         if (ierr /= 0) return         
         do i = 1,n
            cont(i) = y(i) + z2(i)
         end do
         call fcn(n,x + c2*h,cont,z2,lrpar,rpar,lipar,ipar,ierr)            
         if (ierr /= 0) return         
         do i = 1,n
            cont(i) = y(i) + z3(i)
         end do
         call fcn(n,xph,cont,z3,lrpar,rpar,lipar,ipar,ierr)
         if (ierr /= 0) return         
         nfcn = nfcn + 3
         if (call_debug_routine) call debug('get_rhs', 0, ierr)
      end subroutine get_rhs
      
      
      integer function accept_or_reject(ierr) result(result)
         integer, intent(out) :: ierr
         ierr = 0
         ! estimate error
         call estrad(n,nerror,fjac,ldjac,mljac,mujac,fmas,ldmas,mlmas,mumas,
     &          h,dd1,dd2,dd3,fcn,nfcn,y0,y,ijob,x,m1,m2,n_minus_m1,
     &          e1,lde1,z1,z2,z3,cont,f1,f2,ip1,iphes,scal,err,
     &          first,reject,fac1,lrpar,rpar,lipar,ipar,ierr,
     &          mle,mue,mbjac,mbb,mdiag,mdiff,mbdiag,
     &          decsol,decsols,nzmax,isparse,ia,ja,sa,
     &          lrd,rpar_decsol,lid,ipar_decsol)
         if (ierr /= 0) then
            if (lout /= 0) write(lout,'(a,/)') 'ierr from estrad'
            return
         end if
         fac = min(safe,cfac/(newt + 2*nit))
         quot = max(facr,min(facl,err**.25d0/fac))
         hnew = h/quot
         if (err < 1d0) then ! step is accepted
            if (lout /= 0) write(lout,'(5(a,i5,3x),2(a,1pe10.3,3x)/)') 
     >            'accept', nstep, 'naccpt', naccpt, 'nrejct', nrejct, 'njac', njac, 
     >            'ndec', ndec, 'hnew/h', hnew/h, 'err', err
            result = do_step_is_accepted(ierr)
            if (call_debug_routine) call debug('do_step_is_accepted', 0, ierr)
         else
            if (lout /= 0) write(lout,'(5(a,i5,3x),2(a,1pe10.3,3x),/)') 
     >            'reject', nstep, 'naccpt', naccpt, 'nrejct', nrejct, 'njac', njac, 
     >            'ndec', ndec, 'hnew/h', hnew/h, 'err', err
            result = do_step_is_rejected(ierr)
            if (call_debug_routine) call debug('do_step_is_rejected', 0, ierr)
         end if
      end function accept_or_reject

      
      integer function do_step_is_rejected(ierr) result(result)
         integer, intent(out) :: ierr
         ierr = 0
         reject = .true.
         last = .false.
         if (first) then
             h = h*0.1d0
             hhfac = 0.1d0
         else 
             hhfac = hnew/h
             h = hnew
         end if
         if (naccpt > 0) nrejct = nrejct + 1
         if (caljac) then
            result = start_with_old_jacobian
         else
            result = start_with_new_jacobian
         end if
      end function do_step_is_rejected
      
      
      integer function do_step_is_accepted(ierr) result(result)
         integer, intent(out) :: ierr
         ierr = 0
         first = .false.
         naccpt = naccpt + 1
         if (pred) call predictive_controller
         call prepare_for_next_step
         ! note: for mesa, I've moved the following call to fcn up a few lines 
         ! so when we call solout or return from radau5, we have always just 
         ! evaluated the fcn for the final y at the end of the step.
         call fcn(n,x,y,y0,lrpar,rpar,lipar,ipar,ierr)
         if (ierr /= 0) return
         nfcn = nfcn + 1
         if (iout /= 0) then
            call do_solout
            if (irtrn < 0) then
               result = finished
               return
            end if
         end if
         if (last) then
            h = hnew
            idid = 1
            result = finished
            return
         end if
         caljac = .false.
         hnew = posneg*min(abs(hnew),hmaxn)
         hopt = min(h,hnew)
         if (reject) hnew = posneg*min(abs(hnew),abs(h)) 
         reject = .false.
         if ((x + hnew/quot1 - xend)*posneg >= 0d0) then
            h = xend - x
            last = .true.
         else
            qt = hnew/h 
            hhfac = h
            if (theta <= thet .and. qt >= quot1 .and. qt <= quot2) then
               result = keep_going_with_same_time_step
               return
            end if
            h = hnew 
         end if
         hhfac = h
         if (theta <= thet) then
            result = start_with_old_jacobian
         else
            result = start_with_new_jacobian
         end if
      end function do_step_is_accepted
      
      
      subroutine prepare_for_next_step
         xold = x
         hold = h
         x = xph 
         do i = 1,n
            y(i) = y(i) + z3(i)  
            z2i = z2(i)
            z1i = z1(i)
            cont(i+n) = (z2i - z3(i))/c2m1
            ak = (z1i - z2i)/c1mc2
            acont3 = z1i/c1
            acont3 = (ak - acont3)/c2
            cont(i+n2) = (ak - cont(i+n))/c1m1
            cont(i+n3) = cont(i+n2) - acont3
         end do
         if (itol == 0) then
            do i = 1,n
               scal(i) = atol(1) + rtol(1)*abs(y(i))
            end do
         else
            do i = 1,n
               scal(i) = atol(i) + rtol(i)*abs(y(i))
            end do
         end if
      end subroutine prepare_for_next_step


      subroutine predictive_controller ! of gustafsson
         if (naccpt > 1) then
            facgus = (hacc/h)*(err**2/erracc)**0.25d0/safe
            facgus = max(facr,min(facl,facgus))
            quot = max(quot,facgus)
            hnew = h/quot
         end if
         hacc = h
         erracc = max(1.0d0 - 2,err)
      end subroutine predictive_controller
      
      
      subroutine do_solout
         nrsol = naccpt + 1
         xsol = x
         xosol = xold
         do i = 1,n
            cont(i) = y(i)
         end do
         nsolu = n
         hsol = hold
         iwork(1) = nn
         iwork(2) = nn2
         iwork(3) = nn3
         j = 1 + lrc
         rwork(j) = xsol; j = j+1
         rwork(j) = hsol; j = j+1
         rwork(j) = c2m1; j = j+1
         rwork(j) = c1m1; j = j+1
         call solout(nrsol,xosol,xsol,nsolu,y,rwork,iwork,contr5,lrpar,rpar,lipar,ipar,irtrn)
      end subroutine do_solout
      
      
      subroutine prepare_for_retry
         ierr = 0
         h = h*0.5d0 
         hhfac = 0.5d0
         reject = .true.
         last = .false.
         retry = .false.
      end subroutine prepare_for_retry
      
      
      subroutine create_numerical_jacobian(ierr)
         integer, intent(out) :: ierr
         ierr = 0
         if (banded) then
            mujacp = mujac + 1
            md = min(mbjac,m2)
            do mm = 1,m1/m2 + 1
               do k = 1,md
                  j = k + (mm - 1)*m2
                  do
                     f1(j) = y(j)
                     f2(j) = dsqrt(uround*max(1.d - 5,abs(y(j))))
                     y(j) = y(j) + f2(j)
                     j = j + md
                     if (j > mm*m2) exit
                  end do                  
                  call fcn(n,x,y,cont,lrpar,rpar,lipar,ipar,ierr)
                  if (ierr /= 0) return ! retry with a smaller step size
                  j = k + (mm - 1)*m2
                  j1 = k
                  lbeg = max(1,j1 - mujac) + m1                  
                  do
                     lend = min(m2,j1 + mljac) + m1
                     y(j) = f1(j)
                     mujacj = mujacp - j1 - m1
                     do l = lbeg,lend
                        fjac(l + mujacj,j) = (cont(l) - y0(l))/f2(j) 
                     end do
                     j = j + md
                     j1 = j1 + md
                     lbeg = lend + 1
                     if (j > mm*m2) exit
                  end do                  
               end do
            end do
         else ! jacobian is full
            do i = 1,n
               ysafe = y(i)
               delt = dsqrt(uround*max(1.d - 5,abs(ysafe)))
               y(i) = ysafe + delt
               ierr = 0
               call fcn(n,x,y,cont,lrpar,rpar,lipar,ipar,ierr)
               if (ierr /= 0) return ! retry with a smaller step size
               do j = m1 + 1,n
                 fjac(j - m1,i) = (cont(j) - y0(j))/delt
               end do
               y(i) = ysafe
            end do
         end if
         if (call_debug_routine) call debug('create_numerical_jacobian', 0, ierr)
      end subroutine create_numerical_jacobian
      
      
      subroutine debug(message, info, ierr)
         character (len=*), intent(in) :: message
         integer, intent(in) :: info
         integer, intent(out) :: ierr
         call radau5_debug_routine(
     >      message, info, n, x0, x, y, xend, hmax, h, 
     >      rtol, atol, itol, okay_to_reuse_jac, 
     >      ijac, nzmax, isparse, mljac, mujac, mlmas, mumas, 
     >      ia, ja, sparse_jac, sa, sar, sai, 
     >      nmax, uround, safe, thet, fnewt, quot1, quot2, nit, ijob, startn, 
     >      nind1, nind2, nind3, pred, facl, facr, m1, m2, n_minus_m1, nerror, 
     >      implct, banded, ldjac, lde1, ldmas, z1, z2, z3, 
     >      y0, scal, f1, f2, f3, fjac, e1, e2r, e2i, fmas, ip1, ip2, 
     >      rwork, nfcn, njac, nstep, naccpt, nrejct, ndec, nsol, newt, ierr_cnt,
     >      caljac, first, last, err, theta, faccon, dyno,
     >      lrpar, rpar, lipar, ipar, lout, ierr)
      end subroutine debug
      

      end subroutine radcor
!
! ***********************************************************
!
      real(dp) function contr5(i,x,rwork,iwork,ierr) 

!     this function can be used for coninuous output. it provides an
!     approximation to the i - th component of the solution at x.
!     it gives the value of the collocation polynomial, defined for
!     the last successfully computed step (by radau5).
      implicit none
      integer, intent(in) :: i
      real(dp), intent(in) :: x
      real(dp), intent(inout), target :: rwork(*)
      integer, intent(inout), target :: iwork(*)
      integer, intent(out) :: ierr
      
      !dimension cont(lrc),ns(lns),vs(lvs)
!      common /conra5/nn,nn2,nn3,nn4,xsol,hsol,c2m1,c1m1
      real(dp) :: xsol, hsol, c2m1, c1m1, s
      real(dp), pointer :: cont(:)
      integer :: nn, nn2, nn3, lrc, j
      ierr = 0
      nn = iwork(1)
      nn2 = iwork(2)
      nn3 = iwork(3)
      lrc = 4*nn

      cont  = > rwork(1:lrc); j = 1 + lrc
      xsol = rwork(j); j = j+1
      hsol = rwork(j); j = j+1
      c2m1 = rwork(j); j = j+1
      c1m1 = rwork(j); j = j+1

      s = (x - xsol)/hsol
      contr5 = cont(i) + s*(cont(i + nn) + (s - c2m1)*(cont(i + nn2) + (s - c1m1)*cont(i + nn3)))
      return
      end function contr5

! ***********************************************************      


      end module mod_radau5
