! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module mebdfi_support
      use const_def, only: dp
      
		implicit none

		contains
		
		
		subroutine matrix_solver(mf, equil, equed_flag, mbnd, 
     >			pw, pw1, pw_R, pw_C, ipiv, n, x, rcond, recip_pivot_growth_factor, 
     >         verbosity, lout, info)
			integer, intent(in) :: n, mf, equil, verbosity, lout, mbnd(4)
			integer, intent(out) :: info
			integer, intent(inout) :: equed_flag, ipiv(n)
			
			real(dp), intent(inout) :: pw(*), pw1(*), pw_R(n), pw_C(n), x(n), 
     >			rcond, recip_pivot_growth_factor
			character (len=1) :: equed_char, fact_char
			
			real(dp) :: b(n), ferr(1), berr(1), work(4*n)
			integer :: iwork(n)

			info = 0
			if (equil /= 0) then ! do equilibration
				select case (equed_flag)
					case (0) ! first time
						fact_char = 'E'
					case (1)
						fact_char = 'F'
						equed_char = 'N'
					case (2)
						fact_char = 'F'
						equed_char = 'R'
					case (3)
						fact_char = 'F'
						equed_char = 'C'
					case default
						fact_char = 'F'
						equed_char = 'B'
				end select
				b(1:n) = x(1:n)
				
				if (mf >= 23) then ! banded matrix
     				call DGBSVX(fact_char, 'No transpose', n, mbnd(1), mbnd(2), 
     >						1, pw1, mbnd(3), pw, mbnd(4), 
     >						ipiv, equed_char, pw_R, pw_C, b, n, X, n,
     >						rcond, ferr, berr, work, iwork, info )
					if (verbosity /= 0 .and. lout > 0) then
						write(lout,*) 'DGBSVX rcond', rcond, 
     >						'  ferr', ferr, '   berr', berr, 
     >                  '  equed_char ', equed_char,
     >                  '  recip_pivot_growth_factor ', work(1)
						if (info /= 0) write(lout,*) 'DGBSVX failed: ', info
					end if
				else
      			call DGESVX(fact_char, 'No transpose', n, 1, pw1, n, pw, n, ipiv,
     >						equed_char, pw_R, pw_C, b, n, X, n, 
     >						rcond, ferr, berr, work, iwork, info )
					if (verbosity /= 0 .and. lout > 0) then
						write(lout,*) 'DGESVX rcond', rcond,
     >						'  ferr', ferr, '   berr', berr, 
     >                  '  equed_char ', equed_char,
     >                  '  recip_pivot_growth_factor ', work(1)
						if (info /= 0) write(lout,*) 'DGESVX failed: ', info
					end if
				end if

				if (rcond == 0) then ! singular matrix
					info = -1001
				else if (equed_char == 'N') then
					equed_flag = 1
				else if (equed_char == 'R') then
					equed_flag = 2
				else if (equed_char == 'C') then
					equed_flag = 3
				else if (equed_char == 'B') then
					equed_flag = 4
				else
					info = -999
				end if
			else if (mf >= 23) then ! banded matrix
      		call DGBTRS( 'No transpose', n, mbnd(1), mbnd(2), 1, pw, 
     >				mbnd(4), ipiv, x, n, info )
				if (info /= 0 .and. verbosity > 0 .and. lout > 0) then
					write(lout,*) 'DGBTRS failed: ', info
				end if
	      else ! square matrix
     			call DGETRS( 'No transpose', n, 1, pw, n, ipiv, x, n, info )
				if (info /= 0 .and. verbosity > 0 .and. lout > 0) then
					write(lout,*) 'DGETRS failed: ', info
				end if
	      end if
	      
	      recip_pivot_growth_factor = work(1)
			
		end subroutine matrix_solver
	

      subroutine itrat2(qqq,y,yprime,n,t,h,con,errbnd,arh,crate,tcrate,
     +     m,worked,ymax,error,save1,save2,scale,pw,pw1,pw_R,pw_C,mf,mbnd,nind1,
     +     nind2,nind3,ipiv,min_iter,itol,num_tol,rtol,atol,inf_nrms,
     >     num_rpar,rpar,num_ipar,ipar,hused,nbsol,
     +     nre,nqused,nstep,equil,equed_flag,rcond,
     >     pderv,resid,call_back,monitor,stage,verbosity,lout,ierr)

			integer, intent(in) :: n, mbnd(4), nind1, nind2, nind3, mf, itol, 
     >			nqused, nstep, num_tol, num_ipar, num_rpar, verbosity, lout, stage, min_iter
			integer, intent(inout) :: ipar(num_ipar),equil,equed_flag
	      integer, intent(out) :: ipiv(n),ierr,nbsol,nre
	      integer, intent(out) :: m ! count of number of extra iterations done

	      real(dp), intent(in) :: hused,qqq,t,h,con,errbnd,rtol(num_tol),atol(num_tol),inf_nrms(4)
	      real(dp), intent(inout) :: arh(n),rpar(num_rpar),ymax(n),y(n,12),rcond
	      real(dp), intent(out) :: error(n),pw(*),pw1(*),pw_R(n),pw_C(n),save1(n),save2(n),
     +     scale(n),yprime(n),tcrate,crate

	      logical, intent(out) :: worked
			
			interface
			
				include "mebdfi_call_back.dek"
				include "mebdfi_resid.dek"
				include "mebdfi_pderv.dek"
				include "mebdfi_monitor.dek"
			
			end interface

			real(dp) :: d, d1, norminf
			real(dp), parameter :: zero = 0d0
			
			call set_scale(y, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)
			
			m = 0
	      save1(1:n) = y(1:n,1)
	      error(1:n) = zero
			worked = .true.
			d1 = 0d0

	      newton_iteration: do
	
				call do_one ! set d to weighted residual norm
				if (.not. worked) return
				
				if (m+1 >= min_iter) then ! see if it has converged
			      if ((d*dmin1(1d0,2d0*crate)) < errbnd/dble(nqused)) then
   		         ! check infinity norm of last correction before accept as converged
         			norminf = maxval(abs(save2(1:n)/scale(1:n)))
         	      if (norminf < inf_nrms(stage)) then
		               worked = .true.
   					   return
   					end if
					   if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				   'newton_iteration norminf too large', norminf
		         end if
	         end if
	
		      if (m > 0 .and. d > d1) then ! not converging, so quit
	            worked = .false.
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'norm of corrections larger than for previous iteration'
	            return
				end if
	
		      if (m == 4) then ! have run out of chances
		         worked = .false.
					if (verbosity > 0 .and. lout > 0) write(lout,*)
     >					'itrat2 failed to converge in 4 tries'
		         return         
		      end if 
		
		      d1 = d ! d1 holds the previous residual norm for comparison
		      m = m + 1
		      yprime(1:n) = (save1(1:n)-arh(1:n))/qqq
		
			end do newton_iteration
			
			contains
				
			subroutine do_one
				integer :: info, i
				real(dp) :: rcond, recip_pivot_growth_factor
					! estimate of the reciprocal condition number of the matrix
		
		      call resid(n,t,h,con,save1,save2,yprime,num_rpar,rpar,num_ipar,ipar,stage,m+1,ierr)
		      nre = nre + 1
		      if (ierr /= 0) then
	            worked = .false.
					if (verbosity > 0 .and. lout > 0) then
						write(lout,*) 'itrat2 failed because of nonzero resid ierr: ', ierr
					end if
	            return
	         end if

				info = 0
				call matrix_solver(mf, equil, equed_flag, mbnd, 
     >				pw, pw1, pw_R, pw_C, ipiv, n, save2, rcond, recip_pivot_growth_factor,
     >            verbosity, lout, info)
		      if (info < 0) then
	            worked = .false.
	            return
	         end if
		      nbsol = nbsol + 1
		
		      d = sum((save2(1:n)/(scale(1:n)))**2)
		      if (itol == 1) d = d/(rtol(1)**2)
		      if (m /= 0  .and.  d1 /= zero) crate = dmax1(0.9d0*crate,d/d1)		
		      tcrate = tcrate + crate

		      error(1:n) = error(1:n) - save2(1:n)
		      save1(1:n) = y(1:n,1) + error(1:n)
		
				call monitor(
     >				n,t,h,con,y,save1,error,save2,yprime,scale,d,crate,errbnd/dble(nqused),
     >            pw_R,pw_C,ipiv,rcond,recip_pivot_growth_factor,
     >				num_rpar,rpar,num_ipar,ipar,nqused,nstep,nbsol,stage,m+1,ierr)
		      if (ierr /= 0) then
	            worked = .false.
					if (verbosity > 0 .and. lout > 0) then
						write(lout,*) 'itrat2 failed because of nonzero monitor ierr: ', ierr
					end if
	            return
	         end if
		
			end subroutine do_one

      end subroutine itrat2


      subroutine interp(n,nqused,h,t,y,tout,y0)
	      integer, intent(in) :: n
	      integer, intent(in) :: nqused ! last order used
	      real(dp), intent(in) :: h,t,tout,y(n,12)
	      real(dp), intent(out) :: y0(n)

	      integer :: j,l
			real(dp) :: s, s1

	      y0(1:n) = y(1:n,1)
	
	      l = nqused + 2
	      s = (tout-t)/h
	      s1 = 1.0d0
	      do j = 2,l
	         s1 = s1 * (s+dble(j-2))/dble(j-1)
	         y0(1:n) = y0(1:n) + s1*y(1:n,j)
	      end do
	      
      end subroutine interp


      subroutine coset(nq,el,elst,tq,ncoset,maxord)
	  		integer, intent(in) :: nq	! current order
			integer, intent(inout) :: ncoset ! number of times the order has been changed
			integer, intent(inout) :: maxord	! the maximum order used so far in the integration
			real(dp), intent(out) :: el(10)  ! nq+1 coefficients for BDF
			real(dp), intent(out) :: elst(10)  ! nq+2 coefficients for EBDF
			real(dp), intent(out) :: tq(5)  ! coefficients for truncation error estimates
			! ------------------------------------------------------------------
			! coset is called by the integrator and sets the coefficients used
			! by the conventional backward differentiation scheme and the
			! modified extended backward differentiation scheme.  the vector
			! el of length nq+1 determines the basic bdf method while the vector
			! elst of length nq+2 determines the mebdf.  the vector tq of
			! length 4 is involved in adjusting the stepsize in relation to the
			! truncation error.  its values are given by the pertst array.  the
			! vectors el and tq both depend on meth and nq.  the
			! coefficients in pertst need to be given to only about one percent
			! accuracy.  the order in which the groups appear below is:
			! coefficients for order nq-1, coefficients for order nq,
			! coefficients for order nq+1.
			! -----------------------------------------------------------------
      	real(dp) :: pertst(8,3)
      	data  pertst(1,1)/1./,pertst(2,1)/2./,pertst(3,1)/4.5/,
     +      pertst(4,1)/7.333/,pertst(5,1)/10.42/,pertst(6,1)/13.7/,
     +      pertst(7,1)/17.15/,pertst(8,1)/20.74/
      	data  pertst(1,2)/2./,pertst(2,2)/4.5/,pertst(3,2)/7.333/,
     +      pertst(4,2)/10.42/,pertst(5,2)/13.7/,pertst(6,2)/17.15/,
     +      pertst(7,2)/20.74/,pertst(8,2)/24.46/
      	data  pertst(1,3)/4.5/,pertst(2,3)/7.333/,pertst(3,3)/10.42/,
     +      pertst(4,3)/13.7/,pertst(5,3)/17.15/,pertst(6,3)/20.74/,
     +      pertst(7,3)/24.46/,pertst(8,3)/1./

	      if (nq > maxord) maxord = nq
	      ncoset = ncoset + 1
			! -------------------------------------------------------------------
			! the following coefficients should be defined to machine accuracy.
			! their derivation is given in reference 1.
			! -------------------------------------------------------------------
			select case (nq)
			case (2)
		      el(1) = 6.6666666666667d-01
		      el(3) = 3.3333333333333d-01
		      elst(1) = 9.5652173913043d-01
		      elst(3) = 2.1739130434782d-01
		      elst(4) = -1.7391304347826d-01
			case (3)
		      el(1) = 5.4545454545455d-01
		      el(3) = 4.5454545454545d-01
		      el(4) = 1.8181818181818d-01
		      elst(1) = 7.6142131979695d-01
		      elst(3) = 3.2994923857868d-01
		      elst(4) = 8.6294416243654d-02
		      elst(5) = -9.1370558375634d-02
			case (4)
		      el(1) = 0.48d0
		      el(3) = 0.52d0
		      el(4) = 0.28d0
		      el(5) = 0.12d0
		      elst(1) = 6.5733706517393d-01
		      elst(3) = 4.0023990403838d-01
		      elst(4) = 1.5793682526989d-01
		      elst(5) = 4.4382247101159d-02
		      elst(6) = -5.7576969212315d-02
			case (5)
		      el(1) = 4.3795620437956d-01
		      el(3) = 5.62043795620436d-01
		      el(4) = 3.43065693430656d-01
		      el(5) = 1.97080291970802d-01
		      el(6) = 8.75912408759123d-02
		      elst(1) = 5.9119243917152d-01
		      elst(3) = 4.4902473356122d-01
		      elst(4) = 2.1375427307460d-01
		      elst(5) = 9.0421610027481503d-02
		      elst(6) = 2.6409276761177d-02
		      elst(7) = -4.0217172732757d-02
			case (6)
		      el(1) = 4.08163265306120d-01
		      el(3) = 5.91836734693874d-01
		      el(4) = 3.87755102040813d-01
		      el(5) = 2.51700680272107d-01
		      el(6) = 1.49659863945577d-01
		      el(7) = 6.80272108843534d-02
		      elst(1) = 5.4475876041119d-01
		      elst(3) = 4.8525549636077d-01
		      elst(4) = 2.5789750131312d-01
		      elst(5) = 1.3133738525800d-01
		      elst(6) = 5.7677396763462d-02
		      elst(7) = 1.7258197643881d-02
		      elst(8) = -3.0014256771967d-02
			case (7)
		      el(1) = 3.85674931129476d-01
		      el(3) = 6.14325068870521d-01
		      el(4) = 4.21487603305783d-01
		      el(5) = 2.9292929292929d-01
		      el(6) = 1.96510560146923d-01
		      el(7) = 1.19375573921028d-01
		      el(8) = 5.50964187327820d-02
		      elst(1) = 5.0999746293734d-01
		      elst(3) = 5.1345839935281d-01
		      elst(4) = 2.9364346131937d-01
		      elst(5) = 1.6664672120553d-01
		      elst(6) = 8.8013735242353d-02
		      elst(7) = 3.9571794884069d-02
		      elst(8) = 1.2039080338722d-02
		      elst(9) = -2.3455862290154d-02
			case default
		      el(1) = 1d0
		      elst(1) = 1.5d0
		      elst(3) = -0.5d0
			end select
		
	      tq(1:3) = pertst(nq,1:3)
	      tq(4) = 0.5d0*tq(2)/dble(nq)
	      if (nq /= 1) tq(5) = pertst(nq-1,1)

      end subroutine coset


      subroutine pset(y,yprime,n,h,t,uround,epsjac,con,miter,mbnd,
     >		nind1,nind2,nind3,ier,pderv,resid,call_back,
     >		ymax,save1,save2,save3,pw,pw1,pw_R,pw_C,equil,equed_flag,
     >		wrkspc,ipiv,itol,num_tol,rtol,atol,npset,nje,nre,
     >		ndec,num_rpar,rpar,num_ipar,ipar,jdebug,verbosity,lout,ierr)   

	      real(dp), intent(in) :: h,t,uround,epsjac,con,rtol(num_tol),
     >			atol(num_tol)
	      real(dp), intent(inout) :: rpar(num_rpar),y(n,12)
      	real(dp), intent(out) :: save1(n),save2(n),save3(n),
     >			pw(*),pw1(*),
     >			yprime(n),wrkspc(n),ymax(n),pw_R(n),pw_C(n)

	      integer, intent(in) :: miter, nind1, nind2, nind3, n, itol, 
     >			mbnd(4), jdebug, num_tol, num_ipar, num_rpar, verbosity, lout
	      integer, intent(inout) :: ipar(num_ipar), equil, equed_flag
	      integer, intent(out) :: ndec,nre,nje,npset,ier,ierr,ipiv(n)
			
			interface
			
				include "mebdfi_call_back.dek"
				include "mebdfi_resid.dek"
				include "mebdfi_pderv.dek"
			
			end interface
		
			! -----------------------------------------------------------------
			! pset is called by stiff to compute and process the matrix
			! pd=dg/dy + (1/con)dg/dy'. this matrix is then subjected to lu
			! decomposition in preparation for later solution of linear systems
			! of algebraic equations with lu as the coefficient matrix.  the
			! matrix pd is found by the user-supplied routine pderv if miter=1
			! or 3 or by finite differencing if miter = 2 or 4.
			! in addition to variables described previously, communication with
			! pset uses the following ..
			! epsjac = dsqrt(uround), used in numerical jacobian increments.
			! -----------------------------------------------------------------
			! Reminder:
			!        mbnd(1) = ml
			!        mbnd(2) = mu
			!        mbnd(3) = mu + ml + 1
			!        mbnd(4) = 2*ml + mu + 1

      	real(dp), parameter :: del_abs = 1d-10, del_rel = 1d-10
			real(dp) :: yp, yj, yi, r, r0
      	integer i, j, j1, i1, i2, ii, jj, mba, mu, ml, mbnd3, mbnd4

	      npset = npset + 1 ! number of times a new coefficient matrix is made
	
			if (miter < 3) then ! not banded
				ml = 0
				mu = 0
				mbnd3 = n
				mbnd4 = n
			else
	      	ml = mbnd(1)
	      	mu = mbnd(2)
				mbnd3 = mbnd(3)
				mbnd4 = mbnd(4)
			end if

			pw(1:n*mbnd4) = 0.0d0
			pw1(1:n*mbnd3) = 0.0d0
			pw_R(1:n) = 0d0
			pw_C(1:n) = 0d0
			equed_flag = 0

		   nje = nje + 1 ! count jacobians created.

	      if (miter /= 2  .and.  miter /= 4) then ! doing analytical jacobians.  call pderv.
	
		      if ( miter /= 3 ) then ! square matrix
		         call pderv(t,h,con,y,pw,n,yprime,n,num_rpar,rpar,num_ipar,ipar,ierr)
					if (ierr /= 0) then
						if (verbosity > 0 .and. lout > 0) then
							write(lout,*) 'pset failed because of nonzero pderv ierr: ', ierr
						end if
						return
					end if
		         pw1(1:n*n)=pw(1:n*n)
		      else ! banded
		         call pderv(t,h,con,y,pw,n,yprime,mbnd4,num_rpar,rpar,num_ipar,ipar,ierr)
					if (ierr /= 0) then
						if (verbosity > 0 .and. lout > 0) then
							write(lout,*) 'pset failed because of nonzero pderv ierr: ', ierr
						end if
						return
					end if
					call save_unfactored_analytic_j(n,mbnd4,pw,mbnd3,pw1)
					call shift_analytic_j(n,mbnd4,mbnd3,pw)
		      end if
		      call LU_factor(n,mbnd4,pw,equed_flag)
				return

	      end if
	
			! do numerical differences to form the jacobian
      
	      if (itol == 2) then
	         ymax(1:n) = dabs(y(1:n,1))*rtol(1) + atol(1)
	      else if (itol == 3) then
	         ymax(1:n) = dabs(y(1:n,1))*rtol(1) + atol(1:n)
	      else if (itol == 4) then
	         ymax(1:n) = dabs(y(1:n,1))*rtol(1:n) + atol(1)
	      else if (itol == 5) then
	         ymax(1:n) = dabs(y(1:n,1))*rtol(1:n) + atol(1:n)
	      end if
	
	      j1 = 0
	
	      if (miter /= 4) then ! square matrix
		      call resid(n,t,h,con,y,save2,yprime,num_rpar,rpar,num_ipar,ipar,0,0,ierr) 
				if (ierr /= 0) then
					if (verbosity > 0 .and. lout > 0) then
						write(lout,*) 'pset failed because of nonzero resid ierr: ', ierr
					end if
					return
				end if
		      nre=nre+1
		      do j = 1,n
		         yj = y(j,1)
		         yp=yprime(j)        
					!r = r0*max(1d-4,abs(yj))
					r = sign(1d0,yj)*(del_abs + del_rel*abs(yj))
		         y(j,1) = yj + r
		         yprime(j)=yprime(j)+r/con                           
		         call resid(n,t,h,con,y,wrkspc,yprime,num_rpar,rpar,num_ipar,ipar,-1,j,ierr)
					if (ierr /= 0) then
						if (verbosity > 0 .and. lout > 0) then
							write(lout,*) 'pset failed because of nonzero resid ierr: ', ierr
						end if
						return
					end if
		         pw(1+j1:n+j1) = (wrkspc(1:n)-save2(1:n))/r
		         y(j,1) = yj
		         yprime(j)=yp
		         j1 = j1 + n
		      end do
		      nre = nre + n 
				call do_jdebug
		      pw1(1:n*n)=pw(1:n*n)
		      call LU_factor(n,mbnd4,pw,equed_flag)
				return
	      end if

			! banded matrix
	      call resid(n,t,h,con,y,save2,yprime,num_rpar,rpar,num_ipar,ipar,0,0,ierr)
			if (ierr /= 0) then
				if (verbosity > 0 .and. lout > 0) then
					write(lout,*) 'pset failed because of nonzero resid ierr: ', ierr
				end if
				return
			end if
	      nre = nre+1
	      mba = min0(mbnd3,n)
	      do j=1,mba
 1			format(a20,e26.16)	
	         do i=j,n,mbnd3
	            save1(i) = y(i,1)
	            save3(i) = yprime(i)
	            yi=y(i,1)
					!r = r0*max(1d-4,abs(yi))
					r = sign(1d0,yi)*(del_abs + del_rel*abs(yi))
	            y(i,1)=yi+r
	            yprime(i)=yprime(i)+r/con
	         end do
	         call resid(n,t,h,con,y,wrkspc,yprime,num_rpar,rpar,num_ipar,ipar,-1,j,ierr)
				if (ierr /= 0) then
					if (verbosity > 0 .and. lout > 0) then
						write(lout,*) 'pset failed because of nonzero resid ierr: ', ierr
					end if
					return
				end if
	
	         do jj=j,n,mbnd3
	            y(jj,1)=save1(jj)
	            yprime(jj)=save3(jj)
	            yi = y(jj,1)
					!r = r0*max(1d-4,abs(yi))
					r = sign(1d0,yi)*(del_abs + del_rel*abs(yi))
	            i1 = max0(jj-mu,1)
	            i2 = min0(jj+ml,n)
	            ii = jj*(mbnd4-1)-ml
	            pw(ii+i1:ii+i2) = (wrkspc(i1:i2) - save2(i1:i2))/r
	         end do
	
	      end do

	      nre = nre + mbnd3
			call do_jdebug
			call save_unfactored_numerical_j(n,mbnd4,pw,mbnd3,pw1)
			call LU_factor(n,mbnd4,pw,equed_flag)
			
			contains

			subroutine shift_analytic_j(n,ldaf,lda,af)
				integer, intent(in) :: n, ldaf, lda
				real(dp), intent(inout) :: af(ldaf,n)
				
				integer :: i, j
				
				j = ldaf - lda
				do i = lda, 1, -1
					af(i+j,1:n) = af(i,1:n)
				end do
				
			end subroutine shift_analytic_j

			subroutine save_unfactored_analytic_j(n,ldaf,af,lda,a)
				integer, intent(in) :: n, ldaf, lda
				real(dp), intent(in) :: af(ldaf,n)
				real(dp), intent(out) :: a(lda,n)
				a(1:lda,1:n) = af(1:lda,1:n)
			end subroutine save_unfactored_analytic_j

			subroutine save_unfactored_numerical_j(n,ldaf,af,lda,a)
				integer, intent(in) :: n, ldaf, lda
				real(dp), intent(in) :: af(ldaf,n)
				real(dp), intent(out) :: a(lda,n)
				a(1:lda,1:n) = af(mu+1:ldaf,1:n)
			end subroutine save_unfactored_numerical_j
			
			subroutine do_jdebug
				if ( jdebug /= 1 ) return 
		      call pderv(t,h,con,y,pw,n,yprime,mbnd4,num_rpar,rpar,num_ipar,ipar,ierr)
			end subroutine do_jdebug
			
			subroutine LU_factor(n,ldaf,af,eflag)
				integer, intent(in) :: n, ldaf
				integer, intent(out) :: eflag
				real(dp) :: af(ldaf,n)
				if (equil /= 0) then 
					eflag = 0 ! signal to equilibrate and factor on next solve cycle
		      else if (miter > 2) then ! banded
					call DGBTRF( n, n, ml, mu, af, ldaf, ipiv, ier )
					if (ier /= 0 .and. verbosity > 0 .and. lout > 0) then
						do j = 1, n
							if (af(ml+mu+1,j) == 0) 
     >							write(lout,*) 'zero on diagonal of jacobian at row=col=', j
						end do
						write(lout,*) 'DGBTRF failed: ', ier
					end if
		      else ! square
					call DGETRF( n, n, af, n, ipiv, ier )
					if (ier /= 0 .and. verbosity > 0 .and. lout > 0) then
						do j = 1, n
							if (af(j,j) == 0)
     >						write(lout,*) 'zero on diagonal of jacobian at row=col=', j
						end do
						write(lout,*) 'DGETRF failed: ', ier
					end if
		      end if
				! ier is set nonzero in case of trouble
		      ndec = ndec + 1 ! keep count of number of matrix decompositions
			end subroutine LU_factor

      end subroutine pset


      subroutine errors(n,tq,edn,e_present,eup,bnd,eddn)
      	integer, intent(in) :: n
      	real(dp), intent(in) :: tq(5)
      	real(dp), intent(out) ::edn,e_present,eup,bnd,eddn

			real(dp) :: sqhol
			
      	sqhol = dble(n)
      	edn = tq(1)*tq(1)*sqhol
			! ** error associated with method of order one lower.
      	e_present = tq(2)*tq(2)*sqhol
			! ** error associated with present order
      	eup = tq(3)*tq(3)*sqhol
			! ** error associated with higher order method
      	bnd = tq(4)*tq(4)*sqhol*0.5d0
      	eddn = tq(5)*tq(5)*sqhol
			! ** error associated with method of order two lower.

      end subroutine errors


      subroutine prdict(t,h,y,l,n)

	      integer, intent(in) :: l,n
	      real(dp), intent(in) :: h
	      real(dp), intent(inout) :: t,y(n,12)
			! -----------------------------------------------------------------
			! predicts a value for y at (t+h) given the history array at t.
			! -----------------------------------------------------------------
			integer :: j2
	      do j2 = 2,l         
	      	y(1:n,1) = y(1:n,1) + y(1:n,j2)          
	      end do        
	      t = t + h
	
	   end subroutine prdict


		subroutine set_scale(y, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)
			integer, intent(IN) :: n, itol, nind1, nind2, nind3
			real(dp), intent(IN) :: y(n,12), ymax(n), rtol(*), atol(*), hused
			real(dp), intent(OUT) :: scale(n)
			
			! sets the scale vector for calculating norms
		
			select case (itol)
			case (2)
			   scale(1:n) = rtol(1)*dabs(y(1:n,1)) + atol(1)
			case (3)
			   scale(1:n) = rtol(1)*dabs(y(1:n,1)) + atol(1:n)
			case (4)
			   scale(1:n) = rtol(1:n)*dabs(y(1:n,1)) + atol(1)
			case (5)
			   scale(1:n) = rtol(1:n)*dabs(y(1:n,1)) + atol(1:n)
			case default
			   scale(1:n) = ymax(1:n)
			end select
			
         if (nind2 /= 0) then
            scale(nind1+1:nind1+nind2) = scale(nind1+1:nind1+nind2)/hused
         end if

         if (nind3 /= 0) then
            scale(nind1+nind2+1:n) = scale(nind1+nind2+1:n)/(hused**2)
         end if

		end subroutine set_scale


      subroutine rscale(n,l,rh,y)
	      integer, intent(in) :: l,n
	      real(dp), intent(in) :: rh
	      real(dp), intent(inout) :: y(n,12)

	      integer ::  i,j
	      real(dp) :: di(8,8), ta, tb, tc, td, te, tf

			! for rescaling the history array after a change in stepsize

			! n      order of the problem
			! l      number of terms in the history array to be rescaled
			! rh     ratio of the stepsize change (i.e. rh = hnew/hold)
			! y()    the history array
			
			! polynomials in rh are defined in ref(1).

      	real(dp), parameter :: zero = 0d0

	      di(2,2) = rh
	      if (l > 2) then
	         ta = rh*rh
	         di(2,3) = rh* (1d0-rh)/2d0
	         di(3,3) = ta
	         if (l > 3) then
	            tb = ta*rh
	            di(2,4) = rh* ((rh-3d0)*rh+2d0)/6d0
	            di(3,4) = ta* (1d0-rh)
	            di(4,4) = tb
	            if (l > 4) then
	               tc = tb*rh
	               di(2,5) = - (((rh-6d0)*rh+11d0)*rh-6d0)*rh/
     +                      	24d0
	               di(3,5) = ta* ((7d0*rh-18d0)*rh+11d0)/12d0
	               di(4,5) = 1.5d0*tb* (1d0-rh)
	               di(5,5) = tc
	               if (l > 5) then
	                  td = tc*rh
	                  di(2,6) = ((((rh-10d0)*rh+35d0)*rh-50d0)
     +                        *rh+24d0)*rh/120d0
                  	di(3,6) = - (((3d0*rh-14d0)*rh+21d0)*rh
     +                       -10d0)*ta/12d0
	                  di(4,6) = ((5d0*rh-12d0)*rh+7d0)*tb/4d0
	                  di(5,6) = 2d0*tc* (1d0-rh)
	                  di(6,6) = td
	                  if (l > 6) then
	                     te = td*rh
	                     di(2,7) = -rh* (rh-1d0)* (rh-2d0)*
     +                          (rh-3d0)*(rh-4d0)*(rh-5d0)/
     +                           720d0
                     	di(3,7) = ta* ((((62d0*rh-450d0)*rh+
     +                         1190d0)*rh-1350d0)*rh+548d0)
     +                         /720d0
                     	di(4,7) = tb* (((-18d0*rh+75d0)*rh
     +                          -102d0)*rh+45d0)/24d0
                     	di(5,7) = tc* ((13d0*rh-30d0)*rh+17d0)
     +                         /6d0
	                     di(6,7) = 2.5d0*td* (1d0-rh)
	                     di(7,7) = te
	                     if (l > 7) then
	                        tf = te*rh
	                        di(2,8) = rh*(rh-1d0)*(rh-2d0)*(rh
     +                            -3d0)*(rh-4d0)*(rh-5d0)
     +                            *(rh-6d0)/5040d0
                        	di(3,8) = ta* ((((((-126d0*rh)+1302d0)*rh-
     +                            5250d0)*rh+10290d0)*rh-9744d0
     +                            )*rh+3528d0)/5040d0
                        	di(4,8) = tb* ((((43d0*rh-270d0)*rh+
     +                            625d0)*rh-630d0)*rh+232d0)
     +                           /120d0
                        	di(5,8) = tc* (((-10d0*rh+39d0)*rh-
     +                            50d0)*rh+21d0)/6d0
                        	di(6,8) = td* ((20d0*rh-45d0)*rh+25d0
     +                            )/6d0
	                        di(7,8) = 3d0*te* (1d0-rh)
	                        di(8,8) = tf
	                     end if

	                  end if

	               end if

	            end if

	         end if

	      end if

	      do i = 1,n
				do j = 2,l
					y(i,j) = dot_product(di(j,j:l),y(i,j:l))
	   		end do
	      end do
      
		end subroutine rscale


      subroutine hchose(rh,h,hstpsz,ovride)
			! called with info about a proposed new stepsize

	      real(dp), intent(in) :: h ! the current stepsize
	      real(dp), intent(inout) :: rh ! ratio of new h to old h
	      real(dp), intent(inout) :: hstpsz(2,14) ! history of h's and rh's
				! hstpsz(1,:) = old rh values
				! hstpsz(2,:) = old h values
	      logical, intent(out) :: ovride ! true if decide to change rh based on history

			integer :: i, i2
			
	      if (h /= hstpsz(2,1)) then
				! move all elements down one place			
	         do i=12,2,-1
	            i2=i-1
	            hstpsz(1,i)=hstpsz(1,i2)
	         	hstpsz(2,i)=hstpsz(2,i2)
				end do
				! now insert value of h used before this call
	         hstpsz(1,2)=h/hstpsz(2,1)
	         hstpsz(2,1)=h
	      end if

			! decide on the new change
	      if (rh > 1.0d0) then
	         ovride=.false.
	      else if (hstpsz(1,2) <= 1.0d0) then
	         ovride=.false.
	      else if ((rh*h) <= hstpsz(2,2)) then
	         ovride=.false.
	      else
	         rh=hstpsz(2,2)/h
	         ovride=.true.
	      end if
	      hstpsz(1,1)=rh
      
      end subroutine hchose


      end module mebdfi_support
