! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module test_burn
      use const_def
      use chem_def
      use chem_lib
      use net_def
      use rates_def, only: num_rvs
      use test_net_support
      
      implicit none

      
      ! for burner
      double precision, pointer :: x_initial(:) ! (species)

      double precision :: burn_T, burn_Rho, burn_neu_total, burn_lnE, burn_lnS
      double precision :: burn_logT, burn_logRho, burn_eta, burn_cp
      
      ! info at start of step
      double precision :: T_prev, time_prev, eps_nuc_prev, eps_neu_prev, cp_prev
      double precision, pointer :: x_previous(:) ! (species)

      integer :: eos_handle
      double precision :: total_energy ! u + P/rho - binding energy; for conservation checks
      double precision :: eps_nuc_max


      integer, parameter :: i_caller_id = 1
      integer, parameter :: i_handle = 2
      integer, parameter :: i_screening_mode = 3
      integer, parameter :: i_reuse_rates = 4
      integer, parameter :: i_net_lwork = 5
      integer, parameter :: burn_lipar = 5

      
      integer, parameter :: r_temp = 1
      integer, parameter :: r_lgT = 2
      integer, parameter :: r_rho = 3
      integer, parameter :: r_lgRho = 4
      integer, parameter :: r_eta = 5
      integer, parameter :: r_theta = 6
      integer, parameter :: burn_lrpar = 6
      
      logical, parameter :: dbg = .true.
      
      logical :: show_by_step
      
      
      contains

      
      subroutine Do_One_Test_Burn(net_file_in, show)
         use num_lib, only: isolve_name
         use mtx_def
         use eos_lib
         
         character (len=*), intent(in) :: net_file_in
         logical, intent(in) :: show
         
         character (len=256) :: net_file
         double precision :: logRho, logT, Rho, T, xsum, 
     >     eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, 
     >     eps_nuc_categories(num_rvs, num_categories)
         double precision, dimension(:, :), pointer :: rate_screened, rate_raw
         integer :: i, j, info
         
         integer :: which_solver ! as defined in num_def.f
         integer, parameter :: num_times = 1
         double precision, dimension(:), pointer :: times, dxdt_source_term
         double precision, dimension(:,:), pointer :: log10Ts_f, log10Rhos_f, etas_f, log10Ps_f

         double precision, pointer :: rate_factors(:) ! (num_reactions)
         double precision :: category_factors(num_categories)
         integer :: screening_mode
         double precision :: theta_e_for_graboske_et_al
         
         ! args to control the solver -- see num/public/num_isolve.dek
         double precision :: h 
         double precision :: max_step_size ! maximal step size.
         integer :: max_steps ! maximal number of allowed steps.
         ! absolute and relative error tolerances
         double precision :: rtol(1) ! relative error tolerance(s)
         double precision :: atol(1) ! absolute error tolerance(s)
         integer :: itol ! switch for rtol and atol
         
         double precision, pointer :: ending_x(:) ! (species)
         integer :: nfcn    ! number of function evaluations
         integer :: njac    ! number of jacobian evaluations
         integer :: nstep   ! number of computed steps
         integer :: naccpt  ! number of accepted steps
         integer :: nrejct  ! number of rejected steps
         integer, parameter :: solver_name_len = 10
         character (len=solver_name_len) :: solver_name
         
         integer :: ierr, iout, caller_id
         double precision :: time_doing_net
         logical :: do_test, clip
         
         include 'formats.dek'
         
         ierr = 0
         show_by_step = show
         net_file = net_file_in
         clip = .true.
            
         eps_nuc_max = 0
         
         do_test = (trim(net_file) == 'do test')
         if (do_test) net_file = 'ns_he.net'
         call test_net_setup(net_file)
         
         allocate(
     >      rate_factors(num_reactions), ending_x(species), 
     >      x_initial(species), x_previous(species), times(num_times),
     >      log10Ts_f(4,num_times), log10Rhos_f(4,num_times),
     >      etas_f(4,num_times), log10Ps_f(4,num_times),
     >      stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed for Do_One_Test_Burn'
            stop 1
         end if

         xin = 0
         eta = 0
         rate_factors(:) = 1
         category_factors(:) = 1
         screening_mode = extended_screening
         which_rates(:) = rates_NACRE_if_available
         etas_f(1,1) = 0
         theta_e_for_graboske_et_al = 1
         max_step_size = 0
         max_steps = 10000
         iout = 1
         itol = 0         
         
         !which_solver = 4 ! ros3pl
         !which_solver = 6 ! rodas4
         which_solver = 8 ! seulex
         !which_solver = 9 ! sodex
         
         rtol(:) = 1d-5
         atol(:) = 1d-6
         
         if (do_test) then
         
            if (.not. qt) write(*,*) 'do test'
            
            times(1) = 1d4
            h = 1d-14
            logT = 8.75d0
            logRho = 4.58d0
            
            rtol(:) = 1d-10
            atol(:) = 1d-10
            
            xin(net_iso(ihe4)) =  0.930d0
            xin(net_iso(ic12)) =  0.045d0
            xin(net_iso(in14)) =  0.023d0
            xin(net_iso(io16)) =  0.002d0
         
         else if (net_file == 'basic.net') then

            times(1) = 1d10*secyer
            h = 1d-14
            logT = 7.28d0
            logRho = 2d0
            
            rtol(:) = 1d-12
            atol(:) = 1d-14
            
            ! solar
            xin = 0
            xin(net_iso(ih1)) = chem_Xsol("h1")
            xin(net_iso(ihe3)) = chem_Xsol("he3")
            xin(net_iso(ihe4)) = chem_Xsol("he4")
            xin(net_iso(ic12)) = chem_Xsol("c12")
            xin(net_iso(in14)) = chem_Xsol("n14")
            xin(net_iso(io16)) = chem_Xsol("o16")
            xin(net_iso(ine20)) = chem_Xsol("ne20")
            xin(net_iso(img24)) = chem_Xsol("mg24")

            ! He burn
            times(1) = 1d10*secyer
            h = 1d-14
            logT = 8.1d0
            logRho = 4d0
            xin = 0
            xin(net_iso(ihe4)) =  0.98d0
            xin(net_iso(in14)) =  0.02d0

            xin(species) = 1d0 - sum(xin(1:species-1))
            do i=1,species
               if (.not. qt) write(*,1) chem_isos% name(chem_id(i)), xin(i)
            end do
            if (.not. qt) write(*,1) 'sum xin', sum(xin(1:species))
         
         else if (net_file == 'pp_and_cno_extras.net') then
            
            ! very low mass star (0.2M range)
            times(1) = 1d12*secyer
            logT = 6.8d0
            logRho = 2.3d0
            
            ! intermediate mass star (7M range)
            times(1) = 1d8*secyer
            logT = 7.5d0
            logRho = 1.5d0
            
            ! low mass star (1M range)
            times(1) = 1d10*secyer
            logT = 7.28d0
            logRho = 2d0
            
            h = 1d-14
            rtol(:) = 1d-12
            atol(:) = 1d-14
            
            xin = 0
     
            xin(net_iso(ih1)) = chem_Xsol("h1")
            xin(net_iso(ih2)) = chem_Xsol("h2")
            xin(net_iso(ihe3)) = chem_Xsol("he3")
            xin(net_iso(ihe4)) = chem_Xsol("he4")
            xin(net_iso(ili7)) = chem_Xsol("li7")
            xin(net_iso(ibe7)) = chem_Xsol("be7")
            xin(net_iso(ib8)) = chem_Xsol("b8")
            xin(net_iso(ic12)) = chem_Xsol("c12")
            xin(net_iso(ic13)) = chem_Xsol("c13")
            xin(net_iso(in13)) = chem_Xsol("n13")
            xin(net_iso(in14)) = chem_Xsol("n14")
            xin(net_iso(in15)) = chem_Xsol("n15")
            xin(net_iso(io14)) = chem_Xsol("o14")
            xin(net_iso(io15)) = chem_Xsol("o15")
            xin(net_iso(io16)) = chem_Xsol("o16")
            xin(net_iso(io17)) = chem_Xsol("o17")
            xin(net_iso(io18)) = chem_Xsol("o18")
            xin(net_iso(if17)) = chem_Xsol("f17")
            xin(net_iso(if18)) = chem_Xsol("f18")
            xin(net_iso(if19)) = chem_Xsol("f19")
            xin(net_iso(ine18)) = chem_Xsol("ne18")
            xin(net_iso(ine19)) = chem_Xsol("ne19")
            xin(net_iso(ine20)) = chem_Xsol("ne20")
            xin(net_iso(img22)) = chem_Xsol("mg22")
            xin(net_iso(img24)) = chem_Xsol("mg24")
            
            xin(species) = 1d0 - sum(xin(1:species-1))
            if (.not. qt) then
               do i=1,species
                  write(*,1) chem_isos% name(chem_id(i)), xin(i)
               end do
               write(*,1) 'sum xin', sum(xin(1:species))
               write(*,1) 'x', xin(net_iso(ih1)) + xin(net_iso(ih2))
               write(*,1) 'y', xin(net_iso(ihe3)) + xin(net_iso(ihe4))
               write(*,1) 'z', 1d0 - 
     >            (xin(net_iso(ih1)) + xin(net_iso(ih2)) + xin(net_iso(ihe3)) + xin(net_iso(ihe4)))
               write(*,*)
            end if
            !stop
         
         else if (net_file == 'cno_extras_o18_to_mg26.net') then

               ! He burn
               times(1) = 1d10*secyer
               h = 1d-14
               logT = 8.1d0
               logRho = 4d0
               xin = 0
               xin(net_iso(ihe4)) =  0.98d0
               xin(net_iso(in14)) =  0.02d0
               
               rtol(:) = 1d-12
               atol(:) = 1d-14
               
               do i=1,species
                  write(*,1) chem_isos% name(chem_id(i)), xin(i)
               end do
               screening_mode = classic_screening
         
         else if (net_file == 'o18_to_mg26.net') then

               ! He burn
               times(1) = 1d10*secyer
               h = 1d-14
               logT = 8.1d0
               logRho = 4d0
               xin = 0
               xin(net_iso(ihe4)) =  0.98d0
               xin(net_iso(in14)) =  0.02d0
               
               rtol(:) = 1d-12
               atol(:) = 1d-14
               
               do i=1,species
                  write(*,1) chem_isos% name(chem_id(i)), xin(i)
               end do
               screening_mode = classic_screening
         
         else if (net_file == 'approx20.net') then

               ! Neon burn
               times(1) = 1d12
               h = 1d-8
               logT = log10(1.7d9)
               logRho = log10(3d6)
               xin = 0
               xin(net_iso(ine20)) =  1
               rtol(:) = 1d-9
               atol(:) = 1d-10
               
                   
               ! C burn
               times(1) = 1d20
               h = 1d-14
               logT = 8.9d0
               logRho = 5.5d0
               xin = 0
               xin(net_iso(ic12)) =  1

               
               ! PURE HE
               times(1) = 1d20
               h = 1d-14
               logT = 8.1d0
               logRho = 4d0
               xin = 0
               xin(net_iso(ihe4)) =  1
                   
               ! O burn
               times(1) = 10**9.5
               h = 1d-8
               logT = 9.3d0
               logRho = 6.9d0
               xin = 0
               xin(net_iso(io16)) =  1

                   
               ! Si28 burn
               times(1) = 10**3.5
               h = 1d-7
               logT = 9.6d0
               logRho = 8d0
               xin = 0
               xin(net_iso(isi28)) =  1
               
                   
               ! 98% Helium burn
               times(1) = 10
               h = 1d-14
               logT = log10(2.6d9)
               logRho = log10(2d5)
               
               times(1) = 1d10*secyer
               logT = 8d0
               logRho = 5d0
               
               xin = 0
               xin(net_iso(ihe4)) =  0.98d0
               xin(net_iso(in14)) =  0.02d0
               
               rtol(:) = 1d-8
               atol(:) = 1d-9
               
               screening_mode = extended_screening
               
         else
            
            write(*, *) 'need to define setup for net_file ', trim(net_file)
            stop 'Do_One_Test_Burn'
         
         end if
         
         Rho = 10**logRho
         T = 10**logT
         
         burn_T = T
         burn_Rho = Rho
         
         log10Ts_f(1,1) = logT
         log10Rhos_f(1,1) = logRho
         x_initial(1:species) = xin(1:species)
         x_previous(1:species) = xin(1:species)
         burn_neu_total = 0
         caller_id = 0
         dxdt_source_term => null()
         
         if (.not. qt) then
            write(*,*) 'Do_One_Test_Burn ' // trim(net_file)
            
            if (show_by_step) then
               write(*,*) 'max_steps', max_steps
               write(*,fmt='(a7,99(a26,1x))',advance='no') 'i', 'lg_time', 'lg_eps_nuc', 'lg_ergs'
               do j=1,species
                  write(*,fmt='(a26,1x)',advance='no') 'lg_' // trim(chem_isos% name(chem_id(j)))
               end do
               write(*,fmt='(99(a26,1x))') 'lg_abs_1_sub_sum_x', 'ye', 'total_e_div_e_start',
     >            'logT', 'logRho', 'logS', 'logE', 'h17', 'lg_neu_total', 'lg_enthalpy',
     >            'lg_age'   
            end if
         end if
         
         time_doing_net = 0
         call net_1_zone_burn(
     >         handle, which_solver, species, num_reactions, 0d0, times(1), xin, clip,
     >         num_times, times, log10Ts_f, log10Rhos_f, etas_f,
     >         dxdt_source_term, rate_factors, category_factors, std_reaction_Qs, std_reaction_neuQs,
     >         screening_mode, theta_e_for_graboske_et_al,
     >         h, max_step_size, max_steps, rtol, atol, itol,
     >         lapack, caller_id, burn_solout, iout, ending_x,
     >         nfcn, njac, nstep, naccpt, nrejct, time_doing_net, ierr)

         if (ierr /= 0) then
            write(*,*) 'net_1_zone_burn ierr', ierr
            stop 1
         end if
         
         if (.not. qt) then
            write(*,*)
            call isolve_name(which_solver, solver_name_len, solver_name)
            write(*,2) 'using ' // trim(solver_name)
            write(*,1) 'rtol', rtol(1)
            write(*,1) 'atol', atol(1)
            do i=1,num_times
               write(*,*)
               write(*,1) 'at time (sec)', times(i)
               write(*,1) 'at time (yr)', times(i)/secyer
               write(*,1) 'temperature', 10**log10Ts_f(1,i)
               write(*,1) 'density', 10**log10Rhos_f(1,i)
               write(*,*)
               if (show_by_step) then
                  write(*,'(a64,99a26)') 'init mass fraction', 'final mass fraction', 'change since start'
                  do j=1,species
                     if (chem_id(j) /= ihe4) then
                        if (x_initial(j) < 1d-5 .and. ending_x(j) < 1d-5) cycle
                        if (abs(ending_x(j) - x_initial(j)) < 1d-10) cycle
                     end if
                     write(*,1) trim(chem_isos% name(chem_id(j))), x_initial(j), ending_x(j), 
     >               ending_x(j) - x_initial(j)
                  end do
                  write(*,*)
                  write(*,2) 'nfcn', nfcn
                  write(*,2) 'njac', njac
                  write(*,2) 'nstep', nstep
                  write(*,2) 'naccpt', naccpt
                  write(*,2) 'nrejct', nrejct
               else
                  do j=1,species
                     if (chem_id(j) /= ihe4) cycle
                     write(*,1) trim(chem_isos% name(chem_id(j))), x_initial(j), ending_x(j), 
     >               ending_x(j) - x_initial(j)
                  end do
               end if
            end do
         
         
            write(*,*)
            
         end if
         
         deallocate(
     >      rate_factors, ending_x,
     >      x_initial, x_previous, times,
     >      log10Ts_f, log10Rhos_f,
     >      etas_f, log10Ps_f)
         
         
         contains
         
         subroutine turn_off(rate_id)
            integer, intent(in) :: rate_id
            integer :: rn
            rn = reaction_table(rate_id)
            if (rn == 0) return
            rate_factors(rn) = 0
            write(*,*) 'turn off ' // trim(reaction_name(rate_id))
         end subroutine turn_off
         
      end subroutine Do_One_Test_Burn


      subroutine burn_solout(
     >         nr, told, time, n, x, rwork_y, iwork_y, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         use chem_lib
         use chem_def
         use num_lib, only: safe_log10
         integer, intent(in) :: nr, n, lrpar, lipar
         double precision, intent(in) :: told, time
         double precision, intent(inout) :: x(n)
         ! x can be modified if necessary to keep it in valid range of possible solutions.
         double precision, intent(inout), target :: rpar(lrpar), rwork_y(*)
         integer, intent(inout), target :: ipar(lipar), iwork_y(*)
         interface
            include 'num_interp_y.dek'
         end interface
         integer, intent(out) :: irtrn ! < 0 causes solver to return to calling program.


         double precision :: logRho, logT
         double precision :: eps_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT,
     >     eps_nuc_categories(num_rvs, num_categories)
         double precision, dimension(:, :), pointer :: reaction_eps_nuc
         double precision, dimension(:, :), pointer :: rate_screened, rate_raw

         double precision :: Rho, T, xsum, d_eps_nuc_dx(species), dx, enuc, 
     >         dt, burn_ergs, energy, enthalpy, xh, xhe, approx_abar, approx_zbar
         
         integer :: info, i, j, lwork, adjustment_iso, cid, ierr
         double precision, dimension(species) :: dabar_dx, dzbar_dx, eps_nuc_dx
         double precision, pointer :: work(:), rate_factors(:), category_factors(:)

         include 'formats.dek'
         
         irtrn = 0
         if (time == 0) return
         
         x(1:n) = max(0d0, min(1d0, x(1:n)))
         xsum = sum(x(1:n))
         x(1:n) = x(1:n)/xsum

         info = 0
         lwork = net_work_size(handle, info) 
         if (info /= 0) stop 1
         
         allocate(work(lwork), 
     >         rate_factors(num_reactions), category_factors(num_categories),
     >         rate_screened(num_rvs, num_reactions),     
     >         reaction_eps_nuc(num_rvs, num_reactions),      
     >         rate_raw(num_rvs, num_reactions), stat=info)
         if (info /= 0) stop 2
         
         T = burn_T
         Rho = burn_Rho
         logT = log10(T)
         logRho = log10(Rho)
         xin = x

         call composition_info(
     >            species, chem_id, xin, xh, xhe, abar, zbar, z2bar, ye, 
     >            approx_abar, approx_zbar, xsum, dabar_dx, dzbar_dx)
         
         rate_factors(:) = 1
         category_factors(:) = 1
         call net_get(handle, species, num_reactions, 
     >            xin, T, logT, Rho, logRho, 
     >            abar, zbar, z2bar, ye, eta, rate_factors, category_factors, 
     >            std_reaction_Qs, std_reaction_neuQs,
     >            eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx, 
     >            dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx, 
     >            screening_mode, theta_e_for_graboske_et_al,     
     >            rate_screened, rate_raw, reuse_given_rates, 
     >            reaction_eps_nuc, eps_nuc_categories, eps_neu_total,
     >            lwork, work, info)
         if (info /= 0) then
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*, *) 'bad return from net_get'
            write(*, *) trim(alert_message)
            stop 1
         end if
         if (.false. .and. time > 1d45) then
            write(*,1) 'time', time
            write(*,1) 'eps_nuc', eps_nuc
            write(*,1) 'T =', T
            write(*,1) 'logT =', logT
            write(*,1) 'Rho =', Rho
            write(*,1) 'logRho =', logRho
            write(*,1) 'abar =', abar
            write(*,1) 'zbar =', zbar
            write(*,1) 'z2bar =', z2bar
            write(*,1) 'ye =', ye
            write(*,1) 'eta =', eta
            write(*,*)
            do j=1,species
               write(*,1) 'xin(net_iso(i' // trim(chem_isos% name(chem_id(j))) // ')) =', x(j)
            end do
            write(*,*)
            do j=1,num_reactions
               if (abs(reaction_eps_nuc(i_rate,j)) < 1d8) cycle
               write(*, 1) trim(reaction_name(reaction_id(j))), 
     >               reaction_eps_nuc(i_rate,j), rate_screened(i_rate,j)
            end do
            write(*,*)
            stop 'solout'
         end if
         deallocate(work, rate_factors, category_factors, rate_screened, reaction_eps_nuc, rate_raw)
         
         dt = time - told

         ! set burn_ergs according to change from initial abundances
         burn_neu_total = burn_neu_total + eps_neu_total*dt
         burn_ergs = 0
         do i=1,species
            cid = chem_id(i)
            dx = x(i) - x_initial(i)
            burn_ergs = burn_ergs + 
     >            (chem_isos% binding_energy(cid) - chem_isos% Z(cid)*del_Mp - 
     >                 chem_isos% N(cid)*del_Mn)*dx/chem_isos% W(cid)
         end do
         burn_ergs = burn_ergs*Qconv - burn_neu_total

         energy = 0
         enthalpy = 0
         total_energy = 1
         burn_logT=0; burn_logRho=0; burn_lnS=0; burn_lnE=0
         
         x_previous(1:species) = x(1:species)

         if (.not. qt .and. show_by_step) then
            write(*,'(i7,99(1pe26.16,1x))') nr, safe_log10(time), safe_log10(eps_nuc), safe_log10(burn_ergs),
     >         safe_log10(max(1d-16,x(1:n))), safe_log10(abs(1d0 - sum(xin(1:n)))), ye, 
     >         energy/total_energy, burn_logT, burn_logRho, burn_lnS/ln10, burn_lnE/ln10, 
     >         enthalpy*1d-17, safe_log10(burn_neu_total), safe_log10(enthalpy),
     >         safe_log10(time/secyer)     
         end if

      end subroutine burn_solout


      end module test_burn

