! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_one_zone_support
      use chem_def
      use chem_lib
      use net_def
      use net_lib
      use alert_lib
      use const_def
      use rates_def
      use screen_def
      
      implicit none


      character(len=256):: final_abundances_filename
      logical :: save_final_abundances

      character(len=256):: initial_abundances_filename
      logical :: read_initial_abundances

   	
      character(len=256):: burn_filename
      double precision :: burn_tend, burn_h, burn_rho, burn_temp, burn_rtol, burn_atol, burn_P
      double precision :: min_for_show_peak_abundances
   	
   	integer, parameter :: max_num_burn_isos_to_show = 1000
      character(len=iso_name_length) :: names_of_isos_to_show(max_num_burn_isos_to_show)
   	integer :: num_names_of_isos_to_show

   	integer, parameter :: max_num_isos_for_Xinit = 1000
      character(len=iso_name_length) :: names_of_isos_for_Xinit(max_num_isos_for_Xinit)
      double precision :: values_for_Xinit(max_num_isos_for_Xinit)
   	integer :: num_isos_for_Xinit
   	logical :: uniform_Xinit

      double precision :: theta_e_for_graboske_et_al
      integer :: screening_mode
      integer, pointer :: which_rates(:)

   	integer, parameter :: io_out = 35
   	double precision :: data_output_min_t


      character (len=32) :: which_solver
      integer :: decsol_switch
      character (len=32) :: small_mtx_decsol, large_mtx_decsol


   	logical :: show_net_reactions_info
      integer :: which_rates_choice
   	
   	double precision :: rattab_logT_lower_bound, rattab_logT_upper_bound

      character(len=256):: mesa_data_dir, data_filename, data_heading_line
      character (len=64) :: net_file, cache_suffix
      
      integer :: handle
      type (Net_General_Info), pointer  :: g
      integer :: species, num_reactions
      
      integer, dimension(:), pointer :: net_iso, chem_id

      double precision :: abar, zbar, z2bar, ye, eta, eps_neu_total
      double precision, dimension(:), pointer :: &
            xin, xin_copy, d_eps_nuc_dx, dxdt, d_dxdt_dRho, d_dxdt_dT
      double precision, pointer :: d_dxdt_dx(:, :)  

      integer :: max_steps ! maximal number of allowed steps.

      double precision, pointer :: x_initial(:) ! (species)

      double precision :: burn_neu_total, burn_lnE, burn_lnS
      double precision :: burn_logT, burn_logRho, burn_eta, burn_cp
      
      double precision :: T_prev, time_prev, eps_nuc_prev, eps_neu_prev, cp_prev
      double precision, pointer :: x_previous(:) ! (species)

      double precision, dimension(:), pointer :: peak_abundance, peak_time


      logical :: burn_at_constant_P, clip
      double precision :: starting_temp, pressure
      
      

      integer :: eos_handle


      integer, parameter :: i_caller_id = 1
      integer, parameter :: i_handle = 2
      integer, parameter :: i_screening_mode = 3
      integer, parameter :: i_reuse_rates = 4
      integer, parameter :: i_net_lwork = 5
      integer, parameter :: burn_lipar = 5

      
      integer, parameter :: r_temp = 1
      integer, parameter :: r_lgT = 2
      integer, parameter :: r_rho = 3
      integer, parameter :: r_lgRho = 4
      integer, parameter :: r_eta = 5
      integer, parameter :: r_theta = 6
      integer, parameter :: burn_lrpar = 6
      
      logical, parameter :: dbg = .true.
      
      
      contains
      
      
      integer function burn_isos_for_Xinit(i)
         integer, intent(in) :: i 
         burn_isos_for_Xinit = chem_get_iso_id(names_of_isos_for_Xinit(i))
      end function burn_isos_for_Xinit
      
      
      integer function burn_isos_to_show(i)
         integer, intent(in) :: i 
         burn_isos_to_show = chem_get_iso_id(names_of_isos_to_show(i))
      end function burn_isos_to_show

      
      subroutine Do_One_Zone_Burn(net_file_in)
         use num_lib, only: solver_option
         use mtx_lib, only: decsol_option
         use eos_lib
         use mtx_def
         
         character (len=*), intent(in) :: net_file_in
         
         character (len=256) :: net_file
         double precision :: logRho, logT, Rho, T, xsum, &
           eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, &
           eps_nuc_categories(num_rvs, num_categories)
         double precision, dimension(:,:), pointer :: &
            rate_screened, rate_raw
         integer :: i, j, info, iounit, decsol_choice, solver_choice
         
         integer, parameter :: num_times = 1

         double precision, dimension(:), pointer :: times, dxdt_source_term
         double precision, dimension(:,:), pointer :: &
            log10Ts_f, log10Rhos_f, etas_f, log10Ps_f

         double precision, pointer :: rate_factors(:) ! (num_reactions)
         double precision :: category_factors(num_categories)
         
         ! args to control the solver -- see num/public/num_isolve.dek
         double precision :: h 
         double precision :: max_step_size ! maximal step size.
         ! absolute and relative error tolerances
         double precision :: rtol(1) ! relative error tolerance(s)
         double precision :: atol(1) ! absolute error tolerance(s)
         integer :: itol ! switch for rtol and atol
         
         double precision, pointer :: ending_x(:) ! (species)
         integer :: nfcn    ! number of function evaluations
         integer :: njac    ! number of jacobian evaluations
         integer :: nstep   ! number of computed steps
         integer :: naccpt  ! number of accepted steps
         integer :: nrejct  ! number of rejected steps
         
         integer :: ierr, iout, caller_id, cid
         integer :: time0, time1, clock_rate
         
         double precision :: ending_temp, ending_rho, initial_rho, dt
         double precision :: time_doing_net, time_doing_eos
         
         include 'formats.dek'
         
         ierr = 0
         net_file = net_file_in

         Rho = burn_rho
         T = burn_temp     
         logT = log10(T)
         logRho = log10(burn_rho)
         
         call test_net_setup(net_file, logT)
         
         allocate( &
            rate_factors(num_reactions), ending_x(species), &
            x_initial(species), x_previous(species), times(num_times), &
            log10Ts_f(4,num_times), log10Rhos_f(4,num_times), &
            etas_f(4,num_times), log10Ps_f(4,num_times), &
            peak_abundance(species), peak_time(species), &
            stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed for Do_One_Zone_Burn'
            stop 1
         end if
      	peak_abundance(:) = 0

         xin = 0
         eta = 0
         rate_factors(:) = 1
         category_factors(:) = 1
         
         max_step_size = 0
         iout = 1
         itol = 0         
         
         times(1) = burn_tend
         h = 1d-14
         
         rtol(:) = burn_rtol
         atol(:) = burn_atol
         
      	xin = 0
      	if (read_initial_abundances) then
      	   call read_X(ierr)
      	   if (ierr /= 0) return
      	else if (uniform_Xinit) then
      	   xin(:) = 0.5d0/(species-1)
            j = net_iso(ih1)
            if (j <= 0) stop 'where is the h?'
      	   xin(j) = 0.5d0
      	else
         	do i = 1, num_isos_for_Xinit
               cid = burn_isos_for_Xinit(i)
               j = net_iso(cid)
         	   if (j <= 0 .or. j > species) then
         	      write(*,*) 'bad names_of_isos_for_Xinit ' // trim(names_of_isos_for_Xinit(i))
         	      stop 1
         	   end if
         	   xin(j) = values_for_Xinit(i)
         	end do
      	end if
      	xin(:) = xin(:)/sum(xin(:))
         
         log10Ts_f(1,1) = logT
         log10Ts_f(2:4,1) = 0
         log10Rhos_f(1,1) = logRho
         log10Rhos_f(2:4,1) = 0
         etas_f(1:4,1) = 0
         
         x_initial(1:species) = xin(1:species)
         x_previous(1:species) = xin(1:species)
         burn_neu_total = 0
         caller_id = 0
         dxdt_source_term => null()

	      write(*,*)
	      write(*,*)
         write(*,1) 'one zone burn ' // trim(net_file)
	      write(*,*)
	      write(*,2) 'number of species', species
	      write(*,*)
	      write(*,1) 'rho', burn_rho
	      write(*,1) 'temp', burn_temp
	      write(*,1) 'tend', burn_tend
	      write(*,1) 'tend/secyer', burn_tend/secyer
	      write(*,*)
         write(*,1) 'using ' // trim(which_solver)
         write(*,1) 'rtol', rtol(1)
         write(*,1) 'atol', atol(1)
	      write(*,*)
	      write(*,1) 'initial abundances'
	      call show_X(xin,.false.)
            
         write(io_out,'(a)') trim(data_heading_line)
         write(io_out,'(a7,99(a26,1x))',advance='no') &
            'i', 'lg_ergs_total', 'lg_ergs_nuc', 'lg_ergs_neu', &
            'lg_eps_nuc_total', 'lg_eps_nuc', 'lg_eps_neu', &
            'time', 'lg_time', 'lg_yrs', 'dt', 'lg_dt', 'ye', 'xsum_sub_1'
         do i=1,num_names_of_isos_to_show
            cid = burn_isos_to_show(i)
            j = net_iso(cid)
            if (j == 0) cycle
            write(io_out,'(a26,1x)',advance='no') 'lg_' // trim(chem_isos% name(cid))
         end do
         write(io_out,*) 

         if (show_net_reactions_info) then
            write(*,'(a)') ' species'
            do j=1,species
               write(*,'(i6,3x,a)') j, chem_isos% name(chem_id(j))
            end do
            write(*,*)
            call show_net_reactions(handle, 6, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in show_net_reactions'
               stop 1
            end if
            write(*,*)
         end if
         
         write(*,1) 'h', h
         write(*,1) 'max_step_size', max_step_size
         write(*,2) 'max_steps', max_steps
         write(*,2) 'screening_mode', screening_mode
         write(*,1) 'theta_e_for_graboske_et_al', theta_e_for_graboske_et_al
         
         call system_clock(time0,clock_rate)
         time_doing_net = -1
         time_doing_eos = -1

         if (species >= decsol_switch) then
            decsol_choice = decsol_option(large_mtx_decsol, ierr)
            if (ierr /= 0) then
               write(*,*) 'ERROR: unknown large_mtx_decsol ' // trim(large_mtx_decsol)
               return
            end if
         else
            decsol_choice = decsol_option(small_mtx_decsol, ierr)
            if (ierr /= 0) then
               write(*,*) 'ERROR: unknown small_mtx_decsol ' // trim(small_mtx_decsol)
               return
            end if
         end if
         
         solver_choice = solver_option(which_solver, ierr)
         if (ierr /= 0) then
            write(*,*) 'ERROR: unknown value for which_solver ' // trim(which_solver)
            return
         end if

         if (burn_at_constant_P) then
            write(*,1) 'pressure', pressure
            write(*,1) 'starting_temp', starting_temp
            log10Ps_f(1,1) = log10(pressure)
            call net_1_zone_burn_const_P( &
               handle, eos_handle, species, num_reactions, &
               solver_choice, starting_temp, xin, clip, &
               num_times, times, log10Ps_f, &
               rate_factors, category_factors, std_reaction_Qs, std_reaction_neuQs, &
               screening_mode, &
               h, max_step_size, max_steps, rtol, atol, itol,  &
               decsol_choice, caller_id, burn_solout, iout, &
               ending_x(:), ending_temp, ending_rho, initial_rho, &
               nfcn, njac, nstep, naccpt, nrejct, time_doing_net, time_doing_eos, ierr)
            write(*,*)
            write(*,1) 'initial_rho', initial_rho
            write(*,1) 'ending_temp', ending_temp
            write(*,1) 'ending_rho', ending_rho
         else
            call net_1_zone_burn( &
               handle, solver_choice, species, num_reactions, 0d0, burn_tend, xin, clip, &
               num_times, times, log10Ts_f, log10Rhos_f, etas_f, &
               dxdt_source_term, rate_factors, category_factors, std_reaction_Qs, std_reaction_neuQs, &
               screening_mode, theta_e_for_graboske_et_al, & 
               h, max_step_size, max_steps, rtol, atol, itol, & 
               decsol_choice, caller_id, burn_solout, iout, ending_x, & 
               nfcn, njac, nstep, naccpt, nrejct, time_doing_net, ierr)
         end if
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate

         if (ierr /= 0) then
            write(*,*) 'net_1_zone_burn ierr', ierr
            stop 1
         end if
         
         write(*,*)
         write(*,*)
         write(*,2) 'number of species', species
         write(*,1) 'final abundances'
         call show_X(ending_x(:),.true.)
	      write(*,*)
         write(*,2) 'nfcn', nfcn
         write(*,2) 'njac', njac
         write(*,2) 'naccpt', naccpt
         write(*,2) 'nrejct', nrejct
         write(*,*)
         write(*,2) 'number of steps', nstep
         write(*,*)
         write(*,*)
         write(*,1) 'output file ' // trim(data_filename)
         write(*,*)
         write(*,'(/,a30,99f18.3,/)') 'runtime (seconds)', dt
         write(*,*)
         
         if (save_final_abundances) then
            iounit = 33
            write(*,*) 'save final abundances to ' // trim(final_abundances_filename)
            open(unit=iounit, file=trim(final_abundances_filename), iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open final_abundances_filename ' // trim(final_abundances_filename)
            else
               write(iounit,2) 'species', species
               do j = 1, species
         	      write(iounit,1) trim(chem_isos% name(chem_id(j))), max(0d0,ending_x(j))
               end do
               close(iounit)
            end if
         end if
         
         
         deallocate( &
            rate_factors, ending_x, &
            x_initial, x_previous, times, &
            log10Ts_f, log10Rhos_f, &
            etas_f, log10Ps_f, &
            peak_abundance, peak_time)
         
      	
      	contains
      	
      	
      	subroutine read_X(ierr)
            use utils_def
            use utils_lib
      	   integer, intent(out) :: ierr
            character (len=256) :: buffer, string
            integer :: i, n, iounit, t, num_isos, id, k
            
            include 'formats.dek'
            
      	   iounit = 33
      	   write(*,*) 'read initial abundances from ' // trim(initial_abundances_filename)
            open(unit=iounit, file=trim(initial_abundances_filename), &
               action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open'
               return
            end if
            
            n = 0
            i = 0
            t = token(iounit, n, i, buffer, string)
            if (t /= name_token .or. string /= 'species') then
               write(*,*) 'expect to find specification of number of species at start of file'
               ierr = -1; return
      	   end if
            t = token(iounit, n, i, buffer, string)
            read(string,fmt=*,iostat=ierr) num_isos
            if (t /= name_token .or. ierr /= 0) then
               write(*,*) 'expect to find specification of number of species at start of file'
               ierr = -1; return
      	   end if
      	   if (num_isos /= species) then
               write(*,2) 'expect to find number of species equal to those in current net', species
               ierr = -1; return
      	   end if
            do k = 1, species
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  write(*,*) 'failed to find iso name at start of line: ' // trim(string)
                  ierr = -1; return
               end if
               id = get_nuclide_index(string)
               if (id <= 0) then
                  write(*,*) 'failed to recognize name of iso ' // trim(string)
                  ierr = -1
                  return
               end if
               j = net_iso(id)
         	   if (j <= 0 .or. j > species) then
         	      write(*,*) 'iso not in current net ' // trim(string)
         	      ierr = 1
         	      return
         	   end if
               t = token(iounit, n, i, buffer, string)
               if (t /= name_token) then
                  write(*,*) 'failed to read iso abundance: ' // &
                     trim(chem_isos% name(id)) // ' ' // trim(string)
                  ierr = -1; return
               end if
               read(string,fmt=*,iostat=ierr) xin(j)
               if (ierr /= 0) then
                  write(*,*) 'failed to read iso abundance: ' &
                     // trim(chem_isos% name(id)) // ' ' // trim(string)
               end if
            end do
      	   close(iounit)
      	end subroutine read_X
      	
	
      	subroutine show_X(X,show_peak)
      	   double precision :: X(:)
      	   logical, intent(in) :: show_peak
      	   include 'formats.dek'
      	   integer :: j
      	   double precision :: xsum
      	   xsum = 0
         	do j=1, species
         	   if (x(j) > min_for_show_peak_abundances) &
         	      write(*,1) trim(chem_isos% name(chem_id(j))), x(j)
         	   if (x(j) > 1.1 .or. x(j) < -0.1) then
         	      write(*,1) 'bad x for ' // trim(chem_isos% name(chem_id(j))), x(j)
         	      stop 1
         	   end if
         	   xsum = xsum + x(j)
         	end do
         	write(*,1) 'xsum', xsum
         	write(*,*)
         	if (.not. show_peak) return
         	write(*,*)
         	write(*,1) 'peak x and time'
         	do j=1, species
         	   if (peak_abundance(j) >= min_for_show_peak_abundances) &
         	      write(*,1) trim(chem_isos% name(chem_id(j))), &
         	         peak_abundance(j), peak_time(j)
         	end do
         	write(*,*)
         end subroutine show_X


      end subroutine Do_One_Zone_Burn


      subroutine burn_solout( &
               step, told, time, n, x, rwork_y, iwork_y, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         use num_lib, only: safe_log10
         integer, intent(in) :: step, n, lrpar, lipar
         double precision, intent(in) :: told, time
         double precision, intent(inout) :: x(n)
         ! x can be modified if necessary to keep it in valid range of possible solutions.
         double precision, intent(inout), target :: rpar(lrpar), rwork_y(*)
         integer, intent(inout), target :: ipar(lipar), iwork_y(*)
         interface
            include 'num_interp_y.dek'
         end interface
         integer, intent(out) :: irtrn ! < 0 causes solver to return to calling program.


         double precision :: logRho, logT
         double precision :: eps_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, &
           eps_nuc_categories(num_rvs, num_categories)
         double precision, dimension(:, :), pointer :: reaction_eps_nuc
         double precision, dimension(:, :), pointer :: rate_screened, rate_raw

         double precision :: Rho, T, xsum, d_eps_nuc_dx(species), dx, enuc,  &
               dt, burn_ergs, energy, enthalpy, burn_ergs_total, &
               xh, xhe, approx_abar, approx_zbar
         
         integer :: info, i, j, lwork, adjustment_iso, cid, ierr
         double precision, dimension(species) :: dabar_dx, dzbar_dx, eps_nuc_dx
         double precision, pointer :: work(:), rate_factors(:), category_factors(:)

         logical, parameter :: reuse_given_rates = .false.

         include 'formats.dek'
         
         irtrn = 0
         if (time == 0) return
         
         x(1:n) = max(0d0, min(1d0, x(1:n)))
         xsum = sum(x(1:n))
         x(1:n) = x(1:n)/xsum
      	
      	if (step > 1 .and. mod(step,50) == 0) write(*,2) 'step, time', step, time

         info = 0
         lwork = net_work_size(handle, info) 
         if (info /= 0) stop 1
         
         allocate(work(lwork),  &
               rate_factors(num_reactions), category_factors(num_categories), &
               rate_screened(num_rvs, num_reactions),  &    
               reaction_eps_nuc(num_rvs, num_reactions),  &     
               rate_raw(num_rvs, num_reactions), stat=info)
         if (info /= 0) stop 2
         
         T = burn_temp
         Rho = burn_Rho
         logT = log10(T)
         logRho = log10(Rho)
         xin = x

         call composition_info( &
               species, chem_id, xin, xh, xhe, abar, zbar, z2bar, ye, &
               approx_abar, approx_zbar, xsum, dabar_dx, dzbar_dx)
         
         rate_factors(:) = 1
         category_factors(:) = 1
         call net_get(handle, species, num_reactions,  &
               xin, T, logT, Rho, logRho,  &
               abar, zbar, z2bar, ye, eta, rate_factors, category_factors,  &
               std_reaction_Qs, std_reaction_neuQs, &
               eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx,  &
               dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx,  &
               screening_mode, theta_e_for_graboske_et_al,     &
               rate_screened, rate_raw, reuse_given_rates,  &
               reaction_eps_nuc, eps_nuc_categories, eps_neu_total, &
               lwork, work, info)
         if (info /= 0) then
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*, *) 'bad return from net_get'
            write(*, *) trim(alert_message)
            stop 1
         end if
         deallocate(work, rate_factors, category_factors, rate_screened, reaction_eps_nuc, rate_raw)
         
         dt = time - told

         ! set burn_ergs according to change from initial abundances
         burn_neu_total = burn_neu_total + eps_neu_total*dt
         burn_ergs = 0
         eps_nuc = 0
         xsum = 0
         do i=1,species
            cid = chem_id(i)
            dx = x(i) - x_initial(i)
            xsum = xsum + x(i)
            burn_ergs = burn_ergs +  &
               (chem_isos% binding_energy(cid) - chem_isos% Z(cid)*del_Mp -  &
                  chem_isos% N(cid)*del_Mn)*dx/chem_isos% W(cid)
            
            
            if (.false.) then
               write(*,4) 'x_current', step, i, cid, x(i)
               write(*,4) 'x_initial', step, i, cid, x_initial(i)
               write(*,4) 'dx', step, i, cid, dx
               write(*,4) 'burn_ergs', step, i, cid, burn_ergs
               write(*,4) 'dxdt', step, i, cid, dxdt(i)
               do j=1,n
                  write(*,3) 'd_dxdt_dx', i, j, d_dxdt_dx(i,j)
               end do
            end if
            
            
            
            dx = x(i) - x_previous(i)
            eps_nuc = eps_nuc +  &
               (chem_isos% binding_energy(cid) - chem_isos% Z(cid)*del_Mp -  &
                  chem_isos% N(cid)*del_Mn)*dx/chem_isos% W(cid)
         end do
         eps_nuc = eps_nuc*Qconv/dt - eps_neu_total
         burn_ergs_total = burn_ergs*Qconv - burn_neu_total
         burn_ergs = burn_ergs_total - burn_neu_total
         
         if (.false.) then
            write(*,1) 'eps_nuc', eps_nuc
            write(*,1) 'burn_ergs', burn_ergs
            write(*,1) 'burn_ergs_total', burn_ergs_total
            write(*,*)
            !stop
         end if

         energy = 0
         enthalpy = 0
         burn_logT=0; burn_logRho=0; burn_lnS=0; burn_lnE=0
         
         x_previous(1:species) = x(1:species)
      	
      	if (time < data_output_min_t) return

         write(io_out,'(i7,99(1pe26.16,1x))',advance='no') &
            step, &
            safe_log10(burn_ergs_total), &
            safe_log10(burn_ergs), &
            safe_log10(burn_neu_total), &
            safe_log10(eps_nuc + eps_neu_total), &
            safe_log10(eps_nuc), &
            safe_log10(eps_neu_total), &
            time, safe_log10(time), safe_log10(time/secyer), &
            time - told, safe_log10(time - told), ye, xsum-1
         do i=1,num_names_of_isos_to_show
            j = net_iso(burn_isos_to_show(i))
            if (j == 0) cycle
            write(io_out,'(1pe26.16,1x)',advance='no') safe_log10(x(j))
         end do
         write(io_out,*) 

      	do j=1, species
      	   if (x(j) > peak_abundance(j)) then
      	      peak_abundance(j) = x(j)
      	      peak_time(j) = time
      	   end if
      	end do

      end subroutine burn_solout
      
      
      subroutine test_net_setup(net_file_in, logT)
         character (len=*), intent(in) :: net_file_in
         double precision, intent(in) :: logT
         integer :: info, i
         
         include 'formats.dek'
         
         net_file = net_file_in

         call net_init(mesa_data_dir, info)
         if (info /= 0) stop 1
         
         if (rattab_logT_lower_bound <= 0) rattab_logT_lower_bound = logT
         if (rattab_logT_upper_bound <= 0) rattab_logT_upper_bound = logT
         if (rattab_logT_upper_bound < rattab_logT_lower_bound) then
            write(*,1) 'rattab_logT_upper_bound', rattab_logT_upper_bound
            write(*,1) 'rattab_logT_lower_bound', rattab_logT_lower_bound
            stop 'lower > upper'
         end if
         call set_rattab_range(rattab_logT_lower_bound, rattab_logT_upper_bound)
         
         handle = alloc_net_handle(info)
         if (info /= 0) then
            write(*,*) 'alloc_net_handle failed'
            stop 2
         end if
         
         call net_start_def(handle, info)
         if (info /= 0) then
            write(*,*) 'net_start_def failed'
            stop 2
         end if
         
         call read_net_file(net_file, handle, info)
         if (info /= 0) then
            write(*,*) 'read_net_file failed ', trim(net_file)
            stop 2
         end if
         
         call net_finish_def(handle, info)
         if (info /= 0) then
            write(*,*) 'net_finish_def failed'
            stop 2
         end if
         
         call net_ptr(handle, g, info)
         if (info /= 0) then
            write(*,*) 'net_ptr failed'
            stop 2
         end if
         
         species = g% num_isos
         num_reactions = g% num_reactions

         allocate(which_rates(rates_reaction_id_max))
         which_rates(:) = which_rates_choice
         call net_set_which_rates(handle, which_rates, info)
         if (info /= 0) then
            write(*,*) 'net_set_which_rate_f17pg failed'
            stop 2
         end if

         call net_setup_tables(handle, 'rate_tables', cache_suffix, info)
         if (info /= 0) then
            write(*,*) 'net_setup_tables failed'
            stop 2
         end if
         
         call get_chem_id_table(handle, species, chem_id, info)
         if (info /= 0) then
            write(*,*) 'get_chem_id_table failed'
            stop 2
         end if
         
         call get_net_iso_table(handle, net_iso, info)
         if (info /= 0) then
            write(*,*) 'get_net_iso_table failed'
            stop 2
         end if
         
         allocate( &
               xin(species), xin_copy(species), d_eps_nuc_dx(species),  &
               dxdt(species), d_dxdt_dRho(species), d_dxdt_dT(species), d_dxdt_dx(species, species))
     
      end subroutine test_net_setup


      end module mod_one_zone_support



      program one_zone_burn
      use chem_lib
      use net_def
      use net_lib
      use rates_lib, only: rates_init
      use weak_lib, only: weak_init
      use reaclib_lib, only: reaclib_init
      use screen_def
      use const_lib
      use utils_lib
      use mtx_def

      use mod_one_zone_support
      
      implicit none
      
   	integer :: ierr, unit
   	character(len=256) :: net_name, filename

      namelist /one_zone/ &
         mesa_data_dir, net_name, num_names_of_isos_to_show, names_of_isos_to_show, &
         final_abundances_filename, save_final_abundances, &
         initial_abundances_filename, read_initial_abundances, &
         num_isos_for_Xinit, names_of_isos_for_Xinit, values_for_Xinit, uniform_Xinit, &
         burn_tend, burn_h, burn_rho, burn_temp, burn_rtol, burn_atol, &
         min_for_show_peak_abundances, data_output_min_t, data_filename, &
         which_solver, screening_mode, which_rates_choice, &
         theta_e_for_graboske_et_al, data_heading_line, show_net_reactions_info, &
         rattab_logT_lower_bound, rattab_logT_upper_bound, max_steps, &
         decsol_switch, small_mtx_decsol, large_mtx_decsol, &
         burn_at_constant_P, starting_temp, pressure, cache_suffix
   	
   	include 'formats.dek'
   	
      filename = 'inlist_one_zone_burn'
      
      ! set defaults
      
      mesa_data_dir = '../../data'   	
      net_name = 'test.net'
      cache_suffix = '0'
      final_abundances_filename = ''
      save_final_abundances = .false.
      initial_abundances_filename = ''
      read_initial_abundances = .false.
   	burn_tend = 10 ! seconds
   	burn_h = 0
   	burn_rho = 2d5
   	burn_temp = 2.6d9
      burn_rtol = 1d-8
      burn_atol = 1d-9
      show_net_reactions_info = .false.
      decsol_switch = 15
         ! if current number of species <= switch,
            ! then use small_mtx_decsol,
            ! else use large_mtx_decsol.
      small_mtx_decsol = 'lapack'
      large_mtx_decsol = 'klu'
      which_solver = 'seulex'
      rattab_logT_lower_bound = -1
      rattab_logT_upper_bound = -1
      max_steps = 10000
      uniform_Xinit = .false.
      burn_at_constant_P = .false.
      starting_temp = -1
      pressure = -1
      
      min_for_show_peak_abundances = 1d-3 ! show if peak is > this
      
      data_filename = 'one_zone_burn.data'
      data_output_min_t = -99
      
      num_names_of_isos_to_show = 0

      num_isos_for_Xinit = 4
      names_of_isos_for_Xinit(1:num_isos_for_Xinit) = (/ &
         'he4', 'c12', 'n14', 'o16' /)
      values_for_Xinit(1:num_isos_for_Xinit) = (/ &
         0.95d0, 0.005d0, 0.035d0, 0.010d0 /)
      
      screening_mode = extended_screening
      theta_e_for_graboske_et_al = 1
      which_rates_choice = rates_NACRE_if_available


      ! read inlist
      
      unit=alloc_iounit(ierr)
      if (ierr /= 0) stop 'failed in alloc_iounit'

      open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
      if (ierr /= 0) then
         write(*, *) 'Failed to open control namelist file ', trim(filename)
         stop 1
      else
         read(unit, nml=one_zone, iostat=ierr)  
         close(unit)
         if (ierr /= 0) then
            write(*, *) 'Failed while trying to read control namelist file ', trim(filename)
            write(*, '(a)') &
               'The following runtime error message might help you find the problem'
            write(*, *) 
            open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
            read(unit, nml=one_zone)
            stop 1
         end if  
      end if
      call free_iounit(unit)
      
      ! do initialization
      
      call const_init

   	ierr = 0
   	call chem_init(mesa_data_dir, 'isotopes.data_approx', ierr)
   	if (ierr /= 0) then
   	   write(*,*) 'chem_init failed'
   	   stop 1
   	end if
   	
   	call weak_init(mesa_data_dir, ierr)   
   	if (ierr /= 0) then
   	   write(*,*) 'weak_init failed'
   	   stop 1
   	end if
         
      call reaclib_init(mesa_data_dir, ierr)   
   	if (ierr /= 0) then
   	   write(*,*) 'reaclib_init failed'
   	   stop 1
   	end if
   	
   	allocate(net_iso(num_chem_isos), chem_id(num_chem_isos))

      call rates_init(mesa_data_dir, 'reactions.list', ierr)
      if (ierr /= 0) stop 1
         
      open(unit=io_out, file=trim(data_filename), action='write', iostat=ierr)
      if (ierr /= 0) stop 1
      
      call Do_One_Zone_Burn(net_name)
         
      open(unit=io_out, file=trim(data_filename), action='write', iostat=ierr)
      if (ierr /= 0) stop 1
      
      end program one_zone_burn