! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_def
      
      implicit none
      
      
      ! nets
      
         ! the 'basic net' is for standard hydrogen and helium burning.
         
            ! it includes 3 PP chains, 4 CNO cycles, triple alpha, and alpha captures up to mg24.      
            ! the 8 isotopes included are h1, he3, he4, c12, n14, o16, ne20, and mg24.
            ! see basic_net.dek for the details
               
         ! to go beyond the basics, use any combination of the following extensions.
         
            ! to add C13, see add_c13.dek
            ! to add Ne22, see add_ne22.dek
            ! to add O18 and Ne22, see add_o18_and_ne22.dek
            ! for more PP reactions, see add_pp_extras.dek
            ! for more CNO reactions including hot CNO, see add_cno_extras.dek
            ! for carbon/oxygen burning, see add_co_burn.dek
            ! for links of alpha chain to ni56, see add_alpha_s32.dek, etc.
            
         ! there are also several special purpose nets you can use.
            
            ! 'net_18_to_mg24' includes 18 isotopes from h1 to mg24.
            ! it gives you all the energy generation you need
            ! for hydrogen and helium burning, so for stars with less 
            ! than 6-8 Msun this is good to get you to the WD stage.
            ! includes basic PP chains, CNO cycles, and helium burning.
            
            ! 'net_16_to_ni56' uses only the basic 8 isotopes plus
            ! 8 more "multi-alpha" isotopes from si28 to ni56
            ! to cover evolution of massive stars from zams 
            ! through advance burning by alpha captures up to ni56.
            
            ! 'net_18_to_ni56' adds o18 and ne22 to net_16_to_ni56.
      


   ! reactions

      
      ! predefined nets and extensions      
         
         integer, parameter :: bit_for_Basic = 0  ! net number bits start at 0
         integer, parameter :: num_Basic = 2**bit_for_Basic
         integer, parameter :: num_isos_for_Basic = 8
            
         ! bits 1 to 29 are for extensions to the basic net -- only some numbers are used.
         
         integer, parameter :: bit_for_PP_extras = 1  
         integer, parameter :: bit_for_CNO_extras = 2  
         integer, parameter :: bit_for_C13 = 3  
         integer, parameter :: bit_for_O18_and_Ne22 = 4  
         integer, parameter :: bit_for_CO_burn = 5  

         integer, parameter :: bit_for_alpha_to_S32 = 10
         integer, parameter :: bit_for_alpha_to_Ar36 = 11
         integer, parameter :: bit_for_alpha_to_Ca40 = 12
         integer, parameter :: bit_for_alpha_to_Ti44 = 13 
         integer, parameter :: bit_for_alpha_to_Cr48 = 14  
         integer, parameter :: bit_for_alpha_to_Fe52 = 15
         integer, parameter :: bit_for_alpha_to_Ni56 = 16

         integer, parameter :: bit_for_add_Na23_to_alpha_Mg24 = 17
         integer, parameter :: bit_for_add_Al27_to_alpha_Si28 = 18
         integer, parameter :: bit_for_add_P31_to_alpha_S32 = 19
         integer, parameter :: bit_for_add_Cl35_to_alpha_Ar36 = 20
         integer, parameter :: bit_for_add_K39_to_alpha_Ca40 = 21
         integer, parameter :: bit_for_add_Sc43_to_alpha_Ti44 = 22
         integer, parameter :: bit_for_add_V47_to_alpha_Cr48 = 23
         integer, parameter :: bit_for_add_Mn51_to_alpha_Fe52 = 24
         ! note: Co55 is automatically included in alpha_Ni56

         integer, parameter :: bit_for_specific_nets = 30 
         
         ! "special nets" are nets that are not combos of the basic net plus extensions
         ! NOTE: all nets are required to include the basic net isos.
         ! beyond that, the special nets can do anything they want.
         ! 2**bit_for_specific_nets = 2**30 = 1073741824
         
         integer, parameter :: num_19_to_Ni56 = 2**bit_for_specific_nets ! 1073741824
         integer, parameter :: num_18_to_Mg24 = 2**bit_for_specific_nets+1 ! 1073741825

         
      type Net_General_Info ! things that are constant for the particular net
      ! it is okay to have multiple threads using the same instance of this simultaneously.

         integer :: num_isos ! total number in current net            
         integer :: num_reactions ! total number of reactions for current net
         
         character (len=32) :: cache_suffix

         ! isotopes
         integer, pointer :: net_iso(:) ! maps chem id to net iso number
         ! index from 1 to num_chem_isos
         ! value is 0 if the iso is not in the current net
         ! else is value between 1 and num_isos in current net
         integer, pointer :: chem_id(:) ! maps net iso number to chem id
         ! index from 1 to num_isos in current net
         ! value is between 1 and num_chem_isos         

         ! reactions

         integer, pointer :: which_rates(:) ! maps reaction id to small integer indicating choice for rate
                  
         integer, pointer :: net_reaction(:) ! maps reaction id to net reaction number
         ! index from 1 to num_net_reactions
         ! value is 0 if the reaction is not in the current net
         ! else is value between 1 and num_reactions in current net
         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         ! index from 1 to num_reactions in current net
         ! value is between 1 and num_net_reactions     

         
         ! extra info
         
         ! rates cutoff smoothly for logT < logTcut_lim
         double precision :: logTcut_lim
         ! rates are zero logT < logTcut_lo
         double precision :: logTcut_lo
   
         ! the following is private info for the implementation
         
         ! tables for graboske screening
         double precision, pointer :: zg1(:) ! (num_reactions)
         double precision, pointer :: zg2(:) ! (num_reactions)
         double precision, pointer :: zg3(:) ! (num_reactions)
         double precision, pointer :: zg4(:) ! (num_reactions)
         
         ! tables for screen5
         double precision, pointer :: zs13(:) ! (num_reactions) ! zs13 = (z1+z2)**(1./3.)
         double precision, pointer :: zhat(:) ! (num_reactions)
         double precision, pointer :: zhat2(:) ! (num_reactions)
         double precision, pointer :: lzav(:) ! (num_reactions)
         double precision, pointer :: aznut(:) ! (num_reactions)
         double precision, pointer :: zs13inv(:) ! (num_reactions) ! zs13inv = 1 / zs13
   
         ! info for evaluation of the raw reaction rates
         double precision, pointer :: rattab(:,:) ! (num_reactions, nrattab)
         double precision, pointer :: ttab(:) ! (nrattab)
         double precision, pointer :: logttab(:) ! (nrattab)
         double precision, pointer :: rattab_f(:,:,:) ! (4, nrattab, num_reactions) ! for interpolation

         ! info for evaluation of weak rates by weaklib
         integer :: num_weaklib_rates
         integer, pointer :: weaklib_ids(:), weak_reaction_index(:), weak_reaction_num(:)
         
         ! top level file name for net
         character (len=256) :: net_filename
         
         ! timing
         logical :: doing_timing
         logical :: doing_derivs_timing
         ! the following are sums of results from system_clock.  divide by clock_rate to get seconds.
         ! must set all of these to 0 before change doing_timing to true.
         integer :: clock_net_eval
         integer :: clock_net_weak_rates
         integer :: clock_net_rate_tables
         integer :: clock_net_screen
         integer :: clock_net_derivs
         integer :: clock_derivs_select
         integer :: clock_derivs_setup
         integer :: clock_derivs_general
         integer :: clock_net_get
         
         ! bookkeeping
         integer :: handle
         logical :: net_has_been_defined
         logical :: in_use

      end type Net_General_Info
               
               
      type Net_Info
         ! this is private working storage for the nuclear reaction calculations

         type (Net_General_Info), pointer  :: g

         double precision, pointer :: reaction_Qs(:) ! if null, use standard values         
         double precision, pointer :: reaction_neuQs(:) ! if null, use standard values

         ! molar fractions and their rates of change
         double precision, pointer :: y(:) ! units [moles/gram]     (num_isos)
         double precision, pointer :: dydt(:, :) ! units [moles/(gram-second)]  (num_rvs, num_isos)
         double precision, pointer :: d_dydt_dy(:, :) ! units [1/second] (num_isos, num_isos)

         integer :: screening_mode
         double precision :: theta_e_for_graboske_et_al
         double precision, pointer :: graboske_cache(:, :, :)

         double precision, pointer :: rate_screened(:, :) ! (num_rvs, num_rates)
         ! the units here depend on the number of reactants.
         ! in all cases, the rate_screened times as many molar fractions as there are reactants
            ! gives a number with the same units as dy/dt.
         ! so for a 2-body reaction, there are 2 Y factors, each with units [moles/gram]
            ! and the rate_screened units for such a reaction are [grams/(mole-sec)], 
            ! which when multiplied by [moles/gram]^2 gives the same units as dydt.
         ! for a 1-body reaction (e.g., a decay), there is only 1 Y factor, so the units are [1/second].
         ! similarly, a 3 body reaction will have rate_screened with units of [gram^2/(mole^2-sec)].

         double precision, pointer :: rate_raw(:, :) ! (num_rvs, num_rates)
         ! raw rates are unscreened (but include density factors)

         double precision, pointer :: reaction_eps_nuc(:, :) ! (num_rvs, num_rates)
         
         double precision, pointer :: d_eps_nuc_dy(:) ! (num_isos)

         double precision, pointer :: eps_nuc_categories(:, :) ! (num_rvs, num_categories)
         ! eps_nuc subtotals for each reaction category

         double precision :: eps_neu_total
         
         ! weaklib results
         double precision, dimension(:), pointer ::
     >      ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, 
     >      lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, 
     >      lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, 
     >      lambda, dlambda_dlnT, dlambda_dlnRho, 
     >      Q, dQ_dlnT, dQ_dlnRho, 
     >      Qneu, dQneu_dlnT, dQneu_dlnRho
     
         ! args for get1_derivs
         double precision ::
     >         rpen, rnep, spen, snep, 
     >         d_rhe4_breakup_dneut, d_rhe4_breakup_dprot,
     >         d_rhe4_rebuild_dneut, d_rhe4_rebuild_dprot,
     >         d_rfe52neut_to_fe54_dneut, d_rfe54g_to_fe52_dneut,
     >         d_rfe54prot_to_ni56_dprot, d_rni56gprot_to_fe54_dprot,
     >         d_rfe52aprot_to_fe54_dprot, d_rfe54prot_to_fe52_dprot,
     >         d_rfe52aprot_to_ni56_dprot, d_rni56gprot_to_fe52_dprot,
     >         d_rfe54neut_to_fe56_dneut, d_rfe56g_to_fe54_dneut

      end type Net_Info
      

   ! private to the implementation
      integer, parameter :: max_net_handles = 10
      type (Net_General_Info), target :: net_handles(max_net_handles)
      
      character (len=256) :: net_dir

      
      contains


      subroutine net_def_init(data_dir)
         character (*), intent(in) :: data_dir
         integer :: i
         net_dir = trim(data_dir) // '/net_data'
         do i=1, max_net_handles
            net_handles(i)% handle = i
            net_handles(i)% in_use = .false.
            net_handles(i)% net_has_been_defined = .false.
            net_handles(i)% num_isos = 0
            net_handles(i)% num_reactions = 0
         end do
      end subroutine net_def_init


      integer function do_alloc_net(ierr)
         use rates_def
         use alert_lib, only:alert
         integer, intent(out) :: ierr
         integer :: i
         type (Net_General_Info), pointer :: g
         ierr = 0
         do_alloc_net = -1
!$omp critical (net_handle)
         do i = 1, max_net_handles
            if (.not. net_handles(i)% in_use) then
               net_handles(i)% in_use = .true.
               do_alloc_net = i
               exit
            end if
         end do
!$omp end critical (net_handle)
         if (do_alloc_net == -1) then
            ierr = -1
            call alert(ierr, 'no available net handle')
            return
         end if
         if (net_handles(do_alloc_net)% handle /= do_alloc_net) then
            ierr = -1
            call alert(ierr, 'broken handle for net')
            return
         end if
         g => net_handles(do_alloc_net)
         nullify(g% which_rates)
         nullify(g% net_iso)
         nullify(g% chem_id)
         nullify(g% net_reaction)
         nullify(g% reaction_id)
         g% net_has_been_defined = .false.
         g% doing_timing = .false.
         g% doing_derivs_timing = .false.
         g% num_isos = 0
         g% num_reactions = 0
         g% logTcut_lo = rattab_tlo
         g% logTcut_lim = rattab_tlo + 0.1d0
         g% cache_suffix = '0'
      end function do_alloc_net
      
      
      subroutine do_free_net(handle)
         use rates_def
         integer, intent(in) :: handle
         type (Net_General_Info), pointer :: g
         if (handle >= 1 .and. handle <= max_net_handles) then
            g => net_handles(handle)
            deallocate(g% which_rates)
            deallocate(g% net_iso)
            deallocate(g% chem_id)
            deallocate(g% net_reaction)
            deallocate(g% reaction_id)
            deallocate(g% zg1)
            deallocate(g% zg2)
            deallocate(g% zg3)
            deallocate(g% zg4)
            deallocate(g% zs13)
            deallocate(g% zhat)
            deallocate(g% zhat2)
            deallocate(g% lzav)
            deallocate(g% aznut)
            deallocate(g% zs13inv)
            deallocate(g% rattab)
            deallocate(g% ttab)
            deallocate(g% logttab)
            deallocate(g% rattab_f)
            if (associated(g% weaklib_ids)) deallocate(g% weaklib_ids)
            if (associated(g% weak_reaction_num)) deallocate(g% weak_reaction_num)
            if (associated(g% weak_reaction_index)) deallocate(g% weak_reaction_index)
            g% in_use = .false.
            g% net_has_been_defined = .false.
            g% num_isos = 0
            g% num_reactions = 0
            g% num_weaklib_rates = 0
         end if
      end subroutine do_free_net
      

      subroutine get_net_ptr(handle, g, ierr)
         use alert_lib, only:alert
         integer, intent(in) :: handle
         type (Net_General_Info), pointer :: g
         integer, intent(out):: ierr         
         if (handle < 1 .or. handle > max_net_handles) then
            ierr = -1
            call alert(ierr, 'invalid net handle')
            return
         end if
         g => net_handles(handle)
         ierr = 0
      end subroutine get_net_ptr


      integer function get_net_timing_total(g)
         type (Net_General_Info), pointer :: g
         get_net_timing_total = 0
         if (.not. g% doing_timing) return
         get_net_timing_total = 
     >      g% clock_net_eval +
     >      g% clock_net_weak_rates +
     >      g% clock_net_rate_tables + 
     >      g% clock_net_screen + 
     >      g% clock_net_derivs
      end function get_net_timing_total


      subroutine zero_net_timing(g)
         type (Net_General_Info), pointer :: g
         g% clock_net_eval = 0
         g% clock_net_weak_rates = 0
         g% clock_net_rate_tables = 0 
         g% clock_net_screen = 0 
         g% clock_net_derivs = 0
         
         g% clock_derivs_setup = 0
         g% clock_derivs_select = 0
         g% clock_derivs_general = 0
         g% clock_net_get = 0
      end subroutine zero_net_timing


      end module net_def

