! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_screen
      use const_def
      use chem_def
      use net_def
      use rates_def
      
      implicit none
      

      contains


      subroutine make_screening_tables(net, ierr)
         type (Net_Info), pointer :: net
         integer, intent(out) :: ierr
         double precision :: y(num_chem_isos)
         y = 0
         call screen_net(
     >      net, num_chem_isos, y, 1d0, 1d0, 0d0, 0d0, .true., 0d0, 0d0, 0d0, 1d0, ierr)
      end subroutine make_screening_tables
      

      subroutine screen_net(
     >         n, num_isos, y, temp, den, logT, logRho, init, 
     >         zbar, abar, z2bar, ye, ierr)

         !..this routine computes the screening factors
         !..and applies them to the raw reaction rates, 
         !..producing the final reaction rates used by the
         !..right hand sides and jacobian matrix elements

         use screen_def, only: Screen_Info
         use screen_lib, only: screen_set_context
         
         type (Net_Info), pointer :: n
         integer, intent(in) :: num_isos
         double precision, intent(in) :: y(num_isos), temp, den, logT, logRho, 
     >         zbar, abar, z2bar, ye
         logical, intent(in) :: init
         integer, intent(out) :: ierr

         type (Net_General_Info), pointer  :: g
         type (Screen_Info), target :: scrn_info
         type (Screen_Info), pointer :: sc
         integer :: jscr, num_reactions, i, ir, j
         double precision :: Tfactor, iso_z(num_isos)
         logical :: all_okay
         
         include 'formats.dek'
         
         
         sc => scrn_info
         
         if (.not. init) then
            iso_z(:) = chem_isos% Z(n% g% chem_id(:))
            call screen_set_context(
     >         sc, temp, den, logT, logRho, zbar, abar, z2bar, 
     >         n% screening_mode, n% graboske_cache, 
     >         n% theta_e_for_graboske_et_al, num_isos, y, iso_z)
         end if
   
         g => n% g
         num_reactions = g% num_reactions
         
         ierr = 0

         n% rate_screened(1:num_rvs, 1:num_reactions) = 
     >         n% rate_raw(1:num_rvs, 1:num_reactions)
         
         jscr = 0
         do i = 1, num_reactions
            ir = g% reaction_id(i)
            if (ir == 0) then
               write(*,*) 'g% reaction_id(i) == 0', i, num_reactions
               stop 'screen_net'
            end if
            if (reaction_screening_info(3,ir) > 0) then
               call eval_screen_triple( 
     >            init, jscr, 
     >            reaction_screening_info(1,ir), 
     >            reaction_screening_info(2,ir),  
     >            reaction_screening_info(3,ir),  
     >            i, n, sc, ir, ierr)
            else if (reaction_screening_info(2,ir) > 0) then
               call eval_screen_pair( 
     >            init, jscr, 
     >            reaction_screening_info(1,ir), 
     >            reaction_screening_info(2,ir),  
     >            i, n, sc, ir, ierr)
               if (.false. .and. .not. init .and. reaction_Name(ir) == 'r_b8_pg_c9') then
                  write(*,2) trim(reaction_Name(ir)) // ' screen factor', 
     >                  ir, n% rate_screened(i_rate, i)/n% rate_raw(i_rate,i)
                  write(*,2) trim(reaction_Name(ir)) // ' raw rate', 
     >                  ir, n% rate_raw(i_rate, i)
                  write(*,2) trim(reaction_Name(ir)) // ' screened rate', 
     >                  ir, n% rate_screened(i_rate, i)
                  write(*,2) trim(reaction_Name(ir)) // ' screened dr_dlnRho', 
     >                  ir, n% rate_screened(i_rate_dRho, i)*den
               end if
            end if
            if (ierr /= 0) return
         end do
         
         call set_combo_screen_rates(n, num_isos, ye, y, sc)
               
         if (nrattab > 1 .and. (logT < g% logTcut_lim .or. logT <= g% logTcut_lo)) then
            ! rates cutoff smoothly for logT < logTcut_lim
            if (logT <= g% logTcut_lo) then
               n% rate_screened(1:num_rvs, 1:num_reactions) = 0
            else
               Tfactor = (logT - g% logTcut_lo)/(g% logTcut_lim - g% logTcut_lo)
               Tfactor = 0.5d0*(1 - cos(pi*Tfactor**2))
               n% rate_screened(1:num_rvs, 1:num_reactions) = 
     >            Tfactor * n% rate_screened(1:num_rvs, 1:num_reactions)
            end if
         end if
         
         if (.false.) then
            do j = 1, num_reactions
               write(*,2) 'rate screened ' // trim(reaction_Name(g% reaction_id(j))), 
     >            j, n% rate_screened(i_rate, j), n% rate_raw(i_rate, j)
            end do
         end if

      end subroutine screen_net


      subroutine set_combo_screen_rates(n, num_isos, ye, y, sc)
         use screen_def, only: Screen_Info
         type (Net_Info), pointer :: n
         integer, intent(in) :: num_isos
         double precision, intent(in) :: ye, y(num_isos)
         type (Screen_Info), pointer :: sc

         type (Net_General_Info), pointer  :: g
         integer, pointer :: rtab(:)
         double precision :: rateII, rateIII, rsum, fII, fIII

         include 'formats.dek'
          
         g => n% g
         rtab => g% net_reaction
         
         if (rtab(ir34_pp2) /= 0 .and. rtab(ir34_pp3) /= 0) then
            if (n% rate_screened(i_rate, rtab(ir34_pp2)) /= n% rate_screened(i_rate, rtab(ir34_pp3))) then
               write(*,1) 'n% rate_screened(i_rate, rtab(ir34_pp2))', n% rate_screened(i_rate, rtab(ir34_pp2))
               write(*,1) 'n% rate_screened(i_rate, rtab(ir34_pp3))', n% rate_screened(i_rate, rtab(ir34_pp3))
               write(*,*) 'should be the same'
               stop 'set_combo_screen_rates'
            end if
            if (rtab(ir_be7_ec_li7) /= 0) then
               rateII  = ye * n% rate_screened(i_rate, rtab(ir_be7_ec_li7))
            else if (rtab(irbe7ec_li7_aux) /= 0) then
               rateII  = ye * n% rate_screened(i_rate, rtab(irbe7ec_li7_aux))
            else
               write(*,*) 'need either rbe7ec or rbe7ec_li7'
               stop 'set_combo_screen_rates'
            end if
            if (rtab(ir_be7_pg_b8) /= 0) then
               rateIII = y(g% net_iso(ih1)) * n% rate_screened(i_rate, rtab(ir_be7_pg_b8))
            else if (rtab(irbe7pg_b8_aux) /= 0) then
               rateIII = y(g% net_iso(ih1)) * n% rate_screened(i_rate, rtab(irbe7pg_b8_aux))
            else
               write(*,*) 'need either rbe7pg or rbe7pg_b8'
               stop 'set_combo_screen_rates'
            end if
            rsum = rateII + rateIII
            if (rsum < 1d-50) then
               fII = 0.5d0
            else
               fII = rateII / rsum
            end if
            fIII = 1d0 - fII
            n% rate_screened(:, rtab(ir34_pp2)) = fII*n% rate_screened(:, rtab(ir34_pp2))
            n% rate_screened(:, rtab(ir34_pp3)) = fIII*n% rate_screened(:, rtab(ir34_pp3))
         end if


! n14(p,g)o15(e+nu)n15(p,a)c12         
         if (rtab(irn14_to_c12) /= 0) 
     >      call rate_for_pg_pa_branches(
     >               n, rtab(irn14pg_aux), rtab(irn15pg_aux), rtab(irn15pa_aux), 
     >               0, rtab(irn14_to_c12))         

! n14(p,g)o15(e+nu)n15(p,g)o16         
         if (rtab(irn14_to_o16) /= 0)
     >      call rate_for_pg_pa_branches(
     >               n, rtab(irn14pg_aux), rtab(irn15pg_aux), rtab(irn15pa_aux), 
     >               rtab(irn14_to_o16), 0)               
      
             
! o16(o16,p)p31(p,a)si28           
         if (rtab(ir1616ppa) /= 0) 
     >      call rate_for_pg_pa_branches(
     >               n, rtab(ir1616p_aux), rtab(irp31pg_aux), rtab(irp31pa_aux), 
     >               0, rtab(ir1616ppa))         
      
! o16(o16,p)p31(p,g)s32           
         if (rtab(ir1616ppg) /= 0) 
     >      call rate_for_pg_pa_branches(
     >               n, rtab(ir1616p_aux), rtab(irp31pg_aux), rtab(irp31pa_aux), 
     >               rtab(ir1616ppg), 0)         

! o16 => c12           
         call rate_for_alpha_gp(
     >               n, iro16gp_aux, irn15pg_aux, irn15pa_aux, 
     >               iro16gp_to_c12)         

! o16 => ne20           
         call rate_for_alpha_ap(
     >               n, iro16ap_aux, irf19pg_aux, irf19pa_aux, 
     >               iro16ap_to_ne20)    

! ne20 => o16           
         call rate_for_alpha_gp(
     >               n, irne20gp_aux, irf19pg_aux, irf19pa_aux, 
     >               irne20gp_to_o16)         
              
! ne20 => mg24           
         call rate_for_alpha_ap(
     >               n, irne20ap_aux, irna23pg_aux, irna23pa_aux, 
     >               irne20ap_to_mg24)         
                                       
! mg24 => ne20           
         call rate_for_alpha_gp(
     >               n, irmg24gp_aux, irna23pg_aux, irna23pa_aux, 
     >               irmg24gp_to_ne20)               
      
! mg24 => si28           
         call rate_for_alpha_ap(
     >               n, irmg24ap_aux, iral27pg_aux, iral27pa_aux, 
     >               irmg24ap_to_si28)         
                                       
! si28 => mg24           
         call rate_for_alpha_gp(
     >               n, irsi28gp_aux, iral27pg_aux, iral27pa_aux, 
     >               irsi28gp_to_mg24)                      
                           
! si28 => s32            
         call rate_for_alpha_ap(
     >               n, irsi28ap_aux, irp31pg_aux, irp31pa_aux, 
     >               irsi28ap_to_s32)         
                
! s32 => si28
         call rate_for_alpha_gp(
     >               n, irs32gp_aux, irp31pg_aux, irp31pa_aux, 
     >               irs32gp_to_si28)         
            
! s32 => ar36
         call rate_for_alpha_ap(
     >               n, irs32ap_aux, ircl35pg_aux, ircl35pa_aux, 
     >               irs32ap_to_ar36)         
               
! ar36 => s32
         call rate_for_alpha_gp(
     >               n, irar36gp_aux, ircl35pg_aux, ircl35pa_aux, 
     >               irar36gp_to_s32)         
                    
! ar36 => ca40
         call rate_for_alpha_ap(
     >               n, irar36ap_aux, irk39pg_aux, irk39pa_aux, 
     >               irar36ap_to_ca40)         

! ca40 => ar36
         call rate_for_alpha_gp(
     >               n, irca40gp_aux, irk39pg_aux, irk39pa_aux, 
     >               irca40gp_to_ar36)         

! ca40 => ti44
         call rate_for_alpha_ap(
     >               n, irca40ap_aux, irsc43pg_aux, irsc43pa_aux, 
     >               irca40ap_to_ti44)         

! ti44 => ca40
         call rate_for_alpha_gp(
     >               n, irti44gp_aux, irsc43pg_aux, irsc43pa_aux, 
     >               irti44gp_to_ca40)         

! ti44 => cr48
         call rate_for_alpha_ap(
     >               n, irti44ap_aux, irv47pg_aux, irv47pa_aux, 
     >               irti44ap_to_cr48)         
            
! cr48 => ti44
         call rate_for_alpha_gp(
     >               n, ircr48gp_aux, irv47pg_aux, irv47pa_aux, 
     >               ircr48gp_to_ti44)         

! cr48 => fe52
         call rate_for_alpha_ap(
     >               n, ircr48ap_aux, irmn51pg_aux, irmn51pa_aux, 
     >               ircr48ap_to_fe52)         
            
! fe52 => cr48
         call rate_for_alpha_gp(
     >               n, irfe52gp_aux, irmn51pg_aux, irmn51pa_aux, 
     >               irfe52gp_to_cr48)                          

! fe52 => ni56
         call rate_for_alpha_ap(
     >               n, irfe52ap_aux, irco55pg_aux, irco55pa_aux, 
     >               irfe52ap_to_ni56)         
            
! ni56 => fe52
         call rate_for_alpha_gp(
     >               n, irni56gp_aux, irco55pg_aux, irco55pa_aux, 
     >               irni56gp_to_fe52)                          


      end subroutine set_combo_screen_rates
      
      
      subroutine eval_screen_pair(init, jscr, i1, i2, i, n, sc, ir, ierr)
         use screen_def, only: Screen_Info
         logical, intent(in) :: init
         integer, intent(inout) :: jscr
         type (Net_Info), pointer :: n
         type (Screen_Info), pointer :: sc
         integer, intent(in) :: i1, i2 ! chem id's for the isotopes
         integer, intent(in) :: i ! rate number
         integer, intent(in) :: ir
         integer, intent(out) :: ierr
         double precision :: sc1a, sc1adt, sc1add, a1, z1, a2, z2
         type (Net_General_Info), pointer  :: g
         include 'formats.dek'
         g => n% g
         a1 = chem_isos% W(i1)
         z1 = dble(chem_isos% Z(i1))
         a2 = chem_isos% W(i2)
         z2 = dble(chem_isos% Z(i2))
         call screening_pair(
     >            init, jscr, n, sc, a1, z1, a2, z2, sc1a, sc1adt, sc1add, ierr)
         if (ierr /= 0) return
         if (init) return
         call set_rate_screening(n, i, sc1a, sc1adt, sc1add)
         
         if (.false. .and. reaction_Name(ir) == 'r_b8_pg_c9') then
            write(*,2) 'scr 2 ' // trim(reaction_Name(ir)) 
     >         // ' ' // trim(chem_isos% name(i1))
     >         // ' ' // trim(chem_isos% name(i2)), 
     >         ir, sc1a, sc1adt, sc1add
         end if
         
      end subroutine eval_screen_pair
      
      
      subroutine eval_screen_triple(init, jscr, i1_in, i2_in, i3_in, i, n, sc, ir, ierr)
         use screen_def, only: Screen_Info
         logical, intent(in) :: init
         integer, intent(inout) :: jscr
         type (Net_Info), pointer :: n
         type (Screen_Info), pointer :: sc
         integer, intent(in) :: i1_in, i2_in, i3_in ! chem id's for the isotopes
         integer, intent(in) :: i ! rate number
         integer, intent(in) :: ir
         integer, intent(out) :: ierr
         integer :: i1, i2, i3, ii
         double precision :: sc1, sc1dt, sc1dd
         double precision :: sc2, sc2dt, sc2dd
         double precision :: scor, scordt, scordd
         double precision :: a1, z1, a2, z2, a3, z3
         type (Net_General_Info), pointer  :: g
         include 'formats.dek'
         ierr = 0
         i1 = i1_in; i2 = i2_in; i3 = i3_in
         g => n% g
         a1 = chem_isos% W(i1)
         z1 = dble(chem_isos% Z(i1))
         a2 = chem_isos% W(i2)
         z2 = dble(chem_isos% Z(i2))
         a3 = chem_isos% W(i3)
         z3 = dble(chem_isos% Z(i3))
         if (z2 == 0) then
            if (z1 == 0) return ! n + n + A
            ! have A + n + B
            ! swap 1 and 2 so have n + A + B
            ii = i2; i2 = i1; i1 = ii
            a1 = chem_isos% W(i1)
            z1 = dble(chem_isos% Z(i1))
            a2 = chem_isos% W(i2)
            z2 = dble(chem_isos% Z(i2))
         end if
         if (z3 == 0) then ! have A + B + n
            ! swap 1 and 3 so have n + A + B
            ii = i1; i1 = i3; i3 = ii
            a1 = chem_isos% W(i1)
            z1 = dble(chem_isos% Z(i1))
            a3 = chem_isos% W(i3)
            z3 = dble(chem_isos% Z(i3))
         end if
         call screening_pair(
     >         init, jscr, n, sc, a2, z2, a3, z3, sc2, sc2dt, sc2dd, ierr)
         if (ierr /= 0) return
         if (z1 == 0) then
            if (init) return
            call set_rate_screening(n, i, sc2, sc2dt, sc2dd)
         
            if (.false.) write(*,2) 'scr 2 ' // trim(reaction_Name(ir)) 
     >            // ' ' // trim(chem_isos% name(i2))
     >            // ' ' // trim(chem_isos% name(i3)), 
     >            ir, sc2
         
            return ! n + (A + B)
         end if
         a2 = a2 + a3
         z2 = z2 + z3
         call screening_pair(
     >         init, jscr, n, sc, a1, z1, a2, z2, sc1, sc1dt, sc1dd, ierr)
         if (init) return
	      scor = sc1*sc2
	      scordt = sc1*sc2dt + sc1dt*sc2
	      scordd = sc1*sc2dd + sc1dd*sc2
         call set_rate_screening(n, i, scor, scordt, scordd)
         
         if (.false.) write(*,2) 'scr 3 ' // trim(reaction_Name(ir))
     >            // ' ' // trim(chem_isos% name(i1))
     >            // ' ' // trim(chem_isos% name(i2))
     >            // ' ' // trim(chem_isos% name(i3)), 
     >         ir, scor
         
      end subroutine eval_screen_triple
      
      
      subroutine screening_pair(
     >         init, jscr, n, sc, a1, z1, a2, z2, scor, scordt, scordd, ierr)
         use screen_lib, only: screen_init_AZ_info, screen_pair
         use screen_def, only: Screen_Info
         logical, intent(in) :: init
         integer, intent(inout) :: jscr
         type (Net_Info), pointer :: n
         type (Screen_Info), pointer :: sc
         double precision, intent(in) :: a1, z1, a2, z2
         double precision, intent(out) :: scor, scordt, scordd
         integer, intent(out) :: ierr
         type (Net_General_Info), pointer  :: g
         g => n% g
         jscr = jscr + 1
         if (init) then
            call screen_init_AZ_info(
     >         a1, z1, a2, z2,
     >         g% zg1(jscr), g% zg2(jscr), g% zg3(jscr), g% zg4(jscr), g% zs13(jscr), 
     >         g% zhat(jscr), g% zhat2(jscr), g% lzav(jscr), g% aznut(jscr), g% zs13inv(jscr),
     >         ierr)
         else
            call screen_pair(
     >         sc, a1, z1, a2, z2, n% screening_mode,
     >         g% zg1(jscr), g% zg2(jscr), g% zg3(jscr), g% zg4(jscr), g% zs13(jscr), 
     >         g% zhat(jscr), g% zhat2(jscr), g% lzav(jscr), g% aznut(jscr), g% zs13inv(jscr),
     >         n% theta_e_for_graboske_et_al, n% graboske_cache, scor, scordt, scordd, ierr) 
         end if         
      end subroutine screening_pair
      
      
      subroutine set_rate_screening(n, i, sc1a, sc1adt, sc1add)
         type (Net_Info), pointer :: n
         integer, intent(in) :: i
         double precision, intent(in) :: sc1a, sc1adt, sc1add
         include 'formats.dek'
         if (i == 0) return         
         n% rate_screened(i_rate, i) = n% rate_raw(i_rate,i)*sc1a
         n% rate_screened(i_rate_dT,i) = 
     >         n% rate_raw(i_rate_dT,i)*sc1a + n% rate_raw(i_rate,i)*sc1adt
         n% rate_screened(i_rate_dRho,i)  = 
     >         n% rate_raw(i_rate_dRho,i)*sc1a + n% rate_raw(i_rate,i)*sc1add
      end subroutine set_rate_screening


      subroutine rate_for_alpha_ap(n, ir_start, irpg, irpa, ir_with_pg)
         type (Net_Info), pointer :: n
         integer, intent(in) :: ir_start, irpg, irpa, ir_with_pg
         type (Net_General_Info), pointer  :: g
         integer, pointer :: rtab(:)
         if (ir_start == 0) return
         g => n% g
         rtab => g% net_reaction
         if (rtab(ir_with_pg) == 0) return
         call rate_for_pg_pa_branches(n, 
     >         rtab(ir_start), rtab(irpg), rtab(irpa), rtab(ir_with_pg), 0)
      end subroutine rate_for_alpha_ap


      subroutine rate_for_alpha_gp(n, ir_start, irpg, irpa, ir_with_pa)
         type (Net_Info), pointer :: n
         integer, intent(in) :: ir_start, irpg, irpa, ir_with_pa         
         type (Net_General_Info), pointer  :: g
         integer, pointer :: rtab(:)
         if (ir_start == 0) return
         g => n% g
         rtab => g% net_reaction
         if (rtab(ir_with_pa) == 0) return
         call rate_for_pg_pa_branches(n, 
     >         rtab(ir_start), rtab(irpg), rtab(irpa), 0, rtab(ir_with_pa))
      end subroutine rate_for_alpha_gp
         

      subroutine rate_for_pg_pa_branches(n, ir_start, irpg, irpa, ir_with_pg, ir_with_pa)
         type (Net_Info), pointer :: n
         integer, intent(in) :: ir_start, irpg, irpa, ir_with_pg, ir_with_pa
            
         double precision :: pg_raw_rate, pa_raw_rate, pg_frac, pa_frac
         double precision :: d_pg_frac_dT, d_pg_frac_dRho, d_pa_frac_dT, d_pa_frac_dRho
         double precision :: r, drdT, drdd
         
         if (ir_start == 0) then
            write(*,*) 'ir_start', ir_start
            if (irpg /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(irpg))) // ' irpg'
            if (irpa /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(irpa))) // ' irpa'
            if (ir_with_pg /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pg))) // ' ir_with_pg'
            if (ir_with_pa /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pa))) // ' ir_with_pa'
            stop 'rate_for_pg_pa_branches'
         end if
         
         if (irpg == 0) then
            write(*,*) 'irpg', irpg
            if (ir_with_pg /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pg))) // ' ir_with_pg'
            if (ir_with_pa /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pa))) // ' ir_with_pa'
            stop 'rate_for_pg_pa_branches'
         end if
         
         if (irpa == 0) then
            write(*,*) 'irpg', irpg
            if (ir_with_pg /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pg))) // ' ir_with_pg'
            if (ir_with_pa /= 0) write(*,*) trim(reaction_Name(n% g% reaction_id(ir_with_pa))) // ' ir_with_pa'
            stop 'rate_for_pg_pa_branches'
         end if
         
         pg_raw_rate = n% rate_raw(i_rate, irpg)
         pa_raw_rate = n% rate_raw(i_rate, irpa)
         
         if (pg_raw_rate + pa_raw_rate < 1d-99) then ! avoid divide by 0
            pg_raw_rate = 1; pa_raw_rate = 1
         end if
         
         pg_frac = pg_raw_rate / (pg_raw_rate + pa_raw_rate)
         pa_frac = 1 - pg_frac
         
         d_pg_frac_dT = 
     >      (pa_raw_rate*n% rate_raw(i_rate_dT, irpg) - pg_raw_rate*n% rate_raw(i_rate_dT, irpa)) /
     >      (pg_raw_rate + pa_raw_rate)**2
         d_pa_frac_dT = -d_pg_frac_dT
         
         d_pg_frac_dRho = 
     >      (pa_raw_rate*n% rate_raw(i_rate_dRho, irpg) - pg_raw_rate*n% rate_raw(i_rate_dRho, irpa)) /
     >      (pg_raw_rate + pa_raw_rate)**2
         d_pa_frac_dRho = -d_pg_frac_dRho
         
         r    = n% rate_screened(i_rate, ir_start)
         drdT = n% rate_screened(i_rate_dT, ir_start)
         drdd = n% rate_screened(i_rate_dRho, ir_start)
         
         if (ir_with_pg /= 0) then
            n% rate_screened(i_rate, ir_with_pg) = r*pg_frac
            n% rate_screened(i_rate_dT, ir_with_pg) = r*d_pg_frac_dT + drdT*pg_frac
            n% rate_screened(i_rate_dRho, ir_with_pg) = r*d_pg_frac_dRho + drdd*pg_frac
         end if
         
         if (ir_with_pa /= 0) then
            n% rate_screened(i_rate, ir_with_pa)  = r*pa_frac
            n% rate_screened(i_rate_dT, ir_with_pa) = r*d_pa_frac_dT + drdT*pa_frac
            n% rate_screened(i_rate_dRho, ir_with_pa) = r*d_pa_frac_dRho + drdd*pa_frac
         end if
               
      end subroutine rate_for_pg_pa_branches
      

      end module net_screen

