! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_eval
      use const_def
      use chem_def
      use net_def
      implicit none
      
      
      contains
      

      subroutine eval_net(
     >         g, num_isos, num_reactions, num_weaklib_rates,
     >         x, atemp, alogtemp, arho, alogrho, 
     >         abar, zbar, z2bar, ye, eta, rate_factors, category_factors,
     >         reaction_Qs, reaction_neuQs,
     >         eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_eps_nuc_dx, 
     >         dxdt, d_dxdt_dRho, d_dxdt_dT, d_dxdt_dx, 
     >         screening_mode, theta_e_for_graboske_et_al, 
     >         rate_screened, rate_raw, reuse_given_rates,
     >         reaction_eps_nuc, eps_nuc_categories, eps_neu_total,
     >         lwork, work, ierr)
         use net_initialize
         use net_screen
         use net_derivs
         use rates_lib, only: eval_using_rate_tables
         use alert_lib

         type (Net_General_Info), pointer :: g
         integer, intent(in) :: num_isos
         integer, intent(in) :: num_reactions, num_weaklib_rates
         double precision, intent(in)  :: x(num_isos)
         double precision, intent(in)  :: atemp, alogtemp
         double precision, intent(in)  :: arho, alogrho
         double precision, intent(in)  :: abar  ! mean number of nucleons per nucleus
         double precision, intent(in)  :: zbar  ! mean charge per nucleus
         double precision, intent(in)  :: z2bar ! mean charge squared per nucleus
         double precision, intent(in)  :: ye    
         double precision, intent(in)  :: eta
         double precision, intent(in) :: rate_factors(num_reactions)
         double precision, intent(in) :: category_factors(num_categories)
         double precision, pointer, intent(in) :: reaction_Qs(:) ! (rates_reaction_id_max)
         double precision, pointer, intent(in) :: reaction_neuQs(:) ! (rates_reaction_id_max)
         double precision, intent(out) :: eps_nuc ! ergs/gram/second from burning 
         double precision, intent(out) :: d_eps_nuc_dT
         double precision, intent(out) :: d_eps_nuc_dRho
         double precision, intent(out) :: d_eps_nuc_dx(num_isos) 
         double precision, intent(out) :: dxdt(num_isos)
         double precision, intent(out) :: d_dxdt_dRho(num_isos)
         double precision, intent(out) :: d_dxdt_dT(num_isos)
         double precision, intent(out) :: d_dxdt_dx(num_isos, num_isos)
         double precision, intent(out), target :: reaction_eps_nuc(num_rvs, num_reactions)
         double precision, intent(out), target :: eps_nuc_categories(num_rvs, num_categories)
         double precision, intent(inout), dimension(num_rvs, num_reactions), target :: 
     >         rate_screened, rate_raw
         double precision, intent(out) :: eps_neu_total
         logical, intent(in) :: reuse_given_rates
         integer, intent(in) :: screening_mode
         double precision, intent(in)  :: theta_e_for_graboske_et_al
         integer, intent(in) :: lwork
         double precision, target :: work(lwork)
         integer, intent(out) :: ierr

         type (Net_Info), target :: netinf
         type (Net_Info), pointer:: n
         integer, parameter :: max_z_for_cache = 14
         double precision, target :: 
     >         graboske_cache(3, max_z_for_cache, max_z_for_cache)
         double precision :: enuc, temp, logtemp, rho, logrho, total, prev, curr, prev_T
         double precision :: btemp, bden, eps_total
         integer :: ci, i, j
         integer, pointer :: chem_id(:)
         integer :: time0, time1
         logical :: doing_timing
         double precision, dimension(num_weaklib_rates), target :: 
     >         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, 
     >         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, 
     >         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, 
     >         lambda, dlambda_dlnT, dlambda_dlnRho, 
     >         Q, dQ_dlnT, dQ_dlnRho, 
     >         Qneu, dQneu_dlnT, dQneu_dlnRho
         double precision, parameter :: 
     >         deltap     = 7.288969d0, 
     >         deltan     = 8.071323d0
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'

         doing_timing = g% doing_timing
         if (doing_timing) then
            call system_clock(time0)
            g% doing_timing = .false.
         end if

         ierr = 0
         
         eps_nuc = 0
         d_eps_nuc_dT = 0
         d_eps_nuc_dRho = 0
         d_eps_nuc_dx(:) = 0
                  
         dxdt(:) = 0
         d_dxdt_dRho(:) = 0
         d_dxdt_dT(:) = 0
         d_dxdt_dx(: ,:) = 0
         reaction_eps_nuc = 0
         eps_nuc_categories = 0
         eps_neu_total = 0

         temp = atemp; logtemp = alogtemp; rho = arho; logrho = alogrho
         call get_T_rho_args(temp, logtemp, rho, logrho, ierr)
         if (ierr /= 0) return
         
         if (logtemp < rattab_tlo) then ! clip to table so can eval beta decays
            logtemp = rattab_tlo
            temp = 10**logtemp
            !write(*,1) 'logtemp', logtemp
            !write(*,1) 'rattab_tlo', rattab_tlo
            !return 
         end if
         
         chem_id => g% chem_id
         n => netinf
         n% graboske_cache => graboske_cache
         n% reaction_Qs => reaction_Qs
         n% reaction_neuQs => reaction_neuQs
         n% eps_neu_total = 0
         if (dbg) write(*,*) 'call setup_net_info'
         call setup_net_info(
     >         g, n, num_reactions, reaction_eps_nuc, eps_nuc_categories, 
     >         screening_mode, theta_e_for_graboske_et_al,
     >         rate_screened, rate_raw, lwork, work, reuse_given_rates, ierr)
         if (ierr /= 0) return
         
         if (.not. g% net_has_been_defined) then
            ierr = -1
            call alert(ierr, 'must define net')
            return
         end if
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            time0 = time1
         end if
         
         if (num_weaklib_rates > 0) then
            if (dbg) write(*,*) 'call get_weaklib_rates'
            call get_weaklib_rates(ierr)
            if (ierr /= 0) return
         end if

         n% d_eps_nuc_dy(:) = 0

         if (dbg) write(*,*) 'call set_molar_abundances'
         call set_molar_abundances(g, num_isos, x, n% y, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if

         !..limit range of temperatures and densities
         btemp = min(10**rattab_thi, max(temp, 10**rattab_tlo))
         bden = min(1.0d11, max(rho, 1.0d-10))
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            time0 = time1
         end if

         if (.not. reuse_given_rates) then
            call get_rates_with_screening(ierr)
            if (ierr /= 0) then
               call dealloc; return
            end if
         end if
         
         !..get the derivatives
         if (dbg) write(*,*) 'call get_derivs'
         call get_derivs( 
     >       n, category_factors, eta, ye, btemp, bden, abar, zbar, 
     >       num_reactions, rate_factors, ierr)
         if (ierr /= 0) then
            call dealloc; return
         end if
         
         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_derivs = g% clock_net_derivs + (time1 - time0)
            time0 = time1
         end if

         !..convert the reaction_eps_nuc Qs to ergs/g/sec
         n% reaction_eps_nuc(: ,:) = Qconv*n% reaction_eps_nuc(: ,:)

         !..convert the eps_nuc_categories
         n% eps_nuc_categories(: ,:) = Qconv*n% eps_nuc_categories(: ,:)

         !..store the results
         do i = 1, num_isos
            ci = chem_id(i)
            dxdt(i) = chem_isos% W(ci)*n% dydt(i_rate, i)
            d_dxdt_dRho(i) = chem_isos% W(ci)*n% dydt(i_rate_dRho, i)
            d_dxdt_dT(i) = chem_isos% W(ci)*n% dydt(i_rate_dT, i)
            do j=1, num_isos
               d_dxdt_dx(i, j) = chem_isos% W(ci)*n% d_dydt_dy(i, j) / chem_isos% W(chem_id(j))
            end do
         enddo
   
         eps_nuc = sum(n% reaction_eps_nuc(i_rate,:))
         d_eps_nuc_dT = sum(n% reaction_eps_nuc(i_rate_dT,:))
         d_eps_nuc_dRho = sum(n% reaction_eps_nuc(i_rate_dRho,:))         
         d_eps_nuc_dx(:) = Qconv*n% d_eps_nuc_dy(:) / chem_isos% W(chem_id(:))
         
         eps_neu_total = n% eps_neu_total*Qconv

! calculating eps_nuc from eps_binding is wrong when have weaklib Q significantly different than expected Q.
! e.g. when have very high density (> 1e10), mg24 -> na24 Q from weaklib is very large
!         eps_binding = 0.0d0
!         do i=1,num_isos
!            ci = chem_id(i)
!            eps_bind1 = 
!     >         (chem_isos% binding_energy(ci) - chem_isos% Z(ci)*deltap - chem_isos% N(ci)*deltan)
!     >            *dxdt(i)/chem_isos% W(ci)
!            eps_binding = eps_binding + eps_bind1
!            write(*,'(a30,2d26.16)') 
!     >            'binding eps nuc ' // trim(chem_isos% name(chem_id(i))), eps_bind1, dxdt(i)
!         enddo
!         eps_binding = eps_binding*Qconv

         call dealloc

         if (doing_timing) then
            call system_clock(time1)
            g% clock_net_eval = g% clock_net_eval + (time1 - time0)
            g% doing_timing = .true.
         end if
         
         
         if (.false.) then
            do i = 1, num_isos
               write(*,'(a30,1d26.16)') 
     >                     'xin(net_iso(i' // trim(chem_isos% name(chem_id(i))) // ')) = ', x(i)
            enddo
            write(*,*)
            if (.false.) then
            do i = 1, num_isos
               write(*,'(a30,1d26.16)') 
     >                     'dydt(i' // trim(chem_isos% name(chem_id(i))) // ')', n% dydt(i_rate,i)
            enddo
            write(*,*)
            do i = 1, num_isos
               write(*,'(a30,1d26.16)') 
     >                     'xin(net_iso(i' // trim(chem_isos% name(chem_id(i))) // ')) = ', 
     >                     n% y(i)*chem_isos% W(chem_id(i))
            enddo
            write(*,*)
            do i = 1, num_isos
               ci = chem_id(i)
               do j=1, num_isos
                  if (n% d_dydt_dy(i,j) /= 0) write(*,'(a30,1e26.16)') 'dfdy_' //  
     >               trim(chem_isos% name(ci)) // '_' // 
     >                     trim(chem_isos% name(chem_id(j))), n% d_dydt_dy(i,j)
               end do
            enddo
            end if
            write(*,*)
            write(*,1) 'rho =', rho
            write(*,1) 'T =', temp
            write(*,1) 'logrho =', logrho
            write(*,1) 'logT =', logtemp
            write(*,1) 'abar =', abar
            write(*,1) 'zbar =', zbar
            write(*,1) 'z2bar =', z2bar
            write(*,1) 'ye =', ye
            write(*,1) 'eta =', eta
            write(*,1) 'eps_nuc', eps_nuc
            write(*,1) 'eps_neu_total', eps_neu_total
            write(*,*)
            stop 'net eval'
         end if
         
         
         contains
         
         
         subroutine dealloc
         end subroutine dealloc
         
         
         subroutine get_rates_with_screening(ierr)
            integer, intent(out) :: ierr
            !..get the raw reaction rates
            if (dbg) write(*,*) 'call eval_using_rate_tables'
            call eval_using_rate_tables(num_reactions, g% reaction_id, g% rattab, g% rattab_f, 
     >               ye, logtemp, btemp, bden, rate_factors, g% logttab, n% rate_raw)
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_rate_tables = g% clock_net_rate_tables + (time1 - time0)
               time0 = time1
            end if
            !..get the reaction rates including screening factors
            if (dbg) write(*,*) 'call screen_net'
            call screen_net(
     >            n, num_isos, n% y, btemp, bden, logtemp, logrho, .false., 
     >            zbar, abar, z2bar, ye, ierr)
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_screen = g% clock_net_screen + (time1 - time0)
               time0 = time1
            end if
         end subroutine get_rates_with_screening
         
         
         subroutine get_weaklib_rates(ierr)
            use weak_lib, only: eval_weak_reaction_info
            use weak_def, only: weak_lhs_nuclide_name, weak_rhs_nuclide_name
            use num_lib, only: safe_log10
            integer, intent(out) :: ierr
            double precision :: d_eta_dlnT, d_eta_dlnRho
            double precision, parameter :: lntwo = log(2d0) ! 0.6931471805599453d0
            integer :: i, id
            include 'formats.dek'
            n% ldecay => ldecay
            n% d_ldecay_dT9 => d_ldecay_dT9
            n% d_ldecay_dlYeRho => d_ldecay_dlYeRho
            n% lcapture => lcapture
            n% d_lcapture_dT9 => d_lcapture_dT9
            n% d_lcapture_dlYeRho => d_lcapture_dlYeRho
            n% lneutrino => lneutrino
            n% d_lneutrino_dT9 => d_lneutrino_dT9
            n% d_lneutrino_dlYeRho => d_lneutrino_dlYeRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% lambda => lambda
            n% dlambda_dlnT => dlambda_dlnT
            n% dlambda_dlnRho => dlambda_dlnRho
            n% Q => Q
            n% dQ_dlnT => dQ_dlnT
            n% dQ_dlnRho => dQ_dlnRho
            n% Qneu => Qneu
            n% dQneu_dlnT => dQneu_dlnT
            n% dQneu_dlnRho => dQneu_dlnRho
            d_eta_dlnT = 0
            d_eta_dlnRho = 0
            call eval_weak_reaction_info(
     >         g% weaklib_ids(1:num_weaklib_rates), temp*1d-9, ye*rho,
     >         eta, d_eta_dlnT, d_eta_dlnRho,
     >         n% ldecay, n% d_ldecay_dT9, n% d_ldecay_dlYeRho,
     >         n% lcapture, n% d_lcapture_dT9, n% d_lcapture_dlYeRho,
     >         n% lneutrino, n% d_lneutrino_dT9, n% d_lneutrino_dlYeRho,
     >         n% lambda, n% dlambda_dlnT, n% dlambda_dlnRho,
     >         n% Q, n% dQ_dlnT, n% dQ_dlnRho,
     >         n% Qneu, n% dQneu_dlnT, n% dQneu_dlnRho,
     >         ierr)
            if (doing_timing) then
               call system_clock(time1)
               g% clock_net_weak_rates = g% clock_net_weak_rates + (time1 - time0)
               time0 = time1
            end if

         end subroutine get_weaklib_rates
      
      end subroutine eval_net

         
      subroutine set_molar_abundances(g, num_isos, x, y, ierr)
         use alert_lib
         type (Net_General_Info), pointer :: g
         integer, intent(in) :: num_isos
         double precision, intent(in) :: x(num_isos)
         double precision, intent(out) :: y(num_isos)
         integer, intent(out) :: ierr
         
         double precision :: sum
         integer :: i, ci
         character (len=256) :: message
         include 'formats.dek'
         sum = 0
         do i = 1, g% num_isos
            sum = sum + x(i)
            ci = g% chem_id(i)
            if (ci <= 0) then
               write(*,*) 'problem in set_molar_abundances'
               write(*,*) 'i', i
               write(*,*) 'g% num_isos', g% num_isos
               write(*,*) 'g% chem_id(i)', g% chem_id(i)
               stop 'set_molar_abundances' 
            end if
            y(i) = min(1d0, max(x(i), 0d0)) / chem_isos% W(ci)
         enddo
   
         if (abs(sum - 1d0) > 1d-2) then
            ierr = -1
            !do i = 1, g% num_isos
            !   ci = g% chem_id(i)
            !   write(*,2) chem_isos% name(ci), i, x(i)
            !end do
            return
         end if
      
      end subroutine set_molar_abundances


      
      
      subroutine get_T_rho_args(temp, logtemp, rho, logrho, info)
         use alert_lib
         double precision, intent(inout) :: temp, logtemp ! log10 of temp
         double precision, intent(inout) :: rho, logrho ! log10 of rho
         integer, intent(out) :: info
         info = 0
         if (temp == arg_not_provided .and. logtemp == arg_not_provided) then
            info = -2
            call alert(info, 'failed to provide either T or logT for nuclear net')
            return
         end if
         if (logtemp == arg_not_provided) logtemp = LOG10(temp)
         if (temp == arg_not_provided) temp = 10**logtemp
         if (temp <= 0) then
            info = -1
            call alert(info, 'T <= 0 for net')
            return
         end if
         if (rho == arg_not_provided .and. logrho == arg_not_provided) then
            info = -3
            call alert(info, 'failed to provide either Rho or logRho for nuclear net')
            return
         end if
         if (logrho == arg_not_provided) logrho = LOG10(rho)
         if (rho == arg_not_provided) rho = 10**logrho
         if (rho <= 0) then
            info = -1
            call alert(info, 'rho <= 0 for net')
            return
         end if
      end subroutine get_T_rho_args
      
      
      subroutine do_clean_up_fractions(nzlo, nzhi, species, nz, xa, max_sum_abs, xsum_tol, ierr)
         integer, intent(in) :: nzlo, nzhi, species, nz
         double precision, intent(inout) :: xa(species, nz)
         double precision, intent(in) :: max_sum_abs, xsum_tol
         integer, intent(out) :: ierr
         integer :: k, op_err
         ierr = 0
         if (nzlo == nzhi) then
            call do_clean1(species, xa(1: species, nzlo), nzlo, max_sum_abs, xsum_tol, ierr)
            return
         end if         
!$OMP  PARALLEL DO PRIVATE(k, op_err)
         do k = nzlo, nzhi
            op_err = 0
            call do_clean1(species, xa(1: species, k), k, max_sum_abs, xsum_tol, op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP  END PARALLEL DO
      end subroutine do_clean_up_fractions
      

      subroutine do_clean1(species, xa, k, max_sum_abs, xsum_tol, ierr)
         use utils_lib
         use alert_lib
         integer, intent(in) :: species, k
         double precision, intent(inout) :: xa(species)
         double precision, intent(in) :: max_sum_abs, xsum_tol
         integer, intent(out) :: ierr
         integer :: j
         double precision :: xsum
         double precision, parameter :: tiny_x = 1d-99
         character (len=256) :: message
 2       format(a40,i6,1pe26.16)
         if (max_sum_abs > 1) then ! check for crazy values
            xsum = sum(abs(xa(1: species)))
            if (is_bad_num(xsum) .or. xsum > max_sum_abs) then
               ierr = -1
               write(message,2) 'do_clean1: bad xsum', k, xsum
               call alert(ierr,message)
               return
            end if
         end if
         ierr = 0
         do j = 1, species
            if (xa(j) < tiny_x) xa(j) = tiny_x
            if (xa(j) > 1) xa(j) = 1
         end do
         
         xsum = sum(xa(1: species))         
         if (abs(xsum-1) > xsum_tol) then
            ierr = -1
            write(message,2) 'do_clean1: xsum-1', k, xsum-1
            call alert(ierr,message)
            return
         end if
         xa(1: species) = xa(1: species)/xsum
      end subroutine do_clean1
      

      end module net_eval

