! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
! ***********************************************************************

      module net_derivs_support
      use net_def
      use const_def
      use chem_def
      use rates_def

      implicit none
      
      
      double precision, parameter :: r_min = 1d-99
      
      logical, parameter :: show_rate = .false.
      logical, parameter :: show_jac = .false.
      logical, parameter :: show_neuQs = .false.
      
      logical, parameter :: show_dydt = .false.
      logical, parameter :: show_d_dydt_dRho = .false.
      logical, parameter :: show_d_dydt_dT = .false.
      
      logical, parameter :: show_d_eps_nuc_dy = .false.
      
      logical, parameter :: checkQs = .false.
      
      double precision, parameter :: checkQ_frac = 1d-4
      
      
      contains
      
      
      double precision function isoB(ci)
         integer, intent(in) :: ci
         double precision, parameter :: deltap = 7.288969d0, deltan = 8.071323d0
         isoB = chem_isos% binding_energy(ci) - chem_isos% Z(ci)*deltap - chem_isos% N(ci)*deltan
      end function isoB


      subroutine do_two_two(
     >         n, i, c1, i1, c2, i2, r_in, c3, i3, c4, i4, 
     >         idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 -> c3 * i3 + c4 * i4
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4 ! net isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, d2, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, cid4, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0
         
         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         d2  = dr2 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)

         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, idr2, d2, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_lhs_iso(n, i, c2, i2, rvs, idr1, d1, idr2, d2, 0, 0d0)             
         lhs = lhs + c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c2*rvs(i_rate), -c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_rhs_iso(n, i, c3, i3, rvs, idr1, d1, idr2, d2, 0, 0d0)       
         rhs = c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c3*rvs(i_rate), c3, rvs(i_rate), r

         call check(i4, 'i4')
         cid4 = chem_id(i4)
         call do_rhs_iso(n, i, c4, i4, rvs, idr1, d1, idr2, d2, 0, 0d0)
         rhs = rhs + c4*(chem_isos% Z(cid4) + chem_isos% N(cid4))
         if (cid4 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c4*rvs(i_rate), c4, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = n% reaction_Qs(reaction_id) - n% reaction_neuQs(reaction_id)
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + n% reaction_neuQs(reaction_id)*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     n% reaction_neuQs(reaction_id)*rvs(i_rate),
     >                     n% reaction_Qs(reaction_id)*rvs(i_rate),
     >                     n% reaction_neuQs(reaction_id),
     >                     n% reaction_Qs(reaction_id)
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)
     
         n% d_eps_nuc_dy(idr2) = n% d_eps_nuc_dy(idr2) + d2*Q
         if (chem_id(idr2) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d2*Q, n% d_eps_nuc_dy(idr2), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)
         
         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) + 
     >         n% reaction_eps_nuc(:, i)
                  
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c3*isoB(cid3) + c4*isoB(cid4) - c1*isoB(cid1) - c2*isoB(cid2)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_two_two checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         if (.false.) then
            write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id), rvs(1)*Q, r, rvs(1)
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_two_two: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_two_two


      subroutine do_two_three(
     >         n, i, c1, i1, c2, i2, r_in, c3, i3, c4, i4, c5, i5, 
     >         idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 -> c3 * i3 + c4 * i4 + c5 * i5
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4, i5 ! net isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4, c5 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, d2, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, cid4, cid5, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0
         
         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         d2  = dr2 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)

         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, idr2, d2, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_lhs_iso(n, i, c2, i2, rvs, idr1, d1, idr2, d2, 0, 0d0)             
         lhs = lhs + c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c2*rvs(i_rate), -c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_rhs_iso(n, i, c3, i3, rvs, idr1, d1, idr2, d2, 0, 0d0)       
         rhs = c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c3*rvs(i_rate), c3, rvs(i_rate), r

         call check(i4, 'i4')
         cid4 = chem_id(i4)
         call do_rhs_iso(n, i, c4, i4, rvs, idr1, d1, idr2, d2, 0, 0d0)
         rhs = rhs + c4*(chem_isos% Z(cid4) + chem_isos% N(cid4))
         if (cid4 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c4*rvs(i_rate), c4, rvs(i_rate), r

         call check(i5, 'i5')
         cid5 = chem_id(i5)
         call do_rhs_iso(n, i, c5, i5, rvs, idr1, d1, idr2, d2, 0, 0d0)
         rhs = rhs + c5*(chem_isos% Z(cid5) + chem_isos% N(cid5))
         if (cid5 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c5*rvs(i_rate), c5, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = n% reaction_Qs(reaction_id) - n% reaction_neuQs(reaction_id)
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + n% reaction_neuQs(reaction_id)*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     n% reaction_neuQs(reaction_id)*rvs(i_rate),
     >                     n% reaction_Qs(reaction_id)*rvs(i_rate),
     >                     n% reaction_neuQs(reaction_id),
     >                     n% reaction_Qs(reaction_id)
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)
     
         n% d_eps_nuc_dy(idr2) = n% d_eps_nuc_dy(idr2) + d2*Q
         if (chem_id(idr2) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d2*Q, n% d_eps_nuc_dy(idr2), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)

         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) + 
     >         n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c3*isoB(cid3) + c4*isoB(cid4) + c5*isoB(cid5) - c1*isoB(cid1) - c2*isoB(cid2)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_two_three checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         if (.false.) then
            write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id), rvs(1)*Q, r, rvs(1)
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_two_three: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_two_three


      subroutine do_three_one(
     >         n, i, c1, i1, c2, i2, c3, i3, r_in, c4, i4, 
     >         idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 + c3 * i3 -> c4 * i4
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         integer :: j
         j = n% g% reaction_id(i)
         call do_three_one_neu(
     >         n, i, c1, i1, c2, i2, c3, i3, r_in, c4, i4, idr1, dr1, idr2, dr2,  
     >         n% reaction_Qs(j), n% reaction_neuQs(j), deriv_flgs, category_factors)
      end subroutine do_three_one


      subroutine do_three_one_neu(
     >         n, i, c1, i1, c2, i2, c3, i3, r_in, c4, i4, 
     >         idr1, dr1, idr2, dr2, reaction_Q, Qneu, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 + c3 * i3 -> c4 * i4
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         double precision, intent(in) :: reaction_Q, Qneu
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)

         double precision :: rvs(num_rvs), d1, d2, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, cid4, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0

         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         d2  = dr2 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, idr2, d2, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_lhs_iso(n, i, c2, i2, rvs, idr1, d1, idr2, d2, 0, 0d0)             
         lhs = lhs + c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c2*rvs(i_rate), -c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_lhs_iso(n, i, c3, i3, rvs, idr1, d1, idr2, d2, 0, 0d0)       
         lhs = lhs + c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c3*rvs(i_rate), -c3, rvs(i_rate), r

         call check(i4, 'i4')
         cid4 = chem_id(i4)
         call do_rhs_iso(n, i, c4, i4, rvs, idr1, d1, idr2, d2, 0, 0d0)
         rhs = c4*(chem_isos% Z(cid4) + chem_isos% N(cid4))
         if (cid4 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c4*rvs(i_rate), c4, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = reaction_Q - Qneu
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + Qneu*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     Qneu*rvs(i_rate), reaction_Q*rvs(i_rate), Qneu, reaction_Q
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), reaction_Q, Qneu

         n% d_eps_nuc_dy(idr2) = n% d_eps_nuc_dy(idr2) + d2*Q
         if (chem_id(idr2) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d2*Q, n% d_eps_nuc_dy(idr2), reaction_Q, Qneu

         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) + 
     >         n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c4*isoB(cid4) - c1*isoB(cid1) - c2*isoB(cid2) - c3*isoB(cid3)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_three_one checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_three_one: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_three_one_neu


      subroutine do_three_two(
     >         n, i, c1, i1, c2, i2, c3, i3, r_in, c4, i4, c5, i5,
     >         idr1, dr1, idr2, dr2, idr3, dr3, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 + c3 * i3 -> c4 * i4 + c5 * i5
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4, i5 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4, c5 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         integer, intent(in) :: idr3 ! isotope number for dr3
         double precision, intent(in) :: dr3 ! coefficient for Jacobian entries d_dydt_dy(idr3)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)

         double precision :: rvs(num_rvs), d1, d2, d3, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, cid4, cid5, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0

         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         d2  = dr2 * n% rate_screened(i_rate, i) * category_factors(icat)
         d3  = dr3 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, idr2, d2, idr3, d3)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_lhs_iso(n, i, c2, i2, rvs, idr1, d1, idr2, d2, idr3, d3)             
         lhs = lhs + c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c2*rvs(i_rate), -c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_lhs_iso(n, i, c3, i3, rvs, idr1, d1, idr2, d2, idr3, d3)       
         lhs = lhs + c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c3*rvs(i_rate), -c3, rvs(i_rate), r


         call check(i4, 'i4')
         cid4 = chem_id(i4)
         call do_rhs_iso(n, i, c4, i4, rvs, idr1, d1, idr2, d2, idr3, d3)
         rhs = c4*(chem_isos% Z(cid4) + chem_isos% N(cid4))
         if (cid4 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c4*rvs(i_rate), c4, rvs(i_rate), r

         call check(i5, 'i5')
         cid5 = chem_id(i5)
         call do_rhs_iso(n, i, c5, i5, rvs, idr1, d1, idr2, d2, idr3, d3)
         rhs = rhs + c5*(chem_isos% Z(cid5) + chem_isos% N(cid5))
         if (cid5 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c5*rvs(i_rate), c5, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = n% reaction_Qs(reaction_id) - n% reaction_neuQs(reaction_id)
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + n% reaction_neuQs(reaction_id)*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     n% reaction_neuQs(reaction_id)*rvs(i_rate),
     >                     n% reaction_Qs(reaction_id)*rvs(i_rate),
     >                     n% reaction_neuQs(reaction_id),
     >                     n% reaction_Qs(reaction_id)
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)

         n% d_eps_nuc_dy(idr2) = n% d_eps_nuc_dy(idr2) + d2*Q
         if (chem_id(idr2) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d2*Q, n% d_eps_nuc_dy(idr2), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)

         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) + 
     >         n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c4*isoB(cid4) + c5*isoB(cid5) - c1*isoB(cid1) - c2*isoB(cid2) - c3*isoB(cid3)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_three_two checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         if (.false.) then
            write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id), rvs(1)*Q, r, rvs(1)
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_three_two: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_three_two


      subroutine do_two_one(
     >         n, i, c1, i1, c2, i2, r_in, c3, i3, idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 -> c3 * i3
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         integer :: j
         j = n% g% reaction_id(i)
         call do_two_one_neu(
     >         n, i, c1, i1, c2, i2, r_in, c3, i3, idr1, dr1, idr2, dr2, 
     >         n% reaction_Qs(j), n% reaction_neuQs(j), deriv_flgs, category_factors)
      end subroutine do_two_one
      

      subroutine do_two_one_neu(
     >      n, i, c1, i1, c2, i2, r_in, c3, i3, idr1, dr1, idr2, dr2, 
     >      reaction_Q, Qneu, deriv_flgs, category_factors)
         ! reaction of form c1 * il + c2 * i2 -> c3 * i3
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! coefficient of rate for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! coefficient for Jacobian entries d_dydt_dy(idr1)
         integer, intent(in) :: idr2 ! isotope number for dr2
         double precision, intent(in) :: dr2 ! coefficient for Jacobian entries d_dydt_dy(idr2)
         double precision, intent(in) :: reaction_Q, Qneu
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, d2,  lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0

         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         d2  = dr2 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, idr2, d2, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_lhs_iso(n, i, c2, i2, rvs, idr1, d1, idr2, d2, 0, 0d0)             
         lhs = lhs + c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c2*rvs(i_rate), -c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_rhs_iso(n, i, c3, i3, rvs, idr1, d1, idr2, d2, 0, 0d0)       
         rhs = c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c3*rvs(i_rate), c3, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = reaction_Q - Qneu
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + Qneu*rvs(i_rate)
         if (Qneu /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >            Qneu*rvs(i_rate), reaction_Q*rvs(i_rate), Qneu, reaction_Q
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), reaction_Q, Qneu

         n% d_eps_nuc_dy(idr2) = n% d_eps_nuc_dy(idr2) + d2*Q
         if (chem_id(idr2) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d2*Q, n% d_eps_nuc_dy(idr2), reaction_Q, Qneu

         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) 
     >         + n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c3*isoB(cid3) - c1*isoB(cid1) - c2*isoB(cid2)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_two_one checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_two_one: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check

      end subroutine do_two_one_neu


      subroutine do_one_three(
     >         n, i, c1, i1, r_in, c2, i2, c3, i3, c4, i4, idr1, dr1, deriv_flgs, category_factors)
         ! reaction of form c1 * il -> c2 * i2 + c3 * i3 + c4 * i4
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3, i4 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3, c4 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! rate info for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! for Jacobian entries d_dydt_dy(idr1)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, cid4, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0

         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, 0, 0d0, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_rhs_iso(n, i, c2, i2, rvs, idr1, d1, 0, 0d0, 0, 0d0)
         rhs = c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c2*rvs(i_rate), c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_rhs_iso(n, i, c3, i3, rvs, idr1, d1, 0, 0d0, 0, 0d0)       
         rhs = rhs + c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c3*rvs(i_rate), c3, rvs(i_rate), r

         call check(i4, 'i4')
         cid4 = chem_id(i4)
         call do_rhs_iso(n, i, c4, i4, rvs, idr1, d1, 0, 0d0, 0, 0d0)       
         rhs = rhs + c4*(chem_isos% Z(cid4) + chem_isos% N(cid4))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c4*rvs(i_rate), c4, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = n% reaction_Qs(reaction_id) - n% reaction_neuQs(reaction_id)
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + n% reaction_neuQs(reaction_id)*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     n% reaction_neuQs(reaction_id)*rvs(i_rate),
     >                     n% reaction_Qs(reaction_id)*rvs(i_rate),
     >                     n% reaction_neuQs(reaction_id),
     >                     n% reaction_Qs(reaction_id)
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)
         
         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) 
     >         + n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c2*isoB(cid2) + c3*isoB(cid3) + c4*isoB(cid4) - c1*isoB(cid1)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_one_two checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         if (.false.) then
            write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id), rvs(1)*Q, r, rvs(1)
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_one_two: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_one_three


      subroutine do_one_two(
     >         n, i, c1, i1, r_in, c2, i2, c3, i3, idr1, dr1, deriv_flgs, category_factors)
         ! reaction of form c1 * il -> c2 * i2 + c3 * i3
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2, i3 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2, c3 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! rate info for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! for Jacobian entries d_dydt_dy(idr1)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, cid3, reaction_id
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0

         g => n % g
         chem_id => g% chem_id

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, 0, 0d0, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_rhs_iso(n, i, c2, i2, rvs, idr1, d1, 0, 0d0, 0, 0d0)
         rhs = c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c2*rvs(i_rate), c2, rvs(i_rate), r

         call check(i3, 'i3')
         cid3 = chem_id(i3)
         call do_rhs_iso(n, i, c3, i3, rvs, idr1, d1, 0, 0d0, 0, 0d0)       
         rhs = rhs + c3*(chem_isos% Z(cid3) + chem_isos% N(cid3))
         if (cid3 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c3*rvs(i_rate), c3, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = n% reaction_Qs(reaction_id) - n% reaction_neuQs(reaction_id)
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + n% reaction_neuQs(reaction_id)*rvs(i_rate)
         if (n% reaction_neuQs(reaction_id) /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >                     n% reaction_neuQs(reaction_id)*rvs(i_rate),
     >                     n% reaction_Qs(reaction_id)*rvs(i_rate),
     >                     n% reaction_neuQs(reaction_id),
     >                     n% reaction_Qs(reaction_id)
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id)
         
         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) 
     >         + n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c2*isoB(cid2) + c3*isoB(cid3) - c1*isoB(cid1)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_one_two checkQ ' // trim(reaction_Name(g% reaction_id(i))), 
     >            n% reaction_Qs(reaction_id), checkQ
               !stop
            end if
         end if
         
         if (.false.) then
            write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         n% reaction_Qs(reaction_id), n% reaction_neuQs(reaction_id), rvs(1)*Q, r, rvs(1)
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_one_two: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_one_two


      subroutine do_one_one(n, i, c1, i1, r_in, c2, i2, idr1, dr1, deriv_flgs, category_factors)
         ! reaction of form c1 * il -> c2 * i2
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! rate info for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! for Jacobian entries d_dydt_dy(idr1)
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         integer :: j
         j = n% g% reaction_id(i)
         call do_one_one_neu(
     >         n, i, c1, i1, r_in, c2, i2, idr1, dr1, 
     >         n% reaction_Qs(j), n% reaction_neuQs(j), 
     >         deriv_flgs, category_factors)
      end subroutine do_one_one


      subroutine do_one_one_neu(
     >      n, i, c1, i1, r_in, c2, i2, idr1, dr1, reaction_Q, Qneu, deriv_flgs, category_factors)
         ! reaction of form c1 * il -> c2 * i2
         type (Net_Info), pointer :: n
         integer, intent(in) :: i ! the reaction number
         integer, intent(in) :: i1, i2 ! isotope numbers for the reaction
         double precision, intent(in) :: c1, c2 ! isotope coefficients in reaction equation
         double precision, intent(in) :: r_in ! rate info for the reaction
         integer, intent(in) :: idr1 ! isotope number for dr1
         double precision, intent(in) :: dr1 ! for Jacobian entries d_dydt_dy(idr1)
         double precision, intent(in) :: reaction_Q, Qneu
         logical, pointer :: deriv_flgs(:)
         double precision, intent(in) :: category_factors(num_categories)
         
         double precision :: rvs(num_rvs), d1, lhs, rhs, r, Q, checkQ
         type (Net_General_Info), pointer  :: g
         integer, pointer :: chem_id(:)
         integer :: cid1, icat, cid2, reaction_id
         logical :: dbg
         
         include 'formats.dek'
         
         r = r_in; icat = reaction_categories(n% g% reaction_id(i))
         if (r < r_min .or. n% rate_screened(i_rate, i) < r_min) r = 0
         
         g => n % g
         chem_id => g% chem_id
         
      	dbg = .false. ! (reaction_Name(g% reaction_id(i)) == 'r_h1_h1_wk_h2')

         d1  = dr1 * n% rate_screened(i_rate, i) * category_factors(icat)
         rvs = r * n% rate_screened(1:num_rvs, i) * category_factors(icat)
         
         call check(i1, 'i1')
         cid1 = chem_id(i1)
         call do_lhs_iso(n, i, c1, i1, rvs, idr1, d1, 0, 0d0, 0, 0d0)
         lhs = c1*(chem_isos% Z(cid1) + chem_isos% N(cid1))
         if (cid1 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         -c1*rvs(i_rate), -c1, rvs(i_rate), r

         call check(i2, 'i2')
         cid2 = chem_id(i2)
         call do_rhs_iso(n, i, c2, i2, rvs, idr1, d1, 0, 0d0, 0, 0d0)       
         rhs = c2*(chem_isos% Z(cid2) + chem_isos% N(cid2))
         if (cid2 == ib8 .and. show_rate) write(*,1) trim(reaction_Name(g% reaction_id(i))), 
     >         c2*rvs(i_rate), c2, rvs(i_rate), r
         
         reaction_id = g% reaction_id(i)
         Q = reaction_Q - Qneu
         n% reaction_eps_nuc(:, i) = rvs*Q
         n% eps_neu_total = n% eps_neu_total + Qneu*rvs(i_rate)
         if (Qneu /= 0 .and. show_neuQs) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' neu', 
     >               Qneu*rvs(i_rate), reaction_Q*rvs(i_rate), Qneu, reaction_Q
         
         n% d_eps_nuc_dy(idr1) = n% d_eps_nuc_dy(idr1) + d1*Q
         if (chem_id(idr1) == ic9 .and. show_d_eps_nuc_dy) 
     >         write(*,1) trim(reaction_Name(g% reaction_id(i))) // ' d_epsnuc_dy', 
     >            d1*Q, n% d_eps_nuc_dy(idr1), reaction_Q, Qneu
         
         n% eps_nuc_categories(:, icat) = n% eps_nuc_categories(:, icat) 
     >         + n% reaction_eps_nuc(:, i)
         
         call check_balance(n, i, lhs, rhs)
         deriv_flgs(i) = .true.
         
         if (checkQs) then
            checkQ = c2*isoB(cid2) - c1*isoB(cid1)
            if (abs(n% reaction_Qs(reaction_id) - checkQ) > checkQ_frac*abs(checkQ)) then
               write(*,1) 'do_one_one_neu checkQ ' // trim(reaction_Name(g% reaction_id(i))),
     >            n% reaction_Qs(reaction_id), checkQ, c1, isoB(cid1), c2, isoB(cid2)
               !stop
            end if
         end if
         
         contains
         
         subroutine check(i, str)
            integer, intent(in) :: i
            character (len=*), intent(in) :: str
            if (i <= 0) then
               write(*,*) 
     >            'do_one_one_neu: bad iso num for ' // trim(str) //
     >            ' in ' // trim(reaction_Name(g% reaction_id(i)))
               stop 1
            end if
         end subroutine check
      
      end subroutine do_one_one_neu
      
         
      subroutine do_lhs_iso(n, i, c, i1, rvs, i2, dr2, i3, dr3, i4, dr4)
         type (Net_Info), pointer :: n
         integer, intent(in) :: i, i1, i2, i3, i4
         double precision, intent(in) :: c, rvs(num_rvs), dr2, dr3, dr4
         
         ! i1, i2, i3, and 14 are isotope numbers
         ! dr2 = dr/dy(i2); dr3 = dr/dy(i3); dr4 = dr/dy(i4)
         ! -c * r   = dydt(i1)
         ! -c * dr2 = d_dydt(i1)_dy(i2)
         ! -c * dr3 = d_dydt(i1)_dy(i3)
         ! -c * dr4 = d_dydt(i1)_dy(i4)
         
         integer, pointer :: chem_id(:)
         chem_id => n% g% chem_id
         
         include 'formats.dek'

         ! update the dydt terms for i1
         n% dydt(1:num_rvs, i1) = n% dydt(1:num_rvs, i1) - c * rvs
         if (chem_id(i1) == ic9 .and. show_dydt)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' dydt '
     >            // trim(chem_isos% name(chem_id(i1))), -c * rvs(i_rate), n% dydt(i_rate, i1)
         if (chem_id(i1) == ic9 .and. show_d_dydt_dRho)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dRho '
     >            // trim(chem_isos% name(chem_id(i1))), -c * rvs(i_rate_dRho), n% dydt(i_rate_dRho, i1)
         if (chem_id(i1) == if17 .and. show_d_dydt_dT)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dT '
     >            // trim(chem_isos% name(chem_id(i1))), -c * rvs(i_rate_dT), n% dydt(i_rate_dT, i1)
     
         ! update the Jacobian for d_dydt(i1)_dy(i2)
         n% d_dydt_dy(i1, i2) = n% d_dydt_dy(i1, i2)  - c * dr2
         
         if (chem_id(i1) == imn50 .and. chem_id(i2) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy l2 dr2 ' 
     >     // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i2))), 
     >     - c * dr2, n% d_dydt_dy(i1, i2)
            
         if (i3 <= 0) return
            
         ! update the Jacobian for d_dydt(i1)_dy(i3)
         n% d_dydt_dy(i1, i3) = n% d_dydt_dy(i1, i3)  - c * dr3
         
         if (chem_id(i1) == imn50 .and. chem_id(i3) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy l3 dr3 ' 
     >     // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i3))), 
     >     - c * dr3, n% d_dydt_dy(i1, i3)
            
         if (i4 <= 0) return
            
         !write(*,4) trim(reaction_Name(n% g% reaction_id(i))) // ' do_lhs_iso', i, i1, i4
         ! update the Jacobian for d_dydt(i1)_dy(i4)
         n% d_dydt_dy(i1, i4) = n% d_dydt_dy(i1, i4)  - c * dr4
         
         if (chem_id(i1) == imn50 .and. chem_id(i4) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy l4 dr4 ' 
     >     // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i4))), 
     >     - c * dr4, n% d_dydt_dy(i1, i4)
         
      end subroutine do_lhs_iso
            

      subroutine do_rhs_iso(n, i, c, i1, rvs, i2, dr2, i3, dr3, i4, dr4)
         type (Net_Info), pointer :: n
         integer, intent(in) :: i, i1, i2, i3, i4
         double precision, intent(in) :: c, rvs(num_rvs), dr2, dr3, dr4
         
         ! i1, i2, i3, and 14 are isotope numbers
         ! dr2 = dr/dy(i2); dr3 = dr/dy(i3); dr4 = dr/dy(i4)
         ! c * r   = dydt(i1)
         ! c * dr2 = d_dydt(i1)_dy(i2)
         ! c * dr3 = d_dydt(i1)_dy(i3)
         ! c * dr4 = d_dydt(i1)_dy(i4)

         integer, pointer :: chem_id(:)
         chem_id => n% g% chem_id
         
         include 'formats.dek'

         ! update the dydt terms for i1
         n% dydt(1:num_rvs, i1) = n% dydt(1:num_rvs, i1) + c * rvs
         if (chem_id(i1) == ic9 .and. show_dydt)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' dydt '
     >      // trim(chem_isos% name(chem_id(i1))), c * rvs(i_rate), n% dydt(i_rate, i1)
         if (chem_id(i1) == ic9 .and. show_d_dydt_dRho)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dRho '
     >            // trim(chem_isos% name(chem_id(i1))), c * rvs(i_rate_dRho), n% dydt(i_rate_dRho, i1)
         if (chem_id(i1) == if17 .and. show_d_dydt_dT)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dT '
     >            // trim(chem_isos% name(chem_id(i1))), c * rvs(i_rate_dT), n% dydt(i_rate_dT, i1)

         ! update the Jacobian for d_dydt(i1)_dy(i2)
         n% d_dydt_dy(i1, i2) = n% d_dydt_dy(i1, i2)  + c * dr2
         
         if (chem_id(i1) == imn50 .and. chem_id(i2) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy r2 dr2 '
     >      // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i2))), 
     >     c * dr2, n% d_dydt_dy(i1, i2)
            
         if (i3 <= 0) return
            
         ! update the Jacobian for d_dydt(i1)_dy(i3)
         n% d_dydt_dy(i1, i3) = n% d_dydt_dy(i1, i3)  + c * dr3
         
         if (chem_id(i1) == imn50 .and. chem_id(i3) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy r3 dr3 '
     >      // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i3))), 
     >     c * dr3, n% d_dydt_dy(i1, i3)
            
         if (i4 <= 0) return
            
         ! update the Jacobian for d_dydt(i1)_dy(i4)
         n% d_dydt_dy(i1, i4) = n% d_dydt_dy(i1, i4)  + c * dr4
         
         if (chem_id(i1) == imn50 .and. chem_id(i4) == ih1 .and. show_jac)
     >         write(*,1) trim(reaction_Name(n% g% reaction_id(i))) // ' d_dydt_dy r4 dr4 '
     >      // trim(chem_isos% name(chem_id(i1))) // ' ' // trim(chem_isos% name(chem_id(i4))), 
     >     c * dr4, n% d_dydt_dy(i1, i4)
         
      end subroutine do_rhs_iso
      
         
      subroutine check_balance(n, i, lhs, rhs) ! check conservation of nucleons
         type (Net_Info), pointer :: n
         integer, intent(in) :: i
         double precision, intent(in) :: lhs, rhs
         if (lhs == rhs) return
         if (abs(lhs-rhs) < 1d-6) return
         write(*,'(2a)') 'non-conservation of nucleons in reaction ', 
     >         reaction_Name(n% g% reaction_id(i))
         write(*,*) 'lhs aion sum', lhs
         write(*,*) 'rhs aion sum', rhs
         stop
      end subroutine check_balance

      
      
      subroutine get_he4_breakup_and_rebuild(
     >      n, g, y, num_reactions, rate_factors,
     >      d_rhe4_breakup_dneut, d_rhe4_breakup_dprot,
     >      d_rhe4_rebuild_dneut, d_rhe4_rebuild_dprot)        
         type (Net_Info), pointer :: n
         type (Net_General_Info), pointer  :: g
         double precision, pointer :: y(:)
         integer, intent(in) :: num_reactions
         double precision, intent(in) ::
     >      rate_factors(num_reactions)         
         double precision, intent(out) ::
     >      d_rhe4_breakup_dneut, d_rhe4_breakup_dprot,
     >      d_rhe4_rebuild_dneut, d_rhe4_rebuild_dprot

         integer, pointer :: rtab(:), itab(:)
         double precision, pointer :: rates(:, :)
         double precision :: 
     >      rhegn, rhegp, rdgn,
     >      rheng, rdpg, rhng,
     >      d_rhegn_dt, d_rhegp_dt, d_rdgn_dt,
     >      d_rheng_dt, d_rdpg_dt, d_rhng_dt,
     >      d_rhegn_dd, d_rhegp_dd, d_rdgn_dd,
     >      d_rheng_dd, d_rdpg_dd, d_rhng_dd,
     >      denom, denomdt, denomdd, denomdprot, denomdneut,
     >      zz, d_zz_dt, d_zz_dd, d_zz_dprot, d_zz_dneut
         integer :: neut, prot, rhe4_breakup, rhe4_rebuild
         integer :: rhe4g_neut_aux, rhe3gprot_aux, rh2g_neut_aux
         integer :: rprot_neutg_aux, rh2_protg_aux, rhe3_neutg_aux

         include 'formats.dek'
          
         rates => n% rate_screened
         rtab => g% net_reaction
         itab => g% net_iso

         neut = itab(ineut)
         prot = itab(iprot)
         
         rhe4_breakup = rtab(irhe4_breakup)
         rhe4_rebuild = rtab(irhe4_rebuild)
         
         if (rhe4_breakup /= 0) rates(:, rhe4_breakup) = 0
         if (rhe4_rebuild /= 0) rates(:, rhe4_rebuild) = 0
         
         rhe4g_neut_aux = rtab(irhe4g_neut_aux)
         if (rhe4g_neut_aux == 0) stop 'need rhe4g_neut_aux'
         
         rhe3gprot_aux = rtab(irhe3gprot_aux)
         if (rhe3gprot_aux == 0) stop 'need rhe3gprot_aux'
         
         rh2g_neut_aux = rtab(irh2g_neut_aux)
         if (rh2g_neut_aux == 0) stop 'need rh2g_neut_aux'
         
         rprot_neutg_aux = rtab(irprot_neutg_aux)
         if (rprot_neutg_aux == 0) stop 'need rprot_neutg_aux'
         
         rh2_protg_aux = rtab(irh2_protg_aux)
         if (rh2_protg_aux == 0) stop 'need rh2_protg_aux'
         
         rhe3_neutg_aux = rtab(irhe3_neutg_aux)
         if (rhe3_neutg_aux == 0) stop 'need rhe3_neutg_aux'

         rhegn = rates(i_rate,rhe4g_neut_aux)*rate_factors(rhe4g_neut_aux)
         rhegp = rates(i_rate,rhe3gprot_aux)*rate_factors(rhe3gprot_aux)
         rdgn = rates(i_rate,rh2g_neut_aux)*rate_factors(rh2g_neut_aux)
         
         rheng = rates(i_rate,rhe3_neutg_aux)*rate_factors(rhe3_neutg_aux)
         rdpg = rates(i_rate,rh2_protg_aux)*rate_factors(rh2_protg_aux)
         rhng = rates(i_rate,rprot_neutg_aux)*rate_factors(rprot_neutg_aux)
         
         denom  = rhegp*rdgn + y(neut)*rheng*rdgn + y(neut)*y(prot)*rheng*rdpg
         
         if (.false.) then
            write(*,1) 'rhegn', rhegn
            write(*,1) 'rhegp', rhegp
            write(*,1) 'rdgn', rdgn
            write(*,1) 'rheng', rheng
            write(*,1) 'rdpg', rdpg
            write(*,1) 'rhng', rhng
            write(*,1) 'denom', denom
            write(*,*)
            !stop 'get_he4_breakup_and_rebuild'
         end if
         
         d_rhegn_dt = rates(i_rate_dT,rhe4g_neut_aux)*rate_factors(rhe4g_neut_aux)
         d_rhegp_dt = rates(i_rate_dT,rhe3gprot_aux)*rate_factors(rhe3gprot_aux)
         d_rdgn_dt = rates(i_rate_dT,rh2g_neut_aux)*rate_factors(rh2g_neut_aux)

         d_rhegn_dd = rates(i_rate_dRho,rhe4g_neut_aux)*rate_factors(rhe4g_neut_aux)
         d_rhegp_dd = rates(i_rate_dRho,rhe3gprot_aux)*rate_factors(rhe3gprot_aux)
         d_rdgn_dd = rates(i_rate_dRho,rh2g_neut_aux)*rate_factors(rh2g_neut_aux)
         
         d_rheng_dt = rates(i_rate_dT,rhe3_neutg_aux)*rate_factors(rhe3_neutg_aux)
         d_rdpg_dt = rates(i_rate_dT,rh2_protg_aux)*rate_factors(rh2_protg_aux)
         d_rhng_dt = rates(i_rate_dT,rprot_neutg_aux)*rate_factors(rprot_neutg_aux)
         
         d_rheng_dd = rates(i_rate_dRho,rhe3_neutg_aux)*rate_factors(rhe3_neutg_aux)
         d_rdpg_dd = rates(i_rate_dRho,rh2_protg_aux)*rate_factors(rh2_protg_aux)
         d_rhng_dd = rates(i_rate_dRho,rprot_neutg_aux)*rate_factors(rprot_neutg_aux)
            
         if (denom > 1) then
            denomdt =  
     1            d_rhegp_dt*rdgn 
     1          + rhegp*d_rdgn_dt  
     2          +  y(neut) * (d_rheng_dt*rdgn + rheng*d_rdgn_dt) 
     4          +  y(neut)*y(prot) * (d_rheng_dt*rdpg + rheng*d_rdpg_dt)
            denomdd =  
     1            d_rhegp_dd*rdgn 
     1          + rhegp*d_rdgn_dd  
     2          +  y(neut) * (d_rheng_dd*rdgn + rheng*d_rdgn_dd) 
     4          +  y(neut)*y(prot) * (d_rheng_dd*rdpg + rheng*d_rdpg_dd)
            denomdprot = y(neut)*rheng*rdpg
            denomdneut = rheng*rdgn + y(prot)*rheng*rdpg
         else
            denom = 1
            denomdt = 0
            denomdd = 0
            denomdprot = 0
            denomdneut = 0
         end if

         zz = 1/denom  
         d_zz_dt = -denomdt/denom**2
         d_zz_dd = -denomdd/denom**2
         d_zz_dprot = -denomdprot/denom**2
         d_zz_dneut = -denomdneut/denom**2
         
         if (rhe4_breakup /= 0) then
            rates(i_rate,rhe4_breakup) = rhegn*rhegp*rdgn*zz
            rates(:,rhe4_breakup) = rates(:,rhe4_breakup)*rate_factors(rhe4_breakup)

            rates(i_rate_dT,rhe4_breakup) = 
     >         d_rhegn_dt*rhegp*rdgn*zz +         
     >         rhegn*d_rhegp_dt*rdgn*zz +         
     >         rhegn*rhegp*d_rdgn_dt*zz +         
     >         rhegn*rhegp*rdgn*d_zz_dt       

            rates(i_rate_dRho,rhe4_breakup) = 
     >         d_rhegn_dd*rhegp*rdgn*zz +         
     >         rhegn*d_rhegp_dd*rdgn*zz +         
     >         rhegn*rhegp*d_rdgn_dd*zz +         
     >         rhegn*rhegp*rdgn*d_zz_dd       

            d_rhe4_breakup_dprot = rhegn*rhegp*rdgn*d_zz_dprot
            d_rhe4_breakup_dneut = rhegn*rhegp*rdgn*d_zz_dneut
         
         else

            d_rhe4_breakup_dprot = 0
            d_rhe4_breakup_dneut = 0
            
         end if
         
         if (rhe4_rebuild /= 0) then

            rates(i_rate,rhe4_rebuild) = rheng*rdpg*rhng*zz
            rates(:,rhe4_rebuild) = rates(:,rhe4_rebuild)*rate_factors(rhe4_rebuild)

            rates(i_rate_dT,rhe4_rebuild) = 
     >         d_rheng_dt*rdpg*rhng*zz +
     >         rheng*d_rdpg_dt*rhng*zz +
     >         rheng*rdpg*d_rhng_dt*zz +
     >         rheng*rdpg*rhng*d_zz_dt
     
            rates(i_rate_dRho,rhe4_rebuild) = 
     >         d_rheng_dd*rdpg*rhng*zz +
     >         rheng*d_rdpg_dd*rhng*zz +
     >         rheng*rdpg*d_rhng_dd*zz +
     >         rheng*rdpg*rhng*d_zz_dd

            d_rhe4_rebuild_dprot = rheng*rdpg*rhng*d_zz_dprot
            d_rhe4_rebuild_dneut = rheng*rdpg*rhng*d_zz_dneut
            
         else
         
            d_rhe4_rebuild_dprot = 0
            d_rhe4_rebuild_dneut = 0

         end if

      end subroutine get_he4_breakup_and_rebuild
         

      subroutine get_rates_for_fe52_fe54(
     >      n, g, temp, y, num_reactions, rate_factors,
     >      d_rfe52neut_to_fe54_dneut, d_rfe54g_to_fe52_dneut)        
         type (Net_Info), pointer :: n
         type (Net_General_Info), pointer  :: g
         double precision, pointer :: y(:)
         integer, intent(in) :: num_reactions
         double precision, intent(in) ::
     >      temp, rate_factors(num_reactions)         
         double precision, intent(out) ::
     >      d_rfe52neut_to_fe54_dneut, d_rfe54g_to_fe52_dneut

         double precision, pointer :: rates(:, :)
         integer, pointer :: rtab(:), itab(:)
         integer :: neut, fe52, fe54, rfe52ng_aux, rfe53ng_aux, rfe53gn_aux, rfe54gn_aux,
     >      rfe52neut_to_fe54, rfe54g_to_fe52
         double precision ::  
     >      denom, denomdt, denomdd, denomdneut, 
     >      zz, d_zz_dt, d_zz_dd, d_zz_dneut,
     >      rfe52ng, rfe53ng, rfe53gn, rfe54gn,  
     >      d_rfe52ng_dt, d_rfe53ng_dt, d_rfe53gn_dt, d_rfe54gn_dt, 
     >      d_rfe52ng_dd, d_rfe53ng_dd, d_rfe53gn_dd, d_rfe54gn_dd,
     >      fac, fr, rr

         include 'formats.dek'
          
         rates => n% rate_screened
         rtab => g% net_reaction
         itab => g% net_iso

         neut = itab(ineut)
         if (neut == 0) stop 'need neut'

         fe52 = itab(ife52)
         if (neut == 0) stop 'need fe52'

         fe54 = itab(ife54)
         if (neut == 0) stop 'need fe54'
         
         rfe52neut_to_fe54 = rtab(irfe52neut_to_fe54)
         rfe54g_to_fe52 = rtab(irfe54g_to_fe52)
         
         rfe52ng_aux = rtab(irfe52ng_aux)
         if (rfe52ng_aux == 0) stop 'need rfe52ng_aux'
         
         rfe53ng_aux = rtab(irfe53ng_aux)
         if (rfe53ng_aux == 0) stop 'need rfe53ng_aux'
         
         rfe53gn_aux = rtab(irfe53gn_aux)
         if (rfe53gn_aux == 0) stop 'need rfe53gn_aux'
         
         rfe54gn_aux = rtab(irfe54gn_aux)
         if (rfe54gn_aux == 0) stop 'need rfe54gn_aux'
         
         rfe52ng = rates(i_rate,rfe52ng_aux)*rate_factors(rfe52ng_aux)
         rfe53ng = rates(i_rate,rfe53ng_aux)*rate_factors(rfe53ng_aux)
         rfe53gn = rates(i_rate,rfe53gn_aux)*rate_factors(rfe53gn_aux)
         rfe54gn = rates(i_rate,rfe54gn_aux)*rate_factors(rfe54gn_aux)
         
         denom = rfe53gn + y(neut)*rfe53ng
         
         rates(:,rfe52neut_to_fe54) = 0
         d_rfe52neut_to_fe54_dneut = 0
         
         rates(:,rfe54g_to_fe52) = 0
         d_rfe54g_to_fe52_dneut = 0
         
         !write(*,1) 'denom', denom
         
         if (denom < 1d-12 .or. temp < 1.5d9) return
         
         d_rfe52ng_dt = rates(i_rate_dT,rfe52ng_aux)*rate_factors(rfe52ng_aux)
         d_rfe53ng_dt = rates(i_rate_dT,rfe53ng_aux)*rate_factors(rfe53ng_aux)
         d_rfe53gn_dt = rates(i_rate_dT,rfe53gn_aux)*rate_factors(rfe53gn_aux)
         d_rfe54gn_dt = rates(i_rate_dT,rfe54gn_aux)*rate_factors(rfe54gn_aux)
         
         d_rfe52ng_dd = rates(i_rate_dRho,rfe52ng_aux)*rate_factors(rfe52ng_aux)
         d_rfe53ng_dd = rates(i_rate_dRho,rfe53ng_aux)*rate_factors(rfe53ng_aux)
         d_rfe53gn_dd = rates(i_rate_dRho,rfe53gn_aux)*rate_factors(rfe53gn_aux)
         d_rfe54gn_dd = rates(i_rate_dRho,rfe54gn_aux)*rate_factors(rfe54gn_aux)
         
         denomdt = rates(i_rate_dT,rfe53gn_aux) + y(neut)*rates(i_rate_dT,rfe53ng_aux)
         denomdd = rates(i_rate_dRho,rfe53gn_aux) + y(neut)*rates(i_rate_dRho,rfe53ng_aux)
         denomdneut = rfe53ng
         
         zz = 1/denom
         d_zz_dt = -denomdt/denom**2
         d_zz_dd = -denomdd/denom**2
         d_zz_dneut = -denomdneut/denom**2
         
         if (rfe54g_to_fe52 /= 0) then
         
            rates(i_rate,rfe54g_to_fe52) = rfe54gn*rfe53gn*zz
            
            rates(i_rate_dT,rfe54g_to_fe52) = 
     >         d_rfe54gn_dt*rfe53gn*zz +
     >         rfe54gn*d_rfe53gn_dt*zz +
     >         rfe54gn*rfe53gn*d_zz_dt

            rates(i_rate_dRho,rfe54g_to_fe52) = 
     >         d_rfe54gn_dd*rfe53gn*zz +
     >         rfe54gn*d_rfe53gn_dd*zz +
     >         rfe54gn*rfe53gn*d_zz_dd

            d_rfe54g_to_fe52_dneut = rfe54gn*rfe53gn*d_zz_dneut

            rates(:,rfe54g_to_fe52) = rates(:,rfe54g_to_fe52)*rate_factors(rfe54g_to_fe52)
            d_rfe54g_to_fe52_dneut = d_rfe54g_to_fe52_dneut*rate_factors(rfe54g_to_fe52)

         end if

         if (rfe52neut_to_fe54 /= 0) then
         
            rates(i_rate,rfe52neut_to_fe54) = rfe52ng*rfe53ng*zz
            
            rates(i_rate_dT,rfe52neut_to_fe54) = 
     >         d_rfe52ng_dt*rfe53ng*zz +
     >         rfe52ng*d_rfe53ng_dt*zz +
     >         rfe52ng*rfe53ng*d_zz_dt

            rates(i_rate_dRho,rfe52neut_to_fe54) = 
     >         d_rfe52ng_dd*rfe53ng*zz +
     >         rfe52ng*d_rfe53ng_dd*zz +
     >         rfe52ng*rfe53ng*d_zz_dd

            d_rfe52neut_to_fe54_dneut = rfe52ng*rfe53ng*d_zz_dneut
         
            rates(:,rfe52neut_to_fe54) = rates(:,rfe52neut_to_fe54)*rate_factors(rfe52neut_to_fe54)
            d_rfe52neut_to_fe54_dneut = d_rfe52neut_to_fe54_dneut*rate_factors(rfe52neut_to_fe54)
            
         end if

      end subroutine get_rates_for_fe52_fe54


      subroutine get_rates_for_co55_combos(  
     >      n, g, temp, y, num_reactions, rate_factors,
     >      d_rfe54prot_to_ni56_dprot, d_rni56gprot_to_fe54_dprot,
     >      d_rfe52aprot_to_fe54_dprot, d_rfe54prot_to_fe52_dprot,
     >      d_rfe52aprot_to_ni56_dprot, d_rni56gprot_to_fe52_dprot)    
         type (Net_Info), pointer :: n
         type (Net_General_Info), pointer  :: g
         double precision, pointer :: y(:)
         integer, intent(in) :: num_reactions
         double precision, intent(in) :: temp, rate_factors(num_reactions)         
         double precision, intent(out) ::    
     >         d_rfe54prot_to_ni56_dprot, d_rni56gprot_to_fe54_dprot,
     >         d_rfe52aprot_to_fe54_dprot, d_rfe54prot_to_fe52_dprot,
     >         d_rfe52aprot_to_ni56_dprot, d_rni56gprot_to_fe52_dprot

         double precision, pointer :: rates(:, :)
         integer, pointer :: rtab(:), itab(:)
         integer :: prot, rfe54prot_to_ni56, rni56gprot_to_fe54,
     >      rfe52aprot_to_fe54, rfe54prot_to_fe52,
     >      rfe52aprot_to_ni56, rni56gprot_to_fe52,
     >      rfe52aprot_aux, rfe54protg_aux, rco55gprot_aux,
     >      rco55protg_aux, rco55prota_aux, rni56gprot_aux
         double precision ::  
     >      denom, denomdt, denomdd, denomdprot, 
     >      zz, d_zz_dt, d_zz_dd, d_zz_dprot,
     >      rfe52aprot, rfe54protg, rco55g_prot,
     >      rco55protg, rco55prot_a, rni56g_prot,
     >      d_rfe52aprot_dt, d_rfe54protg_dt, d_rco55gprot_dt,
     >      d_rco55protg_dt, d_rco55prota_dt, d_rni56gprot_dt,
     >      d_rfe52aprot_dd, d_rfe54protg_dd, d_rco55gprot_dd,
     >      d_rco55protg_dd, d_rco55prota_dd, d_rni56gprot_dd

         include 'formats.dek'
          
         rates => n% rate_screened
         rtab => g% net_reaction
         itab => g% net_iso

         prot = itab(iprot)
         if (prot == 0) stop 'need prot'
         
         rfe54prot_to_ni56 = rtab(irfe54prot_to_ni56) ! fe54(prot,g)co55(prot,g)ni56
         rni56gprot_to_fe54 = rtab(irni56gprot_to_fe54) ! ni56(g,prot)co55(g,prot)fe54         
         rfe52aprot_to_fe54 = rtab(irfe52aprot_to_fe54) ! fe52(a,prot)co55(g,prot)fe54         
         rfe54prot_to_fe52 = rtab(irfe54prot_to_fe52) ! fe54(prot,g)co55(prot,a)fe52         
         rfe52aprot_to_ni56 = rtab(irfe52aprot_to_ni56) ! fe52(a,prot)co55(prot,g)ni56         
         rni56gprot_to_fe52 = rtab(irni56gprot_to_fe52) ! ni56(g,prot)co55(prot,a)fe52
                  
         rfe52aprot_aux = rtab(irfe52aprot_aux)
         if (rfe52aprot_aux == 0) stop 'need rfe52aprot_aux'
         
         rfe54protg_aux = rtab(irfe54protg_aux)
         if (rfe54protg_aux == 0) stop 'need rfe54protg_aux'
         
         rco55gprot_aux = rtab(irco55gprot_aux)
         if (rco55gprot_aux == 0) stop 'need rco55gprot_aux'
         
         rco55protg_aux = rtab(irco55protg_aux)
         if (rco55protg_aux == 0) stop 'need rco55protg_aux'
         
         rco55prota_aux = rtab(irco55prota_aux)
         if (rco55prota_aux == 0) stop 'need rco55prota_aux'
         
         rni56gprot_aux = rtab(irni56gprot_aux)
         if (rni56gprot_aux == 0) stop 'need rni56gprot_aux'
         
         rfe52aprot = rates(i_rate,rfe52aprot_aux)*rate_factors(rfe52aprot_aux)
         rfe54protg = rates(i_rate,rfe54protg_aux)*rate_factors(rfe54protg_aux)
         rco55g_prot = rates(i_rate,rco55gprot_aux)*rate_factors(rco55gprot_aux)
         rco55protg = rates(i_rate,rco55protg_aux)*rate_factors(rco55protg_aux)
         rco55prot_a = rates(i_rate,rco55prota_aux)*rate_factors(rco55prota_aux)
         rni56g_prot = rates(i_rate,rni56gprot_aux)*rate_factors(rni56gprot_aux)

         denom = rco55g_prot + y(prot)*(rco55protg + rco55prot_a)
         
         if (rfe54prot_to_ni56 /= 0) then
            rates(:,rfe54prot_to_ni56) = 0
            d_rfe54prot_to_ni56_dprot = 0
         end if

         if (rni56gprot_to_fe54 /= 0) then
            rates(:,rni56gprot_to_fe54) = 0
            d_rni56gprot_to_fe54_dprot = 0
         end if

         if (rfe52aprot_to_fe54 /= 0) then
            rates(:,rfe52aprot_to_fe54) = 0
            d_rfe52aprot_to_fe54_dprot = 0
         end if

         if (rfe54prot_to_fe52 /= 0) then
            rates(:,rfe54prot_to_fe52) = 0
            d_rfe54prot_to_fe52_dprot = 0
         end if

         if (rfe52aprot_to_ni56 /= 0) then
            rates(:,rfe52aprot_to_ni56) = 0
            d_rfe52aprot_to_ni56_dprot = 0
         end if

         if (rni56gprot_to_fe52 /= 0) then
            rates(:,rni56gprot_to_fe52) = 0
            d_rni56gprot_to_fe52_dprot = 0
         end if
         
         if (denom < 1d-12 .or. temp < 1.5d9) return
         
         d_rfe52aprot_dt = rates(i_rate_dT,rfe52aprot_aux)*rate_factors(rfe52aprot_aux)
         d_rfe54protg_dt = rates(i_rate_dT,rfe54protg_aux)*rate_factors(rfe54protg_aux)
         d_rco55gprot_dt = rates(i_rate_dT,rco55gprot_aux)*rate_factors(rco55gprot_aux)
         d_rco55protg_dt = rates(i_rate_dT,rco55protg_aux)*rate_factors(rco55protg_aux)
         d_rco55prota_dt = rates(i_rate_dT,rco55prota_aux)*rate_factors(rco55prota_aux)
         d_rni56gprot_dt = rates(i_rate_dT,rni56gprot_aux)*rate_factors(rni56gprot_aux)
         
         d_rfe52aprot_dd = rates(i_rate_dRho,rfe52aprot_aux)*rate_factors(rfe52aprot_aux)
         d_rfe54protg_dd = rates(i_rate_dRho,rfe54protg_aux)*rate_factors(rfe54protg_aux)
         d_rco55gprot_dd = rates(i_rate_dRho,rco55gprot_aux)*rate_factors(rco55gprot_aux)
         d_rco55protg_dd = rates(i_rate_dRho,rco55protg_aux)*rate_factors(rco55protg_aux)
         d_rco55prota_dd = rates(i_rate_dRho,rco55prota_aux)*rate_factors(rco55prota_aux)
         d_rni56gprot_dd = rates(i_rate_dRho,rni56gprot_aux)*rate_factors(rni56gprot_aux)

         denomdt = d_rco55gprot_dt + y(prot)*(d_rco55protg_dt + d_rco55prota_dt)
         denomdd = d_rco55gprot_dd + y(prot)*(d_rco55protg_dd + d_rco55prota_dd)
         denomdprot = rco55protg + rco55prot_a
         
         zz = 1/denom
         d_zz_dt = -denomdt/denom**2
         d_zz_dd = -denomdd/denom**2
         d_zz_dprot = -denomdprot/denom**2
         
         if (rfe54prot_to_ni56 /= 0) then ! fe54(prot,g)co55(prot,g)ni56
         
            rates(i_rate,rfe54prot_to_ni56) = rfe54protg*rco55protg*zz
            
            rates(i_rate_dT,rfe54prot_to_ni56) = 
     >         d_rfe54protg_dt*rco55protg*zz +
     >         rfe54protg*d_rco55protg_dt*zz +
     >         rfe54protg*rco55protg*d_zz_dt

            rates(i_rate_dRho,rfe54prot_to_ni56) = 
     >         d_rfe54protg_dd*rco55protg*zz +
     >         rfe54protg*d_rco55protg_dd*zz +
     >         rfe54protg*rco55protg*d_zz_dd

            d_rfe54prot_to_ni56_dprot = rfe54protg*rco55protg*d_zz_dprot
         
            rates(:,rfe54prot_to_ni56) = rates(:,rfe54prot_to_ni56)*rate_factors(rfe54prot_to_ni56)
            d_rfe54prot_to_ni56_dprot = d_rfe54prot_to_ni56_dprot*rate_factors(rfe54prot_to_ni56)
            
         end if
         
         if (rni56gprot_to_fe54 /= 0) then ! ni56(g,prot)co55(g,prot)fe54
         
            rates(i_rate,rni56gprot_to_fe54) = rni56g_prot*rco55g_prot*zz
            
            rates(i_rate_dT,rni56gprot_to_fe54) = 
     >         d_rni56gprot_dt*rco55g_prot*zz +
     >         rni56g_prot*d_rco55gprot_dt*zz +
     >         rni56g_prot*rco55g_prot*d_zz_dt

            rates(i_rate_dRho,rni56gprot_to_fe54) = 
     >         d_rni56gprot_dd*rco55g_prot*zz +
     >         rni56g_prot*d_rco55gprot_dd*zz +
     >         rni56g_prot*rco55g_prot*d_zz_dd

            d_rni56gprot_to_fe54_dprot = rni56g_prot*rco55g_prot*d_zz_dprot
         
            rates(:,rni56gprot_to_fe54) = rates(:,rni56gprot_to_fe54)*rate_factors(rni56gprot_to_fe54)
            d_rni56gprot_to_fe54_dprot = d_rni56gprot_to_fe54_dprot*rate_factors(rni56gprot_to_fe54)
 
         end if
         
         if (rfe52aprot_to_fe54 /= 0) then ! fe52(a,prot)co55(g,prot)fe54
         
            rates(i_rate,rfe52aprot_to_fe54) = rfe52aprot*rco55g_prot*zz
            
            rates(i_rate_dT,rfe52aprot_to_fe54) = 
     >         d_rfe52aprot_dt*rco55g_prot*zz +
     >         rfe52aprot*d_rco55gprot_dt*zz +
     >         rfe52aprot*rco55g_prot*d_zz_dt

            rates(i_rate_dRho,rfe52aprot_to_fe54) = 
     >         d_rfe52aprot_dd*rco55g_prot*zz +
     >         rfe52aprot*d_rco55gprot_dd*zz +
     >         rfe52aprot*rco55g_prot*d_zz_dd

            d_rfe52aprot_to_fe54_dprot = rfe52aprot*rco55g_prot*d_zz_dprot
         
            rates(:,rfe52aprot_to_fe54) = rates(:,rfe52aprot_to_fe54)*rate_factors(rfe52aprot_to_fe54)
            d_rfe52aprot_to_fe54_dprot = d_rfe52aprot_to_fe54_dprot*rate_factors(rfe52aprot_to_fe54)
 
         end if
         
         if (rfe54prot_to_fe52 /= 0) then ! fe54(prot,g)co55(prot,a)fe52
         
            rates(i_rate,rfe54prot_to_fe52) = rfe54protg*rco55prot_a*zz
            
            rates(i_rate_dT,rfe54prot_to_fe52) = 
     >         d_rfe54protg_dt*rco55prot_a*zz +
     >         rfe54protg*d_rco55prota_dt*zz +
     >         rfe54protg*rco55prot_a*d_zz_dt

            rates(i_rate_dRho,rfe54prot_to_fe52) = 
     >         d_rfe54protg_dd*rco55prot_a*zz +
     >         rfe54protg*d_rco55prota_dd*zz +
     >         rfe54protg*rco55prot_a*d_zz_dd

            d_rfe54prot_to_fe52_dprot = 
     >            rfe54protg*rco55prot_a*d_zz_dprot
         
            rates(:,rfe54prot_to_fe52) = rates(:,rfe54prot_to_fe52)*rate_factors(rfe54prot_to_fe52)
            d_rfe54prot_to_fe52_dprot = d_rfe54prot_to_fe52_dprot*rate_factors(rfe54prot_to_fe52)
 
         end if
         
         if (rfe52aprot_to_ni56 /= 0) then ! fe52(a,prot)co55(prot,g)ni56
         
            rates(i_rate,rfe52aprot_to_ni56) = rfe52aprot*rco55protg*zz
            
            rates(i_rate_dT,rfe52aprot_to_ni56) = 
     >        d_rfe52aprot_dt*rco55protg*zz  +
     >        rfe52aprot*d_rco55protg_dt*zz  +
     >        rfe52aprot*rco55protg*d_zz_dt

            rates(i_rate_dRho,rfe52aprot_to_ni56) = 
     >        d_rfe52aprot_dd*rco55protg*zz  +
     >        rfe52aprot*d_rco55protg_dd*zz  +
     >        rfe52aprot*rco55protg*d_zz_dd

            d_rfe52aprot_to_ni56_dprot = rfe52aprot*rco55protg*d_zz_dprot

            rates(:,rfe52aprot_to_ni56) = rates(:,rfe52aprot_to_ni56)*rate_factors(rfe52aprot_to_ni56)
            d_rfe52aprot_to_ni56_dprot = d_rfe52aprot_to_ni56_dprot*rate_factors(rfe52aprot_to_ni56)
 
         end if
         
         if (rni56gprot_to_fe52 /= 0) then ! ni56(g,prot)co55(prot,a)fe52
         
            rates(i_rate,rni56gprot_to_fe52) = rni56g_prot*rco55prot_a*zz
            
            rates(i_rate_dT,rni56gprot_to_fe52) = 
     >         d_rni56gprot_dt*rco55prot_a*zz +
     >         rni56g_prot*d_rco55prota_dt*zz +
     >         rni56g_prot*rco55prot_a*d_zz_dt

            rates(i_rate_dRho,rni56gprot_to_fe52) = 
     >         d_rni56gprot_dd*rco55prot_a*zz +
     >         rni56g_prot*d_rco55prota_dd*zz +
     >         rni56g_prot*rco55prot_a*d_zz_dd

            d_rni56gprot_to_fe52_dprot = rni56g_prot*rco55prot_a*d_zz_dprot
         
            rates(:,rni56gprot_to_fe52) = rates(:,rni56gprot_to_fe52)*rate_factors(rni56gprot_to_fe52)
            d_rni56gprot_to_fe52_dprot = d_rni56gprot_to_fe52_dprot*rate_factors(rni56gprot_to_fe52)
 
         end if

      end subroutine get_rates_for_co55_combos
         

      subroutine get_rates_for_fe54_fe56(
     >      n, g, temp, y, num_reactions, rate_factors,
     >      d_rfe54neut_to_fe56_dneut, d_rfe56g_to_fe54_dneut)        
         type (Net_Info), pointer :: n
         type (Net_General_Info), pointer  :: g
         double precision, pointer :: y(:)
         integer, intent(in) :: num_reactions
         double precision, intent(in) ::
     >      temp, rate_factors(num_reactions)         
         double precision, intent(out) ::
     >      d_rfe54neut_to_fe56_dneut, d_rfe56g_to_fe54_dneut

         double precision, pointer :: rates(:, :)
         integer, pointer :: rtab(:), itab(:)
         integer :: neut, fe54, fe56, rfe54ng_aux, rfe55ng_aux, rfe55gn_aux, rfe56gn_aux,
     >      rfe54neut_to_fe56, rfe56g_to_fe54
         double precision ::  
     >      denom, denomdt, denomdd, denomdneut, 
     >      zz, d_zz_dt, d_zz_dd, d_zz_dneut,
     >      rfe54ng, rfe55ng, rfe55gn, rfe56gn,  
     >      d_rfe54ng_dt, d_rfe55ng_dt, d_rfe55gn_dt, d_rfe56gn_dt, 
     >      d_rfe54ng_dd, d_rfe55ng_dd, d_rfe55gn_dd, d_rfe56gn_dd,
     >      fac, fr, rr

         include 'formats.dek'
          
         rates => n% rate_screened
         rtab => g% net_reaction
         itab => g% net_iso

         neut = itab(ineut)
         if (neut == 0) stop 'need neut'

         fe54 = itab(ife54)
         if (neut == 0) stop 'need fe54'

         fe56 = itab(ife56)
         if (neut == 0) stop 'need fe56'
         
         rfe54neut_to_fe56 = rtab(irfe54ng_to_fe56)
         rfe56g_to_fe54 = rtab(irfe56gn_to_fe54)
         
         rfe54ng_aux = rtab(irfe54ng_aux)
         if (rfe54ng_aux == 0) stop 'need rfe54ng_aux'
         
         rfe55ng_aux = rtab(irfe55ng_aux)
         if (rfe55ng_aux == 0) stop 'need rfe55ng_aux'
         
         rfe55gn_aux = rtab(irfe55gn_aux)
         if (rfe55gn_aux == 0) stop 'need rfe55gn_aux'
         
         rfe56gn_aux = rtab(irfe56gn_aux)
         if (rfe56gn_aux == 0) stop 'need rfe56gn_aux'
         
         rfe54ng = rates(i_rate,rfe54ng_aux)*rate_factors(rfe54ng_aux)
         rfe55ng = rates(i_rate,rfe55ng_aux)*rate_factors(rfe55ng_aux)
         rfe55gn = rates(i_rate,rfe55gn_aux)*rate_factors(rfe55gn_aux)
         rfe56gn = rates(i_rate,rfe56gn_aux)*rate_factors(rfe56gn_aux)
         
         denom = rfe55gn + y(neut)*rfe55ng
         
         rates(:,rfe54neut_to_fe56) = 0
         d_rfe54neut_to_fe56_dneut = 0
         
         rates(:,rfe56g_to_fe54) = 0
         d_rfe56g_to_fe54_dneut = 0
         
         !write(*,1) 'denom', denom
         
         if (denom < 1d-12 .or. temp < 1.5d9) return
         
         d_rfe54ng_dt = rates(i_rate_dT,rfe54ng_aux)*rate_factors(rfe54ng_aux)
         d_rfe55ng_dt = rates(i_rate_dT,rfe55ng_aux)*rate_factors(rfe55ng_aux)
         d_rfe55gn_dt = rates(i_rate_dT,rfe55gn_aux)*rate_factors(rfe55gn_aux)
         d_rfe56gn_dt = rates(i_rate_dT,rfe56gn_aux)*rate_factors(rfe56gn_aux)
         
         d_rfe54ng_dd = rates(i_rate_dRho,rfe54ng_aux)*rate_factors(rfe54ng_aux)
         d_rfe55ng_dd = rates(i_rate_dRho,rfe55ng_aux)*rate_factors(rfe55ng_aux)
         d_rfe55gn_dd = rates(i_rate_dRho,rfe55gn_aux)*rate_factors(rfe55gn_aux)
         d_rfe56gn_dd = rates(i_rate_dRho,rfe56gn_aux)*rate_factors(rfe56gn_aux)
         
         denomdt = rates(i_rate_dT,rfe55gn_aux) + y(neut)*rates(i_rate_dT,rfe55ng_aux)
         denomdd = rates(i_rate_dRho,rfe55gn_aux) + y(neut)*rates(i_rate_dRho,rfe55ng_aux)
         denomdneut = rfe55ng
         
         zz = 1/denom
         d_zz_dt = -denomdt/denom**2
         d_zz_dd = -denomdd/denom**2
         d_zz_dneut = -denomdneut/denom**2
         
         if (rfe56g_to_fe54 /= 0) then
         
            rates(i_rate,rfe56g_to_fe54) = rfe56gn*rfe55gn*zz
            
            rates(i_rate_dT,rfe56g_to_fe54) = 
     >         d_rfe56gn_dt*rfe55gn*zz +
     >         rfe56gn*d_rfe55gn_dt*zz +
     >         rfe56gn*rfe55gn*d_zz_dt

            rates(i_rate_dRho,rfe56g_to_fe54) = 
     >         d_rfe56gn_dd*rfe55gn*zz +
     >         rfe56gn*d_rfe55gn_dd*zz +
     >         rfe56gn*rfe55gn*d_zz_dd

            d_rfe56g_to_fe54_dneut = rfe56gn*rfe55gn*d_zz_dneut

            rates(:,rfe56g_to_fe54) = rates(:,rfe56g_to_fe54)*rate_factors(rfe56g_to_fe54)
            d_rfe56g_to_fe54_dneut = d_rfe56g_to_fe54_dneut*rate_factors(rfe56g_to_fe54)

         end if

         if (rfe54neut_to_fe56 /= 0) then
         
            rates(i_rate,rfe54neut_to_fe56) = rfe54ng*rfe55ng*zz
            
            rates(i_rate_dT,rfe54neut_to_fe56) = 
     >         d_rfe54ng_dt*rfe55ng*zz +
     >         rfe54ng*d_rfe55ng_dt*zz +
     >         rfe54ng*rfe55ng*d_zz_dt

            rates(i_rate_dRho,rfe54neut_to_fe56) = 
     >         d_rfe54ng_dd*rfe55ng*zz +
     >         rfe54ng*d_rfe55ng_dd*zz +
     >         rfe54ng*rfe55ng*d_zz_dd

            d_rfe54neut_to_fe56_dneut = rfe54ng*rfe55ng*d_zz_dneut
         
            rates(:,rfe54neut_to_fe56) = rates(:,rfe54neut_to_fe56)*rate_factors(rfe54neut_to_fe56)
            d_rfe54neut_to_fe56_dneut = d_rfe54neut_to_fe56_dneut*rate_factors(rfe54neut_to_fe56)
            
         end if

      end subroutine get_rates_for_fe54_fe56



      end module net_derivs_support

