! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module net_derivs
      use net_def
      use const_def
      use chem_def
      use net_derivs_support
      use rates_def
      use rates_lib

      implicit none
      private
      public :: get_derivs


      logical, parameter :: combine_beta_rates = .true.
      
      double precision, parameter :: tiny_rate = 1d-50
      
      contains


      subroutine get_derivs(
     >         n, category_factors, eta, ye, temp, den, abar, zbar,
     >         num_reactions, rate_factors, ierr)
         use rates_lib, only: eval_ecapnuc_rate
         type (Net_Info), pointer :: n
         integer, intent(in) :: num_reactions
         double precision, intent(in) :: 
     >      category_factors(num_categories), eta, ye, temp, den, abar, zbar,
     >      rate_factors(num_reactions)         
         integer, intent(out) :: ierr

         logical :: all_okay
         integer, pointer :: itab(:), rtab(:)
         integer :: ir, i
         logical, target :: deriv_flgs_data(num_reactions)
         logical, pointer :: deriv_flgs(:)
         type (Net_General_Info), pointer  :: g
         double precision, pointer :: y(:)
         integer :: time0, time1
         
         ierr = 0
         
         y => n% y
         g => n% g

         if (g% doing_derivs_timing) call system_clock(time0)

         itab => g% net_iso
         rtab => g% net_reaction
         
         deriv_flgs => deriv_flgs_data
         deriv_flgs(:) = .false. ! we'll check this later to see if we got them all.

         n% dydt = 0
         n% d_dydt_dy = 0

         if (rtab(irprot_to_neut) /= 0 .or. rtab(irneut_to_prot) /= 0) then
            call eval_ecapnuc_rate(eta, temp, n% rpen, n% rnep, n% spen, n% snep)
         end if
         
         if (rtab(irhe4_breakup) /= 0 .or. rtab(irhe4_rebuild) /= 0) then
            call get_he4_breakup_and_rebuild(n, g, y, num_reactions, rate_factors,
     >         n% d_rhe4_breakup_dneut, n% d_rhe4_breakup_dprot,
     >         n% d_rhe4_rebuild_dneut, n% d_rhe4_rebuild_dprot)
         end if
         
         if (rtab(irfe52neut_to_fe54) /= 0 .or. rtab(irfe54g_to_fe52) /= 0) then
            call get_rates_for_fe52_fe54(
     >         n, g, temp, y, num_reactions, rate_factors,
     >         n% d_rfe52neut_to_fe54_dneut, n% d_rfe54g_to_fe52_dneut)
         end if
         
         if (rtab(irfe54prot_to_ni56) /= 0 .or.
     >       rtab(irni56gprot_to_fe54) /= 0 .or.
     >       rtab(irfe52aprot_to_fe54) /= 0 .or.
     >       rtab(irfe54prot_to_fe52) /= 0 .or.
     >       rtab(irfe52aprot_to_ni56) /= 0 .or.
     >       rtab(irni56gprot_to_fe52) /= 0) then
            call get_rates_for_co55_combos(
     >         n, g, temp, y, num_reactions, rate_factors,
     >         n% d_rfe54prot_to_ni56_dprot, n% d_rni56gprot_to_fe54_dprot,
     >         n% d_rfe52aprot_to_fe54_dprot, n% d_rfe54prot_to_fe52_dprot,
     >         n% d_rfe52aprot_to_ni56_dprot, n% d_rni56gprot_to_fe52_dprot
     >         )
         end if
         
         if (rtab(irfe54ng_to_fe56) /= 0 .or. rtab(irfe56gn_to_fe54) /= 0) then
            call get_rates_for_fe54_fe56(
     >         n, g, temp, y, num_reactions, rate_factors,
     >         n% d_rfe54neut_to_fe56_dneut, n% d_rfe56g_to_fe54_dneut)
         end if         

         if (g% doing_derivs_timing) then
            call system_clock(time1)
            g% clock_derivs_setup = g% clock_derivs_setup + (time1 - time0)
         end if
         
         do i=1, n% g% num_reactions
            call get1_derivs( 
     >         n, i, category_factors, eta, ye, temp, den, abar, zbar, 
     >         num_reactions, rate_factors,  
     >         deriv_flgs, ierr)
            if (ierr /= 0) return
         end do
         
         contains
         
         subroutine Check
            integer :: nrates
            nrates = n% g% num_reactions
            
            do ir = 1, nrates
               if (.not. deriv_flgs(ir)) then
                  all_okay = .false.
                  write(*,'(a,i4,2x,a)') 'missing derivs for ', ir,
     >                  trim(reaction_Name(n% g% reaction_id(ir)))
               end if
            end do
         
         end subroutine Check
      
      end subroutine get_derivs


      subroutine get1_derivs(
     >         n, i, category_factors, eta, ye, temp, den, abar, zbar,
     >         num_reactions, rate_factors,  
     >         deriv_flgs, ierr)
         type (Net_Info), pointer :: n
         integer, intent(in) :: i, num_reactions
         double precision, intent(in) :: category_factors(num_categories), 
     >         eta, ye, temp, den, abar, zbar, rate_factors(num_reactions)
         logical, pointer :: deriv_flgs(:)
         integer, intent(out) :: ierr
         
         integer :: ir, j, prot, neut, h1, he4, c14, n14, ne20, ne22, 
     >         mg21, mg22, mg23, mg24, al23, al24, si24, si25, si26,  
     >         s28, s29, s30, cl31, ar32, ar33, ar34, k35, ca36, ca37, ca38, 
     >         ti41, ti42, v43, cr44, cr45, cr46, cr56, 
     >         fe48, fe49, fe50, fe51, fe52, fe54, fe56, 
     >         ni52, ni53, ni54, ni55, ni56, 
     >         zn57, zn58, zn59, zn60, ge62, ge63, ge64, 
     >         se68, kr72, sr76, mo84, sn104
         integer :: weak_id, num_reaction_inputs, in1, in2, in3, in4, in5
         integer :: cin1, cin2, cin3, cin4, cin5
         double precision :: din1, din2, din3, din4, din5
         integer :: num_reaction_outputs, out1, out2, out3, out4, out5
         integer :: cout1, cout2, cout3, cout4, cout5
         double precision :: dout1, dout2, dout3, dout4, dout5
         type (Net_General_Info), pointer  :: g
         integer, pointer :: itab(:), reaction_id(:)
         double precision, pointer :: y(:)
         integer :: i1, i2, i3, idr1, idr2, idr3, o1, o2, o3
         double precision :: r, dr1, dr2, dr3, rn56ec, sn56ec, Qneu, rn14ec
         double precision :: halflife, halflife1, halflife2
         double precision :: beta_rate, capture_rate, G00, Gp1, Gp2, mu00, mup1, b1, b2
         double precision, parameter :: lntwo = log(2d0)

         integer, parameter :: npart = 24	! no. entries in partition fcn table
         double precision, dimension(npart) :: Tpart = (/
     >         0.10,0.15,0.20,0.30,0.40,0.50,0.60,0.70,0.80,0.90,1.00,1.50,
     >         2.00,2.50,3.00,3.50,4.00,4.50,5.00,6.00,7.00,8.00,9.00,10.0 /)            
                  
         double precision, parameter :: ge64_halflife = 64.0
         
         double precision, parameter :: se66_halflife = 0.033
         double precision, parameter :: se67_halflife = 0.133
         double precision, parameter :: se68_halflife = 35.5
         
         double precision, parameter :: kr70_halflife = 0.057
         double precision, parameter :: kr71_halflife = 0.100
         double precision, parameter :: kr72_halflife = 17.1
         
         double precision, parameter :: sr74_halflife = 0.044
         double precision, parameter :: sr75_halflife = 0.088
         double precision, parameter :: sr76_halflife = 7.89
         
         double precision, parameter :: y76_halflife = 2d-7
         
         double precision, parameter :: zr78_halflife = 2d-7
         double precision, parameter :: zr79_halflife = 0.056
         double precision, parameter :: nb81_halflife = 2d-7
         
         double precision, parameter :: mo83_halflife = 0.023
         double precision, parameter :: mo84_halflife = 3.8
         
         double precision, parameter :: sn102_halflife = 4.5
         
         double precision, parameter :: ge64_Qneu = 4 ! ?
         double precision, parameter :: se66_Qneu = 4 ! ?
         double precision, parameter :: se67_Qneu = 4 ! ?
         double precision, parameter :: se68_Qneu = 4 ! ?
         double precision, parameter :: kr70_Qneu = 4 ! ?
         double precision, parameter :: kr71_Qneu = 4 ! ?
         double precision, parameter :: kr72_Qneu = 4 ! ?
         double precision, parameter :: y76_Qneu = 4 ! ?
         double precision, parameter :: sr74_Qneu = 4 ! ?
         double precision, parameter :: sr75_Qneu = 4 ! ?
         double precision, parameter :: sr76_Qneu = 4 ! ?
         double precision, parameter :: zr78_Qneu = 4 ! ?
         double precision, parameter :: zr79_Qneu = 4 ! ?
         double precision, parameter :: nb81_Qneu = 4 ! ?
         double precision, parameter :: mo83_Qneu = 4 ! ?
         double precision, parameter :: mo84_Qneu = 4 ! ?
         double precision, parameter :: Tc85_Qneu = 4 ! ?
         double precision, parameter :: Ru87_Qneu = 4 ! ?
         double precision, parameter :: Ru88_Qneu = 4 ! ?
         double precision, parameter :: Pd91_Qneu = 4 ! ?
         double precision, parameter :: Pd92_Qneu = 4 ! ?
         double precision, parameter :: Ag94_Qneu = 4 ! ?
         double precision, parameter :: Ag95_Qneu = 4 ! ?
         double precision, parameter :: In98_Qneu = 4 ! ?
         double precision, parameter :: Sn100_Qneu = 4 ! ?
         double precision, parameter :: Sn101_Qneu = 4 ! ?
         double precision, parameter :: Sn102_Qneu = 4 ! ?
         double precision, parameter :: Sb104_Qneu = 4 ! ?    
         
         logical :: done
         integer :: time0, time1

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         y => n% y
         g => n% g
         itab => g% net_iso
         reaction_id => g% reaction_id

         ierr = 0

         ir = reaction_id(i)         

         if (reaction_outputs(1,ir) == 0) return ! skip aux reactions
         
         if (dbg) write(*,'(/,a)') ' reaction name <' // trim(reaction_Name(ir)) // '>'

         if (g% doing_derivs_timing) call system_clock(time0)
         
         select case(ir) ! check for special cases

            case(irn14ag_lite) ! n14 + 1.5 alpha => ne20
               he4 = itab(ihe4)
               n14 = itab(in14)
               ne20 = itab(ine20)
               r = y(n14) * y(he4)
               dr1 = y(n14)
               dr2 = y(he4)               
               call do_two_one(n, i, 1d0, n14, 1.5d0, he4, 
     >            r, 1d0, ne20, he4, dr1, n14, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irne22ga_fake) ! fake reaction to get ne22 to the c12 alpha chain: ne22(g,0.5a)ne20
               he4 = itab(ihe4)
               ne20 = itab(ine20)
               ne22 = itab(ine22)
               r = y(ne22)
               dr1 = 1d0       
               call do_one_two(
     >            n, i, 1d0, ne22, r, 0.5d0, he4, 1d0, ne20, ne22, dr1, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irne22ag_fake) ! fake reaction to get ne22 to the c12 alpha chain: ne22(0.5a,g)mg24
               he4 = itab(ihe4)
               ne22 = itab(ine22)
               mg24 = itab(img24)
               r = y(ne22) * y(he4)
               dr1 = y(ne22)
               dr2 = y(he4)
               call do_two_one(n, i, 1d0, ne22, 0.5d0, he4, 
     >                  r, 1d0, mg24, he4, dr1, ne22, dr2, deriv_flgs, category_factors)       
               if (g% doing_derivs_timing) call done_select
               return
                  
            case(irprot_to_neut) ! prot(e-nu)neut
               neut = itab(ineut)
               prot = itab(iprot)
               n% rate_raw(i_rate,i) = n% rpen*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               r = y(prot) 
               dr1 = 1  
               if (n% rpen > 1d-50) then
                  Qneu = n% spen/n% rpen
               else
                  Qneu = 0
               end if
               call do_one_one_neu(
     >            n, i, 1d0, prot, r, 1d0, neut, prot, dr1, 
     >            n% reaction_Qs(ir), Qneu, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irneut_to_prot) ! neut(e+nu)prot
               neut = itab(ineut)
               prot = itab(iprot)
               n% rate_raw(i_rate,i) = n% rnep*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               r = y(neut)
               dr1 = 1 
               if (n% rnep > 1d-50) then
                  Qneu = n% snep/n% rnep
               else
                  Qneu = 0
               end if
               call do_one_one_neu(
     >            n, i, 1d0, neut, r, 1d0, prot, neut, dr1, 
     >            n% reaction_Qs(ir), Qneu, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irhe4_breakup) ! he4(g,neut)he3(g,prot)h2(g,neut)prot
               ! since rate depends on abundances of neut and prot,
               ! need to write this as a 3 to 2 reaction
               neut = itab(ineut)
               prot = itab(iprot)
               he4 = itab(ihe4)
               r = y(he4)
               dr1 = 1d0 ! dr/dhe4
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr2 = r*n% d_rhe4_breakup_dneut/n% rate_screened(i_rate,i)
                  dr3 = r*n% d_rhe4_breakup_dprot/n% rate_screened(i_rate,i)
               else
                  dr2 = 0
                  dr3 = 0
               end if
               call do_three_two(
     >            n, i, 1d0, he4, 1d0, neut, 1d0, prot,
     >            r, 3d0, neut, 3d0, prot, 
     >            he4, dr1, neut, dr2, prot, dr3,
     >            deriv_flgs, category_factors)                   
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irhe4_rebuild) ! prot(neut,g)h2(prot,g)he3(neut,g)he4
               neut = itab(ineut)
               prot = itab(iprot)
               he4 = itab(ihe4)
               r = y(neut)**2 * y(prot)**2
               dr1 = 2*y(neut)**2*y(prot)
               dr2 = 2*y(neut)*y(prot)**2
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr1 = dr1 + r*n% d_rhe4_rebuild_dprot/n% rate_screened(i_rate,i)
                  dr2 = dr2 + r*n% d_rhe4_rebuild_dneut/n% rate_screened(i_rate,i)
               end if
               call do_two_one(n, i, 2d0, neut, 2d0, prot, 
     >            r, 1d0, he4, prot, dr1, neut, dr2, deriv_flgs, category_factors)       
               if (g% doing_derivs_timing) call done_select
               return

            case(irfe52neut_to_fe54) ! fe52(neut,g)fe53(neut,g)fe54
               neut = itab(ineut)
               fe52 = itab(ife52)
               fe54 = itab(ife54)
               r = y(fe52) * y(neut)**2
               dr1 = 2*y(fe52)*y(neut)
               dr2 = y(neut)**2
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr1 = dr1 + r*n% d_rfe52neut_to_fe54_dneut/n% rate_screened(i_rate,i)
               end if
               call do_two_one(n, i, 1d0, fe52, 2d0, neut, 
     >            r, 1d0, fe54, neut, dr1, fe52, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irfe54g_to_fe52) ! fe54(g,neut)fe53(g,neut)fe52
               ! since rate depends on abundance of neut,
               ! need to write this as a 2 to 2 reaction
               neut = itab(ineut)
               fe52 = itab(ife52)
               fe54 = itab(ife54)
               r = y(fe54)
               dr1 = 1d0
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr2 = r*n% d_rfe54g_to_fe52_dneut/n% rate_screened(i_rate,i)
               else
                  dr2 = 0
               end if
               call do_two_two(n, i, 1d0, fe54, 1d0, neut, 
     >            r, 3d0, neut, 1d0, fe52, fe54, dr1, neut, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irfe54prot_to_ni56) ! fe54(prot,g)co55(prot,g)ni56
               prot = itab(iprot)
               fe54 = itab(ife54)
               ni56 = itab(ini56)
               r = y(fe54) * y(prot)**2
               dr1 = 2 * y(fe54) * y(prot)
               dr2 = y(prot)**2
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr1 = dr1 + r*n% d_rfe54prot_to_ni56_dprot/n% rate_screened(i_rate,i)
               end if
               call do_two_one(n, i, 1d0, fe54, 2d0, prot, 
     >            r, 1d0, ni56, prot, dr1, fe54, dr2, deriv_flgs, category_factors)       
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irni56gprot_to_fe54) ! ni56(g,prot)co55(g,prot)fe54
               prot = itab(iprot)
               fe54 = itab(ife54)
               ni56 = itab(ini56)
               r = y(ni56)
               dr1 = 1d0        
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr2 = r*n% d_rni56gprot_to_fe54_dprot/n% rate_screened(i_rate,i)
               else
                  dr2 = 0
               end if
               call do_two_two(n, i, 1d0, ni56, 1d0, prot, 
     >            r, 3d0, prot, 1d0, fe54, ni56, dr1, prot, dr2, deriv_flgs, category_factors)       
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irfe52aprot_to_fe54) ! fe52(a,prot)co55(g,prot)fe54
               ! since rate depends on abundance of prot,
               ! need to write this as a 3 to 2 reaction
               prot = itab(iprot)
               he4 = itab(ihe4)
               fe52 = itab(ife52)
               fe54 = itab(ife54)
               r = y(fe52) * y(he4)
               dr1 = y(he4)
               dr2 = y(fe52)
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr3 = r*n% d_rfe52aprot_to_fe54_dprot/n% rate_screened(i_rate,i)
               else
                  dr3 = 0
               end if
               call do_three_two(
     >            n, i, 1d0, fe52, 1d0, he4, 1d0, prot,
     >            r, 1d0, fe54, 3d0, prot, 
     >            fe52, dr1, he4, dr2, prot, dr3,
     >            deriv_flgs, category_factors) 
               if (g% doing_derivs_timing) call done_select
               return
               
            case(irfe54prot_to_fe52) ! fe54(prot,g)co55(prot,a)fe52
               prot = itab(iprot)
               he4 = itab(ihe4)
               fe52 = itab(ife52)
               fe54 = itab(ife54)
               r = y(fe54) * y(prot)**2
               dr1 = 2 * y(fe54)* y(prot)
               dr2 = y(prot)**2
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr1 = dr1 + r*n% d_rfe54prot_to_fe52_dprot/n% rate_screened(i_rate,i)
               end if
               call do_two_two(n, i, 1d0, fe54, 2d0, prot, 
     >            r, 1d0, he4, 1d0, fe52, prot, dr1, fe54, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irfe52aprot_to_ni56) ! fe52(a,prot)co55(prot,g)ni56
               ! since rate depends on abundance of prot,
               ! need to write this as a 3 to 2 reaction
               prot = itab(iprot)
               he4 = itab(ihe4)
               fe52 = itab(ife52)
               ni56 = itab(ini56)
               r = y(fe52) * y(he4) * y(prot)
               dr1 = y(he4) * y(prot)
               dr2 = y(fe52) * y(prot)
               dr3 = y(fe52) * y(he4)
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr3 = dr3 + r*n% d_rfe52aprot_to_ni56_dprot/n% rate_screened(i_rate,i)
               end if
               call do_three_two(
     >            n, i, 1d0, fe52, 1d0, he4, 1d0, prot,
     >            r, 1d0, ni56, 1d0, prot, 
     >            fe52, dr1, he4, dr2, prot, dr3,
     >            deriv_flgs, category_factors)                   
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irni56gprot_to_fe52) ! ni56(g,prot)co55(prot,a)fe52
               prot = itab(iprot)
               he4 = itab(ihe4)
               fe52 = itab(ife52)
               ni56 = itab(ini56)
               r = y(ni56) * y(prot)
               dr1 = y(prot)
               dr2 = y(ni56)
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr2 = dr2 + r*n% d_rni56gprot_to_fe52_dprot/n% rate_screened(i_rate,i)
               end if
               call do_two_three(n, i, 1d0, ni56, 1d0, prot, 
     >            r, 1d0, prot, 1d0, he4, 1d0, fe52, 
     >            ni56, dr1, prot, dr2, deriv_flgs, category_factors)       
               if (g% doing_derivs_timing) call done_select
               return

            case(irfe54ng_to_fe56) ! fe54(neut,g)fe55(neut,g)fe56
               neut = itab(ineut)
               fe54 = itab(ife54)
               fe56 = itab(ife56)
               r = y(fe54) * y(neut)**2
               dr1 = 2*y(fe54)*y(neut)
               dr2 = y(neut)**2
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr1 = dr1 + r*n% d_rfe54neut_to_fe56_dneut/n% rate_screened(i_rate,i)
               end if
               call do_two_one(n, i, 1d0, fe54, 2d0, neut, 
     >            r, 1d0, fe56, neut, dr1, fe54, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return
            
            case(irfe56gn_to_fe54) ! fe56(g,neut)fe55(g,neut)fe54
               ! since rate depends on abundance of neut,
               ! need to write this as a 2 to 2 reaction
               neut = itab(ineut)
               fe54 = itab(ife54)
               fe56 = itab(ife56)
               r = y(fe56)
               dr1 = 1d0
               if (n% rate_screened(i_rate,i) > tiny_rate) then
                  dr2 = r*n% d_rfe56g_to_fe54_dneut/n% rate_screened(i_rate,i)
               else
                  dr2 = 0
               end if
               call do_two_two(n, i, 1d0, fe56, 1d0, neut, 
     >            r, 3d0, neut, 1d0, fe54, fe56, dr1, neut, dr2, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return

            case(irni56ec_to_fe54) ! ni56 + 2 e- => 56/54 fe54 in place of fe56
               neut = itab(ineut)
               fe54 = itab(ife54)
               ni56 = itab(ini56)
               r = y(ni56)
               call eval_mazurek_rate(temp,den,r,ye,rn56ec,sn56ec)
               n% rate_raw(i_rate,i) = rn56ec*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               dr1 = 1d0
               ! use eval_neutrino_Q to estimate Qneu
               !          Qneu ni56->co56    6.6873372603963444E-01
               !          Qneu co56->fe56    1.9289212685118804E+00
               !                      sum    2.5976549945515148E+00
!               Qneu = 2.6
!               call do_one_one_neu(n, i, 1d0, ni56, 
!     >            r, 56d0/54d0, fe54, ni56, dr1, Qneu, deriv_flgs, category_factors)
               call do_one_one(n, i, 1d0, ni56, 
     >            r, 56d0/54d0, fe54, ni56, dr1, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return

            case(ir_he4_he4_he4_to_c12) ! triple alpha
               if (g% which_rates(ir) == use_rate_3a_FL87) then 
                  ! use Fushiki and Lamb, Apj, 317, 368-388, 1987
                  call do_FL_3alf(i)
                  if (g% doing_derivs_timing) call done_select
                  return
               end if

            case(ir1212_to_ne20) ! c12 + c12 -> ne20 + he4
               if (g% which_rates(ir) == use_rate_1212_G05) then 
                  ! use Gasques, et al. Phys Review C, 72, 025806 (2005)
                  call do_G05_1212_to_ne20(i)
                  if (g% doing_derivs_timing) call done_select
                  return
               end if

            case(irni56ec_to_fe56) ! ni56 + 2 e- => fe56
               neut = itab(ineut)
               fe56 = itab(ife56)
               ni56 = itab(ini56)
               r = y(ni56)
               call eval_mazurek_rate(temp,den,r,ye,rn56ec,sn56ec)
               n% rate_raw(i_rate,i) = rn56ec*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               dr1 = 1d0
               call do_one_one(n, i, 1d0, ni56, 
     >            r, 1d0, fe56, ni56, dr1, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return

            case(irfe56ec_fake_to_cr56) ! fe56 + 2 e- => cr56
               neut = itab(ineut)
               cr56 = itab(icr56)
               fe56 = itab(ife56)
               r = y(fe56)
               call eval_mazurek_rate(temp,den,r,ye,rn56ec,sn56ec) ! use ni56 ec rate
               n% rate_raw(i_rate,i) = rn56ec*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               dr1 = 1d0
               call do_one_one(n, i, 1d0, fe56, 
     >            r, 1d0, cr56, fe56, dr1, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return

            case(irn14ecap) ! n14(e-,nu)c14
               n14 = itab(in14)
               c14 = itab(ic14)
               r = y(n14)
               call eval_n14_electron_capture_rate(temp,den,1/ye,rn14ec)
               n% rate_raw(i_rate,i) = rn14ec*rate_factors(i)
               n% rate_raw(i_rate_dT,i) = 0
               n% rate_raw(i_rate_dRho,i) = 0
               n% rate_screened(:,i) = n % rate_raw(:,i)
               dr1 = 1d0
               call do_one_one(n, i, 1d0, n14, 
     >            r, 1d0, c14, n14, dr1, deriv_flgs, category_factors)
               if (g% doing_derivs_timing) call done_select
               return

         end select
         
         if (g% doing_derivs_timing) call done_select

         num_reaction_inputs = get_num_reaction_inputs(ir)
         num_reaction_outputs = get_num_reaction_outputs(ir)

         if (dbg) write(*,*) 'num_reaction_inputs', num_reaction_inputs
         if (dbg) write(*,*) 'num_reaction_outputs', num_reaction_outputs
         if (dbg) write(*,*)

         ! NB: num_reaction_inputs is NOT the number of input particles.
         ! each input consists of 1 or more particles of a particular type.
         ! e.g., for triple alpha, num_reaction_inputs = 1, with num=3 and type=he4
         
         if (num_reaction_inputs == 1) then
            
            cin1 = reaction_inputs(1,ir); din1 = cin1
            in1 = reaction_inputs(2,ir)
            i1 = itab(in1)
            
            if (i1 == 0) then
               write(*,*) trim(reaction_Name(ir))
               write(*,2) 'num_reaction_inputs', num_reaction_inputs
               stop 'get1_derivs: itab(in1) = 0'
            end if
            
            if (cin1 == 1) then
               r = y(i1)
               idr1 = i1
               dr1 = 1
            else if (cin1 == 3 .and. in1 /= ih1) then ! 3 he4
               !write(*,'(/,a)') '1/6*r  reaction name <' // trim(reaction_Name(ir)) // '>'
               r = (1d0/6d0)*y(i1)**3
               idr1 = i1
               dr1 = 0.5d0*y(i1)**2
            else ! 2 body
               !write(*,'(/,a)') '1/2*r  reaction name <' // trim(reaction_Name(ir)) // '>'
               !write(*,'(i3,3x,99e20.10)') i, n% rate_raw(i_rate,i), n% rate_screened(i_rate,i)
               r = 0.5d0*y(i1)**2
               idr1 = i1
               dr1 = y(i1)
               !stop
            end if
            
         else if (num_reaction_inputs == 2 .or. num_reaction_inputs == 3) then
            
            cin1 = reaction_inputs(1,ir); din1 = cin1
            in1 = reaction_inputs(2,ir)
            i1 = itab(in1)            
            if (i1 == 0) then
               write(*,*) trim(reaction_Name(ir))
               write(*,2) 'num_reaction_inputs', num_reaction_inputs
               stop 'get1_derivs: itab(in1) = 0'
            end if
            
            cin2 = reaction_inputs(3,ir); din2 = cin2
            in2 = reaction_inputs(4,ir)
            i2 = itab(in2)
            if (i2 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(in2) = 0'
            end if
            
            if (reaction_ye_rho_exponents(2,ir) == 0) then
               ! treat as 1 body reaction
               r = y(i1)
               idr1 = i1
               dr1 = 1
               idr2 = i2
               dr2 = 0
               !write(*,*) 'get1_derivs rho=0: ' // trim(reaction_Name(ir))
               !stop 'net_derivs'
            else if ((cin1 == 1 .and. cin2 == 1) .or. reaction_ye_rho_exponents(2,ir) == 1) then
               ! treat as 2 body reaction
               r = y(i1)*y(i2)
               dr1 = y(i1)
               idr1 = i2
               dr2 = y(i2)
               idr2 = i1
            else if (cin1 == 2 .and. cin2 == 1) then 
               r = 0.5d0*y(i1)**2*y(i2)
               dr1 = 0.5d0*y(i1)**2
               idr1 = i2
               dr2 = y(i1)*y(i2)
               idr2 = i1
            else if (cin1 == 1 .and. cin2 == 2) then 
               ! e.g., rhe4p, r_neut_he4_he4_to_be9, r_neut_h1_h1_to_h1_h2
               r = y(i1)*0.5d0*y(i2)**2
               dr1 = y(i1)*y(i2)
               idr1 = i2
               dr2 = 0.5d0*y(i2)**2
               idr2 = i1
            else if (cin1 == 2 .and. cin2 == 2) then 
               ! e.g., r_neut_neut_he4_he4_to_h3_li7, r_h1_h1_he4_he4_to_he3_be7
               r = 0.5d0*y(i1)**2*0.5d0*y(i2)**2
               dr1 = 0.5d0*y(i1)**2*y(i2)
               idr1 = i2
               dr2 = y(i1)*0.5d0*y(i2)**2
               idr2 = i1
            else
               write(*,*) 'get1_derivs: ' // trim(reaction_Name(ir)) // ' invalid coefficient'
               stop 'get1_derivs'
            end if            
            
            if (num_reaction_inputs == 3) then
               ! we assume that the 3rd kind of input is just "along for the ride"
               ! e.g., some compound reactions such as r34_pp2 are in this category.
               cin3 = reaction_inputs(5,ir); din3 = cin3
               in3 = reaction_inputs(6,ir)
               i3 = itab(in3)
               dr3 = 0
               idr3 = i3
               if (i3 == 0) then
                  write(*,*) trim(reaction_Name(ir))
                  stop 'get1_derivs: itab(in3) = 0'
               end if
            end if

         else

            write(*,*) 'get1_derivs: ' // trim(reaction_Name(ir)) // ' invalid specification'
            stop 'get1_derivs'

         end if

         if (num_reaction_outputs >= 1) then
            cout1 = reaction_outputs(1,ir); dout1 = cout1
            out1 = reaction_outputs(2,ir)
            o1 = itab(out1)
            if (o1 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out1) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 2) then
            cout2 = reaction_outputs(3,ir); dout2 = cout2
            out2 = reaction_outputs(4,ir)
            o2 = itab(out2)
            if (o2 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out2) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 3) then
            cout3 = reaction_outputs(5,ir); dout3 = cout3
            out3 = reaction_outputs(6,ir)
            o3 = itab(out3)
            if (o3 == 0) then
               write(*,*) trim(reaction_Name(ir))
               stop 'get1_derivs: itab(out3) = 0'
            end if
         end if
            
         if (num_reaction_outputs >= 4) then
            write(*,*) trim(reaction_Name(ir))
            stop 'get1_derivs: num_reaction_outputs >= 4'
         end if
         
         if (num_reaction_inputs == 1) then
         
            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 -> dout1 out1
               if (dbg) write(*,*) ' do_one_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               
               weak_id = g% weak_reaction_index(i)
               done = .false.
               if (weak_id > 0) then
                  if (g% weaklib_ids(weak_id) > 0) then ! 0 means not included in weaklib
!                     write(*,1) 'weak ' // trim(reaction_Name(ir)) // ' old, new 1/2 life',
!     >                  log(2d0)/max(n% rate_screened(i_rate,i), 1d-99),
!     >                  log(2d0)/max(n% lambda(weak_id), 1d-99)
                     n% rate_screened(i_rate,i) = n% lambda(weak_id)
                     n% rate_screened(i_rate_dT,i) = 0
                     n% rate_screened(i_rate_dRho,i) = 0
                     !write(*,*)
                     !stop
                     call do_one_one_neu(
     >                  n, i, din1, i1, r, dout1, o1, 
     >                  idr1, dr1, n% Q(weak_id), n% Qneu(weak_id), deriv_flgs, category_factors)
                     done = .true.
                     if (.false. .and. reaction_Name(ir) == 'r_h1_h1_ec_h2') then
                        write(*,'(i3,3x,a,2x,99e20.10)') i, 
     >                     trim(chem_isos% name(g% chem_id(i1))) // ' => ' // 
     >                     trim(chem_isos% name(g% chem_id(o1))), 
     >                     n% rate_screened(i_rate,i), n% Qneu(weak_id)
                        write(*,1) 'weak ' // trim(reaction_Name(ir)) // ' old, new 1/2 life',
     >                     log(2d0)/max(n% rate_screened(i_rate,i), 1d-99),
     >                     log(2d0)/max(n% lambda(weak_id), 1d-99)
                        stop
                     end if
                  end if
               end if
               if (.not. done) then
                  call do_one_one(
     >               n, i, din1, i1, r, dout1, o1, 
     >               idr1, dr1, deriv_flgs, category_factors)
               end if
     
            else if (num_reaction_outputs == 2) then
               ! reaction of form cin1 in1 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_one_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_one_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))

               if (.false. .and. reaction_Name(ir) == 'r_c9_gp_b8') then
                  write(*,2) trim(reaction_Name(ir)) // ' rate r', ir, n% rate_screened(i_rate,i), r
                  write(*,*)
                  !stop
               end if
               call do_one_two(
     >            n, i, din1, i1, r, dout1, o1, dout2, o2, 
     >            idr1, dr1, deriv_flgs, category_factors)
            else if (num_reaction_outputs == 3) then
               ! reaction of form cin1 in1 -> dout1 out1 + dout2 out2 + dout3 out3
               if (dbg) write(*,*) ' do_one_three din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) 'do_one_three dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_one_three dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               if (dbg) write(*,*) 'do_one_three dout3', dout3, trim(chem_isos% name(g% chem_id(o3)))
               call do_one_three(
     >            n, i, din1, i1, r, dout1, o1, dout2, o2, dout3, o3, 
     >            idr1, dr1, deriv_flgs, category_factors)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 1'
            end if
            
         else if (num_reaction_inputs == 2) then
         
            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1
               if (dbg) write(*,*) ' do_two_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_one din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (.false. .and. reaction_Name(ir) == 'r_h3_pg_he4') then
                  write(*,'(i3,3x,a,2x,99e20.10)') i, 
     >               'do_two_one ' // trim(reaction_Name(ir)) // ' ' //
     >               trim(chem_isos% name(g% chem_id(i1))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(i2))) // ' => ' // 
     >               trim(chem_isos% name(g% chem_id(o1))), 
     >               r, dr1, dr2
                  stop
               end if
               call do_two_one(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
            else if (num_reaction_outputs == 2) then
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_two_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_two din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_two_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               if (.false.) write(*,'(i3,3x,a,2x,99e20.10)') i, 
     >               trim(chem_isos% name(g% chem_id(i1))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(i2))) // ' => ' // 
     >               trim(chem_isos% name(g% chem_id(o1))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(o2))) // ' ', 
     >               n% rate_raw(i_rate,i)/den, den
               call do_two_two(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, dout2, o2, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
            else if (num_reaction_outputs == 3) then
               ! reaction of form din1 in1 + din2 in2 -> dout1 out1 + dout2 out2 + dout3 out3
               if (dbg) write(*,*) ' do_two_three din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_two_three din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) 'do_two_three dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_two_three dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               if (dbg) write(*,*) 'do_two_three dout3', dout3, trim(chem_isos% name(g% chem_id(o3)))
               
               if (.false. .and. reaction_Name(ir) == 'h_h1_he4_to_h2_he3') then
                  write(*,2) trim(reaction_Name(ir)) // ' rate_screened r, dr1, dr2', 
     >                  ir, n% rate_screened(i_rate,i), r, dr1, dr2
                  write(*,*)
                  stop
               end if
               
               call do_two_three(
     >            n, i, din1, i1, din2, i2, r, dout1, o1, dout2, o2, dout3, o3, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 2'
            end if
            
         else if (num_reaction_inputs == 3) then

            if (num_reaction_outputs == 1) then 
               ! reaction of form din1 in1 + din2 in2 + din3 in3 -> dout1 out1
               if (dbg) write(*,*) ' do_three_one din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_three_one din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) ' do_three_one din3', din3, trim(chem_isos% name(g% chem_id(i3)))
               if (dbg) write(*,*) 'do_three_one dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (.false. .and. reaction_Name(ir) == '') then
                  write(*,'(i3,3x,a,2x,99e20.10)') i, 
     >               'do_three_one' // ' ' //trim(chem_isos% name(g% chem_id(i1))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(i2))) // ' + ' // 
     >               trim(chem_isos% name(g% chem_id(i3))) // ' => ' // 
     >               trim(chem_isos% name(g% chem_id(o1))), 
     >               n% rate_raw(i_rate,i)/den, den
               end if
               call do_three_one(
     >            n, i, din1, i1, din2, i2, din3, i3, r, dout1, o1, 
     >            idr1, dr1, idr2, dr2, deriv_flgs, category_factors)
            else if (num_reaction_outputs == 2) then
               ! reaction of form din1 in1 + din2 in2 + din3 in3 -> dout1 out1 + dout2 out2
               if (dbg) write(*,*) ' do_three_two din1', din1, trim(chem_isos% name(g% chem_id(i1)))
               if (dbg) write(*,*) ' do_three_two din2', din2, trim(chem_isos% name(g% chem_id(i2)))
               if (dbg) write(*,*) ' do_three_two din3', din3, trim(chem_isos% name(g% chem_id(i3)))
               if (dbg) write(*,*) 'do_three_two dout1', dout1, trim(chem_isos% name(g% chem_id(o1)))
               if (dbg) write(*,*) 'do_three_two dout2', dout2, trim(chem_isos% name(g% chem_id(o2)))
               call do_three_two(
     >            n, i, din1, i1, din2, i2, din3, i3, r, dout1, o1, dout2, o2,
     >            idr1, dr1, idr2, dr2, idr3, dr3, deriv_flgs, category_factors)
            else
               write(*,*) trim(reaction_Name(ir))
               stop 'too many reaction_outputs for num_reaction_inputs == 3'
            end if
            
         else
            stop 'too many reaction_inputs'
         end if
         


         if (g% doing_derivs_timing) then
            call system_clock(time1)
            g% clock_derivs_general = g% clock_derivs_general + (time1 - time0)
            time0 = time1
         end if


               if (.false. .and. (reaction_Name(ir) == 'r_n14_an_f17' .or. 
     >               reaction_Name(ir) == 'r_f17_na_n14')) then
                  write(*,*)
                  write(*,2) trim(reaction_Name(ir)) // ' rate/rho', 
     >               ir, n% rate_raw(i_rate,i)/den
                  write(*,2) trim(reaction_Name(ir)) // ' rate dr_dlnRho/rho', 
     >               ir, n% rate_raw(i_rate_dRho,i)
                  write(*,2) trim(reaction_Name(ir)) // ' rate dr_dlnT/rho', 
     >               ir, n% rate_raw(i_rate_dT,i)*temp/den

                  write(*,2) trim(reaction_Name(ir)) // ' rho rate', ir, n% rate_raw(i_rate,i)
                  write(*,2) trim(reaction_Name(ir)) // ' rho rate dr_dlnRho',  
     >               ir, n% rate_raw(i_rate_dRho,i)*den
                  write(*,2) trim(reaction_Name(ir)) // ' rho rate dr_dlnT',  
     >               ir, n% rate_raw(i_rate_dT,i)*temp
                  
                  write(*,2) trim(reaction_Name(ir)) // ' screened rate',  
     >               ir, n% rate_screened(i_rate,i)
                  write(*,2) trim(reaction_Name(ir)) // ' screened dr_dlnRho',  
     >               ir, n% rate_screened(i_rate_dRho,i)*den
                  write(*,2) trim(reaction_Name(ir)) // ' screened dr_dlnT',  
     >               ir, n% rate_screened(i_rate_dT,i)*temp
                  write(*,*)
                  !stop
               end if
               
               
         
         contains

         
         subroutine done_select
            call system_clock(time1)
            g% clock_derivs_select = g% clock_derivs_select + (time1 - time0)
            time0 = time1
         end subroutine done_select
               
               
         double precision function get_Qneu(lhs, rhs)
            use weak_lib, only: get_weak_info_list_id
            use weak_def, only: weak_info_list_Qneu, weak_info_life_halflife
			   character (len=*), intent(in) :: lhs, rhs
			   character (len=iso_name_length) :: lhs_name, rhs_name
            integer :: i
            lhs_name = lhs; rhs_name = rhs
            i = get_weak_info_list_id(lhs_name, rhs_name)
            if (i > 0) then
               get_Qneu = weak_info_list_Qneu(i)
            else
               get_Qneu = 0
               write(*,'(a)') 
     >            'failed to find weak info for ' // trim(lhs) // ' ' // trim(rhs)
            end if
         end function get_Qneu
               
               
         subroutine get_Qneu_and_beta_rate(lhs, rhs, Qneu, beta_rate)
            use weak_lib, only: get_weak_info_list_id
            use weak_def, only: weak_info_list_Qneu, weak_info_life_halflife
			   character (len=*), intent(in) :: lhs, rhs
			   double precision, intent(out) :: Qneu, beta_rate
			   character (len=iso_name_length) :: lhs_name, rhs_name
            integer :: i
            lhs_name = lhs; rhs_name = rhs
            i = get_weak_info_list_id(lhs_name, rhs_name)
            if (i > 0) then
               Qneu = weak_info_list_Qneu(i)
               beta_rate = lntwo/weak_info_life_halflife(i)
            else
               Qneu = 0
               beta_rate = 0
               write(*,'(a)') 
     >            'failed to find weak info for ' // trim(lhs) // ' ' // trim(rhs)
            end if
         end subroutine get_Qneu_and_beta_rate
         
         
         subroutine do_FL_3alf(i) ! Fushiki and Lamb, Apj, 317, 368-388, 1987
            integer, intent(in) :: i
            integer :: he4, c12
            double precision :: UE, XHe4, YHe4, 
     >            FLeps_nuc, dFLeps_nuc_dT, dFLeps_nuc_dRho, r, drdT, drdRho, conv
            include 'formats.dek'
            he4 = itab(ihe4)
            c12 = itab(ic12)
            UE = abar/zbar
            YHe4 = y(he4)
            XHe4 = 4*YHe4
            if (YHe4 < 1d-50) then
               n% rate_raw(:,i) = 0
               n% rate_screened(:,i) = 0
               r = 0
               dr1 = 0
            else
               call eval_FL_epsnuc_3alf(temp, den, XHe4, UE, FLeps_nuc, dFLeps_nuc_dT, dFLeps_nuc_dRho)
               conv = Qconv*n% reaction_Qs(ir)
               r = YHe4**3/6d0
               dr1 = 0.5d0*YHe4**2
               n% rate_raw(i_rate,i) = FLeps_nuc/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dT,i) = dFLeps_nuc_dT/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dRho,i) = dFLeps_nuc_dRho/r*rate_factors(i)/conv         
               n% rate_screened(:,i) = n % rate_raw(:,i)
            end if
            call do_one_one(n, i, 3d0, he4, r, 1d0, c12, he4, dr1, deriv_flgs, category_factors)
         end subroutine do_FL_3alf
         
         
         subroutine do_G05_1212_to_ne20(i) ! use Gasques, et al. Phys Review C, 72, 025806 (2005)
            integer, intent(in) :: i
            integer :: he4, c12, ne20
            double precision :: X12, Y12, eps, deps_dT, deps_dRho, r, drdT, drdRho, conv
            include 'formats.dek'
            he4 = itab(ihe4)
            c12 = itab(ic12)
            ne20 = itab(ine20)
            Y12 = y(c12)
            X12 = 12*Y12
            if (Y12 < 1d-50) then
               n% rate_raw(:,i) = 0
               n% rate_screened(:,i) = 0
               r = 0
               dr1 = 0
            else
               call eval_G05_epsnuc_CC(temp, den, X12, eps, deps_dT, deps_dRho)
               conv = Qconv*n% reaction_Qs(ir)
               r = Y12**2/2d0
               dr1 = Y12
               n% rate_raw(i_rate,i) = eps/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dT,i) = deps_dT/r*rate_factors(i)/conv
               n% rate_raw(i_rate_dRho,i) = deps_dRho/r*rate_factors(i)/conv         
               n% rate_screened(:,i) = n % rate_raw(:,i)
            end if
            call do_one_two( n, i, 2d0, c12, r,
     >         1d0, ne20, 1d0, he4, c12, dr1, deriv_flgs, category_factors)
      
            if (.true. .or. den < 1.6d10) return
            write(*,1) 'temp', temp
            write(*,1) 'den', den
            write(*,1) 'X12', X12
            write(*,1) 'eps', eps
            write(*,1) 'deps_dT', deps_dT
            write(*,1) 'deps_dRho', deps_dRho
            write(*,*) 
            stop 'do_G05_1212_to_ne20'
            
         end subroutine do_G05_1212_to_ne20

          
      end subroutine get1_derivs


      end module net_derivs















