/*
 * -- SuperLU MT routine (version 1.0) --
 * Univ. of California Berkeley, Xerox Palo Alto Research Center,
 * and Lawrence Berkeley National Lab.
 * August 15, 1997
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include "pdsp_defs.h"
#include "slu_mt_util.h"
#include "slu_mt_Cnames.h"



typedef long long fptr;  /* 64-bit by default */

typedef struct {
    SuperMatrix *L;
    SuperMatrix *U;
    int *perm_c;
    int *perm_r;
} factors_t;



void
copy_pdgssv(int *iopt, int nprocs, SuperMatrix *A, int *perm_c, int *perm_r, 
       SuperMatrix *L, SuperMatrix *U, SuperMatrix *B,
		 fptr *f_factors, /* a handle containing the address
				     pointing to the factored matrices */
		 int *info )
{
    trans_t  trans;
    NCformat *Astore;
    DNformat *Bstore;
    SCPformat *Lstore;
    NCPformat *Ustore;
    SuperMatrix *AA; /* A in NC format used by the factorization routine.*/
    SuperMatrix AC; /* Matrix postmultiplied by Pc */
    int i, n, panel_size, relax;
    fact_t   fact;
    yes_no_t refact, usepr;
    double diag_pivot_thresh, drop_tol;
    void *work;
    int lwork;
    superlumt_options_t superlumt_options;
    Gstat_t  Gstat;
    double   t; /* Temporary time */
    double   *utime;
    flops_t  *ops, flopcnt;

    /* ------------------------------------------------------------
       Test the input parameters.
       ------------------------------------------------------------*/
    Astore = A->Store;
    Bstore = B->Store;
    *info = 0;
    if ( nprocs <= 0 ) *info = -1;
    else if ( A->nrow != A->ncol || A->nrow < 0 || 
	      (A->Stype != SLU_NC && A->Stype != SLU_NR) ||
	      A->Dtype != SLU_D || A->Mtype != SLU_GE )
	*info = -2;
    else if ( B->ncol < 0 || Bstore->lda < SUPERLU_MAX(1, A->nrow) )*info = -7;
    if ( *info != 0 ) {
        i = -(*info);
	xerbla_("pdgssv", &i);
	return;
    }

    fact               = EQUILIBRATE;
    refact             = NO;
    trans              = NOTRANS;
    panel_size         = sp_ienv(1);
    relax              = sp_ienv(2);
    diag_pivot_thresh  = 1.0;
    usepr              = NO;
    drop_tol           = 0.0;
    work               = NULL;
    lwork              = 0;

    /* ------------------------------------------------------------
       Allocate storage and initialize statistics variables. 
       ------------------------------------------------------------*/
    n = A->ncol;
    
    StatAlloc(n, nprocs, panel_size, relax, &Gstat);
    StatInit(n, nprocs, &Gstat);
    utime = Gstat.utime;
    ops = Gstat.ops;

    /* ------------------------------------------------------------
       Convert A to NC format when necessary.
       ------------------------------------------------------------*/
    if ( A->Stype == SLU_NR ) {
	NRformat *Astore = A->Store;
	AA = (SuperMatrix *) SUPERLU_MALLOC( sizeof(SuperMatrix) );
	dCreate_CompCol_Matrix(AA, A->ncol, A->nrow, Astore->nnz, 
			       Astore->nzval, Astore->colind, Astore->rowptr,
			       SLU_NC, A->Dtype, A->Mtype);
	trans = TRANS;
    } else if ( A->Stype == SLU_NC ) AA = A;

    /* ------------------------------------------------------------
       Initialize the option structure superlumt_options using the
       user-input parameters;
       Apply perm_c to the columns of original A to form AC.
       ------------------------------------------------------------*/
       
       //printf("pdgstrf_init nprocs %i ncol %i nrow %i nnz %i\n", nprocs, A->ncol, A->nrow, Astore->nnz);
       
    pdgstrf_init(nprocs, fact, trans, refact, panel_size, relax,
		 diag_pivot_thresh, usepr, drop_tol, perm_c, perm_r,
		 work, lwork, AA, &AC, &superlumt_options, &Gstat);

    /* ------------------------------------------------------------
       Compute the LU factorization of A.
       The following routine will create nprocs threads.
       ------------------------------------------------------------*/
       
       
       
       //printf("pdgstrf\n");
       
       
       
    pdgstrf(&superlumt_options, &AC, perm_r, L, U, &Gstat, info);

    flopcnt = 0;
    for (i = 0; i < nprocs; ++i) flopcnt += Gstat.procstat[i].fcops;
    ops[FACT] = flopcnt;

#if ( PRNTlevel==1 )
    printf("nprocs = %d, flops %e, Mflops %.2f\n",
	   nprocs, flopcnt, flopcnt/utime[FACT]*1e-6);
    printf("Parameters: w %d, relax %d, maxsuper %d, rowblk %d, colblk %d\n",
	   sp_ienv(1), sp_ienv(2), sp_ienv(3), sp_ienv(4), sp_ienv(5));
    fflush(stdout);
#endif
    
    
    
    
    
    //printf("\t#NZ in L+U = %d\n", Lstore->nnz + Ustore->nnz - L->ncol);

    /* ------------------------------------------------------------
       Solve the system A*X=B, overwriting B with X.
       ------------------------------------------------------------*/
    if ( *info == 0 ) {
        t = SuperLU_timer_();


       //printf("dgstrs\n");



	dgstrs (trans, L, U, perm_r, perm_c, B, &Gstat, info);
	utime[SOLVE] = SuperLU_timer_() - t;
	ops[SOLVE] = ops[TRISOLVE];
    }

    /* ------------------------------------------------------------
       Deallocate storage after factorization.
       ------------------------------------------------------------*/
       //printf("pxgstrf_finalize\n");
       
    pxgstrf_finalize(&superlumt_options, &AC);
    if ( A->Stype == SLU_NR ) {


       //printf("Destroy_SuperMatrix_Store\n");


	Destroy_SuperMatrix_Store(AA);
	SUPERLU_FREE(AA);
    }

    /* ------------------------------------------------------------
       Print timings, then deallocate statistic variables.
       ------------------------------------------------------------*/
#ifdef PROFILE
    {
	SCPformat *Lstore = (SCPformat *) L->Store;
	ParallelProfile(n, Lstore->nsuper+1, Gstat.num_panels, nprocs, &Gstat);
    }
#endif


// uncomment the following to get statistics

//	Lstore = (SCPformat *) L->Store;
//	Ustore = (NCPformat *) U->Store;
//    	printf("#NZ in factor L = %d\n", Lstore->nnz);
//    	printf("#NZ in factor U = %d\n", Ustore->nnz);
//    	printf("#NZ in L+U = %d\n", Lstore->nnz + Ustore->nnz - L->ncol);

//    PrintStat(&Gstat);

    StatFree(&Gstat);
}



void
c_bridge_pdgssv_(int *iopt, int *nprocs, int *n, int *nnz, int *nrhs, double *values,
		 int *rowind, int *colptr, double *b, int *ldb,
		 fptr *f_factors, /* a handle containing the address
				     pointing to the factored matrices */
		 int *info)
{
    SuperMatrix A, B, L, U;
    SCformat *Lstore;
    NCformat *Ustore;
    NCformat *Astore;
    int      *perm_r; /* row permutations from partial pivoting */
    int      *perm_c; /* column permutation vector */
    int      panel_size, permc_spec, i;
    superlu_memusage_t superlu_memusage;


       //printf("c_bridge_pdgssv iopt %i nprocs %i n %i nnz %i\n", *iopt, *nprocs, *n, *nnz);



    /* Adjust to 0-based indexing */
    for (i = 0; i < *nnz; ++i) --rowind[i];
    for (i = 0; i <= *n; ++i) --colptr[i];
    
    
    //printf("dCreate_CompCol_Matrix\n");
    dCreate_CompCol_Matrix(&A, *n, *n, *nnz, values, rowind, colptr, 
			   SLU_NC, SLU_D, SLU_GE);
    Astore = A.Store;
    //printf("Dimension %dx%d; # nonzeros %d\n", A.nrow, A.ncol, Astore->nnz);
			   
			   
			   
    //printf("dCreate_Dense_Matrix\n");
    dCreate_Dense_Matrix(&B, *n, *nrhs, b, *ldb, SLU_DN, SLU_D, SLU_GE);

    //printf("perm_r malloc\n");
    if ( !(perm_r = intMalloc(*n)) ) { *info = -1; return; } //ABORT("Malloc fails for perm_r[].");
    //printf("perm_c malloc\n");
    if ( !(perm_c = intMalloc(*n)) ) { *info = -1; return; } //ABORT("Malloc fails for perm_c[].");

    /*
     * Get column permutation vector perm_c[], according to permc_spec:
     *   permc_spec = 0: natural ordering 
     *   permc_spec = 1: minimum degree ordering on structure of A'*A
     *   permc_spec = 2: minimum degree ordering on structure of A'+A
     *   permc_spec = 3: approximate minimum degree for unsymmetric matrices
     */    	
     permc_spec = 3;
    //printf("get_perm_c\n");
    get_perm_c(permc_spec, &A, perm_c);

    //printf("panel_size\n");
    panel_size = sp_ienv(1);
    
    
    
    
    //printf("copy_pdgssv\n");
    
    
    copy_pdgssv(iopt, *nprocs, &A, perm_c, perm_r, &L, &U, &B, f_factors, info);
    
    
    
    
    
    
    if ( *info == 0 ) {
	Lstore = (SCformat *) L.Store;
	Ustore = (NCformat *) U.Store;
    }

    SUPERLU_FREE (perm_r);
    SUPERLU_FREE (perm_c);
    Destroy_SuperMatrix_Store(&A);
    Destroy_SuperMatrix_Store(&B);
    Destroy_SuperNode_Matrix(&L);
    Destroy_CompCol_Matrix(&U);

    /* Restore to 1-based indexing */
    for (i = 0; i < *nnz; ++i) ++rowind[i];
    for (i = 0; i <= *n; ++i) ++colptr[i];

}



