! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


#ifdef DBLE
      module thomas_block_dble
      use my_lapack95_dble
      use utils_lib, only: set_pointer_2
#else
      module thomas_block_quad
      use my_lapack95_quad
      use utils_lib, only: set_quad_pointer_2
#endif
      
      use const_def, only: dp

      implicit none
      
      contains

#ifdef DBLE
#define set_ptr_2 set_pointer_2
#else
#define set_ptr_2 set_quad_pointer_2
#endif


      subroutine thomas_factor(id, mblk, nblk, lblk, dblk, ublk, ipiv, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
         integer, intent(in) :: id, nblk, mblk, lid, lrd
         real(fltp), pointer, intent(inout) :: lblk(:,:,:), dblk(:,:,:), ublk(:,:,:)
            ! row(i) of mtx has lblk(:,:,i), dblk(:,:,i), ublk(:,:,i)
            ! lblk(:,:,1) is not used; ublk(:,:,nz) is not used.
         integer, pointer, intent(out) :: ipiv(:,:) ! (mblk,nblk)
         real(dp), pointer, intent(inout) :: rpar_decsol(:) ! (lrd)
         integer, pointer, intent(inout) :: ipar_decsol(:) ! (lid)
         integer, intent(out) :: ierr

         integer :: ns, ns1, i, j, nrhs
         character :: fact, trans, equed
         integer, pointer :: ipivot(:), iwork(:)
         real(fltp), pointer :: dmat(:,:), umat(:,:), lmat(:,:), &
            blk_copy(:,:), work(:), ferr(:), berr(:), rcond(:)
         real(fltp) :: sfmin
#ifdef DBLE
         real(fltp) :: dlamch
         sfmin = dlamch('S')  
#else
         real(fltp) :: qlamch
         sfmin = qlamch('S')  
#endif
         
         include 'formats.dek'

         ierr = 0

         if (size(lblk,1) /= mblk .or. size(lblk,2) /= mblk .or. size(lblk,3) < nblk) then
            write(*,*) 'thomas_factor lblk wrong dims'
            write(*,2) 'size(lblk,1)', size(lblk,1)
            write(*,2) 'size(lblk,2)', size(lblk,2)
            write(*,2) 'mblk', mblk
            write(*,2) 'size(lblk,3)', size(lblk,3)
            write(*,2) 'nblk', nblk
            ierr = -1
            return
         end if
         if (size(dblk,1) /= mblk .or. size(dblk,2) /= mblk .or. size(dblk,3) < nblk) then
            write(*,*) 'thomas_factor dblk wrong dims'
            write(*,2) 'size(dblk,1)', size(dblk,1)
            write(*,2) 'size(dblk,2)', size(dblk,2)
            write(*,2) 'mblk', mblk
            write(*,2) 'size(dblk,3)', size(dblk,3)
            write(*,2) 'nblk', nblk
            ierr = -1
            return
         end if
         if (size(ublk,1) /= mblk .or. size(ublk,2) /= mblk .or. size(ublk,3) < nblk) then
            write(*,*) 'thomas_factor ublk wrong dims'
            write(*,2) 'size(ublk,1)', size(ublk,1)
            write(*,2) 'size(ublk,2)', size(ublk,2)
            write(*,2) 'mblk', mblk
            write(*,2) 'size(ublk,3)', size(ublk,3)
            write(*,2) 'nblk', nblk
            ierr = -1
            return
         end if
         if (size(ipiv,1) /= mblk .or. size(ipiv,2) < nblk) then
            write(*,*) 'thomas_factor ipiv wrong dims'
            ierr = -1
            return
         end if
         if (nblk <= 0 .or. mblk <= 0) return
         
         do ns = nblk, 1, -1
            dmat => dblk(:,:,ns)
            ipivot => ipiv(:,ns)
            call my_getf2(mblk, dmat, mblk, ipivot, sfmin, ierr)
            if (ierr /= 0) then
               exit
               !          < 0:  if info = -i, the i-th argument had an illegal value
               !          > 0:  if info = i, u(i,i) is exactly zero. the factorization
               write (6, '(x,a,i4)') 'error factoring matrix in thomas_factor: block = ', ns
               if (ierr < 0) write (6,'(i4, a)') ierr, 'th argument has illegal value'
               if (ierr > 0) write (6,'(i4, a)') ierr, 'th diagonal factor exactly zero'
               exit
            end if
            if (ns == 1) exit
            lmat => lblk(:,:,ns)
            call my_getrs(mblk, mblk, dmat, mblk, ipivot, lmat, mblk, ierr)
            if (ierr /= 0) exit
            ns1 = ns-1 
            umat => ublk(:,:,ns1)
            dmat => dblk(:,:,ns1)
            call my_gemm(mblk,mblk,mblk,umat,mblk,lmat,mblk,dmat,mblk)
         end do


      end subroutine thomas_factor

      
      subroutine thomas_solve(id, mblk, nblk, lblk, dblk, ublk, brhs, nrhs, ipiv, &
               lrd, rpar_decsol, lid, ipar_decsol, ierr)
         integer, intent(in) :: id, nblk, mblk, lrd, lid
         real(fltp), pointer, intent(in) :: lblk(:,:,:), dblk(:,:,:), ublk(:,:,:) ! (mblk,mblk,nblk)
         real(fltp), pointer, intent(inout) :: brhs(:,:,:) ! (mblk,nrhs,nblk)
         integer, intent(in) :: nrhs
         integer, pointer, intent(in) :: ipiv(:,:) ! (mblk,nblk)
         real(dp), pointer, intent(inout) :: rpar_decsol(:) ! (lrd)
         integer, pointer, intent(inout) :: ipar_decsol(:) ! (lid)
         integer, intent(out) :: ierr

         integer :: ns, ns1, i, j
         integer, pointer :: ipivot(:), iwork(:)
         real(fltp), pointer, dimension(:,:) :: dmat, umat, lmat, bptr, x1, b_copy
         real(fltp), pointer :: b1_copy(:), work(:), ferr(:), berr(:)
         real(fltp) :: rcond
         character :: fact, trans, equed
         
         ierr = 0

         if (size(lblk,1) /= mblk .or. size(lblk,2) /= mblk .or. size(lblk,3) < nblk) then
            write(*,*) 'thomas_solve lblk wrong dims'
            ierr = -1
            return
         end if
         if (size(dblk,1) /= mblk .or. size(dblk,2) /= mblk .or. size(dblk,3) < nblk) then
            write(*,*) 'thomas_solve dblk wrong dims'
            ierr = -1
            return
         end if
         if (size(ublk,1) /= mblk .or. size(ublk,2) /= mblk .or. size(ublk,3) < nblk) then
            write(*,*) 'thomas_solve ublk wrong dims'
            ierr = -1
            return
         end if
         if (size(ipiv,1) /= mblk .or. size(ipiv,2) < nblk) then
            write(*,*) 'thomas_solve ipiv wrong dims'
            ierr = -1
            return
         end if
         
         do ns = nblk, 1, -1
            bptr => brhs(:,:,ns)
            dmat => dblk(:,:,ns)
            ipivot => ipiv(:,ns)
            call my_getrs(mblk, nrhs, dmat, mblk, ipivot, bptr, mblk, ierr)
            if (ierr /= 0) exit
            if (ns == 1) exit
            ns1 = ns-1
            umat => ublk(:,:,ns1)
            bptr => brhs(:,:,ns1)
            x1 => brhs(:,:,ns)
            ! bptr = bptr - matmul(umat,x1)
            if (nrhs == 1) then
               call my_gemv(mblk,mblk,umat,mblk,x1(:,1),bptr(:,1))
            else
               call my_gemm(mblk,nrhs,mblk,umat,mblk,x1,mblk,bptr,mblk)
            end if
         end do

         if (ierr /= 0) return

         do ns = 2, nblk
            ns1 = ns-1
            bptr => brhs(:,:,ns)
            x1 => brhs(:,:,ns1)
            lmat => lblk(:,:,ns)
            ! bptr = bptr - matmul(lmat,x1)
            if (nrhs == 1) then
               call my_gemv(mblk,mblk,lmat,mblk,x1(:,1),bptr(:,1))
            else
               call my_gemm(mblk,nrhs,mblk,lmat,mblk,x1,mblk,bptr,mblk)
            end if
         end do

      end subroutine thomas_solve

      
      subroutine thomas_solve1(id, mblk, nblk, lblk, dblk, ublk, brhs, ipiv, ierr) ! nrhs = 1
         integer, intent(in) :: id, nblk, mblk
         real(fltp), pointer, intent(in) :: lblk(:,:,:), dblk(:,:,:), ublk(:,:,:) ! (mblk,mblk,nblk)
         real(fltp), pointer, intent(inout) :: brhs(:,:) ! (mblk,nblk)
         integer, pointer, intent(in) :: ipiv(:,:) ! (mblk,nblk)
         integer, intent(out) :: ierr

         integer :: ns, ns1
         integer, pointer :: ipivot(:)
         real(fltp), pointer :: dmat(:,:), umat(:,:), lmat(:,:), bptr(:), b2ptr(:,:), x1(:)
         
         ierr = 0
         if (size(lblk,1) /= mblk .or. size(lblk,2) /= mblk .or. size(lblk,3) < nblk) then
            write(*,*) 'thomas_solve1 lblk wrong dims'
            ierr = -1
            return
         end if
         if (size(dblk,1) /= mblk .or. size(dblk,2) /= mblk .or. size(dblk,3) < nblk) then
            write(*,*) 'thomas_solve1 dblk wrong dims'
            ierr = -1
            return
         end if
         if (size(ublk,1) /= mblk .or. size(ublk,2) /= mblk .or. size(ublk,3) < nblk) then
            write(*,*) 'thomas_solve1 ublk wrong dims'
            ierr = -1
            return
         end if
         if (size(ipiv,1) /= mblk .or. size(ipiv,2) < nblk) then
            write(*,*) 'thomas_solve1 ipiv wrong dims'
            ierr = -1
            return
         end if
         
         do ns = nblk, 1, -1
            bptr => brhs(:,ns)
            dmat => dblk(:,:,ns)
            ipivot => ipiv(:,ns) 
            call set_ptr_2(b2ptr, bptr, mblk, 1)
            call my_getrs(mblk, 1, dmat, mblk, ipivot, b2ptr, mblk, ierr)
            if (ierr /= 0) return
            if (ns == 1) exit
            ns1 = ns-1
            umat => ublk(:,:,ns1)
            bptr => brhs(:,ns1)
            x1 => brhs(:,ns)
            ! bptr = bptr - matmul(umat,x1)
            call my_gemv(mblk,mblk,umat,mblk,x1,bptr)
         end do

         do ns = 2, nblk
            ns1 = ns-1
            bptr => brhs(:,ns)
            x1 => brhs(:,ns1)
            lmat => lblk(:,:,ns)
            ! bptr = bptr - matmul(lmat,x1)
            call my_gemv(mblk,mblk,lmat,mblk,x1,bptr)
         end do

      end subroutine thomas_solve1

      
      subroutine thomas_stats( &
            id, nblk, factor, refactor, rejected_refactor, solve, sum_nz, max_nz, min_rcond, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: nblk ! number of blocks 
         integer, intent(out) :: factor ! number of times did block factor 
         integer, intent(out) :: refactor ! number of times did block refactor 
         integer, intent(out) :: rejected_refactor ! number of times rejected block refactor
         integer, intent(out) :: solve ! number of times did block solve 
         integer, intent(out) :: sum_nz ! sum of non-zeros in factored blocks 
         integer, intent(out) :: max_nz ! max number of non-zeros in a factored block 
         real(fltp), intent(out) :: min_rcond
         integer, intent(out) :: ierr
         integer :: ns
         ierr = 0
         nblk = 0
         factor = 0
         refactor = 0
         rejected_refactor = 0
         solve = 0
         min_rcond = 0
         sum_nz = 0
         max_nz = 0
      end subroutine thomas_stats
      
      
      integer function thomas_handle(ierr)
         integer, intent(out) :: ierr
         ierr = 0
         thomas_handle = -1
      end function thomas_handle
      
      
      subroutine thomas_dealloc(id,ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine thomas_dealloc


#ifdef DBLE
      end module thomas_block_dble
#else
      end module thomas_block_quad
#endif
