      module test_mlt_support
      use mlt_lib
      use mlt_def
      use alert_lib
      use const_def

      implicit none
      
      logical, parameter :: do_standard_test = .false.

      ! inputs
      real(dp) :: cgrav
      real(dp) :: m       ! enclosed mass
      real(dp) :: r       ! radius
      real(dp) :: T       ! temperature
      real(dp) :: rho     ! density
      real(dp) :: L       ! luminosity
      real(dp) :: grad_star ! the actual dlnT/dlnP
      real(dp) :: P       ! pressure
      real(dp) :: chiRho  ! dlnP_dlnd at constant T
      real(dp) :: chiT    ! dlnP_dlnT at constant Rho
      real(dp) :: Cp      ! specific heat capacity at constant pressure
      real(dp) :: opacity
      real(dp) :: grada ! dlnT/dlnP at constant entropy (adiabatic gradient)
      real(dp) :: alpha_semiconvection
      real(dp) :: gradmu ! actual dlnmu/dlnP
      real(dp) :: dlnmu_dlnT  ! dlnmu/dlnT in surroundings
      real(dp) :: dlnmu_dlnrho  ! dlnmu/dlnRho in surroundings
      real(dp) :: grad_rho ! dlnRho/dlnP in surroundings
      real(dp) :: grad_temp ! dlnT/dlnP in surroundings
      real(dp) :: gradr_factor
      real(dp) :: Cv
      real(dp) :: X
      real(dp) :: mixing_length_alpha, thermo_haline_coeff
      logical :: use_Ledoux_criterion, alt_scale_height, use_Henyey_MLT
      real(dp) :: gradL_composition_term
      real(dp) :: Henyey_y_param, Henyey_nu_param, tau, max_convective_velocity
      
      character (len=64) :: thermohaline_option
      
      
      ! results
      integer :: mixing_type
      real(dp) :: mlt_basics(num_mlt_results)
      real(dp) :: mlt_partials(num_mlt_partials,num_mlt_results)
      real(dp) :: gradT, gradr, scale_height, convection_velocity, &
         d_gradT_dlnd, d_gradT_dlnT, d_gradT_dlnR, d_gradT_dL, d_gradT_dlnm, &
         log10_convection_velocity, log10_scale_height,  &
         L_conv, L_rad, L_total


      contains
      
      
      subroutine Do_MLT_Test
         integer :: ierr
         
 1       format(a40,e18.6)
         
         use_Ledoux_criterion = .false.
         gradL_composition_term = 0
         alt_scale_height = .true.
         use_Henyey_MLT = .true.
         thermo_haline_coeff = 0
         Henyey_y_param = one_third
         Henyey_nu_param = 8
         tau = 1
         max_convective_velocity = 1d99
         
         thermohaline_option = 'Traxler'
         X = 0.7d0
         
         gradr_factor = 1
         alpha_semiconvection = 0
         gradmu = 0
         dlnmu_dlnT = 0
         dlnmu_dlnrho = 0
         grad_rho = 0
         grad_temp = 0
         
         ! test data from near top of central convective region of 4.0M ZAMS, Z=0.02
                             m =     6.0708798764978617D+32
                             r =     1.8104022869294819D+10
                             T =     1.6390574971606171D+07
                           rho =     2.1498086433161234D+01
                             L =     1.4437988700909283D+35
                             P =     6.3673785913580768D+16
                        chiRho =     1.0016146515292350D+00
                          chiT =     1.0039723441618589D+00
                            Cp =     4.5860132601655614D+08
                       opacity =     7.1145152433753656D-01
                         grada =     3.9495940002582286D-01
           mixing_length_alpha =     2.0000000000000000D+00
 
         ! expected results
                         gradT =     3.9495941506490573D-01
                         gradr =     5.8865301921953783D-01
           convection_velocity =     4.7462650334641930D+03
                  scale_height =     2.3964312124944618D+10
                  d_gradT_dlnd =     2.9566266663535595D-03
                  d_gradT_dlnT =    -9.8632412487315978D-03
                  d_gradT_dlnR =     1.1076053742688285D-18
                    d_gradT_dL =     2.1104128893990745D-43
                  d_gradT_dlnm =    -3.0470117470354128D-08
                  
         log10_convection_velocity = log10(convection_velocity)
         log10_scale_height = log10(scale_height)
         max_convective_velocity = 1d99
         
         Cv = Cp ! need Cv
         
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term,  &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            write(*,'(a)') trim(alert_message)
            stop 1
         end if
      
         call show_mlt_input
         
         gradr = mlt_Paczynski_gradr(P,opacity,L,m,cgrav,crad*T**4/3,tau,T,r,rho)
         write(*,*) 'mlt_Paczynski_gradr', gradr
         write(*,*)
         
         call check_gradT
         call check_convection_velocity
         call check_gradr
         write(*,*) 'MLT Results'
         call show_mlt_results
         write(*,*)

      end subroutine Do_MLT_Test
      
         
         subroutine show_mlt_input
            write(*,*)
            write(*,*) 'MLT Input'
            write(*,*)
            write(*,'(99a24)') 'm (Msun)', 'r (Rsun)', 'logT', 'logRho',  &
               'logP', 'log Cp', 'log kap', 'log L', 'grada'
            write(*,'(99(1pe24.8))') m / Msun, r / Rsun, log10(T),  &
               log10(Rho), log10(P), log10(Cp), log10(opacity),  &
               log10(L), grada
            write(*,*)
         end subroutine show_mlt_input
         
         subroutine show_mlt_results
            write(*,*)
            write(*,'(99a24)')  &
                  'mixing_type', 'gradT', 'log velocity', 'gradr', 'log scale height', 'D'
            write(*,'(i24,99(1pe24.8))')  &
               mixing_type, mlt_basics(mlt_gradT), &
               log10(mlt_basics(mlt_convection_velocity)), &
               mlt_basics(mlt_gradr), &
               log10(mlt_basics(mlt_scale_height)),  &
               mlt_basics(mlt_D)
         end subroutine show_mlt_results
         
         subroutine check_gradT
 1          format(a40,e18.6)
            if (abs(mlt_basics(mlt_gradT) - gradT) > 1d-2 * gradT) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected gradT', gradT
               write(*,1) ' but got gradT', mlt_basics(mlt_gradT)
               write(*,1) '    relative error',  &
                  abs(mlt_basics(mlt_gradT) - gradT) / gradT
            end if
         end subroutine check_gradT

         subroutine check_convection_velocity
 1          format(a40,e18.6)
            if (abs(log10(mlt_basics(mlt_convection_velocity)) - log10_convection_velocity)  &
                  > 1d-2 * log10_convection_velocity) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected log10_convection_velocity',  &
                     log10_convection_velocity
               write(*,1) ' but got log10_convection_velocity',  &
                     log10(mlt_basics(mlt_convection_velocity))
               write(*,1) '    relative error',  &
                     abs(log10(mlt_basics(mlt_convection_velocity)) - log10_convection_velocity)  &
                     / log10_convection_velocity
            end if
         end subroutine check_convection_velocity

         subroutine check_gradr
 1          format(a40,e18.6)
            if (abs(mlt_basics(mlt_gradr) - gradr) > 1d-2 * gradr) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected gradr', gradr
               write(*,1) ' but got gradr', mlt_basics(mlt_gradr)
               write(*,1) '    relative error',  &
                  abs(mlt_basics(mlt_gradr) - gradr) / gradr
            end if
         end subroutine check_gradr
         
         
      
      
      
      subroutine do_semiconvection_test
         integer :: ierr
         real(dp) :: gradr, gradT, conv_vel, mixing_length, tau
         
         include 'formats.dek'
         
         gradr_factor = 1
         use_Ledoux_criterion = .false.
         gradL_composition_term = 0
         alt_scale_height = .false.
         use_Henyey_MLT = .false.
         thermo_haline_coeff = 0
         Henyey_y_param = one_third
         Henyey_nu_param = 8
         
                                     m =    1.0116700550284785D+33
                                     r =    6.5892058129907646D+09
                                     T =    1.0781315546869174D+07
                                   rho =    4.4761282612718418D+01
                                     L =    2.1073670369450749D+34
                                     P =    2.9743597373734936D+16
 
                                chiRho =    1.0021609189370979D+00
                                  chiT =    1.0074132838429284D+00
                                    Cp =    1.5677760528753823D+08
                                    Cv =    9.4368731396228209D+07
 
                               opacity =    2.1886934997593572D+00
                                 grada =    3.9514282118006483D-01
 
                  alpha_semiconvection =    1.0000000000000000D-04
                   thermo_haline_coeff =    0.0000000000000000D+00
                                gradmu =    0.0000000000000000D+00

         dlnmu_dlnT = 0
         dlnmu_dlnrho = 0
         grad_rho = 0
         grad_temp = 0
         X = 0.7
         thermohaline_option = 'Traxler'
         tau = 1
         max_convective_velocity = 1d99
 
                   mixing_length_alpha =    2.0000000000000000D+00
         
         write(*,*) 'call mlt_eval'
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term,  &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            write(*,'(a)') trim(alert_message)
            stop 1
         end if
         
         gradr = mlt_basics(mlt_gradr)
         gradT = mlt_basics(mlt_gradT)
         conv_vel = mlt_basics(mlt_convection_velocity)
         mixing_length = mlt_basics(mlt_Lambda)
         write(*,*)
         write(*,1) 'gradr-grada', gradr-grada
         write(*,1) 'gradr', gradr
         write(*,1) 'grada', grada
         write(*,1) 'gradT', gradT
         write(*,*)
         write(*,1) 'conv_vel', conv_vel
         write(*,1) 'mixing_length', mixing_length
         write(*,*)


         write(*,*)
         write(*,2) 'mixing_type', mixing_type
         write(*,2) 'semiconvective_mixing', semiconvective_mixing
         write(*,*)
         stop 'do_semiconvection_test'
         
      end subroutine do_semiconvection_test
      
      
      subroutine do_thermo_haline_test
         integer :: ierr
         real(dp) :: gradr, gradT, conv_vel, mixing_length, tau
         
         include 'formats.dek'
                  
                                     m =    5.3093667591952712D+32
                                     r =    3.5937315878003525D+10
                                     T =    3.5159289248464070D+06
                                   rho =    3.9034257032852071D-02
                                     L =    1.4334378463284934D+35
                                     P =    1.8419976844386375D+13
                                chiRho =    9.7838128052383377D-01
                                  chiT =    1.0649794602918055D+00
                                    Cp =    3.8705686857243329D+08
                                    Cv =    2.3151991478630376D+08
                               opacity =    1.8307670950889652D+00
                                 grada =    3.7731969748478839D-01
 
                                gradmu =   -3.5799013317669598D-08
                            dlnmu_dlnT =   -1.6210452152229940D-04
                          dlnmu_dlnrho =   -6.3049035907311377D-06
                              grad_rho =    6.9102258843244790D-01
                             grad_temp =    3.0435962148269896D-01
 
                          gradr_factor =    1.0000000000000000D+00
                     use_Ledoux_criterion = .true.
                gradL_composition_term =   -2.9325763341978774D-08
                  alpha_semiconvection =    4.0000000000000001D-02
                   thermo_haline_coeff =    2.0000000000000000D+00
                   mixing_length_alpha =    1.6000000000000001D+00
                        alt_scale_height = .false.
 
                          use_Henyey_MLT = .false.
                        Henyey_y_param =    3.3333332999999998D-01
                       Henyey_nu_param =    8.0000000000000000D+00
                                   tau =    1
                  max_convective_velocity = 1d99

         X = 0.7
         thermohaline_option = 'Traxler'
         
         write(*,*) 'call mlt_eval'
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term,  &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            write(*,'(a)') trim(alert_message)
            stop 1
         end if
         
         gradr = mlt_basics(mlt_gradr)
         gradT = mlt_basics(mlt_gradT)
         conv_vel = mlt_basics(mlt_convection_velocity)
         mixing_length = mlt_basics(mlt_Lambda)
         write(*,*)
         write(*,1) 'gradr-grada', gradr-grada
         write(*,1) 'gradr', gradr
         write(*,1) 'gradT', gradT
         write(*,1) 'conv_vel', conv_vel
         write(*,1) 'D', mlt_basics(mlt_D)
         write(*,1) 'mixing_length', mixing_length
         write(*,2) 'mixing_type', mixing_type
         write(*,*)


         write(*,*)
         if (mixing_type == thermo_haline_mixing) then
            write(*,*) 'mixing_type == thermo_haline_mixing'
         else
            write(*,*) 'mixing_type /= thermo_haline_mixing'
         end if
         write(*,*)
         stop 'do_thermo_haline_test'
         
      end subroutine do_thermo_haline_test
      
      end module


      program test_mlt
      use test_mlt_support
      use const_lib
      
      call const_init
      cgrav = standard_cgrav
      
      !call do_semiconvection_test
      
      !call do_thermo_haline_test
      
      !call do_special_test
      
      call Do_MLT_Test
      
      end program




