! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mlt_lib

      use mlt_def
      use const_def, only: dp
      
      implicit none

      contains 

      ! temperature gradient calculation based on standard mixing length theory
      ! as, for example, described in Cox & Giuli, Chapter 14.  (see references below to C&G)
      
      subroutine mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term,  &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)

         use mlt,only:do_mlt_eval

         real(dp), intent(in) :: cgrav   ! gravitational constant
         real(dp), intent(in) :: m       ! enclosed mass
         real(dp), intent(in) :: r       ! radius
         real(dp), intent(in) :: T       ! temperature
         real(dp), intent(in) :: rho     ! density
         real(dp), intent(in) :: L       ! luminosity
         real(dp), intent(in) :: P       ! pressure
         real(dp), intent(in) :: chiRho  ! dlnP_dlnd at constant T and mu
         real(dp), intent(in) :: chiT    ! dlnP_dlnT at constant Rho and mu
         real(dp), intent(in) :: Cp      ! specific heat capacity at constant pressure
         real(dp), intent(in) :: Cv      ! specific heat capacity at constant volume
         real(dp), intent(in) :: X ! hydrogen mass fraction
         real(dp), intent(in) :: opacity
         real(dp), intent(in) :: grada   ! dlnT/dlnP at constant entropy (adiabatic gradient)
         real(dp), intent(in) :: gradmu  ! dlnmu/dlnP in surroundings
         real(dp), intent(in) :: dlnmu_dlnT  ! dlnmu/dlnT in surroundings
         real(dp), intent(in) :: dlnmu_dlnrho  ! dlnmu/dlnRho in surroundings
         real(dp), intent(in) :: grad_rho ! dlnRho/dlnP in surroundings
         real(dp), intent(in) :: grad_temp ! dlnT/dlnP in surroundings
         real(dp), intent(in) :: gradr_factor ! multiply standard gradr by this
            ! usually 1, but not when have rotation.
         
         logical, intent(in) :: use_Ledoux_criterion
            ! if false, use Schwarzschild criterion for convection
            ! if true, use Ledoux criterion for convection
         real(dp), intent(in) :: gradL_composition_term
            ! gradL = grada + gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection
            ! determines efficiency of semiconvective mixing.
            ! values of 1d-1 to 1d-4 are used.
         real(dp), intent(in) :: thermo_haline_coeff 
            ! 0 means no thermo-haline mixing.
            ! > 0 determines efficiency of therm-haline mixing.
            ! values of up to 1000 are used.
         real(dp), intent(in) :: mixing_length_alpha 
         logical, intent(in) :: alt_scale_height
            ! if false, then use the usual definition -- P/(g*rho)
            ! if true, use min of the usual and sound speed * hydro time scale, sqrt(P/G)/rho
         character (len=*), intent(in) :: thermohaline_option 
            ! determines which method to use for calculating thermohaline diffusion coef
            ! if == 'Traxler', use method of Traxler, Garaud, & Stellmach, ApJ Letters, 728:L29 (2011).
            ! if == 'Kippenhahn', use method of Kippenhahn, R., Ruschenplatt, G., & Thomas, H.-C. 1980, A&A, 91, 175.
         logical, intent(in) :: use_Henyey_MLT
            ! if false, then use an algorithm based on Cox & Giuli.
            ! if true, then use an algorithm based on Henyey, Vardya, & Bodenheimer 1965, ApJ, 142, 841.
            ! This treatment allows the convective efficiency (Gamma) to vary with the opaqueness
            ! of the convective element, which is measured by the dimensionless variable omega.
            ! Note the reference omits Rho from the definition of omega--this is a misprint.
            ! omega << 1 means transparent; omega >> 1 means opaque
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param 
         real(dp), intent(in) :: max_convective_velocity
         real(dp), intent(in) :: tau
            ! if tau < 2/3, then, following B. Paczynski, 1969, Acta Astr., vol. 19, 1.  (eqn 14),
            ! the radiative temperature gradient, gradr, includes a factor to approximate the effect
            ! of the dilution of the radiation in the atmosphere.
            ! you should also use Henyey MLT in order to account for optically thin turbulent elements.
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or thermohaline
            ! values defined in const_def
         real(dp), intent(out) :: mlt_basics(num_mlt_results)
         real(dp), intent(out) :: mlt_partials(num_mlt_partials, num_mlt_results)
            ! e.g., mlt_partials(mlt_dlnT,mlt_gradT) has partial wrt lnT of gradT
         integer, intent(out) :: ierr

         call do_mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term,  &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)
         
      end subroutine mlt_eval
      
      
      real(dp) function mlt_Paczynski_gradr(P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
         use mlt,only:get1_Paczynski_gradr
         real(dp), intent(in) :: P,opacity,L,m,cgrav,Pr,tau,T,r,rho
         mlt_Paczynski_gradr = get1_Paczynski_gradr(P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
      end function mlt_Paczynski_gradr
      
      
      end module mlt_lib

