! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mlt
      use mlt_def
      use alert_lib
      use const_def
      
      implicit none

      ! largely based on standard mixing length theory
      ! as, for example, described in Cox & Giuli, Chapter 14.  (C&G)
      
      integer, parameter :: nvbs = num_mlt_partials

      contains      


      subroutine do_mlt_eval(  &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, X, opacity, grada,  &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
            gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
            alpha_semiconvection, thermo_haline_coeff, &
            mixing_length_alpha, alt_scale_height, thermohaline_option, &
            use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
            max_convective_velocity, tau, &
            mixing_type, mlt_basics, mlt_partials, ierr)

         real(dp), intent(in) :: cgrav, m, r, T, Rho, L, P
         real(dp), intent(in) :: chiRho, chiT, Cp, Cv, X, opacity, grada, &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, gradr_factor
         logical, intent(in) :: use_Ledoux_criterion
         real(dp), intent(in) :: gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection, thermo_haline_coeff
         real(dp), intent(in) :: mixing_length_alpha
         logical, intent(in) :: alt_scale_height, use_Henyey_MLT
         character (len=*), intent(in) :: thermohaline_option 
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param, max_convective_velocity, tau
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or salt finger
         real(dp), intent(out) :: mlt_basics(num_mlt_results)
         real(dp), intent(out) :: mlt_partials(num_mlt_partials,num_mlt_results)
            ! e.g., mlt_partials(mlt_dlnT,mlt_gradT) has partial wrt lnT of gradT
         integer, intent(out) :: ierr
         
         ierr = 0
         mlt_basics = 0
         mlt_partials = 0
         call Get_results( &
                  cgrav, m, r, T, Rho, L, P, &
                  chiRho, chiT, Cp, Cv, X, opacity, grada, &
                  gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
                  gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
                  alpha_semiconvection, thermo_haline_coeff, &
                  mixing_length_alpha, alt_scale_height, thermohaline_option, &
                  use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
                  max_convective_velocity, tau, mixing_type, &
                  mlt_basics(mlt_gradT), mlt_partials(:,mlt_gradT), &
                  mlt_basics(mlt_gradr), mlt_partials(:,mlt_gradr), &
                  mlt_basics(mlt_gradL), mlt_partials(:,mlt_gradL), &
                  mlt_basics(mlt_scale_height), mlt_partials(:,mlt_scale_height), &
                  mlt_basics(mlt_Lambda), mlt_partials(:,mlt_Lambda), &
                  mlt_basics(mlt_convection_velocity), mlt_partials(:,mlt_convection_velocity), &
                  mlt_basics(mlt_D), mlt_partials(:,mlt_D), &
                  mlt_basics(mlt_gamma), mlt_partials(:,mlt_gamma), &
                  mlt_basics(mlt_conv_dP_term), mlt_partials(:,mlt_conv_dP_term), &
                  ierr)
         if (mlt_basics(mlt_D) < 0) then
            stop 'do_mlt_eval: mlt_basics(mlt_D) < 0'
         end if
            
      end subroutine do_mlt_eval


      subroutine Get_results( &
                  cgrav, m, r, T, Rho, L, P, &
                  chiRho, chiT, Cp, Cv, xh, opacity, grada,  &
                  gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, &
                  gradr_factor, use_Ledoux_criterion, gradL_composition_term, &
                  alpha_semiconvection, thermo_haline_coeff, &
                  mixing_length_alpha, alt_scale_height, thermohaline_option, &
                  use_Henyey_MLT, Henyey_y_param, Henyey_nu_param, &
                  max_convective_velocity, tau, mixing_type, &
                  gradT, d_gradT_dvb, &
                  gradr, d_gradr_dvb, &
                  gradL, d_gradL_dvb, &
                  scale_height, d_scale_height_dvb, &
                  Lambda, d_Lambda_dvb, &
                  conv_vel, d_conv_vel_dvb, & ! convection velocity
                  D, d_D_dvb, &
                  Gamma, d_Gamma_dvb, &
                  conv_P, d_conv_P_dvb, &
                  ierr)

         use utils_lib,only:is_bad_num

         real(dp), intent(in) :: cgrav, m, r, T, Rho, L, P
         real(dp), intent(in) :: chiRho, chiT, Cp, Cv, xh, opacity, grada, &
            gradmu, dlnmu_dlnT, dlnmu_dlnrho, grad_rho, grad_temp, gradr_factor
         logical, intent(in) :: use_Ledoux_criterion
         real(dp), intent(in) :: gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection, thermo_haline_coeff
         real(dp), intent(in) :: mixing_length_alpha
         logical, intent(in) :: alt_scale_height, use_Henyey_MLT
         character (len=*), intent(in) :: thermohaline_option 
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param, max_convective_velocity, tau
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or salt finger
         real(dp), intent(out) :: gradT, d_gradT_dvb(nvbs)
         real(dp), intent(out) :: gradr, d_gradr_dvb(nvbs)
         real(dp), intent(out) :: gradL, d_gradL_dvb(nvbs)
         real(dp), intent(out) :: scale_height, d_scale_height_dvb(nvbs)
         real(dp), intent(out) :: Lambda, d_Lambda_dvb(nvbs)
         real(dp), intent(out) :: conv_vel, d_conv_vel_dvb(nvbs)
         real(dp), intent(out) :: D, d_D_dvb(nvbs)
         real(dp), intent(out) :: Gamma, d_Gamma_dvb(nvbs) ! convective efficiency
         real(dp), intent(out) :: conv_P, d_conv_P_dvb(nvbs)
         
         integer, intent(out) :: ierr

         ! locals
         real(dp) :: scale_height1, scale_height2
         real(dp) :: Pg, Pr, dP_dvb(nvbs), dPg_dvb(nvbs), dPr_dvb(nvbs), dRho_dvb(nvbs)
         real(dp) :: dT_dvb(nvbs), alpha, phi, dgrad, denom, tmp
         
         real(dp) :: grav, d_grav_dvb(nvbs)
         real(dp) :: diff_grads, d_diff_grads_dvb(nvbs)
         real(dp) :: convective_conductivity, d_cc_dvb(nvbs)
         real(dp) :: radiative_conductivity, d_rc_dvb(nvbs)
         real(dp) :: surf, dsurf_dvb(nvbs)
         real(dp) :: beta, d_beta_dvb(nvbs)
         real(dp) :: chi, d_chi_dvb(nvbs)
         real(dp) :: D_div_B, d_D_div_B_dvb(nvbs)
         real(dp) :: Q, dQ_dvb(nvbs)
         real(dp) :: A, dA_dvb(nvbs)
         real(dp) :: Bcubed, d_Bcubed_dvb(nvbs)			
         real(dp) :: Zeta, d_Zeta_dvb(nvbs)
         real(dp) :: d_Cp_dvb(nvbs)
         real(dp) :: dR_dvb(nvbs)
         real(dp) :: d_opacity_dvb(nvbs)
         real(dp) :: d_grada_dvb(nvbs)
         real(dp) :: Dconv, d_Dconv_dvb(nvbs)
         real(dp) :: delta, d_delta_dvb(nvbs)
         real(dp) :: f, f0, d_f0_dvb(nvbs)
         real(dp) :: f1, d_f1_dvb(nvbs)
         real(dp) :: f2, d_f2_dvb(nvbs)
         real(dp) :: x, d_x_dvb(nvbs)

         real(dp) :: d_chiT_dvb(nvbs), d_chiRho_dvb(nvbs)

         !real(dp), parameter :: a0 = (9d0 / 4d0)
         !for Henyey MLT:
         real(dp) :: a0, omega, theta
         real(dp) :: d_omega_dvb(nvbs), d_theta_dvb(nvbs) !, d_a0_dvb(nvbs)
         
         integer :: i
         real(dp), parameter :: tiny = 1d-30, min_D_th = 1d-3
         character (len=256) :: message        
         logical :: Schwarzschild_stable, Ledoux_stable, quit
         real(dp) :: diff_grad, K, gamma0, L_ratio, frac, s, dilution_factor
         real(dp) :: K_T, K_mu, nu_rad, nu_mol, nu, grad_mu, R0, r_th
         
         logical, parameter :: debug = .false.

         include 'formats.dek'
      
         ierr = 0

         if (.not. use_Henyey_MLT) then
            a0 = 9d0/4d0
            !d_a0_dvb = 0d0
         endif
         
         call set_no_mixing

         d_grada_dvb = 0
         d_grada_dvb(mlt_dgrada) = 1
                     
         grav = cgrav*m / r**2
         d_grav_dvb = 0
         d_grav_dvb(mlt_dlnR) = -2*grav
		   d_grav_dvb(mlt_dlnm) = grav

			dP_dvb = 0
			dP_dvb(mlt_dP) = 1
			
			dRho_dvb = 0
			dRho_dvb(mlt_dlnd) = Rho
			
			dT_dvb = 0
			dT_dvb(mlt_dlnT) = T
			
			dR_dvb = 0
			dR_dvb(mlt_dlnR) = r

         d_chiT_dvb = 0
         d_chiT_dvb(mlt_dchiT) = 1

         d_chiRho_dvb = 0
         d_chiRho_dvb(mlt_dchiRho) = 1

         d_Cp_dvb = 0
         d_Cp_dvb(mlt_dCp) = 1

         d_opacity_dvb = 0
         d_opacity_dvb(mlt_dopacity) = 1
			
         surf = pi4*r**2
         if (debug) write(*,1) 'surf', surf
         dsurf_dvb = 8*pi*r*dR_dvb
         
         scale_height = P / (grav*rho)
         d_scale_height_dvb = scale_height*(dP_dvb/P - d_grav_dvb/grav - dRho_dvb/Rho)
         if (alt_scale_height) then
            ! consider sound speed*hydro time scale as an alternative scale height
            ! (this comes from Eggleton's code.)
            scale_height2 = sqrt(P / cgrav) / rho
            !if (P > 1d17 .and. m < 1d31) then
            !   write(*,1) 'scale_height2/scale_height', scale_height2/scale_height
            !end if
            if (scale_height2 < scale_height) then
      		   scale_height = scale_height2
      		   d_scale_height_dvb = scale_height*(0.5d0*dP_dvb/P - dRho_dvb/Rho)
            end if
         end if
         if (debug) write(*,1) 'scale_height', scale_height

         ! mixing length, Lambda
         Lambda = mixing_length_alpha*scale_height
         if (debug) write(*,1) 'Lambda', Lambda
			d_Lambda_dvb = mixing_length_alpha*d_scale_height_dvb
                  
         if (mixing_length_alpha <= 0) then
            call set_no_mixing
            return
         end if

         ! 'Q' param  C&G 14.24
         Q = chiT/chiRho
         dQ_dvb = Q*( d_chiT_dvb/chiT - d_chiRho_dvb/chiRho )
         if (Q <= 0) then
            call set_no_mixing
            return
         end if

         Pr = one_third*crad*T**4
         if (debug) write(*,1) 'Pr', Pr
         dPr_dvb = 0
         dPr_dvb(mlt_dlnT) = 4*Pr
         
         gradr = get1_Paczynski_gradr(P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
         gradr = gradr*gradr_factor
      
         d_gradr_dvb(mlt_dP) = gradr/P
         d_gradr_dvb(mlt_dopacity) = gradr/opacity
         d_gradr_dvb(mlt_dL) = gradr_factor*P*opacity/(16*pi*clight*m*cgrav*Pr)
         d_gradr_dvb(mlt_dlnm) = -gradr
         d_gradr_dvb(mlt_dlnT) = -4*gradr
                     
         diff_grads = gradr - grada
         d_diff_grads_dvb = d_gradr_dvb - d_grada_dvb

         Pg = P - Pr
         if (debug) write(*,1) 'Pg', Pg
         if (Pg < tiny .or. r < tiny .or. opacity < tiny .or. rho < tiny .or. Cp < tiny) then
            call set_no_mixing
            return
         end if
         
         dPg_dvb = dP_dvb - dPr_dvb
                     
         radiative_conductivity = (4*crad*clight / 3)*T**3 / (opacity*rho) ! erg / (K cm sec)
         if (debug) write(*,1) 'radiative_conductivity', radiative_conductivity
         d_rc_dvb = 0
         d_rc_dvb(mlt_dlnd) = -radiative_conductivity
         d_rc_dvb(mlt_dlnT) = 3*radiative_conductivity
         d_rc_dvb(mlt_dopacity) = -radiative_conductivity/opacity

      	beta = Pg / P
         if (debug) write(*,1) 'beta', beta
      	d_beta_dvb = beta*(dPg_dvb/Pg - dP_dvb/P)
         
         ! Ledoux temperature gradient
         gradL = grada + gradL_composition_term
         d_gradL_dvb = d_grada_dvb ! ignore partials of composition term
         
         Schwarzschild_stable = (gradr < grada)
         Ledoux_stable = (gradr < gradL)
         if (debug) write(*,1) 'grada', grada
         if (debug) write(*,1) 'gradr', gradr
         if (debug) write(*,1) 'gradL', gradL
         if (debug) write(*,1) 'grada-gradr', grada-gradr
         if (debug) write(*,1) 'gradL-gradr', gradL-gradr
         if (debug) write(*,1) 'gradL_composition_term', gradL_composition_term
         if (debug) write(*,1) 'gradmu', gradmu
         if (debug) write(*,1) 'beta', beta
         if (debug) write(*,*) 'Schwarzschild_stable', Schwarzschild_stable
         if (debug) write(*,*) 'Ledoux_stable', Ledoux_stable
         if (debug) write(*,*) 'use_Ledoux_criterion', use_Ledoux_criterion
         if (debug) write(*,1) 'gradL-grada', gradL-grada
         
         if (Schwarzschild_stable .and. &
                  ((.not. use_Ledoux_criterion) .or. Ledoux_stable)) then ! radiatative or thermohaline
            call set_no_mixing ! also sets gradT = gradr
            if (use_Ledoux_criterion .and. gradL < grada) then ! composition inversion => thermohaline mixing
               if (thermo_haline_coeff > 0) then
                  diff_grad = max(1d-40, grada - gradT) ! positive since Schwarzschild stable               
                  K = 4*crad*clight*T**3/(3*opacity*rho) ! thermal conductivity
                  
                  if (thermohaline_option == 'Traxler') then 
                     ! Traxler, Garaud, & Stellmach, ApJ Letters, 728:L29 (2011).
                     ! also see Denissenkov. ApJ 723:563–579, 2010.
                     K_T = K/(Cp*rho) ! radiative diffusivity
                     K_mu = 1.84d-17*(1 + 7*xh)*T**2.5d0/rho ! molecular diffusivity  
                     nu_rad = 4*crad*T**4/(15*clight*opacity*rho**2) ! radiative viscosity
                     nu_mol = K_mu ! molecular visosity
                     nu = nu_mol + nu_rad ! total viscosity
                     ! estimate grad_mu assuming ideal gas + radiation
                     grad_mu = min(-1d-40,gradL_composition_term*(4 - 3*beta)/beta) 
                     R0 = (gradT - grada)/grad_mu
                     r_th = (R0 - 1)/(K_T/K_mu - 1)
                     D = 101d0*sqrt(K_mu*nu)*exp(-3.6d0*r_th)*(1 - r_th)**1.1d0 ! eqn 24
                     if (D < min_D_th) then
                        call set_no_mixing
                        return
                     end if
                     
                  else if (thermohaline_option == 'Kippenhahn') then 
                     ! Kippenhahn, R., Ruschenplatt, G., & Thomas, H.-C. 1980, A&A, 91, 175
                     D = -thermo_haline_coeff*3*K/(2*rho*cp)*gradL_composition_term/diff_grad
                     if (D < min_D_th) then
                        if (debug) write(*,1) 'D < min_D_th', D, min_D_th
                        if (debug) write(*,1) 'thermo_haline_coeff', thermo_haline_coeff
                        if (debug) write(*,1) 'gradL_composition_term', gradL_composition_term
                        if (debug) write(*,1) 'diff_grad', diff_grad
                        if (debug) write(*,1) 'grada', grada
                        if (debug) write(*,1) 'gradT', gradT
                        if (debug) write(*,1) 'T', T
                        if (debug) write(*,1) 'rho', rho
                        if (debug) write(*,1) 'cp', cp
                        if (debug) write(*,1) 'opacity', opacity
                        call set_no_mixing
                        return
                     end if
                     if (debug) then
                        if (debug) write(*,1) 'rho', rho
                        if (debug) write(*,1) 'T', T
                        if (debug) write(*,1) 'cp', cp
                        if (debug) write(*,1) 'opacity', opacity
                        if (debug) write(*,1) 'grada', grada
                        if (debug) write(*,1) 'gradT', gradT
                        if (debug) write(*,1) 'grada-gradT', grada-gradT
                        if (debug) write(*,1) 'crad', crad
                        if (debug) write(*,1) 'thermo_haline_coeff', thermo_haline_coeff
                        if (debug) write(*,1) '1st part', D/gradL_composition_term
                        if (debug) write(*,1) '2nd part', gradL_composition_term
                     end if
                  else
                     D = 0
                     ierr = -1
                     write(*,*) 'unknown value for MLT thermohaline_option' // trim(thermohaline_option)
                     return                     
                  end if
                  d_D_dvb = 0
                  conv_vel = 3*D/Lambda 
                  d_conv_vel_dvb = 0
                  call set_conv_P
                  mixing_type = thermo_haline_mixing 
                  if (debug) then
                     write(*,*) 'thermo_haline_mixing'
                     write(*,*)
                     write(*,1) 'thermo_haline_coeff', thermo_haline_coeff
                     write(*,1) 'D', D
                     write(*,1) 'conv_vel', conv_vel
                     write(*,*)
                  end if
               end if
            end if       
            call set_conv_P     
            return
         end if
         
         ! from here on, Schwarzschild and/or Ledoux unstable
         ! convection or semiconvection
         
         if (use_Ledoux_criterion .and. Ledoux_stable) then ! semiconvection
            call semiconvection
            call set_conv_P
            return
         end if
         
         ! from here on, doing convection

         ! need to set gradT, d_gradT_dvb, conv_vel, d_conv_vel_dvb
         call standard_scheme
         call set_conv_P
         if (quit) return
         
         D = conv_vel*Lambda/3     ! diffusion coefficient [cm^2/sec]
         if (debug) write(*,1) 'D', D
         d_D_dvb = (d_conv_vel_dvb*Lambda + conv_vel*d_Lambda_dvb)/3
      
         mixing_type = convective_mixing
         
         if (debug .or. D < 0) then
            write(*,*) 'get_gradT: convective_mixing'
            write(*,1) 'D', D
            write(*,1) 'conv_vel', conv_vel
            write(*,*)
				stop 'get_gradT'
         end if
                 
         
         contains
         
         
         subroutine set_conv_P
            conv_P = rho*conv_vel**2/(3*P) ! see C&G (14.69) for Pturb/P
            d_conv_P_dvb = rho*2*d_conv_vel_dvb*conv_vel/(3*P) + &
               dRho_dvb*conv_vel**2/(3*P) - conv_P*dP_dvb/P
         end subroutine set_conv_P
         
         
         subroutine standard_scheme
            ! need to set gradT, d_gradT_dvb, conv_vel, d_conv_vel_dvb
            include 'formats.dek'
            
            quit = .false.

            x = Q*Rho / (2*P)
            d_x_dvb = x*(drho_dvb/rho + dQ_dvb/Q - dP_dvb/P)
         
            convective_conductivity = Cp*grav*Lambda**2*Rho*(sqrt(x)) / 9 ! erg / (K cm sec)
            if (debug) write(*,1) 'convective_conductivity', convective_conductivity
            d_cc_dvb = convective_conductivity* &
                 (d_Cp_dvb/Cp + d_grav_dvb/grav + 2*d_Lambda_dvb/Lambda + dRho_dvb/rho + d_x_dvb/(2*x))
         
            if (use_Henyey_MLT) then ! based on Henyey, Vardya, & Bodenheimer 1965, ApJ, 142, 841
               ! This treatment allows the convective efficiency (Gamma) to vary with the opaqueness
               ! of the convective element, which is measured by the dimensionless variable omega.
               ! Note the reference omits Rho from the definition of omega--this is a misprint.
               ! omega << 1 means transparent; omega >> 1 means opaque
               omega = Lambda*Rho*opacity !dimensionless
               d_omega_dvb = omega*( d_Lambda_dvb/Lambda + dRho_dvb/Rho + d_opacity_dvb/opacity)
               theta = omega / ( 1d0 + Henyey_y_param*omega**2 )
               d_theta_dvb = d_omega_dvb*(1d0 - Henyey_y_param*omega**2 ) / ( ( 1d0 + Henyey_y_param*omega**2 )**2 )

               a0 = 0.75d0*omega*theta
               !d_a0_dvb = a0*( d_omega_dvb/omega + d_theta_dvb/theta )

               A = sqrt(P*Q*rho/Henyey_nu_param)*(Cp*mixing_length_alpha)/(2*crad*clight*T**3*theta)
               dA_dvb = A*( dP_dvb/(2*P) + dQ_dvb/(2*Q) + drho_dvb/(2*rho) + d_Cp_dvb/Cp - 3*dT_dvb/T - d_theta_dvb/theta )

            else ! this assumes optically thick

            ! 'A' param is ratio of convective to radiative conductivities   C&G 14.98
               A = convective_conductivity / radiative_conductivity !  unitless.

               if (debug) write(*,1) 'A', A
               dA_dvb = (d_cc_dvb - d_rc_dvb*A) / radiative_conductivity
            
            end if

            ! 'B' param  C&G 14.81
            Bcubed = (A**2 / a0)*diff_grads         
            d_Bcubed_dvb = (A**2 / a0)*d_diff_grads_dvb + (2*A*dA_dvb / a0)*diff_grads
         
            if (debug) write(*,1) 'Bcubed', Bcubed

            ! now solve cubic equation for convective efficiency, Gamma
            ! a0*Gamma^3 + Gamma^2 + Gamma - a0*Bcubed == 0   C&G 14.82, 
            ! rewritten in terms of Gamma
            ! leave it to Mathematica to find an expression for the root we want (with a0 = 9/4)
         
            delta = a0*Bcubed
            d_delta_dvb = a0*d_Bcubed_dvb !+ Bcubed*d_a0_dvb
         
            if (debug) write(*,1) 'a0', a0
            if (debug) write(*,1) 'delta', delta
      
            f = -2 + 9*a0 + 27*a0**2*delta !ignoring derivative wrt a0
            if (f > 1d100) then
               f0 = f
               d_f0_dvb = 27*a0**2*d_delta_dvb
            else
               f0 = sqrt(f**2 + 4*(-1 + 3*a0)**3)         
               d_f0_dvb = 27*a0**2*f*d_delta_dvb / f0
            end if
         
            if (debug) write(*,1) 'f0', f0

            f1 = -2 + 9*a0 + 27*a0**2*delta + f0  
            if (f1 < 0 .or. is_bad_num(f1)) then
               call set_no_mixing
               quit = .true.
               return
            end if   
            f1 = f1**one_third        
   			d_f1_dvb = (27*a0**2*d_delta_dvb + d_f0_dvb) / (3*f1**2)

            f2 = 2*2**one_third*(1 - 3*a0) / f1       
   			d_f2_dvb = -f2*d_f1_dvb / f1

            Gamma = (4**one_third*f1 + f2 - 2) / (6*a0)
   			d_Gamma_dvb = (4**one_third*d_f1_dvb + d_f2_dvb) / (6*a0)

				if (is_bad_num(Gamma) .or. Gamma < 0) then
               call set_no_mixing
               quit = .true.
               return
				end if
   			
            ! average convection velocity, vbar   C&G 14.86b
            ! vbar = vsound*Sqrt(Q)*alpha*Gamma / (2*Sqrt(2*Gamma1)*A)
            ! vsound = Sqrt(Gamma1*P / rho), so
            ! vbar = Sqrt(Q*P / (8*rho))*alpha*Gamma / A

            x = Q*P / (8*rho)
            conv_vel = sqrt(x)*mixing_length_alpha*Gamma / A
            if (debug) write(*,1) 'conv_vel', conv_vel
				if (conv_vel < 0) then
					write(*,1) 'conv_vel', conv_vel
					write(*,1) 'mixing_length_alpha', mixing_length_alpha
					write(*,1) 'x', x
					write(*,1) 'A', A
					write(*,1) 'Gamma', Gamma
					stop 'standard_scheme'
				end if
            if (conv_vel > max_convective_velocity) then ! reduce Gamma
               ! note that this will also push gradT toward gradr by reducing Zeta
               if (x < 1d-50 .or. A <= 1d-99) then
                  Gamma = 1d25
                  d_Gamma_dvb = 0
               else
                  Gamma = max_convective_velocity*A/(sqrt(x)*mixing_length_alpha)
                  d_Gamma_dvb = Gamma*dA_dvb/A
               end if
               conv_vel = max_convective_velocity
            end if
            d_conv_vel_dvb = 0.5d0*conv_vel* &
                    (-2*dA_dvb / A + 2*d_Gamma_dvb / Gamma +  &
                    dP_dvb / P + dQ_dvb / Q - drho_dvb / rho)  
                          
            Zeta = Gamma**3 / Bcubed  ! C&G 14.80
   			d_Zeta_dvb = Zeta*(3*d_Gamma_dvb / Gamma - d_Bcubed_dvb / Bcubed)

            ! C&G 14.79
            gradT = (1 - Zeta)*gradr + Zeta*grada 
            if (debug) write(*,1) 'gradT', gradT
            d_gradT_dvb = (1 - Zeta)*d_gradr_dvb + Zeta*d_grada_dvb + &
         					(grada - gradr)*d_Zeta_dvb

            if (is_bad_num(gradT)) then
               call set_no_mixing
               quit = .true.
               return
            end if
         
         end subroutine standard_scheme
            
            
         subroutine semiconvection
            ! need to set gradT, d_gradT_dvb, conv_vel, d_conv_vel_dvb
            include 'formats.dek'
         
            call set_no_mixing ! also sets gradT = gradr
            if (alpha_semiconvection > 0) then ! Langer 1983 & 1985
               do i=1,4 ! estimate L_ratio and gradT by doing a few iterations
                  diff_grad = max(1d-20, gradL - gradT) ! positive since Ledoux stable
                  L_ratio = alpha_semiconvection*(gradT - grada)/(2*gradT*diff_grad)* &
                     ((gradT - grada) - gradmu*beta*(8 - 3*beta)/(32 - 24*beta - beta**2))
                  if (L_ratio <= -1) exit
                  gradT = gradr/(1 + L_ratio)
                  d_gradT_dvb = d_gradr_dvb/(1 + L_ratio) ! ignore partials of L_ratio
               end do
               D = alpha_semiconvection*radiative_conductivity/(6*Cp*rho) &
                  *(gradT - grada)/(gradL - gradT)                             
               d_D_dvb = 0
            else ! use a fraction of convective velocity
               frac = -alpha_semiconvection
               conv_vel = frac*conv_vel
               D = frac*D
               d_D_dvb = frac*d_D_dvb
            end if
            
            if (D <= 0) then
               call set_no_mixing
               return
            end if
            
            conv_vel = 3*D/Lambda 
            d_conv_vel_dvb = 0
            mixing_type = semiconvective_mixing

            if (debug) then
               write(*,*) 'semiconvective_mixing'
               write(*,1) 'D', D
               write(*,1) 'conv_vel', conv_vel
               write(*,*) 
            end if
         
         end subroutine semiconvection
                  
         
         subroutine set_no_mixing
            mixing_type = no_mixing
            gradT = gradr
            d_gradT_dvb = d_gradr_dvb
            conv_vel = 0
            d_conv_vel_dvb = 0
            D = 0
            d_D_dvb = 0
            conv_P = 0
            d_conv_P_dvb = 0
         end subroutine set_no_mixing
         
         
         subroutine show_args
 1          format(a30,1pe26.16)
            
            write(*,1) 'm = ', m
            write(*,1) 'r = ', r 
            write(*,1) 'T = ', T 
            write(*,1) 'Rho = ', Rho 
            write(*,1) 'L  = ', L 
            write(*,1) 'P = ', P 
            write(*,1) 'chiRho = ', chiRho 
            write(*,1) 'chiT = ', chiT
            write(*,1) 'Cp = ', Cp 
            write(*,1) 'opacity = ', opacity 
            write(*,1) 'grada = ', grada
            write(*,1) 'mixing_length_alpha = ', mixing_length_alpha
            
         end subroutine show_args


      end subroutine Get_results
      
      
      real(dp) function get1_Paczynski_gradr( &
            P,opacity,L,m,cgrav,Pr,tau,T,r,rho)
         real(dp), intent(in) :: P,opacity,L,m,cgrav,Pr,tau,T,r,rho
         real(dp) :: dilution_factor, s, f
         get1_Paczynski_gradr = P*opacity*L / (16*pi*clight*m*cgrav*Pr)
         if (tau < 2d0/3d0) then ! B. Paczynski, 1969, Acta Astr., vol. 19, 1., eqn 14.
            s = (2*crad*T**3*sqrt(r))/(3*cgrav*m*rho)*(L/(8*pi*boltz_sigma))**0.25d0  ! eqn 15
            f = 1 - 1.5d0*tau ! Paczynski, 1969, eqn 8
            dilution_factor = (1 + f*s*(4*pi*cgrav*clight*m)/(opacity*L))/(1 + f*s)
            get1_Paczynski_gradr = get1_Paczynski_gradr*dilution_factor
         end if
      end function get1_Paczynski_gradr


      end module mlt
