      module test_kap_support
      
      use kap_lib
      use alert_lib
      use num_lib, only: safe_log10
      
      implicit none

      logical, parameter :: use_shared_data_dir = .true. ! if false, then test using local version data
      !logical, parameter :: use_shared_data_dir = .false.

      integer :: handle
      
   
      integer :: nz
      double precision, pointer, dimension(:) :: &
            x, y, z, c, n, o, ne, lgT, lgd, &
            old_lgKap, old_opacity, old_d_opacity_dlnd, old_d_opacity_dlnT
      double precision, pointer, dimension(:,:) :: &
            new_lgKap, new_opacity, new_d_opacity_dlnd, new_d_opacity_dlnT
      integer, parameter :: num_new = 2
      
      
      logical :: cubic_interpolation_in_X = .true.
      logical :: cubic_interpolation_in_Z = .false.
      logical :: include_electron_conduction = .true.
      
      character(len=256) :: kappa_file_prefix = 'gs98'
      character (len=256) :: kappa_CO_prefix = 'gn93_co'
      !character (len=256) :: kappa_lowT_prefix = 'lowT_Freedman11'
      character (len=256) :: kappa_lowT_prefix = 'lowT_fa05_gs98'
      real*8 :: kappa_blend_logT_upper_bdy = 4.1d0
      real*8 :: kappa_blend_logT_lower_bdy = 3.93d0

      contains
      
      
      subroutine Do_One(quietly)
         logical, intent(in) :: quietly         
         integer :: ierr
         ierr = 0         
         call setup(quietly)         
         call test1(quietly, 1, 'fixed metals', ierr)
         call test1(quietly, 2, 'C/O enhanced', ierr)
         if (ierr /= 0) stop 1         
      end subroutine Do_One
      
      
      subroutine Do_Test1(quietly)
         logical, intent(in) :: quietly 
         integer :: ierr
         ierr = 0         
         
         !cubic_interpolation_in_Z = .false.
         cubic_interpolation_in_Z = .true.
         
         kappa_file_prefix = 'OP'
         call setup(quietly)         
         
         call test1(quietly, 0, '', ierr) ! testing
         stop
         
         call test1(quietly, 1, 'fixed metals', ierr)
         call test1(quietly, 2, 'C/O enhanced', ierr)

         if (ierr /= 0) stop 1         
      end subroutine Do_Test1
   
   
      subroutine test1(quietly, which, test_str, ierr)
         logical, intent(in) :: quietly         
         integer, intent(in) :: which 
         character (len=*), intent(in) :: test_str       
         integer, intent(out) :: ierr
         double precision :: &
            zbar, zbase, xh, dxc, dxo, frac, abar, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            logT, logRho, logR, kap, log10kap, dlnkap_dlnRho, dlnkap_dlnT
            
         logical :: CO_enhanced
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         CO_enhanced = .false.
         dxc = 0d0
         dxo = 0d0
         
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         
         if (which == 1) then ! not enhanced in C/O

            logT =    6
            logRho =   -6
            zbase =    0.01d0
            xh =    0.7d0
             
         else if (which == 2) then ! C/O enhanced
         
            CO_enhanced = .true.
            logT =    6
            logRho =   -6
            zbase =    0.01d0
            xh =    0.7d0
            dxc =   0.021d0
            dxo =   0.019d0
                          
         else ! DEBUGGING
 
            CO_enhanced = .false.
            logT = 6.3d0
            logRho = -0.6d0
            zbase = 0.02d0
            xh =  0.70d0
            dxc =   0.0
            dxo =   0.0
            
            write(*,*) 'test'
             
         end if
         
         call get_composition_info(zbase + dxc + dxo, xh, abar, zbar)
                  
         if (CO_enhanced) then
            call kap_get_Type2( &
                  handle, zbar, xh, zbase, dxc, dxo, logRho, logT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         else
            call kap_get_Type1( &
                  handle, zbar, xh, Zbase, logRho, logT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         end if
      

         log10kap = safe_log10(kap)
         
         if (.not. quietly) then
            write(*,*) 'test number', which
            write(*,*) trim(test_str)
            write(*,*)
            call show_args
            call show_results
         end if
      
         contains
      
         subroutine show_args
            1 format(a40,1pe26.16)
            !write(*,*) 'CO_enhanced', CO_enhanced
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*,1) 'zbase', zbase
            write(*,1) 'xh', xh
            write(*,1) 'dxc', dxc
            write(*,1) 'dxo', dxo
            write(*,*)
         end subroutine show_args
      
         subroutine show_results
            use utils_lib
            1 format(a40,1pe26.16)
            write(*,1) 'log10kap', log10kap
            write(*,1) 'dlnkap_dlnRho', dlnkap_dlnRho
            write(*,1) 'dlnkap_dlnT', dlnkap_dlnT
            write(*,*)
            write(*,1) 'kap', kap
            write(*,1) 'dkap_dlnd', dlnkap_dlnRho*kap
            write(*,1) 'dkap_dlnT', dlnkap_dlnT*kap
            write(*,*)
            if (is_bad_num(log10kap)) then
               write(*,*) trim(alert_message)
               write(*,*) 'bad log10kap'
            end if
         end subroutine show_results

      end subroutine test1
      

      subroutine setup(quietly)
         use chem_lib
         use const_lib
         logical, intent(in) :: quietly
         !..allocate and load the opacity tables

         character (len=256) :: data_dir, kap_dir, opal_dir, cbeg_ferg
         integer :: ierr
         logical, parameter :: use_cache = .true.
         
         if (use_shared_data_dir) then
            data_dir = '../../data' ! test using shared data
         else
            
            write(*,*)
            write(*,*) 'TESTING WITH LOCAL DATA'
            write(*,*)
            
            data_dir = '../data' ! test using local data
         end if
         
         call const_init
      	call chem_init(data_dir, 'isotopes.data_approx', ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'chem_init failed'
      	   stop 1
      	end if

         !kappa_file_prefix = 'gn93_af94'
         !write(*,*) 'HAVE CHANGED kappa_file_prefix = gn93_af94'

         call kap_init( &
            data_dir, kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, use_cache, ierr) 
         if (ierr /= 0) stop 1
         
         handle = alloc_kap_handle(ierr) 
         if (ierr /= 0) stop 1
         
         call kap_set_choices( &
            handle, cubic_interpolation_in_X, &
            cubic_interpolation_in_Z, include_electron_conduction, ierr)
         if (ierr /= 0) stop 1
      
      end subroutine setup


      subroutine finish
         call free_kap_handle(handle)
         call kap_shutdown
      end subroutine finish
      
      
      
      
      
      subroutine get_composition_info(Z, X, abar, zbar)
         double precision, intent(in) :: Z, X
         double precision, intent(out) :: abar, zbar

         double precision :: Y
         
         integer, parameter :: ionmax = 7
         double precision :: aion(ionmax),zion(ionmax),xmass(ionmax),ymass(ionmax),zbarxx,ytot1

         double precision, parameter :: Zfrac_C = 0.173312d0
         double precision, parameter :: Zfrac_N = 0.053177d0
         double precision, parameter :: Zfrac_O = 0.482398d0
         double precision, parameter :: Zfrac_Ne = 0.098675d0
         double precision, parameter :: Zfrac_Mg = 1d0 - (Zfrac_C + Zfrac_N + Zfrac_O + Zfrac_Ne)
         
         integer :: i
         integer :: ih1,ihe4,ic12,in14,io16,ine20,img24

         
         Y = 1 - (X+Z)
         if (Y < 0) then ! adjust XC and XO
            write(*,*) 'bad args to get_composition_info'
            stop 1
         end if
      
         ih1        = 1
         zion(ih1)  = 1.0d0
         aion(ih1)  = 1.0d0 
         xmass(ih1) = X

         ihe4        = 2
         zion(ihe4)  = 2.0d0
         aion(ihe4)  = 4.0d0 
         xmass(ihe4) = Y

         ic12        = 3
         zion(ic12)  = 6.0d0
         aion(ic12)  = 12.0d0 
         xmass(ic12) = Z * Zfrac_C

         in14        = 4
         zion(in14)  = 7.0d0
         aion(in14)  = 14.0d0 
         xmass(in14) = Z * Zfrac_N

         io16        = 5
         zion(io16)  = 8.0d0
         aion(io16)  = 16.0d0 
         xmass(io16) = Z * Zfrac_O

         ine20       = 6
         zion(ine20)  = 10.0d0
         aion(ine20)  = 20.0d0 
         xmass(ine20) = Z * Zfrac_Ne

         img24       = 7
         zion(img24)  = 12.0d0
         aion(img24)  = 24.0d0 
         xmass(img24) = Z * Zfrac_Mg

         zbarxx  = 0.0d0
         ytot1   = 0.0d0
         do i=1,ionmax
            ymass(i) = xmass(i)/aion(i)
            ytot1    = ytot1 + ymass(i)
            zbarxx   = zbarxx + zion(i) * ymass(i)
         enddo
         abar   = 1.0d0/ytot1
         zbar   = zbarxx * abar

      end subroutine get_composition_info
      
      
      subroutine write_plot_data
         use chem_def
         integer, parameter :: io_unit0 = 40
         character (len=256) :: dir
         double precision, pointer, dimension(:,:,:) :: output_values, co_output_values
         double precision :: kap_elect, Zbase, xh, dxc, dxo, abar, zbar, &
            logT_max, logT_min, logRho_max, logRho_min, dlogT, dlogRho, logT, logRho
         integer :: logT_points, logRho_points, num_out, io_params, io_rho, io_tmp, &
            io_first, io_last, i, j, k, ierr, io
            
         logical, parameter :: compare_to_CO = .false.
         
         dir = 'plot_data'
         write(*,*) 'write data for opacity plots to ' // trim(dir)

         Zbase = 0.02d0
         xh = 0.30d0

         dxc = 0.0
         dxo = 0.0

         logT_points = 101
         logRho_points = 101
         
         logT_max = 7.3d0
         logT_min = 6.9d0
         logRho_max = 4.1d0
         logRho_min = 3.5d0
         
         logT_max = 9.2d0
         logT_min = 1.9d0
         logRho_max = 14d0
         logRho_min = -14d0
         
         logT_max = 10.2d0
         logT_min = 1.7d0
         logRho_max = 11d0
         logRho_min = -11d0
         
         if (.true.) then
            dxc =     0
            dxo =     0
            Zbase = 0.02d0
            xh = 0.0

            logT_max = 9
            logT_min = 3
            logRho_max = 8
            logRho_min = -10

            logT_points = 201
            logRho_points = 201
         end if

         call get_composition_info(zbase + dxc + dxo, xh, abar, zbar)
         
         kap_elect = 0.2d0*(1 + xh)
         
         io_params = io_unit0
         io_rho = io_unit0+1
         io_tmp = io_unit0+2
         io_first = io_unit0+3
         call Open_Plot_Outfiles(io_first, io_last, io_params, io_rho, io_tmp, dir, compare_to_CO)
         num_out = io_last - io_first + 1
         
         allocate(output_values(logRho_points,logT_points,num_out), &
            co_output_values(logRho_points,logT_points,num_out))

         write(io_params, '(99(f16.6,6x))') Zbase, xh, dxc, dxo
         write(io_params, '(99(i16,6x))') logRho_points, logT_points
         close(io_params)


         dlogT = (logT_max - logT_min)/(logT_points-1)
         dlogRho = (logRho_max - logRho_min)/(logRho_points-1)
         
         ierr = 0
         
         do j=1, logT_points
            logT = logT_min + dlogT*(j-1)
            do i=1,logRho_points
               logRho = logRho_min + dlogRho*(i-1)
               if (dxc /= 0 .or. dxo /= 0) then
                  call do1_CO_plot_data( &
                     logRho, logT, zbar, xh, Zbase, dxc, dxo, output_values, i, j, compare_to_CO, ierr)
               else
                  call do1_plot_data( &
                     logRho, logT, zbar, xh, Zbase, output_values, i, j, compare_to_CO, ierr)
                  if (compare_to_CO) then
                     call do1_CO_plot_data( &
                        logRho, logT, zbar, xh, Zbase, dxc, dxo, co_output_values, i, j, compare_to_CO, ierr)
                  end if
               end if
               if (ierr /= 0) exit
            end do
         end do

 01   format(e30.22)
         ! write out the results
         do j=1,logT_points
            write(io_tmp,01) logT_min + dlogT*(j-1)
         end do
         close(io_tmp)

         do i=1,logRho_points
            write(io_rho,01) logRho_min + dlogRho*(i-1)
         enddo
         close(io_rho)
         
         if (compare_to_CO) then
            write(*,*) 1
            write(io_first,'(e14.6)') output_values(1:logRho_points,1:logT_points,1)
            write(*,*) 2
            write(io_first+1,'(e14.6)') co_output_values(1:logRho_points,1:logT_points,1)
            write(*,*) 3
            write(io_first+2,'(e14.6)') &
               output_values(1:logRho_points,1:logT_points,1) - &
               co_output_values(1:logRho_points,1:logT_points,1)
         else
            do k = 1, num_out
               write(*,*) k
               write(io_first+k-1,'(e14.6)') output_values(1:logRho_points,1:logT_points,k)
            end do
         end if
      
         do io=io_first,io_last
            close(io)
         end do
         
         deallocate(output_values, co_output_values)
         
      end subroutine write_plot_data
      

      subroutine Open_Plot_Outfiles( &
            io_first, io_last, io_params, io_rho, io_tmp, dir, compare_to_CO)
         integer, intent(IN) :: io_first, io_params, io_rho, io_tmp
         integer, intent(OUT) :: io_last
         character (len=256), intent(IN) :: dir
         logical, intent(in) :: compare_to_CO
         character (len=256) :: fname
         integer :: io
         
         fname = trim(dir) // '/params.data'
         open(unit=io_params,file=trim(fname))
         
         fname = trim(dir) // '/logRho.data'
         open(unit=io_rho,file=trim(fname))
         
         fname = trim(dir) // '/logT.data'
         open(unit=io_tmp,file=trim(fname))
         
      
         if (compare_to_CO) then
            io = io_first
            fname = trim(dir) // '/kap.data'
            open(unit=io,file=trim(fname))
            fname = trim(dir) // '/kapCO.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/kap_sub_kapCO.data'
            io = io+1; open(unit=io,file=trim(fname))
         else 
            io = io_first
            fname = trim(dir) // '/logK.data'
            open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogK_dlogRho.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogK_dlogT.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/logKec.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogKec_dlogRho.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogKec_dlogT.data'
            io = io+1; open(unit=io,file=trim(fname))
         end if
            
         io_last = io
      
      end subroutine Open_Plot_Outfiles
         
      
      
      subroutine do1_plot_data( &
            lgd, lgT, zbar, xh, Zbase, output_values, i, j, compare_to_CO, ierr)
         use utils_lib, only: is_bad_num
         double precision, intent(in) :: lgd, lgT, zbar
         integer, intent(in) :: i, j
         double precision, intent(in) :: xh, Zbase
         double precision, intent(out) :: output_values(:,:,:)
         logical, intent(in) :: compare_to_CO
         integer, intent(out) :: ierr
         
         double precision :: kap, dlnkap_dlnRho, dlnkap_dlnT, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         ierr = 0
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         call kap_get_Type1( &
                  handle, zbar, xh, Zbase, lgd, lgT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,:) = -99
            ierr = 0
            return
         end if

         
         if (is_bad_num(kap)) then
            write(*,*) 'kap', kap
            stop 'do1_plot_data'
         end if

         if (compare_to_CO) then
            output_values(i,j,1) = kap
         else
            output_values(i,j,1) = safe_log10(kap)
         end if
         output_values(i,j,2) = dlnkap_dlnRho
         output_values(i,j,3) = dlnkap_dlnT

         call kap_get_elect_cond_opacity( &
                  zbar, lgd, lgT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,4) = -99
            output_values(i,j,5) = -99
            output_values(i,j,6) = -99
            ierr = 0
            return
         end if
         
         if (is_bad_num(kap)) then
            write(*,*) 'kap_get_elect_cond_opacity kap', kap
            stop 'do1_plot_data'
         end if

         output_values(i,j,4) = safe_log10(kap)
         output_values(i,j,5) = dlnkap_dlnRho
         output_values(i,j,6) = dlnkap_dlnT

      end subroutine do1_plot_data
         
      
      subroutine do1_CO_plot_data( &
            lgd, lgT, zbar, xh, zbase, dxc, dxo, output_values, i, j, compare_to_CO, ierr)
         use utils_lib, only: is_bad_num
         double precision, intent(in) :: lgd, lgT, zbar
         integer, intent(in) :: i, j
         double precision, intent(in) :: xh, zbase, dxc, dxo
         double precision, intent(out) :: output_values(:,:,:)
         logical, intent(in) :: compare_to_CO
         integer, intent(out) :: ierr
         double precision :: kap, dlnkap_dlnRho, dlnkap_dlnT, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         ierr = 0
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         call kap_get_Type2( &
                  handle, zbar, xh, zbase, dxc, dxo, lgd, lgT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,1) = -99
            output_values(i,j,2) = -99
            output_values(i,j,3) = -99
            ierr = 0
            return
         end if
         
         if (is_bad_num(kap)) then
            write(*,*) 'kap', kap, lgd, lgT, zbar, xh, zbase, dxc, dxo
            stop 'do1_CO_plot_data'
         end if
         
         if (is_bad_num(dlnkap_dlnRho)) then
            write(*,*) 'dlnkap_dlnRho', dlnkap_dlnRho, lgd, lgT, zbar, xh, zbase, dxc, dxo
            stop 'do1_CO_plot_data'
         end if
         
         if (is_bad_num(dlnkap_dlnT)) then
            write(*,*) 'dlnkap_dlnT', dlnkap_dlnT, lgd, lgT, zbar, xh, zbase, dxc, dxo
            stop 'do1_CO_plot_data'
         end if

         if (compare_to_CO) then
            output_values(i,j,1) = kap
         else
            output_values(i,j,1) = safe_log10(kap)
         end if
         output_values(i,j,2) = dlnkap_dlnRho
         output_values(i,j,3) = dlnkap_dlnT
      end subroutine do1_CO_plot_data
      

      end module test_kap_support

