! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module kap_lib
      ! library for calculating opacities

      use kap_def
      ! the data interface for the library is defined in kap_def
      
      implicit none


      contains ! the procedure interface for the library
      ! client programs should only call these routines.
            
      
      ! call this routine to initialize the kap module. 
      ! only needs to be done once at start of run.
      ! Reads data from the 'kap' directory in the data_dir.
      ! If use_cache is true and there is a 'kap/cache' directory, it will try that first.
      ! If it doesn't find what it needs in the cache, 
      ! it reads the data and writes the cache for next time.    
      subroutine kap_init( &
            data_dir, kap_file_prefix, CO_prefix, lowT_prefix, &
            blend_logT_upper_bdy, blend_logT_lower_bdy, use_cache, info)      
         use kap_def, only : kap_def_init, kap_is_initialized
         use load_kap, only : Setup_Kap_Tables
         character(*), intent(in) :: &
            data_dir, kap_file_prefix, CO_prefix, lowT_prefix
         real*8, intent(in) :: blend_logT_upper_bdy, blend_logT_lower_bdy
         logical, intent(in) :: use_cache
         integer, intent(out) :: info ! 0 means AOK.         
         logical, parameter :: load_on_demand = .true.
         info = 0    
         if (kap_is_initialized) return
         call kap_def_init(lowT_prefix)
         call Setup_Kap_Tables( &
            data_dir, kap_file_prefix, CO_prefix, lowT_prefix, &
            blend_logT_upper_bdy, blend_logT_lower_bdy, use_cache, load_on_demand, info)
         if (info /= 0) return
         kap_is_initialized = .true.
         !write(*,*) 'using opacity table set ', trim(kap_prefix)
      end subroutine kap_init

      
      subroutine kap_shutdown
         use load_kap, only : Free_Kap_Tables
         call Free_Kap_Tables  
         kap_is_initialized = .false.
      end subroutine kap_shutdown

      
      ! after kap_init has finished, you can allocate a "handle".
      
      integer function alloc_kap_handle(ierr)
         use kap_def,only:do_alloc_kap
         integer, intent(out) :: ierr
         alloc_kap_handle = do_alloc_kap(ierr)
      end function alloc_kap_handle      
      
      subroutine free_kap_handle(handle)
         ! frees the handle and all associated data
         use kap_def,only:Kap_General_Info,do_free_kap,get_kap_ptr
         integer, intent(in) :: handle
         call do_free_kap(handle)
      end subroutine free_kap_handle

      
      ! for typical use, you won't need the pointer corresponding to a handle,
      ! but just in case, this routine lets you get it.
      subroutine kap_ptr(handle,rq,ierr)
         use kap_def,only:Kap_General_Info,get_kap_ptr
         integer, intent(in) :: handle ! from alloc_kap_handle
         type (Kap_General_Info), pointer :: rq
         integer, intent(out):: ierr
         call get_kap_ptr(handle,rq,ierr)
      end subroutine kap_ptr
      

      ! kap evaluation
      ! you can call these routines after you've setup the tables for the handle.
      ! NOTE: the structures referenced via the handle are read-only
      ! for the evaulation routines, so you can do multiple evaluations in parallel
      ! using the same handle. 
      
      


      subroutine kap_set_choices( &
            handle, cubic_interpolation_in_X, &
            cubic_interpolation_in_Z, include_electron_conduction, ierr)
         integer, intent(in) :: handle ! from alloc_kap_handle
         logical, intent(in) :: &
            cubic_interpolation_in_X, cubic_interpolation_in_Z, include_electron_conduction
         integer, intent(out) :: ierr ! 0 means AOK.
         type (Kap_General_Info), pointer :: rq
         ierr = 0
         call kap_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         rq% cubic_interpolation_in_X = cubic_interpolation_in_X
         rq% cubic_interpolation_in_Z = cubic_interpolation_in_Z
         rq% include_electron_conduction = include_electron_conduction
      end subroutine kap_set_choices
           

      
      ! Zbase is the metallicity for the purpose of calculating opacites.
      ! there is no assumption that Zbase is equal to Z, where Z is 1 - (X + Y).
      ! the opacity includes electron conduction.
      subroutine kap_get_Type1( &
            handle, zbar, X, Zbase, log10_rho, log10_T,  &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         
         ! INPUT
         integer, intent(in) :: handle ! from alloc_kap_handle
         double precision, intent(in) :: zbar ! average ionic charge (for electron conduction)
         double precision, intent(in) :: X ! the hydrogen mass fraction
         double precision, intent(in) :: Zbase ! the metallicity
         double precision, intent(in) :: log10_rho ! the density
         double precision, intent(in) :: log10_T ! the temperature
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons
         
         ! OUTPUT
         double precision, intent(out) :: kap ! opacity
         double precision, intent(out) :: dlnkap_dlnRho ! partial derivative at constant T
         double precision, intent(out) :: dlnkap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
         
         double precision, parameter :: frac_Type1 = 1d0, dXC = 0d0, dXO = 0d0

         ierr = 0
         call kap_get_blend_1_2( &
            handle, zbar, X, Zbase, dXC, dXO, frac_Type1, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         
      end subroutine kap_get_Type1
           

      
      ! Zbase is the metallicity for the purpose of calculating opacites.
      ! there is no assumption that Zbase is equal to Z, where Z is 1 - (X + Y).
      ! dXC and dXO are the excess mass fractions of elemental carbon and oxygen.
      ! the opacity includes electron conduction.
      subroutine kap_get_Type2( &
            handle, zbar, X, Zbase, dXC, dXO, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         
         ! INPUT
         integer, intent(in) :: handle ! from alloc_kap_handle
         double precision, intent(in) :: zbar ! average ionic charge (for electron conduction)
         double precision, intent(in) :: X, Zbase, dXC, dXO ! composition    
         double precision, intent(in) :: log10_rho ! density
         double precision, intent(in) :: log10_T ! temperature
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons
         
         ! OUTPUT
         double precision, intent(out) :: kap ! opacity
         double precision, intent(out) :: dlnkap_dlnRho ! partial derivative at constant T
         double precision, intent(out) :: dlnkap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
         
         double precision, parameter :: frac_Type1 = 0d0

         ierr = 0
         call kap_get_blend_1_2( &
            handle, zbar, X, Zbase, dXC, dXO, frac_Type1, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
                     
      end subroutine kap_get_Type2


      subroutine kap_get_blend_1_2( &
            handle, zbar, X, Zbase, dXC, dXO, frac_Type1, log10_rho, log10_T, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         use kap_eval, only: Get_kap_Results

         ! INPUT
         integer, intent(in) :: handle ! from alloc_kap_handle
         double precision, intent(in) :: zbar ! average ionic charge (for electron conduction)
         double precision, intent(in) :: X, Zbase, dXC, dXO ! composition    
         double precision, intent(in) :: frac_Type1
         double precision, intent(in) :: log10_rho ! density
         double precision, intent(in) :: log10_T ! temperature
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons
         
         ! OUTPUT
         double precision, intent(out) :: kap ! opacity
         double precision, intent(out) :: dlnkap_dlnRho ! partial derivative at constant T
         double precision, intent(out) :: dlnkap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
         
         type (Kap_General_Info), pointer :: rq
         real :: logRho, logT, Rho, T

         ierr = 0
         call kap_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         logRho = real(log10_rho); Rho = 10**logRho
         logT = real(log10_T); T = 10**logT
         call Get_kap_Results( &
            rq, real(zbar), real(X), real(dXC), real(dXO), real(Zbase), &
            frac_Type1, Rho, logRho, T, logT,  &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
                     
      end subroutine kap_get_blend_1_2
      
      
      subroutine kap_get_elect_cond_opacity( &
            zbar, log10_rho, log10_T,  &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         use mod_condint, only: do_electron_conduction
         
         ! INPUT
         double precision, intent(in) :: zbar ! average ionic charge (for electron conduction)
         double precision, intent(in) :: log10_rho ! the density
         double precision, intent(in) :: log10_T ! the temperature
         
         ! OUTPUT
         double precision, intent(out) :: kap ! electron conduction opacity
         double precision, intent(out) :: dlnkap_dlnRho ! partial derivative at constant T
         double precision, intent(out) :: dlnkap_dlnT   ! partial derivative at constant Rho
         integer, intent(out) :: ierr ! 0 means AOK.
         
         real*8 :: kap_ec, dlnkap_ec_dlnRho, dlnkap_ec_dlnT

         ierr = 0
         call do_electron_conduction( &
            real(zbar), real(log10_rho), real(log10_T), &
            kap_ec, dlnkap_ec_dlnRho, dlnkap_ec_dlnT, ierr)
         kap = dble(kap_ec)
         dlnkap_dlnRho = dble(dlnkap_ec_dlnRho)
         dlnkap_dlnT = dble(dlnkap_ec_dlnT)
         
      end subroutine kap_get_elect_cond_opacity
      
      
      subroutine kap_get_compton_opacity( &
            Rho, T, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         use kap_eval, only: Compton_Opacity
         
         ! INPUT
         double precision, intent(in) :: Rho, T
         double precision, intent(in) :: lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
            ! free_e := total combined number per nucleon of free electrons and positrons
         
         ! OUTPUT
         double precision, intent(out) :: kap ! electron conduction opacity
         double precision, intent(out) :: dlnkap_dlnRho, dlnkap_dlnT
         integer, intent(out) :: ierr ! 0 means AOK.

         ierr = 0
         call Compton_Opacity(Rho, T, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         
      end subroutine kap_get_compton_opacity
		
		
      end module kap_lib

