! ***********************************************************************
!
!   Copyright (C) 2009  Bill Paxton

!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module create_fixed_metal_tables
		use const_def
		use kap_support
      implicit none


      contains
      
      
      subroutine Do_Test(data_dir, type1_table)
         character (len=*), intent(in) :: data_dir, type1_table
         double precision :: Zbase, X, XC, XO, Rho, logRho, T, logT, logK, logR
         integer :: info
         logical, parameter :: co_enhanced = .false.
         
         1 format(a40,1p e26.16)
         
         Zbase = 0.00d0
         X = 0.00d0
         XC = 0d0
         XO = 0d0
	
         call setup( &
            co_enhanced, data_dir, type1_table, Zbase, X, XC, XO)
         
         ! NOTE: you must have a logR that matches the OPAL tables
         logR = -1.4d0
         logT = 8.7d0
         logRho = logR + 3*logT - 18
         T = 10**logT
         rho = 10**logRho
         
         write(*,1) 'logR', logR
         write(*,1) 'logT', logT
         write(*,1) 'logRho', logRho
         write(*,1) 'z', z
         write(*,1) 'xh', x
         write(*,*) 'call Get_Results'
         write(*,*)

         call Get_Results( &
            Zbase, X, XC, XO, Rho, logRho, T, logT,  &
            logK, co_enhanced, data_dir, type1_table, info)
            
         write(*,*) 'info', info
         write(*,1) 'logK', logK
         write(*,1) 'kap', 10**logK
         write(*,*)
         stop 'Do_Test'
         
      end subroutine Do_Test


      subroutine Write_Files( &
            Z_in, which_x, output_dir, data_dir, type1_table, header_info, &
            file_prefix, table_version)
          double precision, intent(in) :: Z_in, which_x
          character (len=*), intent(in) :: &
            output_dir, data_dir, type1_table, header_info, file_prefix
          integer, intent(in) :: table_version
          integer, parameter :: max_num_Xs = 10
          double precision :: Z, Xs(max_num_Xs)
          integer :: ix, iz, num_Xs
         
          Z = Z_in
          if (which_x < 0) then
             Xs(1:max_num_Xs-1) = &
               (/ 0.00d0, 0.10d0, 0.20d0, 0.35d0, 0.50d0, 0.70d0, 0.80d0, 0.90d0, 0.95d0 /)
             num_Xs = max_num_Xs
          else
             Xs(1) = which_x
             num_Xs = 1
          end if
	       
	       if (num_Xs > 1) then
             iz = floor(Z*1d4 + 0.1d0)
             select case (iz)
             case (0)              ! Z = 0.0
                Xs(num_Xs) = 1
             case (1)              ! Z = 0.0001
                Xs(num_Xs) = 0.9999d0
             case (3)              ! Z = 0.0003
                Xs(num_Xs) = 0.9997d0
             case (10)             ! Z = 0.001
                Xs(num_Xs) = 0.999d0
             case (20)             ! Z = 0.002
                Xs(num_Xs) = 0.998d0
             case (40)             ! Z = 0.004
                Xs(num_Xs) = 0.996d0
             case (100)            ! Z = 0.01
                Xs(num_Xs) =  0.99d0
             case (200)            ! Z = 0.02
                Xs(num_Xs) =  0.98d0
             case (300)            ! Z = 0.03
                Xs(num_Xs) =  0.97d0
             case (400)            ! Z = 0.04
                Xs(num_Xs) =  0.96d0
             case (600)            ! Z = 0.06 
                num_Xs = max_num_Xs-1
                Xs(num_Xs) = 0.94d0
             case (800)            ! Z = 0.08
                num_Xs = max_num_Xs-1
                Xs(num_Xs) = 0.92d0
             case (1000)           ! Z = 0.1
                num_Xs = max_num_Xs-2
                Xs(num_Xs) = 0.90d0
             case default
                if (Freedman_flag) then ! use Ferguson Z=0.1 to fill gap for higher T
                  num_Xs = max_num_Xs-2
                  Xs(num_Xs) = 0.90d0
                else
                  write(*,*) 'unknown Z value for fixed metal', Z
                  stop 'Write_Files'
                end if
             end select
          end if
          
          do ix = 1,num_Xs
            call Do_Table( &
               Z, Xs(ix), output_dir, data_dir, type1_table, header_info, file_prefix, &
               table_version)
          end do
          
      end subroutine Write_Files
		
		
      subroutine Do_Table( &
         Z_in, X, output_dir, data_dir, type1_table, header_info, file_prefix, &
         table_version)
      double precision, intent(in) :: Z_in, X
      character (len=*), intent(in) :: output_dir, data_dir, type1_table
      character (len=*), intent(in) :: header_info, file_prefix
      integer, intent(in) :: table_version
      integer, parameter :: file_type = 1
      double precision, parameter :: XC = 0, XO = 0
      logical, parameter :: co_enhanced = .false.
      
      double precision :: logR, logT, T, logRho, Rho, logK
      integer :: i, j, io_unit, ios, info
      character (len=256) :: fname

      Z = Z_in

      io_unit = 34
			
      call create_fname(Z, X, output_dir, file_prefix, fname)
      
      open(unit=io_unit, file=trim(fname), action='write', status='replace', iostat=ios)
      if (ios /= 0) then
         write(*,*) 'fixed_metal Do_Table failed to open ', trim(fname)
         stop 1
      end if
	
      call setup( &
         co_enhanced, data_dir, type1_table, Z, X, XC, XO)

      ! header
      write(io_unit,'(a)') trim(header_info)
      write(io_unit,'(A8,99(2x,A10))') 'form','version', 'X   ','Z   ', 'logRs', 'logR_min', 'logR_max',&
         'logTs', 'logT_min', 'logT_max'
      write(io_unit,advance='no',fmt='(i8,2x,i10,2(2x,F10.6))') file_type, table_version, X, Z
      write(io_unit,advance='no',fmt='(2x,I10,2(2x,F10.6))') logR_points, logR_min, logR_max
      write(io_unit,fmt='(2x,I10,2(2x,F10.6))') logT_points, logT_min, logT_max
      
      ! data
      write(io_unit,'(/,a)') '   logT                       logR = logRho - 3*logT + 18'

      write(io_unit, advance='NO', fmt='(8x)')
      do j=1,logR_points
         logR = logR_min + dlogR*(j-1)
         write(io_unit, advance='NO', fmt='(F8.3)') logR
      enddo
      write(io_unit, *)
      write(io_unit, *)
      
      write(*,*) trim(fname)
			
      do i=1, logT_points
         logT = output_logTs(i)
         T = 10 ** logT
         write(io_unit, advance='NO', fmt='(F8.3)') logT
         do j=1,logR_points
            logR = logR_min + dlogR*(j-1)
            logRho = logR + 3*logT - 18
            Rho = 10**logRho
            call Get_Results(Z, X, XC, XO, Rho, logRho, T, logT,  &
               logK, co_enhanced, data_dir, type1_table, info)
            if (info /= 0 .or. logK > 99d0 .or. logK < -99d0 .or. logK-logK /= 0d0) then
               logK = -99.999d0
            end if
            write(io_unit, advance='NO', fmt='(F8.3)') logK
         enddo
         write(io_unit,*)
      enddo
      write(io_unit,*)
      
      close(io_unit)
      
      end subroutine Do_Table
		
		
      subroutine create_fname(Z, X, output_dir, file_prefix, fname)
         double precision, intent(in) :: Z, X
         character (len=*), intent(in) :: output_dir, file_prefix
         character (len=*), intent(out) :: fname
         character (len=16) :: zstr, xstr
         if (Freedman_flag) then
            call get_Freedman_output_Zstr(Z, zstr)			
            fname = trim(output_dir) // '/' // trim(file_prefix)// '_z' // &
               trim(zstr) // '.data'
         else		
            call get_output_Zstr(Z, zstr)			
            call get_output_Xstr(X, xstr)
            fname = trim(output_dir) // '/' // trim(file_prefix)// '_z' // &
               trim(zstr) // '_x' // trim(xstr) // '.data'
         end if
      end subroutine create_fname
		
		
      subroutine get_Freedman_output_Zstr(Z, zstr)
         double precision, intent(in) :: Z
         character (len=*),intent(out) :: zstr
         integer :: iz
         iz = floor(Z*1d5 + 0.1d0)
         select case (iz)
         case (1000)
            zstr = '0.01'
         case (2000)
            zstr = '0.02'
         case (4000)
            zstr = '0.04'
         case (10000)
            zstr = '0.10'
         case (20000)
            zstr = '0.20'
         case (63000)
            zstr = '0.63'
         case (100000)
            zstr = '1.00'
         case default
            write(*,*) 'unexpected Z value for Freedman data', Z
            stop 'get_Zstr'
         end select
      end subroutine get_Freedman_output_Zstr

		
      subroutine get_output_Zstr(Z, zstr)
         double precision, intent(in) :: Z
         character (len=*),intent(out) :: zstr
         integer :: iz
         iz = floor(Z*1d5 + 0.1d0)
         select case (iz)
         case (0)
            zstr = '0m0'
         case (1)
            zstr = '1m5'
         case (3)
            zstr = '3m5'
         case (10)
            zstr = '1m4'
         case (30)
            zstr = '3m4'
         case (100)
            zstr = '1m3'
         case (200)
            zstr = '2m3'
         case (400)
            zstr = '4m3'
         case (1000)
            zstr = '1m2'
         case (2000)
            zstr = '2m2'
         case (3000)
            zstr = '3m2'
         case (4000)
            zstr = '4m2'
         case (5000)
            zstr = '5m2'
         case (6000)
            zstr = '6m2'
         case (8000)
            zstr = '8m2'
         case (10000)
            zstr = '1m1'
         case default
            write(*,*) 'unexpected Z value', Z
            stop 'get_Zstr'
         end select
         
      end subroutine get_output_Zstr
      	
      subroutine get_output_Xstr(X, xstr)
         double precision, intent(in) :: X
         character (len=*),intent(out) :: xstr
         integer :: ix
         ix = floor(X*1d5 + 0.1d0)
         
!     0,  .1, .2, .35, .5, .7, .8, .9, .95, 1-Z
         select case (ix)
         case (0)
            xstr = '00'
         case (10000)
            xstr = '10'
         case (20000)
            xstr = '20'
         case (35000)
            xstr = '35'
         case (50000)
            xstr = '50'
         case (70000)
            xstr = '70'
         case (80000)
            xstr = '80'
         case (90000)
            xstr = '90'
         case (92000)
            xstr = '92'
         case (94000)
            xstr = '94'
         case (95000)
            xstr = '95'
         case (96000)
            xstr = '96'
         case (97000)
            xstr = '97'
         case (98000)
            xstr = '98'
         case (99000)
            xstr = '99'
         case (99600)
            xstr = '99.6'
         case (99800)
            xstr = '99.8'
         case (99900)
            xstr = '99.9'
         case (99970)
            xstr = '99.97'
         case (99990)
            xstr = '99.99'
         case (99997)
            xstr = '99.997'
         case (99999)
            xstr = '99.999'
         case (100000)
            xstr = '100'
         case default
            write(*,*) 'unknown X value', X, ix
            stop 'get_output_Xstr'
         end select
         
      end subroutine get_output_Xstr
      
      end module create_fixed_metal_tables  

