! ***********************************************************************
!
!   Copyright (C) 2009  Bill Paxton

!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module create_co_enhanced_tables
		use const_def
		use kap_support
      implicit none

      contains
      
      
      subroutine Do_CO_Test(data_dir, type1_table)
         character (len=*), intent(in) :: data_dir, type1_table
         double precision :: Zbase, X, dXC, dXO, Rho, logRho, T, logT, logK, opac_rad
         integer :: info
         logical, parameter :: co_enhanced = .true.
         
         1 format(a40,1pe26.16)
         
         Zbase = 0.02d0
         X = 0.70
         dXC = 0
         dXO = 0
	
	      call setup(co_enhanced, data_dir, type1_table, Zbase, X, dXC, dXO)
	      
	      logRho = -6
	      rho = 10**logRho
	      logT = 6
	      T = 10**logT

	      call Get_Results( &
	         Zbase, X, dXC, dXO, Rho, logRho, T, logT,  &
         	logK, co_enhanced, data_dir, type1_table, info)
         
         write(*,*) 'co_enhanced', co_enhanced
         write(*,*) 'info', info
         write(*,1) 'logT', logT
         write(*,1) 'logRho', logRho
         write(*,1) 'z', z
         write(*,1) 'xh', x
         write(*,*)
         write(*,1) 'kap', 10**logK
         write(*,1) 'logK', logK
         write(*,*)
         
      end subroutine Do_CO_Test


      subroutine Write_CO_Files( &
            Zbase, which_x, output_dir, data_dir, type1_table, & 
            header_info, file_prefix, table_version)
         double precision, intent(in) :: Zbase, which_x
         character (len=*), intent(in) :: &
            output_dir, data_dir, type1_table, header_info, file_prefix
         integer, intent(in) :: table_version
         integer, parameter :: max_num_Xs = 4
         double precision :: Xs(max_num_Xs)
         double precision :: X
         integer :: ix, num_Xs
         
         if (which_x < 0) then
            num_Xs = 4
            Xs(1:num_Xs) = (/ 0.00, 0.10, 0.35, 0.70 /) ! ferg data missing X = 0.03
         else
            num_Xs = 1
            Xs(1) = which_x
         end if
         
         do ix = 1,num_Xs
            X = Xs(ix)
            call Do_CO_Tables(Zbase, X, output_dir, data_dir, type1_table, & 
                   header_info, file_prefix, table_version)
         end do
         
      end subroutine Write_CO_Files
		
      
      subroutine Create_Filename(Z, X, output_dir, file_prefix, fname)
         double precision, intent(in) :: Z, X
         character (len=*),intent(in) :: output_dir, file_prefix
         character (len=*),intent(out) :: fname
         
         character (len=4) :: zstr, xstr
         integer :: i
         
         i = floor(X*1d5+0.1d0)
         select case (i)
         case (0)
            xstr = '00'
         case (10000)
            xstr = '10'
         case (20000)
            xstr = '20'
         case (35000)
            xstr = '35'
         case (50000)
            xstr = '50'
         case (70000)
            xstr = '70'
         case (80000)
            xstr = '80'
         case (90000)
            xstr = '90'
         case (92000)
            xstr = '92'
         case (94000)
            xstr = '94'
         case (95000)
            xstr = '95'
         case (96000)
            xstr = '96'
         case (97000)
            xstr = '97'
         case (98000)
            xstr = '98'
         case (99000)
            xstr = '99'
         case (99600)
            xstr = '99.6'
         case (99800)
            xstr = '99.8'
         case (99900)
            xstr = '99.9'
         case (99970)
            xstr = '99.97'
         case (99990)
            xstr = '99.99'
         case (99997)
            xstr = '99.997'
         case (99999)
            xstr = '99.999'
         case (100000)
            xstr = '100'
         case default
            write(*,*) 'unknown X value', X, i
            stop 'get_output_Xstr'
         end select
         
         i = floor(Z*1d5 + 0.1d0)
         select case (i)
         case (0)
            zstr = '0m0'
         case (1)
            zstr = '1m5'
         case (3)
            zstr = '3m5'
         case (10)
            zstr = '1m4'
         case (30)
            zstr = '3m4'
         case (100)
            zstr = '1m3'
         case (200)
            zstr = '2m3'
         case (400)
            zstr = '4m3'
         case (1000)
            zstr = '1m2'
         case (2000)
            zstr = '2m2'
         case (3000)
            zstr = '3m2'
         case (4000)
            zstr = '4m2'
         case (5000)
            zstr = '5m2'
         case (6000)
            zstr = '6m2'
         case (8000)
            zstr = '8m2'
         case (10000)
            zstr = '1m1'
         case default
            write(*,*) 'unknown Z value for ferg data', Z, i
            stop 'get_Zstr'
         end select
         
         write(fname,'(a)') &
            trim(output_dir) // &
            '/' // trim(file_prefix) // '_z' // &
            trim(zstr) // '_x' // trim(xstr) // '.data'
			   
      end subroutine Create_Filename
		
		
      subroutine Do_CO_Tables( &
            Zbase, X, output_dir, data_dir, type1_table, header_info, file_prefix, &
            table_version)
         double precision, intent(in) :: Zbase, X
         character (len=*),intent(in) :: output_dir, data_dir, type1_table, header_info, file_prefix
         integer, intent(in) :: table_version
         integer, parameter :: num_dXCOs = 8
         double precision :: dXCOs(num_dXCOs), Y, dXC, dXO, mid, XC_base, XO_base
         integer :: i, j, num_Tables, io_unit, pass, num1, num2, num3, ios
         character (len=256) :: fname
         double precision, parameter :: tiny = 1d-10
			
         integer, parameter :: file_type = 2

         io_unit = 33
         call Create_Filename(Zbase, X, output_dir, file_prefix, fname)
         open(unit=io_unit, file=trim(fname), action='write', status='replace', iostat=ios)
         if (ios /= 0) then
            write(*,*) 'failed to open ', trim(fname)
            stop 1
         end if		
         write(*,*) 'creating ', trim(fname)
			
         dXCOs(1:num_dXCOs) = (/ 0.00d0, 0.01d0, 0.03d0, 0.10d0, 0.20d0, 0.40d0, 0.60d0, 1.0d0 /)
         Y = 1 - (X + Zbase)    ! this sets upper limit on sum of dXC + dXO			
         mid = Y * 0.5d0
         
         XC_base = Zbase*Zfrac_C
         XO_base = Zbase*Zfrac_O
         
         do pass = 1, 3 
           ! count the tables on the 1st pass, write a list of tables on 2nd,
           ! and write the actual tables on the 3rd. 
            
            if (pass == 2) then ! write the header
               write(io_unit,'(a)') trim(header_info)
               write(io_unit,'(A8,2(2x,A8),4(2x,A16),99(2x,A8))') &
                  'form', 'version', 'Tables','X  ','Zbase', 'XC_base', 'XO_base', &
                  'logRs','min','max','logTs','min','max'
               write(io_unit,advance='no',fmt='(I8,2(2x,I8),4(2x,E16.6))') &
                  file_type, table_version, num_Tables, X, Zbase, XC_base, XO_base
               write(io_unit,advance='no',fmt='(2x,I8,2(2x,F8.2))') logR_points, logR_min, logR_max
               write(io_unit,fmt='(2x,I8,2(2x,F8.2))') logT_points, logT_min, logT_max
            end if
				
            num_Tables = 0				
            
            if (pass == 2) write(io_unit,'(/,i3,a,/,4(a12))') num1, ' tables with dXC > dXO', 'Num',  &
            'Y   ', 'dXC ', 'dXO '
            ! 1) dXC > dXO
            do j = 1, num_dXCOs
               dXO = dXCOs(j)
               if (dXO >= mid-tiny) exit
               do i = j+1, num_dXCOs
                  dXC = dXCOs(i)
                  if (dXC+dXO > Y) dXC = Y-dXO
                  num_Tables = num_Tables + 1
                  if (pass == 2) then
                     write(io_unit,'(i12,3f12.4)') num_Tables, Y-(dXC+dXO), dXC, dXO
                  else if (pass == 3) then
                     call Do_CO_Table(Zbase,X,dXC,dXO,num_Tables,io_unit,data_dir,type1_table)
                  end if
                  if (dXC+dXO > Y-tiny) exit
               end do
            end do
            if (pass == 1) num1 = num_Tables
            
            if (pass == 2) write(io_unit,'(/,i3,a,/,4(a12))') num2, ' tables with dXC == dXO',  &
                  'Num', 'Y   ', 'dXC ', 'dXO '
            ! 2) dXC == dXO
            do i = 1, num_dXCOs
               if (dXCOs(i) > mid+tiny) exit
               num_Tables = num_Tables + 1
               dXC = dXCOs(i); dXO = dXC
               if (pass == 2) then
                  write(io_unit,'(i12,3f12.4)') num_Tables, Y-(dXC+dXO), dXC, dXO
               else if (pass == 3) then
                  call Do_CO_Table(Zbase, X, dXC, dXO, num_Tables, io_unit, data_dir, type1_table)
               end if
               if (dXCOs(i) >= mid-tiny) exit
            end do
            if (pass == 1) num2 = num_Tables - num1
            
            if (pass == 2) write(io_unit,'(/,i3,a,/,4(a12))') num3,  &
                     ' tables with dXC < dXO', 'Num', 'Y   ', 'dXC ', 'dXO '
            ! 3) dXC < dXO
            do j = 1, num_dXCOs
               dXC = dXCOs(j)
               if (dXC >= mid-tiny) exit
               do i = j+1, num_dXCOs
                  dXO = dXCOs(i)
                  if (dXC+dXO > Y) dXO = Y-dXC
                  num_Tables = num_Tables + 1
                  if (pass == 2) then
                     write(io_unit,'(i12,3f12.4)') num_Tables, Y-(dXC+dXO), dXC, dXO
                  else if (pass == 3) then
                     call Do_CO_Table(Zbase,X,dXC,dXO,num_Tables,io_unit,data_dir,type1_table)
                  end if
                  if (dXC+dXO > Y-tiny) exit
               end do
            end do
            if (pass == 1) num3 = num_Tables - (num1+num2)
            
            if (pass == 2) write(io_unit,'(/,a,/)')  &
            '---------------------------------------------------------------------------'
            
         end do
         
         close(io_unit)
         
      end subroutine Do_CO_Tables
		
      
      subroutine Do_CO_Table(Zbase, X, XC, XO, table_num, io_unit, data_dir, type1_table)
         double precision, intent(in) :: Zbase, X, XC, XO
         integer, intent(in) :: table_num, io_unit
         character (len=*), intent(in) :: data_dir, type1_table
			
         integer i,j,info
         double precision :: opac_rad, logR, logT, T, logRho, Rho, logK
         logical, parameter :: co_enhanced = .true.
         
         call setup(co_enhanced, data_dir, type1_table, Zbase, X, XC, XO)
      
         write(io_unit,'(A10,10x,10(A11))') 'Table', 'X  ',  'Y  ', 'Zbase', 'dXC ', 'dXO '
         write(io_unit,'(6x,I2,17x,5(F6.3,5x))') table_num, X, Y, Zbase, XC, XO
         write(io_unit,'(/,a)') '   logT                       logR = logRho - 3*logT + 18'

         write(io_unit, advance='NO', fmt='(8x)')
         do j=1,logR_points
            logR = logR_min + dlogR*(j-1)
            write(io_unit, advance='NO', fmt='(F8.3)') logR
         enddo
         write(io_unit, *)
         write(io_unit, *)
			
         do i=1, logT_points
            logT = output_logTs(i)
            T = 10 ** logT
            write(io_unit, advance='NO', fmt='(F8.3)') logT
            do j=1,logR_points
               logR = logR_min + dlogR*(j-1)
               logRho = logR + 3*logT - 18
               Rho = 10**logRho
               call Get_Results(Zbase, X, XC, XO, Rho, logRho, T, logT,  &
                  logK, co_enhanced, data_dir, type1_table, info)
               if (info /= 0 .or. logK > 99d0 .or. logK < -99d0 .or. logK-logK /= 0d0) logK = -99.999d0
               
               write(io_unit, advance='NO', fmt='(F8.3)') logK
            enddo
            write(io_unit,*)
         enddo
         write(io_unit,*)
            
      end subroutine Do_CO_Table


      end module create_co_enhanced_tables
