      subroutine do_mkbicub_db(x,nx,y,ny,f,nf2,
     >   ibcxmin,bcxmin,ibcxmax,bcxmax,
     >   ibcymin,bcymin,ibcymax,bcymax,
     >   ilinx,iliny,ier)
C
C  setup bicubic spline, dynamic allocation of workspace
C  fortran-90 fixed form source
C
C  --NOTE-- dmc 22 Feb 2004 -- rewrite for direct calculation of
C  coefficients, to avoid large transient use of memory.
C
C
      implicit NONE
C
C  input:
      integer nx                        ! length of x vector
      integer ny                        ! length of y vector
      double precision x(nx)                        ! x vector, strict ascending
      double precision y(ny)                        ! y vector, strict ascending
C
      integer nf2                       ! 2nd dimension of f, nf2.ge.nx
C  input/output:
      double precision f(4,nf2,ny)                  ! data & spline coefficients
C
C  on input:  f(1,i,j) = f(x(i),y(j))
C  on output:  f(1,i,j) unchanged
C              f(2,i,j) = d2f/dx2(x(i),y(j))
C              f(3,i,j) = d2f/dy2(x(i),y(j))
C              f(4,i,j) = d4f/dx2dy2(x(i),y(j))
C
C  and the interpolation formula for (x,y) in (x(i),x(i+1))^(y(j),y(j+1))
C  is:
C        hx = x(i+1)-x(i)   hy = y(j+1)-y(j)
C        dxp= (x-x(i))/hx   dxm= 1-dxp     dxp,dxm in (0,1)
C        dyp= (x-x(i))/hx   dym= 1-dyp     dyp,dym in (0,1)
C        dx3p = dxp**3-dxp  dx3m = dxm**3-dxm     dxp3,dxm3 in (0,1)
C
C   finterp = dxm*(dym*f(1,i,j)+dyp*f(1,i,j+1))
C            +dxp*(dym*f(1,i+1,j)+dyp*f(1,i+1,j+1))
C     +1/6*hx**2*
C            dx3m*(dym*f(2,i,j)+dyp*f(2,i,j+1))
C           +dx3p*(dym*f(2,i+1,j)+dyp*f(2,i+1,j+1))
C     +1/6*hy**2*
C            dxm*(dy3m*f(3,i,j)+dy3p*f(3,i,j+1))
C           +dxp*(dy3m*f(3,i+1,j)+dy3p*f(3,i+1,j+1))
C     +1/36*hx**2*hy**2*
C            dx3m*(dym*f(4,i,j)+dyp*f(4,i,j+1))
C           +dx3p*(dym*f(4,i+1,j)+dyp*f(4,i+1,j+1))
C
C  where the f(2:4,*,*) are cleverly evaluated to assure
C  (a) finterp is continuous and twice differentiable across all
C      grid cell boundaries, and
C  (b) all boundary conditions are satisfied.
C
C  input bdy condition data:
      integer ibcxmin                   ! bc flag for x=xmin
      double precision bcxmin(ny)                   ! bc data vs. y at x=xmin
      integer ibcxmax                   ! bc flag for x=xmax
      double precision bcxmax(ny)                   ! bc data vs. y at x=xmax
C
      integer ibcymin                   ! bc flag for y=ymin
      double precision bcymin(nx)                   ! bc data vs. x at y=ymin
      integer ibcymax                   ! bc flag for y=ymax
      double precision bcymax(nx)                   ! bc data vs. x at y=ymax
C
C  with interpretation:
c   ibcxmin -- indicator for boundary condition at x(1):
c    bcxmin(...) -- boundary condition data
c     =-1 -- periodic boundary condition
c     =0 -- use "not a knot"
c     =1 -- match slope, specified at x(1),th(ith) by bcxmin(ith)
c     =2 -- match 2nd derivative, specified at x(1),th(ith) by bcxmin(ith)
c     =3 -- boundary condition is slope=0 (df/dx=0) at x(1), all th(j)
c     =4 -- boundary condition is d2f/dx2=0 at x(1), all th(j)
c     =5 -- match 1st derivative to 1st divided difference
c     =6 -- match 2nd derivative to 2nd divided difference
c     =7 -- match 3rd derivative to 3rd divided difference
c           (for more detailed definition of BCs 5-7, see the
c           comments of subroutine mkspline)
c   NOTE bcxmin(...) referenced ONLY if ibcxmin=1 or ibcxmin=2
c
c   ibcxmax -- indicator for boundary condition at x(nx):
c    bcxmax(...) -- boundary condition data
c     (interpretation as with ibcxmin, bcxmin)
c   NOTE:  if ibcxmin=-1, ibcxmax is ignored! ...and the BC is periodic.
c
C  and analogous interpretation for ibcymin,bcymin,ibcymax,bcymax
C  (df/dy or d2f/dy2 boundary conditions at y=ymin and y=ymax).
C
C  output linear grid flags and completion code (ier=0 is normal):
C
      integer ilinx                     ! =1: x grid is "nearly" equally spaced
      integer iliny                     ! =1: y grid is "nearly" equally spaced
C  ilinx and iliny are set to zero if corresponding grids are not equally
C  spaced
C
      integer ier                       ! =0 on exit if there is no error.
C
C  if there is an error, ier is set and a message is output on unit 6.
C  these are considered programming errors in the calling routine.
C
C  possible errors:
C    x(...) not strict ascending
C    y(...) not strict ascending
C    nx .lt. 4
C    ny .lt. 4
C    invalid boundary condition flag
C
C-----------------------
      integer ierx,iery
C
      double precision, dimension(:,:), allocatable :: fwk
      double precision :: zbcmin,zbcmax
      integer ix,iy,ibcmin,ibcmax
c
      double precision, dimension(:,:,:), allocatable :: fcorr
      integer iflg2
      double precision zdiff(2),hy
C
C-----------------------
c
c  see if 2nd pass is needed due to inhomogeneous d/dy bdy cond.
c
      iflg2=0
      if(ibcymin.ne.-1) then
         if((ibcymin.eq.1).or.(ibcymin.eq.2)) then
            do ix=1,nx
               if (bcymin(ix).ne.0.0) iflg2=1
            enddo
         endif
         if((ibcymax.eq.1).or.(ibcymax.eq.2)) then
            do ix=1,nx
               if (bcymax(ix).ne.0.0) iflg2=1
            enddo
         endif
      endif
c
c  check boundary condition specifications
c
      ier=0
c
      call ibc_ck_db(ibcxmin,'bcspline','xmin',-1,7,ier)
      if(ibcxmin.ge.0) call ibc_ck_db(ibcxmax,'bcspline','xmax',0,7,ier)
      call ibc_ck_db(ibcymin,'bcspline','ymin',-1,7,ier)
      if(ibcymin.ge.0) call ibc_ck_db(ibcymax,'bcspline','ymax',0,7,ier)
c
c  check ilinx & x vector
c
      call splinck_db(x,nx,ilinx,1.0d-3,ierx)
      if(ierx.ne.0) ier=2
c
c      if(ier.eq.2) then
c         write(6,'('' ?bcspline:  x axis not strict ascending'')')
c      endif
c
c  check iliny & y vector
c
      call splinck_db(y,ny,iliny,1.0d-3,iery)
      if(iery.ne.0) ier=3
c
c      if(ier.eq.3) then
c         write(6,'('' ?bcspline:  y axis not strict ascending'')')
c      endif
c
      if(ier.ne.0) return
c
c------------------------------------
      allocate(fwk(2,max(nx,ny)))
c
c  evaluate fxx (spline in x direction)
c
      zbcmin=0
      zbcmax=0
      do iy=1,ny
         fwk(1,1:nx) = f(1,1:nx,iy)
         if((ibcxmin.eq.1).or.(ibcxmin.eq.2)) zbcmin=bcxmin(iy)
         if((ibcxmax.eq.1).or.(ibcxmax.eq.2)) zbcmax=bcxmax(iy)
         call mkspline_db(x,nx,fwk,
     >      ibcxmin,zbcmin,ibcxmax,zbcmax,ilinx,ier)
         if(ier.ne.0) then
            deallocate(fwk)
            return
         end if
         f(2,1:nx,iy)=fwk(2,1:nx)
      enddo
c
c  evaluate fyy (spline in y direction)
c  use homogeneous boundary condition; correction done later if necessary
c
      zbcmin=0
      zbcmax=0
      ibcmin=ibcymin
      ibcmax=ibcymax
      do ix=1,nx
         fwk(1,1:ny) = f(1,ix,1:ny)
         if(iflg2.eq.1) then
            if((ibcymin.eq.1).or.(ibcymin.eq.2)) ibcmin=0
            if((ibcymax.eq.1).or.(ibcymax.eq.2)) ibcmax=0
         endif
         call mkspline_db(y,ny,fwk,
     >      ibcmin,zbcmin,ibcmax,zbcmax,iliny,ier)
         if(ier.ne.0) then
            deallocate(fwk)
            return
         end if
         f(3,ix,1:ny)=fwk(2,1:ny)
      enddo
c
c  evaluate fxxyy (spline fxx in y direction; BC simplified; avg
c  d2(d2f/dx2)/dy2 and d2(df2/dy2)/dx2
c
      zbcmin=0
      zbcmax=0
      ibcmin=ibcymin
      ibcmax=ibcymax
      do ix=1,nx
         fwk(1,1:ny) = f(2,ix,1:ny)
         if(iflg2.eq.1) then
            if((ibcymin.eq.1).or.(ibcymin.eq.2)) ibcmin=0
            if((ibcymax.eq.1).or.(ibcymax.eq.2)) ibcmax=0
         endif
         call mkspline_db(y,ny,fwk,
     >      ibcmin,zbcmin,ibcmax,zbcmax,iliny,ier)
         if(ier.ne.0) then
            deallocate(fwk)
            return
         end if
         f(4,ix,1:ny)= fwk(2,1:ny)
      enddo
c
      if(iflg2.eq.1) then
         allocate(fcorr(2,nx,ny))
c
c  correct for inhomogeneous y boundary condition
c
         do ix=1,nx
            !  the desired inhomogenous BC is the difference btw the 
            !  requested derivative (1st or 2nd) and the current value

            zdiff(1)=0.0
            if(ibcymin.eq.1) then
               hy=y(2)-y(1)
               zdiff(1)=(f(1,ix,2)-f(1,ix,1))/hy +
     >            hy*(-2*f(3,ix,1)-f(3,ix,2))/6
               zdiff(1)=bcymin(ix)-zdiff(1)
            else if(ibcymin.eq.2) then
               zdiff(1)=bcymin(ix)-f(3,ix,1)
            endif

            zdiff(2)=0.0
            if(ibcymax.eq.1) then
               hy=y(ny)-y(ny-1)
               zdiff(2)=(f(1,ix,ny)-f(1,ix,ny-1))/hy + 
     >            hy*(2*f(3,ix,ny)+f(3,ix,ny-1))/6
               zdiff(2)=bcymax(ix)-zdiff(2)
            else if(ibcymax.eq.2) then
               zdiff(2)=bcymax(ix)-f(3,ix,ny)
            endif
c
            fwk(1,1:ny)=0.0  ! values are zero; only BC is not
            call mkspline_db(y,ny,fwk,ibcymin,zdiff(1),ibcymax,zdiff(2),
     >         iliny,ier)
            if(ier.ne.0) then
               deallocate(fwk,fcorr)
               return
            end if
            fcorr(1,ix,1:ny)=fwk(2,1:ny)  ! fyy-correction
         enddo
c
         zbcmin=0
         zbcmax=0
         do iy=1,ny
            fwk(1,1:nx)=fcorr(1,1:nx,iy)
            call mkspline_db(x,nx,fwk,ibcxmin,zbcmin,ibcxmax,zbcmax,
     >         ilinx,ier)
            if(ier.ne.0) then
               deallocate(fwk,fcorr)
               return
            end if
            fcorr(2,1:nx,iy)=fwk(2,1:nx)  ! fxxyy-correction
         enddo
c
         f(3:4,1:nx,1:ny)=f(3:4,1:nx,1:ny)+fcorr(1:2,1:nx,1:ny)
c
         deallocate(fcorr)        
      endif
c
c  correction spline -- f=fxx=zero; fyy & fxxyy are affected
c
      deallocate(fwk)
c------------------------------------
C
C  thats all
C
      return
      end subroutine do_mkbicub_db



      subroutine mkspline_db(x,nx,fspl,ibcxmin,bcxmin,ibcxmax,bcxmax,
     >   ilinx,ier)
      implicit none
C
C  make a 2-coefficient 1d spline
C
C  only 2 coefficients, the data and its 2nd derivative, are needed to
C  fully specify a spline.  See e.g. Numerical Recipies in Fortran-77
C  (2nd edition) chapter 3, section on cubic splines.
C
C  input:
      integer nx                        ! no. of data points
      double precision x(nx)                        ! x axis data, strict ascending order
C
C  input/output:
      double precision fspl(2,nx)                   ! f(1,*):  data in; f(2,*):  coeffs out
C
C     f(1,j) = f(x(j))  on input (unchanged on output)
C     f(2,j) = f''(x(j)) (of interpolating spline) (on output).
C
C  ...boundary conditions...
C
C  input:
C
      integer ibcxmin                   ! b.c. flag @ x=xmin=x(1)
      double precision bcxmin                       ! b.c. data @xmin
C
      integer ibcxmax                   ! b.c. flag @ x=xmax=x(nx)
      double precision bcxmax                       ! b.c. data @xmax
C
C  ibcxmin=-1 -- periodic boundary condition
C                (bcxmin,ibcxmax,bcxmax are ignored)
C
C                the output spline s satisfies
C                s'(x(1))=s'(x(nx)) ..and.. s''(x(1))=s''(x(nx))
C
C  if non-periodic boundary conditions are used, then the xmin and xmax
C  boundary conditions can be specified independently:
C
C  ibcxmin (ibcxmax) = 0 -- this specifies a "not a knot" boundary
C                condition, see "cubsplb.for".  This is a common way
C                for inferring a "good" spline boundary condition
C                automatically from data in the vicinity of the
C                boundary.  ... bcxmin (bcxmax) are ignored.
C
C  ibcxmin (ibcxmax) = 1 -- boundary condition is to have s'(x(1))
C                ( s'(x(nx)) ) match the passed value bcxmin (bcxmax).
C
C  ibcxmin (ibcxmax) = 2 -- boundary condition is to have s''(x(1))
C                ( s''(x(nx)) ) match the passed value bcxmin (bcxmax).
C
C  ibcxmin (ibcxmax) = 3 -- boundary condition is to have s'(x(1))=0.0
C                ( s'(x(nx))=0.0 )
C
C  ibcxmin (ibcxmax) = 4 -- boundary condition is to have s''(x(1))=0.0
C                ( s''(x(nx))=0.0 )
C
C  ibcxmin (ibcxmax) = 5 -- boundary condition is to have s'(x(1))
C                ( s'(x(nx)) ) match the 1st divided difference
C                e.g. at x(1):  d(1)/h(1), where
C                           d(j)=f(1,j+1)-f(1,j)
C                           h(j)=x(j+1)-x(j)
C
C  ibcxmin (ibcxmax) = 6 -- BC is to have s''(x(1)) ( s''(x(nx)) )
C                match the 2nd divided difference
C                e.g. at x(1):
C                     e(1) = [d(2)/h(2) - d(1)/h(1)]/(0.5*(h(1)+h(2)))
C
C  ibcxmin (ibcxmax) = 7 -- BC is to have s'''(x(1)) ( s'''(x(nx)) )
C                match the 3rd divided difference
C                e.g. at x(1): [e(2)-e(1)]/(0.33333*(h(1)+h(2)+h(3)))
C
C  output:
C
      integer ilinx                     ! =1: hint, x axis is ~evenly spaced
C
C  let dx[avg] = (x(nx)-x(1))/(nx-1)
C  let dx[j] = x(j+1)-x(j), for all j satisfying 1.le.j.lt.nx
C
C  if for all such j, abs(dx[j]-dx[avg]).le.(1.0d-3*dx[avg]) then
C  ilinx=1 is returned, indicating the data is (at least nearly)
C  evenly spaced.  Even spacing is useful, for speed of zone lookup,
C  when evaluating a spline.
C
C  if the even spacing condition is not satisfied, ilinx=2 is returned.
C
      integer ier                       ! exit code, 0=OK
C
C  an error code is returned if the x axis is not strict ascending,
C  or if nx.lt.4, or if an invalid boundary condition specification was
C  input.
C
C------------------------------------
C
C  this routine calls traditional 4 coefficient spline software, and
C  translates the result to 2 coefficient form.
C
C  this could be done more efficiently but we decided out of conservatism
C  to use the traditional software.
C
C------------------------------------
      integer i, inwk
C  workspaces -- f90 dynamically allocated
C
      double precision, dimension(:,:), allocatable :: fspl4 ! traditional 4-spline
      double precision, dimension(:), allocatable :: wk ! cspline workspace
C
C------------------------------------
C
      allocate(fspl4(4,nx),wk(nx))
C
C  make the traditional call
C
      do i=1,nx
         fspl4(1,i)=fspl(1,i)
         fspl(2,i)=0.0                  ! for now
      enddo
C
      inwk=nx
C
C  boundary conditions imposed by cspline...
C
      call cspline_db(x,nx,fspl4,ibcxmin,bcxmin,ibcxmax,bcxmax,
     >   wk,inwk,ilinx,ier)
C
      if(ier.eq.0) then
C
C  copy the output -- careful of end point.
C
         do i=1,nx-1
            fspl(2,i)=2.0*fspl4(3,i)
         enddo
         fspl(2,nx)=2.0*fspl4(3,nx-1) +
     >        (x(nx)-x(nx-1))*6.0*fspl4(4,nx-1)
      endif
C
      deallocate(fspl4,wk)
C
      return
      end subroutine mkspline_db


      subroutine splinck_db(x,inx,ilinx,ztol,ier)
      implicit none
C
C  check if a grid is strictly ascending and if it is evenly spaced
C  to w/in ztol
C
      integer inx
      double precision x(inx)                       ! input -- grid to check
C
      integer ilinx                     ! output -- =1 if evenly spaced =2 O.W.
C
      double precision ztol                         ! input -- spacing check tolerance
C
      integer ier                       ! output -- =0 if OK
C
C  ier=1 is returned if x(1...inx) is NOT STRICTLY ASCENDING...
C
C-------------------------------
C
      double precision zeps, zdiffx, zdiff, dxavg
      integer ix
      
      ier=0
      ilinx=1
      if(inx.le.1) return
c
      dxavg=(x(inx)-x(1))/(inx-1)
      zeps=abs(ztol*dxavg)
c
      do ix=2,inx
         zdiffx=(x(ix)-x(ix-1))
         if(zdiffx.le.0.0) ier=2
         zdiff=zdiffx-dxavg
         if(abs(zdiff).gt.zeps) then
            ilinx=2
         endif
      enddo
 10   continue
c
      return
      end subroutine splinck_db

      SUBROUTINE V_SPLINE_db(k_bc1,k_bcn,n,x,f,wk)
!***********************************************************************
!V_SPLINE evaluates the coefficients for a 1d cubic interpolating spline
!References:
!  Forsythe, Malcolm, Moler, Computer Methods for Mathematical
!    Computations, Prentice-Hall, 1977, p.76
!  Engeln-Muellges, Uhlig, Numerical Algorithms with Fortran, Springer,
!    1996, p.251
!  W.A.Houlberg, D.McCune 3/2000
!Input:
!  k_bc1-option for BC at x(1)
!       =-1 periodic, ignore k_bcn
!       =0 not-a-knot
!       =1 s'(x1) = input value of f(2,1)
!       =2 s''(x1) = input value of f(3,1)
!       =3 s'(x1) = 0.0
!       =4 s''(x1) = 0.0
!       =5 match first derivative to first 2 points
!       =6 match second derivative to first 3 points
!       =7 match third derivative to first 4 points
!       =else use not-a-knot
!  k_bcn-option for boundary condition at x(n)
!       =0 not-a-knot
!       =1 s'(x1) = input value of f(2,1)
!       =2 s''(x1) = input value of f(3,1)
!       =3 s'(x1) = 0.0
!       =4 s''(x1) = 0.0
!       =5 match first derivative to first 2 points
!       =6 match second derivative to first 3 points
!       =7 match third derivative to first 4 points
!       =else use knot-a-knot
!  n-number of data points or knots-(n.ge.2)
!  x(n)-abscissas of the knots in strictly increasing order
!  f(1,i)-ordinates of the knots
!  f(2,1)-input value of s'(x1) for k_bc1=1
!  f(2,n)-input value of s'(xn) for k_bcn=1
!  f(3,1)-input value of s''(x1) for k_bc1=2
!  f(3,n)-input value of s''(xn) for k_bcn=2
!  wk(n)-scratch work area for periodic BC
!Output:
!  f(2,i)=s'(x(i))
!  f(3,i)=s''(x(i))
!  f(4,i)=s'''(x(i))
!Comments:
!  s(x)=f(1,i)+f(2,i)*(x-x(i))+f(3,i)*(x-x(i))**2/2!
!       +f(4,i)*(x-x(i))**3/3! for x(i).le.x.le.x(i+1)
!  W_SPLINE can be used to evaluate the spline and its derivatives
!  The cubic spline is twice differentiable (C2)
!
!  bugfixes -- dmc 24 Feb 2004:
!    (a) fixed logic for not-a-knot:
!          !    Set f(3,1) for not-a-knot
!                    IF(k_bc1.le.0.or.k_bc1.gt.7) THEN ...
!        instead of
!          !    Set f(3,1) for not-a-knot
!                    IF(k_bc1.le.0.or.k_bc1.gt.5) THEN ...
!        and similarly for logic after cmt
!          !    Set f(3,n) for not-a-knot
!        as required since k_bc*=6 and k_bc*=7 are NOT not-a-knot BCs.
!
!    (b) the BCs to fix 2nd derivative at end points did not work if that
!        2nd derivative were non-zero.  The reason is that in those cases
!        the off-diagonal matrix elements nearest the corners are not
!        symmetric; i.e. elem(1,2).ne.elem(2,1) and 
!        elem(n-1,n).ne.elem(n,n-1) where I use "elem" to refer to
!        the tridiagonal matrix elements.  The correct values for the
!        elements is:   elem(1,2)=0, elem(2,1)=x(2)-x(1)
!                       elem(n,n-1)=0, elem(n-1,n)=x(n)-x(n-1)
!        the old code in effect had these as all zeroes.  Since this
!        meant the wrong set of linear equations was solved, the
!        resulting spline had a discontinuity in its 1st derivative
!        at x(2) and x(n-1).  Fixed by introducing elem21 and elemnn1
!        to represent the non-symmetric lower-diagonal values.  Since
!        elem21 & elemnn1 are both on the lower diagonals, logic to 
!        use them occurs in the non-periodic forward elimination loop
!        only.  DMC 24 Feb 2004.
!***********************************************************************
      IMPLICIT NONE
!Declaration of input variables
      INTEGER        k_bc1,                   k_bcn,
     &               n
      double precision           x(*),                    wk(*),
     &               f(4,*)
!Declaration in local variables
      INTEGER        i,                       ib,
     &               imax,                    imin
      double precision           a1,                      an,
     &               b1,                      bn,
     &               q,                       t,
     &               hn
      double precision           elem21,                  elemnn1    ! (dmc)

!Set default range
      imin=1
      imax=n
!Set first and second BC values
      a1=0.0
      b1=0.0
      an=0.0
      bn=0.0
      IF(k_bc1.eq.1) THEN
        a1=f(2,1)
      ELSEIF(k_bc1.eq.2) THEN
        b1=f(3,1)
      ELSEIF(k_bc1.eq.5) THEN
        a1=(f(1,2)-f(1,1))/(x(2)-x(1))
      ELSEIF(k_bc1.eq.6) THEN
        b1=2.0*((f(1,3)-f(1,2))/(x(3)-x(2))
     &         -(f(1,2)-f(1,1))/(x(2)-x(1)))/(x(3)-x(1))
      ENDIF
      IF(k_bcn.eq.1) THEN
        an=f(2,n)
      ELSEIF(k_bcn.eq.2) THEN
        bn=f(3,n)
      ELSEIF(k_bcn.eq.5) THEN
        an=(f(1,n)-f(1,n-1))/(x(n)-x(n-1))
      ELSEIF(k_bcn.eq.6) THEN
        bn=2.0*((f(1,n)-f(1,n-1))/(x(n)-x(n-1))
     &         -(f(1,n-1)-f(1,n-2))/(x(n-1)-x(n-2)))/(x(n)-x(n-2))
      ENDIF
!Clear f(2:4,n)
      f(2,n)=0.0
      f(3,n)=0.0
      f(4,n)=0.0
      IF(n.eq.2) THEN
!Coefficients for n=2
        f(2,1)=(f(1,2)-f(1,1))/(x(2)-x(1))
        f(3,1)=0.0
        f(4,1)=0.0
        f(2,2)=f(2,1)
        f(3,2)=0.0
        f(4,2)=0.0
      ELSEIF(n.gt.2) THEN
!Set up tridiagonal system for A*y=B where y(i) are the second
!  derivatives at the knots
!  f(2,i) are the diagonal elements of A
!  f(4,i) are the off-diagonal elements of A
!  f(3,i) are the B elements/3, and will become c/3 upon solution
        f(4,1)=x(2)-x(1)
        f(3,2)=(f(1,2)-f(1,1))/f(4,1)
        DO i=2,n-1
          f(4,i)=x(i+1)-x(i)
          f(2,i)=2.0*(f(4,i-1)+f(4,i))
          f(3,i+1)=(f(1,i+1)-f(1,i))/f(4,i)
          f(3,i)=f(3,i+1)-f(3,i)
        ENDDO
!
!  (dmc): save now:
!
        elem21=f(4,1)
        elemnn1=f(4,n-1)
!
!  BC's
!    Left
        IF(k_bc1.eq.-1) THEN
          f(2,1)=2.0*(f(4,1)+f(4,n-1))
          f(3,1)=(f(1,2)-f(1,1))/f(4,1)-(f(1,n)-f(1,n-1))/f(4,n-1)
          wk(1)=f(4,n-1)
          DO i=2,n-3
            wk(i)=0.0
          ENDDO
          wk(n-2)=f(4,n-2)
          wk(n-1)=f(4,n-1)
        ELSEIF(k_bc1.eq.1.or.k_bc1.eq.3.or.k_bc1.eq.5) THEN
          f(2,1)=2.0*f(4,1)
          f(3,1)=(f(1,2)-f(1,1))/f(4,1)-a1
        ELSEIF(k_bc1.eq.2.or.k_bc1.eq.4.or.k_bc1.eq.6) THEN
          f(2,1)=2.0*f(4,1)
          f(3,1)=f(4,1)*b1/3.0
          f(4,1)=0.0  ! upper diagonal only (dmc: cf elem21)
        ELSEIF(k_bc1.eq.7) THEN
          f(2,1)=-f(4,1)
          f(3,1)=f(3,3)/(x(4)-x(2))-f(3,2)/(x(3)-x(1))
          f(3,1)=f(3,1)*f(4,1)**2/(x(4)-x(1))
        ELSE                             ! not a knot:
          imin=2
          f(2,2)=f(4,1)+2.0*f(4,2)
          f(3,2)=f(3,2)*f(4,2)/(f(4,1)+f(4,2))
        ENDIF
!    Right
        IF(k_bcn.eq.1.or.k_bcn.eq.3.or.k_bcn.eq.5) THEN
          f(2,n)=2.0*f(4,n-1)
          f(3,n)=-(f(1,n)-f(1,n-1))/f(4,n-1)+an
        ELSEIF(k_bcn.eq.2.or.k_bcn.eq.4.or.k_bcn.eq.6) THEN
          f(2,n)=2.0*f(4,n-1)
          f(3,n)=f(4,n-1)*bn/3.0
!xxx          f(4,n-1)=0.0  ! dmc: preserve f(4,n-1) for back subst.
          elemnn1=0.0  !  lower diaganol only (dmc)
        ELSEIF(k_bcn.eq.7) THEN
          f(2,n)=-f(4,n-1)
          f(3,n)=f(3,n-1)/(x(n)-x(n-2))-f(3,n-2)/(x(n-1)-x(n-3))
          f(3,n)=-f(3,n)*f(4,n-1)**2/(x(n)-x(n-3))
        ELSEIF(k_bc1.ne.-1) THEN         ! not a knot:
          imax=n-1
          f(2,n-1)=2.0*f(4,n-2)+f(4,n-1)
          f(3,n-1)=f(3,n-1)*f(4,n-2)/(f(4,n-1)+f(4,n-2))
        ENDIF
!  Limit solution for only three points in domain
        IF(n.eq.3) THEN
          f(3,1)=0.0
          f(3,n)=0.0
        ENDIF
        IF(k_bc1.eq.-1) THEN
!Solve system of equations for second derivatives at the knots
!  Periodic BC
!    Forward elimination
          DO i=2,n-2
            t=f(4,i-1)/f(2,i-1)
            f(2,i)=f(2,i)-t*f(4,i-1)
            f(3,i)=f(3,i)-t*f(3,i-1)
            wk(i)=wk(i)-t*wk(i-1)
            q=wk(n-1)/f(2,i-1)
            wk(n-1)=-q*f(4,i-1)
            f(2,n-1)=f(2,n-1)-q*wk(i-1)
            f(3,n-1)=f(3,n-1)-q*f(3,i-1)
          ENDDO
!    Correct the n-1 element
          wk(n-1)=wk(n-1)+f(4,n-2)
!    Complete the forward elimination
!    wk(n-1) and wk(n-2) are the off-diag elements of the lower corner
          t=wk(n-1)/f(2,n-2)
          f(2,n-1)=f(2,n-1)-t*wk(n-2)
          f(3,n-1)=f(3,n-1)-t*f(3,n-2)
!    Back substitution
          f(3,n-1)=f(3,n-1)/f(2,n-1)
          f(3,n-2)=(f(3,n-2)-wk(n-2)*f(3,n-1))/f(2,n-2)
          DO ib=3,n-1
            i=n-ib
            f(3,i)=(f(3,i)-f(4,i)*f(3,i+1)-wk(i)*f(3,n-1))/f(2,i)
          ENDDO
          f(3,n)=f(3,1)
        ELSE
!  Non-periodic BC
!    Forward elimination
!    For Not-A-Knot BC the off-diagonal end elements are not equal
          DO i=imin+1,imax
            IF((i.eq.n-1).and.(imax.eq.n-1)) THEN
              t=(f(4,i-1)-f(4,i))/f(2,i-1)
            ELSE
              if(i.eq.2) then
                 t=elem21/f(2,i-1)
              else if(i.eq.n) then
                 t=elemnn1/f(2,i-1)
              else
                 t=f(4,i-1)/f(2,i-1)
              endif
            ENDIF
            IF((i.eq.imin+1).and.(imin.eq.2)) THEN
              f(2,i)=f(2,i)-t*(f(4,i-1)-f(4,i-2))
            ELSE
              f(2,i)=f(2,i)-t*f(4,i-1)
            ENDIF
            f(3,i)=f(3,i)-t*f(3,i-1)
          ENDDO
!    Back substitution
          f(3,imax)=f(3,imax)/f(2,imax)
          DO ib=1,imax-imin
            i=imax-ib
            IF((i.eq.2).and.(imin.eq.2)) THEN
              f(3,i)=(f(3,i)-(f(4,i)-f(4,i-1))*f(3,i+1))/f(2,i)
            ELSE
              f(3,i)=(f(3,i)-f(4,i)*f(3,i+1))/f(2,i)
            ENDIF
          ENDDO
!    Reset d array to step size
          f(4,1)=x(2)-x(1)
          f(4,n-1)=x(n)-x(n-1)
!    Set f(3,1) for not-a-knot
          IF(k_bc1.le.0.or.k_bc1.gt.7) THEN
            f(3,1)=(f(3,2)*(f(4,1)+f(4,2))-f(3,3)*f(4,1))/f(4,2)
          ENDIF
!    Set f(3,n) for not-a-knot
          IF(k_bcn.le.0.or.k_bcn.gt.7) THEN
            f(3,n)=f(3,n-1)+(f(3,n-1)-f(3,n-2))*f(4,n-1)/f(4,n-2)
          ENDIF
        ENDIF
!f(3,i) is now the sigma(i) of the text and f(4,i) is the step size
!Compute polynomial coefficients
        DO i=1,n-1
          f(2,i)=
     >        (f(1,i+1)-f(1,i))/f(4,i)-f(4,i)*(f(3,i+1)+2.0*f(3,i))
          f(4,i)=(f(3,i+1)-f(3,i))/f(4,i)
          f(3,i)=6.0*f(3,i)
          f(4,i)=6.0*f(4,i)
        ENDDO
        IF(k_bc1.eq.-1) THEN
          f(2,n)=f(2,1)
          f(3,n)=f(3,1)
          f(4,n)=f(4,1)
        ELSE
           hn=x(n)-x(n-1)
           f(2,n)=f(2,n-1)+hn*(f(3,n-1)+0.5*hn*f(4,n-1))
           f(3,n)=f(3,n-1)+hn*f(4,n-1)
           f(4,n)=f(4,n-1)
           IF(k_bcn.eq.1.or.k_bcn.eq.3.or.k_bcn.eq.5) THEN
              f(2,n)=an
           ELSE IF(k_bcn.eq.2.or.k_bcn.eq.4.or.k_bcn.eq.6) THEN
              f(3,n)=bn
           ENDIF
        ENDIF
      ENDIF
      RETURN
      END SUBROUTINE V_SPLINE_db


      subroutine zonfind_db(x,nx,zxget,i)
      implicit none
c
      integer nx
      double precision x(nx),zxget
      integer i
      
      integer nxm, i1, i2, ii, ij
      double precision dx
c
c  find index i such that x(i).le.zxget.le.x(i+1)
c
c  x(1...nx) is strict increasing and x(1).le.zxget.le.x(nx)
c  (this is assumed to already have been checked -- no check here!)
c
      nxm=nx-1
      if((i.lt.1).or.(i.gt.nxm)) then
         i1=1
         i2=nx-1
         go to 10
      endif
c
      if(x(i).gt.zxget) then
c  look down
         dx=x(i+1)-x(i)
         if((x(i)-zxget).gt.4*dx) then
            i1=1
            i2=i-1
            go to 10
         else
            i2=i-1
            do ij=i2,1,-1
               if((x(ij).le.zxget).and.(zxget.le.x(ij+1))) then
                  i=ij
                  return
               endif
            enddo
            i=1
            return
         endif
      else if(x(i+1).lt.zxget) then
c  look up
         dx=x(i+1)-x(i)
         if((zxget-x(i+1)).gt.4*dx) then
            i1=i+1
            i2=nxm
            go to 10
         else
            i2=i+1
            do ij=i2,nxm
               if((x(ij).le.zxget).and.(zxget.le.x(ij+1))) then
                  i=ij
                  return
               endif
            enddo
            ij=nxm
            return
         endif
      else
c  already there...
         return
      endif
c
c---------------------------
c  binary search
c
 10   continue
c
      if(i1.eq.i2) then
c found by proc. of elimination
         i=i1
         return
      endif
c
      ii=(i1+i2)/2
c
      if(zxget.lt.x(ii)) then
         i2=ii-1
      else if(zxget.gt.x(ii+1)) then
         i1=ii+1
      else
c found
         i=ii
         return
      endif
c
      go to 10
c
      end subroutine zonfind_db
      
 
