
c        from PSPLINE by Doug McCune (version as of February, 2004)


c        PSPLINE Home Page:
c        http://w3.pppl.gov/NTCC/PSPLINE

c        Doug McCune, Princeton University
c                dmccune@pppl.gov

c
c  bcspeval -- eval bicubic spline function and/or derivatives
c
      subroutine bcspeval_db(xget,yget,iselect,fval,
     >                    x,nx,y,ny,ilinx,iliny,f,inf3,ier)
c
      implicit none
      integer iselect(6)
      integer ilinx,iliny,nx,ny,inf3,ier
c
      double precision xget,yget
      double precision fval(6)
      double precision x(nx),y(ny),f(4,4,inf3,ny)
c
c  modification -- dmc 11 Jan 1999 -- remove SAVE stmts;
C    break routine into these parts:
C
C    bcspevxy -- find grid cell of target pt.
C    bcspevfn -- evaluate function using output of bcpsevxy
C
C    in cases where multiple functions are defined on the same grid,
C    time can be saved by using bcspevxy once and then bcspevfn
C    multiple times.
c
c  input:
c     (xget,yget)   location where interpolated value is desired
c                   x(1).le.xget.le.x(nx) expected
c                   y(1).le.yget.le.y(ny) expected
c
c     iselect       select desired output
c
c                     iselect(1)=1 -- want function value (f) itself
c                     iselect(2)=1 -- want  df/dx
c                     iselect(3)=1 -- want  df/dy
c                     iselect(4)=1 -- want  d2f/dx2
c                     iselect(5)=1 -- want  d2f/dy2
c                     iselect(6)=1 -- want  d2f/dxdy
c
c              example:  iselect(1)=iselect(2)=iselect(3)=1
c                            f, df/dx, and df/dy all evaluated
c                        iselect(4)=iselect(5)=iselect(6)=0
c                            2nd derivatives not evaluated.
c
c                   the number of non zero values iselect(1:6)
c                   determines the number of outputs...
c                   see fval (output) description.
c
c  new dmc December 2005 -- access to higher derivatives (even if not
c  continuous-- but can only go up to 3rd derivatives on any one coordinate.
c     if iselect(1)=3 -- want 3rd derivatives
c          iselect(2)=1 for d3f/dx3
c          iselect(3)=1 for d3f/dx2dy
c          iselect(4)=1 for d3f/dxdy2
c          iselect(5)=1 for d3f/dy3
c               number of non-zero values iselect(2:5) gives no. of outputs
c     if iselect(1)=4 -- want 4th derivatives
c          iselect(2)=1 for d4f/dx3dy
c          iselect(3)=1 for d4f/dx2dy2
c          iselect(4)=1 for d4f/dxdy3
c               number of non-zero values iselect(2:4) gives no. of outputs
c     if iselect(1)=5 -- want 5th derivatives
c          iselect(2)=1 for d5f/dx3dy2
c          iselect(3)=1 for d5f/dx2dy3
c               number of non-zero values iselect(2:3) gives no. of outputs
c     if iselect(1)=6 -- want 6th derivatives
c          d6f/dx3dy3 -- one value is returned.
c
c     x(1...nx)     independent coordinate x, strict ascending
c     y(1...ny)     independent coordinate y, strict ascending
c
c     ilinx  --  =1: flag that x is linearly spaced (avoid search for speed)
c     iliny  --  =1: flag that y is linearly spaced (avoid search for speed)
c
c  **CAUTION** actual even spacing of x, y is NOT CHECKED HERE!
c
c
c     f             the function values (at grid points) and spline coefs
c
c  evaluation formula:  for point x btw x(i) and x(i+1), dx=x-x(i)
c                             and y btw y(j) and y(j+1), dy=y-y(j),
c
c      spline value =
c        f(1,1,i,j) + dx*f(2,1,i,j) + dx**2*f(3,1,i,j) + dx**3*f(4,1,i,j)
c   +dy*(f(1,2,i,j) + dx*f(2,2,i,j) + dx**2*f(3,2,i,j) + dx**3*f(4,2,i,j))
c   +d2*(f(1,3,i,j) + dx*f(2,3,i,j) + dx**2*f(3,3,i,j) + dx**3*f(4,3,i,j))
c   +d3*(f(1,4,i,j) + dx*f(2,4,i,j) + dx**2*f(3,4,i,j) + dx**3*f(4,4,i,j))
c
c      where d2=dy**2 and d3=dy**3.
c
c  output:
c      up to 6 elements of fval, ordered as follows:
c        fval(1)=function value or lowest order derivative requested
c        fval(2)=next order derivative
c             etc
c        the ordering is a subset of the sequence given under the "iselect"
c        description.
c
c      ier = 0 -- successful completion; = 1 -- an error occurred.
c
c-------------------------------------------------------------------
c  local
c
      integer :: i=0
      integer :: j=0
c
      double precision dx,dy
c
c--------------------------
c
      call bcspevxy_db(xget,yget,x,nx,y,ny,ilinx,iliny,
     >   i,j,dx,dy,ier)
      if(ier.ne.0) return
c
      call bcspevfn_db(iselect,1,1,fval,(/i/),(/j/),
     <   (/dx/),(/dy/),f,inf3,ny)
c
      return
      end subroutine bcspeval_db

c
c-------------------------------------------------------------------------
c  bcspevxy -- look up x-y zone
c
c  this is the "first part" of bcspeval, see comments, above.
c
      subroutine bcspevxy_db(xget,yget,x,nx,y,ny,ilinx,iliny,
     >   i,j,dx,dy,ier)
c
      integer nx,ny                     ! array dimensions
c
      double precision xget,yget                    ! target point
      double precision x(nx),y(ny)                  ! indep. coords.
c
      integer ilinx                     ! =1:  assume x evenly spaced
      integer iliny                     ! =1:  assume y evenly spaced
c
c  output of bcspevxy
c
      integer i,j                       ! index to cell containing target pt
      double precision dx,dy                        ! displacement of target pt w/in cell
                                        ! dx=x-x(i)  dy=y-y(j)
C
      integer ier                       ! return ier.ne.0 on error
c
c------------------------------------
c
      double precision zxget, zyget
      ier=0
c
c  range check
c
      zxget=xget
      zyget=yget
 
      if((xget.lt.x(1)).or.(xget.gt.x(nx))) then
         zxtol=4.0e-7*max(abs(x(1)),abs(x(nx)))
         if((xget.lt.x(1)-zxtol).or.(xget.gt.x(nx)+zxtol)) then
            ier=1
c            write(6,1001) xget,x(1),x(nx)
c 1001       format(' ?bcspeval:  xget=',1pe11.4,' out of range ',
c     >         1pe11.4,' to ',1pe11.4)
         else
c            if((xget.lt.x(1)-0.5*zxtol).or.
c     >         (xget.gt.x(nx)+0.5*zxtol))
c     >      write(6,1011) xget,x(1),x(nx)
c 1011       format(' %bcspeval:  xget=',1pe15.8,' beyond range ',
c     >         1pe15.8,' to ',1pe15.8,' (fixup applied)')
            if(xget.lt.x(1)) then
               zxget=x(1)
            else
               zxget=x(nx)
            endif
         endif
      endif
      if((yget.lt.y(1)).or.(yget.gt.y(ny))) then
         zytol=4.0e-7*max(abs(y(1)),abs(y(ny)))
         if((yget.lt.y(1)-zytol).or.(yget.gt.y(ny)+zytol)) then
            ier=1
c            write(6,1002) yget,y(1),y(ny)
c 1002       format(' ?bcspeval:  yget=',1pe11.4,' out of range ',
c     >         1pe11.4,' to ',1pe11.4)
         else
c         if((yget.lt.y(1)-0.5*zytol).or.(yget.gt.y(ny)+0.5*zytol))
c     >      write(6,1012) yget,y(1),y(ny)
c 1012       format(' %bcspeval:  yget=',1pe15.8,' beyond range ',
c     >         1pe15.8,' to ',1pe15.8,' (fixup applied)')
            if(yget.lt.y(1)) then
               zyget=y(1)
            else
               zyget=y(ny)
            endif
         endif
      endif
      if(ier.ne.0) return
c
c  now find interval in which target point lies..
c
      nxm=nx-1
      nym=ny-1
c
      if(ilinx.eq.1) then
         ii=1+nxm*(zxget-x(1))/(x(nx)-x(1))
         i=min(nxm, ii)
         if(zxget.lt.x(i)) then
            i=i-1
         else if(zxget.gt.x(i+1)) then
            i=i+1
         endif
      else
         if((1.le.i).and.(i.lt.nxm)) then
            if((x(i).le.zxget).and.(zxget.le.x(i+1))) then
               continue  ! already have the zone
            else
               call zonfind_db(x,nx,zxget,i)
            endif
         else
            call zonfind_db(x,nx,zxget,i)
         endif
      endif
c
      if(iliny.eq.1) then
         jj=1+nym*(zyget-y(1))/(y(ny)-y(1))
         j=min(nym, jj)
         if(zyget.lt.y(j)) then
            j=j-1
         else if(zyget.gt.y(j+1)) then
            j=j+1
         endif
      else
         if((1.le.j).and.(j.lt.nym)) then
            if((y(j).le.zyget).and.(zyget.le.y(j+1))) then
               continue  ! already have the zone
            else
               call zonfind_db(y,ny,zyget,j)
            endif
         else
            call zonfind_db(y,ny,zyget,j)
         endif
      endif
c
      dx=zxget-x(i)
      dy=zyget-y(j)
c
      return
      end subroutine bcspevxy_db

c------------------------------------------------------------------------
c  bcspevfn -- OK now evaluate the bicubic spline
c
      subroutine bcspevfn_db(ict,ivec,ivd,fval,iv,jv,dxv,dyv,f,inf3,ny)
c
c  input:
      integer ny
      integer ict(6)                    ! selector:
c        ict(1)=1 for f      (don't evaluate f if ict(1)=0)
c        ict(2)=1 for df/dx   ""
c        ict(3)=1 for df/dy   ""
c        ict(4)=1 for d2f/dx2
c        ict(5)=1 for d2f/dy2
c        ict(6)=1 for d2f/dxdy
c
c    note:  if ict(1)=-1, evaluate f,d2f/dx2,d2f/dy2,d4f/dx2dy2
c
c                   the number of non zero values ict(1:6)
c                   determines the number of outputs...
c
c  new dmc December 2005 -- access to higher derivatives (even if not
c  continuous-- but can only go up to 3rd derivatives on any one coordinate.
c     if ict(1)=3 -- want 3rd derivatives
c          ict(2)=1 for d3f/dx3
c          ict(3)=1 for d3f/dx2dy
c          ict(4)=1 for d3f/dxdy2
c          ict(5)=1 for d3f/dy3
c               number of non-zero values ict(2:5) gives no. of outputs
c     if ict(1)=4 -- want 4th derivatives
c          ict(2)=1 for d4f/dx3dy
c          ict(3)=1 for d4f/dx2dy2
c          ict(4)=1 for d4f/dxdy3
c               number of non-zero values ict(2:4) gives no. of outputs
c     if ict(1)=5 -- want 5th derivatives
c          ict(2)=1 for d5f/dx3dy2
c          ict(3)=1 for d5f/dx2dy3
c               number of non-zero values ict(2:3) gives no. of outputs
c     if ict(1)=6 -- want 6th derivatives
c          d6f/dx3dy3 -- one value is returned.
c
      integer ivec,ivd                  ! vector dimensioning
c
c    ivec-- number of vector pts (spline values to look up)
c    ivd -- 1st dimension of fval, .ge.ivec
c
c output:
      double precision fval(ivd,6)                 ! output array
c
c    v = index to element in vector;
c  fval(v,1) = first item requested by ict(...),
c  fval(v,2) = 2nd item requested,  ...etc...
c
c  input:
      integer iv(ivec),jv(ivec)         ! grid cell indices -- vectors
      double precision dxv(ivec),dyv(ivec)          ! displacements w/in cell -- vectors
c
      integer inf3                      ! 3rd dimension of f -- .ge. nx
      double precision f(4,4,inf3,ny)               ! bicubic fcn spline coeffs array
c
c  usage example:
c
c  1.  for each element (xx(v),yy(v)) in a vector of (x,y) pairs,
c    find the x and y zone indices and displacements with respect
c    to the "lower left corner" of the zone; store these in vectors
c    iv,jv and dxv,dyv.
c
c  2.  set ict(1)=0, ict(2)=1, ict(3)=1, the rest zero -- get only
c      the 1st derivatives.
c
c  3.  ivec is the length of the vector; ivd is the 1st dimension
c      of the array fval to receive the output
c
c      double precision fval(ivd,6)
c      double precision xv(ivd),yv(ivd)
c      integer iv(ivd),jv(ivd)
c      double precision dxv(ivd),dyv(ivd)
c      integer ict(6)
c
c      double precision fspline(4,4,nx,ny)  ! spline coeffs
c      data ict/0,1,1,0,0,0/    ! this call:  want 1st derivatives
c                               ! only ... these will be output to
c                               ! fval(*,1) fval(*,2)
c      ...
c      do iv=1,ivec
c        ...                    ! find indices and displacements
c      enddo
c      call bcspevfn(ict,ivec,ivd,fval,iv,jv,dxv,dyv,fspline,nx,ny)
c
c-------------------------------------------------------------------
c  local:
c
      integer v                         ! vector element index
c
c  OK can now do evaluations
c
      iaval=0  ! fval addressing
c
      if(ict(1).le.2) then
         if((ict(1).gt.0).or.(ict(1).eq.-1)) then
c  evaluate f
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >       f(1,1,i,j)+dy*(f(1,2,i,j)+dy*(f(1,3,i,j)+dy*f(1,4,i,j)))
     >  +dx*(f(2,1,i,j)+dy*(f(2,2,i,j)+dy*(f(2,3,i,j)+dy*f(2,4,i,j)))
     >  +dx*(f(3,1,i,j)+dy*(f(3,2,i,j)+dy*(f(3,3,i,j)+dy*f(3,4,i,j)))
     >  +dx*(f(4,1,i,j)+dy*(f(4,2,i,j)+dy*(f(4,3,i,j)+dy*f(4,4,i,j))))))
            enddo
         endif
c
         if((ict(2).gt.0).and.(ict(1).ne.-1)) then
c  evaluate df/dx
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >         f(2,1,i,j)+dy*(f(2,2,i,j)+dy*(f(2,3,i,j)+dy*f(2,4,i,j)))
     >       +2.0*dx*(
     >         f(3,1,i,j)+dy*(f(3,2,i,j)+dy*(f(3,3,i,j)+dy*f(3,4,i,j)))
     >       +1.5*dx*(
     >         f(4,1,i,j)+dy*(f(4,2,i,j)+dy*(f(4,3,i,j)+dy*f(4,4,i,j)))
     >              ))
            enddo
         endif
c
         if((ict(3).gt.0).and.(ict(1).ne.-1)) then
c  evaluate df/dy
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >         f(1,2,i,j)+dy*(2.0*f(1,3,i,j)+dy*3.0*f(1,4,i,j))
     >      +dx*(f(2,2,i,j)+dy*(2.0*f(2,3,i,j)+dy*3.0*f(2,4,i,j))
     >      +dx*(f(3,2,i,j)+dy*(2.0*f(3,3,i,j)+dy*3.0*f(3,4,i,j))
     >      +dx*(f(4,2,i,j)+dy*(2.0*f(4,3,i,j)+dy*3.0*f(4,4,i,j))
     >              )))
            enddo
         endif
c
         if((ict(4).gt.0).or.(ict(1).eq.-1)) then
c  evaluate d2f/dx2
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >        2.0*(
     >         f(3,1,i,j)+dy*(f(3,2,i,j)+dy*(f(3,3,i,j)+dy*f(3,4,i,j))))
     >       +6.0*dx*(
     >         f(4,1,i,j)+dy*(f(4,2,i,j)+dy*(f(4,3,i,j)+dy*f(4,4,i,j))))
            enddo
         endif
c
         if((ict(5).gt.0).or.(ict(1).eq.-1)) then
c  evaluate d2f/dy2
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              2.0*f(1,3,i,j)+6.0*dy*f(1,4,i,j)
     >              +dx*(2.0*f(2,3,i,j)+6.0*dy*f(2,4,i,j)
     >              +dx*(2.0*f(3,3,i,j)+6.0*dy*f(3,4,i,j)
     >              +dx*(2.0*f(4,3,i,j)+6.0*dy*f(4,4,i,j))))
            enddo
         endif
c
         if((ict(6).gt.0).and.(ict(1).ne.-1)) then
c  evaluate d2f/dxdy
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >            f(2,2,i,j)+dy*(2.0*f(2,3,i,j)+dy*3.0*f(2,4,i,j))
     > +2.*dx*(f(3,2,i,j)+dy*(2.0*f(3,3,i,j)+dy*3.0*f(3,4,i,j))
     >+1.5*dx*(f(4,2,i,j)+dy*(2.0*f(4,3,i,j)+dy*3.0*f(4,4,i,j))
     >              ))
            enddo
         endif
c
         if(ict(1).eq.-1) then
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              4.0*f(3,3,i,j)+12.0*dy*f(3,4,i,j)
     >              +dx*(12.0*f(4,3,i,j)+36.0*dy*f(4,4,i,j))
            enddo
         endif
c
c-----------------------------------
c  access to 3rd derivatives
c
      else if(ict(1).eq.3) then
         if(ict(2).eq.1) then
c  evaluate d3f/dx3 (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              +6.0*(
     >         f(4,1,i,j)+dy*(f(4,2,i,j)+dy*(f(4,3,i,j)+dy*f(4,4,i,j))))
            enddo
         endif
c
         if(ict(3).eq.1) then
c  evaluate d3f/dx2dy
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              2.0*(
     >           f(3,2,i,j)+dy*(2.0*f(3,3,i,j)+dy*3.0*f(3,4,i,j)))
     >              +6.0*dx*(
     >           f(4,2,i,j)+dy*(2.0*f(4,3,i,j)+dy*3.0*f(4,4,i,j)))
            enddo
         endif
c
         if(ict(4).eq.1) then
c  evaluate d3f/dxdy2
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              (2.0*f(2,3,i,j)+6.0*dy*f(2,4,i,j)
     >              +2.0*dx*(2.0*f(3,3,i,j)+6.0*dy*f(3,4,i,j)
     >              +1.5*dx*(2.0*f(4,3,i,j)+6.0*dy*f(4,4,i,j))
     >              ))
            enddo
         endif

         if(ict(5).eq.1) then
c  evaluate d3f/dy3 (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               fval(v,iaval)=6.0*(f(1,4,i,j)+
     >              dx*(f(2,4,i,j)+dx*(f(3,4,i,j)+dx*f(4,4,i,j))))
            enddo
         endif
c
c-----------------------------------
c  access to 4th derivatives
c
      else if(ict(1).eq.4) then
         if(ict(2).eq.1) then
c  evaluate d4f/dx3dy (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              +6.0*(
     >         f(4,2,i,j)+dy*2.0*(f(4,3,i,j)+dy*1.5*f(4,4,i,j)))
            enddo
         endif
c
         if(ict(3).eq.1) then
c  evaluate d4f/dx2dy2
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              4.0*f(3,3,i,j)+12.0*dy*f(3,4,i,j)
     >              +dx*(12.0*f(4,3,i,j)+36.0*dy*f(4,4,i,j))
            enddo
         endif
c
         if(ict(4).eq.1) then
c  evaluate d4f/dxdy3 (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               fval(v,iaval)=
     >              6.0*(f(2,4,i,j)
     >              +2.0*dx*(f(3,4,i,j)+1.5*dx*f(4,4,i,j)))
            enddo
         endif
c
c-----------------------------------
c  access to 5th derivatives
c
      else if(ict(1).eq.5) then
         if(ict(2).eq.1) then
c  evaluate d5f/dx3dy2 (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dy=dyv(v)
               fval(v,iaval)=
     >              +12.0*(f(4,3,i,j)+dy*3.0*f(4,4,i,j))
            enddo
         endif
c
         if(ict(3).eq.1) then
c  evaluate d5f/dx3dy2 (not continuous)
            iaval=iaval+1
            do v=1,ivec
               i=iv(v)
               j=jv(v)
               dx=dxv(v)
               fval(v,iaval)=
     >              12.0*(f(3,4,i,j)+dx*3.0*f(4,4,i,j))
            enddo
         endif
c
c-----------------------------------
c  access to 6th derivatives
c
      else if(ict(1).eq.6) then
c  evaluate d6f/dx3dy3 (not continuous)
         iaval=iaval+1
         do v=1,ivec
            i=iv(v)
            j=jv(v)
            fval(v,iaval)=
     >              36.0*f(4,4,i,j)
         enddo
      endif
c
      return
      end subroutine bcspevfn_db

c----------------------



c  bcspline -- dmc 30 May 1996
c
c  set up coefficients for bicubic spline with following BC's:
c  FULL BC CONTROL at all bdys
c
c  inhomogeneous explicit BCs -- this means setting of 1st or 2nd 
c  derivative at boundary to a non-zero value.
c
c  periodic, not-a-knot, zero derivative, and divided-difference based
c  BCs are "homogeneous"-- i.e. if splines s & t satisfy the BC then
c  the spline (c*s + t) formed as a linear combination of these two
c  splines, also satisfies the BC.
c
c  algorithm note -- handling of inhomogeneous explicit BC's while 
c  maintaining full C2 differentiability is delicate.  Basic method:  use 
c  a fully C2 method based on the "not-a-knot" BC, and then, correct to 
c  meet each user BC by calculating a C2 spline that is zero at all grid
c  points but satisfies a BC which is the difference btw the user spec
c  and the not-a-knot result; add the coeffs of this into the original.
c
c  for this more workspace is needed: nwk .ge. 4*inx*inth +5*max(inx,inth)
c
      subroutine bcspline_db(x,inx,th,inth,fspl,inf3,
     >                    ibcxmin,bcxmin,ibcxmax,bcxmax,
     >                    ibcthmin,bcthmin,ibcthmax,bcthmax,
     >                    wk,nwk,ilinx,ilinth,ier)
c
      implicit none
      integer inx, inth, inf3, nwk, ibcxmin, ibcxmax, ibcthmin, ibcthmax, ilinx,ilinth,ier
      double precision x(inx),th(inth),fspl(4,4,inf3,inth),wk(nwk)
      double precision bcxmin(inth),bcxmax(inth)
      double precision bcthmin(inx),bcthmax(inx)
c
c  input:
c    x(1...inx) -- abscissae, first dimension of data
c   th(1...inth) -- abscissae, second dimension of data  f(x,th)
c   fspl(1,1,1..inx,1..inth) -- function values
c   inf3 -- fspl dimensioning, inf3.ge.inx required.
c
c  boundary conditions input:
c   ibcxmin -- indicator for boundary condition at x(1):
c    bcxmin(...) -- boundary condition data
c     =-1 -- periodic boundary condition
c     =0 -- use "not a knot", bcxmin(...) ignored
c     =1 -- match slope, specified at x(1),th(ith) by bcxmin(ith)
c     =2 -- match 2nd derivative, specified at x(1),th(ith) by bcxmin(ith)
c     =3 -- boundary condition is slope=0 (df/dx=0) at x(1), all th(j)
c     =4 -- boundary condition is d2f/dx2=0 at x(1), all th(j)
c     =5 -- match 1st derivative df/dx to 1st divided difference
c     =6 -- match 2nd derivative d2f/dx2 to 2nd divided difference
c     =7 -- match 3rd derivative d3f/dx3 3rd divided difference
c           (for more detailed definition of BCs 5-7, see the
c           comments of subroutine mkspline)
c   NOTE bcxmin(...) referenced ONLY if ibcxmin=1 or ibcxmin=2
c
c   ibcxmax -- indicator for boundary condition at x(nx):
c    bcxmax(...) -- boundary condition data
c     (interpretation as with ibcxmin, bcxmin)
c   NOTE:  if ibcxmin=-1, ibcxmax is ignored! ...and the BC is periodic.
c
c   ibcthmin -- indicator for boundary condition at th(1):
c    bcthmin(...) -- boundary condition data
c     (interpretation as with ibcxmin, bcxmin)
c   ibcthmax -- indicator for boundary condition at th(inth):
c    bcthmax(...) -- boundary condition data
c     (interpretation as with ibcxmin, bcxmin)
c   NOTE:  if ibcthmin=-1, ibcthmax is ignored! ...and the BC is periodic.
c
c   NOTE the bcxmin,bcxmax,bcthmin,bcthmax arrays are only used if the
c     corresponding boundary condition flags are set to 1 or 2.
c     Carefully note the dimensioning of these arrays!
c
c  output:
c   fspl(*,*,1..inx,1..inth) -- bicubic spline coeffs (4x4)
c   ...fspl(1,1,*,*) is not replaced.
c
c   ilinx -- =1 on output if x(inx) pts are nearly evenly spaced (tol=1e-3)
c   ilinth-- =1 on output if th(inth) evenly spaced (tol=1e-3)
c
c   ier -- completion code, 0 for normal
c
c  workspace:
c   wk -- must be at least 5*max(inx,inth) large
c                          5*max(inx,inth) + 4*inx*inth large
c                          if explicit non-zero d/dth or d2/dth2 BC's
c                          are supplied.
c  nwk -- size of workspace of workspace provided
c
c---------------------------------
c  in what follows, "f" is an abbreviation for "fspl"...
c
c  compute bicubic spline of 2d function, given values at the grid
c  grid crossing points, f(1,1,i,j)=f(x(i),th(j)).
c
c  on evaluation:  for point x btw x(i) and x(i+1), dx=x-x(i)
c                       and th btw th(j) and th(j+1), dt=th-th(j),
c
c      spline =
c        f(1,1,i,j) + dx*f(2,1,i,j) + dx**2*f(3,1,i,j) + dx**3*f(4,1,i,j)
c   +dt*(f(1,2,i,j) + dx*f(2,2,i,j) + dx**2*f(3,2,i,j) + dx**3*f(4,2,i,j))
c   +d2*(f(1,3,i,j) + dx*f(2,3,i,j) + dx**2*f(3,3,i,j) + dx**3*f(4,3,i,j))
c   +d3*(f(1,4,i,j) + dx*f(2,4,i,j) + dx**2*f(3,4,i,j) + dx**3*f(4,4,i,j))
c
c      where d2=dt**2 and d3=dt**3.
c
      integer iselect1(10)
      integer iselect2(10)
      
      
      integer iflg2, ix, itest, ierx, inxo, ith, jth, ii, iadr, ia5w, iaspl, ierth, intho, ic
      integer ibcthmina, ibcthmaxa, iasc, iinc, iawk, jx
      double precision xo2, xo6, zcur, zdiff1, zhxn, zhth, zdiff2
      double precision fval(6)
c
c---------------------------------
c
c  see if 2nd pass is needed due to "non-linear" d/dth bdy cond.
c
      iflg2=0
      if(ibcthmin.ne.-1) then
         if((ibcthmin.eq.1).or.(ibcthmin.eq.2)) then
            do ix=1,inx
               if (bcthmin(ix).ne.0.0) iflg2=1
            enddo
         endif
         if((ibcthmax.eq.1).or.(ibcthmax.eq.2)) then
            do ix=1,inx
               if (bcthmax(ix).ne.0.0) iflg2=1
            enddo
         endif
      endif
c
      ier=0
      itest=5*max(inx,inth)
      if(iflg2.eq.1) then
         itest=itest +4*inx*inth
      endif
c
      if(nwk.lt.itest) then
         write(6,9901) nwk,itest
 9901    format(' ?bcspline:  workspace too small:'/
     >          '  user supplied:  nwk=',i6,'; need at least:  ',i6/
     >          '  nwk=4*nx*ny +5*max(nx,ny) will work for any user'/
     >          '  choice of bdy conditions.')
         ier=1
      endif
      if(inx.lt.4) then
         write(6,'('' ?bcspline:  at least 4 x points required.'')')
         ier=1
      endif
      if(inth.lt.4) then
         write(6,'('' ?bcspline:  need at least 4 theta points.'')')
         ier=1
      endif
c
      call ibc_ck_db(ibcxmin,'bcspline','xmin',-1,7,ier)
      if(ibcxmin.ge.0) call ibc_ck_db(ibcxmax,'bcspline','xmax',0,7,ier)
      call ibc_ck_db(ibcthmin,'bcspline','thmin',-1,7,ier)
      if(ibcthmin.ge.0) call ibc_ck_db(ibcthmax,'bcspline','thmax',0,7,ier)
c
c  check ilinx & x vector
c
      call splinck_db(x,inx,ilinx,1.0d-3,ierx)
      if(ierx.ne.0) ier=2
c
      if(ier.eq.2) then
         write(6,'('' ?bcspline:  x axis not strict ascending'')')
      endif
c
c  check ilinth & th vector
c
      call splinck_db(th,inth,ilinth,1.0d-3,ierth)
      if(ierth.ne.0) ier=3
c
c      if(ier.eq.3) then
c         write(6,'('' ?bcspline:  th axis not strict ascending'')')
c      endif
c
      if(ier.ne.0) return
c
c------------------------------------
c
      xo2=0.5
      xo6=1.0/6.0
c
c  spline in x direction
c
      inxo=4*(inx-1)
      do ith=1,inth
c
c  copy the function in
c
         do ix=1,inx
            wk(4*(ix-1)+1)=fspl(1,1,ix,ith)
         enddo
c
         if(ibcxmin.eq.1) then
            wk(2)=bcxmin(ith)
         else if(ibcxmin.eq.2) then
            wk(3)=bcxmin(ith)
         endif
c
         if(ibcxmax.eq.1) then
            wk(inxo+2)=bcxmax(ith)
         else if(ibcxmax.eq.2) then
            wk(inxo+3)=bcxmax(ith)
         endif
c
c  use Wayne's routine
c
         call v_spline_db(ibcxmin,ibcxmax,inx,x,wk,wk(4*inx+1))
c
c  copy the coefficients out
c
         do ix=1,inx
            fspl(2,1,ix,ith)=wk(4*(ix-1)+2)
            fspl(3,1,ix,ith)=wk(4*(ix-1)+3)*xo2
            fspl(4,1,ix,ith)=wk(4*(ix-1)+4)*xo6
         enddo
c
      enddo
c
c-----------------------------------
c
c  spline in theta direction
c
      intho=4*(inth-1)
      do ix=1,inx
c
c  spline each x coeff
c
         do ic=1,4
c
c  copy ordinates in
c
            do ith=1,inth
               wk(4*(ith-1)+1)=fspl(ic,1,ix,ith)
            enddo
c
c  first pass:  use a linear BC -- if flag indicates BC correction
c  will be needed, it will be done later
c
            wk(2)=0.0
            wk(3)=0.0
            wk(intho+2)=0.0
            wk(intho+3)=0.0
c
            ibcthmina=ibcthmin
            ibcthmaxa=ibcthmax
            if(iflg2.eq.1) then
               if((ibcthmin.eq.1).or.(ibcthmin.eq.2)) ibcthmina=0
               if((ibcthmax.eq.1).or.(ibcthmax.eq.2)) ibcthmaxa=0
            endif
c
            call v_spline_db(ibcthmina,ibcthmaxa,inth,th,wk,wk(4*inth+1))
c
c  copy coeffs out
c
            do ith=1,inth
               fspl(ic,2,ix,ith)=wk(4*(ith-1)+2)
               fspl(ic,3,ix,ith)=wk(4*(ith-1)+3)*xo2
               fspl(ic,4,ix,ith)=wk(4*(ith-1)+4)*xo6
            enddo
c
         enddo
c
      enddo
c
c  now make correction for user BC's if needed
c
      if(iflg2.eq.1) then
c
         iasc=1                         ! wk addr for correction splines
         iinc=4*inth                    ! spacing btw correction splines
         iawk=iasc+4*inth*inx
c
c  last grid zone widths
c
         zhxn=x(inx)-x(inx-1)
         jx=inx-1
         zhth=th(inth)-th(inth-1)
         jth=inth-1
c
         do ii=1,10
            iselect1(ii)=0
            iselect2(ii)=0
         enddo
         if(ibcthmin.eq.1) iselect1(3)=1
         if(ibcthmin.eq.2) iselect1(5)=1
         if(ibcthmax.eq.1) iselect2(3)=1
         if(ibcthmax.eq.2) iselect2(5)=1
c
c  loop over BC's
c
         do ix=1,inx
c
c  (a) d/dth @ th(1) difference btw current BC and user request
c
            if(ibcthmin.eq.1) then
               if(ix.lt.inx) then
                  zcur=fspl(1,2,ix,1)   ! 1st deriv.
               else
                  zcur=fspl(1,2,jx,1)+zhxn*(fspl(2,2,jx,1)+zhxn*
     >               (fspl(3,2,jx,1)+zhxn*fspl(4,2,jx,1)))
               endif
               zdiff1=bcthmin(ix)-zcur
            else if(ibcthmin.eq.2) then
               if(ix.lt.inx) then
                  zcur=2.0*fspl(1,3,ix,1) ! 2nd deriv.
               else
                  zcur=2.0*(fspl(1,3,jx,1)+zhxn*(fspl(2,3,jx,1)+zhxn*
     >               (fspl(3,3,jx,1)+zhxn*fspl(4,3,jx,1))))
               endif
               zdiff1=bcthmin(ix)-zcur
            else
               zdiff1=0.0
            endif
c
c  (b) d/dth @ th(inth) difference btw current BC and user request
c
            if(ibcthmax.eq.1) then
               if(ix.lt.inx) then
c  1st deriv.
                  zcur=fspl(1,2,ix,jth)+zhth*(2.0*fspl(1,3,ix,jth)+
     >               zhth*3.0*fspl(1,4,ix,jth))
               else
                  call bcspeval_db(x(inx),th(inth),iselect2,fval,
     >               x,inx,th,inth,ilinx,ilinth,fspl,inf3,ier)
                  zcur=fval(1)
                  if(ier.ne.0) return
               endif
               zdiff2=bcthmax(ix)-zcur
            else if(ibcthmax.eq.2) then
               if(ix.lt.inx) then
c  2nd deriv.
                  zcur=2.0*fspl(1,3,ix,jth)+
     >               6.0*zhth*fspl(1,4,ix,jth)
               else
                  call bcspeval_db(x(inx),th(inth),iselect2,fval,
     >               x,inx,th,inth,ilinx,ilinth,fspl,inf3,ier)
                  zcur=fval(1)
                  if(ier.ne.0) return
               endif
               zdiff2=bcthmax(ix)-zcur
            else
               zdiff2=0.0
            endif
c
c  ok compute the theta spline with BC's to span the difference(s)
c  these theta "correction splines" are zero at all the grid points
c  but have at least one non-zero 1st or 2nd derivative BC
c
            iadr=iasc+(ix-1)*iinc
            do ith=1,inth
               wk(iadr+4*(ith-1))=0.0
            enddo
c
            wk(iadr+1)=0.0
            wk(iadr+2)=0.0
            wk(iadr+intho+1)=0.0
            wk(iadr+intho+2)=0.0
c
            if(ibcthmin.eq.1) then
               wk(iadr+1)=zdiff1
            else if(ibcthmin.eq.2) then
               wk(iadr+2)=zdiff1
            endif
c
            if(ibcthmax.eq.1) then
               wk(iadr+intho+1)=zdiff2
            else if(ibcthmax.eq.2) then
               wk(iadr+intho+2)=zdiff2
            endif
c
            call v_spline_db(ibcthmin,ibcthmax,inth,th,wk(iadr),wk(iawk))
         enddo
c
c  add in results to main array -- th spline coef corrections
c
         do ix=1,inx
            iadr=iasc+(ix-1)*iinc
            do ith=1,inth-1
               wk(iadr+4*(ith-1)+2)=wk(iadr+4*(ith-1)+2)*xo2
               wk(iadr+4*(ith-1)+3)=wk(iadr+4*(ith-1)+3)*xo6
               if(ix.lt.inx) then
                  fspl(1,2,ix,ith)=fspl(1,2,ix,ith)+wk(iadr+4*(ith-1)+1)
                  fspl(1,3,ix,ith)=fspl(1,3,ix,ith)+wk(iadr+4*(ith-1)+2)
                  fspl(1,4,ix,ith)=fspl(1,4,ix,ith)+wk(iadr+4*(ith-1)+3)
               endif
            enddo
         enddo
c
c  compute the x splines of the th spline correction coeffs
c
         ia5w=iawk+4*inx
c
         do ith=1,inth-1
            do ic=2,4
               do ix=1,inx
                  iaspl=iasc+iinc*(ix-1)
                  wk(iawk+4*(ix-1))=wk(iaspl+4*(ith-1)+(ic-1))
               enddo
c
c  use zero BCs for this correction spline
c
               wk(iawk+1)=0.0
               wk(iawk+2)=0.0
               wk(iawk+inxo+1)=0.0
               wk(iawk+inxo+2)=0.0
c
c  periodic spline of correction spline higher coeffs (1st coeffs are
c  all zero by defn of the correction spline
c
               call v_spline_db(ibcxmin,ibcxmax,inx,x,wk(iawk),wk(ia5w))
c
               do ix=1,inx-1
                  fspl(2,ic,ix,ith)=fspl(2,ic,ix,ith)+
     >               wk(iawk+4*(ix-1)+1)
                  fspl(3,ic,ix,ith)=fspl(3,ic,ix,ith)+
     >               wk(iawk+4*(ix-1)+2)*xo2
                  fspl(4,ic,ix,ith)=fspl(4,ic,ix,ith)+
     >               wk(iawk+4*(ix-1)+3)*xo6
               enddo
c
            enddo
         enddo                          ! ith
c
      endif                             ! BC correction needs test
c
      return
      end subroutine bcspline_db
