! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module interp_1d_mp ! high accuracy monotonicity preserving algorithms
      
      use alert_lib

      implicit none
      private
      public :: m3, m3_on_uniform_grid
      
      contains
      
      
      subroutine m3(x, nx, f, which, slope_only, nwork, work, ierr)  
         ! make piecewise monotonic cubic interpolant
         use interp_1d_def
         use interp_1d_misc
         use interp_1d_pm, only: mk_pmlinear, mk_pmquad
         integer, intent(in) :: nx       ! length of x vector
         double precision, intent(in)    :: x(nx)    ! junction points, strictly monotonic
         double precision, intent(inout) :: f(:,:) ! (4,nx)  ! data & interpolation coefficients
         integer, intent(in) :: which
         logical, intent(in) :: slope_only
         integer, intent(in) :: nwork
         double precision, intent(inout), target :: work(:,:) ! (nx, nwork)
         integer, intent(out) :: ierr
         
         double precision, dimension(:), pointer :: h, s_mid, s, d, d_mid, e_mid, hd_mid, 
     >         spL, spR, t, tmax, tmp, tmp1, tmp2
         double precision, parameter :: tiny = 1e-20
         integer :: i
         character (len=256) :: message

         ierr = 0
         
         if (nx < 2) then
            return
         end if
         
         if (nx == 2) then
            call mk_pmlinear(x, f, slope_only, nwork, work, ierr)
            return
         end if
         
         if (nx == 3) then
            call mk_pmquad(x, f, slope_only, nwork, work, ierr)
            return
         end if
         
         if (size(f,dim=1) /= 4) then
            ierr = -1
            call alert(ierr, 'size(f,dim=1) /= 4 for interpolation')
            return
         end if
         
         if (size(f,dim=2) < nx) then
            ierr = -1
            call alert(ierr, 'size(f,dim=2) < nx for interpolation')
            return
         end if
         
         if (size(work,dim=1) < nx) then
            ierr = -1
            call alert(ierr, 'size(work,dim=1) < nx for interpolation')
            return
         end if
         
         if (size(work,dim=2) < nwork) then
            ierr = -1
            call alert(ierr, 'size(work,dim=2) < nwork for interpolation')
            return
         end if
         
         if (nwork < mp_work_size) then
            ierr = -1
            call alert(ierr, 'nwork too small for m3')
            return
         end if
         
         i = 1
         h => work(1:nx, i); i = i+1
         s_mid => work(1:nx, i); i = i+1
         s => work(1:nx, i); i = i+1
         d => work(1:nx, i); i = i+1
         d_mid => work(1:nx, i); i = i+1
         e_mid => work(1:nx, i); i = i+1
         hd_mid => work(1:nx, i); i = i+1
         spL => work(1:nx, i); i = i+1
         spR => work(1:nx, i); i = i+1
         t => work(1:nx, i); i = i+1
         tmax => work(1:nx, i); i = i+1
         tmp => work(1:nx, i); i = i+1
         tmp1 => work(1:nx, i); i = i+1
         tmp2 => work(1:nx, i); i = i+1

         ! grid spacing
         h(1:nx-1) = x(2:nx) - x(1:nx-1)
         do i = 1, nx-1
            if (h(i) == 0) then
               write(message, '(a,x,2i5,x,a)') 'same x values at', i, i+1, 'for make interpolant'
               ierr = -1
               call alert(ierr, message)
               return
            end if
         end do
                  
         ! divided differences
         s_mid(1:nx-1) = (f(1, 2:nx) - f(1, 1:nx-1)) / h(1:nx-1) ! eqn 2.1      
         d(2:nx-1) = (s_mid(2:nx-1) - s_mid(1:nx-2)) / (x(3:nx) - x(1:nx-2)) ! eqn 3.1
         
         ! need to extend d to full range. simplest way is just to copy from neighbor
         d(1) = d(2)
         d(nx) = d(nx-1)
         
         ! d_mid eqn(3.4) -- modified according to eqn (2.27) of Suresh & Huynh, 1997
         forall (i=1:nx-1)
            tmp1(i) = 4*d(i+1) - d(i)
            tmp2(i) = 4*d(i) - d(i+1)
         end forall
         call minmod4(d_mid(1:nx-1), nx-1, d(2:nx), d(1:nx-1), tmp1(1:nx-1), tmp2(1:nx-1))
         
         hd_mid(1:nx-1) = h(1:nx-1)*d_mid(1:nx-1)
         
         ! spL(i) = p'(i-1/2)(xi) = smid(i-1) + h(i-1)*d_mid(i-1) from Theorem 1
         spL(2:nx) = s_mid(1:nx-1) + hd_mid(1:nx-1)      
         
         ! spR(i) = p'(i+1/2)(xi) = smid(i) - h(i)*d_mid(i) from Theorem 1
         spR(1:nx-1) = s_mid(1:nx-1) - hd_mid(1:nx-1) 
         
         call minmod(s(2:nx-1), nx-2, s_mid(1:nx-2), s_mid(2:nx-1)) ! eqn (2.8)
         call minmod(t(2:nx-1), nx-2, spL(2:nx-1), spR(2:nx-1))
         
         if (which == average) then
         
            f(2, 2:nx-1) = sign(1d0, t(2:nx-1))*
     >            min((dabs(spL(2:nx-1)+spR(2:nx-1)))/2d0, 
     >               max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1))))   

         else if (which == quartic) then
         
            e_mid(2:nx-2) = (d(3:nx-1) - d(2:nx-2)) / (x(4:nx) - x(1:nx-3))  ! eqn 4.1
            ! eqn 3.5b for p'(i); eqn 4.20 for quadratic f'(i)
            f(2, 3:nx-2) = s_mid(3:nx-2) - 
     >         h(3:nx-2) * (d(3:nx-2) + h(2:nx-3)*
     >            (e_mid(2:nx-3)*(x(5:nx)-x(3:nx-2)) 
     >               + e_mid(3:nx-2)*(x(3:nx-2)-x(1:nx-4)))/(x(5:nx)-x(1:nx-4)))
            ! finish off ends with average
            f(2, 2) = sign(1d0, t(2))*
     >            min((dabs(spL(2)+spR(2)))/2d0, max(3*dabs(s(2)), 1.5d0*dabs(t(2)))) 
            f(2, nx-1) = sign(1d0, t(nx-1))*
     >            min((dabs(spL(nx-1)+spR(nx-1)))/2d0, 
     >            max(3*dabs(s(nx-1)), 1.5d0*dabs(t(nx-1))))
            tmp1(2:nx-1) = f(2, 2:nx-1)
            tmp2(2:nx-1) = tmp1(2:nx-1)
            call median(tmp1(2:nx-1), nx-2, tmp2(2:nx-1), spL(2:nx-1), spR(2:nx-1))
            f(2, 2:nx-1) = tmp1(2:nx-1)
            tmax(2:nx-1) = sign(1d0, t(2:nx-1))*
     >            max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1)))
            tmp1(2:nx-1) = f(2, 2:nx-1)
            call minmod(tmp2(2:nx-1), nx-2, tmp1(2:nx-1), tmax(2:nx-1))
            f(2, 2:nx-1) = tmp2(2:nx-1)
         else !if (which == super_bee) then
         
            f(2, 2:nx-1) = sign(1d0, t(2:nx-1))*
     >            min(max(dabs(spL(2:nx-1)), dabs(spR(2:nx-1))), 
     >               max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1))))

         end if
         
         ! slope at i=1
         !f(2, 1) = minmod1(spR(1), 3*s_mid(1)) ! eqn (5.2)
         f(2, 1) = minmod1(s_mid(1), s_mid(2)) ! stablize the ends
         

         ! slope at i=nx
         !f(2, nx) = minmod1(spL(nx), 3*s_mid(nx-1)) ! eqn (5.2)
         f(2, nx) = minmod1(s_mid(nx-2), s_mid(nx-1)) ! stablize the ends
         
         if (slope_only) return

         ! 2nd and 3rd derivatives
         f(3, 1:nx-1) = (3*s_mid(1:nx-1) - 2*f(2, 1:nx-1) - f(2, 2:nx)) / h(1:nx-1)       
         f(4, 1:nx-1) = (f(2, 1:nx-1) + f(2, 2:nx) - 2*s_mid(1:nx-1)) / h(1:nx-1)**2
         
         f(3, nx) = (3*f(1, nx-1) - 3*f(1, nx) + (f(2, nx-1) + 2*f(2, nx)) * h(nx-1)) / h(nx-1)**2
         f(4, nx) = (-2*f(1, nx-1) + 2*f(1, nx) - (f(2, nx-1) + f(2, nx))*h(nx-1)) / h(nx-1)**3
      
      end subroutine m3

      
      subroutine m3_on_uniform_grid(dx, nx, f, which, slope_only, nwork, work, ierr)  
         use interp_1d_def
         use interp_1d_misc
         use interp_1d_pm, only: mk_pmlinear, mk_pmquad
         ! make piecewise monotonic cubic interpolant
         double precision, intent(in) :: dx ! the grid spacing
         integer, intent(in) :: nx       ! length of x vector
         double precision, intent(inout) :: f(:,:) ! (4, nx)  ! data & interpolation coefficients
         integer, intent(in) :: which
         logical, intent(in) :: slope_only
         integer, intent(in) :: nwork
         double precision, intent(inout), target :: work(:,:) ! (nx, nwork)
         integer, intent(out) :: ierr
         
         double precision, dimension(:), pointer :: s_mid, s, d, d_mid, e_mid, spL, spR, t, tmax, 
     >      tmp, tmp1, tmp2         
         double precision, parameter :: tiny = 1e-20
         double precision :: x(3)
         integer :: i

         ierr = 0
         
         if (nx < 2) then
            return
         end if
         
         if (nx == 2) then
            x(1:2) = (/ 0d0, dx /)
            call mk_pmlinear(x, f, slope_only, nwork, work, ierr)
            return
         end if
         
         if (nx == 3) then
            x(1:3) = (/ 0d0, dx, 2*dx /)
            call mk_pmquad(x, f, slope_only, nwork, work, ierr)
            return
         end if
         
         if (size(f,dim=1) /= 4) then
            ierr = -1
            call alert(ierr, 'size(f,dim=1) /= 4 for interpolation')
            return
         end if
         
         if (size(f,dim=2) < nx) then
            ierr = -1
            call alert(ierr, 'size(f,dim=2) < nx for interpolation')
            return
         end if
         
         if (size(work,dim=1) < nx) then
            ierr = -1
            call alert(ierr, 'size(work,dim=1) < nx for interpolation')
            return
         end if
         
         if (size(work,dim=2) < nwork) then
            ierr = -1
            call alert(ierr, 'size(work,dim=2) < nwork for interpolation')
            return
         end if

         if (dx == 0) then
            ierr = -1
            call alert(ierr, 'dx cannot be 0 for make interpolant')
            return
         end if
         
         if (nwork < mp_work_size) then
            ierr = -1
            call alert(ierr, 'nwork too small for m3')
            return
         end if
         
         i = 1
         s_mid => work(1:nx, i); i = i+1
         s => work(1:nx, i); i = i+1
         d => work(1:nx, i); i = i+1
         d_mid => work(1:nx, i); i = i+1
         e_mid => work(1:nx, i); i = i+1
         d_mid => work(1:nx, i); i = i+1
         spL => work(1:nx, i); i = i+1
         spR => work(1:nx, i); i = i+1
         t => work(1:nx, i); i = i+1
         tmax => work(1:nx, i); i = i+1
         tmp => work(1:nx, i); i = i+1
         tmp1 => work(1:nx, i); i = i+1
         tmp2 => work(1:nx, i); i = i+1
         
         ! divided differences
         s_mid(1:nx-1) = (f(1, 2:nx) - f(1, 1:nx-1)) / dx ! eqn 2.1    
         d(2:nx-1) = (s_mid(2:nx-1) - s_mid(1:nx-2)) / (2*dx) ! eqn 3.1
         
         ! need to extend d to full range. simplest way is just to copy from neighbor
         d(1) = d(2)
         d(nx) = d(nx-1)
         
         ! d_mid eqn(3.4) -- modified according to eqn (2.27) of Suresh & Huynh, 1997
         forall (i=1:nx-1)
            tmp1(i) = 4*d(i+1) - d(i)
            tmp2(i) = 4*d(i) - d(i+1)
         end forall
         call minmod4(d_mid(1:nx-1), nx-1, d(2:nx), d(1:nx-1), tmp1(1:nx-1), tmp2(1:nx-1))
         
         ! spL(i) = p'(i-1/2)(xi) = smid(i-1) + h(i-1)*d_mid(i-1) from Theorem 1
         spL(2:nx) = s_mid(1:nx-1) + dx*d_mid(1:nx-1)    
         
         ! spR(i) = p'(i+1/2)(xi) = smid(i) - h(i)*d_mid(i) from Theorem 1
         spR(1:nx-1) = s_mid(1:nx-1) - dx*d_mid(1:nx-1)        
         
         call minmod(s(2:nx-1), nx-2, s_mid(1:nx-2), s_mid(2:nx-1)) ! eqn (2.8)
         call minmod(t(2:nx-1), nx-2, spL(2:nx-1), spR(2:nx-1))
         
         if (which == average) then
         
            f(2, 2:nx-1) = sign(1d0, t(2:nx-1))*
     >            min((dabs(spL(2:nx-1)+spR(2:nx-1)))/2d0, 
     >               max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1))))
   
         else if (which == quartic) then
         
            e_mid(2:nx-2) = (d(3:nx-1) - d(2:nx-2)) / (3*dx)  ! eqn 4.1
            ! eqn 3.5b for p'(i); eqn 4.20 for quadratic f'(i)
            f(2, 3:nx-2) = s_mid(3:nx-2) - 
     >         dx * (d(3:nx-2) + dx*
     >            (e_mid(2:nx-3)*(2*dx) 
     >               + e_mid(3:nx-2)*(2*dx))/(4*dx))
            ! finish off ends with average
            f(2, 2) = sign(1d0, t(2))*
     >            min((dabs(spL(2)+spR(2)))/2d0, max(3*dabs(s(2)), 1.5d0*dabs(t(2)))) 
            f(2, nx-1) = sign(1d0, t(nx-1))*
     >            min((dabs(spL(nx-1)+spR(nx-1)))/2d0, 
     >            max(3*dabs(s(nx-1)), 1.5d0*dabs(t(nx-1))))
            tmp1(2:nx-1) = f(2, 2:nx-1)
            tmp2(2:nx-1) = tmp1(2:nx-1)
            call median(tmp1(2:nx-1), nx-2, tmp2(2:nx-1), spL(2:nx-1), spR(2:nx-1))
            f(2, 2:nx-1) = tmp1(2:nx-1)
            tmax(2:nx-1) = sign(1d0, t(2:nx-1))*
     >            max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1)))
            tmp1(2:nx-1) = f(2, 2:nx-1)
            call minmod(tmp2(2:nx-1), nx-2, tmp1(2:nx-1), tmax(2:nx-1))
            f(2, 2:nx-1) = tmp2(2:nx-1)
            
         else !if (which == super_bee) then
         
            f(2, 2:nx-1) = sign(1d0, t(2:nx-1))*
     >            min(max(dabs(spL(2:nx-1)), dabs(spR(2:nx-1))), 
     >               max(3*dabs(s(2:nx-1)), 1.5d0*dabs(t(2:nx-1))))

         end if
         
         ! slope at i=1
         !f(2, 1) = minmod1(spR(1), 3*s_mid(1)) ! eqn (5.2)
         f(2, 1) = minmod1(s_mid(1), s_mid(2)) ! stablize the ends
         
         ! slope at i=nx
         !f(2, nx) = minmod1(spL(nx), 3*s_mid(nx-1)) ! eqn (5.2)
         f(2, nx) = minmod1(s_mid(nx-2), s_mid(nx-1)) ! stablize the ends
         
         if (slope_only) return
         
         ! 2nd and 3rd derivatives
         f(3, 1:nx-1) = (3*s_mid(1:nx-1) - 2*f(2, 1:nx-1) - f(2, 2:nx)) / dx        
         f(4, 1:nx-1) = (f(2, 1:nx-1) + f(2, 2:nx) - 2*s_mid(1:nx-1)) / dx**2
         
         f(3, nx) = (3*f(1, nx-1) - 3*f(1, nx) + (f(2, nx-1) + 2*f(2, nx)) * dx) / dx**2
         f(4, nx) = (-2*f(1, nx-1) + 2*f(1, nx) - (f(2, nx-1) + f(2, nx))*dx) / dx**3
      
      end subroutine m3_on_uniform_grid


      end module interp_1d_mp
