! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module interp_1d_misc
      
      use alert_lib

      implicit none
      
      contains
      
      
      subroutine do_integrate_values(init_x, nx, f, nv, x, vals, ierr)
         use alert_lib
         double precision, intent(in) :: init_x(nx) ! junction points, strictly monotonic
         integer, intent(in) :: nx ! length of init_x vector
         double precision, intent(in) :: f(:,:) ! (4, nx)  ! data & interpolation coefficients
         integer, intent(in) :: nv ! length of new x vector and vals vector
         double precision, intent(in) :: x(nv)
            ! strictly monotonic in same way as init_x
         double precision, intent(out) :: vals(:) ! (nv)
            ! for i > 1, vals(i) = integral of interpolating poly from x(i-1) to x(i)
            ! vals(1) = 0
         integer, intent(out) :: ierr ! 0 means aok
   
         integer :: k_old, k_new
         double precision :: xk_old, xkp1_old, xk_new, xk_prev, sum
         logical :: increasing              
         increasing = (init_x(1) < init_x(nx))
         
         if (increasing .and. (x(1) < init_x(1) .or. x(nv) > init_x(nx))
     >       .or. ((.not. increasing) .and. (x(1) > init_x(1) .or. x(nv) < init_x(nx)))) then
            ierr = -1
            call alert(ierr, 'no extrapolation allowed for integrate_values')
            return
         end if
         
         if (size(vals,dim=1) < nv) then
            ierr = -1
            call alert(ierr, 'size(vals,dim=1) < nv for interpolation')
            return
         end if
         
         if (size(f,dim=1) /= 4) then
            ierr = -1
            call alert(ierr, 'size(f,dim=1) /= 4 for interpolation')
            return
         end if
         
         if (size(f,dim=2) < nx) then
            ierr = -1
            call alert(ierr, 'size(f,dim=2) < nx for interpolation')
            return
         end if
         
         ierr = 0
         
         k_old = 1; xk_old = init_x(k_old); xkp1_old = init_x(k_old+1)
         sum = 0; xk_prev = x(1); vals(1) = 0
         
         do k_new = 2, nv
         
            xk_new = x(k_new)
            do while ((increasing .and. xk_new > xkp1_old) .or. ((.not. increasing) .and. xk_new < xkp1_old))
               k_old = k_old + 1
               if (k_old >= nx) then
                  k_old = k_old - 1
                  xk_new = xkp1_old
                  exit
               end if
               call add_to_integral(k_old - 1, xkp1_old)
               xk_old = xkp1_old
               xkp1_old = init_x(k_old+1)
            end do
            
            call add_to_integral(k_old, xk_new)
            vals(k_new) = sum
            sum = 0
            
         end do
         
         contains
         
         subroutine add_to_integral(k, x2)
            integer, intent(in) :: k
            double precision, intent(in) :: x2
            
            double precision :: x0, x1, a1, a2, d1, d2, area
            
            x0 = init_x(k)
            x1 = xk_prev
            if (x1 == x2) return
            d2 = x2 - x0
            a2 = d2*(f(1, k) + d2*(f(2, k)/2 
     >               + d2*(f(3, k)/3 + d2*f(4, k)/4)))
            if (x1 > x0) then
               d1 = x1 - x0
               a1 = d1*(f(1, k) + d1*(f(2, k)/2 
     >                     + d1*(f(3, k)/3 + d1*f(4, k)/4)))
               area = a2 - a1
            else
               d1 = 0; a1 = 0; area = a2
            end if
            sum = sum + area
            xk_prev = x2
         
         end subroutine add_to_integral
                     
      
      end subroutine do_integrate_values
      
      
      subroutine do_interp_values(init_x, nx, f, nv, x, vals, ierr)
         double precision, intent(in) :: init_x(nx) ! junction points, strictly monotonic
         integer, intent(in) :: nx ! length of init_x vector
         double precision, intent(in) :: f(:,:) ! (4, nx)  ! data & interpolation coefficients
         integer, intent(in) :: nv ! length of new x vector and vals vector
         double precision, intent(in) :: x(nv)  ! locations where want interpolated values
            ! strictly monotonic in same way as init_x
            ! values out of range of init_x's are clipped to boundaries of init_x's
         double precision, intent(out) :: vals(:) ! (nv)
         integer, intent(out) :: ierr ! 0 means aok
   
         integer :: k_old, k_new
         double precision :: xk_old, xkp1_old, xk_new, delta
         logical :: increasing
         
         ierr = 0
         
         if (nx < 1) then
            ierr = -1
            call alert(ierr, 'failed to supply initial values for interp 1D')
            return
         end if
         
         if (size(vals,dim=1) < nv) then
            ierr = -1
            call alert(ierr, 'size(vals,dim=1) < nv for interpolation')
            return
         end if
         
         if (size(f,dim=1) /= 4) then
            ierr = -1
            call alert(ierr, 'size(f,dim=1) /= 4 for interpolation')
            return
         end if
         
         if (size(f,dim=2) < nx) then
            ierr = -1
            call alert(ierr, 'size(f,dim=2) < nx for interpolation')
            return
         end if
         
         if (nx == 1) then
            vals(1:nv) = f(1,1)
            return
         end if

         increasing = (init_x(1) < init_x(nx))
         
         k_old = 1; xk_old = init_x(k_old); xkp1_old = init_x(k_old+1)
                  
         do k_new = 1, nv
      
            xk_new = x(k_new)
            if (increasing) then
               if (xk_new > init_x(nx)) then
                  xk_new = init_x(nx)
               else if (xk_new < init_x(1)) then
                  xk_new = init_x(1)
               end if
            else ! decreasing
               if (xk_new < init_x(nx)) then
                  xk_new = init_x(nx)
               else if (xk_new > init_x(1)) then
                  xk_new = init_x(1)
               end if
            end if
            do while ((increasing .and. xk_new > xkp1_old) .or. ((.not. increasing) .and. xk_new < xkp1_old))
               k_old = k_old + 1
               if (k_old >= nx) then
                  k_old = k_old - 1
                  xk_new = xkp1_old
                  exit
               end if
               xk_old = xkp1_old
               xkp1_old = init_x(k_old+1)
            end do
         
            delta = xk_new - xk_old
         
            vals(k_new) = 
     >            f(1, k_old) + delta*(f(2, k_old) 
     >               + delta*(f(3, k_old) + delta*f(4, k_old)))
            
         end do

   
      end subroutine do_interp_values
      
      
      subroutine do_interp_values_and_slopes(init_x, nx, f, nv, x, vals, slopes, ierr)
         double precision, intent(in) :: init_x(nx) ! junction points, strictly monotonic
         integer, intent(in) :: nx ! length of init_x vector
         double precision, intent(in) :: f(:,:) ! (4, nx)  ! data & interpolation coefficients
         integer, intent(in) :: nv ! length of new x vector and vals vector
         double precision, intent(in) :: x(nv)  ! locations where want interpolated values
            ! strictly monotonic in same way as init_x
            ! values out of range of init_x's are clipped to boundaries of init_x's
         double precision, intent(out) :: vals(:) ! (nv)
         double precision, intent(out) :: slopes(:) ! (nv)
         integer, intent(out) :: ierr ! 0 means aok
   
         integer :: k_old, k_new
         double precision :: xk_old, xkp1_old, xk_new, delta                  
         logical :: increasing   
         
         ierr = 0          
          
         if (nx < 1) then
            ierr = -1
            call alert(ierr, 'failed to supply initial values for interp 1D')
            return
         end if
         
         if (size(vals,dim=1) < nv) then
            ierr = -1
            call alert(ierr, 'size(vals,dim=1) < nv for interpolation')
            return
         end if
         
         if (size(f,dim=1) /= 4) then
            ierr = -1
            call alert(ierr, 'size(f,dim=1) /= 4 for 1d interpolation: interp_values_and_slopes')
            return
         end if
         
         if (size(f,dim=2) < nx) then
            ierr = -1
            call alert(ierr, 'size(f,dim=2) < nx for 1d interpolation: interp_values_and_slopes')
            return
         end if
         
         if (nx == 1) then
            vals(1:nv) = f(1,1)
            slopes(1:nv) = 0
            return
         end if
         
         k_old = 1; xk_old = init_x(k_old); xkp1_old = init_x(k_old+1)

         increasing = (init_x(1) < init_x(nx))
         
         do k_new = 1, nv
         
            xk_new = x(k_new)
            if (increasing) then
               if (xk_new > init_x(nx)) then
                  xk_new = init_x(nx)
               else if (xk_new < init_x(1)) then
                  xk_new = init_x(1)
               end if
            else ! decreasing
               if (xk_new < init_x(nx)) then
                  xk_new = init_x(nx)
               else if (xk_new > init_x(1)) then
                  xk_new = init_x(1)
               end if
            end if
            do while ((increasing .and. xk_new > xkp1_old) .or. ((.not. increasing) .and. xk_new < xkp1_old))
               k_old = k_old + 1
               if (k_old >= nx) then
                  k_old = k_old - 1
                  xk_new = xkp1_old
                  exit
               end if
               xk_old = xkp1_old
               xkp1_old = init_x(k_old+1)
            end do
            
            delta = xk_new - xk_old
            
            vals(k_new) = 
     >            f(1, k_old) + delta*(f(2, k_old) 
     >               + delta*(f(3, k_old) + delta*f(4, k_old)))
            
            slopes(k_new) = 
     >            f(2, k_old) + 2*delta*(f(3, k_old) + 1.5d0*delta*f(4, k_old))
            
         end do

   
      end subroutine do_interp_values_and_slopes
      
            
      double precision function minmod1(f1, f2)
         double precision, intent(in) :: f1, f2       
         minmod1 = 0.5d0 * (sign(1d0, f1) + sign(1d0, f2)) * min(abs(f1), abs(f2))    
      end function minmod1
      
      
      double precision function median1(f1, f2, f3)
         double precision, intent(in) :: f1, f2, f3
         median1 = f1 + minmod1(f2 - f1, f3 - f1)
      end function median1

      
      subroutine minmod(z, n, f1, f2)
         double precision, intent(out) :: z(n)     
         integer, intent(in) :: n       ! length of vectors
         double precision, intent(in) :: f1(n), f2(n)       
         z(1:n) = 0.5d0 * (sign(1d0, f1(1:n)) + sign(1d0, f2(1:n))) * min(abs(f1(1:n)), abs(f2(1:n)))      
      end subroutine minmod

      
      subroutine minmod4(z, n, f1, f2, f3, f4)
         double precision, intent(out) :: z(n)     
         integer, intent(in) :: n       ! length of vectors
         double precision, intent(in) :: f1(n), f2(n), f3(n), f4(n)
         call minmod(z, n, f1, f2)
         call minmod(z, n, z, f3)
         call minmod(z, n, z, f4)
      end subroutine minmod4
      
      
      subroutine median(z, n, f1, f2, f3)
         double precision, intent(out) :: z(n)     
         integer, intent(in) :: n       ! length of vectors
         double precision, intent(in) :: f1(n), f2(n), f3(n)
         double precision :: tmp1(n), tmp2(n)
         tmp1(1:n) = f2(1:n) - f1(1:n)
         tmp2(1:n) = f3(1:n) - f1(1:n)
         call minmod(z(1:n), n, tmp1(1:n), tmp2(1:n))
         z(1:n) = z(1:n) + f1(1:n)
      end subroutine median


      end module interp_1d_misc
