! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module eos_def
      implicit none
       
      ! cgs units
      
      ! the basic eos results
      
      integer, parameter :: i_lnPgas = 1
            ! gas pressure (total pressure minus radiation pressure)
      integer, parameter :: i_lnE = 2 
            ! internal energy per gram
      integer, parameter :: i_lnS = 3 
            ! entropy per gram
      integer, parameter :: i_grad_ad = 4 
            ! dlnT_dlnP at constant S
      integer, parameter :: i_chiRho = 5 
            ! dlnP_dlnRho at constant T
      integer, parameter :: i_chiT = 6 
            ! dlnP_dlnT at constant Rho
      integer, parameter :: i_Cp = 7 
            ! dh_dT at constant P, specific heat at constant total pressure
            ! where h is enthalpy, h = E + P/Rho
      integer, parameter :: i_Cv = 8 
            ! dE_dT at constant Rho, specific heat at constant volume
      integer, parameter :: i_dE_dRho = 9 
            ! at constant T
      integer, parameter :: i_dS_dT = 10 
            ! at constant Rho
      integer, parameter :: i_dS_dRho = 11 
            ! at constant T
      integer, parameter :: i_mu = 12 
            ! mean molecular weight per gas particle (ions + free electrons)
      integer, parameter :: i_lnfree_e = 13
            ! free_e := total combined number per nucleon of free electrons and positrons
      integer, parameter :: i_gamma1 = 14 
            ! dlnP_dlnRho at constant S
      integer, parameter :: i_gamma3 = 15 
            ! gamma3 - 1 = dlnT_dlnRho at constant S
      integer, parameter :: i_eta = 16 
            ! electron degeneracy parameter (eta > 1 for significant degeneracy)
            ! eta = ratio of electron chemical potential to kT
      
      integer, parameter :: num_eos_basic_results = 16
      
      
      character (len=20) :: eosDT_result_names(num_eos_basic_results)

      
      ! NOTE: the calculation of eta is based on the following equation for ne, 
      ! the mean number of free electrons per cm^3,
      ! assuming non-relativistic electrons (okay for T < 10^9 at least)
      !
      !  ne = 4 Pi / h^3 (2 me k T)^1.5 F[1/2,eta]   -- see, for example, Clayton, eqn 2-57
      !        where F is the fermi-dirac integral: 
      !        F[beta,eta] := Integrate[(u^beta)/(1+E^(u-eta)),{u,0,Infinity}]
      ! 
      ! CAVEAT: when free_e, the mean number of free electrons per nucleon gets really small, 
      ! eta isn't very interesting because there aren't a lot of free electrons to be degenerate!
      ! our calculation of eta gets flaky at this point as well.
      ! we sweep this problem under the rug by making eta tend to a fairly large negative value 
      ! when free_e < 0.01 or so. this roughly corresponds to T < 10^4 or less.

      
      integer, parameter :: num_eos_Zs = 5
      double precision, parameter :: eos_Zs(num_eos_Zs) = (/ 0.00d0, 0.02d0, 0.04d0, 0.20d0, 1.00d0 /)
      integer, parameter :: num_eos_Xs_for_Z(num_eos_Zs) = (/ 6, 5, 5, 5, 1 /)
      
      integer, parameter :: num_eos_Xs = 6
      double precision, parameter :: eos_Xs(num_eos_Xs) =
     >      (/ 0.0d0, 0.2d0, 0.4d0, 0.6d0, 0.8d0, 1.0d0 /)

      integer, parameter :: sz_per_eos_point = 4 ! for bicubic spline interpolation

      type EoS_General_Info
         logical :: include_radiation
         double precision :: mass_fraction_limit_for_PC ! skip any species with abundance < this
         double precision :: logRho1_OPAL_SCVH_limit ! don't use OPAL_SCVH for logRho > this
         double precision :: logRho2_OPAL_SCVH_limit ! full OPAL_SCVH okay for logRho < this
         double precision :: logRho1_PC_limit ! okay for pure PC for logRho > this
         double precision :: logRho2_PC_limit ! don't use PC for logRho < this (>= 2.8)
         ! transition log_Gamma for PC to HELM
         double precision :: log_Gamma_all_HELM ! HELM for log_Gamma <= this
         double precision :: Gamma_all_HELM ! 10**log_Gamma_all_HELM
         double precision :: log_Gamma_all_PC ! PC for log_Gamma >= this
         double precision :: PC_min_Z ! don't use PC for Z < this
         ! transition Z for OPAL to HELM
         double precision :: Z_all_HELM ! HELM for Z >= this
         ! transition temperature zone for OPAL to HELM at high T
         double precision :: logT_all_HELM ! HELM for lgT >= this
         double precision :: logT_all_OPAL ! OPAL for lgT <= this
         ! transition temperature zone for SCVH to HELM at low T
         double precision :: logT_low_all_HELM ! HELM for lgT <= this
         double precision :: logT_low_all_SCVH ! SCVH for lgT >= this
         ! bookkeeping
         integer :: handle
         logical :: in_use
      end type EoS_General_Info

      
      include 'helm_def.dek'


      ! THE FOLLOWING ARE PRIVATE DEFS -- NOT FOR USE BY CLIENTS
      
      
      ! data tables -- read only after they are initialized
      
         type (Helm_Table), pointer :: eos_ht

         type EosDT_XZ_Info
            real :: logQ_min ! logQ = logRho - 2*logT + 12
            real :: logQ_max
            real :: del_logQ ! spacing for the logQs
            integer :: num_logQs
            real :: logT_min
            real :: logT_max
            real :: del_logT ! spacing for the logTs
            integer :: num_logTs
            real, dimension(:), pointer :: logQs, logTs
            real, dimension(:,:,:,:), pointer :: tbl
               ! dimension(sz_per_eos_point, num_eos_vals, num_logQs, num_logTs)
            integer :: version
         end type EosDT_XZ_Info

         type EosPT_XZ_Info
            real :: logW_min ! logW = logPgas - 4*logT
            real :: logW_max
            real :: del_logW ! spacing for the logWs
            integer :: num_logWs
            real :: logT_min
            real :: logT_max
            real :: del_logT ! spacing for the logTs
            integer :: num_logTs
            real, dimension(:), pointer :: logWs, logTs
            real, dimension(:,:,:,:), pointer :: tbl
               ! dimension(sz_per_eos_point, num_eos_vals, num_logWs, num_logTs)
            integer :: version
         end type EosPT_XZ_Info
      
         type (EosDT_XZ_Info), target :: eosDT_XZ_data(num_eos_Xs, num_eos_Zs)
         type (EosPT_XZ_Info), target :: eosPT_XZ_data(num_eos_Xs, num_eos_Zs)
      

      ! parameters for eosDT

      ! internal storage of table info
      integer, parameter :: eosDT_ilnE = 1
      integer, parameter :: eosDT_ilnPgas = 2
      ! the rest are the same for all of the cases
      integer, parameter :: eos_ilnS = 3
      integer, parameter :: eos_igrad_ad = 4
      integer, parameter :: eos_ichiRho = 5
      integer, parameter :: eos_ichiT = 6
      integer, parameter :: eos_iCp = 7
      integer, parameter :: eos_iCv = 8
      integer, parameter :: eos_idE_dRho = 9
      integer, parameter :: eos_idS_dT = 10
      integer, parameter :: eos_idS_dRho = 11
      integer, parameter :: eos_imu = 12
      integer, parameter :: eos_ilnfree_e = 13
      integer, parameter :: eos_igamma1 = 14
      integer, parameter :: eos_igamma3 = 15
      integer, parameter :: eos_ieta = 16

      integer, parameter :: num_eos_vals = 16

      ! we are storing more than we'd need to store if we had unlimited numerical precision.
      ! for example, Cv, the S derivatives, and the gammas can all be derived from the other stuff.
      ! better to be safe, we'll take the storage hit and do them independently.


      ! parameters for eosPT
      integer, parameter :: eosPT_ilnE = 1
      integer, parameter :: eosPT_ilnRho = 2


      ! interpolation info for theta_e
      integer :: theta_e_nx
      double precision, pointer :: f_theta_e(:,:)
      double precision, pointer :: x_theta_e(:)
      
      
      integer, parameter :: max_eos_handles = 10
      type (EoS_General_Info), target :: eos_handles(max_eos_handles)
      
      
      logical :: use_cache_for_eos = .true.
      logical :: eos_root_is_initialized = .false.
      logical :: eosDT_is_initialized = .false.
      logical :: eosPT_is_initialized = .false.
      
      character(len=256) :: data_dir_for_eos
      character(len=32) :: eosDT_file_prefix, eosDT_Z1_suffix
      character(len=32) :: eosPT_file_prefix, eosPT_Z1_suffix

      
      contains
      
      
      subroutine eos_def_init
         integer :: i
         do i=1,max_eos_handles
            eos_handles(i)% handle = i
            eos_handles(i)% in_use = .false.
         end do
         eos_root_is_initialized = .false.
         eosDT_is_initialized = .false.
         eosPT_is_initialized = .false.
         use_cache_for_eos = .true.
         eosDT_file_prefix = 'mesa'
         eosPT_file_prefix = 'mesa'
         eosDT_Z1_suffix = '_CO_1'
         eosPT_Z1_suffix = '_CO_1'
         eosDT_result_names(i_lnPgas) = 'lnPgas'
         eosDT_result_names(i_lnE) = 'lnE'
         eosDT_result_names(i_lnS) = 'lnS'
         eosDT_result_names(i_grad_ad) = 'grad_ad'
         eosDT_result_names(i_chiRho) = 'chiRho'
         eosDT_result_names(i_chiT) = 'chiT'
         eosDT_result_names(i_Cp) = 'Cp'
         eosDT_result_names(i_Cv) = 'Cv'
         eosDT_result_names(i_dE_dRho) = 'dE_dRho'
         eosDT_result_names(i_dS_dT) = 'dS_dT'
         eosDT_result_names(i_dS_dRho) = 'dS_dRho'
         eosDT_result_names(i_mu) = 'mu'
         eosDT_result_names(i_lnfree_e) = 'lnfree_e'
         eosDT_result_names(i_gamma1) = 'gamma1'
         eosDT_result_names(i_gamma3) = 'gamma3'
         eosDT_result_names(i_eta) = 'eta'
      end subroutine eos_def_init

      
      integer function do_alloc_eos(ierr)
         integer, intent(out) :: ierr
         integer :: i
         type (EoS_General_Info), pointer :: rq
         ierr = 0
         do_alloc_eos = -1
!$omp critical (eos_handle)
         do i = 1, max_eos_handles
            if (.not. eos_handles(i)% in_use) then
               eos_handles(i)% in_use = .true.
               do_alloc_eos = i
               exit
            end if
         end do
!$omp end critical (eos_handle)
         if (do_alloc_eos == -1) then
            ierr = -1
            !call alert(ierr, 'no available eos handle')
            return
         end if
         if (eos_handles(do_alloc_eos)% handle /= do_alloc_eos) then
            ierr = -1
            !call alert(ierr, 'broken handle for eos')
            return
         end if
         rq => eos_handles(do_alloc_eos)
         ! set defaults
         rq% include_radiation = .true.
         
         rq% logT_all_HELM = 7.7d0 ! HELM for lgT >= this
         rq% logT_all_OPAL = 7.6d0 ! OPAL for lgT <= this
         
         rq% logT_low_all_HELM = 2.2d0 ! HELM for lgT <= this
         rq% logT_low_all_SCVH = 2.3d0 ! SCVH for lgT >= this

         rq% mass_fraction_limit_for_PC = 1d-2
         
         rq% logRho1_OPAL_SCVH_limit = 3.6d0  ! must be <= 3.7
         rq% logRho2_OPAL_SCVH_limit = 3.0d0
         rq% logRho1_PC_limit = 2.999d0 
            ! keep < logRho2_OPAL_SCVH_limit so can go from table to PC without HELM
         rq% logRho2_PC_limit = 2.8d0 ! must be > 2.8 or so to avoid NaN's from PC

         !rq% Z_all_HELM = 0.040000001d0 ! old
         rq% Z_all_HELM = 1.000000001d0 ! new
         
         rq% log_Gamma_all_HELM = log10(40d0)
         rq% Gamma_all_HELM = 10**rq% log_Gamma_all_HELM
         rq% log_Gamma_all_PC = log10(80d0)
         rq% PC_min_Z = 0.999d0
      end function do_alloc_eos
            
      
      subroutine do_free_eos_handle(handle)
         integer, intent(in) :: handle
         type (EoS_General_Info), pointer :: rq
         if (handle >= 1 .and. handle <= max_eos_handles) then
            rq => eos_handles(handle)
            eos_handles(handle)% in_use = .false.
         end if
      end subroutine do_free_eos_handle
      

      subroutine get_eos_ptr(handle,rq,ierr)
         !use alert_lib,only:alert
         integer, intent(in) :: handle
         type (EoS_General_Info), pointer :: rq
         integer, intent(out):: ierr         
         if (handle < 1 .or. handle > max_eos_handles) then
            ierr = -1
            !call alert(ierr,'invalid eos handle')
            return
         end if
         rq => eos_handles(handle)
         ierr = 0
      end subroutine get_eos_ptr
      

      end module eos_def
      
