! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module eosDT_load_tables
      use alert_lib
      use eos_def
      use utils_lib, only: is_bad_real

      implicit none

      
      ! the file EOS data
      integer, parameter :: jlogPgas = 1
      integer, parameter :: jlogE = 2
      integer, parameter :: jlogS = 3
      integer, parameter :: jchiRho = 4
      integer, parameter :: jchiT = 5
      integer, parameter :: jCp = 6
      integer, parameter :: jCv = 7
      integer, parameter :: jdE_dRho = 8
      integer, parameter :: jdS_dT = 9
      integer, parameter :: jdS_dRho = 10
      integer, parameter :: jmu = 11
      integer, parameter :: jlogfree_e = 12
      integer, parameter :: jgamma1 = 13
      integer, parameter :: jgamma3 = 14
      integer, parameter :: jgrad_ad = 15
      integer, parameter :: jeta = 16
      integer, parameter :: num_eos_file_vals = 16

      integer, parameter :: file_max_num_logQs = 1000

      

      contains
      
      
      subroutine request_user_to_reinstall
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*) 'NOTICE: you need to install a new verion of the eos data.'
         write(*,*) 'Please update by removing the directory mesa/data/eosDT_data,'
         write(*,*) 'and rerunning the mesa ./install script.'
         write(*,*)
         write(*,*)
         stop 1
      end subroutine request_user_to_reinstall
      
      
      subroutine check_for_error_in_eosDT_data(ierr, fname)
         integer, intent(in) :: ierr
         character (len=*) :: fname
         if (ierr == 0) return
         write(*,*) 'load eos tables ' // trim(fname)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,'(a)') 'FATAL ERROR: missing or bad eos data.'
         write(*,'(a)') 'Please update by removing the directories mesa/data/eosDT_data and mesa/data/eosPT_data,'
         write(*,'(a)') 'and rerunning the mesa ./install script.'
         write(*,*)
         stop 1
      end subroutine check_for_error_in_eosDT_data
      
      
      subroutine Load_eosDT_Table(ierr)
         use utils_lib
         integer, intent(out) :: ierr
         
         integer :: iz, ix, i
         ierr = 0
!$OMP CRITICAL
         call do_read
!$OMP END CRITICAL
         
         contains
         
         subroutine do_read
            if (eosDT_is_initialized) return
            do iz = 1, num_eos_Zs
               do ix = 1, num_eos_Xs
                  if (eos_Zs(iz) + eos_Xs(ix) > 1.0000001d0) cycle
                  call read_one(ix,iz,ierr)
                  if (ierr /= 0) return
               end do
            end do
            eosDT_is_initialized = .true.
         end subroutine do_read
         
         subroutine read_one(ix,iz,ierr)
            integer, intent(in) :: ix, iz
            integer, intent(out) :: ierr
            character (len=256) :: fname, cache_filename
            integer :: iounit1, iounit2            
            include 'formats.dek'            
            iounit1 = alloc_iounit(ierr); if (ierr /= 0) return
            iounit2 = alloc_iounit(ierr); if (ierr /= 0) return
            call Get_eosDT_Table_Filenames(
     >         eos_Zs(iz), eos_Xs(ix), data_dir_for_eos, fname, cache_filename)
            call Load1_eosDT_Table(
     >         ix, iz, fname, cache_filename, iounit1, iounit2, use_cache_for_eos, ierr)
            if (ierr /= 0) then
               write(*,*) 'Load1_eosDT_Table ierr', ierr, ix, iz, eos_Xs(ix), eos_Zs(iz)
            end if
            call free_iounit(iounit2)
            call free_iounit(iounit1)
         end subroutine read_one
         
      end subroutine Load_eosDT_Table
      
      
      subroutine Get_eosDT_Table_Filenames(Z, X, data_dir, fname, cache_filename)
         double precision, intent(in) :: Z, X
         character (*), intent(in) :: data_dir
         character (len=*), intent(out) :: fname, cache_filename
         character (len=256) :: Zstr, Xstr, suffix
         
         call setstr(Z,Zstr)
         call setstr(X,Xstr)
         if (Zstr == '100') then
            suffix = eosDT_Z1_suffix
         else
            suffix = ''
         end if
         
         fname = trim(data_dir) // 
     >         '/eosDT_data/' // trim(eosDT_file_prefix) // '-eosDT_' //
     >         trim(Zstr) // 'z' // trim(Xstr) // 'x' // trim(suffix) // '.data'
         cache_filename = trim(data_dir) // 
     >         '/eosDT_data/cache/' // trim(eosDT_file_prefix) // '-eosDT_' //
     >         trim(Zstr) // 'z' // trim(Xstr) // 'x' // trim(suffix) // '.bin'
         
         contains
         
         subroutine setstr(v,str)
            real*8, intent(in) :: v
            character (len=*) :: str
            if (v > 0.99999) then
               str = '100'
            else if (v > 0.09999) then
               write(str, '(i2)') floor(100d0 * v + 0.5)
            else
               write(str, '(a,i1)') '0', floor(100d0 * v + 0.5)
            end if
         end subroutine setstr
              
      end subroutine Get_eosDT_Table_Filenames
      
      
      subroutine Load1_eosDT_Table(
     >      ix, iz, filename, cache_filename, io_unit, cache_io_unit, use_cache, info)
         integer, intent(in) :: ix, iz
         character (*), intent(in) :: filename, cache_filename
         integer, intent(in) :: io_unit, cache_io_unit
         logical, intent(in) :: use_cache
         integer, intent(out) :: info
         
         double precision :: X, Z
         real :: X_in, Z_in, logQ, logT
         integer :: j, i, k, iQ, ios, status
         real, dimension(:,:,:), pointer :: tbl2
         character (len=256) :: message
         double precision, parameter :: tiny = 1e-6
         type (EosDT_XZ_Info), pointer :: ep
         
         include 'formats.dek'

         info = 0    
         
         X = eos_Xs(ix)
         Z = eos_Zs(iz)   
         
         ep => eosDT_XZ_data(ix,iz)     

         write(message,*) 'open ', trim(filename)
         
         open(UNIT=io_unit, FILE=trim(filename), ACTION='READ', STATUS='OLD', IOSTAT=ios)
         call check_for_error_in_eosDT_data(ios, filename)

         read(io_unit,*,iostat=info)
         if (info /= 0) return
         read(io_unit,*,iostat=info) 
     >         ep% version, X_in, Z_in, 
     >         ep% num_logTs, ep% logT_min, ep% logT_max, ep% del_logT, 
     >         ep% num_logQs, ep% logQ_min, ep% logQ_max, ep% del_logQ
         if (info /= 0) return
         read(io_unit,*,iostat=info)
         if (info /= 0) return

         if (abs(X-X_in) > tiny
     >      .or. abs(Z-Z_in) > tiny
     >     ) then
            write(message,*) 'bad header info in ' // trim(filename)
            info = -1
            call alert(info,message)
            close(io_unit)
            if (abs(X-X_in) > tiny) then
               write(*,'(a50,l)') 'abs(X-X_in) > tiny', abs(X-X_in) > tiny
            end if
            if (abs(Z-Z_in) > tiny) then
               write(*,'(a50,l)') 'abs(Z-Z_in) > tiny', abs(Z-Z_in) > tiny
            end if
            write(*,*)
            call request_user_to_reinstall
            return
         end if
         
         allocate(ep% tbl(sz_per_eos_point, num_eos_vals, ep% num_logQs, ep% num_logTs),
     >            ep% logQs(ep% num_logQs), ep% logTs(ep% num_logTs),  
     >            STAT=info)
         if (info /= 0) then
            call alert(info,'allocate failed for eos tables')
            return
         end if
         
         ep% logQs(1) = ep% logQ_min
         do i = 2, ep% num_logQs-1
            ep% logQs(i) = ep% logQs(i-1) + ep% del_logQ
         end do
         ep% logQs(ep% num_logQs) = ep% logQ_max
         
         ep% logTs(1) = ep% logT_min
         do i = 2, ep% num_logTs-1
            ep% logTs(i) = ep% logTs(i-1) + ep% del_logT
         end do
         ep% logTs(ep% num_logTs) = ep% logT_max

         if (use_cache) then
            call Read_EoS_Cache(X, Z, ep, cache_filename, cache_io_unit, ios)
            if (ios == 0) then
               close(io_unit)
               return
            end if
         end if
                  
         status = 0
         allocate(tbl2(num_eos_file_vals, ep% num_logQs, ep% num_logTs), STAT=status)
         if (status .ne. 0) then
            info = -1
            call alert(info,'allocate failed for eos tables')
            return
         end if

         do iQ=1,ep% num_logQs
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            read (io_unit,*,iostat=info) logQ
            if (failed('read logQ')) return
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            do i=1,ep% num_logTs
               read(io_unit,'(a)',iostat=info) message
               if (failed('read line')) then
                  write(*,'(a)') trim(message)
                  write(*,*) trim(filename)
                  write(*,*) 'iQ, i', iQ, i
                  write(*,*) 'logQ', logQ
                  write(*,*) 'bad input line?'
                  stop 1
               end if
               read (message,*,iostat=info) logT, tbl2(1:num_eos_file_vals,iQ,i)
               if (failed('read tbl2')) then
                  write(*,'(a)') trim(message)
                  write(*,*) trim(filename)
                  write(*,*) 'iQ, i', iQ, i
                  write(*,*) 'logQ', logQ
                  write(*,*) 'bad input line?'
                  stop 1
               end if
            enddo
            if(iQ == ep% num_logQs) exit
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
         end do
            
         close(io_unit)
         
         !write(*,*) 'Make_XEoS_Interpolation_Data ' // trim(filename)
         call Make_XEoS_Interpolation_Data(ep, tbl2, info)
         deallocate(tbl2)
         if (failed('Make_XEoS_Interpolation_Data')) return
         
         call Check_XEoS_Interpolation_Data(ep)
         
         if (.not. use_cache) return

         open(unit=cache_io_unit, file=trim(cache_filename), iostat=ios, 
     >         action='write', form='unformatted')

         if (ios == 0) then
            write(*,*) 'write ', trim(cache_filename)
            write(cache_io_unit) 
     >         X_in, Z_in, ep% num_logTs, ep% logT_min, ep% logT_max, ep% del_logT, 
     >         ep% num_logQs, ep% logQ_min, ep% logQ_max, ep% del_logQ, ep% version
            do i = 1, ep% num_logTs
               do j = 1, ep% num_logQs
                  write(cache_io_unit) ep% tbl(1:sz_per_eos_point, 1:num_eos_vals, j, i)
               end do
            end do
            close(cache_io_unit)
         end if
         
         contains
         
         subroutine Check_XEoS_Interpolation_Data(ep)
            use utils_lib,only:is_bad_real
            type (EosDT_XZ_Info), pointer :: ep
            ! for logT > 6.8 and logRho < -10, splines can get bogus higher order terms
            ! replace NaN's and Infinities with 0
            integer :: i, j, iQ, jtemp
            do i = 1, sz_per_eos_point
               do j = 1, num_eos_vals
                  do iQ = 1, ep% num_logQs
                     do jtemp = 1, ep% num_logTs
                        if (is_bad_real(ep% tbl(i,j,iQ,jtemp))) then
                           ep% tbl(i,j,iQ,jtemp) = 0
                        end if
                     end do
                  end do
               end do
            end do
         end subroutine Check_XEoS_Interpolation_Data
         
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (info /= 0)
            if (failed) write(*,*) 'Load1_eosDT_Table failed: ' // trim(str)
         end function failed
         

      end subroutine Load1_eosDT_Table
      
      
      subroutine Make_XEoS_Interpolation_Data(ep, tbl2, info)
         use interp_2d_lib_sg
         use const_def, only: crad, ln10

         type (EosDT_XZ_Info), pointer :: ep
         real :: tbl2(:,:,:) ! (num_eos_file_vals, ep% num_logQs, ep% num_logTs)
            ! ep% tbl gets internal eos data
            ! tbl2 holds eos data from file
         integer, intent(out) :: info

         real :: logQs(ep% num_logQs)              ! x vector, strict ascending
         real :: logTs(ep% num_logTs)                    ! y vector, strict ascending
         real :: Ts(ep% num_logTs)
         real :: f(sz_per_eos_point,ep% num_logQs,ep% num_logTs)              
            ! data & spline coefficients
         integer :: ibcxmin                   ! bc flag for x=xmin
         real :: bcxmin(ep% num_logTs)    ! bc data vs. y at x=xmin
         integer :: ibcxmax                   ! bc flag for x=xmax
         real :: bcxmax(ep% num_logTs)     ! bc data vs. y at x=xmax
         integer :: ibcymin                   ! bc flag for y=ymin
         real :: bcymin(ep% num_logQs)   ! bc data vs. x at y=ymin
         integer :: ibcymax                   ! bc flag for y=ymax
         real :: bcymax(ep% num_logQs)   ! bc data vs. x at y=ymax
         integer :: ili_logQs    ! =1: logRho grid is "nearly" equally spaced
         integer :: ili_logTs      ! =1: logT grid is "nearly" equally spaced
         integer :: ier            ! =0 on exit if there is no error.
         real :: logQ, Rho, logRho, T, P, Cv, chiRho, chiT, logT, logT0, logT1, logQ0, logQ1
         real :: gamma3, gamma1, grad_ad, Prad, E, S
         integer :: iQ, jtemp, ilogT, ilogQ
         real :: fval(num_eos_file_vals), df_dx(num_eos_file_vals), df_dy(num_eos_file_vals)
         
         integer :: v, vlist(3), var, i, j, num_logQs, num_logTs
         character (len=256) :: message
         
         include 'formats.dek'

         info = 0
         
         num_logQs = ep% num_logQs
         num_logTs = ep% num_logTs
         
         if (size(ep% tbl,dim=4) /= size(tbl2,dim=3) .or. size(tbl2,dim=3) /= num_logTs) then
            write(*,*) 'mismatch in last dimension for tbl and tbl2'
            write(*,*) 'size(ep% tbl,dim=4)', size(ep% tbl,dim=4)
            write(*,*) 'size(tbl2,dim=3)', size(tbl2,dim=3)
            write(*,*) 'num_logTs', num_logTs
            info = -1
            return
         end if
         
         if (size(ep% tbl,dim=3) /= size(tbl2,dim=2) .or. size(tbl2,dim=2) /= num_logQs) then
            write(*,*) 'mismatch in next to last dimension for tbl and tbl2'
            write(*,*) 'size(ep% tbl,dim=3)', size(ep% tbl,dim=3)
            write(*,*) 'size(tbl2,dim=2)', size(tbl2,dim=2)
            write(*,*) 'num_logQs', num_logQs
            info = -1
            return
         end if
         
         do iQ = 1, ep% num_logQs
            logQs(iQ) = ep% logQ_min + (iQ-1) * ep% del_logQ
         end do

         do jtemp = 1, ep% num_logTs
            logTs(jtemp) = ep% logT_min + (jtemp-1) * ep% del_logT
         end do
         
         ! copy file eos variables to internal eos interpolation tables
         do j=1,num_logTs
            do i=1,num_logQs
               ep% tbl(1,eosDT_ilnPgas,i,j) = tbl2(jlogPgas,i,j)*ln10
               ep% tbl(1,eosDT_ilnE,i,j) = tbl2(jlogE,i,j)*ln10
               ep% tbl(1,eos_ilnS,i,j) = tbl2(jlogS,i,j)*ln10
               ep% tbl(1,eos_igrad_ad,i,j) = tbl2(jgrad_ad,i,j)
               ep% tbl(1,eos_ichiRho,i,j) = tbl2(jchiRho,i,j)
               ep% tbl(1,eos_ichiT,i,j) = tbl2(jchiT,i,j)
               ep% tbl(1,eos_iCp,i,j) = tbl2(jCp,i,j)
               ep% tbl(1,eos_iCv,i,j) = tbl2(jCv,i,j)
               ep% tbl(1,eos_idE_dRho,i,j) = tbl2(jdE_dRho,i,j)
               ep% tbl(1,eos_idS_dT,i,j) = tbl2(jdS_dT,i,j)
               ep% tbl(1,eos_idS_dRho,i,j) = tbl2(jdS_dRho,i,j)
               ep% tbl(1,eos_imu,i,j) = tbl2(jmu,i,j)
               ep% tbl(1,eos_ilnfree_e,i,j) = tbl2(jlogfree_e,i,j)*ln10
               ep% tbl(1,eos_igamma1,i,j) = tbl2(jgamma1,i,j)
               ep% tbl(1,eos_igamma3,i,j) = tbl2(jgamma3,i,j)
               ep% tbl(1,eos_ieta,i,j) = tbl2(jeta,i,j)
               
               if (.false. .and. ep% tbl(1,eos_igamma1,i,j) < 0 .and. i == 498 .and. j == 72) then
                  write(*,3) 'gamma1', i, j, ep% tbl(1,eos_igamma1,i,j), logQs(i), logTs(j)
                  !stop
               end if
               
            end do
         end do

         ! just use "not a knot" bc's at edges of tables
         ibcxmin = 0; bcxmin(:) = 0
         ibcxmax = 0; bcxmax(:) = 0
         ibcymin = 0; bcymin(:) = 0
         ibcymax = 0; bcymax(:) = 0

         ! create tables for bicubic spline interpolation         
         do v = 1, num_eos_vals
         
            f(1,:,:) = ep% tbl(1,v,:,:)
            call interp_mkbicub_sg(
     >            logQs,ep% num_logQs,logTs,ep% num_logTs,f,ep% num_logQs,
     >            ibcxmin,bcxmin,ibcxmax,bcxmax,
     >            ibcymin,bcymin,ibcymax,bcymax,
     >            ili_logQs,ili_logTs,ier)
            if (ier /= 0) then
               write(message,*) 'Make_XEoS_Interpolation_Data error happened for eos_value', v
               info = 3
               call alert(info,message)
               return
            end if
            ep% tbl(2:4,v,:,:) = f(2:4,:,:)
            
         end do
         
      end subroutine Make_XEoS_Interpolation_Data
      
      
      subroutine Read_EoS_Cache(X, Z, ep, cache_filename, io_unit, ios)
         double precision, intent(in) :: X, Z
         type (EosDT_XZ_Info), pointer :: ep
         character (*), intent(in) :: cache_filename
         integer, intent(in) :: io_unit ! use this for file access
         integer, intent(out) :: ios

         real :: X_in, Z_in, logT_min_in, logT_max_in, del_logT_in, 
     >         logQ_min_in, logQ_max_in, del_logQ_in
         integer :: num_logQs_in, num_logTs_in, version_in, i, j
         real, parameter :: tiny = 1e-6
         
         include 'formats.dek'
         
         ios = 0
         open(unit=io_unit,file=trim(cache_filename),action='read',
     >         status='old',iostat=ios,form='unformatted')
         if (ios /= 0) return
         
         read(io_unit, iostat=ios) 
     >         X_in, Z_in, num_logTs_in, logT_min_in, logT_max_in, del_logT_in, 
     >         num_logQs_in, logQ_min_in, logQ_max_in, del_logQ_in, version_in
         if (ios /= 0) return
         
         if (ep% version /= version_in) then
            ios = 1
            write(*,*) 'read cache failed for version_in'
         end if
         if (ep% num_logQs /= num_logQs_in) then
            ios = 1
            write(*,*) 'read cache failed for ep% num_logQs'
         end if 
         if (ep% num_logTs /= num_logTs_in) then
            ios = 1
            write(*,*) 'read cache failed for ep% num_logTs'
         end if
         if (abs(X-X_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for X_in'
         end if
         if (abs(Z-Z_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for Z_in'
         end if
         if (abs(ep% logT_min-logT_min_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_logT_min'
         end if    
         if (abs(ep% logT_max-logT_max_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_logT_max'
         end if    
         if (abs(ep% del_logT-del_logT_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_del_logT'
         end if    
         if (abs(ep% logQ_min-logQ_min_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_logQ_min'
         end if    
         if (abs(ep% logQ_max-logQ_max_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_logQ_max'
         end if
         if (abs(ep% del_logQ-del_logQ_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for eos_del_logQ'
         end if
         
         if (ios /= 0) then
            close(io_unit); return
         end if
         
         do i = 1, ep% num_logTs
            do j = 1, ep% num_logQs
               read(io_unit, iostat=ios) ep% tbl(1:sz_per_eos_point, 1:num_eos_vals, j, i)
               if (ios /= 0) then
                  close(io_unit); return
               end if
            end do
         end do
         
         close(io_unit)

      end subroutine Read_EoS_Cache
      
      
      end module eosDT_load_tables
