! ***********************************************************************
!
!   Copyright (C) 2009  Bill Paxton
!
!   This file is part of MESA.
!
!   MESA is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module make_PTplots
      
		use eos_def
		use eos_lib
		use const_def
		use alert_lib
		use chem_def
		use num_lib, only: safe_log10
		use helm_opal_scvh_driver
		
      implicit none
      

		logical, parameter :: helm_only = .false.
		logical, parameter :: opal_scvh_only = .false.
		logical, parameter :: opal_only = .false.
		logical, parameter :: search_for_SCVH = .true.
		logical, parameter :: scvh_only = .false.
		
		logical, parameter :: include_radiation = .true.

      
      contains
      
      
      
      subroutine Do_PT_Test
         use utils_lib, only: is_bad_num
         use const_lib
         
         integer :: ierr
         double precision :: &
            X, Z, abar, zbar, logT, T, logPgas, logRho, logQ, &
            Rho, logE, logS, chiRho, chiT, &
            Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, free_e, gamma1, gamma3, grad_ad, eta, Prad, Pgas, P, logNe, logP
         character (len=256) :: dir
	      integer, parameter :: io_unit0 = 40
         
         include 'formats.dek'
         
	      write(*,*) 'Do_Test'
	      
	      call setup_eos

         ierr = 0
         
         X = 1
         logT = 4.02d0         
         logRho = -1.8332d0         
         logP = 10d0
! he 10.00  9.99996E-01  1.64922E-06  -1.3601   8.4902  11.5210  -0.9271   0.9508   0.1659  -0.0656   0.3953
! h  10.00  3.26801E-01  6.72438E-01  -1.8332   9.0376  12.3945  -1.6005   1.0748   0.3884  -0.0956   0.2461

         
         X = 1
         logT = 4.5d0         
         logRho = -1.4506       
         logP = 11d0
! h   11.00  3.31668E-02  5.52835E-01  -1.4506   9.1249  12.9444  -1.3627   1.0628   0.3864  -0.0925   0.2395
         
         X = 0
         logT = 4.5d0         
         logRho = -0.8601d0      
         logP = 11d0
! he  11.00  9.08220E-01  4.58900E-02  -0.8601   8.5180  12.1131  -0.8923   0.9376   0.3219  -0.0629   0.1954

         
         X = 1
         logT = 4.8625418060200669
         logRho = 2.8500908611879505
         logP = 1.3217015777590825E+01

         Z = 1 - X
         call get_azbar(X, Z, abar, zbar)
         
! pure H test
! logT 4.02    
! logP 10.00
! v1 3.26801E-01  
! v2 6.72438E-01  
! lgRho -1.8332   
! lgS 9.0376  
! lgU 12.3945  
! grada 0.2461



         T = 10**logT
         Rho = 10**logRho
         P = 10**logP
         
         logQ = logRho - 2*logT + 12
         
         !include_radiation = 0 ! TESTING
		   
		   write(*,*)
         write(*,*) 'include_radiation', include_radiation
		   write(*,*)

         call helm_opal_scvh( &
            helm_only, opal_scvh_only, opal_only, scvh_only, search_for_SCVH, &
				include_radiation, logT, logRho, T, Rho, abar, zbar, X, Z, &
            logPgas, logE, logS, chiRho, chiT, Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, free_e, gamma1, gamma3, grad_ad, eta, &
            data_dir, ierr)


	      
         write(*,1) 'T', T
         write(*,1) 'logT', logT
         write(*,1) 'Rho', Rho
         write(*,1) 'logRho', logRho
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'X', X
         write(*,1) 'Z', Z
         write(*,1) 'logQ', logQ
         write(*,*)
         
         Prad = crad*T**4/3
         Pgas = 10**logPgas
         P = Pgas + Prad
         
         write(*,1) 'P', P
         write(*,1) 'logP', log10(P)
         write(*,1) 'Pgas', Pgas
         write(*,1) 'logPgas', logPgas
         write(*,1) 'logE', logE
         write(*,1) 'E', 10**logE
         write(*,1) 'logS', logS
         write(*,1) 'chiRho', chiRho
         write(*,1) 'chiT', chiT
         write(*,1) 'Cp', Cp
         write(*,1) 'Cv', Cv
         write(*,1) 'dE_dRho', dE_dRho
         write(*,1) 'dS_dT', dS_dT
         write(*,1) 'dS_dRho', dS_dRho
         write(*,1) 'mu', mu
         write(*,1) 'lnfree_e', safe_log10(free_e)
         write(*,1) 'gamma1', gamma1
         write(*,1) 'gamma3', gamma3
         write(*,1) 'grad_ad', grad_ad
         write(*,1) 'eta', eta

         write(*,*)
         stop 'Do_PT_Test'
			      
      
      end subroutine Do_PT_Test

      

      
      subroutine make_PTplot_files
         use utils_lib, only: is_bad_num
         use scvh_core, only: interp_vals, setup_scvh
         
         integer :: which_eos, logT_points, logP_points, io_params, io_logP, io_logT, &
            io_first, io_last, io, num_vals, j, i, k, ierr
         double precision, pointer :: output_values(:,:,:)
         double precision :: &
            X, Z, abar, zbar, logT_max, logT_min, logP_min, logP_max, dlogT, dlogP, &
            logT, T, logPgas, Pgas, logP, logRho, Prad, P, &
            Rho, logE, logS, chiRho, chiT, &
            Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
            mu, free_e, gamma1, gamma3, grad_ad, eta, E, S, entropy
         character (len=256) :: dir
         logical :: only_densities, DT_flag
         double precision :: &
            den_h,d_den_h_dlogP,d_den_h_dlogT, &
            ener_h,entr_h,dddt_cp_h,dddp_ct_h,dsdt_cp_h,dsdp_ct_h,dtdp_cs_h, &
            den_he,d_den_he_dlogP,d_den_he_dlogT, &
            ener_he,entr_he,dddt_cp_he,dddp_ct_he,dsdt_cp_he,dsdp_ct_he,dtdp_cs_he, &
            xnh,dxnh_dlogT,dxnh_dlogP, &
            xnh2,dxnh2_dlogT,dxnh2_dlogP, &
            xnhe,dxnhe_dlogT,dxnhe_dlogP, &
            xnhep,dxnhep_dlogT,dxnhep_dlogP, &
            dpdd, dsdd, dpdt, dedd, dsdt, dedt, &
            xmassh1,xmasshe4,logNe,logQ
	      integer, parameter :: io_unit0 = 40
         
         include 'formats.dek'
         
         write(*,*) 'make_PTplot_files'
	      
	      call setup_eos
         
         X = 1d0  ! 0.00d0, 0.20d0, 0.40d0, 0.60d0, 0.80d0
         Z = 1 - X
         

         call get_azbar(X, Z, abar, zbar)
                  
         logT_points = 300
         logP_points = 300
         
	      logT_max = 7.06d0
	      logT_min = 2.10d0
	      logP_min = -0.60d0
	      logP_max = 19d0
         
         if (.true.) then ! TESTING
            X = 0.7d0
            Z = 0.02d0
            logT_points = 200
            logP_points = 200
            
   	      logT_max = 3.46001d0
   	      logT_min = 3.45999d0
   	      logP_min = 11.99999d0
   	      logP_max = 12.00001d0
   	      
   	      logT_max = 5d0
   	      logT_min = 3.65d0
   	      logP_min = 9d0
   	      logP_max = 13.5d0
   	      
   	      logT_max = 5.4d0
   	      logT_min = 3.1d0
   	      logP_min = 4.5d0
   	      logP_max = 15d0

	      end if

	      io_params = io_unit0
	      io_logP = io_unit0+1
	      io_logT = io_unit0+2
	      io_first = io_unit0+3
	      
	      dir = 'plotPT_mesa_data'	      
	      write(*,*) 'output to ' // trim(dir)

         call Open_Plot_Outfiles(io_first, io_last, io_params, io_logP, io_logT, dir)
         write(io_params, '(2(f10.6),2(i7))') Z, X, logP_points, logT_points
         close(io_params)
			num_vals  = io_last - io_first + 1
			allocate(output_values(logP_points,logT_points,num_vals))
			
	      dlogT = (logT_max - logT_min)/(logT_points-1)
	      dlogP = (logP_max - logP_min)/(logP_points-1)

	      xmassh1 = X
	      xmasshe4 = 1-xmassh1
         only_densities = .true.


         if (.not. scvh_only) then ! make call to get things initialized
            logT = 5
            T = 10**logT
            logRho = 0
            Rho = 10**logRho
            call helm_opal_scvh( &
               helm_only, opal_scvh_only, opal_only, scvh_only, search_for_SCVH, &
   				include_radiation, logT, logRho, T, Rho, abar, zbar, X, Z, &
               logPgas, logE, logS, chiRho, chiT, Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
               mu, free_e, gamma1, gamma3, grad_ad, eta, &
               data_dir, ierr)			      
   	      if (ierr /= 0) stop 'failed in helm_opal_scvh'    
	      else
            call setup_scvh(data_dir)
	      end if           
      
         do j=1, logT_points
            logT = logT_min + dlogT*(j-1)
            T = 10 ** logT

		      do i=1,logP_points
		         logP = logP_min + dlogP*(i-1)
		         P = 10**logP
               
               if (.not. scvh_only) then

                  call interp_vals(.false., search_for_SCVH, &
                     den_h,d_den_h_dlogP,d_den_h_dlogT, &
                     ener_h,entr_h,dddt_cp_h,dddp_ct_h,dsdt_cp_h,dsdp_ct_h,dtdp_cs_h, &
                     den_he,d_den_he_dlogP,d_den_he_dlogT, &
                     ener_he,entr_he,dddt_cp_he,dddp_ct_he,dsdt_cp_he,dsdp_ct_he,dtdp_cs_he, &
                     xnh,dxnh_dlogT,dxnh_dlogP, &
                     xnh2,dxnh2_dlogT,dxnh2_dlogP, &
                     xnhe,dxnhe_dlogT,dxnhe_dlogP, &
                     xnhep,dxnhep_dlogT,dxnhep_dlogP, &
                     logT,logP,P,ierr)			      
   			      if (ierr /= 0) stop 'failed in interp_vals'          
			           
                  Rho = 1.0/(xmassh1/den_h + xmasshe4/den_he)
                  logRho = log10(Rho)

                  call helm_opal_scvh(helm_only, opal_scvh_only, opal_only, &
                        scvh_only, search_for_SCVH, &
         					include_radiation, logT, logRho, T, Rho, abar, zbar, X, Z, &
                        logPgas, logE, logS, chiRho, chiT, Cp, Cv, dE_dRho, dS_dT, dS_dRho, &
                        mu, free_e, gamma1, gamma3, grad_ad, eta, &
                        data_dir, ierr)			      
   			      if (ierr /= 0) stop 'failed in helm_opal_scvh'
   			   
   			   else
   			   
                  call interp_vals(.false., search_for_SCVH, &
                     den_h,d_den_h_dlogP,d_den_h_dlogT, &
                     ener_h,entr_h,dddt_cp_h,dddp_ct_h,dsdt_cp_h,dsdp_ct_h,dtdp_cs_h, &
                     den_he,d_den_he_dlogP,d_den_he_dlogT, &
                     ener_he,entr_he,dddt_cp_he,dddp_ct_he,dsdt_cp_he,dsdp_ct_he,dtdp_cs_he, &
                     xnh,dxnh_dlogT,dxnh_dlogP, &
                     xnh2,dxnh2_dlogT,dxnh2_dlogP, &
                     xnhe,dxnhe_dlogT,dxnhe_dlogP, &
                     xnhep,dxnhep_dlogT,dxnhep_dlogP, &
                     logT,logP,P,ierr)			      
   			      if (ierr /= 0) stop 'failed in interp_vals'          
			           
                  Rho = 1.0/(xmassh1/den_h + xmasshe4/den_he)
                  logRho = log10(Rho)

                     
                  dddt_cp_h  = Rho / T     * dddt_cp_h
                  dddp_ct_h  = Rho / P    * dddp_ct_h
                  dsdt_cp_h  = entr_h / T  * dsdt_cp_h
                  dsdp_ct_h  = entr_h / P * dsdp_ct_h
                  dtdp_cs_h  = T / P    * dtdp_cs_h

                  ! store the output
                  !..d(P)/ d(den)|t
                  dpdd  = 1.0/dddp_ct_h

                  !..d(entr)/d(den)|t
                  dsdd = dddt_cp_h/(Rho*Rho*dddp_ct_h)

                  !..d(P)/d(temp)|d
                  dpdt = -Rho*Rho*dsdd

                  !..d(ener)/d(rho)|t
                  dedd = (P - T * dpdt)/Rho/Rho

                  !..d(entr)/d(temp)|d
                  dsdt = dsdp_ct_h * dpdt + dsdt_cp_h

                  !..d(ener)/d(temp)|d
                  dedt = T * dsdt

                  logE = log10(ener_h)
                  logS = log10(entr_h)
            
                  gamma3 = 1 + dpdt / (Rho * dedt)
                  grad_ad = dtdp_cs_h/(T/P)
                  gamma1 = (-Rho*Rho*dsdd)/(Rho*dedt)/grad_ad ! C&G 9.88 & 9.89
                  gamma1 = -dddt_cp_h/(dddp_ct_h*dedt*Rho*grad_ad)
      
                  chiRho = dpdd * Rho / P
                  chiT = dpdt * T / P
      
                  Cv = chiT * P / (rho * T * (gamma3 - 1)) ! C&G 9.93
                  Cp = Cv + P * chiT**2 / (Rho * T * chiRho) ! C&G 9.86
      
                  dE_dRho = dedd
      
                  dS_dT = dsdt
                  dS_dRho = dsdd


			      end if            
			      
			      !call show_stuff
			      !stop 'PTplots'
			      
			      call check_num(logRho)
			      call check_num(logE)
			      call check_num(logS)
			      call check_num(chiRho)
			      call check_num(chiT)
			      call check_num(Cp)
			      call check_num(Cv)
			      call check_num(dE_dRho)
			      call check_num(dS_dT)
			      call check_num(dS_dRho)
			      call check_num(mu)
			      call check_num(gamma1)
			      call check_num(gamma3)
			      call check_num(grad_ad)
			      call check_num(eta)
			      
			      logQ = logRho - 2*logT + 12
			      
			      k = 0
			      k = k+1; output_values(i,j,k) = logRho
			      k = k+1; output_values(i,j,k) = logE
			      entropy = (10**logS)*amu/kerg
			      if (is_bad_num(entropy)) then
			         write(*,1) 'entropy', entropy
			         write(*,1) 'amu', amu
			         write(*,1) 'kerg', kerg
			         write(*,1) 'logS', logS
			         write(*,1) 'logP', logP
			         write(*,1) 'logT', logT
			         write(*,1) 'logRho', logRho
			         stop 'PT plots'
			      end if
			      k = k+1; output_values(i,j,k) = entropy
			      k = k+1; output_values(i,j,k) = chiRho
			      k = k+1; output_values(i,j,k) = chiT
			      k = k+1; output_values(i,j,k) = safe_log10(Cp)
			      k = k+1; output_values(i,j,k) = safe_log10(Cv)
			      E = 10**logE
			      S = 10**logS
			      k = k+1; output_values(i,j,k) = dE_dRho*Rho/E
			      k = k+1; output_values(i,j,k) = dS_dT*T/S
			      k = k+1; output_values(i,j,k) = dS_dRho*Rho/S
			      k = k+1; output_values(i,j,k) = mu
			      k = k+1; output_values(i,j,k) = gamma1
			      k = k+1; output_values(i,j,k) = gamma3
			      k = k+1; output_values(i,j,k) = grad_ad
			      k = k+1; output_values(i,j,k) = eta
			      k = k+1; output_values(i,j,k) = logQ
            
	         enddo
         
	      enddo
	
!$OMP PARALLEL DO PRIVATE(k)
			do k = 1, num_vals
				write(*,*) k
				write(io_first+k-1,'(e14.6)') output_values(1:logP_points,1:logT_points,k)
			end do
!$OMP END PARALLEL DO

			do i = 1, logT_points
            logT = logT_min + dlogT*(i-1)
            write(io_logT,*) logT
			end do
	      close(io_logT)
      
	      do j=1,logP_points
	         logP = logP_min + dlogP*(j-1)
	         write(io_logP,*) logP
			end do
	      close(io_logP)
	
	      do io=io_first,io_last
	         close(io)
	      end do
	
			deallocate(output_values)
			
			contains
			
			subroutine check_num(x)
			   double precision, intent(inout) :: x
			   if (is_bad_num(x)) x = 0
			end subroutine check_num
      
			subroutine show_stuff
			   include 'formats.dek'
            write(*,1) 'T', T
            write(*,1) 'logT', logT
            write(*,1) 'Rho', Rho
            write(*,1) 'logRho', logRho
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,1) 'X', X
            write(*,1) 'Z', Z
            write(*,*)
         
            Prad = crad*T**4/3
            Pgas = 10**logPgas
            P = Pgas + Prad
         
            write(*,1) 'P', P
            write(*,1) 'logP', log10(P)
            write(*,1) 'Pgas', Pgas
            write(*,1) 'logPgas', logPgas
            write(*,1) 'logE', logE
            write(*,1) 'E', 10**logE
            write(*,1) 'logS', logS
            write(*,1) 'chiRho', chiRho
            write(*,1) 'chiT', chiT
            write(*,1) 'Cp', Cp
            write(*,1) 'Cv', Cv
            write(*,1) 'dE_dRho', dE_dRho
            write(*,1) 'dS_dT', dS_dT
            write(*,1) 'dS_dRho', dS_dRho
            write(*,1) 'mu', mu
            write(*,1) 'lnfree_e', safe_log10(free_e)
            write(*,1) 'gamma1', gamma1
            write(*,1) 'gamma3', gamma3
            write(*,1) 'grad_ad', grad_ad
            write(*,1) 'eta', eta

            write(*,*)
			end subroutine show_stuff

      end subroutine make_PTplot_files


      subroutine Open_Plot_Outfiles(io_first, io_last, io_params, io_logP, io_logT, dir)
         integer, intent(in) :: io_first, io_params, io_logP, io_logT
         integer, intent(out) :: io_last
         character (len=*), intent(in) :: dir
         character (len=256) :: fname
         integer :: io
         
         fname = trim(dir) // '/params.data'
         open(unit=io_params,file=trim(fname))
         
         fname = trim(dir) // '/logP.data'
         open(unit=io_logP,file=trim(fname))
         
         fname = trim(dir) // '/logT.data'
         open(unit=io_logT,file=trim(fname))
         
         io = io_first-1

         fname = trim(dir) // '/logRho.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/logE.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/entropy.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/chiRho.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/chiT.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/logCp.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/logCv.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/dlnE_dlnRho.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/dlnS_dlnT.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/dlnS_dlnRho.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/mu.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/gamma1.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/gamma3.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/grad_ad.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/eta.data'
         io = io+1; open(unit=io,file=trim(fname))

         fname = trim(dir) // '/logQ.data'
         io = io+1; open(unit=io,file=trim(fname))

         io_last = io
      
      end subroutine Open_Plot_Outfiles
		


      end module make_PTplots
