      module mod_test_diffusion
      use diffusion_lib
      use chem_def
      use chem_lib
      use eos_def
      use eos_lib
      use kap_def
      use kap_lib
      use const_def
      use utils_lib
      use num_lib

      implicit none
      
      !logical, parameter :: use_solar_model = .true.
      logical, parameter :: use_solar_model = .false.

      !logical, parameter :: save_plot_info = .true.
      logical, parameter :: save_plot_info = .false.
            
         
      integer :: nz ! number of zones
      integer :: nzlo, nzhi

      integer :: species ! number of species of ions
      integer :: h1, he4, c12, n14, o16, ne20, mg24 ! species numbers
      
      integer :: c_h, c_he, c_o, c_fe ! class numbers
      integer :: nc ! number of classes of ions
      integer :: m ! number of fluids, nc+1

      integer, pointer :: chem_id(:) ! maps species to chem id
      integer, pointer :: class(:) ! maps species to class number
      integer, pointer :: class_chem_id(:) ! maps class number to "tyical" chem id
      integer, pointer :: net_iso(:) ! maps chem id to species
      
      double precision, pointer :: abar(:) ! (nz)
      double precision, pointer :: zbar(:) ! (nz)
      double precision, pointer :: z2bar(:) ! (nz)
      double precision, pointer :: ye(:) ! (nz)
      
      double precision, pointer :: rho(:) ! (nz)
      double precision, pointer :: lnd(:) ! (nz)
      double precision, pointer :: T(:) ! (nz)
      double precision, pointer :: lnT(:) ! (nz)
      double precision, pointer :: P(:) ! (nz)
      double precision, pointer :: lnP(:) ! (nz)
      double precision, pointer :: r(:) ! (nz)
      double precision, pointer :: lnR(:) ! (nz)
      
      double precision, pointer :: grav(:) ! (nz)
      double precision, pointer :: gamma(:) ! (nz)
      double precision, pointer :: mu(:) ! (nz)
      double precision, pointer :: free_e(:) ! (nz)

      double precision, pointer :: gradr(:) ! (nz)
      double precision, pointer :: lgkap(:) ! (nz)

      double precision :: mstar
      double precision, pointer :: q(:) ! (nz)
      double precision, pointer :: dm(:) ! (nz)
      double precision, pointer :: cell_mass(:) ! (nz)
      
      double precision, pointer :: dlnP_dm(:) ! (nz)
      double precision, pointer :: dlnT_dm(:) ! (nz)
      double precision, pointer :: dlnRho_dm(:) ! (nz)
      double precision, pointer :: L(:) ! (nz)
      
      double precision, pointer :: v(:,:) ! v(i,k) is velocity of species i at point k [cm/sec]
      double precision, pointer :: vgt(:,:) ! vgt(i,k) is gravothermal settling part of v(i,k)
      double precision, pointer :: typical_charge(:,:)

      double precision, pointer :: AP(:,:)
      double precision, pointer :: AT(:,:)
      double precision, pointer :: AX(:,:,:)

      double precision, pointer :: dlnP_dr_mid(:)
      double precision, pointer :: dlnT_dr_mid(:)
      double precision, pointer :: dlnRho_dr_mid(:)
      double precision, pointer :: dlnC_dr_mid(:,:)
      
      double precision, pointer :: xa(:,:) ! mass fractions (species,nz)
      double precision, pointer :: xa_init(:,:)
      double precision, pointer :: dxa(:,:)
      
      double precision, pointer :: X_init(:,:)
      double precision, pointer :: X_final(:,:)
      double precision, pointer :: dX(:,:)

      integer :: eos_handle, kap_handle

      integer, parameter :: h1_solar = 1
      integer, parameter :: he4_solar = 3
      integer, parameter :: c12_solar = 2
      integer, parameter :: n14_solar = 4
      integer, parameter :: o16_solar = 5
      integer, parameter :: species_solar = 5
      integer, parameter :: nz_solar = 1268
      
      ! Columns in the Standard Model table
      double precision, dimension(nz_solar) :: &
         M_solar, R_solar, T_solar, D_solar, P_solar, L_solar, He3_solar
         ! Mass fraction in units of the solar mass
         ! Radius of the zone in units of one solar radius
         ! Temperature in units of deg (K)
         ! Density in units of g/cm^3
         ! Pressure in units of dyn/cm^2
         ! Luminosity fraction in units of the solar luminosity
         ! X(1H): the hydrogen mass fraction
         ! X(4He): the helium 4 mass fraction
         ! X(3He): the helium 3 mass fraction
         ! X(12C): the carbon 12 mass fraction
         ! X(14N): the nitrogen 14 mass fraction
         ! X(16O): the oxygen 16 mass fraction
      double precision, dimension(species_solar, nz_solar) :: XA_solar ! mass fractions

      double precision, pointer :: diffusion_factor(:)
      
      integer :: maxsteps_allowed, steps_used, total_num_retries
      double precision :: total_time, atol, rtol, AD_factor, AD_velocity, vgt_max, &
         gamma_full_on, gamma_full_off, T_full_on, T_full_off, &
         X_full_on, X_full_off, Y_full_on, Y_full_off
      logical :: calculate_ionization 
      character (len=8), pointer :: class_name(:)
      
      
      contains


      subroutine do_test_diffusion(ierr)
         use const_lib
         integer, intent(out) :: ierr
         
         integer :: i, j, k, nmid
         logical :: okay, use_cache
         character (len=256) :: data_dir
         
         1 format(a40,1pe26.16)
         2 format(a40,i6,1pe26.16)
         
         write(*,*)
         write(*,*) 'test_diffusion'
         write(*,*)

         ierr = 0
         
         call const_init
         call chem_init('../../data', 'isotopes.data_approx', ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'FATAL ERROR: failed in chem_init'
      	   stop 1
      	end if
         
         write(*,*) 'load eos data'
         use_cache = .true.
         data_dir = '../../data'
         call eos_init(data_dir, 'mesa', use_cache, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in eos_init'
            stop 1
         end if
         eos_handle = alloc_eos_handle(ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in alloc_eos_handle'
            stop 1
         end if

         ! default values
         maxsteps_allowed = 1000
         atol = 1d-3
         rtol = 1d-3
         AD_factor = 1
         AD_velocity = 1d-6
         vgt_max = 1d-5
         gamma_full_on = 2
         gamma_full_off = 20
         calculate_ionization = .true.

         X_full_on = 1d-5
         X_full_off = 5d-6
         Y_full_on = 1d-5
         Y_full_off = 5d-6
         T_full_off = 5d4
         T_full_on = 1d5
                  
         write(*,*) 'read diffusion.data'
         call read_diffusion_data(ierr)
         if (ierr /= 0) return
         write(*,*)
         
         !total_time = 1d9*secyer
         write(*,1) 'total_time', total_time
         write(*,1) 'total_time/secyer', total_time/secyer
         write(*,1) 'log10(total_time/secyer)', log10(total_time/secyer)
         write(*,*)
         
         write(*,*) 'call solve_diffusion'
         call solve_diffusion( &
            nz, species, nc, m, class, class_chem_id, net_iso, &
            abar, ye, free_e, mstar, dm, cell_mass, &
            T, lnT, rho, lnd, r, dlnP_dm, dlnT_dm, dlnRho_dm, L, &
            total_time, maxsteps_allowed, calculate_ionization, typical_charge, &
            atol, rtol, AD_factor, AD_velocity, vgt_max, &
            gamma, gamma_full_on, gamma_full_off, T_full_on, T_full_off, &
            X_full_on, X_full_off, Y_full_on, Y_full_off, &
            diffusion_factor, &
            xa, steps_used, total_num_retries, nzlo, nzhi, X_init, X_final, &
            AP, AT, AX, dlnP_dr_mid, dlnT_dr_mid, &
            dlnRho_dr_mid, dlnC_dr_mid, v, vgt, ierr )
         write(*,*) 'solve_diffusion ierr', ierr
         write(*,*) '                  nz', nz
         write(*,*) '                nzlo', nzlo
         write(*,*) '                nzhi', nzhi
         write(*,*) '                   n', nzhi-nzlo+1
         write(*,*) '             classes', nc
         write(*,*) '             species', species
         write(*,*) '          steps_used', steps_used
         write(*,*) '   total_num_retries', total_num_retries
            
         write(*,*)
         dX(:,:) = X_final(:,:) - X_init(:,:)
         k = nzlo
         do j=1,nc
            if (dX(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' dX', k, dX(j,k)
         end do
         write(*,*)
         k = (nzlo+nzhi)/2
         do j=1,nc
            if (dX(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' dX', k, dX(j,k)
         end do
         write(*,*)
         k = nzhi
         do j=1,nc
            if (dX(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' dX', k, dX(j,k)
         end do
         write(*,*)
         k = nzlo
         do j=1,nc
            if (X_final(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' X_final', k, X_final(j,k)
         end do
         write(*,*)
         k = (nzlo+nzhi)/2
         do j=1,nc
            if (X_final(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' X_final', k, X_final(j,k)
         end do
         write(*,*)
         k = nzhi
         do j=1,nc
            if (X_final(j,k) < 1d-20) cycle
            write(*,2) class_name(j) // ' X_final', k, X_final(j,k)
         end do
         write(*,*)
         
         if (save_plot_info) call write_diffusion_results
         
         call do_dealloc
         
         write(*, *) 'done'
         
      end subroutine do_test_diffusion 
      
      
      subroutine read_diffusion_data(ierr)
         integer, intent(out) :: ierr

         integer :: i, j, k
         integer, parameter :: iounit = 33
         double precision :: A
         character (len=iso_name_length) :: name
         
         ierr = 0
         open(iounit, file='diffusion.data', status='old', action='read')
         
         
         read(iounit, *, iostat=ierr) nz, nzlo, nzhi, species, nc, maxsteps_allowed
         if (ierr /= 0) then
            write(*, *) 'read1 failed'
            return
         end if

         m = nc+1

         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do i=1,species
            read(iounit, *, iostat=ierr) name
            if (ierr /= 0) then
               write(*, *) 'read2 failed'
               return
            end if
            chem_id(i) = get_nuclide_index(name)
            if (chem_id(i) <= 0) then
               write(*, *) 'read2 failed -- bad chem_id', i, chem_id(i), trim(name)
               return
            end if
         end do
         

         read(iounit, *, iostat=ierr) class(1:species)
         if (ierr /= 0) then
            write(*, *) 'read3 failed'
            return
         end if
         
         
         do i=1,nc
            read(iounit, *, iostat=ierr) name
            if (ierr /= 0) then
               write(*, *) 'read2 failed'
               return
            end if
            class_chem_id(i) = get_nuclide_index(name)
            if (class_chem_id(i) <= 0) then
               write(*, *) 'read4 failed -- bad class_chem_id', i, class_chem_id(i), trim(name)
               return
            end if
         end do
         
         do i=1,nc
            read(iounit, *, iostat=ierr) class_name(i)
            if (ierr /= 0) then
               write(*, *) 'read5 failed'
               return
            end if
         end do
         
         read(iounit, *) i
         calculate_ionization = (i == 1)
         diffusion_factor(:) = 1
         net_iso(:) = 0
         do i=1,species
            net_iso(chem_id(i)) = i
         end do
         h1 = net_iso(ih1)
         he4 = net_iso(ihe4)
         c12 = net_iso(ic12)
         n14 = net_iso(in14)
         o16 = net_iso(io16)
         ne20 = net_iso(ine20)
         mg24 = net_iso(img24)
         
         c_h = class(h1)
         c_he = class(he4)
         c_o = class(o16)
         c_fe = class(mg24)

         read(iounit, '(99(1pe26.16))', iostat=ierr) mstar, total_time, atol, rtol, &
               AD_factor, AD_velocity, vgt_max, gamma_full_on, gamma_full_off, &
               T_full_on, T_full_off, X_full_on, X_full_off, Y_full_on, Y_full_off
         if (ierr /= 0) then
            write(*, *) 'read6 failed'
            return
         end if

         do k=1,nz
            read(iounit, '(99(1pe26.16))', iostat=ierr) &
               gamma(k), abar(k), ye(k), free_e(k), dm(k), cell_mass(k), &
               T(k), lnT(k), Rho(k), lnd(k), r(k), &
               dlnP_dm(k), dlnT_dm(k), dlnRho_dm(k), L(k)
            if (ierr /= 0) then
               write(*, *) 'read7 failed'
               return
            end if
            read(iounit, '(99(1pe26.16))', iostat=ierr) xa(1:species, k)
            if (ierr /= 0) then
               write(*, *) 'read8 failed'
               return
            end if
         end do

         close(iounit)
         
         contains
         
         subroutine read_array1(ptr, sz)
            double precision, pointer :: ptr(:)
            integer, intent(in) :: sz
            integer :: k
            read(iounit, *)
            do k=1, sz
               read(iounit, *) ptr(k)
            end do
         end subroutine read_array1
         
         subroutine read_array1_int(ptr, sz)
            integer, pointer :: ptr(:)
            integer, intent(in) :: sz
            integer :: k
            read(iounit, *)
            do k=1, sz
               read(iounit, *) ptr(k)
            end do
         end subroutine read_array1_int
         
         subroutine read_array2(ptr, sz1, sz2)
            double precision, pointer :: ptr(:, :)
            integer, intent(in) :: sz1, sz2
            integer :: i, k
            read(iounit, *)
            do k=1, sz2
               read(iounit, *) ptr(1:sz1, k)
            end do
         end subroutine read_array2
         
      end subroutine read_diffusion_data
      
      
      subroutine do_alloc(ierr)
         integer, intent(out) :: ierr
         call do_arrays(.true., ierr)
      end subroutine do_alloc
      
      
      subroutine do_dealloc
         integer :: ierr
         call do_arrays(.false., ierr)      
      end subroutine do_dealloc
      
      
      subroutine do_arrays(alloc, ierr)
         logical, intent(in) :: alloc
         integer, intent(out) :: ierr
         ierr = 0
         
         call do1_int(chem_id, species)
         call do1_int(class, species)
         call do1_int(class_chem_id, nc)
         call do1_int(net_iso, num_chem_isos)
         allocate(class_name(nc))
         allocate(diffusion_factor(nc))
         
         call do1(abar, nz)
         call do1(zbar, nz)
         call do1(z2bar, nz)
         call do1(ye, nz)
         
         call do1(rho, nz)
         call do1(lnd, nz)
         call do1(T, nz)
         call do1(lnT, nz)
         call do1(P, nz)
         call do1(lnP, nz)
         call do1(r, nz)
         call do1(lnR, nz)

         call do1(grav, nz)
         call do1(gamma, nz)
         call do1(mu, nz)
         call do1(free_e, nz)
         
         call do1(gradr, nz)
         call do1(lgkap, nz)
         
         call do1(q, nz)
         call do1(dm, nz)
         call do1(cell_mass, nz)

         call do1(dlnP_dm, nz)
         call do1(dlnT_dm, nz)
         call do1(dlnRho_dm, nz)
         call do1(L, nz)
         
         call do2(v, nc, nz)
         call do2(vgt, nc, nz)
         call do2(typical_charge, nc, nz)
         
         call do2(AP, m, nz)
         call do2(AT, m, nz)
         call do3(AX, m, m, nz)
         
         call do1(dlnP_dr_mid, nz)
         call do1(dlnT_dr_mid, nz)
         call do1(dlnRho_dr_mid, nz)
         call do2(dlnC_dr_mid, m, nz)
         
         call do2(xa, species, nz)
         call do2(xa_init, species, nz)
         call do2(dxa, species, nz)
         
         call do2(X_init, nc, nz)
         call do2(X_final, nc, nz)
         call do2(dX, nc, nz)
         
         contains
         
         subroutine do1_int(ptr, sz)
            integer, pointer :: ptr(:)
            integer, intent(in) :: sz
            if (alloc) then
               allocate(ptr(sz), stat=ierr)
               if (ierr /= 0) stop 1
               ptr = 0
            else
               deallocate(ptr)
            end if
         end subroutine do1_int
         
         subroutine do1(ptr, sz)
            double precision, pointer :: ptr(:)
            integer, intent(in) :: sz
            if (alloc) then
               allocate(ptr(sz), stat=ierr)
               if (ierr /= 0) stop 1
               ptr = 0
            else
               deallocate(ptr)
            end if
         end subroutine do1
         
         subroutine do2(ptr, sz1, sz2)
            double precision, pointer :: ptr(:, :)
            integer, intent(in) :: sz1
            integer, intent(in) :: sz2
            if (alloc) then
               allocate(ptr(sz1, sz2), stat=ierr)
               if (ierr /= 0) stop 1
               ptr = 0
            else
               deallocate(ptr)
            end if
         end subroutine do2
         
         subroutine do3(ptr, sz1, sz2, sz3)
            double precision, pointer :: ptr(:, :, :)
            integer, intent(in) :: sz1
            integer, intent(in) :: sz2
            integer, intent(in) :: sz3
            if (alloc) then
               allocate(ptr(sz1, sz2, sz3), stat=ierr)
               if (ierr /= 0) stop 1
               ptr = 0
            else
               deallocate(ptr)
            end if
         end subroutine do3
         
      end subroutine do_arrays
      
      
      subroutine write_diffusion_results
         use utils_lib
         use const_def
         character (len=100) :: filename
         integer :: k, ierr, iounit, i
         double precision :: ad2vh, ad2vhe, ad2vo
         double precision :: tau0  
            ! = 6d13*secyer, characteristic solar diffusion time (seconds)
         1 format(a40,1pe26.16)
         
         filename = 'plot_data/diffusion.data'

         ierr = 0
         tau0 = 6d13*secyer
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            write(*, *) 'write diffusion results to ' // trim(filename)
            write(iounit, fmt='(99(a, 1x))', advance='no') 'r_div_R', 'mass', &
               'xa_h', 'xa_he', 'xa_c', 'xa_n', 'xa_o', 'xa_mg', &
               'X_h', 'X_he', 'X_o', 'X_fe', &
               'X_init_h', 'X_init_he', 'X_init_o', 'X_init_fe', &
               'dX_h', 'dX_he', 'dX_o', 'dX_fe', &
               'v_h', 'v_he', 'v_o', 'v_fe', 'ad2vh', 'ad2vhe', 'ad2vo', &
               'vgt_h', 'vgt_he', 'vgt_o', 'vgt_fe', &
               'lg_yrs_tau_h', 'lg_yrs_he', 'lg_yrs_tau_o', &
               'z_he', 'z_o', 'z_fe', &
               'AP_h', 'AT_h', 'AX_hh', 'AX_ho', &
               'dlnP_dr', 'dlnT_dr', 'dlnCh_dr', 'dlnCo_dr', &
               'AP_h_dlnP', 'AT_h_dlnT', 'AX_hh_dlnCh', 'AX_ho_dlnCo', &
               'lgT', 'lgd', 'lgP', 'lgR', 'dlnP_dm', 'dlnT_dm', 'dlnRho_dm', 'L', &
               'gamma', 'mu', 'free_e', 'ye', 'free_e_div_ye', 'abar', 'zbar', 'q', &
               'gradr', 'lgkap', 'dmass', 'k'
            write(iounit, *)
            do k=1,nzhi !nzlo, nzhi
               if (k > 1 .and. k < nz) then
                  ad2vh = abs(v(c_h, k))*abs(v(c_h, k-1) - 2*v(c_h, k) + v(c_h, k+1)) / &
                     (1d-50 + abs(v(c_h, k-1)) + 2*abs(v(c_h, k)) + abs(v(c_h, k+1)))
                  ad2vhe = abs(v(c_he, k))*abs(v(c_he, k-1) - 2*v(c_he, k) + v(c_he, k+1)) / &
                     (1d-50 + abs(v(c_he, k-1)) + 2*abs(v(c_he, k)) + abs(v(c_he, k+1)))
                  ad2vo = abs(v(c_o, k))*abs(v(c_o, k-1) - 2*v(c_o, k) + v(c_o, k+1)) / &
                     (1d-50 + abs(v(c_o, k-1)) + 2*abs(v(c_o, k)) + abs(v(c_o, k+1)))
               else
                  ad2vh = 0
                  ad2vhe = 0
                  ad2vo = 0
               end if 
               write(iounit, fmt='(99e24.10)', advance='no') r(k)/r(1), mstar*q(k)/Msun, &
                  xa(h1,k), xa(he4,k), xa(c12,k), xa(n14,k), xa(o16,k), xa(mg24,k), &
                  X_final(c_h, k), X_final(c_he, k), X_final(c_o, k), X_final(c_fe, k), &
                  X_init(c_h, k), X_init(c_he, k), X_init(c_o, k), X_init(c_fe, k), &
                  dX(c_h, k), dX(c_he, k), dX(c_o, k), dX(c_fe, k), &
                  !v(c_h, k) / (Rsun/tau0), & ! divide by Rsun/tau0 to get same units as Thoul's plots
                  !v(c_he, k) / (Rsun/tau0), &
                  !v(c_o, k) / (Rsun/tau0), &
                  v(c_h, k), v(c_he, k), v(c_o, k), v(c_fe, k), ad2vh, ad2vhe, ad2vo, &
                  vgt(c_h, k), vgt(c_he, k), vgt(c_o, k), vgt(c_fe, k), &
                  min(12d0,safe_log10(r(k)/max(1d-50,abs(v(c_h,k)))/secyer)), &
                  min(12d0,safe_log10(r(k)/max(1d-50,abs(v(c_he,k)))/secyer)), &
                  min(12d0,safe_log10(r(k)/max(1d-50,abs(v(c_o,k)))/secyer)), &
                  typical_charge(c_he, k), typical_charge(c_o, k), typical_charge(c_fe, k), &
                  AP(c_h,k), AT(c_h,k), AX(c_h,c_h,k), AX(c_h,c_o,k), &
                  dlnP_dr_mid(k), dlnT_dr_mid(k), dlnRho_dr_mid(k), dlnC_dr_mid(c_h,k), dlnC_dr_mid(c_o,k), &
                  AP(c_h,k)*dlnP_dr_mid(k) / (Rsun/tau0), AT(c_h,k)*dlnT_dr_mid(k) / (Rsun/tau0), &
                  AX(c_h,c_h,k)*dlnC_dr_mid(c_h,k) / (Rsun/tau0), AX(c_h,c_o,k)*dlnC_dr_mid(c_o,k) / (Rsun/tau0), &
                  lnT(k)/ln10, lnd(k)/ln10, lnP(k)/ln10, lnR(k)/ln10, &
                  dlnP_dm(k)*Msun, dlnT_dm(k)*Msun, dlnRho_dm(k)*Msun, L(k)/Lsun, &
                  gamma(k), mu(k), free_e(k), ye(k), free_e(k)/ye(k), abar(k), zbar(k), &
                  q(k), gradr(k), lgkap(k), &
                  dot_product(xa(:,k),v(:,k)), dble(k)
               write(iounit, *)
            end do
            close(iounit)
         else
            write(*, *) 'failed to open file ' // trim(filename)
         end if
         call free_iounit(iounit)      
      end subroutine write_diffusion_results


      end module mod_test_diffusion



      program test_diffusion
      use mod_test_diffusion
      implicit none
      integer :: ierr
      ierr = 0
      call do_test_diffusion(ierr)
      if (ierr /= 0) stop 1
      end program




