! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module diffusion_lib


! DIFFUSION handles gravitational settling and chemical and thermal diffusion.
! Radiative levitation is not supported.
! 
! Burger's equation is solved using a routine derived from Anne Thoul's code.
!       Anne A. Thoul, John N. Bahcall, and Abraham Loeb,
!       "Element Diffusion in the Solar Interior",
!       Ap.J. 421, 828-842 (1994)
! 
! Approximate pressure ionization based on Paquette et al.
!       C. Paquette, C. Pelletier, G. Fontaine, and G. Michaud,
!       "Diffusion in White Dwarfs: New Results and Comparative Study",
!       ApJ Supp. Series, 61:197-217, 1986.
! 
! Element diffusion equation is solved using a semi-implicit,
! finite difference scheme described by Iben and MacDonald.
!       Icko Iben and Jim MacDonald,
!       "The Effects of Diffusion Due to Gravity and Due to
!       Composition Gradients on the Rate of Hydrogen Burning
!       in a Cooling Degenerate Dwarf. I. The Case of a Thick
!       Helium Buffer Layer",
!       Ap.J. 296, 540-553 (1985)
! 
! A similar scheme has been used by Althaus and Benvenuto.
!       L.G. Althaus and O.G. Benvenuto,
!       "Diffusion in helium white dwarf stars",
!       MNRAS 317, 952-964 (2000)


      
      implicit none

      integer, parameter :: diffusion_min_nc = 4 ! minimum number of classes
         ! see subroutine get_min_number_of_classes.

      contains
      
      ! diffusion uses OpenMP parallel processing
      
      subroutine solve_diffusion( &
            nz, species, nc, m, class, class_chem_id, net_iso, &
            abar, ye, free_e, mstar, dm, cell_mass, &
            T, lnT, rho, lnd, r, dlnP_dm, dlnT_dm, dlnRho_dm, L, &
            total_time, maxsteps_allowed, calculate_ionization, typical_charge, &
            atol, rtol, AD_factor, AD_velocity, vgt_max, &
            gamma, gamma_full_on, gamma_full_off, T_full_on, T_full_off, &
            X_full_on, X_full_off, Y_full_on, Y_full_off, &
            diffusion_factor, &
            xa, steps_used, total_num_retries, nzlo, nzhi, &
            X_init, X_final, &
            AP, AT, AX, dlnP_dr_mid, dlnT_dr_mid, &
            dlnRho_dr_mid, dlnC_dr_mid, v, vgt, &
            ierr )
         use mod_diffusion, only: do_solve_diffusion
         
         integer, intent(in) :: nz, species, nc, m 
            ! nz = number of points in the model
            ! species = number of isotopes per point
            ! nc = number of classes of isotopes
            ! m = number of "fluids" for Berger's equation (=nc+1)
         
         ! diffusion velocities are calculated for classes of isotopes.
         integer, intent(in) :: class(species), class_chem_id(nc), net_iso(:)
            ! class(i) = class number for species i. class numbers from 1 to nc
            ! class_chem_id(j) = isotope id number from chem_def for "typical" member of class j
            ! net_iso(j) = species number for isotope with id = j,
            !            or 0 if iso j not in current set of species
         
         ! information about the structure of the model
         double precision, intent(in), dimension(nz) :: &
            abar, ye, free_e, dm, cell_mass, T, lnT, rho, lnd, r, dlnP_dm, dlnT_dm, dlnRho_dm, L
            ! abar -- average atomic weight (from chem_lib)
            ! ye -- average charge per baryon = proton fraction (from chem_lib)
            ! free_e -- mean number of free electrons per nucleon (from eos_lib)
            ! dm(k) is the mass (in grams) between points k and k+1
            ! cell_mass(k) is the mass associated with point k
            ! T, lnT, rho, lnd, r, L -- temperature, density, radius, luminosity
            ! dlnP_dm, dlnT_dm, dlnRho_dm -- gradients
         double precision, intent(in) :: mstar ! total mass (grams)
            
         ! controls
         double precision, intent(in) :: total_time ! evolve the abundances for this time (seconds)
         
         ! the code monitors the size of changes in class mass fractions using these tolerances.
         ! the size of the change for class mass fraction X(i,k) is defined as
         !     abs(X(i,k)-X_init(i,k))/(atol + rtol*max(X(i,k),X_init(i,k)))
         ! the average of these changes is required to be less than 1 for each substep,
         ! and the max is required to be less than 0.1/atol.
         ! if these bounds are not met, the substep is retried with a smaller dt.
         double precision, intent(in) :: atol
         double precision, intent(in) :: rtol
         
         double precision, intent(in) :: AD_factor ! controls use of artificial diffusion for smoothing.
         double precision, intent(in) :: AD_velocity ! velocity for mixing by artificial diffusion (cm/sec).
         
         double precision, intent(in) :: vgt_max ! limit on gravo-thermal settling velocity (cm/sec).
         
         ! gamma -- plasma coupling parameter.
         ! turn off diffusion where gamma > gamma_full_off
         ! limited diffusion where gamma > gamma_full_on
         ! normal diffusion where gamma <= gamma_full_on
         ! the diffusion code is valid up to gamma of order 1,
         ! but does not handle strong coupling with gamma >> 1.
         double precision, intent(in) :: gamma(nz) ! (e.g., from eos_lib function Plasma_Coupling_Parameter)
         double precision, intent(in) :: gamma_full_on
         double precision, intent(in) :: gamma_full_off
         
         ! full mixing at surface where T < T_full_off
         ! extra mixing from artificial diffusion where T < T_full_on
         double precision, intent(in) :: T_full_on
         double precision, intent(in) :: T_full_off
         
         ! X = mass fraction for hydrogen class; Y = mass fraction for helium class.
         ! turn off diffusion where X < X_full_off or Y < Y_full_off.
         ! extra mixing from artificial diffusion where X < X_full_on or Y < Y_full_on.
         double precision, intent(in) :: X_full_on, X_full_off
         double precision, intent(in) :: Y_full_on, Y_full_off

         integer, intent(in) :: maxsteps_allowed ! give up if solver needs more substeps than this
         
         logical, intent(in) :: calculate_ionization 
            ! if true, calls ionization module to get typical_charge
            ! if false, uses given values in typical_charge.
         double precision, intent(inout), dimension(nc,nz) :: typical_charge

         double precision, intent(in) :: diffusion_factor(nc) !arbitrarily enhance or inhibit by class
         
         ! in/out
         double precision, intent(inout) :: xa(species,nz) ! mass fractions of species
         integer, intent(inout) :: nzlo ! points outward from nzlo have been fully mixed
         integer, intent(inout) :: nzhi ! points inward from nzhi are unchanged
         
         ! work arrays made public FYI
         
         double precision, intent(out), dimension(nc,nz) :: X_init ! initial mass fraction by class
         double precision, intent(out), dimension(nc,nz) :: X_final ! final mass fraction by class
         double precision, intent(out), dimension(nc,nz) :: v ! diffusion velocity (cm/sec)
         double precision, intent(out), dimension(nc,nz) :: vgt ! gravo-thermal part of v (cm/sec)
            
         ! work arrays for Thoul computation of diffusion velocities
         double precision, intent(out), dimension(nz) :: dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid
         double precision, intent(out), dimension(m,nz) :: AP, AT, dlnC_dr_mid
         double precision, intent(out), dimension(m,m,nz) :: AX
         
         integer, intent(out) :: steps_used ! solver required this many substeps for the evolution
         integer, intent(out) :: total_num_retries ! solver did this many retries
         integer, intent(out) :: ierr ! nonzero in case of error
         
         integer :: k, j
         include 'formats.dek'
         
         if (.false.) then ! dump args for debugging
            write(*,2) 'nz', nz
            write(*,2) 'species', species
            write(*,2) 'nc', nc
            write(*,2) 'm', m
            do k=1,species
               write(*,3) 'class', k, class(k)
            end do
            do k=1,nc
               write(*,3) 'class_chem_id', k, class_chem_id(k)
            end do
            do k=1,size(net_iso,dim=1)
               if (net_iso(k) /= 0) write(*,3) 'net_iso', k, net_iso(k)
            end do
            do k=1,nz
               write(*,2) 'abar', k, abar(k)
               write(*,2) 'ye', k, ye(k)
               write(*,2) 'free_e', k, free_e(k)
               write(*,2) 'dm', k, dm(k)
               write(*,2) 'T', k, T(k)
               write(*,2) 'lnT', k, lnT(k)
               write(*,2) 'rho', k, rho(k)
               write(*,2) 'lnd', k, lnd(k)
               write(*,2) 'r', k, r(k)
               write(*,2) 'dlnP_dm', k, dlnP_dm(k)
               write(*,2) 'dlnT_dm', k, dlnT_dm(k)
               write(*,2) 'gamma', k, gamma(k)
            end do
            write(*,1) 'mstar', mstar
            write(*,1) 'total_time', total_time
            write(*,1) 'atol', atol
            write(*,1) 'rtol', rtol
            write(*,1) 'AD_factor', AD_factor
            write(*,1) 'AD_velocity', AD_velocity
            write(*,1) 'vgt_max', vgt_max
            write(*,1) 'gamma_full_on', gamma_full_on
            write(*,1) 'gamma_full_off', gamma_full_off
            write(*,1) 'T_full_on', T_full_on
            write(*,1) 'T_full_off', T_full_off
            write(*,1) 'X_full_on', X_full_on
            write(*,1) 'X_full_off', X_full_off
            write(*,1) 'Y_full_on', Y_full_on
            write(*,1) 'Y_full_off', Y_full_off
            write(*,2) 'maxsteps_allowed', maxsteps_allowed
            write(*,2) 'nzlo', nzlo
            write(*,2) 'nzhi', nzhi
            do k=1,nc
               if (diffusion_factor(k) /= 1) &
                  write(*,2) 'diffusion_factor', k, diffusion_factor(k)
            end do
            write(*,*) 'calculate_ionization', calculate_ionization
            do k=1,nz
               do j=1,species
                  write(*,3) 'xa', j, k, xa(j,k)
               end do
            end do
            write(*,*)
            stop 'diffusion args'
         end if
         
         call do_solve_diffusion( &
            nz, species, nc, m, class, class_chem_id, net_iso, abar, ye, free_e, mstar, dm, cell_mass, &
            T, lnT, rho, lnd, r, dlnP_dm, dlnT_dm, dlnRho_dm, L, &
            total_time, maxsteps_allowed, &
            calculate_ionization, typical_charge, &
            atol, rtol, AD_factor, AD_velocity, vgt_max, &
            gamma, gamma_full_on, gamma_full_off, T_full_on, T_full_off, &
            X_full_on, X_full_off, Y_full_on, Y_full_off, diffusion_factor, &
            xa, steps_used, total_num_retries, nzlo, nzhi, X_init, X_final, &
            AP, AT, AX, dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid, dlnC_dr_mid, v, vgt, ierr )
            
      end subroutine solve_diffusion
      
      
      ! this routine sets up tables for 4 classes: h, he, o, and fe.
      subroutine get_min_number_of_classes( &
            species, chem_id, class, class_chem_id, class_name)
         use chem_def
         integer, parameter :: nc = diffusion_min_nc
         integer, intent(in) :: species, chem_id(species)
         integer, intent(out) :: class(species), class_chem_id(nc)
         character (len=8), intent(out) :: class_name(nc)
         double precision :: A
         integer :: i, j
         integer, parameter :: c_h = 1, c_he = 2, c_o = 3, c_fe = 4
         class_name(c_h) = 'c_h'
         class_name(c_he) = 'c_he'
         class_name(c_o) = 'c_o'
         class_name(c_fe) = 'c_fe'        
         class_chem_id(c_h) = ih1
         class_chem_id(c_he) = ihe4
         class_chem_id(c_o) = io16
         class_chem_id(c_fe) = ife56        
         do i=1,species
            A = chem_isos% Z_plus_N(chem_id(i))
            if (A < 3) then
               class(i) = c_h
            else if (A < 12) then
               class(i) = c_he
            else if (A < 20) then
               class(i) = c_o
            else
               class(i) = c_fe
            end if
         end do      
      end subroutine get_min_number_of_classes
      
      
      ! this routine sets up tables with a separate class for each isotope.
      subroutine get_max_number_of_classes( &
            species, chem_id, class, class_chem_id, class_name)
         use chem_def
         integer, intent(in) :: species, chem_id(species)
         integer, intent(out) :: class(species), class_chem_id(species)
         character (len=8), intent(out) :: class_name(species)
         integer :: i, ci
         do i=1,species
            ci = chem_id(i)
            class_name(i) = 'c_' // trim(chem_isos% name(ci))
            class(i) = i
            class_chem_id(i) = ci
         end do
      end subroutine get_max_number_of_classes
      
      
      subroutine get_diffusion_classes( &
            nc, species, chem_id, class_chem_id, class_A_cutoff, &
            class, class_name)
         use chem_def
         integer, intent(in) :: nc, species, chem_id(species)
         integer, intent(in) :: class_chem_id(nc)
         double precision, intent(in) :: class_A_cutoff(nc)
         integer, intent(out) :: class(species)
         character (len=8), intent(out) :: class_name(nc)
         double precision :: A
         integer :: i, j
         do i=1,species
            A = chem_isos% Z_plus_N(chem_id(i))
            class(i) = nc
            do j=1,nc-1
               if (A < class_A_cutoff(j)) then
                  class(i) = j
                  exit
               end if
            end do
         end do      
         do j=1,nc
            class_name(j) = 'c_' // trim(chem_isos% name(class_chem_id(j)))
         end do
      end subroutine get_diffusion_classes
      
      
      subroutine set_diffusion_classes( &
            nc, species, chem_id, class_chem_id, class_A_max, &
            class, class_name)
         use chem_def
         integer, intent(in) :: nc, species, chem_id(species)
         integer, intent(in) :: class_chem_id(nc)
         double precision, intent(in) :: class_A_max(nc)
         integer, intent(out) :: class(species)
         character (len=8), intent(out) :: class_name(nc)
         double precision :: A
         integer :: i, j
         do i=1,species
            A = chem_isos% Z(chem_id(i)) + chem_isos% N(chem_id(i))
            class(i) = nc
            do j=1,nc-1
               if (A <= class_A_max(j)) then
                  class(i) = j
                  exit
               end if
            end do
         end do      
         do j=1,nc
            class_name(j) = 'c_' // trim(chem_isos% name(class_chem_id(j)))
         end do
      end subroutine set_diffusion_classes


      end module diffusion_lib

