! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! 
! ***********************************************************************

      module xmod_diffusion_support

      use const_def
      use chem_def
      use utils_lib
      use mod_diff_vel

      implicit none
      
      integer, parameter :: i_n = 1
      integer, parameter :: i_m = 2
      integer, parameter :: i_nc = 3
      integer, parameter :: num_ipars = 3

      integer, parameter :: num_rpars = 0
      
      integer :: ih, ihe
      double precision, pointer, dimension(:) :: A ! (m)
      double precision, pointer, dimension(:,:) :: Z ! (m,n)
      double precision, pointer, dimension(:,:) :: C, X, Xdot ! (m,n)
      double precision, pointer, dimension(:,:) :: matrix ! (ldab,neqns)
      double precision, pointer, dimension(:) :: alfa_vec ! (n)
      double precision :: tiny_C
      double precision, pointer, dimension(:) :: &
         rho_face, T_face, dlnP_dr_face, dlnT_dr_face, gamma_T_limit_face, &
         dm_dr_face, dm_face, cell_dm ! (n)
      double precision :: &
         dt, v_max, tiny_X, X_full_on, X_full_off, Y_full_on, Y_full_off
      double precision, pointer, dimension(:,:) :: xclass ! (nc,n)
      double precision, pointer, dimension(:,:) :: &
         X_face, Z_face, C_face ! (m,n)
      double precision, pointer, dimension(:) :: E_field_face ! (n)
      double precision, pointer, dimension(:,:) :: v_face ! (nc,n) 
      double precision, pointer, dimension(:,:,:) :: dv_dXm1, dv_dX00 ! (nc,nc,n)
      double precision, pointer, dimension(:,:) :: flow ! (nc,n)
      double precision, pointer, dimension(:,:,:) :: dflow_dXm1, dflow_dX00 ! (nc,nc,n)
      
      
      


      logical, parameter :: dbg = .false.      


      contains
      

      subroutine solve_diffusion_ode( &
            m, n, nc, maxsteps_allowed, atol_in, rtol_in, steps_used, ierr)
         integer, intent(in) :: n, m, nc, maxsteps_allowed
         double precision, intent(in) :: atol_in, rtol_in
         integer, intent(out) :: steps_used, ierr

         integer :: nv, lrpar, lipar, k
         
         ierr = 0         
         nv = nc*n
         lrpar = num_rpars
         lipar = num_ipars
         
         forall (k=1:n) xclass(1:nc,k) = X(1:nc,k)
         
         call do_radau5(ierr)
         
         if (ierr /= 0) then
            !write(*,*) 'failed in radau5'
         else
            forall (k=1:n) X(1:nc,k) = xclass(1:nc,k)
         end if
         
         
         contains
         
         
         subroutine do_radau5(ierr)
            use mtx_lib
            use num_lib
            integer, intent(out) :: ierr

            integer :: lrd, lid, lcd, lrdc, lidc, mljac, mujac, nzmax, &
               isparse, max_steps, itol, ijac, imas, mlmas, mumas, iout, lout, idid
            logical :: call_debug_routine            
            double precision :: rtol(1), atol(1), xstart, xend, h, max_step_size
            double precision, target :: rpar(lrpar)
            integer, target :: ipar(lipar)
            integer, pointer :: ipar_decsol(:) ! (lid)
            double precision, pointer :: rpar_decsol(:) ! (lrd)
            complex*16, pointer :: cpar_decsol(:) ! (lcd)
            double precision, pointer :: rpar_decsolc(:) ! (lrdc)
            integer, pointer :: ipar_decsolc(:) ! (lidc)
            integer :: radau5_lwork, radau5_liwork
            double precision, pointer :: radau5_work(:)
            integer, pointer :: radau5_iwork(:)
            
            include 'formats.dek'
         
            ierr = 0

            ipar(i_n) = n
            ipar(i_m) = m
            ipar(i_nc) = nc

            mljac = 2*nc-1
            mujac = mljac
            nzmax = 0
            isparse = -1
                     
            xstart = 0
            xend = dt
            h = dt  ! initial stepsize
            max_step_size = 0 ! 0 means let radau5 decide
            max_steps = maxsteps_allowed

            itol = 0
            atol(1) = atol_in
            rtol(1) = rtol_in
         
            ijac = 1 ! call routine to compute jacobian
            imas = 0 ! identity matrix
            mlmas = 0
            mumas = 0         
            iout = 0 ! 1 means call solout for each step         
            lout = 0
         
            call_debug_routine = .false.

            call lapack_work_sizes(nv, lrd, lid)
            call lapack_zwork_sizes(nv, mujac, mljac, lcd, lrdc, lidc)

            call radau5_work_sizes( &
               nv, nzmax, mljac, mujac, mlmas, mumas, &
               radau5_liwork, radau5_lwork)

            allocate( &
               ipar_decsol(lid), rpar_decsol(lrd), &
               cpar_decsol(lcd), rpar_decsolc(lrdc), ipar_decsolc(lidc), &
               radau5_iwork(radau5_liwork), radau5_work(radau5_lwork), stat=ierr)
               
            if (ierr /= 0) then
               write(*,*) 'allocate ierr', ierr
               return
            end if
            
            radau5_iwork = 0
            radau5_work = 0
            
            radau5_iwork(4) = 1 ! don't extrapolate for initial guess
            
            call radau5( &
               nv, diffusion_fcn, xstart, xclass, xend, &
               h, max_step_size, max_steps, &
               rtol, atol, itol, &
               diffusion_jacob, ijac, null_sjac, nzmax, isparse, &
               mljac, mujac, &
               null_mas, imas, mlmas, mumas, &
               diffusion_solout, iout, &
               null_radau5_debug_routine, call_debug_routine, &
               lapack_decsol, null_decsols, &
               lrd, rpar_decsol, lid, ipar_decsol, &
               lapack_decsolc, null_decsolcs, &
               lcd, cpar_decsol, lrdc, rpar_decsolc, lidc, ipar_decsolc, &
               radau5_work, radau5_lwork, radau5_iwork, radau5_liwork, &
               lrpar, rpar, lipar, ipar, &
               lout, idid)
               
            !write(*,2) 'radau5 idid', idid
            
            if (idid <= 0) ierr = -1
         
            steps_used = radau5_iwork(16) ! number of computed steps
         
            deallocate( &
               ipar_decsol, rpar_decsol, &
               cpar_decsol, rpar_decsolc, ipar_decsolc, &
               radau5_iwork, radau5_work)
         
         end subroutine do_radau5
         
         
      end subroutine solve_diffusion_ode


      subroutine diffusion_solout( &
            nr, xold, x, n, y, rwork_y, iwork_y, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         ! nr is the step number.
         ! x is the current x value; xold is the previous x value.
         ! y is the current y value.
         ! irtrn negative means terminate integration.
         ! rwork_y and iwork_y hold info for interp_y
         ! note that these are not the same as the rwork and iwork arrays for the solver.
         integer, intent(in) :: nr, n, lrpar, lipar
         double precision, intent(in) :: xold, x
         double precision, intent(inout) :: y(n)
         ! y can be modified if necessary to keep it in valid range of possible solutions.
         double precision, intent(inout), target :: rpar(lrpar), rwork_y(*)
         integer, intent(inout), target :: ipar(lipar), iwork_y(*)
         interface
            include 'num_interp_y.dek'
         end interface
         integer, intent(out) :: irtrn ! < 0 causes solver to return to calling program.
         
         include 'formats.dek'
         
         irtrn = 0
         
         write(*,2) 'diffusion_solout step', nr, x/dt, x, dt
         
         
      end subroutine diffusion_solout


      subroutine diffusion_fcn(num_isos, t, x, f, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: num_isos, lrpar, lipar
         double precision, intent(in) :: t
         double precision, intent(inout) :: x(num_isos)
         double precision, intent(out) :: f(num_isos) ! dxdt
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr
         integer, parameter :: ld_dfdx = 0
         double precision :: dfdx(ld_dfdx,num_isos)
         ierr = 0
         call diffusion_jacob(num_isos, t, x, f, dfdx, ld_dfdx, lrpar, rpar, lipar, ipar, ierr)
      end subroutine diffusion_fcn


      subroutine diffusion_jacob( &
            nv, time, v, f, dfdv, ld_dfdv, lrpar, rpar, lipar, ipar, ierr)
         
         integer, intent(in) :: nv, ld_dfdv, lrpar, lipar
         double precision, intent(in) :: time
         double precision, intent(inout) :: v(nv)
         double precision, intent(out) :: f(nv), dfdv(ld_dfdv, nv)
         double precision, intent(inout), target :: rpar(lrpar)
         integer, intent(inout), target :: ipar(lipar)
         integer, intent(out) :: ierr         
         
         integer :: n, m, nc, neqs, idiag, ldab, i, k
         double precision :: dt
         logical :: skip_partials
         double precision, pointer, dimension(:,:) :: xa ! (nc,n)

         include 'formats.dek'
         
         ierr = 0
         f = 0
         dfdv = 0
         
         if (lipar /= num_ipars) then
            write(*,*) 'bad lipar for diffusion_jacob', lipar, num_ipars
            stop 'diffusion_jacob'
            ierr = -1
            return
         end if
         
         n = ipar(i_n)
         m = ipar(i_m)
         nc = ipar(i_nc)
         neqs = nc*n
         
         ldab = ld_dfdv
         idiag = ldab - 2*nc + 1
         
         if (neqs /= nv .or. nc*n /= nv) then
            write(*,*) 'bad nv for diffusion_jacob', nv, neqs
            stop 'diffusion_jacob'
            ierr = -1
            return
         end if
         
         call set_pointer_2(xa, v, nc, n)
         call set_pointer_2(Xdot, f, nc, n)
         call set_pointer_2(matrix, dfdv, ld_dfdv, neqs)
         
         forall (k=1:n) X(1:nc,k) = xa(1:nc,k)
         
         skip_partials = (ldab == 0)
         
         call eval_eqn(m, n, nc, skip_partials, neqs, idiag, ldab, ierr)
         
      end subroutine diffusion_jacob
      

      subroutine eval_eqn(m, n, nc, skip_partials, neqs, idiag, ldab, ierr)
         integer, intent(in) :: n, m, nc, neqs, idiag, ldab
         logical, intent(in) :: skip_partials
         integer, intent(out) :: ierr

         integer :: k, j, jj, i
         double precision :: alfa, beta, X_face, dm
         include 'formats.dek'
         ierr = 0
         call eval_velocities_and_flow_coeffs(m, n, nc, skip_partials, ierr)
         if (ierr /= 0) return
         if (.not. skip_partials) matrix(:,:) = 0         
!$OMP PARALLEL DO PRIVATE(k,j,i,alfa,beta,X_face,dm)
         do k=1,n
            Xdot(:,k) = 0
            dm = cell_dm(k)
            if (k > 1) then ! flow out of cell
               alfa = alfa_vec(k)
               beta = 1 - alfa
               do j=1,nc
                  X_face = alfa*X(j,k-1) + beta*X(j,k)
                  Xdot(j,k) = Xdot(j,k) - flow(j,k)*X_face/dm
                  if (.not. skip_partials) then
                     call dXdot_dXm1(j, j, k, -flow(j,k)*alfa/dm)
                     call dXdot_dX00(j, j, k, -flow(j,k)*beta/dm)
                     do i=1,nc
                        call dXdot_dXm1(j, i, k, -dflow_dXm1(j,i,k)*X_face/dm)
                        call dXdot_dX00(j, i, k, -dflow_dX00(j,i,k)*X_face/dm)
                     end do
                  end if
               end do
            end if
            if (k < n) then ! flow into cell
               alfa = alfa_vec(k+1)
               beta = 1 - alfa
               do j=1,nc
                  X_face = alfa*X(j,k) + beta*X(j,k+1)
                  Xdot(j,k) = Xdot(j,k) + flow(j,k+1)*X_face/dm
                  if (.not. skip_partials) then
                     call dXdot_dX00(j, j, k, flow(j,k+1)*alfa/dm)
                     call dXdot_dXp1(j, j, k, flow(j,k+1)*beta/dm)
                     do i=1,nc
                        call dXdot_dX00(j, i, k, dflow_dXm1(j,i,k+1)*X_face/dm)
                        call dXdot_dXp1(j, i, k, dflow_dX00(j,i,k+1)*X_face/dm)
                     end do
                  end if
               end do
            end if
         end do
!$OMP END PARALLEL DO

         contains
      
         ! dXdot_dX00(i,j,k) is partial of Xdot(i,k) wrt X(j,k)
         subroutine dXdot_dX00(i,j,k,v)
            integer, intent(in) :: i, j, k
            double precision, intent(in) :: v
            integer :: b, q, v00
            b = nc*(k-1)
            q = idiag + b + i
            v00 = b + j
            matrix(q-v00,v00) = matrix(q-v00,v00) + v
         end subroutine dXdot_dX00
      
         ! dXdot_dXm1(i,j,k) is partial of Xdot(i,k) wrt X(j,k-1)
         subroutine dXdot_dXm1(i,j,k,v)
            integer, intent(in) :: i, j, k
            double precision, intent(in) :: v
            integer :: b, q, vm1
            b = nc*(k-1)
            q = idiag + b + i
            vm1 = b + j - nc
            matrix(q-vm1,vm1) = matrix(q-vm1,vm1) + v
         end subroutine dXdot_dXm1  
      
         ! dXdot_dXp1(i,j,k) is partial of Xdot(i,k) wrt X(j,k+1)
         subroutine dXdot_dXp1(i,j,k,v)
            integer, intent(in) :: i, j, k
            double precision, intent(in) :: v
            integer :: b, q, vp1
            b = nc*(k-1)
            q = idiag + b + i
            vp1 = b + j + nc
            matrix(q-vp1,vp1) = matrix(q-vp1,vp1) + v
         end subroutine dXdot_dXp1

      end subroutine eval_eqn
      

      subroutine eval_velocities_and_flow_coeffs(m, n, nc, skip_partials, ierr)
         integer, intent(in) :: n, m, nc
         logical, intent(in) :: skip_partials
         integer, intent(out) :: ierr

         integer :: k, im, j, i, jj, op_err, maxlocm1(2), maxloc00(2)
         double precision :: dv_im, tmp
         include 'formats.dek'
         ierr = 0
         ! concentration C is number density of ion divided by number density of free electrons
         do k=1,n
            tmp = sum(Z(1:nc,k)*X(1:nc,k)/A(1:nc)) ! number density of free electrons
            forall (j=1:nc) C(j,k) = X(j,k)/(A(j)*tmp)
            C(m,k) = 1
         end do
!$OMP PARALLEL DO PRIVATE(k,i,op_err)
         do k = 2, n
            op_err = 0
            call eval1_v(k, nc, m, op_err)
            if (op_err /= 0) ierr = op_err
            flow(1:nc,k) = dm_dr_face(k)*v_face(1:nc,k)
         end do
!$OMP END PARALLEL DO
         if (.not. skip_partials) then
!$OMP PARALLEL DO PRIVATE(k,i,op_err)
            do k = 2, n, 2 ! get partials for k even using unmodified k odd
               op_err = 0
               call eval1_partials(k, nc, m, n, op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
!$OMP PARALLEL DO PRIVATE(k,i,op_err)
            do k = 1, n, 2 ! get partials for k odd using unmodified k even
               op_err = 0
               call eval1_partials(k, nc, m, n, op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
            if (.false.) then
               do k = 1, n
                  maxlocm1 = maxloc(abs(dflow_dXm1(:,:,k)))
                  maxloc00 = maxloc(abs(dflow_dX00(:,:,k)))
                  write(*,2) 'max abs dflow_dXm1, dflow_dX00', k, &
                     dflow_dXm1(maxlocm1(1),maxlocm1(2),k), &
                     dflow_dX00(maxloc00(1),maxloc00(2),k)
               end do
            end if
         end if
         v_face(:,1) = 0 ! not used
         dv_dXm1(:,:,1) = 0
         dv_dX00(:,:,1) = 0
         flow(:,1) = 0
         dflow_dXm1(:,:,1) = 0 
         dflow_dX00(:,:,1) = 0 
         E_field_face(1) = E_field_face(2)
         
      end subroutine eval_velocities_and_flow_coeffs
      
      
      subroutine eval1_partials(k, nc, m, n, ierr)
         integer, intent(in) :: nc, m, n, k
         integer, intent(out) :: ierr
         
         double precision, dimension(m) :: C_std, X_std
         double precision, dimension(m) :: C_face_00, X_face_00
         double precision, dimension(m) :: C_face_p1, X_face_p1
         double precision, dimension(nc) :: v_face_00, v_face_p1
         double precision :: E_field_face_00, E_field_face_p1, tmp, dX
         integer :: i, j
         double precision, parameter :: del_frac = 1d-7
         include 'formats.dek'
         
         ierr = 0
         
         X_std = X(:,k)
         C_std = C(:,k)
         
         v_face_00 = v_face(:,k)
         E_field_face_00 = E_field_face(k)
         C_face_00 = C_face(:,k)
         X_face_00 = X_face(:,k)
         if (k < n) then
            v_face_p1 = v_face(:,k+1)
            E_field_face_p1 = E_field_face(k+1)
            C_face_p1 = C_face(:,k+1)
            X_face_p1 = X_face(:,k+1)
         end if
         
         do i=1,nc ! calculate partials wrt X(i,k)
            dX = del_frac
            if (X(i,k) > 0.9999d0) dX = -dX
            X(i,k) = X(i,k) + dX
            X(1:nc,k) = X(1:nc,k)/sum(X(1:nc,k))
            tmp = sum(Z(1:nc,k)*X(1:nc,k)/A(1:nc)) ! number density of free electrons
            forall (j=1:nc) C(j,k) = X(j,k)/(A(j)*tmp)
            if (k > 1) then
               call eval1_v(k, nc, m, ierr)
               if (ierr /= 0) return
               dv_dX00(:,i,k) = (v_face(:,k) - v_face_00)/dX
               dflow_dX00(:,i,k) = dm_dr_face(k)*dv_dX00(:,i,k)
            else
               dv_dX00(:,i,k) = 0
               dflow_dX00(:,i,k) = 0
            end if
            if (k < n) then
               call eval1_v(k+1, nc, m, ierr)
               if (ierr /= 0) return
               dv_dXm1(:,i,k+1) = (v_face(:,k+1) - v_face_p1)/dX
               dflow_dXm1(:,i,k+1) = dm_dr_face(k+1)*dv_dXm1(:,i,k+1)
            end if
            X(:,k) = X_std(:)
            C(:,k) = C_std(:)
         end do         
         
         v_face(:,k) = v_face_00
         E_field_face(k) = E_field_face_00
         C_face(:,k) = C_face_00
         X_face(:,k) = X_face_00
         if (k < n) then
            v_face(:,k+1) = v_face_p1
            E_field_face(k+1) = E_field_face_p1
            C_face(:,k+1) = C_face_p1
            X_face(:,k+1) = X_face_p1
         end if

      end subroutine eval1_partials
      
      
      subroutine eval1_v(k, nc, m, ierr)
         integer, intent(in) :: nc, m, k
         integer, intent(out) :: ierr
         
         double precision, dimension(m) :: AP, AT, dlnC_dr_face, dCface_dr
         double precision, dimension(m,m) :: AX
         double precision, dimension(m,m) :: clg
         integer :: i, j, im
         logical :: dbg_solve
         double precision :: coef, dv_im, vgt, limit_coeff, X_Y_limit, alfa, beta
         double precision :: tau0
            ! = 6d13*secyer, characteristic solar diffusion time (seconds)
         double precision, parameter :: rho_unit = 1d2
         double precision, parameter :: T_unit = 1d7

         include 'formats.dek'
         
         dbg_solve = .false.
         
         ierr = 0

         X(m,k) = A(m)/dot_product(A(1:nc),C(1:nc,k))
         dCface_dr(:) = (C(:,k-1) - C(:,k))*dm_dr_face(k)/dm_face(k)
         
         ! get C and X at face
         alfa = alfa_vec(k)
         beta = 1 - alfa
         X_face(:,k) = alfa*X(:,k-1) + beta*X(:,k)
         Z_face(:,k) = alfa*Z(:,k-1) + beta*Z(:,k)
         C_face(:,k) = alfa*C(:,k-1) + beta*C(:,k)
         dlnC_dr_face(:) = dCface_dr(:)/max(tiny_C,C_face(:,k))
         
         call get_X_Y_limit(m, X_face(ih,k), X_face(ihe,k), X_Y_limit)
         
         limit_coeff = X_Y_limit*gamma_T_limit_face(k)
         tau0 = 6d13*secyer
         coef = limit_coeff*Rsun*(T_face(k)/T_unit)**2.5d0/(rho_face(k)/rho_unit)*(Rsun/tau0) 
            ! converts to cgs units
         if (coef == 0) then
            v_face(:,k) = 0
            E_field_face(k) = 0
            return
         end if

         call get_coulomb_logs(m, A, Z_face(:,k), X_face(:,k), rho_face(k), T_face(k), clg)

         call do1_solve_thoul( &
            dbg_solve, 2*m+2, m, ihe, A, Z_face(:,k), X_face(:,k), C_face(:,k), &
            clg, AP, AT, AX, E_field_face(k), ierr)
         if (ierr /= 0) then
            write(*,2) 'failed in do1_solve_thoul', k, ierr
            stop 'eval1_gradient_coeffs'
         end if

         do i=1,nc
            vgt = coef*(AP(i)*dlnP_dr_face(k) + AT(i)*dlnT_dr_face(k))
            v_face(i,k) = vgt + sum(coef*AX(i,1:nc)*dlnC_dr_face(1:nc))
            if (abs(v_face(i,k)) > v_max) v_face(i,k) = sign(1d0,v_face(i,k))*v_max
         end do
         
         ! final fixup for velocity of most abundant to give exact local mass conservation.
         im = maxloc(X_face(1:nc,k),dim=1)
         dv_im = -dot_product(X_face(1:nc,k), v_face(1:nc,k))/X_face(im,k)
         v_face(im,k) = v_face(im,k) + dv_im
         
         
         contains
         

         subroutine get_X_Y_limit(m, xh, xhe, X_Y_limit_face)
            ! only compute diffusion velocities in regions with non-degenerate electrons
            ! decrease coeffs to 0 as X goes from X_full_on to X_full_off
            ! decrease coeffs to 0 as Y goes from Y_full_on to Y_full_off
            integer, intent(in) :: m
            double precision, intent(in) :: xh, xhe
            double precision, intent(out) :: X_Y_limit_face
            double precision :: X_term, Y_term   
            if (xh >= X_full_on) then
               X_term = 1
            else if (xh <= X_full_off) then
               X_term = 0
            else
               X_term = (xh - X_full_off) / (X_full_on - X_full_off)
            end if
            if (xhe >= Y_full_on) then
               Y_term = 1
            else if (xhe <= Y_full_off) then
               Y_term = 0
            else
               Y_term = (xhe - Y_full_off) / (Y_full_on - Y_full_off)
            end if
            if (X_term*Y_term == 0) then
               X_Y_limit_face = 0
            else if (X_term*Y_term == 1) then
               X_Y_limit_face = 1
            else
               X_Y_limit_face = 0.5d0*(1 - cos(pi*X_term*Y_term))
            end if
         end subroutine get_X_Y_limit


      end subroutine eval1_v
         

      subroutine setup( &
            dt_in, ih_in, ihe_in, A_in, C_in, X_in, Z_in, alfa_vec_in, tiny_C_in, &
            rho_face_in, T_face_in, dlnP_dr_face_in, dlnT_dr_face_in, &
            dm_dr_face_in, dm_face_in, cell_dm_in, &
            gamma_T_limit_face_in, v_max_in, &
            X_full_on_in, X_full_off_in, Y_full_on_in, Y_full_off_in, &
            xclass_in, X_face_in, Z_face_in, C_face_in, &
            E_field_face_in, v_face_in, dv_dXm1_in, dv_dX00_in, &
            tiny_X_in, flow_in, dflow_dXm1_in, dflow_dX00_in)
            
         double precision, intent(in) :: dt_in
         integer :: ih_in, ihe_in
         double precision, pointer, dimension(:) :: A_in ! (m)
         double precision, pointer, dimension(:,:) :: Z_in ! (m,n)
         double precision, pointer, dimension(:,:) :: C_in, X_in ! (m,n)
         double precision, pointer, dimension(:) :: alfa_vec_in, cell_dm_in ! (n)
         double precision :: tiny_C_in
         double precision, pointer, dimension(:) :: &
            rho_face_in, T_face_in, dlnP_dr_face_in, dlnT_dr_face_in, gamma_T_limit_face_in, &
            dm_dr_face_in, dm_face_in ! (n)
         double precision :: &
            v_max_in, tiny_X_in, X_full_on_in, X_full_off_in, Y_full_on_in, Y_full_off_in

         double precision, pointer, dimension(:,:) :: xclass_in ! (nc,n)
         double precision, pointer, dimension(:,:) :: &
            X_face_in, Z_face_in, C_face_in ! (m,n)
         double precision, pointer, dimension(:) :: E_field_face_in ! (n)
         double precision, pointer, dimension(:,:) :: v_face_in ! (nc,n) 
         double precision, pointer, dimension(:,:,:) :: dv_dXm1_in, dv_dX00_in ! (nc,nc,n)
         double precision, pointer, dimension(:,:) :: flow_in ! (nc,n)
         double precision, pointer, dimension(:,:,:) :: dflow_dXm1_in, dflow_dX00_in ! (nc,nc,n)
         
         dt = dt_in
         ih = ih_in
         ihe = ihe_in
         A => A_in
         C => C_in
         X => X_in
         Z => Z_in
         alfa_vec => alfa_vec_in
         tiny_C = tiny_C_in
         rho_face => rho_face_in
         T_face => T_face_in
         dlnP_dr_face => dlnP_dr_face_in
         dlnT_dr_face => dlnT_dr_face_in
         dm_dr_face => dm_dr_face_in
         dm_face => dm_face_in
         cell_dm => cell_dm_in
         gamma_T_limit_face => gamma_T_limit_face_in
         v_max = v_max_in
         X_full_on = X_full_on_in
         X_full_off = X_full_off_in
         Y_full_on = Y_full_on_in
         Y_full_off = Y_full_off_in
         xclass => xclass_in
         X_face => X_face_in
         Z_face => Z_face_in
         C_face => C_face_in
         E_field_face => E_field_face_in
         v_face => v_face_in
         dv_dXm1 => dv_dXm1_in
         dv_dX00 => dv_dX00_in
         tiny_X = tiny_X_in
         flow => flow_in
         dflow_dXm1 => dflow_dXm1_in
         dflow_dX00 => dflow_dX00_in
      
      end subroutine setup


      end module xmod_diffusion_support

