! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module mod_diffusion_procs

      use const_def
      use chem_def
      use utils_lib, only:is_bad_num, has_bad_num, return_nan
      use alert_lib, only:alert
      use mod_diffusion_support

      implicit none

      contains
      
      
      subroutine get_struct_mid( &
            nz, nzlo, nzhi, r, rho, T, rho_mid, T_mid, four_pi_r2_rho_mid)
         integer, intent(in) :: nz, nzlo, nzhi
         double precision, dimension(nz), intent(in) :: r, rho, T
         double precision, dimension(nz), intent(out) :: rho_mid, T_mid, four_pi_r2_rho_mid
         double precision :: r2_mid
         integer :: k     
!$OMP PARALLEL DO PRIVATE(k, r2_mid)
         do k=nzlo,nzhi-1
            r2_mid = ((r(k)**3 + r(k+1)**3)/2)**two_thirds
            rho_mid(k) = (rho(k)+rho(k+1))/2
            four_pi_r2_rho_mid(k) = pi4*r2_mid*rho_mid(k)
            T_mid(k) = (T(k)+T(k+1))/2
         end do
!$OMP END PARALLEL DO
         rho_mid(nzhi) = rho_mid(nzhi-1)
         T_mid(nzhi) = T_mid(nzhi-1)
      end subroutine get_struct_mid
      
      
      subroutine get_smooth_PTRho_gradients( &
            nz, nzlo, nzhi, dlnP_dm, dlnT_dm, dlnRho_dm, four_pi_r2_rho_mid, &
            dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid)
         integer, intent(in) :: nz, nzlo, nzhi
         double precision, dimension(nz), intent(in) ::dlnP_dm, dlnT_dm, dlnRho_dm, four_pi_r2_rho_mid
         double precision, dimension(nz), intent(out) :: dlnP_dr_mid, dlnT_dr_mid, dlnRho_dr_mid
         integer :: k, j         
!x$OMP PARALLEL DO PRIVATE(k)
         do k=nzlo,nzhi-1
            dlnP_dr_mid(k) = four_pi_r2_rho_mid(k)*(dlnP_dm(k)+dlnP_dm(k+1))/2
            dlnT_dr_mid(k) = four_pi_r2_rho_mid(k)*(dlnT_dm(k)+dlnT_dm(k+1))/2
            dlnRho_dr_mid(k) = four_pi_r2_rho_mid(k)*(dlnRho_dm(k)+dlnRho_dm(k+1))/2
         end do
!x$OMP END PARALLEL DO         
         ! smooth gradients
         do j = 1, 3
            dlnP_dr_mid(nzlo) = (2*dlnP_dr_mid(nzlo) + dlnP_dr_mid(nzlo+1))/3
            dlnT_dr_mid(nzlo) = (2*dlnT_dr_mid(nzlo) + dlnT_dr_mid(nzlo+1))/3
            dlnRho_dr_mid(nzlo) = (2*dlnRho_dr_mid(nzlo) + dlnRho_dr_mid(nzlo+1))/3
            do k = nzlo+1, nzhi-2
               dlnP_dr_mid(k) = (dlnP_dr_mid(k-1) + dlnP_dr_mid(k) + dlnP_dr_mid(k+1))/3
               dlnT_dr_mid(k) = (dlnT_dr_mid(k-1) + dlnT_dr_mid(k) + dlnT_dr_mid(k+1))/3
               dlnRho_dr_mid(k) = (dlnRho_dr_mid(k-1) + dlnRho_dr_mid(k) + dlnRho_dr_mid(k+1))/3
            end do
            dlnP_dr_mid(nzhi-1) = (2*dlnP_dr_mid(nzhi-1) + dlnP_dr_mid(nzhi-2))/3
            dlnT_dr_mid(nzhi-1) = (2*dlnT_dr_mid(nzhi-1) + dlnT_dr_mid(nzhi-2))/3
            dlnRho_dr_mid(nzhi-1) = (2*dlnRho_dr_mid(nzhi-1) + dlnRho_dr_mid(nzhi-2))/3
            do k = nzhi-2, nzlo+1, -1
               dlnP_dr_mid(k) = (dlnP_dr_mid(k-1) + dlnP_dr_mid(k) + dlnP_dr_mid(k+1))/3
               dlnT_dr_mid(k) = (dlnT_dr_mid(k-1) + dlnT_dr_mid(k) + dlnT_dr_mid(k+1))/3
               dlnRho_dr_mid(k) = (dlnRho_dr_mid(k-1) + dlnRho_dr_mid(k) + dlnRho_dr_mid(k+1))/3
            end do
         end do         
         dlnP_dr_mid(nzhi) = dlnP_dr_mid(nzhi-1)
         dlnT_dr_mid(nzhi) = dlnT_dr_mid(nzhi-1)
         dlnRho_dr_mid(nzhi) = dlnRho_dr_mid(nzhi-1)
      end subroutine get_smooth_PTRho_gradients
      
      
      subroutine get_smooth_Z( &
            nz, nzlo, nzhi, nc, class_chem_id, m, abar, free_e, T, lnT, rho, lnd, &
            calculate_ionization, Z, typical_charge)
         use ionization_lib, only: eval_typical_charge
         integer, intent(in) :: nz, nzlo, nzhi, nc, m
         double precision, dimension(nz), intent(in) :: abar, free_e, T, lnT, rho, lnd
         integer, intent(in) :: class_chem_id(nc)
         logical, intent(in) :: calculate_ionization 
         double precision, dimension(m, nz), intent(out) :: Z ! charge
         double precision, dimension(nc, nz), intent(out) :: typical_charge
         integer :: k, i   
         1 format(a40,1pe26.16)
         if (calculate_ionization) then
!$OMP PARALLEL DO PRIVATE(k,i)
            do k=nzlo,nzhi
               do i=1, nc
                  typical_charge(i,k) = eval_typical_charge( &
                        class_chem_id(i), abar(k), free_e(k)*abar(k), &
                        T(k), lnT(k)/ln10, rho(k), lnd(k)/ln10)
                  Z(i,k) = typical_charge(i,k)
               end do
            end do
!$OMP END PARALLEL DO
         else
            Z(1:nc,1:nz) = typical_charge(1:nc,1:nz)
         end if
         Z(m,nzlo:nzhi) = -1
         if (.not. calculate_ionization) return
         ! smooth Z
         do i = 1, 3
            Z(1:nc,nzlo) = (2*Z(1:nc,nzlo) + Z(1:nc,nzlo+1))/3
            do k = nzlo+1, nzhi-1
               Z(1:nc,k) = (Z(1:nc,k-1) + Z(1:nc,k) + Z(1:nc,k+1))/3
            end do
            Z(1:nc,nzhi) = (2*Z(1:nc,nzhi) + Z(1:nc,nzhi-1))/3
            do k = nzhi-1, nzlo+1, -1
               Z(1:nc,k) = (Z(1:nc,k-1) + Z(1:nc,k) + Z(1:nc,k+1))/3
            end do
         end do         
      end subroutine get_smooth_Z
         
         
      subroutine get_gamma_T_limit_coeffs( &
            nz, nzlo, nzhi, gamma, gamma_full_on, gamma_full_off, &
            T, T_full_on, T_full_off, gamma_T_limit_coeffs)
         ! only compute diffusion velocities in regions with non-degenerate electrons
         ! decrease coeffs to 0 as gamma goes from gamma_full_on to gamma_full_off
         integer, intent(in) :: nz, nzlo, nzhi
         double precision, intent(in) :: gamma(nz), gamma_full_on, gamma_full_off
         double precision, intent(in) :: T(nz), T_full_on, T_full_off
         double precision, intent(out) :: gamma_T_limit_coeffs(nz)
         integer :: k
         double precision :: gamma_term, T_term   
!$OMP PARALLEL DO PRIVATE(k,gamma_term,T_term)
         do k=nzhi, nzlo, -1
            if (gamma(k) >= gamma_full_off) then
               gamma_term = 0
            else if (gamma(k) <= gamma_full_on) then
               gamma_term = 1
            else
               gamma_term = (gamma_full_off - gamma(k)) / (gamma_full_off - gamma_full_on)
            end if
            if (T(k) >= T_full_on) then
               T_term = 1
            else if (T(k) <= T_full_off) then
               T_term = 0
            else
               T_term = (T_full_off - T(k)) / (T_full_off - T_full_on)
            end if
            if (gamma_term*T_term == 0) then
               gamma_T_limit_coeffs(k) = 0
            else if (gamma_term*T_term == 1) then
               gamma_T_limit_coeffs(k) = 1
            else
               gamma_T_limit_coeffs(k) = 0.5d0*(1 - cos(pi*gamma_term*T_term))
            end if
         end do
!$OMP END PARALLEL DO         
      end subroutine get_gamma_T_limit_coeffs
      
      
      subroutine get_A_and_X( &
            nz, nzlo, nzhi, species, nc, m, class, class_chem_id, Z, xa, tiny_X, A, X)
         integer, intent(in) :: nz, nzlo, nzhi, species, nc, m
         integer, intent(in) :: class(species), class_chem_id(nc)
         double precision, intent(in) :: Z(m,nz) ! typical charge
         double precision, intent(in) :: xa(species,nz), tiny_X
         double precision, dimension(m), intent(out) :: A ! atomic number
         double precision, dimension(m,nz), intent(out) :: X ! mass fractions
         integer :: k, j, i
         double precision :: tmp      
         1 format(a40,1pe26.16)
         2 format(a40,i6,1pe26.16)
         3 format(a40,2i6,1pe26.16)
         A(1:nc) = chem_isos% Z_plus_N(class_chem_id(1:nc))
         A(m) = me/amu         
!$OMP PARALLEL DO PRIVATE(k,j,i,tmp)
         do k=1,nz
            X(1:nc,k) = 0
            do j=1,species
               i = class(j)
               X(i,k) = X(i,k) + max(tiny_X, xa(j,k))
            end do
            ! test
            do i=1,nc
               if (X(i,k) <= 0) then
                  write(*,3) 'X(i,k)', i, k, X(i,k)
                  stop 'get_A_and_X'
               end if
            end do
            tmp = sum(X(1:nc,k))
            X(1:nc,k) = X(1:nc,k) / tmp
            X(m,k) = tiny_X
         end do
!$OMP END PARALLEL DO
      end subroutine get_A_and_X
      
      
      subroutine set_new_xa(nz, nzlo, nzhi, species, nc, m, class, X_init, X, xa)
         integer, intent(in) :: nz, nzlo, nzhi, species, nc, m, class(species)
         double precision, intent(in), dimension(nc,nz) :: X_init
         double precision, intent(in), dimension(m,nz) :: X
         double precision, intent(out), dimension(species,nz) :: xa
         integer :: j, k, i
         double precision :: tmp
         1 format(a40,1pe26.16)
         2 format(a40,i6,1pe26.16)
         3 format(a40,2i6,1pe26.16)
!$OMP PARALLEL DO PRIVATE(k,j,i,tmp)
         do k=nzlo,nzhi
            do j=1,species
               i = class(j)
               if (X_init(i,k) <= 0 .or. is_bad_num(X_init(i,k))) then
                  write(*,3) 'X_init(i,k)', i, k, X_init(i,k)
                  stop 'set_new_xa'
               end if
               xa(j,k) = xa(j,k)*X(i,k)/X_init(i,k)
            end do
            tmp = sum(xa(1:species,k))
            if (tmp <= 0 .or. is_bad_num(tmp)) then
               write(*,2) 'tmp', k, tmp
               stop 'set_new_xa'
            end if
            xa(1:species,k) = xa(1:species,k) / tmp
         end do
!$OMP END PARALLEL DO
         forall (j=1:species) xa(j,1:nzlo-1) = xa(j,nzlo)
      end subroutine set_new_xa
      
      
      subroutine do_smooth_where_h_rich(nz, nzlo, nzhi, species, net_iso, xa)
         integer, intent(in) :: nz, nzlo, nzhi, species, net_iso(:)
         double precision, intent(inout), dimension(species,nz) :: xa
         
         integer :: h1, k, kh1, j
         double precision, parameter :: xh1_limit = 0.99d0
         double precision, pointer :: xa_temp(:,:)
         
         h1 = net_iso(ih1)
         kh1 = 0
         do k = 1, nzhi
            if (xa(h1,k) < xh1_limit) exit
            kh1 = k
         end do
         ! kh1 is deepest zone with xh1 >= xh1_limit
         if (kh1 == 0) return
         
         do j=1,10
            do k = 2, kh1
               xa(:,k) = (xa(:,k-1) + xa(:,k) + xa(:,k+1))/3
            end do
         end do

         return
         
         allocate(xa_temp(species,nz))
         
         do j = 1, 2
            xa_temp(:,1:kh1) = xa(:,1:kh1)
            do k = 2, kh1-1
               xa(:,k) = (xa_temp(:,k-1) + xa_temp(:,k) + xa_temp(:,k+1))/3
            end do
            xa_temp(:,1:kh1) = xa(:,1:kh1)
            do k = kh1-1, 2, -1
               xa(:,k) = (xa_temp(:,k-1) + xa_temp(:,k) + xa_temp(:,k+1))/3
            end do
         end do
         
         deallocate(xa_temp)
         
      end subroutine do_smooth_where_h_rich
                  
                  
      end module mod_diffusion_procs

