      module mod_test_chem
      use chem_lib
      use chem_def
      
      implicit none
      

      contains
      
      
      subroutine do_test
         integer :: ierr
         call chem_init('../../data', 'isotopes.data_real', ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'FATAL ERROR: failed in chem_init'
      	   stop 1
      	end if
      	call do_test_chem_isos
         call do_test_chem
         call do_test_Qtotal
         !call write_chem_ids_file
      end subroutine do_test
      
      
      subroutine do_test_Qtotal
      
         integer, parameter :: num_in = 2, num_out = 3
         integer :: reactants(num_in + num_out)
         double precision :: Qtotal
         
         include 'formats.dek'
         
         write(*,*) 'test Qtotal for 2 he3 => 2 h1 + he4'
         write(*,*)
         
         reactants(1) = ihe3
         reactants(2) = ihe3
         reactants(3) = ihe4
         reactants(4) = ih1
         reactants(5) = ih1
         Qtotal = reaction_Qtotal(num_in,num_out,reactants,chem_isos)
         
         write(*,1) 'Qtotal', Qtotal   ! expect 12.86
         write(*,*)
      
      end subroutine do_test_Qtotal
      
      
      subroutine do_test_chem_isos
         integer :: i, ierr, nnuc
         integer, dimension(:), pointer :: Z, A
      	character(len=iso_name_length), dimension(:), pointer :: names
      	character(len=long_name_length), dimension(:), pointer :: long_names
      	type(nuclide_set), dimension(:), pointer :: set
      	double precision, dimension(:), pointer :: X, Y
      	type(nuclide_data) :: nuclides ! contents will be allocated
         
         include 'formats.dek'
         
         write(*,*) 'test chem_isos'
         write(*,*)
         
         i = get_nuclide_index('o16')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index for o16', i
            stop 1
         end if
         call show(i)         
         
         call chem_read_ZA('test_ZA.txt', Z, A, nnuc, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in chem_read_ZA'
            stop 1
         end if
         
         allocate(names(nnuc), long_names(nnuc), set(nnuc), X(nnuc), Y(nnuc))
         
         call generate_nuclide_names(Z, A, names)
         do i=1,nnuc
            write(*,2) trim(names(i))
         end do
         write(*,*)
         
         call generate_long_nuclide_names(Z, A, long_names)
         do i=1,nnuc
            write(*,2) trim(long_names(i))
         end do
         write(*,*)
         
         
         call generate_nuclide_set(names, set)
         
         i = get_nuclide_index_in_set('n14', set)
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index_in_set for n14', i
            stop 1
         end if
         
         write(*,2) trim(set(i)% nuclide)
         write(*,*)
         
         call extract_nuclides_from_chem_isos(set, nuclides, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in extract_nuclides_from_chem_isos'
            stop 1
         end if
         
         
         write(*,2) 'name ' // trim(nuclides% name(i))
         write(*,2) 'Z', nuclides% Z(i) 
         write(*,2) 'N', nuclides% N(i) 
         write(*,2) 'Z_plus_N', nuclides% Z_plus_N(i) 
         write(*,1) 'W', nuclides% W(i)
         write(*,1) 'binding_energy', nuclides% binding_energy(i) 
         write(*,1) 'spin', nuclides% spin(i) 
         
         write(*,*)


         
         i = get_nuclide_index('h1')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index', i
            stop 1
         end if
         call show(i)
         
         i = get_nuclide_index('prot')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index', i
            stop 1
         end if
         call show(i)
         
         i = get_nuclide_index('al26')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index', i
            stop 1
         end if
         call show(i)
         
         i = get_nuclide_index('al26-1')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index', i
            stop 1
         end if
         call show(i)
         
         i = get_nuclide_index('al26-2')
         if (i <= 0 .or. i > num_chem_isos) then
            write(*,*) 'bad result from get_nuclide_index', i
            stop 1
         end if
         call show(i)
         
         deallocate(names, long_names, A, Z, set, X, Y)
         
         call free_nuclide_data(nuclides)
         if (ierr /= 0) then
            write(*,*) 'failed in free_nuclide_data'
            stop 1
         end if
         
         contains
         
         subroutine show(i)
            integer, intent(in) :: i
            include 'formats.dek'
            write(*,2) 'name ' // trim(chem_isos% name(i))
            write(*,2) 'Z', chem_isos% Z(i) 
            write(*,2) 'N', chem_isos% N(i) 
            write(*,2) 'Z_plus_N', chem_isos% Z_plus_N(i) 
            write(*,2) 'isomeric_state', chem_isos% isomeric_state(i) 
            write(*,1) 'W', chem_isos% W(i)
            write(*,1) 'binding_energy', chem_isos% binding_energy(i) 
            write(*,1) 'spin', chem_isos% spin(i) 
            write(*,*)
         end subroutine show
         
      
      end subroutine do_test_chem_isos


      subroutine do_test_chem
         double precision :: c, n, o, cno
         integer :: ic12, in14, io16
 1       format(a40,f26.16)
         
         ic12 = get_nuclide_index('c12')
         in14 = get_nuclide_index('n14')
         io16 = get_nuclide_index('o16')

         write(*,*)
         write(*,*)
         write(*,1) 'chem_W(io16)', chem_isos% W(io16)
         write(*,1) 'chem_Z(io16)', dble(chem_isos% Z(io16))
         write(*,1) 'chem_binding_energy(io16)', chem_isos% binding_energy(io16)
         write(*,*)
         write(*,*) 'chem_name(io16) ', chem_isos% name(io16)
         write(*,*) 'get_nuclide_index("o16") == io16', get_nuclide_index("o16") == io16
         write(*,*)
         write(*,*) 'chem_element_Name(e_he) ', chem_element_Name(e_he)
         write(*,*) 'chem_get_element_id("he") == e_he', chem_get_element_id("he") == e_he
         write(*,*)
         write(*,*)
         write(*,1) 'Anders & Grevesse 1989 zsol', zsol
         write(*,1) 'Anders & Grevesse 1989 yesol', yesol
         write(*,*)
         write(*,*)
         write(*,*) 'cno fraction by mass of Z'
         write(*,*)
         
         c = chem_Xsol('c12')
         n = chem_Xsol('n14')
         o = chem_Xsol('o16')
         !write(*,1) 'c', c
         !write(*,1) 'n', n
         !write(*,1) 'o', o
         cno = c + n + o
         write(*,1) 'Anders & Grevesse 1989', cno/zsol
         write(*,*)         
         cno = GN93_element_zfrac(e_c) + GN93_element_zfrac(e_n) + GN93_element_zfrac(e_o)
         write(*,1) 'Grevesse and Noels 1993', cno
         write(*,*)         
         cno = GS98_element_zfrac(e_c) + GS98_element_zfrac(e_n) + GS98_element_zfrac(e_o)
         write(*,1) 'Grevesse and Sauval 1998', cno
         write(*,*)         
         cno = L03_element_zfrac(e_c) + L03_element_zfrac(e_n) + L03_element_zfrac(e_o)
         write(*,1) 'Lodders 2003', cno
         write(*,*)         
         cno = AGS04_element_zfrac(e_c) + AGS04_element_zfrac(e_n) + AGS04_element_zfrac(e_o)
         write(*,1) 'Asplund, Grevesse and Sauval 2004', cno
         write(*,*)
         write(*,*)
         
      end subroutine do_test_chem 
      
      
      subroutine write_chem_ids_file
         integer, parameter :: iounit = 33
			integer :: ierr
			ierr = 0
         open(unit=iounit, file=trim('chem_ids.list'), action='write', status='replace', iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open file for write_chem_ids'
            stop 1
         end if
         call write_chem_ids(iounit)
         close(iounit)
      end subroutine write_chem_ids_file
      

      subroutine write_chem_ids(iounit)
         integer, intent(in) :: iounit
			integer :: i
         do i = 1, num_chem_isos
            write(iounit,'(5x,i5,3x,a5)') i, chem_isos% name(i)
         end do
         write(iounit,*)
      end subroutine write_chem_ids
      
      
      end module mod_test_chem
      
      
      
      program test_chem
      use mod_test_chem
      
      call do_test

      end program




