! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module chem_lib
      use chem_def, only: chem_has_been_initialized
      
      implicit none


      contains
      
      
      subroutine chem_init(data_dir, isotopes_filename, ierr)
         use chem_def
         use chem_isos_io, only: do_read_chem_isos
         use nuclide_set_mod, only: set_some_isos
         character (len=*), intent(in) :: data_dir, isotopes_filename
         integer, intent(out) :: ierr
         call init_chem_tables
      	call do_read_chem_isos(data_dir, isotopes_filename, ierr)
      	if (ierr /= 0) return
      	call set_some_isos
      	chem_has_been_initialized = .true.
      end subroutine chem_init
      
         
      subroutine composition_info(
     >      num_isos, chem_id, x, xh, xhe, 
     >      abar, zbar, z2bar, ye, approx_abar, approx_zbar, 
     >      sumx, dabar_dx, dzbar_dx)

         ! here's a reminder of definitions:
         ! XA(i) ion mass fraction (g/g)
         ! W(i) ion atomic weight (g/mole)
         ! Z(i) ion charge (number of protons)
         ! Y(i) = X(i)/W(i), ion molar fraction (mole/g)
         ! n(i) = rho*avo*Y(i), ion number density (g/cm^3)*(#/mole)*(mole/g) -> (#/cm^3)
         
         ! abar = 1/sum(Y(i)), g/mole -- average atomic weight
         ! zbar = sum(n(i)*Z(i))/sum(n(i)) -- average charge
         ! z2bar = sum(n(i)*Z(i)^2)/sum(n(i)) -- average charge^2
         ! n = rho*avo/abar -- total number density (#/cm^3)
         ! ye = zbar/abar -- average charge per baryon = proton fraction
         ! xh = mass fraction hydrogen
         ! xhe = mass fraction helium

         use chem_def
         integer, intent(in) :: num_isos
         integer, intent(in) :: chem_id(num_isos) ! the nuclide indices for the entries in x
         double precision, intent(in)  :: x(num_isos) ! mass fractions.  should sum to 1.0
         double precision, intent(out) :: 
     >         xh, xhe, abar, zbar, z2bar, ye, approx_abar, approx_zbar, 
     >         sumx, dabar_dx(num_isos), dzbar_dx(num_isos)
         
         double precision, dimension(num_isos) :: y, z
         integer :: i, iz
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            xh=0; xhe=0; abar=0; zbar=0; z2bar=0; ye=0; sumx=0; dabar_dx=0; dzbar_dx=0
            return
         end if
         xh = 0; xhe = 0
         do i=1,num_isos
            select case(chem_isos% Z(chem_id(i)))
               case (1)
                  xh = xh + x(i)
               case (2)
                  xhe = xhe + x(i)
            end select
         end do
         y(1:num_isos) = x(1:num_isos) / chem_isos% W(chem_id(1:num_isos))
         z(1:num_isos) = chem_isos% Z(chem_id(1:num_isos))
         abar = 1/sum(min(1d0,max(y(1:num_isos),1.0d-50)))
         zbar = abar * sum(y(1:num_isos)*z(1:num_isos))
         z2bar = abar * sum(y(1:num_isos)*z(1:num_isos)**2)
         ye = zbar/abar ! assume complete ionization
         sumx = sum(x(1:num_isos))
         dabar_dx(1:num_isos) = -abar**2/chem_isos% W(chem_id(1:num_isos))
         dzbar_dx(1:num_isos) = abar*(z(1:num_isos)-zbar)/chem_isos% W(chem_id(1:num_isos))
         
         ! use baryon number in place of atomic weight
         y(1:num_isos) = x(1:num_isos) / chem_isos% Z_plus_N(chem_id(1:num_isos)) ! approx y
         approx_abar = 1/sum(min(1d0,max(y(1:num_isos),1.0d-50)))
         approx_zbar = approx_abar * sum(y(1:num_isos)*z(1:num_isos))
         
      end subroutine composition_info
      
      
      double precision function reaction_Qtotal(num_in,num_out,reactants,nuclides)
         use chem_def
         integer, intent(in) :: num_in,num_out,reactants(:)
      	type(nuclide_data), intent(in) :: nuclides
         integer :: j, l
         double precision :: Q
         reaction_Qtotal = 0
			do j=1,num_in+num_out
			   l = reactants(j)
			   Q = nuclides% binding_energy(l) - nuclides% Z(l)*del_Mp - nuclides% N(l)*del_Mn
			   if (j <= num_in) then
			      reaction_Qtotal = reaction_Qtotal - Q
			   else
			      reaction_Qtotal = reaction_Qtotal + Q
			   end if
			end do
      end function reaction_Qtotal
            
      
		integer function chem_get_element_id(cname) 
		! NOTE: this is for elements like 'h', not for isotopes like 'h1'
		! use chem_get_iso_id for looking up isotope names
         use chem_def
         use utils_lib
			character (len=*), intent(in)  :: cname 
		   ! name of the element (e.g. 'h', 'he', 'ne')
		   ! same names as in chem_element_Name
			! returns id for the element if there is a matching name
			! returns 0 otherwise.
			integer :: ierr, value
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            chem_get_element_id = -1
            return
         end if
			call integer_dict_lookup(chem_element_names_dict, cname, value, ierr)
			if (ierr /= 0) value = -1
			chem_get_element_id = value
		end function chem_get_element_id
		
		
		double precision function chem_Xsol(nam)
			character (len=*), intent(in)  :: nam 
			   ! name of the isotope (e.g. 'h1', 'he4', 'ne20')
		   double precision :: z, a, xelem
		   integer :: ierr
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            chem_Xsol = -1
            return
         end if
		   call chem_get_solar(nam, z, a, xelem, ierr)
		   if (ierr /= 0) then
		      chem_Xsol = 0d0
		   else
		      chem_Xsol = xelem
		   end if
		end function chem_Xsol
      

		subroutine chem_get_solar(nam, z, a, xelem, ierr)
         use chem_def
		   use utils_lib
		   ! returns data from Anders and Grevesse, 1989
			character (len=*), intent(in)  :: nam 
			   ! name of the isotope (e.g. 'h1', 'he4', 'ne20')
				! note that these names match those in the nuclear net library iso_Names array
				! but some net isotopes are not here (ex/ be7, n13, o14, o15, f17, f18, ... fe52, ni56 )
				! and some isotopes are here that are not in the nets (ex/ hf176)
			double precision, intent(out) :: z ! charge
			double precision, intent(out) :: a ! number of nucleons (protons and neutrons)
			double precision, intent(out) :: xelem ! elemental mass fraction associated with this isotope
			integer, intent(out) :: ierr ! == 0 means AOK; == -1 means did not find the requested name
			integer :: i
			ierr = 0
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            ierr = -1; z = 0; a = 0; xelem = 0; return
         end if
			call integer_dict_lookup(Xsol_names_dict, nam, i, ierr)
			if (ierr /= 0) then
   			z = 0; a = 0; xelem = 0; return
			end if
			z = dble(izsol(i))
			a = dble(iasol(i))
		   xelem = solx(i)
		end subroutine chem_get_solar

      
      subroutine chem_create_set_from_file(set_filename, default_dir, set, ierr)
         ! looks first in local directory, then in default_dir
         use chem_def, only: nuclide_set, iso_name_length
         use utils_lib
         character (len=*), intent(in) :: set_filename, default_dir
      	type(nuclide_set), dimension(:), pointer :: set ! will be allocated
      	integer, intent(out) :: ierr
         
         integer :: nnuc, iounit
      	character(len=iso_name_length), pointer :: names(:)
         integer, dimension(:), pointer :: Z, A ! will be allocated by chem_read_ZA
         character (len=256) :: filename
         
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            ierr = -1
            return
         end if
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         filename = trim(set_filename)
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         close(iounit)
         if (ierr /= 0) then ! if don't find that file, look in default_dir
            if (len_trim(default_dir) > 0) then
               filename = trim(default_dir) // '/' // trim(set_filename)
               ierr = 0
               open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
               close(iounit)
            end if
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(*,*) 'chem_create_set_from_file failed for open ' // trim(set_filename)
               return
            end if
            call free_iounit(iounit)
         end if

      	call chem_read_ZA(filename, Z, A, nnuc, ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'chem_create_set_from_file failed reading ' // trim(set_filename)
      	   return
      	end if
      	
      	allocate(names(nnuc), set(nnuc), stat=ierr)
      	if (ierr /= 0) return
      	
      	call generate_nuclide_names(Z,A,names)	
      	call generate_nuclide_set(names,set)
      	deallocate(names, Z, A)
      	
      end subroutine chem_create_set_from_file

      
      
      ! create arrays of Z's and A's from file of Z, A_lo, A_hi triples.
      subroutine chem_read_ZA(filename, Z, A, nnuc, ierr)
         use nuclide_set_mod, only: read_ZA
         character (len=*), intent(in) :: filename
         integer, dimension(:), pointer :: Z, A ! will be allocated
         integer, intent(out) :: nnuc, ierr
         call read_ZA(filename, Z, A, nnuc, ierr)
      end subroutine chem_read_ZA
      
      
      ! given an array of Z, A, returns an array of names in chem_isos format
      subroutine generate_nuclide_names(Z, A, names)
         use chem_def
      	use alert_lib
      	integer, dimension(:), intent(in) :: Z, A
      	character(len=iso_name_length), dimension(size(Z)), intent(out) :: names
      	integer :: i, ierr, count_isomer
      	character(len=80) :: message
         logical :: use_al26_isomers

      	count_isomer = 0
         use_al26_isomers = .false.
      	do i = 1, size(Z)
           if (A(i) == 26 .and. Z(i) == 13) count_isomer = count_isomer + 1
         enddo
         if (count_isomer > 1) use_al26_isomers = .true.

      	count_isomer = 1
      	do i = 1, size(Z)
      		select case(Z(i))
      		case (0)
      			names(i) = el_name(Z(i))
      		case (1:max_el_z)
      			write(names(i), '(a, i0)') trim(el_name(Z(i))), A(i)
      			if (A(i) == 26 .and. Z(i) == 13 .and. use_al26_isomers) then
      				count_isomer = count_isomer + 1
      				!if (count_isomer > 1) names(i) = al_isomers(count_isomer)
      				names(i) = al_isomers(count_isomer)
      			end if
      		case default
      			write(message, '(a, i0, a, i0)') 'warning: ', Z(i), ' greater than Zmax = ', max_el_z
      			names(i) = '*****'
      			ierr = nuclide_not_found
      			call alert(abs(ierr), alert_message)
      		end select
      		names(i) = adjustr(names(i))
      	end do
      end subroutine generate_nuclide_names
      

      subroutine generate_long_nuclide_names(Z, A, long_names)
         use chem_def
      	use alert_lib
      	integer, dimension(:), intent(in) :: Z, A
      	character(len=long_name_length), dimension(size(Z)), intent(out) :: long_names
      	integer :: i, ierr, count_isomer
      	character(len=80) :: message
         logical :: use_al26_isomers

      	count_isomer = 0
         use_al26_isomers = .false.
      	do i = 1, size(Z)
           if (A(i) == 26 .and. Z(i) == 13) count_isomer = count_isomer + 1
         enddo
         if (count_isomer > 1) use_al26_isomers = .true.

      	count_isomer = 1
      	do i = 1, size(Z)
      		select case(Z(i))
      		case (0)	! neutrons are special?
      			long_names(i) = el_long_name(Z(i))
      		case (1:max_el_z)
      			write(long_names(i), '(a, "-", i0)') trim(el_long_name(Z(i))), A(i)
      			if (A(i) == 26 .and. Z(i) == 13 .and. use_al26_isomers) then
      				count_isomer = count_isomer + 1
      				!if (count_isomer > 1) long_names(i) = long_al_isomers(count_isomer)
                                    long_names(i) = long_al_isomers(count_isomer)
      			end if
      		case default
      			write(message, '(a, i0, a, i0)') 'warning: ', Z(i), ' greater than Zmax = ', max_el_z
      			long_names(i) = '********'
      			ierr = nuclide_not_found
      			call alert(abs(ierr), alert_message)
      		end select
      	end do
      end subroutine generate_long_nuclide_names
      
      
      ! nuclide information comes from the chem_isos tables
      ! the storage container for the data is called 'chem_isos'
      ! it has name, A, Z, N, spin, and B for each nuclide
      ! use the function get_nuclide_index to find the index given the name.

      ! returns the index of a particular nuclide in the full chem_isos set
      ! returns nuclide_not_found if name not found
      function get_nuclide_index(nuclei) result(indx)
         use chem_def
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: nuclei
      	integer :: indx, ierr
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            indx = nuclide_not_found
            return
         end if
      	ierr = 0
      	call integer_dict_lookup(chem_isos_dict, nuclei, indx, ierr)
      	if (ierr /= 0) indx = nuclide_not_found
      end function get_nuclide_index
            
      
		integer function chem_get_iso_id(cname) ! same as get_nuclide_index
      	character(len=*), intent(in) :: cname
      	chem_get_iso_id = get_nuclide_index(cname)
		end function chem_get_iso_id  
            
      
		integer function lookup_ZN(Z,N)
		   use chem_def, only: chem_isos, num_chem_isos
      	integer, intent(in) :: Z, N
      	lookup_ZN = lookup_ZN_isomeric_state(Z,N,0)
		end function lookup_ZN  
            
      
		integer function lookup_ZN_isomeric_state(Z,N,isomeric_state)
		   use chem_def, only: chem_isos, num_chem_isos
      	integer, intent(in) :: Z, N, isomeric_state
      	integer :: cid, i      	
      	iso_loop: do cid = 1, num_chem_isos
      	   if (chem_isos% Z(cid) == Z .and. chem_isos% N(cid) == N) then
   	         if (chem_isos% isomeric_state(cid) == isomeric_state) then
   	            lookup_ZN_isomeric_state = cid
   	            return
   	         end if
      	      do i = cid+1, num_chem_isos
      	         if (chem_isos% Z(i) /= Z .or. chem_isos% N(i) /= N) exit iso_loop
      	         if (chem_isos% isomeric_state(i) == isomeric_state) then
      	            lookup_ZN_isomeric_state = i
      	            return
      	         end if
      	      end do
      	   end if
      	end do iso_loop
         lookup_ZN_isomeric_state = 0 ! indicating failure
		end function lookup_ZN_isomeric_state  
      

      subroutine generate_nuclide_set(names, set)
         use chem_def
      	use nuclide_set_mod, only: sort_nuclide_set
      	character(len=iso_name_length), dimension(:), intent(in) :: names
      	type(nuclide_set), dimension(size(names)), intent(out) :: set
      	integer :: i
      	set = [(nuclide_set(names(i), i), i=1, size(names))]
      	call sort_nuclide_set(set)
      end subroutine generate_nuclide_set
      

      ! returns the index of a particular nuclide in a particular set
      ! returns nuclide_not_found if name not found
      function get_nuclide_index_in_set(nuclei, set) result(indx)
         use chem_def
      	use nuclide_set_mod
      	character(len=*), intent(in) :: nuclei
      	type(nuclide_set), dimension(:), intent(in) :: set
      	integer :: indx
      	character(len=iso_name_length) :: name
      	name = nuclei
      	indx = rank_in_set(name, set)
      end function get_nuclide_index_in_set
      
	
      subroutine generate_nuclide_permutation(iso_array, set, permutation, ierr)
         use chem_def
      	use nuclide_set_mod, only: rank_in_set
      	character(len=iso_name_length), dimension(:), intent(in) :: iso_array
      	type(nuclide_set), dimension(size(iso_array)), intent(in) :: set
      	integer, dimension(size(iso_array)), intent(out) :: permutation
      	integer, intent(out) :: ierr
      	integer :: i	
      	ierr = 0
      	permutation = [ (rank_in_set(iso_array(i), set), i=1, size(iso_array)) ]
      	if (any(permutation == nuclide_not_found)) ierr = -1
      end subroutine generate_nuclide_permutation
      

      subroutine extract_nuclides_from_chem_isos(set, nuclides, ierr)
         use chem_def
      	use chem_isos_io, only : do_extract_nuclides_from_chem_isos
      	type(nuclide_set), dimension(:), intent(in) :: set
      	type(nuclide_data), intent(out) :: nuclides ! contents will be allocated
      	   ! call free_nuclide_data to release when done
      	integer, intent(out) :: ierr
      	call do_extract_nuclides_from_chem_isos(chem_isos, nuclides, set, ierr)
      end subroutine extract_nuclides_from_chem_isos
      
      
      subroutine create_nuclides_dict(nuclides, nuclides_dict, ierr)
         use chem_def
      	use chem_isos_io, only : do_create_nuclides_dict
      	type(nuclide_data), intent(in) :: nuclides
         type (integer_dict), pointer :: nuclides_dict ! will be allocated
      	integer, intent(out) :: ierr
         call do_create_nuclides_dict(nuclides, nuclides_dict, ierr)
      end subroutine create_nuclides_dict
      
      
      subroutine nuclides_composition_info(Y, nuclides, abar, zbar, z2bar, ye)
         use chem_def
         use nuclide_set_mod, only: do_composition_info
      	double precision, intent(in) :: Y(:)
      	type(nuclide_data), intent(in) :: nuclides
         double precision, intent(out) :: abar, zbar, z2bar, ye
         call do_composition_info(Y, nuclides, abar, zbar, z2bar, ye)
      end subroutine nuclides_composition_info
            
      
		integer function rates_category_id(cname)
		   use chem_def, only: category_names_dict
         use utils_lib
			character (len=*), intent(in)  :: cname 
			! returns id for the category if there is a matching name
			! returns 0 otherwise.
			integer :: ierr, value
			call integer_dict_lookup(category_names_dict, cname, value, ierr)
			if (ierr /= 0) value = 0
			rates_category_id = value
		end function rates_category_id
      

      function binding_energy(nuclides, Y) result (B)
         use chem_def
      	type(nuclide_data), intent(in) :: nuclides
      	double precision, dimension(size(nuclides% binding_energy)), intent(in) :: Y
      	double precision :: B
      	B = dot_product(nuclides% binding_energy, Y)
      end function binding_energy
      

      ! returns the indx corresponding to Tpart just less than T9
      ! T9 is the temperature in units of GK
      ! returns a value of 0 or npart if value is less than the minimum or maximum of the partition function
      ! temperature array Tpart
      function get_partition_fcn_indx(T9) result(indx)
      	use alert_lib
      	use chem_def, only: Tpart, npart
      	double precision, intent(in) :: T9
      	integer :: indx
      	integer, parameter :: max_iterations = 8
      	integer :: low, high, mid, i

      	low = 1
      	high = npart

      	if (T9 < Tpart(low)) then
      		indx = low-1
      		return
      	end if
      	if (T9 > Tpart(high)) then
      		indx = high + 1
      	end if

      	do i = 1, max_iterations
      		if (high-low <= 1) then
      			indx = low
      			return
      		end if
      		mid = (high+low)/2
      		if (T9 < Tpart(mid)) then
      			high = mid
      		else
      			low = mid
      		end if
      	end do
      	! should never get here
      	indx = low-1
      	call alert(1, 'too many cycles')
      	
      end function get_partition_fcn_indx


      function lodders03_element_atom_percent(nuclei) result(percent)
         ! Katharina Lodders,
         ! "Solar System Abundances and Condensation Temperatures of the Elements",
         ! ApJ, 591, 1220-1247 (2003).
         ! Table 6: Abundances of the Isotopes in the Solar System
         
         ! These are element atom percentages (i.e., by number, not by mass)
         
         ! NOTE: The data here stops at ge -- the table in the paper goes to 92U
         
         ! TO DO: add the rest of the info from the paper

         use chem_def
      	use utils_lib, only: integer_dict_lookup
      	character(len=*), intent(in) :: nuclei
      	double precision :: percent
      	integer :: indx, ierr, Z, A
         if (.not. chem_has_been_initialized) then
            write(*,*) 'must call chem_init before calling any other routine in chem_lib'
            percent = -1
            return
         end if
      	percent = 0d0
      	ierr = 0
      	call integer_dict_lookup(chem_isos_dict, nuclei, indx, ierr)
      	if (ierr /= 0) return
      	Z = chem_isos% Z(indx)
      	A = Z + chem_isos% N(indx)
         percent = 0
      	select case (Z)
      	   case(1) ! h
      	      select case (A)
      	         case(1)
                     percent =   99.99806d0
      	         case(2)
                     percent =   0.00194d0
      	      end select
      	   case(2) ! he
      	      select case (A)
      	         case(3)
                     percent =   0.016597d0
      	         case(4)
                     percent =   99.983403d0
      	      end select
      	   case(3) ! li
      	      select case (A)
      	         case(6)
                     percent =   7.589d0
      	         case(7)
                     percent =   92.411d0
      	      end select
      	   case(4) ! be
      	      select case (A)
      	         case(9)
                     percent =   100d0
      	      end select
      	   case(5) ! b
      	      select case (A)
      	         case(10)
                     percent =   19.82d0
      	         case(11)
                     percent =   80.18d0
      	      end select
      	   case(6) ! c
      	      select case (A)
      	         case(12)
                     percent =   98.8922d0
      	         case(13)
                     percent =   1.1078d0
      	      end select
      	   case(7) ! n
      	      select case (A)
      	         case(14)
                     percent =   99.6337d0
      	         case(15)
                     percent =   0.3663d0
      	      end select
      	   case(8) ! o
      	      select case (A)
      	         case(16)
                     percent =   99.7628d0
      	         case(17)
                     percent =   0.0372d0
      	         case(18)
                     percent =   0.20004d0
      	      end select
      	   case(9) ! f
      	      select case (A)
      	         case(19)
                     percent =   100d0
      	      end select
      	   case(10) ! ne
      	      select case (A)
      	         case(20)
                     percent =   92.9431d0
      	         case(21)
                     percent =   0.2228d0
      	         case(22)
                     percent =   6.8341d0
      	      end select
      	   case(11) ! na
      	      select case (A)
      	         case(23)
                     percent =   100d0
      	      end select
      	   case(12) ! mg
      	      select case (A)
      	         case(24)
                     percent =   78.992d0
      	         case(25)
                     percent =   10.003d0
      	         case(26)
                     percent =   11.005d0
      	      end select
      	   case(13) ! al
      	      select case (A)
      	         case(27)
                     percent =   100d0
      	      end select
      	   case(14) ! si
      	      select case (A)
      	         case(28)
                     percent =   92.22968d0
      	         case(29)
                     percent =   4.68316d0
      	         case(30)
                     percent =   3.08716d0
      	      end select
      	   case(15) ! p
      	      select case (A)
      	         case(31)
                     percent =   100d0
      	      end select
      	   case(16) ! s
      	      select case (A)
      	         case(32)
                     percent =   95.018d0
      	         case(33)
                     percent =   0.75d0
      	         case(34)
                     percent =   4.215d0
      	         case(36)
                     percent =   0.017d0
      	      end select
      	   case(17) ! cl
      	      select case (A)
      	         case(35)
                     percent =   75.771d0
      	         case(37)
                     percent =   24.229d0
      	      end select
      	   case(18) ! ar
      	      select case (A)
      	         case(36)
                     percent =   84.5946d0
      	         case(37)
                     percent =   15.3808d0         
      	         case(38)
                     percent =   0.0246d0
      	      end select
      	   case(19) ! k
      	      select case (A)
      	         case(39)
                     percent =   93.25811d0
      	         case(40)
                     percent =   0.011672d0
      	         case(41)
                     percent =   6.73022d0
      	      end select
      	   case(20) ! ca
      	      select case (A)
      	         case(40)
                     percent =   96.941d0
      	         case(42)
                     percent =   0.647d0
      	         case(43)
                     percent =   0.135d0
      	         case(44)
                     percent =   2.086d0
      	         case(46)
                     percent =   0.004d0
      	         case(48)
                     percent =   0.187d0
      	      end select
      	   case(21) ! sc
      	      select case (A)
      	         case(45)
                     percent =   100d0
      	      end select
      	   case(22) ! ti
      	      select case (A)
      	         case(46)
                     percent =   8.249d0
      	         case(47)
                     percent =   7.437d0
      	         case(48)
                     percent =   73.72d0
      	         case(49)
                     percent =   5.409d0
      	         case(50)
                     percent =   5.185d0
      	      end select
      	   case(23) ! v
      	      select case (A)
      	         case(50)
                     percent =   0.2497d0
      	         case(51)
                     percent =   99.7503d0
      	      end select
      	   case(24) ! cr
      	      select case (A)
      	         case(50)
                     percent =   4.3452d0
      	         case(52)
                     percent =   83.7895d0
      	         case(53)
                     percent =   9.5006d0
      	         case(54)
                     percent =   2.3647d0
      	      end select
      	   case(25) ! mn
      	      select case (A)
      	         case(55)
                     percent =   100d0
      	      end select
      	   case(26) ! fe
      	      select case (A)
      	         case(54)
                     percent =   5.845d0
      	         case(56)
                     percent =   91.754d0
      	         case(57)
                     percent =   2.119d0
      	         case(58)
                     percent =   0.282d0
      	      end select
      	   case(27) ! co
      	      select case (A)
      	         case(59)
                     percent =   100d0
      	      end select
      	   case(28) ! ni
      	      select case (A)
      	         case(58)
                     percent =   68.0769d0
      	         case(60)
                     percent =   26.2231d0
      	         case(61)
                     percent =   1.1399d0
      	         case(62)
                     percent =   3.6345d0
      	         case(64)
                     percent =   0.9256d0
      	      end select
      	   case(29) ! cu
      	      select case (A)
      	         case(63)
                     percent =   69.174d0
      	         case(65)
                     percent =   30.826d0
      	      end select
      	   case(30) ! zn
      	      select case (A)
      	         case(64)
                     percent =   48.63d0
      	         case(66)
                     percent =   27.9d0
      	         case(67)
                     percent =   4.10d0
      	         case(68)
                     percent =   18.75d0
      	         case(69)
                     percent =   0.62d0
      	      end select
      	   case(31) ! ga
      	      select case (A)
      	         case(69)
                     percent =   60.1079d0
      	         case(71)
                     percent =   39.8921d0
      	      end select
      	   case(32) ! ge
      	      select case (A)
      	         case(70)
                     percent =   21.234d0
      	         case(72)
                     percent =   27.662d0
      	         case(73)
                     percent =   7.717d0
      	         case(74)
                     percent =   35.943d0
      	         case(76)
                     percent =   7.444d0
      	      end select
      	end select
      end function lodders03_element_atom_percent



      end module chem_lib

