
      module star_adipls_support
      ! routines that call adipls for use with star

      use star_lib
      use star_def
      use const_def
      use utils_lib
      use num_lib, only: safe_log10
      
      implicit none





      real(dp) :: first_mass, first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction
      real(dp) :: min_mass, min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction
      real(dp) :: max_mass, max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction
      logical :: vary_mass, vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction
      real(dp) :: delta_mass, delta_alpha, delta_Y, delta_Z_div_X, delta_f_ov, delta_f0_ov_fraction

      integer :: hooke_itermax = 100
      real(dp) :: hooke_rho = 0.50
      real(dp) :: hooke_eps = 0.05
      real(dp) :: hooke_xscale = 0.75

      logical :: just_call_my_extras_check_model = .false.
            
      character (len=100) :: search_type

      namelist /model_search_controls/ &
         first_mass, first_Z_div_X, first_Y, first_alpha, first_f_ov, first_f0_ov_fraction, &
         min_mass, min_Z_div_X, min_Y, min_alpha, min_f_ov, min_f0_ov_fraction, &
         max_mass, max_Z_div_X, max_Y, max_alpha, max_f_ov, max_f0_ov_fraction, &
         vary_mass, vary_Z_div_X, vary_Y, vary_alpha, vary_f_ov, vary_f0_ov_fraction, &
         delta_mass, delta_alpha, delta_Y, delta_Z_div_X, delta_f_ov, delta_f0_ov_fraction, &
         hooke_itermax, hooke_rho, hooke_eps, hooke_xscale, &
         just_call_my_extras_check_model, search_type

      integer :: star_id, minimize_i_test
      integer :: i_mass, i_alpha, i_Y, i_Z_div_X, i_f_ov, i_f0_ov_fraction, nvar
      real(dp) :: final_mass, final_alpha, final_Y, final_Z_div_X, final_f_ov, final_f0_ov_fraction
      
      real(dp) :: initial_max_years_for_timestep
      logical :: okay_to_restart

      real(dp) :: next_initial_h1_to_try, next_initial_he3_to_try, &
         next_initial_he4_to_try, next_mass_to_try, next_alpha_to_try, &
         next_Z_div_X_to_try, next_Y_to_try, next_f_ov_to_try, next_f0_ov_fraction_to_try







      
      
      integer, parameter :: ivarmd = 6, iaa_arg = 10

      integer :: iounit_dev_null = -1

      integer :: nn_redist ! set from redistrb.c input file
      
      integer :: num_results
      integer, pointer, dimension(:) :: el, order
      real(dp), pointer, dimension(:) :: inertia, cyclic_freq
      
      real(dp), pointer :: x_arg(:), aa_arg(:,:)
      integer :: nn_arg
      real(dp) :: data_arg(8)
      
      logical :: adipls_save_mode_info = .false.
      integer :: adipls_el_to_save, adipls_order_to_save
      character (len=256) :: adipls_save_mode_filename = ''





      integer, parameter :: max_nl0 = 100, max_nl1 = 100, max_nl2 = 100 ! increase as needed
      
      logical :: show_l0_results = .false.
      logical :: show_l1_results = .false.
      logical :: show_l2_results = .false.
      logical :: select_age_by_anchor_l0 = .false.
      logical :: include_nonradial_chi_square = .true.

      real(dp) :: nu_max_obs ! microHz
      real(dp) :: correction_b = 4.90d0, Y_frac_he3 = 1d-4
      real(dp) :: correction_factor = 1
      real(dp) :: rhoend = 1d-4

      
      logical :: do_restribute_mesh = .false.
      ! note: number of zones for redistribute is set in the redistrb.c input file


      ! observed l=0 modes to match to model
      integer :: nl0 
      real(dp) :: l0_obs(max_nl0)
      real(dp) :: l0_obs_sigma(max_nl0)
         
      ! for the radial modes, we require model orders match observation estimated orders.
      ! l0_obs(1) should be the lowest valid order for matching the observations.
      ! if there are gaps in the higher orders of the observed radial modes,
      ! indicate that by putting -1 in l0_obs for missing orders.
      integer :: l0_n_min ! expected order for mode matching l0_obs(1)

      integer :: l0_anchor ! index of radial model with smallest sigma
      logical :: use_lowest_order_l0_as_anchor = .true.
      
      ! observed l=1 modes to match to model
      integer :: nl1
      real(dp) :: l1_obs(max_nl1)
      real(dp) :: l1_obs_sigma(max_nl1)
      
      ! observed l=2 modes to match to model
      integer :: nl2
      real(dp) :: l2_obs(max_nl2)
      real(dp) :: l2_obs_sigma(max_nl2)




      ! working storage for models and search results
      real(dp) :: avg_nu_obs, avg_radial_n, avg_delta_nu_obs ! set at start
      real(dp) :: anchor_l0_freq, prev_anchor_l0_freq, anchor_l0_obs, anchor_l0_sigma, prev_age
      real(dp) :: l0_freq(max_nl0), l0_freq_corr(max_nl0), l0_inertia(max_nl0)
      real(dp) :: l1_freq(max_nl1), l1_freq_corr(max_nl1), l1_inertia(max_nl1)
      real(dp) :: l2_freq(max_nl2), l2_freq_corr(max_nl2), l2_inertia(max_nl2)
      integer*8 :: l0_order(max_nl0), l1_order(max_nl1), l2_order(max_nl2)
      integer :: l0_n_obs(max_nl0) ! order for radial nodes
      
      

      real(dp) :: center_h1_limit ! don't consider models until beyond this limit
      real(dp) :: logg_target, logg_sigma
      real(dp) :: logL_target, logL_sigma
      real(dp) :: Teff_target, Teff_sigma
      real(dp) :: num_sigma_surf
      
      ! give up if exceed any of the following after center H is exhausted.
      integer :: &
         sigmas_coeff_for_logg_limit, &
         sigmas_coeff_for_logL_limit, &
         sigmas_coeff_for_Teff_limit
      
      
      ! evaluate chi^2 when anchor_l0_freq is within this limit of anchor_l0_obs
      real(dp) :: num_sigma_anchor_l0
      
      real(dp) :: max_dt_for_search
      real(dp) :: max_chi2_increase_during_search
      real(dp) :: chi_square_param_P
      
      logical :: &
         include_logg_in_chi_square, &
         include_logL_in_chi_square, &
         include_Teff_in_chi_square
      
      real(dp) :: anchor_l0_atol, anchor_l0_rtol

      integer :: iscan_factor_l0, iscan_factor_l1, iscan_factor_l2
         ! iscan for adipls = this factor times expected number of modes
      real(dp) :: nu_lower_factor, nu_upper_factor
         ! frequency range for adipls is set from observed frequencies times these
      
      
      real(dp) :: min_sample_chi2_so_far = -1
      

      logical :: target_star_is_expanding = .true.

      integer :: el_to_save, order_to_save, save_mode_model_number
      character (len=256) :: save_mode_filename = ''

      namelist /oscillation_controls/ &
         target_star_is_expanding, &
         el_to_save, order_to_save, save_mode_filename, save_mode_model_number, &
         select_age_by_anchor_l0, include_nonradial_chi_square, &
         show_l0_results, show_l1_results, show_l2_results, &
         anchor_l0_atol, anchor_l0_rtol, &
         do_restribute_mesh, &
         iscan_factor_l0, iscan_factor_l1, iscan_factor_l2, &
         use_lowest_order_l0_as_anchor, &
         nu_lower_factor, nu_upper_factor, &
         chi_square_param_P, &
         sigmas_coeff_for_logg_limit, &
         sigmas_coeff_for_logL_limit, &
         sigmas_coeff_for_Teff_limit, &    
         include_logg_in_chi_square, &
         include_logL_in_chi_square, &
         include_Teff_in_chi_square, &
         num_sigma_surf, num_sigma_anchor_l0, &
         max_dt_for_search, &
         max_chi2_increase_during_search, &
         logg_target, logg_sigma, &
         logL_target, logL_sigma, &
         Teff_target, Teff_sigma, &
         nu_max_obs, l0_n_min, &
         nl0, l0_obs, l0_obs_sigma, &
         nl1, l1_obs, l1_obs_sigma, &
         nl2, l2_obs, l2_obs_sigma, &
         center_h1_limit, &
         Y_frac_he3, &
         correction_b, correction_factor, rhoend

      
      ! "best_model" results are for particular mass + parameters
      real(dp) :: &
         best_model_chi_square, &
         best_model_age, &
         best_model_radius, &
         best_model_logL, &
         best_model_Teff, &
         best_model_logg, &
         best_model_diff_anchor_l0, &
         best_model_avg_delta_nu, &
         best_model_a_div_r, &
         best_model_correction_r
      integer :: best_model_model_number
      integer :: &
         best_model_l0_order(max_nl0), &
         best_model_l1_order(max_nl1), &
         best_model_l2_order(max_nl2)
      real(dp), dimension (max_nl0) :: &
         best_model_l0_freq, &
         best_model_l0_freq_corr, &
         best_model_l0_inertia
      real(dp), dimension (max_nl1) :: &
         best_model_l1_freq, &
         best_model_l1_freq_corr, &
         best_model_l1_inertia
      real(dp), dimension (max_nl2) :: &
         best_model_l2_freq, &
         best_model_l2_freq_corr, &
         best_model_l2_inertia

      ! trials are for different masses with same parameters Y, Z/X, and alpha
      integer, parameter :: max_num_trials = 100
      integer :: num_trials
      real(dp), dimension(max_num_trials) :: &
         trial_chi_square, &
         trial_age, &
         trial_radius, &
         trial_logL, &
         trial_Teff, &
         trial_logg, &
         trial_avg_delta_nu, &
         trial_a_div_r, &
         trial_correction_r, &
         trial_mass, &
         trial_init_Y, &
         trial_init_Z_div_X, &
         trial_alpha, &
         trial_f_ov, &
         trial_f0_ov_fraction
      integer, dimension(max_num_trials) :: trial_model_number
      
      integer :: &
         trial_l0_order(max_nl0,max_num_trials), &
         trial_l1_order(max_nl1,max_num_trials), &
         trial_l2_order(max_nl2,max_num_trials)
      real(dp), dimension (max_nl0,max_num_trials) :: &
         trial_l0_freq, &
         trial_l0_freq_corr, &
         trial_l0_inertia
      real(dp), dimension (max_nl1,max_num_trials) :: &
         trial_l1_freq, &
         trial_l1_freq_corr, &
         trial_l1_inertia
      real(dp), dimension (max_nl2,max_num_trials) :: &
         trial_l2_freq, &
         trial_l2_freq_corr, &
         trial_l2_inertia

      ! samples are for different combinations of parameters
      integer, parameter :: max_num_samples = 1000
      integer :: num_samples
      real(dp), dimension(max_num_samples) :: &
         sample_chi_square, &
         sample_age, &
         sample_radius, &
         sample_logL, &
         sample_Teff, &
         sample_logg, &
         sample_avg_delta_nu, &
         sample_a_div_r, &
         sample_correction_r, &
         sample_mass, &
         sample_init_Y, &
         sample_init_Z_div_X, &
         sample_alpha, &
         sample_f_ov, &
         sample_f0_ov_fraction
      integer, dimension(max_num_samples) :: sample_model_number
      
      integer :: &
         sample_l0_order(max_nl0,max_num_samples), &
         sample_l1_order(max_nl1,max_num_samples), &
         sample_l2_order(max_nl2,max_num_samples)
      real(dp), dimension (max_nl0,max_num_samples) :: &
         sample_l0_freq, &
         sample_l0_freq_corr, &
         sample_l0_inertia
      real(dp), dimension (max_nl1,max_num_samples) :: &
         sample_l1_freq, &
         sample_l1_freq_corr, &
         sample_l1_inertia
      real(dp), dimension (max_nl2,max_num_samples) :: &
         sample_l2_freq, &
         sample_l2_freq_corr, &
         sample_l2_inertia
      
      ! info about current model
      real(dp) :: chi_square, a_div_r, correction_r, avg_delta_nu_model, logg
      real(dp) :: initial_Y, initial_Z_div_X

      integer, parameter :: num_extra_log_columns = 5




      
      contains


      ! this can be called from user run_star_extras check model routine
      


      subroutine adipls_get_one_el_info( &
            s, l, nu1, nu2, iscan, R, G, M, &
            redist_mesh, save_mode_info, order_to_save, save_mode_filename, &
            num, l_freq, l_inertia, l_order, ierr)
         use num_lib, only: qsort
         use star_def, only: star_info
         type (star_info), pointer :: s
         integer, intent(in) :: l, iscan
         real(dp), intent(in) :: nu1, nu2, R, G, M
         logical, intent(in) :: redist_mesh, save_mode_info
         integer, intent(in) :: order_to_save
         character (len=*), intent(in) :: save_mode_filename
         integer, intent(out) :: num
         real(dp), pointer, dimension(:) :: l_freq, l_inertia
         integer, pointer, dimension(:) :: l_order
         integer, intent(out) :: ierr
      
         real(dp) :: sig_fac
         integer :: nsel, itrsig, nsig
         real(dp) :: els1, dels, sig1, sig2, dfsig
         integer :: k, i, j
         integer, pointer :: index(:) 

         include 'formats.dek'
         
         ierr = 0
         sig_fac = (2*pi)**2*R**3/(G*M)
         nsel = 0
         dels = 1
         els1 = dble(l)
         itrsig = 1
         sig1 = sig_fac*(nu1*1d-6)**2
         sig2 = sig_fac*(nu2*1d-6)**2
         nsig = 2
         dfsig = sig_fac*s% delta_nu**2
         
         call set_adipls_controls( &
            l, nsel, els1, dels, itrsig, iscan, sig1, sig2, dfsig, nsig)

         adipls_save_mode_info = save_mode_info
         adipls_el_to_save = l
         adipls_order_to_save = order_to_save
         adipls_save_mode_filename = save_mode_filename
      
         num_results = 0
         call run_adipls(s, .false., redist_mesh, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in run_adipls'
            return
         end if
         num = num_results
         
         if (num_results == 0) then
            write(*,*) 'failed to find any modes in specified frequency range'
            return
         end if
         
         ! sort results by increasing frequency
         allocate(index(num_results), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in allocate before calling qsort'
            return
         end if
         call qsort(index, num_results, cyclic_freq)

         if (.not. associated(l_order)) then
            allocate(l_order(num_results))
         else if (num_results >= size(l_order,dim=1)) then ! enlarge
            call realloc_integer(l_order,num_results,ierr)
            if (ierr /= 0) return
         end if

         if (.not. associated(l_freq)) then
            allocate(l_freq(num_results))
         else if (num_results >= size(l_freq,dim=1)) then ! enlarge
            call realloc_double(l_freq,num_results,ierr)
            if (ierr /= 0) return
         end if

         if (.not. associated(l_inertia)) then
            allocate(l_inertia(num_results))
         else if (num_results >= size(l_inertia,dim=1)) then ! enlarge
            call realloc_double(l_inertia,num_results,ierr)
            if (ierr /= 0) return
         end if
         
         do j = 1, num_results
            i = index(j)
            l_freq(j) = cyclic_freq(i)
            l_inertia(j) = inertia(i)
            l_order(j) = order(i)
         end do
         
         deallocate(index)
         
      end subroutine adipls_get_one_el_info


      
      
      subroutine adipls_mode_info( &
            l, order, freq, inertia, x, y, aa, data, nn, iy, iaa, ispcpr)
         integer, intent(in) :: l, order
         real(dp), intent(in) :: freq, inertia
         real(dp), intent(in) :: x(1:nn), y(1:iy,1:nn), aa(1:iaa,1:nn), data(8)
         integer, intent(in) :: nn, iy, iaa, ispcpr
         integer :: iounit, ierr, i, j, skip
         real(dp) :: y_r, y_h
         include 'formats.dek'
         if (.not. adipls_save_mode_info) return
         !write(*,3) 'adipls_mode_info l order freq inertia', l, order, freq, inertia
         if (l /= adipls_el_to_save .or. order /= adipls_order_to_save) return
         if (len_trim(adipls_save_mode_filename) <= 0) adipls_save_mode_filename = 'save_mode.data'
         write(*,*) 'save eigenfunction info to file ' // trim(adipls_save_mode_filename)
         write(*,'(2a8,99a20)') 'el', 'order', 'freq (microHz)', 'inertia'
         write(*,'(2i8,f20.10,e20.10,i20)') l, order, freq, inertia
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(adipls_save_mode_filename), action='write', iostat=ierr)
         if (ierr /= 0) return
         if (abs(x(1)) < 1d-20) then
            skip = 1
         else
            skip = 0
         end if
         write(iounit,'(2a8,99a20)') 'el', 'order', 'freq (microHz)', 'inertia', 'nn'
         write(iounit,'(2i8,f20.10,e20.10,i20)') l, order, freq, inertia, nn-skip
         write(iounit,'(a)') &
            'x = r/R;  y_r = xi_r/R;  y_h = xi_h*l*(l+1)/R;' // &
            '  displacement xi normalized to xi_r = R at surface.'
         write(iounit,'(a6,4a26)') 'i', 'x', 'y_r', 'y_h'
         do i = 1+skip, nn
            y_r = y(1,i)
            if (l > 0) then
               y_h = y(2,i)
            else
               y_h = 0
            end if
            write(iounit,'(i6,4e26.16)') i-skip, x(i), y_r, y_h
         end do
         close(iounit)
         call free_iounit(iounit)         
      end subroutine adipls_mode_info
      
      
      subroutine run_adipls(s, first_time, do_restribute_mesh, ierr)
         type (star_info), pointer :: s
         logical, intent(in) :: first_time, do_restribute_mesh
         integer, intent(out) :: ierr

         common/cstdio/ istdin, istdou, istdpr, istder
         integer :: istdin, istdou, istdpr, istder
         common/cstdio_def/ istdin_def, istdou_def, istdpr_def, istder_def
         integer :: istdin_def, istdou_def, istdpr_def, istder_def
         common/ccgrav/ cgrav
         real(dp) :: cgrav
         
         integer :: i, i_paramset, iriche, iturpr, i_inout, ierr_param
         integer :: nn_arg_0, iounit
         real(dp) :: x_arg0(0), aa_arg0(0,0)
         character (len=256) :: filename
         
         integer :: nn, iconst, ivar, ivers
         real(dp), pointer :: glob(:) ! (iconst)
         real(dp), pointer :: var(:,:) ! (ivar,nn)
         logical :: add_atmosphere
         real(dp), pointer :: aa(:,:) ! (iaa_arg,nn)
         real(dp), pointer :: x(:) ! (nn)
         real(dp) :: data(8)
         
         ierr = 0
         
         iriche = 0
         iturpr = 0
         i_inout = 0
         i_paramset = 1
         ierr_param = 0

         if (first_time) then
            call setup_redist
            call setup_adipls
            return
         end if
         
         if (iounit_dev_null < 0) then
            !write(*,*) 'alloc iounit_dev_null'
            iounit_dev_null = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*,*) 'adipls failed in alloc_iounit for iounit_dev_null'
               stop 'run_adipls'
            end if
            filename = '/dev/null'
            open(unit=iounit_dev_null, file=trim(filename), iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'adipls failed to open ' // trim(filename)
               stop 'run_adipls'
            end if            
            istdou = iounit_dev_null
            istdpr = iounit_dev_null
         end if

         add_atmosphere = .false.
         call star_get_fgong_info( &
            s% id, add_atmosphere, nn, iconst, ivar, glob, var, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_get_fgong_info'
            stop 1
         end if
         
         ivers = 0  ! ??? ! comes from fgong file usually
         
         call fgong_amdl( &
            cgrav, nn, iconst, ivar, ivers, glob, var, data, aa, nn, ierr)
         deallocate(glob, var)
         if (ierr /= 0) then
            write(*,*) 'failed in fgong_amdl'
            stop 1
         end if
         
         call store_amdl(nn, iriche, iturpr, data, aa, x, nn, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in store_amdl'
            stop 1
         end if
         
         call redist_amdl(ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in redist_amdl'
            stop 1
         end if
         
         ! ivarmd and iaa_arg are defined in file with store_amdl         
         call adipls(i_paramset, ierr_param, i_inout, &
               x, aa, data, nn, ivarmd, iaa_arg)
         deallocate(aa, x)

         if (ierr_param < 0) then
            ierr = ierr_param
            write(*,*) 'call to adipls failed'
            stop 'run_adipls'
         end if
         
         
         contains
         
         
         subroutine redist_amdl(ierr)
            integer, intent(out) :: ierr
            real(dp), pointer :: aa_new(:,:)
            real(dp), pointer :: x_new(:)
            integer :: nn_new
            include 'formats.dek'
            ierr = 0
            if (.not. do_restribute_mesh) return
            nn_new = nn_redist ! srdist uses nn from input file
            allocate(aa_new(iaa_arg,nn_new), x_new(nn_new))
            ierr_param = 0
            !write(*,2) 'call srdist: nn_redist', nn_new
            call srdist(i_paramset, ierr_param, i_inout, &
               x, aa, data, x_new, aa_new, &
               nn, nn_new, ivarmd, iaa_arg, iaa_arg)
            !write(*,1) 'done srdist'
            deallocate(aa, x)
            aa => aa_new
            x => x_new
            nn = nn_new
            if (ierr_param < 0) ierr = -1
         end subroutine redist_amdl
         
         
         subroutine setup_adipls
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*,*) 'setup_adipls failed in alloc_iounit'
               stop 'run_adipls'
            end if
            filename = 'adipls.c.pruned.in'
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 'ERROR: failed to open ' // trim(filename)
               write(*,*) 'please convert adipls.c.in to "pruned" form'
               write(*,*) 'e.g., you can run the get-input script from mesa/adipls/adipack.c/bin:'
               write(*,*) './../../adipls/adipack.c/bin/get-input adipls.c.in > adipls.c.pruned.in'
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 
               stop 'run_adipls'
            end if            
            
            write(*,*)
            write(*,'(a)') 'call adipls to read ' // trim(filename)
            call setups_adi
            nn_arg_0 = 0
            istdin = iounit
            call adipls(i_paramset, ierr_param, i_inout, &
                  x_arg0, aa_arg0, data_arg, nn_arg_0, ivarmd, iaa_arg)
            close(iounit)
            
            call free_iounit(iounit)
            
            if (ierr_param < 0) then
               ierr = ierr_param
               write(*,*) '1st call to adipls failed in setup_adipls'
               stop 'run_adipls'
            end if
            
            write(*,*) 'back from 1st call on adipls'
            write(*,*)

         end subroutine setup_adipls

         
         subroutine setup_redist

            common/comgrp/ isprtp
            integer :: isprtp

            if (.not. do_restribute_mesh) return
            
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*,*) 'setup_redist failed in alloc_iounit'
               stop 'run_rdist'
            end if
            filename = 'redistrb.c.pruned.in'
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'setup_redist failed to open ' // trim(filename)
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 'ERROR: failed to open ' // trim(filename)
               write(*,*) 'please convert redistrb.c.in to "pruned" form'
               write(*,*) 'e.g., you can run the get-input script from mesa/adipls/adipack.c/bin:'
               write(*,*) './../../adipls/adipack.c/bin/get-input redistrb.c.in > redistrb.c.pruned.in'
               write(*,*) 
               write(*,*) 
               write(*,*) 
               write(*,*) 
               stop 'run_adipls'
               stop 'run_rdist'
            end if 
            
            read(iounit,*,iostat=ierr) nn_redist
            if (ierr /= 0) then
               write(*,*) 'setup_redist failed to read nn_redist from ' // trim(filename)
               stop 'run_rdist'
            end if 
            write(*,*) 'nn_redist', nn_redist
            
            rewind(iounit)
                       
            write(*,*)
            write(*,'(a)') 'call srdist to read ' // trim(filename)
            
            istdin = iounit
            i_inout = 0
            i_paramset = 1
            ierr_param = 0
            isprtp = 0
            
            call srdist(i_paramset, ierr_param, i_inout, &
               x_arg0, aa_arg0, data_arg, x_arg0, aa_arg0, &
               nn_arg_0, nn_arg_0, ivarmd, iaa_arg, iaa_arg)
                  
            close(iounit)
            
            call free_iounit(iounit)
            
            if (ierr_param < 0) then
               ierr = ierr_param
               write(*,*) '1st call to srdist failed'
               stop 'run_rdist'
            end if
            
            write(*,*) 'back from 1st call on srdist'
            write(*,*)

         end subroutine setup_redist


      end subroutine run_adipls
      
      
      subroutine set_adipls_controls( &
            el, nsel, els1, dels, itrsig, iscan, sig1, sig2, dfsig, nsig)
         integer, intent(in) :: el, nsel, itrsig, iscan, nsig
         real(dp), intent(in) :: els1, dels, sig1, sig2, dfsig

         !  commons for parameter transmission
         common/cadr_param/ &
            para_el, para_els1, para_dels, para_dfsig1, para_dfsig2, &
            para_sig1, para_sig2, para_dfsig, para_eltrw1, para_eltrw2, &
            para_sgtrw1, para_sgtrw2
         real(dp) :: &
            para_el, para_els1, para_dels, para_dfsig1, para_dfsig2, &
            para_sig1, para_sig2, para_dfsig, para_eltrw1, para_eltrw2, &
            para_sgtrw1, para_sgtrw2
         common/cadi_param/ &
            ipara_nsel, ipara_nsig1, ipara_nsig2, ipara_itrsig, ipara_nsig, &
            ipara_istsig, ipara_inomd1, ipara_iscan, ipara_irotkr
         integer :: &
            ipara_nsel, ipara_nsig1, ipara_nsig2, ipara_itrsig, ipara_nsig, &
            ipara_istsig, ipara_inomd1, ipara_iscan, ipara_irotkr
         
         para_el = dble(el)
         ipara_nsel = nsel
         para_els1 = els1
         para_dels = dels
         ipara_itrsig = itrsig
         ipara_iscan = iscan
         para_sig1 = sig1
         para_sig2 = sig2
         para_dfsig = dfsig
         ipara_nsig = nsig
      
      end subroutine set_adipls_controls
      
      
      ! this is called by modmod
      subroutine check_arg_data(nn, data, ldaa, aa, x, ierr)
         integer, intent(in) :: nn, ldaa
         real(dp), intent(in) :: data(8)
         real(dp) :: aa(ldaa,nn)
         real(dp) :: x(nn)
         integer, intent(out) :: ierr
         
         real(dp), parameter :: rtol = 1d-9, atol = 1d-9
         
         integer :: i, j
         
         ierr = 0
         
         if (ldaa /= iaa_arg) then
            write(*,*) 'ldaa /= iaa_arg', ldaa, iaa_arg
            ierr = -1
            stop 1
         end if
         
         if (nn /= nn_arg) then
            write(*,*) 'nn /= nn_arg', nn, nn_arg
            ierr = -1
            stop 1
         end if
         
         do i=1,8
            if (is_bad(data(i),data_arg(i))) then
               write(*,'(a40,i6,99e26.16)') 'data(i) /= data_arg(i)', i, data(i), data_arg(i)
               ierr = -1
               stop 1
            end if
         end do
         
         do j=1,nn
            if (is_bad(x(j),x_arg(j))) then
               write(*,'(a40,i6,99e26.16)') 'x(j) /= x_arg(j)', j, x(j), x_arg(j)
               ierr = -1
               stop 1
            end if
            do i=1,iaa_arg
               if (is_bad(aa(i,j),aa_arg(i,j))) then
                  write(*,'(a40,2i6,99e26.16)') 'aa(i,j) /= aa_arg(i,j)', i, j, aa(i,j), aa_arg(i,j)
                  stop 1
                  ierr = -1
               end if
            end do
         end do
         
         if (ierr /= 0) stop 'check_arg_data'
         
         
         contains
         
         logical function is_bad(v1,v2)
            real(dp), intent(in) :: v1, v2
            real(dp) :: err
            err = abs(v1-v2)/(atol + rtol*max(abs(v1),abs(v2)))
            is_bad = (err > 1d0)
         end function is_bad
         
      
      end subroutine check_arg_data
      
      
      subroutine read_and_store(iriche, iturpr, cgrav)
         integer, intent(inout) :: iriche, iturpr
         real(dp), intent(in) :: cgrav
         character (len=64) :: fname
         integer :: nn, iconst, ivar, ivers, ierr
         real(dp), pointer :: glob(:) ! (iconst)   will be allocated
         real(dp), pointer :: var(:,:) ! (ivar,nn)   will be allocated
         real(dp), pointer :: aa(:,:) ! (iaa_arg,nn)   will be allocated
         real(dp), pointer :: x(:) ! (nn)   will be allocated
         real(dp) :: data(8)
         
         ierr = 0
         fname = 'test.fgong'
         call read_fgong_file(fname, nn, iconst, ivar, ivers, glob, var, ierr)
         if (ierr /= 0) then
            write(*,*) 'read_and_store failed in read_fgong_file'
            stop 1
         end if
         call fgong_amdl( &
            cgrav, nn, iconst, ivar, ivers, glob, var, data, aa, nn, ierr)
         if (ierr /= 0) then
            write(*,*) 'read_and_store failed in fgong_amdl'
            stop 1
         end if
         deallocate(glob, var)
         call store_amdl(nn, iriche, iturpr, data, aa, x, nn, ierr)
         if (ierr /= 0) then
            write(*,*) 'read_and_store failed in store_amdl'
            stop 1
         end if
         
      end subroutine read_and_store
      
      
      subroutine store_amdl(nn_in, iriche, iturpr, data, aa, x, nn, ierr)
         ! derived from adipls readml.n.d.f
         integer, intent(in) :: nn_in, iriche
         integer, intent(inout) :: iturpr
         real(dp), intent(in) :: data(8)
         real(dp), pointer :: aa(:,:)
         real(dp), pointer :: x(:) ! (nn)     will be allocated
         ! nn can be less than nn_in
         integer, intent(out) :: nn, ierr
         
         ! local
         integer :: i, j, nsin, iggt, inp, in, nshift, nnr, n, n1, nstart, idata8
         logical :: sincen, sinsur
         real(dp), pointer :: aa1(:,:)
         real(dp) :: ggt
         
         ierr = 0
         nn = nn_in
         
         allocate(aa1(iaa_arg,nn))
         do i=1,nn
            do j=1,ivarmd
               aa1(j,i) = aa(j,i)
            end do
         end do
      
         ! test for singular centre and/or surface

         sincen=aa1(1,1).eq.0
         sinsur=data(7).ge.0
         nsin=0
         if (sincen) nsin=nsin+1
         if (sinsur) nsin=nsin+1

         ! test for inclusion of g/(g tilde)

	      idata8 = int(data(8)+0.1)
         if (mod(idata8/10,10).eq.2) then
            iggt = 1
            iturpr=8
         else
            iggt=0
         end if

         ! we always take every point in model
         
         ! test for number of nonsingular points

         if (iriche.ne.1.or.mod(nn-nsin,2).eq.1) then
            nshift=0
         else
            nshift=1
         end if
         nnr=nn
         if (nshift.ne.0) then
            nn=nn-nshift
         end if
         
         allocate(x(nn))
         
         if (sincen) then
            x(1)=aa1(1,1)
            do i=1,ivarmd
               aa(i,1)=aa1(i+1,1)
            end do
            do n=2,nnr
               n1=n+nshift
               x(n)=aa1(1,n1)
               do i=1,ivarmd
                  aa(i,n)=aa1(i+1,n1)
               end do
            end do
         else
            do n=1,nnr
               if (n.eq.1) then
                  n1=1
               else
                  n1=n+nshift
               end if
               x(n)=aa1(1,n1)
               do i=1,ivarmd
                  aa(i,n)=aa1(i+1,n1)
               end do
            end do
         end if
         
         deallocate(aa1)

         ! set g/gtilde (=1 in models without turbulent pressure)

         if (iturpr.eq.1) then
            do n=1,nn
               if (x(n).lt.0.999) then
                  ggt=1
               else
                  ggt=1./(x(n)*x(n)*x(n)*aa(1,n))
               end if
               aa(10,n)=ggt
            end do
         else if (iggt.eq.1) then
            do n=1,nn
               aa(10,n)=aa(6,n)
            end do
         else
            do n=1,nn
               aa(10,n)=1
            end do
         end if
         
         x_arg => x
         aa_arg => aa
         nn_arg = nn
         data_arg(:) = data(:)

      end subroutine store_amdl
            
      
      subroutine fgong_amdl( &
            cgrav, nn_in, iconst, ivar, ivers, glob, var, data, aa, nn, ierr)
         ! derived from fgong-amdl.d.f
         real(dp), intent(in) :: cgrav
         integer, intent(in) :: nn_in, iconst, ivar, ivers
         real(dp), pointer :: glob(:) ! (iconst)
         real(dp), pointer :: var(:,:) ! (ivar,nn_in)
         real(dp), intent(out) :: data(8)
         real(dp), pointer :: aa(:,:) ! (iaa_arg,nn)   will be allocated
         integer, intent(out) :: nn, ierr
         
         integer, parameter :: ireset(16) = &
            (/3,4,5,6,8,9,10,11,12,13,14,16,17,18,19,20/)
         integer :: nn1, i, n, ir
         real(dp) :: d2amax, var1(ivar,nn_in+100), q(nn_in+100), x(nn_in+100)
         real(dp), parameter :: pi4 = 4d0*3.14159265358979323846d0
      
         ierr = 0
         nn = nn_in
      
         if (var(1,1).gt.var(1,nn)) then 
            nn1=nn+1
            do i=1,ivar
               do n=1,nn
                  var1(i,n)=var(i,nn1-n)
               end do
               do n=1,nn
                  var(i,n)=var1(i,n)
               end do
            end do
         end if
         
         if (var(1,1).gt.1.d6) then 
            do i=1,ivar
               do n=1,nn
                  var1(i,n+1)=var(i,n)
               end do
            end do
         
            do i=1,ivar
               var1(i,1)=0
            end do
         
            do ir=1,16
               i=ireset(ir)
               var1(i,1)=var1(i,2)
            end do
         
            nn=nn+1 
            do i=1,ivar
               do n=1,nn
                  var(i,n)=var1(i,n)
               end do
            end do
         end if
         
         do n=1,nn
            q(n)=exp(var(2,n))
            x(n)=var(1,n)/glob(2)
         end do
         
         x(1)=0
         q(1)=0
         
         allocate(aa(iaa_arg,nn))
         
         do n=2,nn
            aa(1,n)=x(n)
            aa(2,n)=q(n)/x(n)**3
            aa(3,n)=cgrav*glob(1)*q(n)*var(5,n)/(var(10,n)*var(4,n)*var(1,n))
            aa(4,n)=var(10,n)
            aa(5,n)=var(15,n)
            aa(6,n)=pi4*var(5,n)*var(1,n)**3/(glob(1)*q(n))
         end do
         
         aa(1,1)=0
         aa(2,1)=pi4/3.d0*var(5,1)*glob(2)**3/glob(1)
         aa(3,1)=0
         aa(4,1)=var(10,1)
         aa(5,1)=0
         aa(6,1)=3.d0
         if (aa(5,nn).le.10) then 
            nn=nn-1 
            !write(6,*) 'Chop off outermost point' 
         end if
         data(1)=glob(1)
         data(2)=glob(2)
         data(3)=var(4,1)
         data(4)=var(5,1)
         if (glob(11).lt.0.and.glob(11).gt.-10000) then 
            data(5)=-glob(11)/var(10,1)
            data(6)=-glob(12) 
         else 
            data(5)=pi4/3.d0*cgrav*(var(5,1)*glob(2))**2/(var(4,1)*var(10,1))
            d2amax=0.d0
            do n=2,nn
               d2amax=max(d2amax,aa(5,n)/x(n)**2)
               if (x(n).ge.0.05d0) exit
            end do
            data(6)=d2amax+data(5)
            !write(6,140) data(5), data(6)
         end if
         data(7)=-1.d0
         data(8)=0.d0
      
      end subroutine fgong_amdl
      

      subroutine read_fgong_file(fin, nn, iconst, ivar, ivers, glob, var, ierr)
         character (len=*), intent(in) :: fin
         integer, intent(out) :: nn, iconst, ivar, ivers
         real(dp), pointer :: glob(:) ! (iconst)   will be allocated
         real(dp), pointer :: var(:,:) ! (ivar,nn)   will be allocated
         integer, intent(out) :: ierr
      
         real(dp), pointer :: var1(:,:) ! (ivar,nn)
         integer :: ios, iounit, i, n, ir, nn1
         character*80 :: head
      
  120 format(4i10)
  130 format(5e16.9)

         ierr = 0
         iounit = 55
         if (ierr /= 0) then
            write(*,*) 'failed in read_fgong_file'
            return
         end if
      
         ios = 0
         open(iounit,file=trim(fin),status='old', iostat=ios)
         if (ios /= 0) then
            write(*,*) 'failed to open ' // trim(fin)
            return
         end if

         do i=1,4
            read(iounit,'(a)', iostat=ios) head
            if (ios /= 0) then
               write(*,*) 'failed to read header line ', i
               return
            end if
         end do

         read(iounit,120, iostat=ios) nn, iconst, ivar, ivers
         if (ios /= 0) then
            write(*,*) 'failed to read dimensions'
            return
         end if
      
         allocate(glob(iconst), var(ivar,nn+10))
      
         read(iounit,130, iostat=ios) (glob(i),i=1,iconst)
         if (ios /= 0) then
            write(*,*) 'failed to read globals'
            return
         end if      

         do n=1,nn
            read(iounit,130, iostat=ios) (var(i,n),i=1,ivar)
            if (ios /= 0) exit
         end do
         close(iounit)
      
         if (ios /= 0) then
            write(*,*) 'failed to read vars'
            return
         end if

      end subroutine read_fgong_file
      
      
      ! for testing
      subroutine dump(filename_for_dump,nn,glob,var,ierr)
         character (len=*), intent(in) :: filename_for_dump
         integer, intent(in) :: nn
         real(dp), pointer :: glob(:) ! (iconst)
         real(dp), pointer :: var(:,:) ! (ivar,nn)
         integer, intent(out) :: ierr
      
         real(dp), parameter :: Msun = 1.9892d33, Rsun = 6.9598d10, Lsun = 3.8418d33
         integer :: iounit, k, offset
      
         ierr = 0
         if (len_trim(filename_for_dump) == 0) return

         iounit = 55
         if (ierr /= 0) then
            write(*,*) 'failed in alloc_iounit for dump fgong'
            return
         end if
      
         open(iounit, file=trim(filename_for_dump),  iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'dump fgong failed to open ' // trim(filename_for_dump)
            return
         end if

         write(*,*) 'dump fgong data to ' // trim(filename_for_dump)
      
         if (VAR(1,1) <= 1) then ! skip tny r
            offset = 1
         else
            offset = 0
         end if
      
         write(iounit,'(99a24)') &
            'num_zones', 'star_mass', 'star_radius', 'star_L', 'initial_z',  &
            'mlt_alpha', 'star_age', 'star_Teff'
         write(iounit,fmt='(i24,99e24.12)') &
            nn-offset, GLOB(1)/Msun, GLOB(2)/Rsun, GLOB(3)/Lsun, GLOB(4),  &
            GLOB(6), GLOB(13), GLOB(14)      
      
         write(iounit,'(a5,99a24)')  &
            'i', 'r', 'm', 'temperature', 'pressure', 'density', &
            'xh1', 'luminosity', 'opacity', 'eps', 'gamma1', &
            'grada', 'chiT_div_chiRho', 'cp', 'free_e', 'brunt_A', &
            'dxdt_nuc_h1', 'z', 'dr_to_surf', 'eps_grav', 'xhe3', &
            'xc12', 'xc13', 'xn14', 'xo16', 'xh2', 'xhe4', 'xli7', &
            'xbe7', 'xn15', 'xo17', 'xo18', 'xne20'                               
      
         do k=1+offset,nn
            write(iounit,fmt='(i5,99e24.12)') k-offset, &
               VAR(1,k), &
               exp(VAR(2,k))*GLOB(1), &
               VAR(3,k), &
               VAR(4,k), &
               VAR(5,k), &
               VAR(6,k), &
               VAR(7,k), &
               VAR(8,k), &
               VAR(9,k), &
               VAR(10,k), &
               VAR(11,k), &
               VAR(12,k), &
               VAR(13,k), &
               VAR(14,k), &
               VAR(15,k), &
               VAR(16,k), &
               VAR(17,k), &
               VAR(18,k), &
               VAR(19,k), &
               VAR(21,k), &
               VAR(22,k), &
               VAR(23,k), &
               VAR(24,k), &
               VAR(25,k), &
               VAR(29,k), &
               VAR(30,k), &
               VAR(31,k), &
               VAR(32,k), &
               VAR(33,k), &
               VAR(34,k), &
               VAR(35,k), &
               VAR(36,k)                               
         end do
         close(iounit)
      
      end subroutine dump


      subroutine show_adipls_results
         integer :: k
         include 'formats.dek'
         do k = 1, num_results
            write(*,4) 'ADIPLS', k, el(k), order(k), cyclic_freq(k), inertia(k)
         end do
         write(*,*)
      end subroutine show_adipls_results


      
      
      subroutine init_obs_data
         integer :: i, cnt
         real(dp) :: sum_1, sum_2, sum_3
         
         include 'formats.dek'
         
         ! NOTE: the calculation of avg_delta_nu_obs depends on l0_n_obs
         ! which depends on l0_n_min
         ! so we really need an accurate l0_n_min.
         
         do i=1,nl0
            l0_n_obs(i) = l0_n_min + i - 1
         end do
         
         cnt = 0
         sum_1 = 0
         sum_2 = 0
         l0_anchor = -1
         do i=1,nl0
            if (l0_obs(i) < 0) cycle
            if (l0_anchor < 0) then
               l0_anchor = i
            else if (l0_obs_sigma(i) < l0_obs_sigma(l0_anchor)) then
               l0_anchor = i
            end if
            cnt = cnt + 1
            sum_1 = sum_1 + l0_obs(i)
            sum_2 = sum_2 + l0_n_obs(i)
         end do
         avg_nu_obs = sum_1/cnt
         avg_radial_n = sum_2/cnt
         if (use_lowest_order_l0_as_anchor) l0_anchor = 1
         anchor_l0_obs = l0_obs(l0_anchor)
         anchor_l0_sigma = l0_obs_sigma(l0_anchor)
         
         sum_1 = 0
         sum_2 = 0
         do i=1,nl0
            if (l0_obs(i) < 0) cycle
            sum_1 = sum_1 + (l0_obs(i) - avg_nu_obs)*(l0_n_obs(i) - avg_radial_n)
            sum_2 = sum_2 + (l0_n_obs(i) - avg_radial_n)**2
         end do
         avg_delta_nu_obs = sum_1/sum_2
         
         if (.false.) then
            write(*,1) 'anchor_l0_obs', anchor_l0_obs
            write(*,1) 'anchor_l0_sigma', anchor_l0_sigma
            write(*,1) 'avg_nu_obs', avg_nu_obs
            write(*,1) 'avg_radial_n', avg_radial_n
            write(*,1) 'avg_delta_nu_obs', avg_delta_nu_obs
            write(*,*)
            stop 'init_obs_data'
         end if
            
      end subroutine init_obs_data


      subroutine get_one_el_info(s, l, nu1, nu2, iscan, i1, i2)
         use num_lib, only: safe_log10, qsort
         type (star_info), pointer :: s
         integer, intent(in) :: l, iscan, i1, i2
         real(dp), intent(in) :: nu1, nu2
         
         real(dp) :: nu_obs, dist_j, nu, dist, min_dist, min_freq, avg_nu_model, &
            R, G, M, sig_fac, b, correction_a, sum_1, sum_2, sum_3
         integer :: min_dist_j, min_order, n, ierr, cnt
         integer :: nsel, itrsig, nsig
         real(dp) :: els1, dels, sig1, sig2, dfsig
         integer :: num_l0_terms, k, i, j
         integer, pointer :: index(:) 

         include 'formats.dek'

         R = Rsun*s% photosphere_r
         G = standard_cgrav
         M = Msun*s% star_mass
         sig_fac = (2*pi)**2*R**3/(G*M)
         b = correction_b
         
         ! set controls for adipls
         nsel = 0
         dels = 1
         els1 = dble(l)
         itrsig = 1
         sig1 = sig_fac*(nu1*1d-6)**2
         sig2 = sig_fac*(nu2*1d-6)**2
         dfsig = sig_fac*s% delta_nu**2
         nsig = 2
         
         call set_adipls_controls( &
            l, nsel, els1, dels, itrsig, iscan, sig1, sig2, dfsig, nsig)
      
         ierr = 0
         num_results = 0
         call run_adipls(s, .false., do_restribute_mesh, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in run_adipls'
            stop 'get_one_el_info'
         end if
         
         ! sort results by increasing frequency
         allocate(index(num_results), stat=ierr)
         if (ierr /= 0) then
            stop 'failed in allocate before calling qsort'
         end if
         call qsort(index, num_results, cyclic_freq)
         
         if (l == 0) then
            
            l0_freq(1:nl0) = -1
            min_freq = 1d99
            
            do k = 1, num_results
               i = index(k)
               if (el(i) /= 0) cycle
               if (order(i) == l0_n_min) then
                  min_freq = cyclic_freq(i)
                  exit
               end if
            end do

            if (min_freq > 1d50) then ! failed to find it
               anchor_l0_freq = -1
               avg_delta_nu_model = -1
               !write(*,*) 'failed to find correct anchor l=0 order'
               return
            end if

            ! save indices of l=0 modes
            j = 0
            do k = 1, num_results
               i = index(k)
               if (el(i) /= 0 .or. cyclic_freq(i) < min_freq) cycle
               if (j > 0) then ! check for duplicates at same n
                  if (order(i) == l0_order(j)) then ! is a duplicate
                     if (inertia(i) < l0_inertia(j)) then ! use this one
                        l0_freq(j) = cyclic_freq(i)
                        l0_inertia(j) = inertia(i)
                     end if
                     cycle
                  end if
               end if
               if (j == nl0) exit
               j = j+1
               l0_freq(j) = cyclic_freq(i)
               l0_inertia(j) = inertia(i)
               l0_order(j) = order(i)
            end do
            
            if (j < nl0) then ! fill in missing orders just so can keep going
               do k = j+1, nl0
                  l0_freq(k) = l0_freq(k-1)
                  l0_inertia(k) = l0_inertia(k-1)
                  l0_order(k) = l0_order(k-1) + 1
               end do
            end if
            
            cnt = 0
            sum_1 = 0
            do i=1,nl0
               if (l0_obs(i) < 0) cycle
               cnt = cnt + 1
               sum_1 = sum_1 + l0_freq(i)
            end do
            avg_nu_model = sum_1/cnt
            
            sum_1 = 0
            sum_2 = 0
            sum_3 = 0
            do i=1,nl0
               if (l0_obs(i) < 0) cycle
               sum_1 = sum_1 + (l0_freq(i) - avg_nu_model)*(l0_n_obs(i) - avg_radial_n)
               sum_2 = sum_2 + (l0_n_obs(i) - avg_radial_n)**2
               sum_3 = sum_3 + (l0_obs(i)/nu_max_obs)**b
            end do
            avg_delta_nu_model = sum_1/sum_2
            correction_r = & ! K08 eqn 6
               (b-1)/(b*avg_nu_model/avg_nu_obs - avg_delta_nu_model/avg_delta_nu_obs)
            correction_a = & ! K08 eqn 10
               (avg_nu_obs - correction_r*avg_nu_model)*nl0/sum_3
            a_div_r = correction_a/correction_r
            
            if (is_bad_num(a_div_r)) then
               write(*,1) 'a_div_r', a_div_r
               write(*,1) 'correction_a', correction_a
               write(*,1) 'correction_r', correction_r
               write(*,1) 'avg_delta_nu_model', avg_delta_nu_model
               write(*,1) 'avg_nu_obs', avg_nu_obs
               write(*,1) 'avg_nu_model', avg_nu_model
               write(*,1) 'avg_delta_nu_obs', avg_delta_nu_obs
               write(*,1) 'avg_radial_n', avg_radial_n
               write(*,1) 'b', b
               write(*,1) 'nu_max_obs', nu_max_obs
               stop 'get_one_el_info'
            end if
            
            
            anchor_l0_freq = l0_freq(l0_anchor)
            if (b > 0) anchor_l0_freq = anchor_l0_freq + a_div_r*(l0_freq(1)/nu_max_obs)**b
            if (.false.) then
               write(*,1) 'anchor_l0_freq', anchor_l0_freq
               write(*,1) 'a_div_r', a_div_r
               write(*,1) 'correction_a', correction_a
               write(*,1) 'correction_r', correction_r
               write(*,1) 'nu_max_obs', nu_max_obs
               write(*,1) 'b', b
            end if

         else
         
            do i = i1, i2
               if (l == 1) then
                  nu = l1_obs(i)
               else if (l == 2) then
                  nu = l2_obs(i)
               else
                  stop 'bad value for l in get_one_el_info'
               end if
               
               min_dist = 1d99; min_dist_j = -1
               do j = 1, num_results
                  if (el(j) /= l) cycle
                  dist = abs(cyclic_freq(j) - nu)
                  if (min_dist_j <= 0 .or. dist < min_dist) then
                     min_dist = dist; min_dist_j = j
                  end if
               end do
                
               if (min_dist_j <= 0) then
                  write(*,2) 's% model_number', s% model_number
                  write(*,2) 'num_results', num_results
                  write(*,2) 'l', l
                  write(*,2) 'i', i
                  write(*,1) 'nu1', nu1
                  write(*,1) 'nu2', nu2
                  write(*,2) 'iscan', iscan
                  write(*,2) 'i1', i1
                  write(*,2) 'i2', i2
                  write(*,2) 'min_dist_j', min_dist_j
                  write(*,1) 'nu', nu
                  write(*,1) 'min_dist', min_dist
                  stop 'bad value for min_dist_j in get_one_el_info'
               end if
               j = min_dist_j
               
               if (l == 1) then
                  l1_freq(i) = cyclic_freq(j)
                  l1_inertia(i) = inertia(j)
                  l1_order(i) = order(j)
               else
                  l2_freq(i) = cyclic_freq(j)
                  l2_inertia(i) = inertia(j)
                  l2_order(i) = order(j)
               end if
            end do
            
         end if
            
         deallocate(index)


      end subroutine get_one_el_info
      
      
      subroutine get_freq_corr(a_div_r)
         real(dp) , intent(in) :: a_div_r
         integer :: i
         real(dp) :: Qnl, b
         
         b = correction_b
         
         do i = 1, nl0
            if (l0_obs(i) < 0) cycle
            Qnl = 1
            l0_freq_corr(i) = l0_freq(i)
            if (b > 0) l0_freq_corr(i) = &
               l0_freq_corr(i) + correction_factor*(a_div_r/Qnl)*(l0_freq(i)/nu_max_obs)**b
         end do
         
         do i = 1, nl1
            if (l1_obs(i) < 0) cycle
            Qnl = 2*l1_inertia(i)/(l0_inertia(min(nl0,i)) + l0_inertia(min(nl0,i+1)))
            l1_freq_corr(i) = l1_freq(i)
            if (b > 0) l1_freq_corr(i) = &
               l1_freq_corr(i) + correction_factor*(a_div_r/Qnl)*(l1_freq(i)/nu_max_obs)**b
         end do
         
         do i = 1, nl2
            if (l2_obs(i) < 0) cycle
            Qnl = l2_inertia(i)/l0_inertia(min(nl0,i+1))
            l2_freq_corr(i) = l2_freq(i)
            if (b > 0) l2_freq_corr(i) = &
               l2_freq_corr(i) + correction_factor*(a_div_r/Qnl)*(l2_freq(i)/nu_max_obs)**b
         end do

      end subroutine get_freq_corr


      real(dp) function get_chi_square(a_div_r, Teff, logL, logg, max_el)
         real(dp) , intent(in) :: a_div_r, Teff, logL, logg
         integer, intent(in) :: max_el

         integer :: i, chi2N
         real(dp) :: chi2sum, chi2term
         ! calculate chi^2 following Brandao et al, 2011, eqn 11
         include 'formats.dek'
         
         call get_freq_corr(a_div_r)
         
         chi2sum = 0
         chi2N = 0
         do i = 1, nl0
            if (l0_obs(i) < 0) cycle
            chi2term = ((l0_freq_corr(i) - l0_obs(i))/l0_obs_sigma(i))**2
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end do
         
         if (max_el >= 1) then
            do i = 1, nl1
               if (l1_obs(i) < 0) cycle
               chi2term = ((l1_freq_corr(i) - l1_obs(i))/l1_obs_sigma(i))**2       
               chi2sum = chi2sum + chi2term
               chi2N = chi2N + 1
            end do
         end if
         
         if (max_el >= 2) then
            do i = 1, nl2
               if (l2_obs(i) < 0) cycle
               chi2term = ((l2_freq_corr(i) - l2_obs(i))/l2_obs_sigma(i))**2            
               chi2sum = chi2sum + chi2term
               chi2N = chi2N + 1
            end do
         end if
         
         if (Teff_sigma > 0 .and. include_Teff_in_chi_square) then
            chi2term = ((Teff - Teff_target)/Teff_sigma)**2
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (logL_sigma > 0 .and. include_logL_in_chi_square) then
            chi2term = ((logL - logL_target)/logL_sigma)**2
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (logg_sigma > 0 .and. include_logg_in_chi_square) then
            chi2term = ((logg - logg_target)/logg_sigma)**2
            chi2sum = chi2sum + chi2term
            chi2N = chi2N + 1
         end if
         
         if (chi2N <= chi_square_param_P) then
            write(*,2) 'chi2N <= chi_square_param_P', chi2N, chi_square_param_P
            stop 'get_chi_square'
         end if
         
         get_chi_square = chi2sum / (chi2N - chi_square_param_P)
         
      end function get_chi_square
      
      
      subroutine save_best_model_info(s)
         type (star_info), pointer :: s
         
         include 'formats.dek'
      
         best_model_chi_square = chi_square
         best_model_age = s% star_age
         best_model_radius = s% photosphere_r
         best_model_logL = s% log_surface_luminosity
         best_model_Teff = s% Teff
         best_model_logg = logg
         best_model_diff_anchor_l0 = anchor_l0_freq - anchor_l0_obs
         best_model_avg_delta_nu = avg_delta_nu_model
         best_model_a_div_r = a_div_r
         best_model_correction_r = correction_r

         best_model_model_number = s% model_number

         best_model_l0_order(1:nl0) = l0_order(1:nl0)
         best_model_l0_freq(1:nl0) = l0_freq(1:nl0)
         best_model_l0_freq_corr(1:nl0) = l0_freq_corr(1:nl0)
         best_model_l0_inertia(1:nl0) = l0_inertia(1:nl0)

         best_model_l1_order(1:nl1) = l1_order(1:nl1)
         best_model_l1_freq(1:nl1) = l1_freq(1:nl1)
         best_model_l1_freq_corr(1:nl1) = l1_freq_corr(1:nl1)
         best_model_l1_inertia(1:nl1) = l1_inertia(1:nl1)
      
         best_model_l2_order(1:nl2) = l2_order(1:nl2)
         best_model_l2_freq(1:nl2) = l2_freq(1:nl2)
         best_model_l2_freq_corr(1:nl2) = l2_freq_corr(1:nl2)
         best_model_l2_inertia(1:nl2) = l2_inertia(1:nl2)

         !write(*,2) 'best_model_model_number', best_model_model_number
      
      end subroutine save_best_model_info
      
      
      subroutine zero_best_model_info
      
         best_model_age = 0
         best_model_radius = 0
         best_model_logL = 0
         best_model_Teff = 0
         best_model_logg = 0
         best_model_diff_anchor_l0 = 0
         best_model_avg_delta_nu = 0
         best_model_a_div_r = 0
         best_model_correction_r = 0

         best_model_model_number = 0

         best_model_l0_order(1:nl0) = 0
         best_model_l0_freq(1:nl0) = 0
         best_model_l0_freq_corr(1:nl0) = 0
         best_model_l0_inertia(1:nl0) = 0

         best_model_l1_order(1:nl1) = 0
         best_model_l1_freq(1:nl1) = 0
         best_model_l1_freq_corr(1:nl1) = 0
         best_model_l1_inertia(1:nl1) = 0
      
         best_model_l2_order(1:nl2) = 0
         best_model_l2_freq(1:nl2) = 0
         best_model_l2_freq_corr(1:nl2) = 0
         best_model_l2_inertia(1:nl2) = 0
      
      end subroutine zero_best_model_info
      
      
      subroutine do_show_l0_results
         integer :: i, io
         include 'formats.dek'
         io = 6
         call show_adipls_results
         write(io,'(/,2a6,a20,99a12)') &
            'l=0', 'n', 'l0_freq', 'l0_obs', 'l0_sigma'
         do i = 1, nl0
            if (l0_obs(i) < 0) cycle
            write(io,'(6x, i6, f20.4, 99f12.4)') &
               l0_order(i), l0_freq(i), l0_obs(i), l0_obs_sigma(i)
         end do
         
         stop 'show_l0_results'
         
         
      end subroutine do_show_l0_results
      
      
      subroutine do_show_l1_results
         integer :: i, io
         include 'formats.dek'
         io = 6
         call show_adipls_results
         write(io,'(/,2a6,a20,99a12)') &
            'l=1', 'n', 'l1_freq', 'l1_obs', 'l1_sigma'
         do i = 1, nl1
            if (l1_obs(i) < 0) cycle
            write(io,'(6x, i6, f20.4, 99f12.4)') &
               l1_order(i), l1_freq(i), l1_obs(i), l1_obs_sigma(i)
         end do
         !stop 'do_show_l1_results'
      end subroutine do_show_l1_results
      
      
      subroutine do_show_l2_results
         integer :: i, io
         include 'formats.dek'
         io = 6
         call show_adipls_results
         write(io,'(/,2a6,a20,99a12)') &
            'l=2', 'n', 'l2_freq', 'l2_obs', 'l2_sigma'
         do i = 1, nl2
            if (l2_obs(i) < 0) cycle
            write(io,'(6x, i6, f20.4, 99f12.4)') &
               l2_order(i), l2_freq(i), l2_obs(i), l2_obs_sigma(i)
         end do
      end subroutine do_show_l2_results
      
      
      subroutine show_best_model_info(s, io, mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
         type (star_info), pointer :: s
         integer, intent(in) :: io
         real(dp), intent(in) :: mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction
         real(dp) :: chi2term
         integer :: i, log_iounit, ierr
         
         include 'formats.dek'
         ierr = 0
         log_iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=log_iounit, file=trim('freq.data'), action='write', iostat=ierr)
         if (ierr /= 0) return

         write(log_iounit,1) 'avg_delta_nu_obs', avg_delta_nu_obs
         write(log_iounit,'(2a6,99a12)') &
            'l', 'n', 'chi2term', 'freq', 'corr', 'obs', 'sigma', 'log_E'
         write(io,'(/,2a6,99a12)') &
            'l=0', 'n', 'chi2term', 'l0_freq', 'l0_corr', 'l0_obs', 'l0_sigma', 'log E'
         do i = 1, nl0
            if (l0_obs(i) < 0) cycle
            chi2term = ((best_model_l0_freq_corr(i) - l0_obs(i))/l0_obs_sigma(i))**2
            write(io,'(6x,i6,99f12.4)') &
               best_model_l0_order(i), chi2term, best_model_l0_freq(i), best_model_l0_freq_corr(i), &
               l0_obs(i), l0_obs_sigma(i), safe_log10(best_model_l0_inertia(i))
            write(log_iounit,'(2i6,99f12.4)') &
               0, best_model_l0_order(i), chi2term, best_model_l0_freq(i), best_model_l0_freq_corr(i), &
               l0_obs(i), l0_obs_sigma(i), safe_log10(best_model_l0_inertia(i))
         end do
         
         write(io,*)
         write(io,'(2a6,99a12)') &
            'l=1', 'n', 'chi2term', 'l1_freq', 'l1_corr', 'l1_obs', 'l1_sigma', 'log E'
         do i = 1, nl1
            if (l1_obs(i) < 0) cycle
            chi2term = ((best_model_l1_freq_corr(i) - l1_obs(i))/l1_obs_sigma(i))**2
            if (is_bad_num(chi2term)) cycle
            write(io,'(6x,i6,99f12.4)') &
               best_model_l1_order(i), chi2term, best_model_l1_freq(i), best_model_l1_freq_corr(i), &
               l1_obs(i), l1_obs_sigma(i), safe_log10(best_model_l1_inertia(i))
            write(log_iounit,'(2i6,99f12.4)') &
               1, best_model_l1_order(i), chi2term, best_model_l1_freq(i), best_model_l1_freq_corr(i), &
               l1_obs(i), l1_obs_sigma(i), safe_log10(best_model_l1_inertia(i))
         end do

         write(io,*)
         write(io,'(2a6,99a12)') &
            'l=2', 'n', 'chi2term', 'l2_freq', 'l2_corr', 'l2_obs', 'l2_sigma', 'log E'
         do i = 1, nl2
            if (l2_obs(i) < 0) cycle
            chi2term = ((best_model_l2_freq_corr(i) - l2_obs(i))/l2_obs_sigma(i))**2
            if (is_bad_num(chi2term)) cycle
            write(io,'(6x,i6,99f12.4)') &
               best_model_l2_order(i), chi2term, best_model_l2_freq(i), best_model_l2_freq_corr(i), &
               l2_obs(i), l2_obs_sigma(i), safe_log10(best_model_l2_inertia(i))
            write(log_iounit,'(2i6,99f12.4)') &
               2, best_model_l2_order(i), chi2term, best_model_l2_freq(i), best_model_l2_freq_corr(i), &
               l2_obs(i), l2_obs_sigma(i), safe_log10(best_model_l2_inertia(i))
         end do

         if (Teff_sigma > 0 .and. include_Teff_in_chi_square) then
            chi2term = ((best_model_Teff - Teff_target)/Teff_sigma)**2
            write(io,*)
            write(io,'(a40,99f12.2)') 'Teff chi2term', chi2term
            write(io,'(a40,99f12.2)') 'Teff', best_model_Teff
            write(io,'(a40,99f12.2)') 'Teff_obs', Teff_target
            write(io,'(a40,99f12.2)') 'Teff_sigma', Teff_sigma
         end if
         
         if (logL_sigma > 0 .and. include_logL_in_chi_square) then
            chi2term = ((best_model_logL - logL_target)/logL_sigma)**2
            write(io,*)
            write(io,'(a40,99f12.2)') 'logL chi2term', chi2term
            write(io,'(a40,99f12.2)') 'logL', best_model_logL
            write(io,'(a40,99f12.2)') 'logL_obs', logL_target
            write(io,'(a40,99f12.2)') 'logL_sigma', logL_sigma
         end if
         
         if (logg_sigma > 0 .and. include_logg_in_chi_square) then
            chi2term = ((best_model_logg - logg_target)/logg_sigma)**2
            write(io,*)
            write(io,'(a40,99f12.2)') 'logg chi2term', chi2term
            write(io,'(a40,99f12.2)') 'logg', best_model_logg
            write(io,'(a40,99f12.2)') 'logg_obs', logg_target
            write(io,'(a40,99f12.2)') 'logg_sigma', logg_sigma
         end if
         
         write(io,*)
         write(io,'(a20,99f16.8)') 'log age', log10(best_model_age)
         write(io,'(a20,99f16.8)') 'R/Rsun', best_model_radius
         write(io,'(a20,99f16.8)') 'logL/Lsun', best_model_logL
         write(io,'(a20,99f16.8)') 'Teff', best_model_Teff
         write(io,'(a20,99f16.8)') 'logg', best_model_logg
         write(io,'(a20,99f16.8)') 'diff anchor l0', best_model_diff_anchor_l0
         write(io,'(a20,99f16.8)') 'avg_delta_nu', best_model_avg_delta_nu
         write(io,'(a20,99f16.8)') 'diff avg_delta_nu', best_model_avg_delta_nu - avg_delta_nu_obs
         write(io,'(a20,99f16.8)') 'a_div_r', best_model_a_div_r
         write(io,'(a20,99f16.8)') 'correction_r', best_model_correction_r
         write(io,*)        
         write(io,'(a20,99f16.8)') 'M/Msun', mass
         write(io,'(a20,99f16.8)') 'initial Y', Y
         write(io,'(a20,99f16.8)') 'initial Z/X', Z_div_X
         write(io,'(a20,99f16.8)') 'mixing length alpha', alpha
         write(io,'(a20,99f16.8)') 'f_ov', f_ov
         write(io,'(a20,99f16.8)') 'f0_ov_fraction', f0_ov_fraction
         write(io,*)
         write(io,'(a20,i12)') 'model number', best_model_model_number
         write(io,*)
         write(io,*)
         write(io,'(a20,99f16.8)') 'best chi^2', best_model_chi_square
         write(io,*)
         write(io,*)
         
         close(log_iounit)
         call free_iounit(log_iounit)         

      end subroutine show_best_model_info


      subroutine set_best_trial_results(mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
         real(dp), intent(out) :: mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction
         integer :: itry
         
         include 'formats.dek'
         
         itry = minloc(trial_chi_square(1:num_trials),dim=1)
         
         mass = trial_mass(itry)
         alpha = trial_alpha(itry)
         Y = trial_init_Y(itry)
         Z_div_X = trial_init_Z_div_X(itry)
         f_ov = trial_f_ov(itry)
         f0_ov_fraction = trial_f0_ov_fraction(itry)

         best_model_chi_square = trial_chi_square(itry)
         best_model_age = trial_age(itry)
         best_model_radius = trial_radius(itry)
         best_model_logL = trial_logL(itry)
         best_model_Teff = trial_Teff(itry)
         best_model_logg = trial_logg(itry)
         best_model_radius = trial_radius(itry)
         best_model_diff_anchor_l0 = trial_l0_freq_corr(1,itry) - anchor_l0_obs
         best_model_avg_delta_nu = trial_avg_delta_nu(itry)
         best_model_a_div_r = trial_a_div_r(itry)
         best_model_correction_r = trial_correction_r(itry)
         best_model_logg = trial_logg(itry)
         best_model_model_number = trial_model_number(itry)

         best_model_l0_order(1:nl0) = trial_l0_order(1:nl0,itry)
         best_model_l1_order(1:nl1) = trial_l1_order(1:nl1,itry)
         best_model_l2_order(1:nl2) = trial_l2_order(1:nl2,itry)
      
         best_model_l0_freq(1:nl0) = trial_l0_freq(1:nl0,itry)
         best_model_l0_freq_corr(1:nl0) = trial_l0_freq_corr(1:nl0,itry)
         best_model_l0_inertia(1:nl0) = trial_l0_inertia(1:nl0,itry)
      
         best_model_l1_freq(1:nl1) = trial_l1_freq(1:nl1,itry)
         best_model_l1_freq_corr(1:nl1) = trial_l1_freq_corr(1:nl1,itry)
         best_model_l1_inertia(1:nl1) = trial_l1_inertia(1:nl1,itry)
      
         best_model_l2_freq(1:nl2) = trial_l2_freq(1:nl2,itry)
         best_model_l2_freq_corr(1:nl2) = trial_l2_freq_corr(1:nl2,itry)
         best_model_l2_inertia(1:nl2) = trial_l2_inertia(1:nl2,itry)

         !write(*,2) 'set_best_trial_results', best_model_model_number

      end subroutine set_best_trial_results

      
      subroutine show_best_trial(s,io)
         type (star_info), pointer :: s
         integer, intent(in) :: io
         real(dp) :: mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction
         call set_best_trial_results(mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
         call show_best_model_info(s, io, mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
      end subroutine show_best_trial


      subroutine save_best_for_this_trial(itry, mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction)
         integer, intent(in) :: itry
         real(dp), intent(in) :: mass, alpha, Y, Z_div_X, f_ov, f0_ov_fraction
      
         trial_mass(itry) = mass
         trial_alpha(itry) = alpha
         trial_init_Y(itry) = Y 
         trial_init_Z_div_X(itry) = Z_div_X
         trial_f_ov(itry) = f_ov
         trial_f0_ov_fraction(itry) = f0_ov_fraction

         trial_chi_square(itry) = best_model_chi_square
         trial_age(itry) = best_model_age
         trial_radius(itry) = best_model_radius
         trial_logL(itry) = best_model_logL
         trial_Teff(itry) = best_model_Teff
         trial_logg(itry) = best_model_logg
         trial_a_div_r(itry) = best_model_a_div_r
         trial_avg_delta_nu(itry) = best_model_avg_delta_nu
         trial_correction_r(itry) = best_model_correction_r
         trial_logg(itry) = best_model_logg
         trial_model_number(itry) = best_model_model_number

         trial_l0_order(1:nl0,itry) = best_model_l0_order(1:nl0)
         trial_l1_order(1:nl1,itry) = best_model_l1_order(1:nl1)
         trial_l2_order(1:nl2,itry) = best_model_l2_order(1:nl2)
      
         trial_l0_freq(1:nl0,itry) = best_model_l0_freq(1:nl0)
         trial_l0_freq_corr(1:nl0,itry) = best_model_l0_freq_corr(1:nl0)
         trial_l0_inertia(1:nl0,itry) = best_model_l0_inertia(1:nl0)
      
         trial_l1_freq(1:nl1,itry) = best_model_l1_freq(1:nl1)
         trial_l1_freq_corr(1:nl1,itry) = best_model_l1_freq_corr(1:nl1)
         trial_l1_inertia(1:nl1,itry) = best_model_l1_inertia(1:nl1)
      
         trial_l2_freq(1:nl2,itry) = best_model_l2_freq(1:nl2)
         trial_l2_freq_corr(1:nl2,itry) = best_model_l2_freq_corr(1:nl2)
         trial_l2_inertia(1:nl2,itry) = best_model_l2_inertia(1:nl2)

         num_trials = itry

      end subroutine save_best_for_this_trial


      subroutine save_best_for_sample(i)
         integer, intent(in) :: i         
         integer :: j
         
         j = minloc(trial_chi_square(1:num_trials),dim=1)
      
         sample_mass(i) = trial_mass(j)
         sample_alpha(i) = trial_alpha(j)
         sample_init_Y(i) = trial_init_Y(j)
         sample_init_Z_div_X(i) = trial_init_Z_div_X(j)        
         sample_f_ov(i) = trial_f_ov(j)
         sample_f0_ov_fraction(i) = trial_f0_ov_fraction(j)

         sample_chi_square(i) = trial_chi_square(j)
         sample_age(i) = trial_age(j)
         sample_radius(i) = trial_radius(j)
         sample_logL(i) = trial_logL(j)
         sample_Teff(i) = trial_Teff(j)
         sample_logg(i) = trial_logg(j)
         sample_radius(i) = trial_radius(j)
         sample_avg_delta_nu(i) = trial_avg_delta_nu(j)
         sample_a_div_r(i) = trial_a_div_r(j)
         sample_correction_r(i) = trial_correction_r(j)
         sample_logg(i) = trial_logg(j)
         sample_model_number(i) = trial_model_number(j)

         sample_l0_order(1:nl0,i) = trial_l0_order(1:nl0,j)
         sample_l1_order(1:nl1,i) = trial_l1_order(1:nl1,j)
         sample_l2_order(1:nl2,i) = trial_l2_order(1:nl2,j)
      
         sample_l0_freq(1:nl0,i) = trial_l0_freq(1:nl0,j)
         sample_l0_freq_corr(1:nl0,i) = trial_l0_freq_corr(1:nl0,j)
         sample_l0_inertia(1:nl0,i) = trial_l0_inertia(1:nl0,j)
      
         sample_l1_freq(1:nl1,i) = trial_l1_freq(1:nl1,j)
         sample_l1_freq_corr(1:nl1,i) = trial_l1_freq_corr(1:nl1,j)
         sample_l1_inertia(1:nl1,i) = trial_l1_inertia(1:nl1,j)
      
         sample_l2_freq(1:nl2,i) = trial_l2_freq(1:nl2,j)
         sample_l2_freq_corr(1:nl2,i) = trial_l2_freq_corr(1:nl2,j)
         sample_l2_inertia(1:nl2,i) = trial_l2_inertia(1:nl2,j)

      end subroutine save_best_for_sample

      
      subroutine show_trial_results(io)
         use num_lib, only: safe_log10, qsort
         integer, intent(in) :: io
         integer :: i, j, ierr
         integer, parameter :: skip = 3
         integer, pointer :: index(:)
         include 'formats.dek'
         ! sort results by increasing trial_chi_square
         allocate(index(num_trials), stat=ierr)
         if (ierr /= 0) then
            stop 'failed in allocate before calling qsort from show_trial_results'
         end if
         call qsort(index, num_trials, trial_chi_square)
         write(io,'(15x, a8, 99a20)') &
            'model#', 'chi^2', 'mass', 'diff anchor l0', 'diff avg delta nu', &
            'logg', 'lg L/Lsun', 'Teff', 'R/Rsun', &
            'log age', 'init_Y', 'init_Z_div_X', 'alpha', &
            'f_ov', 'f0_ov_fraction'
         do j=1,num_trials
            i = index(j)
            write(io,fmt='(3x,i5,7x,i8,1pe20.6,0p,99f20.12)',advance='no') &
               i, trial_model_number(i), trial_chi_square(i), trial_mass(i), &
               trial_l0_freq_corr(l0_anchor,i) - l0_obs(l0_anchor), &
               trial_avg_delta_nu(i) - avg_delta_nu_obs, &
               trial_logg(i), trial_logL(i), trial_Teff(i), trial_radius(i), &
               safe_log10(trial_age(i)), trial_init_Y(i), &
               trial_init_Z_div_X(i), trial_alpha(i), &
               trial_f_ov(i), trial_f0_ov_fraction(i)
            write(io,*)
         end do
         if (num_trials > 10) write(io,'(15x, a8, 99a20)') &
            'model#', 'chi^2', 'mass', 'diff anchor l0', 'diff avg delta nu', &
            'logg', 'lg L/Lsun', 'Teff', 'R/Rsun', &
            'log age', 'init_Y', 'init_Z_div_X', 'alpha', &
            'f_ov', 'f0_ov_fraction'
         deallocate(index)
      end subroutine show_trial_results


      integer function do_adipls_extras_check_model(s, id, id_extra)
         use run_star_support, only: initial_h1, initial_he3, initial_he4
         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         
         integer :: max_el_for_chi_square
         real(dp) :: err, target_l0, X, Y, Z, dist, logL_logg_Teff_sigma2_limit
         
         include 'formats.dek'

         do_adipls_extras_check_model = keep_going
          
         if (mod(s% model_number,100) == 0) then
            if (vary_alpha) write(*,'(a20,99f16.8)') 'mixing length alpha', s% mixing_length_alpha
            if (vary_Y) write(*,'(a20,99f16.8)') 'initial Y', initial_Y
            if (vary_Z_div_X) write(*,'(a20,99f16.8)') 'initial Z/X', initial_Z_div_X
            if (vary_f_ov) write(*,'(a20,99f16.8)') 'f_ov', next_f_ov_to_try
            if (vary_f0_ov_fraction) write(*,'(a20,99f16.8)') 'f0_ov_fraction', next_f0_ov_fraction_to_try
            write(*,*)
         end if
         
         if (s% X(s% nz) > center_h1_limit) then
            return
         end if
         
         call check_limits
         if (do_adipls_extras_check_model /= keep_going) return
         
         dist = logL_logg_Teff_sigma2()
         chi_square = dist
         logL_logg_Teff_sigma2_limit = 2*num_sigma_surf**2
         if (dist > logL_logg_Teff_sigma2_limit) then
            !write(*,'(a,i6,6x,99f11.4)') 'logL logg Teff dist too large still', &
            !   s% model_number, dist
            call finish
            return
         end if
                  
         call get_one_el_info(s, 0, &
               nu_lower_factor*l0_obs(1), &
               nu_upper_factor*l0_obs(nl0), &
               iscan_factor_l0*nl0, 1, nl0)
         
         if (anchor_l0_freq <= 0) then
            !write(*,'(a,i6,6x,99f11.4)') 'failed to find anchor_l0_freq', &
            !   s% model_number, anchor_l0_freq
            !call do_show_l0_results
            !stop 'do_adipls_extras_check_model'
            call finish
            return
         end if
         
         if (.not. have_all_l0_freqs()) then
            !write(*,'(a,i6,6x,99f11.4)') 'failed to find required radial modes', &
            !   s% model_number
            !call do_show_l0_results
            !stop 'do_adipls_extras_check_model'
            call finish
            return
         end if
         
         !if (s% model_number == -1) call do_show_l0_results  ! DEBUG

         if (target_star_is_expanding) then     
            if (anchor_l0_freq >= prev_anchor_l0_freq) then
               call finish
               return
            end if
         else
            if (anchor_l0_freq <= prev_anchor_l0_freq) then
               call finish
               return
            end if
         end if
         
         if (show_l0_results) call do_show_l0_results
                  
         if (.not. select_age_by_anchor_l0) then
            if (target_star_is_expanding) then
               target_l0 = anchor_l0_obs + num_sigma_anchor_l0*anchor_l0_sigma
               if (anchor_l0_freq > target_l0) then
                  write(*,'(a,i6,6x,99f11.4)') 'anchor_l0_freq > target', &
                     s% model_number, anchor_l0_freq - target_l0, anchor_l0_freq, target_l0
                  call set_max_timestep(num_sigma_anchor_l0)
                  call finish
                  return
               end if
            else
               target_l0 = anchor_l0_obs - num_sigma_anchor_l0*anchor_l0_sigma
               if (anchor_l0_freq < target_l0) then
                  write(*,'(a,i6,6x,99f11.4)') 'anchor_l0_freq < target', &
                     s% model_number, anchor_l0_freq - target_l0, anchor_l0_freq, target_l0
                  call set_max_timestep(num_sigma_anchor_l0)
                  call finish
                  return
               end if
            end if
            s% max_years_for_timestep = max_dt_for_search
         else
            call set_max_timestep(0d0)
            if (target_star_is_expanding) then ! frequencies are dropping
               if (anchor_l0_freq < anchor_l0_obs + anchor_l0_sigma) then ! close to match for anchor
                  err = abs(anchor_l0_freq - anchor_l0_obs)/ &
                     (anchor_l0_atol + anchor_l0_rtol*anchor_l0_obs)
                  if (err < 1) then ! within tolerance so can stop
                     write(*,'(a,i6,6x,99f11.4)') 'terminate: have good match for anchor l0', &
                        s% model_number, anchor_l0_freq - anchor_l0_obs, anchor_l0_freq, anchor_l0_obs
                     do_adipls_extras_check_model = terminate
                  else if (anchor_l0_freq < anchor_l0_obs) then ! have gone beyond -- so retry
                     write(*,'(a,i6,6x,99f11.4)') 'retry to improve match for anchor l0', &
                        s% model_number, anchor_l0_freq - anchor_l0_obs, anchor_l0_freq, anchor_l0_obs
                     do_adipls_extras_check_model = retry
                     call finish
                     return
                  end if
                  if (anchor_l0_freq < anchor_l0_obs + 0.1*anchor_l0_sigma) &
                     s% max_years_for_timestep = min(max_dt_for_search, s% max_years_for_timestep)
               end if
            else ! contracting so frequencies are rising
               if (anchor_l0_freq > anchor_l0_obs - anchor_l0_sigma) then ! close to match for anchor
                  err = abs(anchor_l0_freq - anchor_l0_obs)/ &
                     (anchor_l0_atol + anchor_l0_rtol*anchor_l0_obs)
                  if (err < 1) then ! within tolerance so can stop
                     write(*,'(a,i6,6x,99f11.4)') 'terminate: have good match for anchor l0', &
                        s% model_number, anchor_l0_freq - anchor_l0_obs, anchor_l0_freq, anchor_l0_obs
                     do_adipls_extras_check_model = terminate
                  else if (anchor_l0_freq > anchor_l0_obs) then ! have gone beyond -- so retry
                     write(*,'(a,i6,6x,99f11.4)') 'retry to improve match for anchor l0', &
                        s% model_number, anchor_l0_freq - anchor_l0_obs, anchor_l0_freq, anchor_l0_obs
                     do_adipls_extras_check_model = retry
                     call finish
                     return
                  end if
                  if (anchor_l0_freq > anchor_l0_obs - 0.1*anchor_l0_sigma) &
                     s% max_years_for_timestep = min(max_dt_for_search, s% max_years_for_timestep)
               end if
            end if
            
            if (do_adipls_extras_check_model /= terminate) then
               write(*,'(a,i6,6x,99f11.4)') 'diff anchor l0/sigma', s% model_number, &
                  (anchor_l0_freq - anchor_l0_obs)/anchor_l0_sigma
               call finish
               return
            end if
         end if
      
               
         if (include_nonradial_chi_square) then
            max_el_for_chi_square = 2
            call get_l1_and_l2_info
         else
            max_el_for_chi_square = 0
         end if

         chi_square = get_chi_square( &
            a_div_r, s% Teff, s% log_surface_luminosity, log10(s% grav(1)), max_el_for_chi_square)
         write(*,'(a,i6,6x,99f11.4)') 'chi^2, diff anchor l0/sigma', s% model_number, &
            chi_square, (anchor_l0_freq - anchor_l0_obs)/anchor_l0_sigma

         if (best_model_chi_square > 0 .and. chi_square > best_model_chi_square) then 
            if (target_star_is_expanding) then
               if (anchor_l0_freq < anchor_l0_obs - 2*num_sigma_anchor_l0*anchor_l0_sigma) then
                  target_l0 = anchor_l0_obs - num_sigma_anchor_l0*anchor_l0_sigma
                  write(*,'(a,i6,6x,99f11.4)') 'terminate: anchor_l0_freq < lower bound', &
                     s% model_number, anchor_l0_freq - target_l0, anchor_l0_freq, target_l0
                  do_adipls_extras_check_model = terminate
               end if  
            else
               if (anchor_l0_freq > anchor_l0_obs + 2*num_sigma_anchor_l0*anchor_l0_sigma) then
                  target_l0 = anchor_l0_obs + num_sigma_anchor_l0*anchor_l0_sigma
                  write(*,'(a,i6,6x,99f11.4)') 'terminate: anchor_l0_freq > upper bound', &
                     s% model_number, anchor_l0_freq - target_l0, anchor_l0_freq, target_l0
                  do_adipls_extras_check_model = terminate
               end if  
            end if
            if (chi_square > max_chi2_increase_during_search*best_model_chi_square .and. &
                chi_square < 10*max_chi2_increase_during_search*best_model_chi_square) then
               ! the "chi_square <" is to filter out bogus false 1 shot values
               write(*,'(a,i6,6x,99f11.4)') 'terminate: chi^2 >> best', &
                  s% model_number, chi_square, best_model_chi_square
               do_adipls_extras_check_model = terminate
            end if         
         end if

         call finish
         
         
         contains
         
         
         subroutine finish
            if (best_model_chi_square < 0 .or. chi_square < best_model_chi_square) &
               call save_best_model_info(s)
            if (do_adipls_extras_check_model == terminate) call show_before_terminate
         end subroutine finish
         
         
         subroutine get_l1_and_l2_info
            call get_one_el_info(s, 1, &
                  nu_lower_factor*l1_obs(1), &
                  nu_upper_factor*l1_obs(nl1), &
                  iscan_factor_l1*nl1, 1, nl1)
            if (show_l1_results) call do_show_l1_results
            if (s% model_number == -1) call do_show_l1_results ! DEBUG
         
            call get_one_el_info(s, 2, &
                  nu_lower_factor*l2_obs(1), &
                  nu_upper_factor*l2_obs(nl2), &
                  iscan_factor_l2*nl2, 1, nl2)
            if (show_l2_results) call do_show_l2_results
            if (s% model_number == -1) call do_show_l2_results ! DEBUG
         end subroutine get_l1_and_l2_info
         
         
         logical function have_all_l0_freqs()
            integer :: i
            real(dp) :: prev
            prev = l0_freq(1)
            do i=2,nl0
               if (l0_obs(i) < 0) cycle
               if (l0_freq(i) == prev) then
                  have_all_l0_freqs = .false.
                  return
               end if
               prev = l0_freq(i)
            end do
            have_all_l0_freqs = .true.
         end function have_all_l0_freqs
         
         
         subroutine show_before_terminate
            X = initial_h1
            Y = initial_he3 + initial_he4
            Z = 1d0 - (X + Y)
            call show_best_model_info( &
               s, 6, s% star_mass, s% mixing_length_alpha, Y, Z/X, &
               next_f_ov_to_try, next_f0_ov_fraction_to_try)
         end subroutine show_before_terminate


         subroutine set_max_timestep(nsigma)
            real(dp), intent(in) :: nsigma ! e.g., num_sigma_anchor_l0 or 0
            real(dp) :: max_dt, target_l0
            logical :: set_max_dt
            include 'formats.dek'
            ! note: this assumes anchor_l0_freq is decreasing with age 
            if (s% star_age <= prev_age .or. prev_anchor_l0_freq <= 0) return      
            if (target_star_is_expanding) then ! frequencies are dropping
               target_l0 = anchor_l0_obs + nsigma*anchor_l0_sigma
               set_max_dt = ( &
                  prev_anchor_l0_freq > anchor_l0_freq .and. anchor_l0_freq > target_l0)
            else ! frequencies are increasing
               target_l0 = anchor_l0_obs - nsigma*anchor_l0_sigma
               set_max_dt = ( &
                  prev_anchor_l0_freq < anchor_l0_freq .and. anchor_l0_freq < target_l0)
            end if
            if (.not. set_max_dt) return
            max_dt = (s% star_age - prev_age) * &
               (anchor_l0_freq - target_l0)/(prev_anchor_l0_freq - target_l0)
            max_dt = max(max_dt, 10*max_dt_for_search)
            if (max_dt < s% max_years_for_timestep .or. &
                  s% max_years_for_timestep <= 0) then
               if (max_dt < 0) then
                  write(*,*) 'confusion in set_max_timestep -- max_dt < 0'
                  stop 1
               end if
               s% max_years_for_timestep = max_dt
            end if
         end subroutine set_max_timestep
         
         
         real(dp) function logL_logg_Teff_sigma2()
            include 'formats.dek'
            logL_logg_Teff_sigma2 = 0
            if (logL_sigma > 0) &
               logL_logg_Teff_sigma2 = logL_logg_Teff_sigma2 + &
                  ((s% log_surface_luminosity - logL_target)/logL_sigma)**2
            if (logg_sigma > 0) &
               logL_logg_Teff_sigma2 = logL_logg_Teff_sigma2 + &
                  ((logg - logg_target)/logg_sigma)**2
            if (Teff_sigma > 0) &
               logL_logg_Teff_sigma2 = logL_logg_Teff_sigma2 + &
                  ((s% Teff - Teff_target)/Teff_sigma)**2
         end function logL_logg_Teff_sigma2
         
         
         subroutine check_limits
            real(dp) :: logg_limit, logL_limit, Teff_limit
            include 'formats.dek'
            logL_limit = logL_target + logL_sigma*sigmas_coeff_for_logL_limit
            if (logL_sigma > 0 .and. s% log_surface_luminosity > logL_limit) then
               write(*,*) 'have reached logL limit'
               write(*,1) 'logL', s% log_surface_luminosity
               write(*,1) 'logL_limit', logL_limit
               write(*,*)
               do_adipls_extras_check_model = terminate
               return
            end if
            Teff_limit = Teff_target + Teff_sigma*sigmas_coeff_for_Teff_limit
            if (Teff_sigma > 0 .and. s% Teff < Teff_limit) then
               write(*,*) 'have reached Teff limit'
               write(*,1) 'Teff', s% Teff
               write(*,1) 'Teff_limit', Teff_limit
               write(*,*)
               do_adipls_extras_check_model = terminate
               return
            end if        
            logg = log10(s% grav(1))
            logg_limit = logg_target + logg_sigma*sigmas_coeff_for_logg_limit
            if (logg_sigma > 0 .and. logg < logg_limit) then
               write(*,*) 'have reached logg limit'
               write(*,1) 'logg', logg
               write(*,1) 'logg_limit', logg_limit
               write(*,*)
               do_adipls_extras_check_model = terminate
               return
            end if
         end subroutine check_limits
         

      end function do_adipls_extras_check_model

   
   
      subroutine read_oscillation_controls(ierr)
         integer, intent(out) :: ierr
         character (len=256) :: filename, message
         integer :: unit
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return
         filename = 'inlist_oscillation_controls'
         write(*,*) 'read ' // trim(filename)
         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(message, *) 'Failed to open control namelist file ', trim(filename)
            call alert(ierr, message)
            write(*,*) trim(message)
         else
            read(unit, nml=oscillation_controls, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') trim(message)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=oscillation_controls)
               close(unit)
               call alert(ierr, message)
            end if  
         end if
         call free_iounit(unit)
      end subroutine read_oscillation_controls



      
      subroutine save_sample_results_to_file(i_test, results_fname, ierr)
         use utils_lib
         integer, intent(in) :: i_test
         character (len=*), intent(in) :: results_fname
         integer, intent(out) :: ierr
         integer :: iounit
         write(*,*) 'save_sample_results_to_file ' // trim(results_fname)
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) stop 'alloc_iounit failed'
         open(unit=iounit, file=trim(results_fname), action='write', iostat=ierr)
         if (ierr /= 0) return
         call show_all_sample_results(iounit,i_test,ierr)
         close(iounit)
         call free_iounit(iounit)         
      end subroutine save_sample_results_to_file
      
      
      subroutine show_all_sample_results(iounit, i_test, ierr)
         use num_lib, only: safe_log10, qsort
         integer, intent(in) :: iounit, i_test
         integer, intent(out) :: ierr
         integer :: i, j
         integer, pointer :: index(:)
         ! sort results by increasing sample_chi_square
         allocate(index(i_test), stat=ierr)
         if (ierr /= 0) then
            stop 'failed in allocate before calling qsort from show_all_sample_results'
         end if
         call qsort(index, i_test, sample_chi_square)
         call show_sample_header
         do j = 1, i_test
            i = index(j)
            write(iounit,'(3x,i5,99f16.8)',advance='no') i, sample_chi_square(i)
            if (vary_mass) write(iounit,'(f16.8)',advance='no') sample_mass(i)
            if (vary_alpha) write(iounit,'(f16.8)',advance='no') sample_alpha(i)
            if (vary_Y) write(iounit,'(f16.8)',advance='no') sample_init_Y(i)
            if (vary_Z_div_X) write(iounit,'(f16.8)',advance='no') sample_init_Z_div_X(i)
            if (vary_f_ov) write(iounit,'(f16.8)',advance='no') sample_f_ov(i)
            if (vary_f0_ov_fraction) write(iounit,'(f16.8)',advance='no') sample_f0_ov_fraction(i)
            write(iounit,'(99f16.8)',advance='no') &
               safe_log10(sample_age(i)), &
               sample_radius(i), &
               sample_logL(i), &
               sample_Teff(i), &
               sample_logg(i), &
               sample_avg_delta_nu(i), &
               sample_a_div_r(i), &
               sample_correction_r(i)
            write(iounit,*)
         end do
         deallocate(index)
         call show_sample_header
         do i = 1, 3
            write(iounit,*)
         end do
         
         contains
         
         subroutine show_sample_header
            write(iounit,'(8x,a16)',advance='no') 'chi_square'
            if (vary_mass) write(iounit,'(a16)',advance='no') 'mass'
            if (vary_alpha) write(iounit,'(a16)',advance='no') 'alpha'
            if (vary_Y) write(iounit,'(a16)',advance='no') 'init_Y'
            if (vary_Z_div_X) write(iounit,'(a16)',advance='no') 'init_Z_div_X'
            if (vary_f_ov) write(iounit,'(a16)',advance='no') 'f_ov'
            if (vary_f0_ov_fraction) write(iounit,'(a16)',advance='no') 'f0_ov_frac'
            write(iounit,'(99a16)',advance='no') &
               'log age', &
               'radius', &
               'logL', &
               'Teff', &
               'logg', &
               'avg_delta_nu', &
               'a_div_r', &
               'correction_r'
            write(iounit,*)
         end subroutine show_sample_header
         
          
      end subroutine show_all_sample_results

      end module star_adipls_support
      