! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras_adipls

      use star_lib
      use star_def
      use const_def
      use utils_lib
      use star_adipls_support
      
      implicit none


      contains
      
      
      subroutine do_run_star_adipls
         use run_star_support
         type (star_info), pointer :: s
         integer :: id, ierr, i
         real(dp) :: chi2
         
         include 'formats.dek'
                  
         ierr = 0
         nullify(el, order, cyclic_freq, inertia)
         okay_to_restart = .true.

         call read_model_search_controls(ierr)
         if (failed('read_model_search_controls')) return

         call read_oscillation_controls(ierr)
         if (failed('read_oscillation_controls')) return

         call init_and_alloc(id, s, ierr)
         if (failed('init_and_alloc')) return
         
         star_id = id
                  
         call star_setup(id, 'inlist', ierr)
         if (failed('star_setup')) return
         
         initial_max_years_for_timestep = s% max_years_for_timestep
         
         call init_obs_data

         call run_adipls(s, .true., do_restribute_mesh, ierr)
         if (ierr /= 0) return
         
         next_mass_to_try = -1      
         next_alpha_to_try = -1
         next_Y_to_try = -1
         next_Z_div_X_to_try = -1
         next_f_ov_to_try = -1   
         next_f0_ov_fraction_to_try = -1   
         
         if (search_type == 'use_first_values' .or. just_call_my_extras_check_model) then
            chi2 = eval1(id, 1, ierr)
         else if (search_type == 'hooke') then
            call do_hooke(ierr)
         else if (search_type == 'bobyqa') then
            call do_bobyqa(ierr)
         else if (search_type == 'scan_grid') then
            call do_scan_grid(s, ierr)
         else 
            write(*,*) 'bad value for search_type ' // trim(search_type)
            ierr = -1
         end if

         
         contains
         

         logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (ierr /= 0)
            if (failed) then
               write(*, *) trim(str) // ' ierr', ierr
               write(*, '(a)') trim(alert_message)
            end if
         end function failed


      end subroutine do_run_star_adipls
   
   
      subroutine read_model_search_controls(ierr)
         integer, intent(out) :: ierr
         character (len=256) :: filename, message
         integer :: unit
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return
         filename = 'inlist_model_search_controls'
         write(*,*) 'read ' // trim(filename)
         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(message, *) 'Failed to open control namelist file ', trim(filename)
            call alert(ierr, message)
            write(*,*) trim(message)
         else
            read(unit, nml=model_search_controls, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(message, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') trim(message)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', delim='quote', status='old', iostat=ierr)
               read(unit, nml=model_search_controls)
               close(unit)
               call alert(ierr, message)
            end if  
         end if
         call free_iounit(unit)
      end subroutine read_model_search_controls

      
      double precision function eval1(id_in, itry, ierr)
         ! run star, save best results in trial(itry)
         ! don't assume final model is the best model.
         
         use run_star_support, only: initial_h1, initial_he3, initial_he4
         use run_star_support, only: run1_star
         use extras_support
         
         integer, intent(in) :: id_in, itry
         integer, intent(out) :: ierr
         logical, parameter :: do_init_and_alloc = .false., do_free_star = .false.
         logical :: restart
         integer :: id, i
         real(dp) :: mass, alpha, X, Y, Z, Z_div_X
         type (star_info), pointer :: s

         include 'formats.dek'
         ierr = 0
         id = id_in

         if (do_fake()) return
         
         ierr = 0
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         ! init for start of run
         best_model_chi_square = -1         
         prev_anchor_l0_freq = -1
         prev_age = -1 
         
         call run1_star( &
            do_init_and_alloc, do_free_star, okay_to_restart, &
            id, restart, &
            adipls_extras_controls, &
            adipls_extras_startup, &
            adipls_extras_check_model, &
            adipls_how_many_extra_log_columns, &
            adipls_data_for_extra_log_columns, &
            adipls_how_many_extra_profile_columns, &
            adipls_data_for_extra_profile_columns, &
            adipls_extras_finish_step, &
            adipls_extras_after_evolve)
         
         okay_to_restart = .false. ! only allow restart on 1st call
         
         eval1 = best_model_chi_square
         
         if (just_call_my_extras_check_model) return
         
         mass = s% star_mass
         alpha = s% mixing_length_alpha
         X = initial_h1
         Y = initial_he3 + initial_he4
         Z = 1 - (X+Y)
         Z_div_X = Z/X
         
         if (best_model_chi_square < 0) then
            write(*,*) 'failed to find chi^2 for this run'
            write(*,1) 'mass', mass
            write(*,1) 'alpha', alpha
            write(*,1) 'Y', Y
            write(*,1) 'Z_div_X', Z_div_X
            call zero_best_model_info
            best_model_chi_square = 999999d0
            call save_best_for_this_trial( &
               itry, mass, alpha, Y, Z_div_X, next_f_ov_to_try, next_f0_ov_fraction_to_try)
            return
         end if
         
         call save_best_for_this_trial( &
            itry, mass, alpha, Y, Z_div_X, next_f_ov_to_try, next_f0_ov_fraction_to_try)
         write(*,*)
         call show_trial_results(6)
         
         
         contains
         
         
         logical function do_fake()
            integer, parameter :: num_to_fake = 0
            real(dp), dimension(num_to_fake) :: fake_mass, fake_chi2
            
            include 'formats.dek'
            
            do_fake = .false.
         
!         fake_mass(:) = (/ &
!            1.130d0, &
!            1.135d0, &
!            1.1319098300562505d0, &
!            1.1330901699437494d0 /)
!            1.30331257d0, &
!            1.2719412752d0, &
!            1.2749034d0, &
!            1.2659311d0, &
!            1.2674368290075d0 /)
!         fake_chi2(:) = (/ &
!            12.3635d0, &
!            16.1301d0, &
!            19.1729d0, &
!            10.1313d0 /)
!            537.8358d0, &
!            105.3729d0, &
!            141.0619d0, &
!            93.5611d0, &
!            91.8541d0 /)
            if (itry <= num_to_fake) then
               if (abs(next_mass_to_try - fake_mass(itry)) > 1d-4) then
                  write(*,2) 'unexpected mass', itry, next_mass_to_try, fake_mass(itry)
                  stop 'eval1'
               end if
               !write(*,2) 'eval1 mass', itry, next_mass_to_try, fake_mass(itry)
               !write(*,*)
               num_trials = itry
               trial_mass(itry) = fake_mass(itry)
               trial_chi_square(itry) = fake_chi2(itry)
               eval1 = trial_chi_square(itry)
               do_fake = .true.
               return
            end if
            if (num_to_fake > 0) then
               do i=1,max(1,num_to_fake) ! try to stop bogus warnings from gfortran
                  if (abs(next_mass_to_try - fake_mass(i)) < 1d-8) then
                     write(*,3) 'repeat mass?', itry, i, next_mass_to_try, fake_mass(i)
                     stop 'eval1'
                  end if
               end do
            end if
         end function do_fake

         
      end function eval1
      
      
      subroutine do_scan_grid(s, ierr)
         use utils_lib
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i_test
         real(dp) :: Z_div_X, Y, alpha, mass, f_ov, f0_ov_fraction, chi2
         real(dp), parameter :: eps = 1d-6
         
         include 'formats.dek'
         
         ierr = 0
         i_test = 0
         
         Z_div_X = min_Z_div_X
         Y = min_Y
         alpha = min_alpha
         mass = min_mass
         f_ov = min_f_ov
         f0_ov_fraction = min_f0_ov_fraction        
         
         do while (f0_ov_fraction <= max_f0_ov_fraction + eps .or. .not. vary_f0_ov_fraction)          
            if (vary_f0_ov_fraction) next_f0_ov_fraction_to_try = f0_ov_fraction         
            do while (f_ov <= max_f_ov + eps .or. .not. vary_f_ov)          
               if (vary_f_ov) next_f_ov_to_try = f_ov        
               do while (Z_div_X <= max_Z_div_X + eps .or. .not. vary_Z_div_X)          
                  if (vary_Z_div_X) next_Z_div_X_to_try = Z_div_X            
                  do while (Y <= max_Y + eps .or. .not. vary_Y)           
                     if (vary_Y) next_Y_to_try = Y            
                     do while (alpha <= max_alpha + eps .or. .not. vary_alpha)                 
                        if (vary_alpha) next_alpha_to_try = alpha                  
                        do while (mass <= max_mass + eps .or. .not. vary_mass)             
                           if (vary_mass) next_mass_to_try = mass                     
                           call do1_grid(ierr)
                           if (ierr /= 0) return
                           if (delta_mass <= 0 .or. .not. vary_mass) exit
                           mass = mass + delta_mass                                       
                        end do
                        mass = min_mass                                    
                        if (delta_alpha <= 0 .or. .not. vary_alpha) exit
                        alpha = alpha + delta_alpha                  
                     end do
                     alpha = min_alpha               
                     if (delta_Y <= 0 .or. .not. vary_Y) exit
                     Y = Y + delta_Y               
                  end do
                  Y = min_Y            
                  if (delta_Z_div_X <= 0 .or. .not. vary_Z_div_X) exit
                  Z_div_X = Z_div_X + delta_Z_div_X            
               end do
               Z_div_X = min_Z_div_X            
               if (delta_f_ov <= 0 .or. .not. vary_f_ov) exit
               f_ov = f_ov + delta_f_ov            
            end do
            f_ov = min_f_ov            
            if (delta_f0_ov_fraction <= 0 .or. .not. vary_f0_ov_fraction) exit
            f0_ov_fraction = f0_ov_fraction + delta_f0_ov_fraction            
         end do
         f0_ov_fraction = min_f0_ov_fraction         
         
         
         contains
         
         
         subroutine do1_grid(ierr)
            integer, intent(out) :: ierr
            include 'formats.dek'
            ierr = 0
                     
            chi2 = eval1(s% id, 1, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in eval1'
               write(*,1) 'next_mass_to_try', next_mass_to_try
               write(*,1) 'next_alpha_to_try', next_alpha_to_try
               write(*,1) 'next_Y_to_try', next_Y_to_try
               write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try
               return
            end if
            
            i_test = i_test + 1
            call save_best_for_sample(i_test)

            write(*,*)
            write(*,*) 'current set of sample results'
            call show_all_sample_results(6,i_test,ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in show_all_sample_results'
               return
            end if

            write(*,*)
            call save_sample_results_to_file(i_test,'sample_results.data',ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in save_sample_results_to_file'
               return
            end if
            
         end subroutine do1_grid

         
      end subroutine do_scan_grid


      subroutine bobyqa_fun(n,x,f)
         integer, intent(in) :: n
         double precision, intent(in) :: x(*)
         double precision, intent(out) :: f
         integer :: ierr
         include 'formats.dek'
         
         ierr = 0
         
         if (vary_mass) then
            next_mass_to_try = bobyqa_param(x(i_mass), first_mass, min_mass, max_mass)
            write(*,1) 'next_mass_to_try', next_mass_to_try
         end if
         
         if (vary_alpha) then
            next_alpha_to_try = bobyqa_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
            write(*,1) 'next_alpha_to_try', next_alpha_to_try, x(i_alpha)
         end if
         
         if (vary_Y) then
            next_Y_to_try = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)
            write(*,1) 'next_Y_to_try', next_Y_to_try, x(i_Y)
         end if

         if (vary_Z_div_X) then
            next_Z_div_X_to_try = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
            write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try, x(i_Z_div_X)
         end if
         
         if (vary_f_ov) then
            next_f_ov_to_try = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
            write(*,1) 'next_f_ov_to_try', next_f_ov_to_try, x(i_f_ov)
         end if
         
         if (vary_f0_ov_fraction) then
            next_f0_ov_fraction_to_try = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
            if (next_f0_ov_fraction_to_try < 0) next_f0_ov_fraction_to_try = 0
            write(*,1) 'next_f0_ov_fraction_to_try', next_f0_ov_fraction_to_try, x(i_f0_ov_fraction)
         end if

         f = eval1(star_id, 1, ierr)
         if (ierr /= 0) then
            write(*,*) 'got ierr from eval1'
            stop 'bobyqa_fun'
         end if
         
         minimize_i_test = minimize_i_test + 1
         call save_best_for_sample(minimize_i_test)

         write(*,*)
         write(*,*) 'current set of sample results'
         call show_all_sample_results(6,minimize_i_test,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in show_all_sample_results'
            stop 'bobyqa_fun'
         end if
         
         min_sample_chi2_so_far = minval(sample_chi_square(1:minimize_i_test))

         write(*,*)
         call save_sample_results_to_file(minimize_i_test,'bobyqa_results.txt',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in save_sample_results_to_file'
            stop 'bobyqa_fun'
         end if
         
      end subroutine bobyqa_fun

      
      real(dp) function bobyqa_param(x, first, min, max)
         real(dp), intent(in) :: x, first, min, max
         if (x > 0) then
            bobyqa_param = first + x*(max-first)
         else
            bobyqa_param = first + x*(first-min)
         end if
      end function bobyqa_param
      
      
      subroutine do_bobyqa(ierr)
         use num_lib
         integer, intent(out) :: ierr
         integer, parameter :: maxfun = 1000, iprint = 0
         real(dp), pointer, dimension(:) :: xl, xu, x, w
         real(dp) :: min_chi2, rhobeg, max_bobyqa_value
         integer :: i, npt
         include 'formats.dek'
         ierr = 0
         minimize_i_test = 0
         nvar = 0
         
         write(*,*)
         write(*,*)
                  
         if (vary_mass) then
            nvar = nvar+1; i_mass = nvar
            if (min_mass >= max_mass) then
               write(*,1) 'min_mass >= max_mass', min_mass, max_mass
               ierr = -1
            end if
         end if
         
         if (vary_alpha) then
            nvar = nvar+1; i_alpha = nvar
            if (min_alpha >= max_alpha) then
               write(*,1) 'min_alpha >= max_alpha', min_alpha, max_alpha
               ierr = -1
            end if
         end if
         
         if (vary_Y) then
            nvar = nvar+1; i_Y = nvar
            if (min_Y >= max_Y) then
               write(*,1) 'min_Y >= max_Y', min_Y, max_Y
               ierr = -1
            end if
         end if
         
         if (vary_Z_div_X) then
            nvar = nvar+1; i_Z_div_X = nvar
            if (min_Z_div_X >= max_Z_div_X) then
               write(*,1) 'min_Z_div_X >= max_Z_div_X', min_Z_div_X, max_Z_div_X
               ierr = -1
            end if
         end if
         
         if (vary_f_ov) then
            nvar = nvar+1; i_f_ov = nvar
            if (min_f_ov >= max_f_ov) then
               write(*,1) 'min_f_ov >= max_f_ov', min_f_ov, max_f_ov
               ierr = -1
            end if
         end if
         
         if (vary_f0_ov_fraction) then
            nvar = nvar+1; i_f0_ov_fraction = nvar
            if (min_f0_ov_fraction >= max_f0_ov_fraction) then
               write(*,1) 'min_f0_ov_fraction >= max_f0_ov_fraction', &
                  min_f0_ov_fraction, max_f0_ov_fraction
               ierr = -1
            end if
         end if
         if (ierr /= 0) return
         
         npt = 2*nvar + 1
         
         allocate(xl(nvar), xu(nvar), x(nvar), w((npt+5)*(npt+nvar)+3*nvar*(nvar+5)/2))
         
         XL(1:nvar) = 0
         X(1:nvar) = 0
         XU(1:nvar) = 1
         
!       RHOBEG and RHOEND must be set to the initial and final values of a trust
!       region radius, so both must be positive with RHOEND no greater than
!       RHOBEG. Typically, RHOBEG should be about one tenth of the greatest
!       expected change to a variable, while RHOEND should indicate the
!       accuracy that is required in the final values of the variables. An
!       error return occurs if any of the differences XU(I)-XL(I), I=1,...,N,
!       is less than 2*RHOBEG.
         rhobeg = 0.45d0
         max_bobyqa_value = 1d6
         call bobyqa(nvar,npt,x,xl,xu,rhobeg,rhoend,iprint,maxfun,w,bobyqa_fun,max_bobyqa_value)
      
         if (vary_mass) &
            final_mass = bobyqa_param(x(i_mass), first_mass, min_mass, max_mass)
         
         if (vary_Y) &
            final_Y = bobyqa_param(x(i_Y), first_Y, min_Y, max_Y)

         if (vary_Z_div_X) &
            final_Z_div_X = bobyqa_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
         
         if (vary_f_ov) &
            final_f_ov = bobyqa_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
         
         if (vary_f0_ov_fraction) &
            final_f0_ov_fraction = &
               bobyqa_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
         
         deallocate(xl, xu, x, w)
      
      end subroutine do_bobyqa




      double precision function hooke_f(x,n)
         integer, intent(in) :: n
         double precision, intent(in) :: x(n)
         
         integer :: ierr
         include 'formats.dek'
         
         ierr = 0
         
         write(*,*)
         write(*,*)
         
         if (vary_mass) then
            next_mass_to_try = hooke_param(x(i_mass), first_mass, min_mass, max_mass)
            write(*,1) 'next_mass_to_try', next_mass_to_try, x(i_mass)
         end if
         
         if (vary_alpha) then
            next_alpha_to_try = hooke_param(x(i_alpha), first_alpha, min_alpha, max_alpha)
            write(*,1) 'next_alpha_to_try', next_alpha_to_try, x(i_alpha)
         end if
         
         if (vary_Y) then
            next_Y_to_try = hooke_param(x(i_Y), first_Y, min_Y, max_Y)
            write(*,1) 'next_Y_to_try', next_Y_to_try, x(i_Y)
         end if

         if (vary_Z_div_X) then
            next_Z_div_X_to_try = hooke_param(x(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
            write(*,1) 'next_Z_div_X_to_try', next_Z_div_X_to_try, x(i_Z_div_X)
         end if
         
         if (vary_f_ov) then
            next_f_ov_to_try = hooke_param(x(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
            write(*,1) 'next_f_ov_to_try', next_f_ov_to_try, x(i_f_ov)
         end if
         
         if (vary_f0_ov_fraction) then
            next_f0_ov_fraction_to_try = &
               hooke_param(x(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)
            if (next_f0_ov_fraction_to_try < 0) next_f0_ov_fraction_to_try = 0
            write(*,1) 'next_f0_ov_fraction_to_try', next_f0_ov_fraction_to_try, x(i_f0_ov_fraction)
         end if
         
         if (maxval(x(1:n)) > 1 .or. minval(x(1:n)) < -1) then
            write(*,*) 'reject because parameter out of range'
            write(*,*)
            write(*,*)
            hooke_f = 1d99
            return
         end if
         
         hooke_f = eval1(star_id, 1, ierr)
         if (ierr /= 0) then
            write(*,*) 'got ierr from eval1'
            stop 'hooke_f'
         end if
         
         minimize_i_test = minimize_i_test + 1
         call save_best_for_sample(minimize_i_test)

         write(*,*)
         write(*,*) 'current set of sample results'
         call show_all_sample_results(6,minimize_i_test,ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in show_all_sample_results'
            stop 'hooke_f'
         end if

         write(*,*)
         call save_sample_results_to_file(minimize_i_test,'hooke_results.txt',ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in save_sample_results_to_file'
            stop 'hooke_f'
         end if
         
      end function hooke_f

      
      real(dp) function hooke_param(x, first, min, max)
         real(dp), intent(in) :: x, first, min, max
         if (x > 0) then
            hooke_param = first + hooke_xscale*x*(max-first)
         else
            hooke_param = first + hooke_xscale*x*(first-min)
         end if
      end function hooke_param
      
      
      subroutine do_hooke(ierr)
         use num_lib
         integer, intent(out) :: ierr
         real(dp), pointer :: startpt(:), endpt(:)
         real(dp) :: rho, eps, final_mass, final_alpha, final_Y, final_Z_div_X
         integer :: itermax, num_iters
         include 'formats.dek'
         
         ierr = 0
         minimize_i_test = 0
         if (vary_mass) then
            nvar = nvar+1; i_mass = nvar
            if (min_mass >= max_mass) then
               write(*,1) 'min_mass >= max_mass', min_mass, max_mass
               ierr = -1
            end if
         end if
         
         if (vary_alpha) then
            nvar = nvar+1; i_alpha = nvar
            if (min_alpha >= max_alpha) then
               write(*,1) 'min_alpha >= max_alpha', min_alpha, max_alpha
               ierr = -1
            end if
         end if
         
         if (vary_Y) then
            nvar = nvar+1; i_Y = nvar
            if (min_Y >= max_Y) then
               write(*,1) 'min_Y >= max_Y', min_Y, max_Y
               ierr = -1
            end if
         end if
         
         if (vary_Z_div_X) then
            nvar = nvar+1; i_Z_div_X = nvar
            if (min_Z_div_X >= max_Z_div_X) then
               write(*,1) 'min_Z_div_X >= max_Z_div_X', min_Z_div_X, max_Z_div_X
               ierr = -1
            end if
         end if
         
         if (vary_f_ov) then
            nvar = nvar+1; i_f_ov = nvar
            if (min_f_ov >= max_f_ov) then
               write(*,1) 'min_f_ov >= max_f_ov', min_f_ov, max_f_ov
               ierr = -1
            end if
         end if
         
         if (vary_f0_ov_fraction) then
            nvar = nvar+1; i_f0_ov_fraction = nvar
            if (min_f0_ov_fraction >= max_f0_ov_fraction) then
               write(*,1) 'min_f0_ov_fraction >= max_f0_ov_fraction', &
                  min_f0_ov_fraction, max_f0_ov_fraction
               ierr = -1
            end if
         end if
         if (ierr /= 0) return
         
         allocate(startpt(nvar), endpt(nvar))         
         ! initialize to 0 means start with the given first_values
         startpt(:) = 0
         
         itermax = hooke_itermax
         rho = hooke_rho
         eps = hooke_eps
         
         write(*,2) 'hooke_itermax', hooke_itermax
         write(*,1) 'hooke_rho', hooke_rho
         write(*,1) 'hooke_eps', hooke_eps

         num_iters = hooke(nvar, startpt, endpt, rho, eps, itermax, hooke_f)
         
         if (vary_mass) &
            final_mass = hooke_param(endpt(i_mass), first_mass, min_mass, max_mass)
         
         if (vary_alpha) &
            final_alpha = hooke_param(endpt(i_alpha), first_alpha, min_alpha, max_alpha)
         
         if (vary_Y) &
            final_Y = hooke_param(endpt(i_Y), first_Y, min_Y, max_Y)

         if (vary_Z_div_X) &
            final_Z_div_X = hooke_param(endpt(i_Z_div_X), first_Z_div_X, min_Z_div_X, max_Z_div_X)
         
         if (vary_f_ov) &
            final_f_ov = hooke_param(endpt(i_f_ov), first_f_ov, min_f_ov, max_f_ov)
         
         if (vary_f0_ov_fraction) &
            final_f0_ov_fraction = &
               hooke_param(endpt(i_f0_ov_fraction), first_f0_ov_fraction, min_f0_ov_fraction, max_f0_ov_fraction)

         deallocate(startpt, endpt)         
      
      end subroutine do_hooke



      end module run_star_extras_adipls
