//!
//! Contains the implementation for the seismic forward model.
//!
//! \file fwdmodel/seismic.cpp
//! \author Nahid Akbar
//! \date 2014
//! \license Affero General Public License version 3 or later
//! \copyright (c) 2014, NICTA
//!

#include "fwdmodel/fwd.hpp"
#include "world/interpolate.hpp"
#include "world/voxelise.hpp"
#include "world/property.hpp"

namespace obsidian
{
  namespace fwd
  {
    //! Generate a cache object for a seismic forward model.
    //!
    //! \param boundaryInterpolation The world model interpolation parameters.
    //! \param worldSpec The world model specification.
    //! \param spec The forward model specification.
    //! \returns Forward model cache object.
    //!
    template<>
    Seismic1dCache generateCache<ForwardModel::SEISMIC1D>(const std::vector<world::InterpolatorSpec>& boundaryInterpolation,
                                                          const WorldSpec& worldSpec, const Seismic1dSpec& spec)
    {
      return
      {
        boundaryInterpolation,
        world::Query(boundaryInterpolation, worldSpec, spec.locations.leftCols(2))
      };
    }

    //! Run a seismic forward model.
    //! 
    //! \param spec The forward model specification.
    //! \param cache The forward model cache generated by generateCache().
    //! \param world The world model parameters.
    //! \returns Forward model results.
    //!
    template<>
    Seismic1dResults forwardModel<ForwardModel::SEISMIC1D>(const Seismic1dSpec& spec, const Seismic1dCache& cache, const WorldParams& world)
    {
      Eigen::MatrixXd transitions = world::getTransitions(cache.boundaryInterpolation, world, cache.query); // mqueries (mlayer) x nlocations
      Eigen::MatrixXd thicknesses = world::thickness(transitions); // (nlocations) x mthicknesses (layers - 1)
      Eigen::VectorXd velocity = world::extractProperty(world, RockProperty::PWaveVelocity); // nlayer x 1
      Seismic1dResults results;
      results.readings.resize(spec.locations.rows());
      for (uint location = 0; location < spec.locations.rows(); location++)
      {
        std::vector<double> reflections(thicknesses.cols());
        for (uint i = 0; i < thicknesses.cols(); i++)
        {
          reflections[i] = 2.0 * thicknesses(location, i) / velocity(i, 0);
        }
        for (uint i = 1; i < thicknesses.cols(); i++)
        {
          reflections[i] += reflections[i - 1];
        }
        results.readings[location].resize(spec.interfaces[location].size());
        for (uint i = 0; i < spec.interfaces[location].size(); i++)
        {
          results.readings[location](i) = reflections[spec.interfaces[location](i)];
        }
      }
      return results;
    }

  } // namespace fwd
} // namespace obsidian
