//!
//! Contains the implementation for the gravity forward model.
//!
//! \file fwdmodel/gravity.cpp
//! \author Darren Shen
//! \author Alistair Reid
//! \date 2014
//! \license Affero General Public License version 3 or later
//! \copyright (c) 2014, NICTA
//!

#include <glog/logging.h>
#include "gravity.hpp"
#include "world/voxelise.hpp"

namespace obsidian
{
  namespace fwd
  {
    //! Generate a cache object for a gravity forward model.
    //!
    //! \param boundaryInterpolation The world model interpolation parameters.
    //! \param worldSpec The world model specification.
    //! \param gravSpec The forward model specification.
    //! \returns Forward model cache object.
    //!
    template<>
    GravCache generateCache<ForwardModel::GRAVITY>(const std::vector<world::InterpolatorSpec>& boundaryInterpolation,
                                                   const WorldSpec& worldSpec, const GravSpec& gravSpec)
    {
      LOG(INFO)<< "Caching grav sensitivity...";
      const VoxelSpec& gravVox = gravSpec.voxelisation;
      world::Query gravQuery(boundaryInterpolation, worldSpec, gravVox.xResolution, gravVox.yResolution,
          gravVox.zResolution, world::SamplingStrategy::noAA);

      GravmagInterpolatorParams interpParams = makeInterpParams(gravVox, gravSpec.locations, worldSpec);

      return // GravCache
      {
        boundaryInterpolation,
        gravQuery,
        fwd::gravSens(gravQuery.edgeX, gravQuery.edgeY,
            gravQuery.edgeZ, interpParams.gridLocations),
        interpParams.sensorIndices,
        interpParams.sensorWeights
      };
    }

    //! Run a gravity forward model.
    //! 
    //! \param spec The forward model specification.
    //! \param cache The forward model cache generated by generateCache().
    //! \param world The world model parameters.
    //! \returns Forward model results.
    //!
    template<>
    GravResults forwardModel<ForwardModel::GRAVITY>(const GravSpec& spec, const GravCache& cache, const WorldParams& world)
    {
      Eigen::MatrixXd densities = world::getVoxels(cache.boundaryInterpolation, world, cache.query, RockProperty::Density);
      GravResults results;
      results.readings = fwd::computeField(cache.sensitivityMatrix, cache.sensorIndices, cache.sensorWeights, world::flatten(densities));
      return results;
    }

    namespace detail
    {
      //! Computes the sensitivity for a particular point in the gravity.
      //!
      //! \param x, y, z The coordinate of the point in 3D space
      //!
      double gravSensFunc(double x, double y, double z, double, double, double)
      {
        // Compute the displacement
        double r = std::sqrt(x * x + y * y + z * z);      // + detail::EPS;

        // Compute the actual sensitivity
        return x * (y + r <= 0 ? 0 : std::log(y + r)) + y * (x + r <= 0 ? 0 : std::log(x + r))
            - z * std::atan((x * y) / (z * r + detail::EPS));
      }
    } // namespace detail

    Eigen::MatrixXd gravSens(const Eigen::VectorXd &xEdges, const Eigen::VectorXd &yEdges, const Eigen::VectorXd &zEdges,
                             const Eigen::MatrixXd &locations)
    {
      constexpr double G = 6.673848e-11;
      constexpr double SI_TO_MILLIGALS = 1.0e5;
      constexpr double GCM3_TO_SI = 1000.0;
      constexpr double MILLIGALS_UNITS = (G * SI_TO_MILLIGALS * GCM3_TO_SI);

      return MILLIGALS_UNITS * detail::computeSensitivity(xEdges, yEdges, zEdges, xEdges, yEdges, zEdges, locations, detail::gravSensFunc);
    }
  } // namespace fwd
} // namespace obsidian
