//!
//! Contains global forward model evaluation functions
//!
//! \file fwdmodel/global.hpp
//! \author Lachlan McCalman
//! \date 2014
//! \license Affero General Public License version 3 or later
//! \copyright (c) 2014, NICTA
//!

#pragma once

#include "datatype/datatypes.hpp"
#include "world/interpolate.hpp"
#include "fwdmodel/fwd.hpp"

namespace obsidian
{
  namespace fwd
  {
    //! Generate a global cache object for all forward models.
    //!
    //! \param boundaryInterpolation The world model interpolation parameters.
    //! \param spec The global world specifications.
    //! \returns Cache object containing the caches of all the forward models.
    //!
    GlobalCache generateGlobalCache(const std::vector<world::InterpolatorSpec>& boundaryInterpolation, const GlobalSpec& spec,
                                    const std::set<ForwardModel>& enabled)
    {
      return
      {
        enabled.count(ForwardModel::GRAVITY) ? generateCache<ForwardModel::GRAVITY>(boundaryInterpolation, spec.world, spec.grav) : GravCache(),
        enabled.count(ForwardModel::MAGNETICS) ? generateCache<ForwardModel::MAGNETICS>(boundaryInterpolation, spec.world, spec.mag) : MagCache(),
        enabled.count(ForwardModel::MTANISO) ? generateCache<ForwardModel::MTANISO>(boundaryInterpolation, spec.world, spec.mt) : MtAnisoCache(),
        enabled.count(ForwardModel::SEISMIC1D) ? generateCache<ForwardModel::SEISMIC1D>(boundaryInterpolation, spec.world, spec.s1d) : Seismic1dCache(),
        enabled.count(ForwardModel::CONTACTPOINT) ? generateCache<ForwardModel::CONTACTPOINT>(boundaryInterpolation, spec.world, spec.cpoint): ContactPointCache(),
        enabled.count(ForwardModel::THERMAL) ? generateCache<ForwardModel::THERMAL>(boundaryInterpolation, spec.world, spec.therm): ThermalCache(),
        enabled.count(ForwardModel::FIELDOBS) ? generateCache<ForwardModel::FIELDOBS>(boundaryInterpolation, spec.world, spec.field): FieldObsCache()
      };
    }

    //! Run all the forward models.
    //!
    //! \param spec The global world specifications.
    //! \param cache The global forward model cache generated by generateCacheGlobal().
    //! \param params The global world parameters.
    //! \returns Results of all the forward models.
    //!
    GlobalResults forwardModelAll(const GlobalSpec& spec, const GlobalCache& cache, const GlobalParams& params,
                                  const std::set<ForwardModel>& enabled)
    {
      return
      {
        enabled.count(ForwardModel::GRAVITY) ? forwardModel<ForwardModel::GRAVITY>(spec.grav, cache.grav, params.world) : GravResults(),
        enabled.count(ForwardModel::MAGNETICS) ? forwardModel<ForwardModel::MAGNETICS>(spec.mag, cache.mag, params.world) : MagResults(),
        enabled.count(ForwardModel::MTANISO) ? forwardModel<ForwardModel::MTANISO>(spec.mt, cache.mt, params.world) : MtAnisoResults(),
        enabled.count(ForwardModel::SEISMIC1D) ? forwardModel<ForwardModel::SEISMIC1D>(spec.s1d, cache.s1d, params.world) : Seismic1dResults(),
        enabled.count(ForwardModel::CONTACTPOINT) ? forwardModel<ForwardModel::CONTACTPOINT>(spec.cpoint, cache.cpoint, params.world) : ContactPointResults(),
        enabled.count(ForwardModel::THERMAL) ? forwardModel<ForwardModel::THERMAL>(spec.therm, cache.therm, params.world) : ThermalResults(),
        enabled.count(ForwardModel::FIELDOBS) ? forwardModel<ForwardModel::FIELDOBS>(spec.field, cache.field, params.world) : FieldObsResults()
      };
    }
  }
}
