//!
//! Contains the implementation for the field observation forward model.
//!
//! \file fwdmodel/fieldobs.cpp
//! \author Richard Scalzo
//! \date 2018
//! \license Affero General Public License version 3 or later
//! \copyright (c) 2014, 2018 NICTA + USyd
//!

#include "fwd.hpp"
#include "world/voxelise.hpp"
#include <glog/logging.h>

namespace obsidian
{
  namespace fwd
  {
    //! Generate a cache object for a field observation forward model.
    //!
    //! \param boundaryInterpolation The world model interpolation parameters.
    //! \param worldSpec The world model specification.
    //! \param spec The forward model specification.
    //! \returns Forward model cache object.
    //!
    template<>
    FieldObsCache generateCache<ForwardModel::FIELDOBS>(const std::vector<world::InterpolatorSpec>& boundaryInterpolation,
                                                                const WorldSpec& worldSpec, const FieldObsSpec& spec)
    {
      LOG(INFO) << "Caching field obs parameters...";
      return
      {
        boundaryInterpolation,
        world::Query(boundaryInterpolation, worldSpec, spec.locations.leftCols(2))
      };
    }

    //! Run a field observation forward model.
    //! 
    //! \param spec The forward model specification.
    //! \param cache The forward model cache generated by generateCache().
    //! \param world The world model parameters.
    //! \returns Forward model results.
    //!
    template<>
    FieldObsResults forwardModel<ForwardModel::FIELDOBS>(const FieldObsSpec& spec, const FieldObsCache& cache,
                                                                 const WorldParams& world)
    {
      Eigen::MatrixXd transitions = world::getTransitions(cache.boundaryInterpolation, world, cache.query); // mqueries (mlayer) x nlocations
      FieldObsResults results;
      results.readings.resize(spec.locations.rows());
      for (uint location = 0; location < spec.locations.rows(); location++)
      {
        // Count down through the layers.  The transition boundaries are lower
        // limits for the rock layers, so the sensor result will be the
        // smallest positive depth.
        results.readings[location] = transitions.rows() - 1;
        for (int layer = 1; layer < transitions.rows(); layer++)
        {
          if (transitions(layer, location) > transitions(0, location))
          {
            results.readings[location] = layer - 1;
            break;
          }
        }
      }
      return results;
    }

  } // namespace fwd
} // namespace obsidian
