# -*- coding: utf-8 -*-
"""
Created on Sat Mar 12 15:08:18 2016

@author: itayshom
"""

import os
import glob
import re
import numpy


def parse_params_file(filename):
    """
    read a file (typically .spe) where each line is of the format 'param = value'.
    return a dictionary of string params and corresponding float values.
    """
    with open(filename) as f:
        d = dict(re.match('\[(\S*)\]\s*=\s*(.*)', line).groups() for line in f)

    for k, v in d.items():
        try:
            d[k] = float(v)
        except ValueError:
            d[k] = v

    return d


def get_sweep_files(folder, prefix, reverse=True):
    """
    Scan for all files named 'prefixNNN.txt' in folder.
    prefix is name of data and NNN is numeric value defining elements of the
    dataset (e.g. different probing powers)

    Also read the accompanying 'prefixNNN.spe' file containing various
    experimental parameters used in data taking.

    Return tuple of lists: values, filenames, parameters
    """
    s = os.path.join(folder, prefix)

    filenames = glob.glob(s + '*.txt')

    if not filenames:
        raise(FileNotFoundError)

    a = [(float(re.match('[A-Za-z_]*([-+0-9.]+)[A-Za-z_]*([-+0-9.]*)\.txt', f[len(s):]).group(1)), f, parse_params_file(f.replace('.txt', '.spe'))) for f in filenames]

    # sort (num, filename, params) tuples by num
    a.sort(key=lambda x: x[0], reverse=reverse)

    return zip(*a)


def concat_sweep_files(folder, prefix, reverse=True, single_freq=True):
    """
    Read complete data and experiemental parameters in files 'prefixNNN'.
    each file contains two columns, X,Y.
    Return concatenated matrices of X values and Y values.
    Also return a dictionary containing lists of experimental parameters.

    single_freq:    If true, all data files are taken to use the same x-axis,
                    returned as a vector.
    """
    nums, filenames, params = get_sweep_files(folder, prefix, reverse=reverse)

    n = len(nums)

    for i in range(n):
        A = numpy.loadtxt(filenames[i], delimiter=',')

        if i==0:
            if single_freq:
                X = A[:,0]
            else:
                X = numpy.zeros((A.shape[0], n))

            Y = numpy.zeros((A.shape[0], n))

        if not single_freq:
            X[:,i] = A[:,0]

        Y[:,i] = A[:,1]

    # convert list of dictionaries to a dictionary of lists
    # keep non-floats as original types
    p = {}
    for k in params[0]:
        try:
            p[k] = numpy.array([i[k] for i in params], dtype=numpy.float64)
        except ValueError:
            p[k] = [i[k] for i in params]
        except KeyError:
            # this occurs if a parameter is absent from a later data,
            # e.g. the pressure was recorded for data point 0 but omitted
            # starting from data point 10
            p[k] = None

    return X, Y, p, n