

##############################################################
#                          Packages                          #
##############################################################

library(metafor)
library(dplyr)
library(devtools)
install_github("daniel1noble/metaAidR")
library(metaAidR)

##############################################################
#                       Functions                            #
##############################################################

# function to calculate the unbiased log response ratio (lnRRu) and variance
lnRRu <- function(mean_f, mean_m, sd_f, sd_m, Nf, Nm){
  
  # lnR components
  logR <- log(mean_f / mean_m)
  bf   <- sd_f^2 / (Nf * mean_f^2)
  bm   <- sd_m^2 / (Nm * mean_m^2)
  # prepare dataframe
  lnRRu           <- data.frame(matrix(0, ncol = 2, nrow = length(mean_f)))
  colnames(lnRRu) <- c("yi","vi")
  # calculate effect size lnRRu
  lnRRu$yi        <- logR + 0.5 * (bf - bm)
  
  # variance components
  vcm <- sd_m^2 / (Nm * mean_m^2)
  vcf <- sd_f^2 / (Nf * mean_f^2)
  b2m <- sd_m^4 / (Nm^2 * mean_m^4)
  b2f <- sd_f^4 / (Nf^2 * mean_f^4)
  # calculate effect size variance
  lnRRu$vi <- vcm + vcf + 0.5 * (b2f - b2m)
  
  return(lnRRu)
}

# function to calculate the log odds ratio (lnOR) and variance
lnOR <- function(win1, fail1, win2, fail2){
  
  # prepare dataframe
  lnOR           <- data.frame(matrix(0, ncol = 2, nrow = length(win1)))
  colnames(lnOR) <- c("yi","vi")
  # calculate log odds
  lnOR$yi        <- log((win1*fail2)/(win2*fail1))
  # calculate variance
  lnOR$vi        <- (1/win1) + (1/win2) + (1/fail1) + (1/fail2)

  return(lnOR)
}

# function to calculate I2 and CIs for two random effects
# based on I2 function by Daniel W. A. Noble
I2_new <- function(model, v){
  
  # extract variance components for both random effects from nullmodel 
  # and use it to simulate (MonteCarlo) 1500 sigmas to calculate CIs
  sigma2           <- matrix(model$sigma2, nrow = 1, ncol = length(model$sigma2))
  colnames(sigma2) <- model$s.names
  Sims             <- data.frame(mapply(function(x, y) 
                                 simMonteCarlo(x,y, sims = 1500), 
                                 sigma2, model$s.nlevels))
  colnames(Sims)   <- colnames(sigma2)
  
  # calculate I2 and upper and lower CI
  wi       <- 1/v
  s2       <- sum((wi) * (length(wi) - 1))/(((sum(wi)^2) - sum((wi)^2)))
  VT       <- rowSums(Matrix::cBind(Sims, s2))
  tau      <- rowSums(Sims)
  I2_total <- Sims/VT
  I2_er    <- s2/VT
  
  tmpMatrix <- Matrix::cBind(between=I2_total[, -match(model$s.names[1], colnames(I2_total))], total = tau/VT, error=I2_er)
  
  # confidence intervals
  CI1 <- t(as.data.frame(quantile(tmpMatrix[,1], c(0.025, 0.975), na.rm = TRUE)))
  CI2 <- t(as.data.frame(quantile(tmpMatrix[,2], c(0.025, 0.975), na.rm = TRUE)))
  CI3 <- t(as.data.frame(quantile(tmpMatrix[,3], c(0.025, 0.975), na.rm = TRUE)))
  
  # I2
  I2  <- t(as.data.frame(colMeans(tmpMatrix)))
  
  # create table to return
  I2_table           <- as.data.frame(rbind(cbind(I2[1,1],CI1),cbind(I2[1,2],CI2),cbind(I2[1,3],CI3)))
  rownames(I2_table) <- c("between", "total", "error")
  colnames(I2_table) <- c("I2","lower 95% CI","upper 95% CI")
  
  
  return(I2_table)
}


##############################################################
#                            Data                            #
##############################################################

# set working directory
setwd()
# load data
sex_diff_raw <- read.csv("RawSexMetaAnalysisFinal.csv")
# check data
str(sex_diff_raw)


#############--------- Data manipulation ---------############

# correct stimulus for A225 and A050
sex_diff_raw$stimulus[which(sex_diff_raw$label == 'A225')]          <- c("position")
sex_diff_raw$stimulus[which(sex_diff_raw$label == 'A050' & 
                            sex_diff_raw$learning.task == 'motor')] <- c("position")

# remove all tasks with only one choice
sex_diff_raw$apparatus[which(sex_diff_raw$apparatus == '1-choice')] <- NA
sex_diff_raw <- sex_diff_raw[complete.cases(sex_diff_raw$apparatus),]

# separate data for males and females and merge together into one dataframe
# to changes data from rows to columns
female        <- subset(sex_diff_raw, sex_diff_raw$sex == "female")
male          <- subset(sex_diff_raw, sex_diff_raw$sex == "male")
sex_diff_proc <- cbind(female, male)
# remove duplicate columns
sex_diff_proc[,9]    <- NULL
for (i in 1:15){
  sex_diff_proc[,24] <- NULL
}

# change column headings
colnames(sex_diff_proc) <- c("study","apparatus","learning.task",
                             "task","stimulus","repeat","number.repeats",
                             "criterion","species","genus","family","order",
                             "age","data.source","N.female","mean.errors.female",
                             "sd.err.female","se.err.female","mean.ttc.female",
                             "sd.ttc.female","se.ttc.female","total.trials.female",
                             "N.total.female","N.male","mean.errors.male",
                             "sd.err.male","se.err.male","mean.ttc.male","sd.ttc.male",
                             "se.ttc.male","total.trials.male","N.total.male")
# check that removal was successful
head(sex_diff_proc)
str (sex_diff_proc)

##############################################################
#                     Descriptive stats                      #
##############################################################

# average sample size and standard deviation
mean(sex_diff_proc$N.total.female) 
mean(sex_diff_proc$N.total.male)   
sd(sex_diff_proc$N.total.female)   
sd(sex_diff_proc$N.total.male)     


##############################################################
#               effect size and its variance                 #
##############################################################

#############--------- TRIALS TO CRITERION ---------############

lnRRu1<-lnRRu(sex_diff_proc$mean.ttc.female, sex_diff_proc$mean.ttc.male,
              sex_diff_proc$sd.ttc.female, sex_diff_proc$sd.ttc.male,
              sex_diff_proc$N.female, sex_diff_proc$N.male)

# change column names and merge with raw data
colnames(lnRRu1) <- c("log.effect.size","log.effect.size.variance")
sex_diff_proc1   <- bind_cols(sex_diff_proc, lnRRu1)
# check if successful
head(sex_diff_proc1, 10)

#############--------- ERRORS TO CRITERION ---------############

lnRRu2<-lnRRu(sex_diff_proc$mean.errors.female, sex_diff_proc$mean.errors.male,
              sex_diff_proc$sd.err.female, sex_diff_proc$sd.err.male,
              sex_diff_proc$N.female, sex_diff_proc$N.male)

# change column names and merge with raw data
colnames(lnRRu2) <- c("log.effect.size.err","log.effect.size.variance.err")
sex_diff_proc2   <- bind_cols(sex_diff_proc, lnRRu2)
# check if successful
head(sex_diff_proc2, 10)

#############--------- PROPORTION OF INDIVIDUALS ---------############
#############---------       THAT LEARNED        ---------############

# calculate number of individuals who failed to learn
sex_diff_proc$fail_f <- sex_diff_proc$N.total.female - sex_diff_proc$N.female
sex_diff_proc$fail_m <- sex_diff_proc$N.total.male   - sex_diff_proc$N.male

or <- lnOR(sex_diff_proc$N.male, sex_diff_proc$fail_m, 
           sex_diff_proc$N.female, sex_diff_proc$fail_f)
# change column names and merge with raw data
colnames(or)   <- c("log.odds.ratio","log.odds.ratio.variance")
sex_diff_proc3 <- bind_cols(sex_diff_proc, or)
# check if successful
head(sex_diff_proc3, 10)


##############################################################
#             Analysis for TRIALS TO CRITERION               #
##############################################################

# remove study A226, did not give trials to criterion
sex_diff_proc1 <- sex_diff_proc1[-which(sex_diff_proc1$study == "A226"),]
# add a column observation to identify each effect size as a single measurement 
# nested in study
sex_diff_proc1[, "obs"] <- c(1:length(sex_diff_proc1$study))
# save raw data-table
write.csv(sex_diff_proc1, file="sex_diff_proc_ttc.csv")

# MLM with two random effects 
# to gain within and between study variance based on random effects 
nullmodel <- rma.mv(log.effect.size, 
                    log.effect.size.variance,
                    random = list(~ 1 | obs, ~ 1 | study),
                    method = "REML", data = sex_diff_proc1)
summary(nullmodel)
# eggers test
summary(lm(residuals.rma(nullmodel) ~ (1/sex_diff_proc1$log.effect.size.variance)))
# confidence interval
confint(lm(residuals.rma(nullmodel) ~ (1/sex_diff_proc1$log.effect.size.variance)))
# I2 and CIs
I2_new(nullmodel, sex_diff_proc1$log.effect.size.variance)


#############--------- moderators ---------############
#############------------ MLMR ------------############

### stimulus
sex_diff_multi_ttc_stimulus <- rma.mv(log.effect.size, 
                                      V = log.effect.size.variance,
                                      mods =  ~ stimulus - 1,
                                      random = list(~ 1 | obs, ~ 1 | study), 
                                      method = "REML", data = sex_diff_proc1)
summary(sex_diff_multi_ttc_stimulus)
# I2 and CIs
I2_new(sex_diff_multi_ttc_stimulus, sex_diff_proc1$log.effect.size.variance)


### learning task
sex_diff_multi_ttc_lt <- rma.mv(log.effect.size, 
                                V = log.effect.size.variance,
                                mods =  ~ learning.task - 1,
                                random = list(~ 1 | obs, ~ 1 | study), 
                                method = "REML", data = sex_diff_proc1)
summary(sex_diff_multi_ttc_lt)
# I2 and CIs
I2_new(sex_diff_multi_ttc_lt, sex_diff_proc1$log.effect.size.variance)


### species
sex_diff_multi_ttc_species <- rma.mv(log.effect.size, 
                                     V = log.effect.size.variance,
                                     mods =  ~ species - 1,
                                     random = list(~ 1 | obs, ~ 1 | study), 
                                     method = "REML", data = sex_diff_proc1)
summary(sex_diff_multi_ttc_species)
# I2 and CIs
I2_new(sex_diff_multi_ttc_species, sex_diff_proc1$log.effect.size.variance)


### genus
sex_diff_multi_ttc_genus <- rma.mv(log.effect.size, 
                                   V = log.effect.size.variance,
                                   mods =  ~ genus - 1,
                                   random = list(~ 1 | obs, ~ 1 | study), 
                                   method = "REML", data = sex_diff_proc1)
summary(sex_diff_multi_ttc_genus)
# I2 and CIs
I2_new(sex_diff_multi_ttc_genus, sex_diff_proc1$log.effect.size.variance)


### family
sex_diff_multi_ttc_family <- rma.mv(log.effect.size, 
                                    V = log.effect.size.variance,
                                    mods =  ~ family - 1,
                                    random = list(~ 1 | obs, ~ 1 | study), 
                                    method = "REML", data = sex_diff_proc1)
summary(sex_diff_multi_ttc_family)
### I2 and CIs
I2_new(sex_diff_multi_ttc_family, sex_diff_proc1$log.effect.size.variance)


##############################################################
#             Analysis for ERRORS TO CRITERION               #
##############################################################

# remove study A050, did not give errors to criterion
sex_diff_proc2 <- sex_diff_proc2[-which(sex_diff_proc2$study== "A050" ),]
# remove study with effect size NaN
sex_diff_proc2 <- sex_diff_proc2[-2,]
# add a column observation to identify each effect size as a single measurement nested in study
sex_diff_proc2[, "obs"] <- c(1:length(sex_diff_proc2$study))
# save raw data-table
write.csv(sex_diff_proc2, file="sex_diff_proc_errors.csv")

# MLM with two random effects 
# to gain within and between study variance based on random effects only
nullmodel2 <- rma.mv(log.effect.size.err, 
                     log.effect.size.variance.err,
                     random = list(~ 1 | obs, ~ 1 | study),
                     method = "REML", data = sex_diff_proc2)
summary(nullmodel2)
# eggers test
summary(lm(residuals.rma(nullmodel2) ~ (1/sex_diff_proc2$log.effect.size.variance)))
# confidence interval
confint(lm(residuals.rma(nullmodel2) ~ (1/sex_diff_proc2$log.effect.size.variance)))
# I2 and CIs
I2_new(nullmodel2, sex_diff_proc2$log.effect.size.variance.err)


#############--------- moderators ---------############
#############------------ MLMR ------------############

### stimulus
sex_diff_multi_err_stimulus <- rma.mv(log.effect.size.err, 
                                      V = log.effect.size.variance.err,
                                      mods =  ~ stimulus - 1,
                                      random = list(~ 1 | obs, ~ 1 | study), 
                                      method = "REML", data = sex_diff_proc2)
summary(sex_diff_multi_err_stimulus)
# I2 and CIs
I2_new(sex_diff_multi_err_stimulus, sex_diff_proc2$log.effect.size.variance.err)


### learning task
sex_diff_multi_err_lt <- rma.mv(log.effect.size.err, 
                                V = log.effect.size.variance.err,
                                mods =  ~ learning.task - 1,
                                random = list(~ 1 | obs, ~ 1 | study), 
                                method = "REML", data = sex_diff_proc2)
summary(sex_diff_multi_err_lt)
# I2 and CIs
I2_new(sex_diff_multi_err_lt, sex_diff_proc2$log.effect.size.variance.err)


### species
sex_diff_multi_err_species <- rma.mv(log.effect.size.err, 
                                     V = log.effect.size.variance.err,
                                     mods =  ~ species - 1,
                                     random = list(~ 1 | obs, ~ 1 | study), 
                                     method = "REML", data = sex_diff_proc2)
summary(sex_diff_multi_err_species)
# I2 and CIs
I2_new(sex_diff_multi_err_species, sex_diff_proc2$log.effect.size.variance.err)


### genus
sex_diff_multi_err_genus <- rma.mv(log.effect.size.err, 
                                   V = log.effect.size.variance.err,
                                   mods =  ~ genus - 1,
                                   random = list(~ 1 | obs, ~ 1 | study), 
                                   method = "REML", data = sex_diff_proc2)
summary(sex_diff_multi_err_genus)
# I2 and CIs
I2_new(sex_diff_multi_err_genus, sex_diff_proc2$log.effect.size.variance.err)


### family
sex_diff_multi_err_family <- rma.mv(log.effect.size.err, 
                                    V = log.effect.size.variance.err,
                                    mods =  ~ family - 1,
                                    random = list(~ 1 | obs, ~ 1 | study), 
                                    method = "REML", data = sex_diff_proc2)
summary(sex_diff_multi_err_family)
# I2 and CIs
I2_new(sex_diff_multi_err_family, sex_diff_proc2$log.effect.size.variance.err)


##############################################################
#             Analysis for PROPORTION LEARNED                #
##############################################################

# remove studies with 0 variance (to be able to calculate I2)
sex_diff_proc3[which(sex_diff_proc3$log.odds.ratio.variance == 0),]   <-NA
sex_diff_proc3[which(sex_diff_proc3$log.odds.ratio.variance == Inf),] <-NA
sex_diff_proc3 <- sex_diff_proc3[complete.cases(sex_diff_proc3$log.odds.ratio.variance),]
# add a column observation to identify each effect size as a single measurement nested in study
sex_diff_proc3[, "obs"] <- c(1:length(sex_diff_proc3$study))
# save raw data-table
write.csv(sex_diff_proc3, file="data/sex_diff_proc_prop.csv")

# MLM with two random effects 
nullmodel3 <- rma.mv(log.odds.ratio, 
                     log.odds.ratio.variance,
                     random = list(~ 1 | obs, ~ 1 | study),
                     method = "REML", data = sex_diff_proc3)
summary(nullmodel3)
# eggers test
summary(lm(residuals.rma(nullmodel3) ~ (1/sex_diff_proc3$log.odds.ratio.variance)))
# cofindence interval
confint(lm(residuals.rma(nullmodel3) ~ (1/sex_diff_proc3$log.odds.ratio.variance)))
# I2 and CIs
I2_new(nullmodel3, sex_diff_proc3$log.odds.ratio.variance)


#############--------- moderators ---------############
#############------------ MLMR ------------############

### stimulus
sex_diff_multi_prop_stimulus <- rma.mv(log.odds.ratio, 
                                       V = log.odds.ratio.variance,
                                       mods =  ~ stimulus - 1,
                                       random = list(~ 1 | obs, ~ 1 | study), 
                                       method = "REML", data = sex_diff_proc3)
summary(sex_diff_multi_prop_stimulus)
# I2 and CIs
I2_new(sex_diff_multi_prop_stimulus, sex_diff_proc3$log.odds.ratio.variance)


### learning task
sex_diff_multi_prop_lt <- rma.mv(log.odds.ratio, 
                                 V = log.odds.ratio.variance,
                                 mods =  ~ learning.task - 1,
                                 random = list(~ 1 | obs, ~ 1 | study), 
                                 method = "REML", data = sex_diff_proc3)
summary(sex_diff_multi_prop_lt)
# I2 and CIs
I2_new(sex_diff_multi_prop_lt, sex_diff_proc3$log.odds.ratio.variance)


### species
sex_diff_multi_prop_species <- rma.mv(log.odds.ratio, 
                                      V = log.odds.ratio.variance,
                                      mods =  ~ species - 1,
                                      random = list(~ 1 | obs, ~ 1 | study), 
                                      method = "REML", data = sex_diff_proc3)
summary(sex_diff_multi_prop_species)
# I2 and CIs
I2_new(sex_diff_multi_prop_species, sex_diff_proc3$log.odds.ratio.variance)


### genus
sex_diff_multi_prop_genus <- rma.mv(log.odds.ratio, 
                                    V = log.odds.ratio.variance,
                                    mods =  ~ genus - 1,
                                    random = list(~ 1 | obs, ~ 1 | study), 
                                    method = "REML", data = sex_diff_proc3)
summary(sex_diff_multi_prop_genus)
# I2 and CIs
I2_new(sex_diff_multi_prop_genus, sex_diff_proc3$log.odds.ratio.variance)


### family
sex_diff_multi_prop_family <- rma.mv(log.odds.ratio, 
                                     V = log.odds.ratio.variance,
                                     mods =  ~ family - 1,
                                     random = list(~ 1 | obs, ~ 1 | study), 
                                     method = "REML", data = sex_diff_proc3)
summary(sex_diff_multi_prop_family)
# I2 and CIs
I2_new(sex_diff_multi_prop_family, sex_diff_proc3$log.odds.ratio.variance)


##############################################################
#                       Funnel plots                         #
##############################################################

# save as tiff
tiff('funnelAll.tiff', width = 1300, height = 500, pointsize = 35, compression = 'none')
# three plots in a row
par(lwd = 2, mar = c(4,4,1,1), mfrow = c(1,3), oma = c(0,0,1,0))
# funnel plot based on MLM trials to criterion
funnel(nullmodel, xlim = c(-2, 2), xlab = '')
# plot heading
mtext("Trials to criterion", side = 3, line = 0.5)
# second plot
par(mar = c(4,2,1,3))
# funnel plot based on MLM errors to criterion
funnel(nullmodel2, xlim = c(-2, 2), ylab = '')
# plot heading
mtext("Errors to criterion", side = 3, line = 0.5)
# third plot
par(mar = c(4,0,1,5))
# funnel plot based on MLM proportion that learnt
funnel(nullmodel3, xlim = c(-4, 4), xlab = '', ylab = '')
# plot heading
mtext("Proportion that learnt", side = 3, line = 0.5)
# finish plotting
dev.off()


##############################################################
#                       Forest plots                         #
##############################################################

##########------- TRIALS TO CRITERION  -------##########

# summary effects and CIs
summary_effects           <- as.data.frame(cbind(sex_diff_multi_ttc_species$b, 
                                                 sex_diff_multi_ttc_species$ci.lb,
                                                 sex_diff_multi_ttc_species$ci.ub))
colnames(summary_effects) <- c("Response.ratio","ciL","ciU")
# order of plotting
summary_effects$num       <- c(length(unique(sex_diff_proc1$species)):1)
# y-axis labels alphabetically decreasing and increasing
names                     <- sort(as.character(unique(sex_diff_proc1$species)), decreasing = TRUE)
names2                    <- sort(as.character(unique(sex_diff_proc1$species)))

# number of effect sizes for each species
n <- c(1:length(unique(sex_diff_proc1$species)))
for(i in 1:length(unique(sex_diff_proc1$species))){
  print(names2[i])
  print(length(unique(sex_diff_proc1$study[sex_diff_proc1$species   == names2[i]])))
  n[i] <- length(unique(sex_diff_proc1$study[sex_diff_proc1$species == names2[i]]))
}
# number of studies done on each species
k <- c(1:length(unique(sex_diff_proc1$species)))
for(i in 1:length(unique(sex_diff_proc1$species))){
  print(names2[i])
  print(length(sex_diff_proc1$study[sex_diff_proc1$species   == names2[i]]))
  k[i] <- length(sex_diff_proc1$study[sex_diff_proc1$species == names2[i]])
}

# create string to plot next to forest plot
# including number of studies, number of effects for each species, 
# effect and confidence intervals
all   <- c(1:length(unique(sex_diff_proc1$species)))
one   <- c(1:length(unique(sex_diff_proc1$species)))
two   <- c(1:length(unique(sex_diff_proc1$species)))
three <- c(1:length(unique(sex_diff_proc1$species)))

for(i in 1:length(unique(sex_diff_proc1$species))){
  one[i]   <-paste(n[i], sprintf("%s],", k[i]), sep='[')
  two[i]   <-sprintf("%s [", round(sex_diff_multi_ttc_species$b[i], digits = 2))
  three[i] <-sprintf("%s]",paste(round(sex_diff_multi_ttc_species$ci.lb[i], digits=2), round(sex_diff_multi_ttc_species$ci.ub[i], digits=2), sep='-'))
  all[i]   <-paste(paste(one[i], two[i]), three[i], sep = "")
}

# save as tiff with transparent background
tiff('forest_plot_ttc_species.tiff', width = 1600, height = 1500, pointsize = 45,
     compression = 'none', bg = "transparent")

# set margins and increase line width
par(lwd = 2, mar = c(4,10,2,6), oma=c(0,0,0,0))
plot(summary_effects$num ~ summary_effects$Response.ratio, 
     type = "n", xlim =c(-0.75,0.75), xlab = "lnRR unbiased", las = 1, 
     yaxt = 'n', ylab = " ", ylim = c(0.5,9.5))
# add summary effects
points(summary_effects$num ~ summary_effects$Response.ratio, pch = 18, cex = 2)
# add vertical line to indicate 0
abline(v = 0)
# add confidence intervals
arrows(x0 = summary_effects$Response.ratio, x1 = summary_effects$ciU, 
       y0 = summary_effects$num, y1 = summary_effects$num, angle = 90, length = 0.15)
arrows(x0 = summary_effects$Response.ratio, x1 = summary_effects$ciL, 
       y0 = summary_effects$num, y1 = summary_effects$num, angle = 90, length = 0.15)
# add y-axis lables left side
axis(2, at = c(1:9), labels = names, las = 1, tick = FALSE, cex.axis = 0.75, font=3)
# right side y-axis heading
mtext("k[n], Effect [LCI to UCI]", cex = 0.5, at=c(1.17), font=2)
# add y-axis lables right side
axis(4, at = c(9:1), labels = all, las = 1, tick = FALSE, cex.axis = 0.5)
# add plot headings
mtext("Trials to criterion", side = 3, line = 0.5)
# finish ploting
dev.off()


##########------- ERRORS TO CRITERION  -------##########

# summary effects and CIs
summary_effects_err           <- as.data.frame(cbind(sex_diff_multi_err_species$b,
                                                     sex_diff_multi_err_species$ci.lb,
                                                     sex_diff_multi_err_species$ci.ub))
colnames(summary_effects_err) <- c("Response.ratio","ciL","ciU")
# order of plotting
summary_effects_err$num       <- c(length(unique(sex_diff_proc2$species)):1)
# y-axis labels alphabetically decreasing and increasing
names_err                     <- sort(as.character(unique(sex_diff_proc2$species)), decreasing = TRUE)
names2_err                    <- sort(as.character(unique(sex_diff_proc2$species)))

# number of effect sizes for each species
n_err <- c(1:length(unique(sex_diff_proc2$species)))
for(i in 1:length(unique(sex_diff_proc2$species))){
  print(names2_err[i])
  print(length(unique(sex_diff_proc2$study[sex_diff_proc2$species       == names2_err[i]])))
  n_err[i] <- length(unique(sex_diff_proc2$study[sex_diff_proc2$species == names2_err[i]]))
}
# number of studies done on each species
k_err <- c(1:length(unique(sex_diff_proc2$species)))
for(i in 1:length(unique(sex_diff_proc2$species))){
  print(names2_err[i])
  print(length(sex_diff_proc2$study[sex_diff_proc2$species       == names2_err[i]]))
  k_err[i] <- length(sex_diff_proc2$study[sex_diff_proc2$species == names2_err[i]])
}

# create string to plot next to forest plot
# including number of studies, number of effects for each species, 
# effect and confidence intervals
all_err   <- c(1:length(unique(sex_diff_proc2$species)))
one_err   <- c(1:length(unique(sex_diff_proc2$species)))
two_err   <- c(1:length(unique(sex_diff_proc2$species)))
three_err <- c(1:length(unique(sex_diff_proc2$species)))

for(i in 1:length(unique(sex_diff_proc2$species))){
  one_err[i]   <- paste(n_err[i], sprintf("%s],", k_err[i]), sep='[')
  two_err[i]   <- sprintf("%s [", round(sex_diff_multi_err_species$b[i], digits = 2))
  three_err[i] <- sprintf("%s]",paste(round(sex_diff_multi_err_species$ci.lb[i], digits=2), round(sex_diff_multi_err_species$ci.ub[i], digits=2), sep='-'))
  all_err[i]   <- paste(paste(one_err[i], two_err[i]), three_err[i], sep = "")
}

# save as tiff with transparent background
tiff('forest_plot_err_species.tiff', width = 1600, height = 1500, pointsize = 45,
     compression = 'none', bg = "transparent")
# set margins and increase line width
par(lwd = 2, mar = c(4,10,2,6), oma=c(0,0,0,0))
plot(summary_effects_err$num ~ summary_effects_err$Response.ratio, 
     type = "n", xlim =c(-1.15,1.15), xlab = "lnRR unbiased", las = 1, 
     yaxt = 'n', ylab = " ", ylim = c(0.5,8.5))
# add summary effects
points(summary_effects_err$num ~ summary_effects_err$Response.ratio, pch = 18, cex = 2)
# add vertical line to indicate 0
abline(v = 0)
# add confidence intervals
arrows(x0 = summary_effects_err$Response.ratio, x1 = summary_effects_err$ciU, 
       y0 = summary_effects_err$num, y1 = summary_effects_err$num, angle = 90, length = 0.15)
arrows(x0 = summary_effects_err$Response.ratio, x1 = summary_effects_err$ciL, 
       y0 = summary_effects_err$num, y1 = summary_effects_err$num, angle = 90, length = 0.15)
# add y-axis lables left side
axis(2, at = c(1:8), labels = names_err, las = 1, tick = FALSE, cex.axis = 0.75, font=3)
# right side y-axis heading
mtext("k[n], Effect [LCI to UCI]", cex = 0.5, at=c(1.8), font=2)
# add y-axis lables right side
axis(4, at = c(8:1), labels = all_err, las = 1, tick = FALSE, cex.axis = 0.5)
# add plot headings
mtext("Errors to criterion", side = 3, line = 0.5)
# finish ploting
dev.off()


##########------- I2 plot -------##########

# read raw data
I2_plot  <- read.csv("plotI2.csv")
# check data
str(I2_plot)
head(I2_plot)

# subset table by measurement of learning
ttc_I2   <- subset(I2_plot, I2_plot$var == "ttc")
error_I2 <- subset(I2_plot, I2_plot$var == "err")
prop_I2  <- subset(I2_plot, I2_plot$var == "prop")

# model names
names_x  <- c("Intercept model","Stimulus","Learning task","Species","Genus","Family")

# specify position of horizontal lines
lineV    <- c(0.0,0.2,0.4,0.6,0.8,1.0)

# save as tiff 
tiff('forest_plot_I2_only.tiff', width = 1700, height = 1000, pointsize = 45, 
     compression = 'none')
# set margins and increase line width
par(mfrow = c(1,3), lwd = 3, oma = c(0,7,3,1))

## three plots next to each other with only a small space in between

# trials to criterion

# set margins
par(mar = c(4,0.5,1,0))
# empty plot in the right size
plot(ttc_I2$num ~ ttc_I2$I2, xlab = "", las = 1, yaxt = 'n', ylab = " ",  type = "n", 
     xlim = c(0,1), ylim = c(0.8,6))
# add vertical lines
for(i in lineV){ 
  abline(v = i, lty = 1, col = "grey", lwd = 1) 
}
# add grey area behind data representin the MLM
rect(-0.036, 0.07, 1.036, 1.5, density = NA, col = "#D3D3D3", border = NA)
# add I2
points(ttc_I2$num ~ ttc_I2$I2, pch = 19, cex = 1.5)
# add confidence intervals
arrows(x0 = ttc_I2$I2, x1 = ttc_I2$uCI, y0 = ttc_I2$num, y1 = ttc_I2$num, 
       angle = 90, length = 0.15)
arrows(x0 = ttc_I2$I2, x1 = ttc_I2$lCI, y0 = ttc_I2$num, y1 = ttc_I2$num, 
       angle = 90, length = 0.15)
# y-axis labels
axis(2, at = c(1:6), labels = names_x, las = 1, tick = FALSE)
# plot title
mtext("Trials to criterion", side=3, line=1.3, cex=0.8)

# errors to criterion

# set margins
par(mar = c(4,0.5,1,0))
# empty plot in the right size
plot(error_I2$num ~ error_I2$I2, xlab = " ", las = 1, yaxt = 'n', ylab = " ",  
     type = "n", xlim = c(0,1), ylim = c(0.8,6))
# add vertical lines
for(i in lineV){ 
  abline(v = i, lty = 1, col = "grey", lwd = 1) 
}
# add grey area behind data representin the MLM
rect(-0.036, 0.07, 1.036, 1.5, density = NA, col = "#D3D3D3", border = NA)
# add I2
points(error_I2$num ~ error_I2$I2, pch = 19, cex = 1.5)
# add confidence intervals
arrows(x0 = error_I2$I2, x1 = error_I2$uCI, y0 = error_I2$num, y1 = error_I2$num, 
       angle = 90, length = 0.15)
arrows(x0 = error_I2$I2, x1 = error_I2$lCI, y0 = error_I2$num, y1 = error_I2$num, 
       angle = 90, length = 0.15)
# plot title
mtext("Errors to criterion", side=3, line=1.3, cex=0.8)
# add centered x-axis title
mtext(expression('I' ^ 2), side = 1, line = 3, family = "serif")

# proportion of individuals that learnt

# set margins
par(mar=c(4,0.5,1,0))
# empty plot in the right size
plot(prop_I2$num ~ prop_I2$I2, xlab = " ", las = 1, yaxt = 'n', ylab = " ",  type = "n", 
     xlim = c(0,1), ylim = c(0.8,6))
# add vertical lines
for(i in lineV){ 
  abline(v = i, lty = 1, col = "grey", lwd = 1) 
}
# add grey area behind data representin the MLM
rect(-0.036, 0.07, 1.036, 1.5, density = NA, col = "#D3D3D3", border = NA)
# add I2
points(prop_I2$num ~ prop_I2$I2, pch = 19, cex = 1.5)
# confidence intervals
arrows(x0 = prop_I2$I2, x1 = prop_I2$uCI, y0 = prop_I2$num, y1 = prop_I2$num, 
       angle = 90, length = 0.15)
arrows(x0 = prop_I2$I2, x1 = prop_I2$lCI, y0 = prop_I2$num, y1 = prop_I2$num, 
       angle = 90, length = 0.15)
# plot title
mtext("Proportion of individuals", side = 3, line = 1.8, cex = 0.8)
mtext("that learnt", side = 3, line = 0.3, cex = 0.8)


# end plotting
dev.off()

