library(survival)
library(readr)
library(vioplot)
library(beanplot)

# Returns the last point at which an asymmptotic survival curve
# stabilizes after a change in the survival probability.
get_stable_point <- function(time, survival, delta=0.05) {
  size = length(time)
  point <- list(time=time[size], survival=survival[size])

  if (size < 2) {
    return(point)
  }

  stable_p <- point$survival
  for (i in (size - 1):1) {
    survival_p <- survival[i]
    if (is.null(survival_p)) next
    if ((survival_p - stable_p) > delta) {
      break
    }
    point <- list(time=time[i], survival=survival_p)
  }
  
  return(point)
}

SECONDS = 86400 * 7 # Each week
stabilization_delta = 0.01

base_path <- "./analysis/raw/"
projects <- c('CenterForOpenScience__osf.io', 'ccnmtl__mediathread', 'ccnmtl__nepi', 'ccnmtl__ssnm', 'ccnmtl__wardenclyffe', 'edx__course-discovery', 'edx__credentials', 'edx__ecommerce', 'edx__edx-analytics-dashboard', 'edx__edx-platform', 'mozilla-services__socorro', 'mozilla__bedrock', 'mozilla__kitsune', 'mozilla__zamboni')
#projects <- c('mozilla__zamboni', 'mozilla__kitsune', 'edx__course-discovery', 'edx__ecommerce', 'edx__edx-platform', 'CenterForOpenScience__osf.io')
#projects <- c('edx__edx-analytics-dashboard')
paths <- paste(base_path, projects, sep="")

filenames = dir(pattern="^rq3-survival.csv$", path = paths, full.names = TRUE, recursive = TRUE)
stable_points = data.frame(matrix(ncol = 2, nrow = 0))

# Colors
col = 1:4

for (i in 1:length(filenames)) {
  data <- read_csv(filenames[i])
  
  repo_name <- data$repo_name[1]
  project = sub("\\.", "_", sub("/", "__", repo_name))
  kmsurvival <- survfit(Surv(ceiling(data$epoch_interval/SECONDS), data$removed) ~ 1)
  survival_summary <- summary(kmsurvival, censored=TRUE)
  # print(repo_name)
  # print(survival_summary)
  # 
  # pdf(paste("survival_", project, ".pdf", sep=""), width=5, height=4)
  # # par needs to be here so the pdf gets the right margins
  # par(mar=c(4, 3.5, 2, 0))
  # plot(kmsurvival, col=col)
  # title(main=repo_name, ylab="Survival Function", xlab="Weeks", mgp=c(2.5, 0, 0), cex.lab=1, cex.main=1.5)
  # box(lwd=1.95)
  # legend("topright", inset=c(0.01, 0), legend=c("Estimate","Lower 95%","Upper 95%"), col=col, lty=1, bty="n", cex=0.8, y.intersp=1.2)
  # dev.off()

  stable_point <- get_stable_point(survival_summary$time, survival_summary$surv, stabilization_delta)
  stable_points = rbind(stable_points, stable_point)
}

#plot(1, type = "n", axes=FALSE, xlab="", ylab="")
#legend(x = "top", inset = 0, legend = c("Esimate", "Lower 95%", "Upper 95%"), col = col, lty = 1, bty = "n", cex = 1.4, pt.cex = 1.9)


print(summary(stable_points))
#
# pdf(paste("removal_stabilization.pdf", sep=""), width=4, height=3)
# par(mfrow = c(1, 2), mar=c(0, 2, 2, 0), mai=c(0.1, 0.5, 0.4, 0.1))
# boxplot(stable_points$time)
# print(boxplot.stats(stable_points$time))
# title("Weeks to stop\nremoving toggles", cex.main=0.8)
# boxplot(stable_points$survival * 100)
# print(boxplot.stats(stable_points$survival))
# title("% of surviving toggles", cex.main=0.8)
# # vioplot(stable_points$time, stable_points$survival, col=rgb(0.1,0.4,0.7), names=c("Weeks", "% of Toggles"))
# # title("Removal stabilization", cex.main=1)
# dev.off()


pdf("removal_stabilization.pdf",  width=2, height=3)
par(mgp=c(2.3, 1, 0), mai=c(0, 0.8, 0, 0))
beanplot(stable_points$time, col=list(c("#880f14")),
         log = 'auto', horizontal = FALSE, ylab = "Weeks to stop removing toggles")

dev.off()

pdf("survival_stabilization.pdf",  width=2, height=3)
par(mgp=c(2.3, 1, 0), mai=c(0, 0.8, 0, 0))
beanplot(stable_points$survival * 100, col=list(c("#E0E0E0")),
         log = 'auto', horizontal = FALSE, cut = 0, ylab = "% of surviving toggles")

dev.off()

pdf(paste("removal_stabilization.pdf", sep=""), width=4, height=3)
par(mfrow = c(1, 2), mai=c(0, 0.5, 0, 0.1))
beanplot(stable_points$time, col=list(c("#880f14")),
         log = 'auto', horizontal = FALSE)
beanplot(stable_points$survival * 100, col=list(c("#E0E0E0")),
         log = 'auto', horizontal = FALSE, cut = 0)

dev.off()
