const proxyquire = require('proxyquire');

const diffSnapshots = (snapshots, gitLogs) => {
  return new Promise((resolve, reject) => {
    snapshots = [].concat(snapshots);

    if (typeof gitLogs !== 'string') {
      // defaults to a single git log execution
      // that resembles the last change from
      // the same commit adding the toggle for
      // the first time
      gitLogs = gitLogs ? gitLogs : snapshots.map((s) => ({
        commit: s.commit.commit,
        hunkRange: '@@ -0,0 +1,2 @@',
      }));
    }

    const getStdout = (gitLog) => [].concat(gitLog).map((log) => `
        >>>>> COMMIT HASH <<<<
        diff --git a/javascript.js b/javascript.js
        --- /dev/null
        +++ b/javascript.js
        >>>>> HUNK RANGE <<<<
        +if (isEnabled('toggle-A') || somethingElse()) {
        +  // bar
      `.replace(/\n\s+/g, '\n').substring(1) // cleanup
      .replace('>>>>> COMMIT HASH <<<<', log.commit)
      .replace('>>>>> HUNK RANGE <<<<', log.hunkRange)
    ).join('\n');

    const TogglesDiff = proxyquire('../lib/toggles-diff', {
      child_process: {
        exec: (command, options, callback) => {
          if (typeof gitLogs !== 'string') {
            const gitLog = gitLogs.shift();
            stdout = getStdout(gitLog);
          } else {
            stdout = gitLogs;
          }
          callback(null, stdout, null);
        }
      },
    });
    const togglesDiff = new TogglesDiff();
    togglesDiff.once('error', reject);
    togglesDiff.once('data', (diff) => resolve(diff));
    snapshots.forEach((snapshot) => togglesDiff.write(snapshot));
    togglesDiff.end();
  });
};

describe('TogglesDiff - simple cases', () => {
  it('traces when a toggle component was added', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }, { filepath: 'api/bar.js' }],
      },
      toggles: [{
        id: 'toggle-id',
        hash: 'toggle-hash-1',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 1, column: 0 },
        end: { line: 1, column: 10 },
      }],
    };

    const { Declaration: diff } = await diffSnapshots(snapshot_t1);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-id');
    expect(diff).to.deep.equal({
      'toggle-id': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        }
      ]
    });
  });

  it('traces when a toggle component was modified', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [{
        id: 'toggle-A',
        common_id: 'same-toggle',
        hash: 'toggle-hash-1',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 1, column: 0 },
        end: { line: 1, column: 10 },
      }],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [{
        id: 'toggle-B',
        common_id: 'same-toggle',
        hash: 'toggle-hash-2',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 1, column: 0 },
        end: { line: 1, column: 10 },
      }],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
    ];

    const { Declaration: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-A');
    expect(diff).to.deep.equal({
      'toggle-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'MODIFIED',
        }
      ]
    });
  });

  it('traces when a toggle component was deleted', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }, { filepath: 'bar.js' }],
      },
      toggles: [
        {
          id: 'toggle-in-foo',
          hash: 'toggle-hash-1',
          type: 'Declaration',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'toggle-in-bar',
          hash: 'toggle-hash-2',
          type: 'Declaration',
          file: 'bar.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        }
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [],
    }

    const { Declaration: diff } = await diffSnapshots([snapshot_t1, snapshot_t2]);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-in-foo', 'toggle-in-bar');
    expect(diff).to.deep.equal({
      'toggle-in-foo': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'DELETED',
        }
      ],

      'toggle-in-bar': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[1],
          operation: 'ADDED',
        }
      ]
    });
  });

  it('traces when a toggle component has not changed', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        }
      ],
    });
  });

  it('traces when a toggle is modified and previous within range of the diff hunk', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [{
        id: 'toggle-A',
        common_id: 'same-toggle',
        hash: 'toggle-hash-1',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 2, column: 0 },
        end: { line: 2, column: 10 },
      }],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [{
        id: 'toggle-B',
        common_id: 'same-toggle',
        hash: 'toggle-hash-2',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 3, column: 0 },
        end: { line: 3, column: 10 },
      }],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        // toggle-A is in the second line of the to-file hunk id
        hunkRange: '@@ -0,0 +1,2 @@',
      },

      // snapshot_t2
      [
        {
          commit: 'commit-hash-t2',
          hunkRange: '@@ -1,2 +2,2 @@',
        },
        {
          commit: 'commit-hash-t1',
          hunkRange: '@@ -0,0 +1,2 @@',
        },
      ]
    ];

    const { Declaration: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-A');
    expect(diff).to.deep.equal({
      'toggle-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'MODIFIED',
        }
      ]
    });
  });
});

describe('TogglesDiff - when common to a group of toggles in the same file', () => {
  it('traces the addition of a toggle', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t2',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        }
      ],

      'router-B': [
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'ADDED',
        },
      ],
    });
  });

  it('traces the modification of a toggle', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'router-C',
          common_id: 'same-router',
          hash: 'toggle-hash-3',
          type: 'Router',
          file: 'foo.js',
          start: { line: 5, column: 5 },
          end: { line: 5, column: 15 },
        },
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
      ],

      // router-C is router-B
      'router-B': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[1],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[1],
          operation: 'MODIFIED',
        },
      ],
    });
  });

  it('traces the addition and modification of toggles', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'router-C',
          common_id: 'same-router',
          hash: 'toggle-hash-3',
          type: 'Router',
          file: 'foo.js',
          start: { line: 5, column: 5 },
          end: { line: 5, column: 15 },
        },
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t2',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      // router-C is router-A
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[1],
          operation: 'MODIFIED',
        },
      ],

      'router-B': [
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'ADDED',
        },
      ],
    });
  });

  it('traces the non-explicit modification of a toggle', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 2, column: 0 },
          end: { line: 2, column: 10 },
        },
        {
          id: 'router-C',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 6, column: 5 },
          end: { line: 6, column: 15 },
        },
      ],
    }

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t2',
        hunkRange: '@@ -0,0 +2,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      // router-C is router-A
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[1],
          operation: 'CONTEXT_CHANGED',
        },
      ],

      'router-B': [
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'ADDED',
        },
      ],
    });
  });

  it('traces the deletion of a toggle', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 5, column: 0 },
          end: { line: 5, column: 10 },
        }
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-C', // location in file has changed
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +5,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +5,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'DELETED',
        }
      ],

      // router-C is router-B
      'router-B': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[1],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'CONTEXT_CHANGED',
        }
      ]
    });
  });

  it('traces the deletion of a toggle after being modified', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 1, column: 0 },
          end: { line: 1, column: 10 },
        },
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-C',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const snapshot_t3 = {
      commit: {
        commit: 'commit-hash-t3',
        authorTs: 6758500,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [],
    };

    const gitLogs = [
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots(
      [snapshot_t1, snapshot_t2, snapshot_t3],
      gitLogs
    );
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'MODIFIED',
        },
        {
          commit: snapshot_t3.commit,
          // TODO: referencing the last version would be more useful
          toggle: snapshot_t1.toggles[0],
          operation: 'DELETED',
        }
      ],
    });
  });

  it('traces multiple modifications of a toggle', async () => {
    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        }
      ],
    };

    const snapshot_t3 = {
      commit: {
        commit: 'commit-hash-t3',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-2',
          type: 'Router',
          file: 'foo.js',
          start: { line: 20, column: 0 },
          end: { line: 20, column: 10 },
        },
      ],
    };

    const snapshot_t4 = {
      commit: {
        commit: 'commit-hash-t4',
        authorTs: 6758500,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-C',
          common_id: 'same-router',
          hash: 'toggle-hash-3',
          type: 'Router',
          file: 'foo.js',
          start: { line: 20, column: 0 },
          end: { line: 20, column: 10 },
        },
      ],
    };

    const gitLogs = Array(3).fill([
      {
        commit: 'commit-hash-t2',
        hunkRange: '@@ -2,1 +4,2 @@',
      },
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
    ]);

    const { Router: diff } = await diffSnapshots(
      [snapshot_t2, snapshot_t3, snapshot_t4],
      gitLogs
    );
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t3.commit,
          toggle: snapshot_t3.toggles[0],
          operation: 'MODIFIED',
        },
        {
          commit: snapshot_t4.commit,
          toggle: snapshot_t4.toggles[0],
          operation: 'MODIFIED',
        },
      ],
    });
  });
});

describe('TogglesDiff - when common to a group of toggles in different files', () => {
  it('traces the addition of toggles', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [
        {
          id: 'router-A',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'foo.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'commit-hash-t2',
        authorTs: 6758499,
        files: [{ filepath: 'bar.js' }],
      },
      toggles: [
        {
          id: 'router-B',
          common_id: 'same-router',
          hash: 'toggle-hash-1',
          type: 'Router',
          file: 'bar.js',
          start: { line: 4, column: 0 },
          end: { line: 4, column: 10 },
        },
      ],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'commit-hash-t1',
        hunkRange: '@@ -0,0 +4,2 @@',
      },

      // snapshot_t2
      {
        commit: 'commit-hash-t2',
        hunkRange: '@@ -0,0 +4,2 @@',
      },
    ];

    const { Router: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('router-A', 'router-B');
    expect(diff).to.deep.equal({
      'router-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        }
      ],

      'router-B': [
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'ADDED',
        },
      ],
    });
  });
});

describe('TogglesDiff - the unexpected', () => {
  it('traces an addition even if the git log shows commit without diff patches', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'commit-hash-t1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }, { filepath: 'api/bar.js' }],
      },
      toggles: [{
        id: 'toggle-id',
        hash: 'toggle-hash-1',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 1, column: 0 },
        end: { line: 1, column: 10 },
      }],
    };

    const gitLog = [
      'commit-hash-t1',
      'diff --git a/javascript.js b/javascript.js',
      '--- /dev/null',
      '+++ b/javascript.js',
      '@@ -0,0 +4,2 @@',
      '+if (isEnabled(\'toggle-A\') || somethingElse()) {',
      '+  // bar',
      '\nsome-weird-commit-hash\n',
    ].join('\n');

    const { Declaration: diff } = await diffSnapshots(snapshot_t1, gitLog);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-id');
    expect(diff).to.deep.equal({
      'toggle-id': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        }
      ]
    });
  });

  it('traces the same toggle even if coming from merges', async () => {
    const snapshot_t1 = {
      commit: {
        commit: 'merge-commit-hash-1',
        authorTs: 123456,
        files: [{ filepath: 'foo.js' }],
        _merged: ['inner-commit'],
      },
      toggles: [{
        id: 'toggle-A',
        common_id: 'same-toggle',
        hash: 'toggle-hash-1',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 20, column: 0 },
        end: { line: 20, column: 10 },
      }],
    };

    const snapshot_t2 = {
      commit: {
        commit: 'merge-commit-hash-2',
        authorTs: 6758499,
        files: [{ filepath: 'foo.js' }],
      },
      toggles: [{
        id: 'toggle-B',
        common_id: 'same-toggle',
        hash: 'toggle-hash-2',
        type: 'Declaration',
        file: 'foo.js',
        start: { line: 50, column: 0 },
        end: { line: 50, column: 10 },
      }],
    };

    const gitLogs = [
      // snapshot_t1
      {
        commit: 'inner-commit',
        hunkRange: '@@ -0,0 +1,2 @@',
      },

      // augment call
      {
        commit: 'inner-commit',
        hunkRange: '@@ -0,0 +1,2 @@',
      },

      // snapshot_t2
      {
        commit: 'inner-commit',
        hunkRange: '@@ -0,0 +1,2 @@',
      },
    ];

    const { Declaration: diff } = await diffSnapshots([snapshot_t1, snapshot_t2], gitLogs);
    expect(diff).to.exist;
    expect(diff).to.have.all.keys('toggle-A');
    expect(diff).to.deep.equal({
      'toggle-A': [
        {
          commit: snapshot_t1.commit,
          toggle: snapshot_t1.toggles[0],
          operation: 'ADDED',
        },
        {
          commit: snapshot_t2.commit,
          toggle: snapshot_t2.toggles[0],
          operation: 'MODIFIED',
        }
      ]
    });
  });
});
