describe('CommitParser', () => {
  it('parses the files as if a project', (done) => {
    const CommitParser = require('proxyquire')('../lib/commit-parser', {
      'child_process': {
        exec: (command, options, callback) => callback(),
      }
    });

    const toggles = ['toggles!'];
    const commit = {
      files: [
        {
          ref: 'foo',
          action: 'A',
          filepath: 'foo.js',
        },
        {
          ref: 'bar',
          action: 'A',
          filepath: 'api/bar.js',
        }
      ],
    };

    projectParser = sinon.stub().resolves(toggles)
    const parser = new CommitParser({ projectParser });

    parser.on('data', (data) => {
      expect(data).to.deep.equal({
        commit,
        toggles,
      });

      done();
    });

    parser.once('error', done);

    parser.write(commit);
  });

  it('augments the merge commit with hashes of the merged commits', (done) => {
    const CommitParser = require('proxyquire')('../lib/commit-parser', {
      'child_process': {
        exec: (command, options, callback) => {
          if (command.indexOf('git log') > -1) {
            return callback(null, [
              'merge-hash',
              'commit-1',
              'commit-2',
            ].join('\n'));
          }

          callback()
        },
      }
    });

    const toggles = ['toggles!'];
    const commit = {
      commit: 'merge-hash',
      parents: 2,
      files: [
        {
          ref: 'foo',
          action: 'A',
          filepath: 'foo.js',
        },
        {
          ref: 'bar',
          action: 'A',
          filepath: 'api/bar.js',
        }
      ],
    };

    projectParser = sinon.stub().resolves(toggles)
    const parser = new CommitParser({ projectParser });

    parser.on('data', (data) => {
      expect(data).to.deep.equal({
        commit: Object.assign({
          _merged: [
            'commit-1',
            'commit-2',
          ],
        }, commit),
        toggles,
      });

      done();
    });

    parser.once('error', done);

    parser.write(commit);
  });

  it('does not emit a file if there was an error parsing it', (done) => {
    const CommitParser = require('proxyquire')('../lib/commit-parser', {
      'child_process': {
        exec: (command, options, callback) => callback(),
      }
    });

    const commit = {
      files: [
        {
          ref: 'foo',
          action: 'A',
          filepath: 'foo.js',
        },
        {
          ref: 'bar',
          action: 'A',
          filepath: 'api/bar.js',
        }
      ],
    };

    const stub = sinon.stub().resolves([
      'a-toggle-in-foo.js',
      {
        "__error__": {
          msg: 'Cannot parse the file',
          filepath: 'api/bar.js',
        }
      }
    ]);

    const parser = new CommitParser({ projectParser: stub });

    let out;
    parser.on('data', (data) => {
      out = data;
    });
    parser.once('error', done);
    parser.once('end', () => {
      expect(out).to.deep.equal({
        commit: {
          files: [
            {
              ref: 'foo',
              action: 'A',
              filepath: 'foo.js',
            },
          ],
        },
        toggles: ['a-toggle-in-foo.js'],
      });

      done();
    });

    parser.end(commit);
  });
});
