const { Transform } = require('stream');
const { exec } = require('child_process');
const debug = require('debug')('toggles-diff');
const debugType = {
  Declaration: require('debug')('toggles-diff:declaration'),
  Point: require('debug')('toggles-diff:point'),
  Router: require('debug')('toggles-diff:router'),
};

const HUNK_START_LINE_REGEXP = /^@@ -[0-9]+,[0-9]+ \+([0-9]+),([0-9]+) @@$/m;

class TogglesDiff extends Transform {
  constructor(options = {}) {
    options.highWaterMark = 1;
    super(Object.assign(options, { objectMode: true }));

    this.diff = {
      'Declaration': {},
      'Router': {},
      'Point': {},
    };

    this.cwd = options.cwd || '';
    this.breakOnFirst = options.breakOnFirst;
  }

  _transform(snapshot, encoding, callback) {
    debug('%s (%s toggles in %s files)', snapshot.commit.commit, snapshot.toggles.length, snapshot.commit.files.length);

    this.pairToggles(snapshot)
      .then((pairedToggles) => {
        pairedToggles.forEach((pair) => {
          const { previous, current } = pair;

          if (!previous && current) {
            this.diff[current.type][current.id] = [{
              commit: snapshot.commit,
              toggle: current,
              operation: 'ADDED',
            }];
            debugType[current.type]('%s (%s) ADDED', current.hash, current.id);
          } else if (previous && !current) {
            this.diff[previous.type][previous.id].push({
              commit: snapshot.commit,
              toggle: previous,
              operation: 'DELETED',
            });
            debugType[previous.type]('%s (%s) DELETED', previous.hash, previous.id);
          } else if (previous.hash !== current.hash) {
            this.diff[previous.type][previous.original_id].push({
              commit: snapshot.commit,
              toggle: current,
              operation: 'MODIFIED',
            });
            debugType[current.type]('%s (%s) - %s (%s) MODIFIED', previous.hash, previous.id, current.hash, current.id);
          } else if (current.id !== previous.id) {
            // toggle was not explicitly changed, but it could've suffered from collateral
            // changes from other changes in the file (e.g. location of toggle in file is
            // now different).
            this.diff[previous.type][previous.original_id].push({
              commit: snapshot.commit,
              toggle: current,
              operation: 'CONTEXT_CHANGED',
            });
            debugType[current.type]('%s (%s) CONTEXT_CHANGED', current.hash, current.id);
          } else {
            // ids are sensitive to any change (i.e. explicit/implicit) so no replacement
            // is needed. The toggle component did not changed at all
            debugType[current.type]('%s (%s) --NOCHANGE--', current.hash, current.id);
          }
        });

        if (this.breakOnFirst === true && this.hasToggles()) {
          this.push(this.diff);
          return process.nextTick(() => {
            this.destroy();
          });
        }

        callback();
      })
      .catch(callback);
  }

  _flush(callback) {
    callback(null, this.diff);
  }

  hasToggles() {
    return Object.keys(this.diff).some(type => Object.keys(this.diff[type]).length > 0)
  }

  // Returns the *first version* of all the living toggles in the system.
  // Toggles are unique per file.
  // The history structure mutates and some toggles become useless,
  // thus, need to compute this on every transform.
  get togglesInFiles() {
    return Object.values(this.diff)
      .reduce((togglesOperations, togglesInType) => {
        togglesOperations.push(...Object.values(togglesInType));
        return togglesOperations;
      }, [])
      .reduce((toggles, toggleOperations) => {
        // Living toggles only
        const lastOperation = toggleOperations[toggleOperations.length - 1];
        const firstOperation = toggleOperations[0];
        if (lastOperation.operation !== 'DELETED') {
          toggles.push(firstOperation.toggle);
        }
        return toggles;
      }, [])
      .reduce((togglesByFile, toggle) => {
        // Use a Set to make sure toggles are not repeated
        togglesByFile[toggle.file] = togglesByFile[toggle.file] || new Set();
        togglesByFile[toggle.file].add(toggle);
        return togglesByFile;
      }, {});
  }

  async pairToggles(snapshot) {
    const previousByFile = snapshot.commit.files
      .map(file => this.togglesInFiles[file.filepath])
      .reduce((toggles, set = []) => { // init when no toggles exist yet
        toggles.push(...set);
        return toggles;
      }, [])
      .reduce((hash, toggle) => {
        hash[toggle.file] = hash[toggle.file] || [];
        hash[toggle.file].push(toggle);
        return hash;
      }, {});

    const currentsByFile = snapshot.toggles
      .reduce((hash, toggle) => {
        hash[toggle.file] = hash[toggle.file] || [];
        hash[toggle.file].push(toggle);
        return hash;
      }, {});

    // Find the previous match of every current toggle
    // IMPORTANT: All async operations must be sequenced.
    // git operations are executed against the same repository in the
    // filesystem without any concurrency control. Don't want to overlap.
    const pairs = [];
    for (const file of snapshot.commit.files) {
      const { filepath } = file;
      const currentToggles = currentsByFile[filepath] || [];
      const previousToggles = previousByFile[filepath] || [];

      for (const current of currentToggles) {
        const commonPreviousToggles = previousToggles.filter((previous) => {
          return (
            previous.common_id === current.common_id &&
            previous.file === current.file
          );
        });

        const previous = await this.findPreviousOfCurrent(
          current, commonPreviousToggles, snapshot.commit.commit
        );
        if (previous) {
          pairs.push({
            previous,
            current,
          });

          // Don't leak the previous toggle
          previousToggles.some((toggle, index, array) => {
            const previousOriginalId = previous.original_id || previous.id;
            if (toggle.id === previousOriginalId) {
              array.splice(index, 1);
              return true;
            }
            return false;
          });
        } else {
          // Augment the toggle to trace it across merges
          if (snapshot.commit._merged) {
            Object.assign(current, await this.augmentToggle(current, snapshot));
          }

          pairs.push({
            current,
          });
        }
      }

      // Drain remaining previous toggles
      previousToggles.forEach((previous) => {
        pairs.push({
          previous,
        });
      });
    }

    return pairs;
  }

  /*
  * Finds the latest previous (t1) toggle of a current (t2) toggle.
  *
  * Procedure:
  * 1. Obtain the start line that originated the toggle at t2 via git log.
  * 2. The first previous (t0) toggle is the current toggle at t2 if the
  *    obtained start line is the same start line in the previous.
  * 3. Find the last previous of t0's history accumulated so far and
  *    reference it as t1's original toggle
  */
  async findPreviousOfCurrent(current, previousToggles, snapshotCommit) {
    if (this.breakOnFirst === true) return;

    const previousToggle = await this.blame(current, snapshotCommit, (gitLogCommitHash, start, end) => {
      const previous = previousToggles.find((toggle) => {
        const toggleCommit = this.diff[toggle.type][toggle.id][0].commit;
        const toggleHash = toggleCommit.commit;
        const mergedHashes = toggleCommit._merged || [];
        
        const samePrevious = (
          toggleHash === gitLogCommitHash &&
          toggle.start.line >= start &&
          toggle.start.line <= end
        );

        if (samePrevious === true) {
          return samePrevious;
        }

        // When a previous comes in a merge because of git log --first-parent and -m
        // we need to check for the merged hashes of its commit and the
        // augmented original area. 
        const sameFromPreviousMerge = (
          mergedHashes.indexOf(gitLogCommitHash) > -1 &&
          toggle._original.start >= start &&
          toggle._original.start <= end
        );

        return sameFromPreviousMerge;
      });

      if (previous) {
        // previous is the *first version* of the previous toggle, but
        // need the last version to correctly compare hashes further
        // down the process
        const operations = this.diff[previous.type][previous.id];
        const lastPrevious = operations[operations.length - 1].toggle;
        lastPrevious.original_id = previous.id;
        return lastPrevious;
      }
    });

    return previousToggle;
  }

  /*
  * Extracts the line where a toggle was introduced from a patch
  * string of `git-log -L`.
  */
  getPreviousLine(patch) {
    let line = {
      start: NaN,
      end: NaN,
    };

    const match = patch.match(HUNK_START_LINE_REGEXP);
    if (match) {
      line.start = parseInt(match[1], 10);
      line.end = line.start + parseInt(match[2], 10);
    }

    return line;
  }

  blame(current, snapshotCommit, callback) {
    return new Promise((resolve, reject) => {
      const { start, file } = current;
      const command = `git log --pretty=format:%H -L ${start.line},${start.line}:${file} ${snapshotCommit}`;
      exec(command, { cwd: this.cwd }, (error, stdout, stderr) => {
        if (error) {
          // Ignore shorter file errors
          if (/fatal: file .+ has only [0-9]+ lines/.test(error.message)) {
            return resolve();
          }

          debug('Upcoming error while processing %s', current.id);
          return reject(error);
        }

        if (stderr) {
          debug('Upcoming git-log error while processing %s', current.id);
          console.error(stderr); // re-stderr
          return resolve();
        }

        const patchedCommits = stdout.split(/^\n/m);
        let i = -1;
        let len = patchedCommits.length - 1;
        // TODO: break early and avoid unnecessary cycles
        while (i < len) {
          i++;
          const patchedCommit = patchedCommits[i];
          const commitPatchLines = patchedCommit.split('\n');

          if (commitPatchLines.length < 3) {
            /*
            * You have find something that looks like a git-log bug!
            * Check this out: git log --pretty=format:%H -L 41,41:website/identifiers/clients/ezid.py 0d22d4d6591493ea9bc32d456d99836347600551
            * in https://github.com/CenterForOpenScience/osf.io
            */
            resolve();
            continue;
          }

          const gitLogCommitHash = commitPatchLines[0];

          // TODO: we're missing the case were the code between start & end lines
          // of a Point gets modified
          const { start, end } = this.getPreviousLine(patchedCommit);
          if (isNaN(start) || isNaN(end)) {
            reject(new Error(
              `Unknown patch format for ${command} -- ${current.id} at ${snapshotCommit}. ${patchedCommit}`
            ));
            continue;
          }

          debug('Blaming with %s', command);
          const result = callback(...[gitLogCommitHash, start, end, { i, patchedCommit }]);

          if (result) {
            resolve(result);
            break;
          }

          if (i === len) {
            resolve();
            break;
          }
        }
      });
    });
  }

  async augmentToggle(toggle, snapshot) {
    const newToggle = Object.assign({}, toggle);
    const blame = await this.blame(toggle, snapshot.commit.commit, (hash, start, end) => {
      return { hash, start, end };
    });

    if (blame) {
      if (blame instanceof Error) {
        throw blame;
      } else {
        newToggle._original = {
          commitHash: hash,
          start,
        };
      }
    }

    return newToggle;
  }
}

module.exports = TogglesDiff;
