const { Transform } = require('stream');
const { exec } = require('child_process');
const debug = require('debug')('commit-parser');

class GitCheckoutError extends Error {
  constructor(...params) {
    super(...params);

    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, GitCheckoutError);
    }

    this.name = 'GitCheckoutError';
  }
}

class CommitParser extends Transform {
  constructor(options = {}) {
    options.highWaterMark = 1;
    super(Object.assign(options, { objectMode: true }));

    this.projectParser = options.projectParser;
    if (!this.projectParser) throw new Error('A project parser is required');
    
    this.cwd = options.cwd || '';
    this.parser = options.parser;
    this.breakOnFirst = options.breakOnFirst;
  }

  /*
  * We do not care about the order messages are returned by this.parser while
  * the returned toggles stay within the same commit. Gadly, _transform() is
  * guaranteed to never be called in parallel, thus every commit is transformed
  * sequentially.
  */
  _transform(bareCommit, encoding, callback) {
    debug('Parsing %s', bareCommit.commit);

    this.augmentCommit(bareCommit, (error, commit) => {
      if (error) return callback(error);

      exec(`git reset HEAD . && git checkout -- . && git checkout ${commit.commit}`, { cwd: this.cwd }, (error, stdout, stderr) => {
        if (error) {
          if (error.message.indexOf('overritten')) {
            // Impossible to checkout.
            // There seems to be a bug in git when you cannot move to another place because `git checkout -- file`
            // will allways show there are pending modifications. Looks related to autoctrl and lineendings.
            // https://stackoverflow.com/a/2016426/638425
            error = new GitCheckoutError('Local changes will be overritten. Cannot checkout.');
          }
          return callback(error);
        }

        const filepaths = commit.files.map(({ filepath }) => filepath);
        const globPattern = filepaths.length === 1 ? filepaths[0] : `{${filepaths.join(',')}}`;
        this.projectParser(globPattern, { cwd: this.cwd, parser: this.parser, breakOnFirst: this.breakOnFirst, commit })
          .then((togglesSets) => {
            const toggles = togglesSets.filter((fileToggles) => {
              const error = fileToggles.__error__;
              if (error) {
                debug(`Parsing error: ${error.msg} (${error.filepath})`);
                const index = commit.files.findIndex(file => file.filepath === error.filepath);
                if (index > -1) {
                  commit.files.splice(index, 1);
                } else {
                  throw new Error('Cannot find a file in the commit');
                }
                return false;
              }

              return true;
            })

            callback(null, {
              commit,
              toggles,
            });
          })
          .catch(callback);
      });
    });
  }

  augmentCommit(bareCommit, callback) {
    const commit = Object.assign({}, bareCommit);
    if (!bareCommit.parents || bareCommit.parents == 1) {
      return callback(null, commit);
    }

    const hash = commit.commit;
    const maxBuffer = 10 * 1024 * 1024; // 10MB. Some projects have loooong merge histories
    exec(`git log --pretty=format:%H "${hash}^..${hash}" `, { cwd: this.cwd, maxBuffer }, (error, stdout, stderr) => {
      if (error) return callback(error);
      commit._merged = stdout.split('\n').slice(1); // First commit will be the merge itself
      callback(null, commit);
    });
  }
}

module.exports = CommitParser;
