/*
* Traverse all commits and diff toggle areas
* 1. For every commit c1 in git-log, older first
* 2. parseProject of files in c1
* 3. parseProject of files in c1.parent
* 4. Diff ToggleAreas between 2 & 3 and store changeset
*/

const { spawn, exec } = require('child_process');
const _path = require('path');
const pump = require('pump');
const { LogParser, CommitParser, TogglesDiff } = require('./lib');
const parseProject = require('./project-parser');
const debug = require('debug')('history');
const { Writable, Readable } = require('stream');

function parseProjectHistory(path, { cwd, parser, breakOnFirst }, callback) {
  debug('Parsing %s', path);
  const command = 'git';
  const args = [
    'log',
    '--date-order',
    '--raw',
    breakOnFirst === true ? '' : '--reverse', // break quickly
    '--no-abbrev',
    '--pretty=format:%H %at %ct %P',
    '--first-parent',
    '-m',
    '--no-renames',
    `${_path.join(cwd, path)}`
  ].filter(arg => !!arg);

  debug('%s %s', command, args.join(' '));
  const gitLog = spawn(command, args, { cwd });
  let history;

  let stdout = '';
  gitLog.stdout.on('data', (data) => {
    stdout += data.toString('utf-8');
  });

  gitLog.once('close', (code) => {
    if (code !== 0) {
      return callback(new Error(`Process exited with process ${code}`));
    }

    require('fs').writeFileSync('./git-log.out', stdout);

    const readable = new Readable()
    readable._read = () => {};
    readable.push(Buffer.from(stdout, 'utf-8'));
    readable.push(null);

    pump(
      readable,
      new LogParser(),
      new CommitParser({ projectParser: parseProject, cwd, parser, breakOnFirst }),
      new TogglesDiff({ cwd, breakOnFirst }),
      new Writable({
        objectMode: true,
        write(diff, encoding, callback) {
          history = diff;
          callback();
        },
      }),
      (error) => {
        if (error) return callback(error);
        callback(null, history);
      }
    );
  });

  gitLog.stderr.once('data', (data) => {
    process.stderr.write(data.toString());
  });
}

module.exports = (filepath, options) => {
  const { cwd } = options;
  return new Promise((resolve, reject) => {
    debug('Checking repository into master...');

    const command = [
      'branch=`git symbolic-ref refs/remotes/origin/HEAD | sed \'s@^refs/remotes/origin/@@\'` &&',
      'git reset HEAD . &&',
      'git checkout -- . &&',
      'git clean -f -d &&',
      'git checkout $branch',
    ].join(' ');

    exec(command, { cwd }, (error) => {
      if (error) return reject(error);
      parseProjectHistory(filepath, options, (err, toggles) => {
        if (err) return reject(err);
        resolve(toggles);
      });
    });
  });
};
