import numpy
import pytest


from numina.array.peaks import peakdet

from .._kernels import kernel_peak_function


def test_pycapsule():
    m = kernel_peak_function(20)
    # This seems to be the only way to check that something is a PyCapsule
    sm = type(m).__name__
    assert sm == 'PyCapsule'


@pytest.mark.parametrize("window", [3, 5, 7, 9])
def test_return_weights(benchmark, window):
    result = benchmark(peakdet.return_weights, window)

    assert result.shape == (3, window)


@pytest.mark.parametrize("window", [3, 5, 7, 9])
def test_generate_weights(benchmark, window):
    result = benchmark(peakdet.generate_weights, window)

    assert result.shape == (3, window)


@pytest.mark.parametrize("window", [3, 5, 7, 9])
def test_weights_equal(window):
    result1 = peakdet.generate_weights(window)
    result2 = peakdet.return_weights(window)
    assert numpy.allclose(result1, result2)


@pytest.fixture(scope='module')
def spectrum():
    data = [-2.08104797e+00, -5.80629604e+00, -4.94689993e+00, -2.31206252e+00, 6.48687947e+00, 2.94904991e-01,
            -1.03945204e+00, 6.38293132e+00, 9.74415575e+00, 9.70397372e+00, 6.22966244e+00, 8.03908447e+00,
            1.97606508e+00, -5.23152482e-01, 7.76846279e+00, -2.35127959e+00, 4.66855800e+00, 5.66663032e+00,
            7.57543334e+00, 8.01355777e-01, 4.63607849e+00, -4.55448819e-01, 6.94442534e+00, 1.85184520e+01,
            -1.39092335e+00, 7.57635434e+00, 9.51165372e+00, -1.55861741e+00, 8.56277799e+00, 1.36267645e+01,
            8.67644395e+00, -2.80050351e+00, 2.98801810e+00, 1.17648220e+00, -2.34784679e+00, 1.60354767e+00,
            6.49891281e+00, 3.51392281e+00, -1.50658593e+00, 1.04116619e+01, 4.99372996e+00, 5.47938310e+00,
            6.70509260e+00, 8.50938089e-01, 2.35020481e+00, 2.86004895e+00, 1.18354156e+01, 1.11009771e+01,
            -1.36889689e+00, 7.02077423e+00, 9.84550536e+00, 5.76702704e+00, 8.83654826e+00, -6.34703661e-01,
            3.62101407e+00, 2.00395870e+00, 1.04152120e+01, 8.28524635e-01, 9.09174368e+00, 1.07493255e+01,
            -7.91062017e-01, 8.43875042e+00, 6.39436922e+00, -5.28260275e+00, 4.97028355e+00, -6.54342734e-01,
            5.30122603e+00, 7.40879592e+00, -3.22643058e+00, 1.69076614e+01, 3.27942569e-01, 2.37227235e+00,
            -1.51976756e-01, 1.78537738e+00, 1.77846257e+00, 1.72211287e+00, -3.20274063e+00, 9.86600391e+00,
            4.12103521e+00, -6.12795093e+00, 4.09639494e+00, 2.21906660e+00, 5.48755213e+00, 1.05701765e+00,
            1.12899598e+01, 7.47781577e+00, -1.05044556e+00, -4.48071071e+00, 6.02272071e+00, 2.42328711e+00,
            8.84821453e+00, 1.13142723e+01, 2.52500759e+00, 5.63110338e+00, -4.00220635e+00, 1.16903907e+00,
            4.65563542e+00, -3.17895223e+00, 3.13947996e+00, 6.27857771e+00, -2.14389856e+00, 1.48693219e+00,
            6.72598450e+00, 4.79902602e+00, 2.23846533e+00, 1.25377348e+01, 3.20915847e+00, 3.07532830e+00,
            2.78905042e+00, 5.80141484e+00, 4.53094045e+00, -3.80690662e+00, 5.14782230e+00, 2.52995296e+00,
            2.98780964e-01, 3.28678755e+00, -1.48964007e+00, 4.47698898e+00, 3.45939903e+00, 6.93136804e+00,
            -1.72978524e+00, 5.90786469e+00, 5.42619041e+00, -2.51949213e+00, 1.26382639e+00, -2.38972853e+00,
            1.26178141e+01, 1.25352816e+01, 5.49638531e+00, 2.60718988e-01, -2.23898153e+00, 8.33552163e+00,
            1.63936552e+00, -4.14892886e+00, 1.47730401e+00, 1.26302224e+01, 6.04214773e+00, -3.29734441e+00,
            1.03456707e+01, -5.85548039e+00, 1.78447751e+00, 6.01533324e+00, 1.02506847e+01, 5.56831144e+00,
            9.44809588e+00, -6.48216100e-01, 9.83799180e+00, 6.52928943e+00, 6.31827875e+00, 4.55354172e-01,
            5.22329210e+00, -5.74150478e+00, 2.37269840e+00, -4.91857020e+00, 9.32946299e+00, 5.39749776e+00,
            1.14798266e+01, 1.05928801e+01, 7.38102830e+00, 1.29249985e+00, 3.89338854e+00, 1.35452815e+01,
            5.22709366e+00, 2.94109484e+00, 6.55245883e+00, 4.81584636e+00, 4.53468011e+00, 2.45806324e+00,
            9.97614550e-01, 1.90502807e+00, 8.96430902e+00, 3.88681999e+00, 1.35442979e+00, -5.96112260e+00,
            4.09759506e+00, 5.53931692e+00, 2.48365471e-01, 5.71932139e-01, 4.84864761e+00, -4.11719592e+00,
            5.40153453e+00, 5.70421517e+00, 1.14683003e+01, 8.93810436e+00, 1.27552493e+01, -1.50965164e+00,
            1.19981761e+00, 1.01056426e+01, 7.70060709e+00, 2.13040884e-01, 3.21297413e+00, 2.65616235e+00,
            8.92570660e+00, 5.66292156e+00, 6.79010158e+00, 7.82014242e+00, -2.09724455e+00, 3.44395718e+00,
            6.99877474e+00, 1.80400840e+00, 2.70977531e+00, 1.81691418e+00, -1.27765255e-01, 1.36742853e+00,
            2.22484072e+00, -3.07919217e+00, -7.51244950e-01, 5.11256663e+00, 5.49162568e+00, 6.72372628e+00,
            1.67795010e+00, 3.91602355e+00, 1.59115917e+01, 7.14464785e+00, 3.87171313e+00, 8.54097948e-01,
            9.84512498e+00, 5.41854516e+00, -2.07772729e+00, -2.87849298e-01, -1.78408655e+00, 1.20037909e+01,
            1.55296579e+00, 6.85300294e+00, 3.36221450e+00, 1.82597105e+00, 1.96882977e+00, 1.25568308e+01,
            1.43198148e+01, 2.31668004e+00, -1.28167898e+00, -1.16903069e+01, 4.47363864e+00, 1.42484413e+01,
            8.92168185e+00, 2.63480303e+00, -4.61857253e+00, -7.07287023e+00, -8.57518026e-01, 4.29831572e+00,
            -4.55114049e+00, 6.09082141e+00, 2.34678593e+00, 7.10283221e+00, 1.00441856e+00, 1.08789207e+00,
            6.96475035e+00, 2.67442399e+00, 1.20062626e+01, -1.34420282e+00, 7.66181738e+00, 6.15672940e+00,
            -6.35014394e+00, 6.27041791e+00, -2.40098100e+00, 2.48405458e+00, -4.72760068e+00, 9.93944896e-01,
            6.83402749e+00, 7.73608890e+00, 1.27999818e+01, 9.44373595e+00, 8.94217475e+00, 5.68647536e+00,
            3.76622946e+00, 3.15044069e+00, 1.71336343e+01, 9.96848231e+00, -2.88510796e+00, 5.89027902e+00,
            3.16143204e+00, -4.79779226e-01, 4.02659215e+00, 6.39477863e+00, 9.64921850e+00, -3.99852024e+00,
            9.00383136e-01, -2.08551354e+00, 5.80523078e+00, -2.46474620e-01, -4.20198563e-01, 4.83553812e+00,
            7.90108581e+00, -1.73817467e+00, 6.82986959e+00, 4.45413253e-02, -1.50342076e+00, 2.82067624e+00,
            4.04670459e+00, 3.33934736e-01, -3.61950378e+00, 5.96250542e-01, -9.91932496e-01, 7.14437444e-02,
            2.70597621e+00, -5.93951485e+00, 8.36064230e+00, -5.36144854e+00, 5.13248025e-01, 8.85520521e+00,
            2.67517220e+00, 1.25540491e+01, 1.57658561e+01, 2.51172251e+00, 5.09312173e+00, 2.57507512e+00,
            3.27635309e+00, -1.09094652e+00, 4.49980711e+00, 6.63488740e+00, 1.16822665e+01, 8.56292605e-01,
            3.98182413e+00, 3.88608610e+00, 1.65257376e+00, -6.69772466e-01, -2.51235810e+00, -2.28628740e+00,
            5.65228465e+00, 6.03490353e+00, 7.10765597e+00, 7.13783134e+00, -1.15282794e+00, 1.93979556e+00,
            4.85536520e+00, 1.17273717e+01, 8.92730414e+00, 6.82377524e+00, 4.99359610e+00, 1.56220729e+00,
            -1.79963267e+00, 3.98046109e+00, 5.58542681e-01, 1.54864978e+00, 2.86963929e+00, -2.08188837e+00,
            1.01693350e+01, -1.71996355e+00, -1.65029926e+00, 8.51103356e+00, 2.04850077e+01, 1.78329491e+00,
            1.36747949e+00, 6.20436401e+00, 7.47256726e+00, 8.36013153e+00, 8.69326420e+00, 1.77319608e-01,
            -3.63119656e-01, 5.78815887e+00, 2.77326561e+00, -1.59222167e+00, -3.65042013e+00, 1.07508803e+01,
            2.06122692e+00, 4.76152301e+00, 1.15262908e+01, 4.46552107e+00, -3.22075701e+00, 3.83552154e+00,
            6.36237688e-01, 1.03534586e+01, -1.93433880e+00, 1.11502110e+01, -1.36885121e+00, -6.16962146e+00,
            -1.27983727e+00, 2.09027153e+00, 5.59207391e+00, 1.08824549e+01, -2.01269235e+00, -3.74633011e+00,
            5.17474499e+00, -1.95544254e+00, 1.54315201e+01, 3.24766211e+00, 4.79780930e+00, 1.92944572e+00,
            -4.32312848e+00, -4.04393771e-01, 7.75887668e+00, 2.39767587e+00, -1.15096183e+00, 8.10971469e+00,
            1.87253844e+01, 7.52361411e+00, -3.27091735e+00, 7.09922927e+00, 6.65533823e-01, 2.41329851e-02,
            1.23394407e+01, 2.31006438e+00, 3.78704414e+00, 1.63656459e+01, -3.15243809e-01, 1.77046403e+00,
            3.08641730e+00, -3.62097921e+00, 1.14586560e+01, 5.16648434e+00, 8.42503219e+00, -1.29942164e+00,
            2.57878104e+00, 3.36818305e+00, -3.43689800e+00, 7.95592056e+00, 1.22866126e+01, 1.02936397e+00,
            9.87955320e+00, 5.91410531e+00, 5.47207153e+00, -9.01749785e-01, 3.13838267e-01, 3.44526263e+00,
            -9.63949828e-01, -1.45820249e+00, 6.95890317e-01, -4.83048574e+00, 4.88354889e+00, 7.41763269e+00,
            1.21330154e+01, 1.55928100e+01, 2.25981867e+00, -2.52443438e+00, 9.49063694e+00, -7.47075331e+00,
            9.46675918e+00, 8.58777859e+00, 7.92427135e+00, -1.32864260e+01, 6.65156931e+00, 1.07744178e+01,
            8.93410947e+00, 7.20447137e+00, 2.11607553e+00, 3.31485042e+00, 1.17652233e+00, 9.09661387e+00,
            5.07061534e+00, 1.37276652e+01, 8.62901128e+00, 9.23927112e-01, 6.35422069e+00, -2.34508297e+00,
            9.58836910e+00, 1.67691397e+00, 1.95741166e+00, 1.48345359e+00, 1.44507643e+01, 3.32641099e+00,
            1.69800151e-01, 2.67319266e+00, 7.38791890e-02, 7.84221475e-01, 3.83362424e+00, -2.75089494e+00,
            1.16902439e+01, 8.63710625e-01, 6.34220705e+00, 9.08983102e+00, 2.86328875e+00, -9.07094534e-01,
            9.29874013e+00, 3.02724099e+00, -1.11212299e+00, 1.65865032e+01, 1.00899888e+01, 4.63317427e+00,
            1.01681392e+01, -4.65105908e+00, -3.27985888e-01, 5.11336557e+00, 1.59912639e+01, 1.47678267e+02,
            7.49374763e+02, 2.69729136e+03, 6.56334355e+03, 1.10701165e+04, 1.24716843e+04, 9.56485814e+03,
            5.01373368e+03, 1.87715384e+03, 4.27780005e+02, 6.68845308e+01, 2.35509412e+01, 3.54206977e+00,
            1.58667331e+00, 5.96335808e+00, 2.39282801e+00, 2.59698653e+00, 1.79786491e+00, 1.32021973e+01,
            1.15882710e+01, 4.42080294e+00, 6.80627966e+00, 7.70679708e+00, 1.36527312e+01, -5.70716863e+00,
            4.68927349e+00, 3.65575351e+00, 6.28770272e-01, 6.10914358e+00, -3.92340956e-02, 4.87374089e+00,
            -4.25122542e+00, 5.73388142e+00, -2.09612791e+00, 9.47953148e+00, 6.95384220e+00, 8.25534391e+00,
            2.29652719e+00, 6.49109785e+00, -1.77396077e+00, 1.56748364e+00, -2.92663787e+00, 1.65423839e-01,
            1.05135975e+01, 2.22818499e+00, 7.80711603e+00, -2.04801454e+00, 3.97474318e+00, 9.82064428e-01,
            2.47457468e+00, 5.63026447e+00, 8.36434344e+00, -2.31064419e-01, -9.48855449e-01, -2.43926359e+00,
            1.34832441e+00, 7.24774756e+00, 6.72618580e+00, 7.72582464e+00, 3.08980472e+00, 1.26262168e+00,
            5.85192699e+00, 4.41339514e+00, 8.32363696e+00, 2.92662744e+00, -1.84846645e+00, 1.12430006e+01,
            -6.15179945e+00, 4.29401999e+00, 1.15957805e+01, 1.34005719e+00, 5.74521914e+00, 4.12228780e-01,
            -9.15760628e+00, 3.04191859e+00, 1.43908118e+00, 1.05297395e+01, 5.22992437e+00, 1.10648874e+01,
            4.16600033e+00, 3.19867549e+00, 7.23681616e-01, 2.50633012e+00, 1.37751964e+00, -2.64175350e+00,
            1.34179026e+01, 8.22734143e+00, -1.34508616e+00, 6.13618277e+00, 7.65509069e+00, -6.65516567e-01,
            1.35599818e+01, 4.00785203e+00, 3.60866457e+00, 4.01259336e+00, 2.90012210e+00, -2.65001344e+00,
            7.25903247e+00, 3.63173354e-01, 7.79938869e+00, 1.36444918e+00, 5.52017734e+00, 6.57068243e+00,
            2.81012650e-01, 5.84050407e+00, -6.45464146e+00, 8.49619111e+00, 8.39752651e+00, 2.41005335e+00,
            5.95012253e+00, 1.50824996e+00, -2.12303514e+00, -3.07506041e-01, 5.81485450e+00, 1.08607538e+01,
            4.07705213e+00, -3.25045695e+00, -7.21869607e+00, 4.76485241e+00, 4.23366142e+00, 4.74541204e+00,
            -6.80215095e+00, 1.29572526e+01, -5.49898480e+00, 1.63951763e+00, 3.37657929e+00, 4.09081182e+00,
            5.52784870e+00, -1.44355243e+00, 5.45958492e+00, 1.89697368e+00, -5.74158681e-01, 7.97199749e-01,
            1.03387336e+00, 1.50586951e+01, 3.23950561e+00, 1.07177346e+01, -3.08076320e+00, 7.23081875e+00,
            2.90995641e+00, -7.82405950e+00, 4.10791326e+00, 4.52336977e+00, -7.55037486e+00, 3.00014326e+00,
            5.12585227e+00, -1.45608736e+00, 1.11126849e+01, -4.62956303e+00, -7.98276074e-01, 6.55007446e+00,
            6.13855446e+00, 3.11686056e+00, 6.05006057e+00, -3.61282842e+00, 6.47071580e+00, 1.08373862e+01,
            4.29514688e+00, -3.38704293e+00, 1.69991738e+00, 9.76139011e+00, 1.70781898e+00, 6.14074907e-01,
            1.48036515e+00, 8.99501488e+00, -1.00482905e+00, -6.17430221e-01, 4.19701358e+00, 4.29642491e+00,
            -5.16380950e+00, 5.66934363e+00, -4.90537271e+00, 8.94692055e+00, 7.60960625e+00, 1.98802972e+00,
            6.29924563e+00, 1.34973318e+00, 7.21848339e+00, 5.42607266e+00, 2.51090123e+00, 4.89239801e+00,
            3.16724893e+00, 1.94853663e+00, -5.32551548e+00, -3.10225024e-01, -1.28402382e+00, 6.34672686e+00,
            -4.01877524e+00, 1.02953967e+01, 3.77654962e+00, 4.41293448e+00, -9.68224249e-01, -3.42863670e+00,
            9.37965179e+00, 3.68827179e+00, 5.89144343e+00, -5.30740994e-01, 1.30628750e+00, 5.69734494e+00,
            1.42673818e+01, -1.14968206e+00, 1.20125403e+01, -2.65585554e+00, 1.22045930e+01, -4.05453836e+00,
            1.98093742e-01, -1.80998959e+00, 1.78355250e+00, 3.21002134e+00, 1.00582865e+00, 5.88715116e+00,
            -1.82655357e+00, 1.45572913e+01, 1.73833446e+00, 1.00735796e+01, -2.72440324e+00, -3.44251299e+00,
            5.53260338e+00, 4.95007823e+00, 5.90285771e-01, 3.96038233e+00, 1.60437008e+00, 2.21441376e+00,
            6.30057913e+00, 7.39963152e+00, 8.16269155e+00, 1.22770912e+01, -1.26700163e+00, 5.54857417e+00,
            1.69381433e+01, -7.54427228e-01, -3.05197082e+00, 6.22921436e+00, -4.55449660e+00, 1.19285538e+01,
            2.22768417e+00, 2.75997960e+00, -1.88213218e+00, 5.55112275e+00, 2.34790012e+00, -3.28433266e-01,
            -3.11497696e+00, 5.16275846e+00, 1.76459851e+01, 6.69065488e+00, 4.13722770e+00, 7.13004435e+00,
            9.93242285e-01, -4.65209171e+00, 7.33601731e+00, 3.56879540e+00, 3.67677024e+00, 1.14621023e+01,
            -6.54428847e+00, -1.15723477e+01, 2.48962007e+00, 6.28055697e+00, 1.23901994e+01, 5.99341176e+00,
            6.47338982e+00, 5.91767383e+00, -9.48871282e-01, 3.66817657e+00, 8.91205550e-01, 3.01187955e+00,
            1.97115134e+00, 4.83005401e+00, 7.22955845e+00, 1.05292007e+01, 9.51800078e+00, 5.49824177e+00,
            1.06177125e+01, 1.00215205e+01, 1.54098903e+00, -2.44051685e-01, -9.96489768e-01, 2.95440171e+00,
            -2.38742476e+00, 1.81793010e+01, -2.79634699e+00, 6.05431949e+00, 5.86358738e+00, 9.16586783e+00,
            -5.14298477e+00, 4.42579328e+00, -8.20883791e+00, 2.57817629e+00, 5.08015111e+00, -2.45318475e-02,
            3.74649213e+00, 2.38917364e-01, 7.37247899e+00, -9.63664320e-02, 9.75012577e+00, -5.23513223e-01,
            2.34219154e+00, 7.08452695e+00, 5.51451269e-01, 3.49642636e+00, 4.25958200e+00, -4.33300449e+00,
            3.94025075e+00, 2.00101048e+00, 1.32345650e+01, 3.80495281e+00, 1.39039486e+01, 7.85153790e+00,
            3.19252226e+00, 3.95715648e+00, 4.73835196e+00, 7.96306874e-01, 5.08522721e+00, 5.18047442e+00,
            1.20604504e+00, 5.36217467e+00, 1.32619617e+01, 2.04628117e+00, 6.96513327e+00, 2.97725004e+00,
            5.17224502e+00, -8.56885588e+00, 5.74278124e+00, 1.02460530e+01, -1.11620957e+01, 6.49394766e+00,
            1.12680681e+01, 2.71820375e+00, 1.14542724e+01, 1.49975613e+00, -1.59611736e+00, -2.85596091e+00,
            -1.91183897e+00, 7.94031067e+00, 1.07136608e+01, -5.91023753e+00, 1.06026015e+01, 7.24472232e+00,
            -3.57590972e-01, 1.43114061e+00, -2.28066005e+00, 7.86513682e+00, -7.98140638e+00, -1.78013010e-01,
            -8.33522524e+00, 1.02811834e+01, 5.57907464e+00, -3.09846218e-01, -3.21811985e+00, 9.11939887e+00,
            4.98510338e+00, -7.52582247e-01, 1.15392527e+00, -3.50912883e+00, -6.98737263e+00, -2.93958545e+00,
            1.00730608e+01, 3.37193500e+00, 3.76560698e+00, -1.44199166e+00, -5.12144274e+00, -5.51845151e+00,
            1.64065284e+00, -3.92761670e-01, 2.65905107e+00, 7.34947565e+00, 8.10257893e+00, 9.18856352e-01,
            1.11841061e+01, 1.89539173e+00, 9.14383804e+00, 4.14050720e+00, 9.72760085e+00, 9.97202084e-01,
            -5.52896499e+00, 4.52869249e+00, 2.19163271e+00, 2.53639442e+00, 6.03152953e+00, 1.75829412e+00,
            2.23620276e+00, 4.49312907e+00, 9.88218114e+00, 9.63933301e+00, -3.62188518e-01, 6.43685586e+00,
            6.70312163e+00, 5.92784593e+00, 1.12960536e+01, -3.00505873e-01, 7.47159191e+00, 1.22598829e+01,
            -5.29728673e+00, 6.33257423e+00, 9.28434248e+00, 8.48226169e+00, 2.15766724e+00, 6.92430796e+00,
            -2.14010622e+00, 1.59312969e+00, 2.21523412e-01, 4.57640098e+00, 1.25491079e+00, 1.28940731e+00,
            6.39484295e+00, 1.34865331e+01, 9.98991326e+00, 1.90813710e+00, -4.61727318e+00, -4.50205367e-01,
            5.32318484e+00, 8.65353796e-01, -3.85990039e+00, 3.03398251e+00, 3.79396971e+00, -6.50657488e+00,
            3.28513163e+00, 7.22563473e+00, 4.19853139e+00, 8.23404168e+00, 1.64928175e+01, 1.61127394e+01,
            1.28645854e+02, 7.92066251e+02, 2.86999548e+03, 7.28537196e+03, 1.21376341e+04, 1.42317884e+04,
            1.14850802e+04, 6.50383760e+03, 2.32143124e+03, 5.60896487e+02, 1.08966377e+02, 7.84402870e+00,
            1.77970162e+00, 1.10635176e+00, 5.78875444e+00, 6.43154223e+00, -2.64163123e+00, 7.83529519e+00,
            6.63670601e+00, 1.26725187e+00, 1.65750431e+00, 2.30218338e+00, 2.88038920e+00, 9.96273532e+00,
            2.29138576e+00, 4.27695140e+00, 6.90503023e+00, 7.35579573e+00, 4.78486834e+00, 3.96788947e+00,
            -3.53465793e+00, 7.56544949e+00, 1.04354283e+01, 8.41418256e-01, -1.18938943e+00, 2.80896438e+00,
            1.62398090e+01, 2.01586204e-01, 4.99303327e+00, -1.98290862e+00, -1.00299719e+01, 1.51785993e+00,
            1.08553900e+00, -4.14839946e+00, 3.98925982e+00, 7.53723254e+00, 3.23587988e+00, 4.66961990e+00,
            -8.16988507e-01, 1.38497858e-01, -9.31068408e+00, 6.15481350e+00, 4.97000176e+00, 2.86086651e+00,
            7.99349298e-02, -3.30503292e+00, 8.01764075e+00, 5.30840236e-01, 4.65746924e+00, 6.40800974e+00,
            4.40604860e+00, -8.77101883e-01, -1.91627585e-01, 1.05554739e+01, 5.86918045e+00, 5.64205621e-01,
            6.91237258e+00, -2.24075205e+00, 1.01031741e+01, 8.40857645e+00, -6.21058256e+00, 1.69372468e+00,
            7.22612286e+00, 3.81985457e+00, 2.66635337e+00, 1.09585968e+01, -5.33846695e-01, 1.15425105e+01,
            -4.43838456e+00, 5.18808202e-01, 1.22326524e+00, -7.36476584e+00, 1.17801252e+01, 6.06577102e+00,
            4.61141209e+00, 3.30914129e+00, 7.83382196e+00, -2.65026746e+00, 8.90342216e+00, 2.88060118e+00,
            1.15507379e+00, 2.36621574e+00, 4.90725244e+00, -1.46885055e+01, -4.79407746e+00, 4.60663776e+00,
            -5.01507267e+00, 1.12543612e+01, -1.68545565e+00, 6.38343695e+00, 2.04615607e+01, 1.17200941e+01,
            1.28917229e+01, 5.75309149e+00, -3.12933090e+00, 1.87689815e+01, -4.70591949e+00, 3.75641971e+00,
            -1.06756836e+01, 1.98905926e+00, 2.72103925e+00, 9.22649129e+00, 6.95360847e+00, -1.90362123e+00,
            4.15150412e-02, 1.40765569e+00, 3.30089149e+00, 5.60048106e+00, 1.41612232e+00, 3.38523806e+00,
            -8.10899953e-01, 1.03245047e+01, 1.12049560e+01, -7.52148590e-01, -2.90735449e+00, -5.85387792e-01,
            2.84863281e+00, 9.80802072e+00, 1.33471351e+01, 1.66334881e+02, 8.63246468e+02, 3.23121852e+03,
            8.02073431e+03, 1.39652066e+04, 1.67416035e+04, 1.34372844e+04, 7.54228382e+03, 2.99364551e+03,
            7.37376379e+02, 1.40009845e+02, 3.02611138e+01, 8.73607545e+00, 1.30569612e+01, -2.59169993e-01,
            1.25636811e+01, 4.31520009e+00, 2.91192756e+00, 6.20729046e-01, 1.58425122e+01, 9.04482831e+00,
            7.38863960e-01, 1.37595319e+01, 2.51981556e+00, 2.98832308e+00, 3.36326496e+00, -5.64212453e+00,
            3.73833170e+00, 4.18712783e+00, 5.31945581e+00, 3.53771405e+00, 1.12949709e+01, 1.07655858e+01,
            -3.85707344e+00, 3.11169323e+00, 1.02534368e+01, 6.45289358e-01, -3.33017264e+00, 1.09593408e+01,
            -1.29026040e+00, 2.74072984e-01, 7.38297829e+00, -5.24956185e+00, 8.11739298e+00, 3.05887191e+00,
            1.24330348e+01, -1.05929086e+01, 7.38134448e+00, -6.82912221e+00, 1.67109074e+00, -2.86790230e+00,
            5.84127573e+00, -2.90424085e+00, -5.95759015e+00, 6.83750582e+00, 6.84390357e+00, 3.50625464e-01,
            1.50467849e+01, 8.70299660e+01, 4.82831360e+02, 2.03933567e+03, 5.68816295e+03, 1.09170324e+04,
            1.42176420e+04, 1.30542511e+04, 8.23367012e+03, 3.67129861e+03, 1.07591554e+03, 2.15464025e+02,
            3.68217819e+01, 1.43350735e+01, -4.11115993e+00, 3.04674648e+00, 3.58483330e+00, -1.17492711e+00,
            9.24247904e+00, -1.79561848e+00, 8.95707279e+00, 2.80826244e+00, -1.42608212e+00, 1.34341140e+01,
            9.68639373e+00, 4.91476929e+00, 3.70343195e+00, 9.60453575e+00, 7.31546036e+00, -1.11059025e+01,
            1.42531764e+01, -5.51595150e+00, -2.21270909e+00, 7.51910069e+00, -6.91807473e+00, -5.06839174e-01,
            5.71139491e+00, 1.68062217e+01, -4.10451879e+00, 7.64832152e+00, -1.09827783e+00, 4.06412120e+00,
            5.70043772e-01, 1.28535331e+00, 1.76726321e+01, 3.91191856e+00, 1.55087360e+01, -2.33195784e+00,
            2.94690213e+00, -2.19184142e+00, -3.37690865e+00, 6.78237639e+00, -7.08819613e+00, -1.76467159e+00,
            5.44981560e+00, 1.01816294e+01, -4.02040363e-01, 4.81004473e-01, -4.75279663e-01, 1.18935612e+01,
            3.73716704e+00, 1.77295410e+01, 7.00073711e+00, 1.14330854e+00, -6.40403589e-01, 4.56552114e+00,
            3.37059208e+00, -2.38818498e+00, 4.37235309e+00, 4.69753907e+00, 6.45111318e+00, 9.81350025e+00,
            1.83739468e+00, 9.84567384e+00, 1.07308923e+01, 5.47213328e+00, 2.13439995e+00, 3.03249885e+00,
            -8.76683469e+00, 3.33157025e+00, -2.55642250e+00, -6.21944578e+00, 5.73025653e+00, 9.41926294e+00,
            -6.74102671e-02, 5.67188872e+00, 3.96285985e+00, 4.32543708e+00, -1.15495512e+01, 3.86949673e+00,
            1.10496856e+01, 8.01577208e+00, 7.13554683e+00, -3.58025560e+00, 3.70512535e+00, 7.18949196e-01,
            4.25444536e+00, 6.32289626e+00, -1.78581318e+00, 1.67565219e+00, 3.34050738e+00, -1.68181169e+00,
            2.90020829e+00, 3.78887585e+00, -2.87433257e+00, 9.58712393e+00, 8.81673487e+00, -2.52101348e+00,
            5.94222558e+00, -3.38923243e+00, 4.94461601e-01, 4.70332268e+00, 1.43425393e+00, 9.22429726e+00,
            1.14075577e+00, 4.67699658e+00, -3.57386866e+00, 8.60721307e+00, 1.04949702e+01, 9.15620222e+00,
            7.21693490e+00, -4.36052056e+00, 8.21199765e+00, 5.99360857e+00, 6.38567097e+00, 4.68783484e+00,
            7.74568845e+00, -5.28779069e+00, -3.95485517e-01, 1.86974247e+00, -2.02240429e+00, 3.93069137e+00,
            -8.28203599e+00, 3.25675268e+00, 2.07658223e+00, 1.15097428e+01, 6.58510882e+00, 7.09824539e+00,
            8.55713877e+00, 8.47133940e+00, 9.53237296e+00, 2.52553268e+00, 4.26240436e-01, 7.45255516e+00,
            8.10022431e+00, 9.23454129e-01, 3.57153765e+00, 1.60717586e+00, 8.89257848e-01, 2.04806363e+00,
            6.36342127e+00, -1.89448766e+00, -4.45179736e+00, -2.64286869e+00, 3.93316301e+00, 1.06407304e+00,
            4.99166543e+00, 3.39356417e+00, 4.80316507e+00, 4.34284333e+00, 8.55959529e+00, -2.33596794e+00,
            5.70500347e+00, 1.42627842e+01, 4.90040657e+00, 1.40640622e+00, -1.55208733e+00, 1.18141241e+01,
            4.95744214e-01, 1.12847506e+00, 9.37845202e-02, -2.59763770e+00, -4.30854913e+00, -1.96935162e+00,
            -4.38096167e-01, 1.14257465e+01, 8.64007210e+00, 7.53149925e+00, -3.98365824e+00, -3.41512024e+00,
            -1.57271443e+00, 7.38198606e+00, 3.84033613e+00, 2.84436339e+00, 9.67664698e+00, -5.68972764e+00,
            1.32410112e+01, 4.64772386e+00, 4.70599201e+00, 8.33599634e+00, 1.11203258e+01, -7.52109094e-01,
            4.95669738e+00, 2.46868344e+00, 3.20405574e+00, 4.81359804e+00, 2.53386985e+00, 6.09834021e+00,
            3.87167277e+00, -5.54800285e-01, 2.95817634e+00, 6.14326994e+00, 1.95887123e+00, 4.89037765e+00,
            2.11648097e+00, -1.30912552e+01, 7.88410605e+00, 2.98911926e+00, 6.16593009e+00, -2.15432374e+00,
            7.42782284e+00, 6.07851046e+00, 8.59889866e+00, 4.42814778e+00, 1.42848782e+01, -1.73231345e+00,
            7.19680443e+00, 1.24922908e+01, 9.43366098e+00, 1.82347480e+00, 1.17118128e+01, -8.67188007e-01,
            -6.83913709e-01, 2.95289065e+00, 3.00928405e+00, 5.69225328e+00, -1.69725508e+00, -4.49367553e+00,
            9.73113534e-01, 1.10956210e+00, 9.90449138e+00, -1.38624351e+00, 7.74652983e+00, 1.73921398e+00,
            3.84269856e+00, 7.43255389e+00, 4.43152798e+00, -3.54633571e+00, 8.59601286e+00, -2.43094227e+00,
            1.24487095e+01, 1.27141443e+00, -2.16034096e+00, 3.86079881e+00, -7.52878645e-01, 2.48032248e+00,
            2.92095232e+00, 2.75831547e+00, -1.83618363e+00, 1.15323400e+00, 5.47162940e+00, 4.95982331e+00,
            4.98497924e+00, 2.84957023e+00, 9.57819225e+00, -1.27105049e+00, 3.56836182e+00, 7.58075255e-01,
            1.06752829e+01, 1.47600823e+00, 8.55296824e+00, 2.31764634e+00, 4.16087626e+00, -5.40077118e+00,
            -1.75416818e-01, 5.07324304e-01, 2.78279931e+00, -3.58086831e-01, 4.77449735e+00, 7.54635742e+00,
            1.09285746e+01, 3.91632483e+00, 1.25431405e+01, 2.56376988e+00, 8.94823438e+00, 5.65907288e+00,
            1.33800753e+01, -7.55027881e+00, 1.54602392e+01, 5.27644581e+00, 2.16384532e+00, 5.24175835e+00,
            5.65644233e+00, 7.51318478e+00, 6.05669495e-01, 1.26657196e+00, 6.63475355e+00, 1.02315480e+01,
            -2.63372890e-01, -7.91102582e+00, 1.15101661e+01, 2.92451977e+00, -4.42450148e-01, 5.91702213e+00,
            5.03780268e+00, 4.95005637e+00, 2.99471270e+00, 1.64505458e+00, 4.07983263e-03, 2.37689033e+00,
            1.07614430e+01, 1.01525516e+01, -1.16437658e+00, 4.88316600e+00, 1.58206430e+00, 7.09713811e+00,
            -1.78550535e+00, 4.26190699e+00, 1.20250847e+01, -4.16793211e+00, -2.32672278e+00, -3.33335598e+00,
            8.41671111e-01, 8.90328839e+00, 8.65145519e+00, 7.11596415e-01, 4.17934574e+00, 2.85740835e+00,
            4.80374696e+00, 7.12043475e+00, 1.49777719e+01, 1.17604115e+01, 3.83748792e+00, 2.69274870e+00,
            6.64376785e+00, 3.54786512e+00, -2.74339529e+00, -7.99474860e-01, 3.43974255e+00, 1.92946965e+00,
            6.05443248e+00, 1.31932217e-01, 2.18474340e+00, -6.58837188e+00, 9.08178232e+00, 3.73679393e+00,
            -9.40423502e+00, 9.14302896e+00, 1.07802825e+01, 4.57524762e-01, -3.90073150e-01, 3.49115835e-01,
            -1.42455833e+00, 4.91859031e+00, 4.63522657e+00, -1.04478984e+00, -2.85186453e+00, 8.67523901e+00,
            4.90606719e+00, 2.47034746e+00, 8.01549790e+00, 1.23966833e+01, 4.87357246e+00, 3.57884482e+00,
            -8.38867343e+00, 9.31344071e+00, 7.50058798e+00, 7.00005127e+00, 1.50602526e+01, 7.05145199e+00,
            -8.72753990e+00, 7.17133503e+00, 5.40680502e+00, 7.98161835e+00, 1.43485079e+00, 5.64257179e+00,
            2.82003579e-01, 2.09609205e+00, 9.16195487e-01, 2.31419394e+00, 1.91534606e+00, -4.83468117e+00,
            -2.74026273e+00, 6.24147122e+00, 2.35590265e+00, 1.62417041e+00, -3.39144356e+00, -2.89187175e-01,
            5.37074486e+00, 6.02581252e+00, 9.17039470e-01, -9.42679662e-01, 6.86199018e-01, 2.00860067e+00,
            1.22773692e+01, 5.81808665e+00, 2.41376885e+00, 5.86624265e+00, 1.90185789e+01, 6.32288978e+00,
            1.65494336e+01, 2.48932472e+00, -1.23794041e+01, 5.97713901e+00, 1.20825272e+00, 1.10587287e+01,
            -3.48617751e+00, -4.59019612e+00, 3.34209780e+00, 2.86493636e+00, 4.35478913e+00, -5.42707776e+00,
            3.05224972e+00, -2.29588651e+00, 5.00210428e+00, 7.22546424e+00, 2.42356902e+00, 8.39666834e+00,
            -4.96221353e+00, 2.88699957e+00, 1.42716371e+01, 6.33155754e+00, -6.98268229e-01, 1.89043602e+00,
            7.22874255e+00, 5.04171910e+00, 9.00266749e+00, -1.51971828e+00, -3.43509159e+00, 3.55381907e+00,
            8.45589050e+00, 7.75625378e+00, 1.65693721e+00, 5.65016128e+00, 2.26264916e+00, 7.46891837e+00,
            7.35359572e+00, -4.30339267e+00, 5.99328476e+00, 1.17077849e+01, 1.12099621e+01, 3.51810025e+00,
            5.58284912e+00, -1.06665422e+01, -8.97179950e+00, 2.22784504e+00, 1.39996814e+00, 1.10358830e+01,
            9.20452310e-01, -3.07722131e+00, -6.23719378e-01, 6.78465404e+00, 3.25770790e+00, -2.62789119e+00,
            1.91927325e+00, -9.52039299e+00, 5.05574944e+00, 5.98174352e-01, 7.61602486e+00, 5.54723212e+00,
            6.80759929e+00, -2.65714594e-01, 8.54714105e+00, 6.17920162e+00, 3.02597671e+00, 5.43008839e+00,
            3.31352136e+00, 5.66338656e+00, 5.75040242e+00, 1.48234259e+01, 5.59520669e+00, 5.27487799e+00,
            4.86875860e+00, 9.42330601e+00, 6.26443397e+00, -1.19388088e+00, 7.31280509e+00, -3.90314840e+00,
            7.05132171e-01, 1.41605419e+00, 7.78068430e+00, 1.44565836e+01, -6.09522206e+00, 2.78836190e+00,
            4.43500057e-01, -1.21213906e+00, 2.68682192e+00, -5.49402867e+00, -1.32414016e+01, -7.90246464e-02,
            1.70598454e+01, 4.29641741e+00, -1.38636054e+00, 4.61083858e-01, 3.49737416e+00, 4.25686182e+00,
            1.01909168e+01, -3.31804310e+00, 6.82974500e+00, 8.63444730e-02, 4.58861579e+00, 7.92246837e+00,
            4.43169754e+00, 1.16869502e+01, -2.02216733e+00, 3.04213403e+00, -1.04209530e+00, 6.72334958e+00,
            -5.63345878e+00, -6.66777917e+00, 2.45856389e-01, 7.17694815e+00, -1.11614591e+00, 3.85567821e+00,
            2.22733877e+00, 1.54735783e+00, -2.15570148e+00, -2.13538155e+00, 5.49656951e+00, -1.37352239e+00,
            -2.56249560e+00, 2.55859023e+00, 8.64746175e+00, 1.07560487e+01, -4.87912459e+00, 4.63421047e+00,
            4.66637786e+00, 1.05300549e+01, 4.35846101e+00, -2.11980780e+00, 2.58036393e+00, 1.51770912e+00,
            8.28389417e+00, 6.33170308e+00, 6.17375202e+00, 1.58901757e+01, 2.13338950e+00, -2.35956852e-01,
            8.61352371e+00, -2.34462864e-02, 5.50596073e+00, 1.44371262e+00, 9.07173949e+00, 9.22908568e+00,
            5.59469899e+00, -4.33670298e-01, 2.03490745e+00, 3.19197512e+00, 2.40242658e+00, 2.60992958e+00,
            -5.14406192e-01, -2.60009371e+00, 6.34028623e+00, 5.13582186e+00, 1.15461783e+01, 1.75628970e+01,
            -3.91215072e+00, 3.85472542e+00, 2.42725697e+00, 1.71056283e+00, 2.52049567e+00, -2.25087684e+00,
            -7.20890304e+00, -1.52125117e+00, 1.74293990e+01, -6.35855939e+00, 4.48524010e+00, -6.88399185e-01,
            1.05399474e+01, -9.67943668e-01, 9.24200453e+00, 1.39643526e+01, 5.31075061e+00, -1.25399043e+00,
            8.41646294e+00, -1.39321416e+00, -2.60219617e+00, 8.94887111e+00, 1.09322265e+01, 7.39372663e+00,
            -5.17913342e-01, 2.02705359e-03, 4.90969956e+00, 2.11157559e+00, 2.36826707e+00, 1.04696139e+01,
            3.77014173e+00, 4.92577999e+00, 8.14895468e+00, 1.84753048e+00, 1.12498355e+01, -3.04216457e+00,
            -1.72719629e+00, 6.41595028e+00, 4.38590917e+00, 1.39743010e+00, -9.00147788e-01, 9.41474715e+00,
            3.84232658e+00, 4.02400351e+00, -2.08013629e+00, -4.22667255e+00, 9.81768590e-02, 3.83307313e+00,
            8.87683306e+00, 3.12915827e+00, 4.14615368e+00, 6.56673517e+00, -5.66539082e-01, -8.81470783e-01,
            1.15900885e+01, -3.41414188e+00, 4.72707279e+00, 8.17458609e-01, 5.78898640e+00, 7.13216299e+00,
            -1.13610580e+00, -8.07908373e-01, 3.10875576e+00, 8.34166589e+00, 6.33451255e+00, 3.21260820e+00,
            5.22071017e+00, 9.84759269e-01, 2.84638750e+00, 8.07671916e-01, 7.11388733e+00, 1.49336634e+01,
            -8.73554442e-01, 6.33369927e+00, -5.25926618e-01, -4.16550744e+00, 1.17937081e+00, 6.23528522e+00,
            8.61484308e+00, -1.77805345e+00, 6.59073697e+00, 5.92521786e+00, -1.30421409e+00, 9.43553294e+00,
            9.75084866e+00, -5.10497801e-01, 2.95984608e+00, 3.32041553e+00, 3.87558361e-01, 4.90065120e+00,
            -2.04098543e+00, 4.35720492e+00, 3.43309793e+00, 6.77584818e+00, -4.24987172e+00, -1.82328779e+00,
            5.56723371e+00, 9.69586808e+00, 7.70299826e+00, 1.36587424e+01, 3.00611460e+00, 5.76166092e+00,
            1.03553432e+01, 1.24641519e+00, 1.12501920e+01, 1.37236359e+01, 3.46425672e-01, 8.83623721e+00,
            6.54636777e+00, -1.19809098e+01, -2.52610506e+00, 1.95671400e+00, 9.46297429e+00, 3.82390416e+00,
            1.18763142e+00, 4.06944148e+00, -4.12106173e+00, 1.07572054e+01, -1.27508861e+01, 2.80189382e+00,
            1.50111036e+00, -3.54604569e+00, 2.60212811e+00, 9.76001271e+00, 1.17958258e+00, -1.98247589e+00,
            -1.53261865e+00, 4.21181617e+00, 6.86521330e+00, -3.80813430e+00, 1.63391395e+00, 2.98748897e+00,
            1.07879552e+01, 8.56151447e+00, -4.85088008e+00, 1.13641415e+01, 4.18349656e+00, 7.88307082e+00,
            9.91773039e+00, 9.10635570e+00, 7.42747616e+00, 1.62294658e+00, 2.29108605e+00, 8.51248114e+00,
            9.23252092e+00, -3.45439193e+00, 9.42742507e+00, 7.17437419e+00, 1.15827568e+00, -5.86770289e+00,
            6.31495355e+00, 1.22136510e+00, 2.60935727e+00, -6.24705701e+00, 4.41822925e-01, 1.71985239e+00,
            3.17492304e+00, 4.41574610e+00, -4.64711700e+00, 4.86130397e+00, -1.13594277e+00, 3.44180902e+00,
            -1.20836991e+01, 7.62085390e+00, 2.98847023e+00, 1.65304046e+00, -7.14049051e+00, 5.25888821e+00,
            5.97311943e+00, 6.86354267e+00, 5.53493051e+00, 1.01807441e+01, 6.63252405e+00, 2.41677366e+00,
            7.60246156e+00, 1.18060225e+00, 5.95669773e+00, -6.90933592e-01, 1.14997641e+01, -2.97337561e-01,
            -1.39067778e+00, -1.92868237e+00, 1.74852411e+00, 3.04861879e+00, 4.30765955e+00, 6.80719063e+00,
            -2.10775548e+00, 3.80926613e+00, -4.06639921e+00, 1.32545843e+01, 1.18720440e+01, 1.03565765e+01,
            1.24508996e+00, 9.52038176e-01, 8.76936699e+00, -7.88943856e-02, 2.85421356e+00, 7.76892665e+00,
            5.43217529e+00, 5.25840598e+00, -8.98933727e+00, -1.14400915e+00, 5.05431526e+00, 5.09087451e+00,
            4.63050651e+00, 4.77639591e+00, 6.67643810e+00, -7.04023427e+00, 8.46613277e+00, 2.87500460e+00,
            4.26003010e+00, -5.11479358e+00, -6.11997475e+00, -6.75612550e-01, -2.64210232e+00, 6.70855409e+00,
            7.64138292e-01, 4.63122523e+00, -9.68034287e-01, 3.35221526e+00, 5.40319757e+00, 4.43063485e+00,
            8.90200585e+00, -3.57765863e+00, 5.90246391e+00, 1.90174545e+00, 7.05192341e-02, 2.62427112e+00,
            3.47853042e+00, 4.04881983e+00, 9.13277814e+00, 1.19342478e+01, 3.71747590e-01, 1.20790563e+00,
            1.83575617e+00, 6.39948274e+00, 6.79022692e+00, 1.40492212e+00, 8.19971657e+00, 1.68080003e+00,
            1.18838391e+01, -6.12361175e-01, 3.21944997e+00, 5.73152713e+00, 5.80417886e+00, -3.39417194e+00,
            2.29205318e+00, 1.48472534e+01, 1.69271605e+00, 8.48744861e+00, -2.06612691e+00, -6.99897181e+00,
            -1.28145176e+00, -4.73523187e-01, -1.93723488e+00, 1.39025931e+01, -3.63544698e+00, 2.67501933e-01,
            3.99772463e+00, 1.29137058e+01, -2.88116565e+00, 4.22585160e+00, 1.87756840e+00, 1.00664656e+01,
            1.23701807e+00, 8.32791224e+00, -1.29003796e+00, -4.46555839e+00, 3.17992926e+00, 7.41340549e+00,
            -3.59891332e+00, -5.02294131e+00, 1.15382479e+01, 1.19865205e+01, 3.09759238e+00, 7.04308558e+00,
            1.07574528e+01, 1.95813488e+00, 5.49820500e+00, 3.05508844e+00, 6.28975291e-01, -5.40630188e+00,
            7.60431718e+00, -1.23968872e+00, 6.00556901e+00, 2.66789089e+00, -1.91937341e+00, 5.35826996e+00,
            5.86690764e+00, 6.42770284e+00, 1.35187158e-01, 1.20130349e+01, -2.33337949e+00, -3.26250685e+00,
            1.22135681e+01, 1.12092160e+01, -1.85465364e+00, 6.03947050e+00, 1.73578790e+01, 3.14668702e+00,
            9.35797577e+00, 1.52310145e+01, 1.11571386e+01, 4.21372474e+00, 2.37077573e+00, 6.24393601e+00,
            -1.74369396e+00, 1.03033446e+01, -8.08853187e+00, 3.53615990e+00, 1.20183152e+01, 5.72814284e+00,
            2.89159526e+00, 1.71356823e+00, -3.61107027e+00, 8.88858057e+00, -2.51359825e+00, 1.66391825e+00,
            2.98762286e+00, 9.04978742e+00, -1.28351665e+00, 6.35065558e+00, 8.62339365e+00, 4.87479903e+00,
            8.02642870e+00, 6.83635794e+00, 6.62970256e+00, 1.89710199e+00, 5.53305014e+00, 3.71065800e+00,
            1.69470997e+01, -2.69581534e+00, -2.26568866e-01, 3.86742120e+00, 2.31966603e-01, 9.56012619e+00,
            6.42275273e+00, 6.82093259e-01, 2.89362371e+00, 6.79605654e+00, 7.71052970e-01, 4.97541060e+00,
            5.67202257e+00, -8.41706776e-01, 6.19415960e+00, 7.17574289e+00, 7.41719926e+00, -1.24340672e+00,
            8.65044929e+00, 1.24448096e+01, 6.13255127e+00, 4.05119831e+00, 4.93555734e+00, 2.97340178e-01,
            7.67197157e+00, -9.69020752e+00, 2.63805769e+00, 4.57817785e+00, -1.74679157e+00, 1.18714378e+00,
            -1.12738613e+00, 1.05393345e+01, 2.27337406e+00, -1.50008783e+00, 7.91404146e+00, -4.62858046e+00,
            8.66830090e+00, 7.17980790e+00, 4.43377202e+00, 3.71490480e+00, -7.05408926e-01, 7.22392749e+00,
            1.49396585e+00, -4.77514633e-01, -5.91077267e+00, -6.73422945e+00, -1.17958850e+00, 1.32558597e+01,
            2.57875189e+00, 4.61328884e+00, 1.22125447e+01, 7.28425800e+00, 6.75496827e+00, -9.60603048e-01,
            -4.68698490e-01, 9.37762980e+00, 4.40806222e+00, 7.30404968e-01, -1.63420965e+00, -7.86931943e+00,
            1.46456428e+01, 6.85256219e+00, 5.73731394e+00, -1.20993542e+00, 1.10090348e+01, 8.07119577e+00,
            1.94409941e+00, 5.51158876e+00, 1.62096553e+01, 9.83699332e+01, 5.61606779e+02, 2.31740899e+03,
            6.73839581e+03, 1.47012523e+04, 2.23847519e+04, 2.43145454e+04, 1.85666387e+04, 9.99364762e+03,
            3.94846467e+03, 1.01844850e+03, 1.95822316e+02, 4.74508038e+01, 6.50216614e+00, 1.64731196e+01,
            -2.48960700e+00, -9.22718139e-01, 9.78531802e+00, 7.58687637e+00, 1.06368577e+00, 6.04362126e+00,
            1.08971652e+01, -7.89882732e-01, 1.00115682e+01, 5.91975254e+00, 6.35831004e+00, 1.14250879e+01,
            7.51711855e+00, 2.55569520e+00, 1.70202206e+00, 3.56805746e+00, 8.19396030e+00, -4.47080100e-01,
            -5.87080311e-01, -2.74938375e+00, 1.29690966e+01, 4.56414662e+00, 1.17481112e+01, 3.36203854e+00,
            1.04853559e+01, 5.41825501e-01, 5.84102664e+00, 1.63488779e+00, 1.63241547e+00, 7.94804899e+00,
            9.36575572e+00, 3.05619126e+00, -3.29884322e+00, -6.86309439e+00, -8.86669769e-01, -3.74675017e-01,
            6.59908289e+00, 3.91817578e+00, 1.94810249e+00, 6.63151240e+00, 8.99990313e+00, -6.28258026e+00,
            -6.89040906e-02, 1.24559617e+01, 4.87541078e+00, -5.75005672e+00, 2.20792093e+00, -4.12073256e+00,
            7.23754971e+00, -3.00366753e+00, -2.46057891e-02, -4.53182415e+00, -9.80394785e-01, -6.22950247e+00,
            3.97247307e+00, 8.37754873e-02, 5.05643348e+00, 6.65718830e+00, -1.78432096e+00, 2.72909774e+00,
            -3.68660918e+00, 1.36273170e+00, 2.02452137e+00, 6.81655155e+00, 7.66872248e+00, 8.02841248e+00,
            2.84816162e-01, -4.93761820e+00, 3.81575195e-01, 9.35201756e+00, 2.27149175e+00, 2.69843405e+00,
            1.19331893e+01, 6.46867089e+00, -2.05161850e+00, 3.88014374e+00, 4.66593872e+00, -2.19102489e-01,
            4.11253841e+00, 1.21567920e+01, 1.80064512e+00, 1.02989472e+01, -5.82569457e-01, 1.29006370e+01,
            -4.43821010e+00, 1.27669399e+00, 1.98384869e+00, 2.54538400e+00, -1.70084469e+00, -2.20947697e+00,
            1.19929896e+01, 6.42264557e+00, -4.27772075e+00, -1.57475958e+00, 5.27122436e+00, 9.68747500e-01,
            6.72630891e+00, 7.93174118e+00, -1.75011394e+00, 1.36996518e+01, 4.98421752e+00, -9.73493041e+00,
            9.55254778e+00, 3.57828962e+00, 3.48404993e+00, 1.02313612e+01, 7.34177146e+00, -1.09276277e+01,
            5.11138127e+00, 1.60654577e+00, 6.70441097e+00, 1.22593820e-01, 3.38834101e+00, 1.21407658e-01,
            1.34975313e+01, 4.62419034e+00, 4.01266425e+00, 2.70287333e+00, 6.37539741e+00, -4.17486820e+00,
            6.10791004e+00, 3.11389202e+00, 9.31787998e+00, 1.92888649e+00, -1.24514959e-01, 2.18903640e+00,
            6.17837765e+00, 8.66504578e+00, 1.31084896e+01, 7.99573826e+00, 9.10205630e+00, 4.96772298e-01,
            3.63490780e+00, 6.17724449e+00, -2.43534404e+00, 4.63809467e+00, 6.29438279e+00, 5.25184057e+00,
            -4.00804509e+00, 1.05229713e+01, 6.19807934e+00, -1.29261805e+00, 8.70324571e+00, -1.03094868e+00,
            2.30178441e+00, -3.10250172e-01, 3.56743638e+00, -3.44012282e-02, 2.89349871e+00, 5.11395535e+00,
            5.47629569e-01, -3.45560037e+00, 8.46774707e+00, 2.28966307e+01, 9.63419820e+00, -1.06302079e+01,
            -7.13750032e+00, -7.42658523e-01, 7.82220386e+00, 1.03778496e+01, -4.23198106e+00, -8.77601196e+00,
            7.15345133e+00, 1.06721094e+01, 5.97761159e-01, 3.34889028e+00, 3.38933382e+00, 2.11661537e+00,
            5.40218264e+00, 6.21524599e+00, -1.09016503e+01, 5.10203692e+00, -3.83641585e+00, 1.17212644e+01,
            8.80282076e+00, 7.33787818e+00, 5.66553793e+00, 5.54797108e+00, -1.54557338e+00, 1.90693610e+00,
            -2.88641059e+00, -6.65266836e-01, 5.54771610e+00, -3.12967455e+00, -5.08225752e+00, 5.40011914e+00,
            5.07878893e+00, 7.05487456e+00, 4.96335208e+00, -2.19338829e+00, 2.96370190e+00, 5.52857105e+00,
            6.91210328e+00, -8.35644099e-01, 9.27558284e+00, 1.78553939e+00, 7.99552615e+00, 6.14208148e-01,
            1.33068052e+01, 8.24990261e+00, 9.25263128e+00, 7.26721872e+00, 5.94576749e+00, -1.53902894e+00,
            9.42901030e+00, 9.04486170e-01, 7.53053337e+00, 4.50438978e+00, 1.09156215e+01, -1.71033210e-01,
            2.51515377e+00, 7.52307475e+00, -6.66915083e+00, -5.31879469e+00, 4.39075246e+00, -8.24940265e+00,
            4.14812091e+00, -5.72150293e-03, 1.14883404e+01, -2.70660026e+00, 6.13394198e+00, 5.36069786e+00,
            1.05967207e+01, 8.68751603e+00, 3.43452565e+00, -3.30368064e+00, -1.14669500e+00, 9.43252092e+00,
            -1.07752085e+00, -5.90410780e-01, 1.45095299e+00, 1.26553756e+00, 6.50780352e+00, 1.79278032e-02,
            -1.29241665e+00, 1.04173869e+01, 4.32692092e+00, -2.56524919e+00, 3.48627469e+00, 8.86323588e-01,
            1.04664255e+01, -1.83509797e+00, 3.84627457e+00, 1.13583305e+01, 7.05496710e-01, -1.36020564e+00,
            1.17393061e+01, 1.24423570e+01, -1.82427669e+00, -3.04300235e+00, 6.27752684e+00, -6.71720599e+00,
            5.27698894e+00, -3.04047688e+00, 6.96153794e+00, 1.16722766e+01, 5.69275473e+00, -4.22482730e+00,
            1.43384403e+00, 1.79587613e+01, 1.12817779e+00, 1.50098873e+01, -7.61558247e-01, 4.44292975e+00,
            3.35853202e+00, 2.78362550e+00, 1.12913617e+01, 1.16742206e+01, -1.68944327e+00, 4.35337846e+00,
            7.25327904e+00, 3.90649310e+01, 2.87808050e+02, 1.33641638e+03, 4.72337069e+03, 1.20081850e+04,
            2.18968450e+04, 2.85980391e+04, 2.66898969e+04, 1.81065136e+04, 8.48000172e+03, 2.92604282e+03,
            7.36086856e+02, 1.38200867e+02, 2.99405601e+01, 5.87759390e+00, 1.09002683e+01, 4.52211500e+00,
            -2.69953454e+00, -9.72884641e-02, 3.07264627e+00, -2.16345216e+00, 5.47381206e+00, 7.95909009e+00,
            7.96995321e+00, 7.28787763e+00, 5.27314493e+00, 5.91967908e+00, 8.84030696e+00, 1.94176460e+00,
            1.14496463e+01, 2.78070899e+00, 1.11066588e+01, 9.67118332e+00, 2.64180436e+00, 6.81597384e-01,
            -6.33459846e+00, 7.60523344e+00, -4.86404880e+00, 6.98506794e+00, 1.50009212e+01, 5.60727386e+00,
            8.10405648e+00, -2.43949962e+00, 4.29152175e+00, 9.32789095e+00, 3.01591313e+00, 3.16415588e+00,
            9.03529046e+00, 1.40386316e+00, 1.57186783e+01, 6.56812498e+00, 6.97119375e+00, 1.44522378e+00,
            2.38103612e+00, 3.07439076e+00, -7.70374122e+00, 4.55595053e+00, 3.91712771e+00, 1.56091480e+00,
            1.61594547e+00, 1.05166518e+01, -1.38222667e-02, 3.19810874e+00, 7.11588246e+00, 6.39392123e-01,
            6.78173731e-01, -7.31178171e-01, 9.86862186e+00, 2.97582323e-01, 1.55760842e+01, 6.70682416e+00,
            9.48210257e-01, 8.20593102e+00, 2.14422470e+00, 2.61137022e+00, 1.71273539e+00, 7.38972549e+00,
            -1.87181253e+00, 5.26302296e+00, 1.29012259e+01, -3.41769637e+00, -2.68067881e+00, 8.93659707e+00,
            2.02798293e+00, -5.45958099e+00, -1.80828738e+00, 2.90620796e+00, 7.08641339e+00, 8.62234118e+00,
            2.40181791e+00, 2.97489310e+00, 3.16811994e+00, -8.02364608e-01, -4.20350397e+00, 1.57948108e+00,
            -4.30361090e+00, 1.80264262e+00, 9.71550398e+00, -2.46499346e+00, 4.29444265e+00, 8.08719401e+00,
            1.70902173e+00, 3.94588787e+00, 5.22240426e+00, 9.30748133e+00, 1.64179557e+01, 4.21644689e+00,
            2.76868925e+00, -4.72546633e+00, 1.38648042e+00, 7.57527934e+00, 7.63714478e+00, 6.01813107e+00,
            -2.22073718e+00, -2.90084752e+00, -5.70163370e-01, -2.22837781e+00, -1.12817090e+00, 2.12855732e-01,
            1.06784157e+01, -3.27164780e-01, 6.95131747e+00, 2.90638172e+00, -5.54208017e+00, -1.82518497e+00,
            1.14285121e+01, 6.33659293e-01, -1.10681809e+00, 8.41272392e+00, 3.75174253e+00, 6.64068762e+00,
            3.47250714e+00, 2.68186508e+00, -5.74069597e-01, 6.94396045e-01, 1.04202866e+00, 2.77420673e+00,
            1.26479617e+01, 1.07669603e+00, 5.53770641e+00, 2.40377505e+00, 1.10700044e+01, -2.94868464e+00,
            -7.37642980e+00, -1.57832121e+00, -7.96388233e+00, 1.57930019e+00, 2.12048456e+00, 6.82124019e+00,
            1.45423742e+00, 2.60791875e+00, 9.75887577e+00, 3.86370643e+00, 3.27827750e+00, 6.47605851e+00,
            1.59099636e+01, -8.20694014e-01, 6.65536055e+00, 2.76515430e+00, -2.02830290e+00, 9.59246423e+00,
            2.91719385e+00, 7.85819758e+00, -3.92784133e+00, 1.00547440e+01, 5.33304542e+00, 2.04058758e+00,
            2.48885081e+00, 1.62799824e+00, 1.09276823e+01, 9.66635848e+00, 1.04301995e+00, 1.76172812e-01,
            4.70428790e+00, 4.13214200e+00, 1.10186733e+01, 7.60777939e+00, 4.59400745e+00, 2.99995196e+00,
            -9.45472654e+00, 5.87182626e+00, 1.63315793e-03, -2.24321984e+00, 7.64107895e-01, 1.68372823e+00,
            1.87858496e+01, -2.49536771e-01, 7.22844445e-04, 1.45421102e+01, -1.84266517e+00, 7.89960177e+00,
            3.78919162e+00, 8.01655291e+00, 8.85592556e+00, 5.41882506e+00, 5.66877825e+00, 1.61566778e+00,
            4.88473566e+00, -5.23153643e+00, 1.34966162e+01, 7.63145291e+00, 5.09332339e+00, 2.66213507e+00,
            1.09939203e+00, 1.23120828e+00, 2.49972044e+00, 5.87111276e+00, -9.92747427e-01, 4.50424014e+00,
            -7.14280280e-01, 1.40904621e+00, 9.54356586e-01, 5.35834917e+00, -1.29592009e+00, 2.62252010e+00,
            1.42152868e+01, 1.68942196e+01, 2.80141366e+00, 4.09416848e+00, 9.45487844e+00, 4.83975014e+00,
            -3.38350309e+00, 1.25935514e-01, 5.98518141e+00, 6.35359362e+00, 3.45468159e+00, 1.15528559e+01,
            2.39997412e+00, 9.58461322e+00, 5.31723711e+00, 7.16107612e+00, 3.52365070e+00, 4.13274319e+00,
            2.13037122e+00, 1.53724884e+00, 6.51502999e-01, -1.96824785e+00, 4.98473337e+00, 3.13453168e+00,
            7.17892800e+00, 3.97384584e+00, 6.77305122e+00, 1.69396244e+00, -3.70784429e+00, 8.34886018e+00,
            8.71763304e+00, 7.48567162e+00, -1.60856822e+00, -3.38335256e+00, 7.27922930e+00, 5.78142968e+00,
            2.56787810e+00, 2.77672068e+00, -4.69577278e+00, -4.43655665e-01, 8.31041579e+00, -3.17807052e+00,
            8.79399985e+00, 3.36685164e+00, -4.20331473e+00, -2.02554714e+00, 1.00810755e+01, -4.83228176e+00,
            9.96178474e-01, -2.59170668e+00, 5.38125195e+00, -2.56287551e+00, 8.05747990e+00, -1.15830875e+00,
            3.64536395e+00, 1.45394732e+01, -3.78514660e-01, 2.38920944e-02, -1.64938610e+00, 1.63219326e+00,
            1.79681134e+00, 1.06463444e+01, 1.13067646e+01, 1.83989352e+00, 4.00504799e+00, 4.51050110e+00,
            4.62280738e+00, 3.89291055e+00, 6.74552176e+00, 6.67855674e+00, 3.70626075e+00, 3.45797456e+00,
            -1.63458775e+00, 3.34340604e+00, 1.19346649e+00, 3.07876119e+00, 9.60941104e+00, 6.08467107e+00,
            1.17684230e+01, 4.31790321e+00, 2.59213289e+00, 9.16108244e+00, 4.18343285e+00, -1.96724833e+00,
            8.89162872e+00, 8.41597537e+00, -8.79335436e-01, 2.35564488e+00, 1.19953641e+01, 1.37486473e+01,
            5.49732858e+00, 1.09248238e+01, 4.95773981e+00, 1.62764998e+00, 2.76046298e-01, 3.82619932e+00,
            7.04994410e-01, -2.15234590e+00, 1.35801294e+00, 4.88669278e-01, -2.43381233e+00, 1.33005503e+01,
            -2.74138449e+00, -3.43256333e+00, -9.57557095e+00, 1.18622317e+01, 9.10809149e+00, 9.34564244e+00,
            1.31268810e+01, -2.16356901e+00, 1.01440796e+01, 4.96871731e+00, 8.82295664e+00, 1.61969386e+02,
            8.61784507e+02, 3.71510210e+03, 1.23427280e+04, 3.04354970e+04, 5.48672695e+04, 7.23514424e+04,
            6.97383135e+04, 4.88383810e+04, 2.56618525e+04, 9.72660522e+03, 2.58724883e+03, 5.23794318e+02,
            1.01382216e+02, 1.24474979e+01, 4.79165374e+00, 4.89252074e-01, 5.36360899e+00, -5.81391960e+00,
            -8.62961997e-01, -1.99335178e+00, -3.79674085e-01, -3.59611879e+00, -4.67065586e-01, 7.24683403e-01,
            3.66772245e+00, -1.98184751e+00, 8.17949120e+00, -4.28908204e+00, 4.82795523e+00, 3.50360574e+00,
            6.79821876e+00, -8.52887173e+00, 5.19397656e-01, -9.36849426e-02, 1.25555220e+01, 4.91681634e+00,
            5.95122766e+00, -2.98483328e+00, 5.90698396e+00, 1.88533255e+00, 9.18567700e+00, 2.89396823e+00,
            9.77807010e+00, 4.29138282e+00, 1.00882558e+01, 1.65723309e+00, 4.46934245e+00, 3.60245812e+00,
            5.49169012e-01, -3.91809221e-01, -6.16767404e+00, 1.20627690e+01, 5.32377539e+00, 6.99049257e+00,
            9.75805969e+00, 8.58870570e+00, -4.49519410e+00, 8.54586869e+00, 9.58626091e+00, 1.18419855e+01,
            6.50919501e+00, 6.15751010e-01, 4.97487772e+00, 1.16964110e+01, 1.12792281e+00, -2.65468318e+00,
            5.68902443e+00, 4.60941927e+00, -7.01146906e-01, 9.76168993e+00, -4.72250013e+00, 8.89286865e+00,
            -1.57543907e+00, -1.60008677e+00, 1.56915441e-01, 6.14028669e+00, -3.49466423e-01, 8.17649271e+00,
            4.11598426e+00, 4.34546607e+00, 3.05806894e+00, 5.35613660e+00, 2.52992678e+00, 6.34578227e+00,
            1.14979071e+01, 1.13189794e+01, 9.05295302e+00, 7.89528595e+00, 7.21717326e+00, 3.45730499e+00,
            -4.15164795e+00, 7.64086177e-01, -6.37679116e-02, 7.10500591e-01, 5.74601391e+00, 1.12310944e+01,
            4.88318957e+00, -2.93462483e+00, 4.40813419e+00, 6.63947932e+00, 1.21888082e+01, 8.65187353e-01,
            -4.42899925e+00, 1.47950855e+00, -2.08240565e+00, 5.81759050e+00, -1.31825000e+00, 7.38487927e+00,
            2.81586245e+00, 1.02034062e+01, 1.31695363e+01, 5.48838948e+00, 1.05654171e+01, 9.71251256e-01,
            7.19182693e+00, 2.50426255e+00, -5.02756118e+00, 9.03266463e+00, -3.97609433e+00, -5.85378570e+00,
            -1.59052648e+00, -3.51363751e+00, 3.58174913e+00, 4.81892750e+00, -7.56706778e-02, -8.58757377e-01,
            -1.95089860e+00, 1.65790175e+00, 1.37896328e+01, 5.40484128e+00, -6.62076915e+00, 5.33499293e+00,
            -3.63471750e+00, -6.84178847e+00, 4.12397199e+00, 6.75756368e+00, -8.49629795e+00, 3.00193131e+00,
            3.17391649e+00, 1.10223418e+01, 7.12839360e+00, -2.81799653e+00, 5.40630116e+00, 2.17941825e+00,
            1.27341739e+01, 6.51549662e+00, 6.44617259e+00, 1.13551083e+01, 6.63474115e+01, 3.86121553e+02,
            2.10857672e+03, 7.40881111e+03, 1.98053767e+04, 3.82607075e+04, 5.32938257e+04, 5.43516236e+04,
            4.00086861e+04, 2.12953966e+04, 8.36411865e+03, 2.33167634e+03, 4.52667289e+02, 7.81244030e+01,
            7.65698924e+00, 3.63732664e+00, 2.51219459e+00, -3.74248033e+00, -4.82371385e+00, -1.18347792e-01,
            3.53504946e+00, -1.50522381e+00, 1.52991373e+01, 4.48905496e+00, -7.94818120e+00, 3.14291216e+00,
            5.99197386e+00, -3.28715680e+00, 4.68749337e+00, 1.11305209e+01, 6.85797358e+00, -8.17537605e+00,
            -5.04548139e+00, 4.51724565e+00, 2.96310346e+00, 2.26219542e+00, -1.32354710e-01, 1.72017116e+01,
            4.48769121e+00, 5.31503892e+00, -4.58291551e+00, 1.24422137e+01, 8.26075153e+00, 1.66900042e+01,
            1.04038638e+01, 1.05981132e+01, 1.35781992e+01, -7.64637566e-01, 5.22356144e-01, 5.21149674e+00,
            -6.84835785e-01, 1.00750661e+01, 3.05493725e+00, 9.14986926e+00, 3.19172270e+00, 2.72496476e-01,
            8.38874428e-01, 1.64777772e-01, 1.11910668e+01, 2.30186877e+00, 5.59138813e+00, -1.42397938e+00,
            -2.89931764e+00, 1.12445295e+00, -3.02351274e+00, -3.25657675e+00, -5.13297730e+00, 6.94056164e+00,
            1.31066946e+01, 8.35392158e+00, 8.20032154e-01, 6.25323133e+00, 1.03558664e+01, 4.59742868e+00,
            3.04848676e+00, 1.08094936e+01, 1.87929603e+00, 1.74842194e+01, 8.65607388e+00, 7.08105951e+00,
            -1.67544702e+00, 4.42578457e+00, 1.30521859e+01, 3.42368249e+00, -1.92993122e+00, 1.16898309e+01,
            4.66755425e+00, 1.08091909e+01, 2.16156719e+00, 5.67208197e-01, 5.70929499e+00, 1.32764684e+01,
            -8.48447942e-01, 1.78839707e+00, 1.42601482e+01, -1.35210742e+00, -5.66255015e+00, 3.16279472e+00,
            3.94759767e-01, 6.17635971e+00, -5.14559445e+00, 3.51484943e+00, 1.15763519e+01, 3.15545981e+00,
            1.03475033e+01, 6.33403951e+00, 2.06689547e+00, 4.48729373e+00, 5.71170524e+00, 4.27238414e+00,
            5.37453200e+00, -4.48705627e+00, 1.25303837e-01, 2.87946814e+00, -1.78357539e+00, 5.93248471e+00,
            9.04192677e-01, 4.92583591e+00, 2.81829644e+00, 1.07251605e+01, 9.52666723e-04, 1.07055009e+01,
            6.32571234e+00, 6.32819414e+00, 3.71684902e+00, 4.76133204e+00, -1.41008452e+00, 1.41162661e+00,
            1.68753911e+01, 1.09953858e+01, 1.14189920e+01, 2.63493671e+00, 1.11921290e+01, 5.69233203e+00,
            -1.01737336e-01, -1.84819803e+00, 3.65862718e+00, 1.41608114e+01, 1.31760573e-01, 8.70589174e-01,
            8.48052715e+00, 4.65483710e+00, 4.08447484e-01, 1.61766127e+00, -1.39921923e+00, -6.61436426e-01,
            3.35155073e+00, 1.07654543e+01, 7.33835338e+00, 2.26978363e+00, -4.77386752e+00, 3.95176070e-01,
            -2.57246819e+00, 1.57055053e+01, 2.12208829e+00, 8.13202192e+00, 2.36589017e+00, 4.67830405e+00,
            6.74015828e+00, 6.00960090e-01, 2.85320914e+00, 9.87438354e+00, 1.13590624e+00, 4.89563320e+00,
            5.38799940e+00, -5.39626622e+00, 9.59013220e+00, 7.07480054e+00, 2.72385807e+00, 1.36987696e+01,
            2.06623905e+00, 8.84384561e-01, -5.12893213e+00, 2.00981050e+00, 1.21573401e+01, 3.41206073e+00,
            -1.62245657e+00, 9.63750292e+00, 1.75368590e+01, 8.88105106e+00, -4.29928610e+00, 5.42095478e+00,
            4.89498126e+00, 4.19835305e+00, 8.58785264e+00, -3.46274545e+00, -3.57347438e+00, -6.96846232e+00,
            7.23468419e+00, -3.17512521e+00, 8.06712163e+00, 5.58520590e+00, -9.19003853e-03, -6.59443404e-01,
            1.19386632e+01, -2.99691527e+00, -4.31706169e-02, -3.89619219e+00, 6.93643871e-01, 3.17158464e+00,
            4.54345828e+00, 9.18687186e-01, 6.38362406e+00, 2.85949125e+00, 1.48644562e+01, 4.42284576e+00,
            7.67758117e+00, -3.54202849e+00, 8.85835298e-01, 2.18270835e+00, -2.62935290e+00, 4.64063760e+00,
            9.02633709e+00, 2.24088491e+00, -2.87589712e+00, 1.05860105e+00, 2.12711570e+00, 1.47260427e+01,
            -2.00013767e+00, 8.71304526e+00, 7.60639541e+00, -1.42252080e+00, 6.65283599e+00, 4.97900307e+00,
            4.93698393e+00, 2.47612664e+00, 6.50512746e+00, 4.99962176e+00, -8.70394778e-01, 1.20836157e+00,
            1.74716524e+00, 1.20418910e+01, 2.47062864e+00, -2.92963119e-01, 9.09488805e+00, 2.54671849e+01,
            1.89654993e+02, 9.31154923e+02, 3.72305473e+03, 1.08791402e+04, 2.36877845e+04, 3.64655373e+04,
            4.20429360e+04, 3.48749855e+04, 2.14515924e+04, 9.42330337e+03, 3.21574489e+03, 7.57506876e+02,
            1.26676198e+02, 2.09595636e+01, 3.20214191e+00, 8.38254325e+00, 2.56705074e+00, 3.53038085e+00,
            -6.13510593e+00, -2.81993816e+00, 9.33902780e+00, -3.25922058e+00, -4.67559002e+00, 4.41487092e+00,
            8.05433743e+00, -5.76436680e+00, 7.96927613e+00, 1.24294438e+01, 6.80780383e+00, 1.03264270e+01,
            8.95793473e+00, 1.00983131e+01, 3.85469793e+00, 2.41597557e+00, 1.63634407e+00, -1.08159281e+00,
            8.33500909e+00, 1.18118485e+01, 9.46416596e+00, 2.50873870e-01, 1.50314927e+00, 9.63804195e+00,
            6.12906017e+00, -2.95970707e+00, -3.78896690e-01, 1.06193677e+01, 7.21683942e-01, 1.81208846e+01,
            4.43593928e+00, 4.91661031e+00, 4.74679156e+00, 7.27200156e+00, 3.47263430e+00, 1.73422015e-01,
            -3.04629997e+00, 4.21939047e+00, 4.53549942e+00, 1.29039518e+01, 9.87250232e+00, 5.86137644e+00,
            -1.49314465e+00, -1.43330953e+00, 3.36981904e+00, 1.27561476e+01, 1.07824175e+01, 9.48997733e+00,
            4.44855296e+00, -2.63028242e+00, 5.83967511e+00, -3.43285230e+00, 5.68138594e+00, 3.89061646e+00,
            -3.09986069e+00, -1.26316265e+01, 5.57939956e+00, 3.37179812e-01, 7.12468143e+00, 6.23492078e+00,
            -4.64993423e+00, 4.47987305e+00, 1.27379897e+01, 2.10766529e+00, 4.60980244e-01, 5.87638574e-01,
            -9.27998786e-02, -3.55992702e+00, 1.54528914e+01, 3.95923094e+00, 3.45952639e+00, 2.47175631e+00,
            5.37568496e+00, 2.06179417e+00, -2.21723540e+00, 3.68410740e+00, 3.60904394e+00, 1.07933698e+01,
            -2.08714576e-01, -7.77480756e+00, 4.29810677e+00, -2.72853887e+00, 1.17174775e+00, 5.62125307e+00,
            7.94257561e+00, 8.41981377e+00, 4.10655130e+00, 3.13357279e-01, 4.94726364e+00, -3.18491682e+00,
            6.99476382e-01, 7.02516315e+00, 4.32131936e+00, 7.10914870e+00, 6.08003547e+00, -5.17038298e+00,
            -2.26062689e+00, -6.77291666e+00, 5.48835499e+00, 5.28913425e+00, -9.91551892e-03, 1.79636776e+00,
            8.02002856e-01, 2.89403124e-01, 3.00925245e+00, 2.82909652e+00, 9.07368281e+00, -1.22143925e+00,
            4.94173258e-01, -9.74016749e+00, -5.21703689e+00, -6.36195106e+00, 2.68442140e+00, 4.42218386e+00,
            1.04089748e+01, -1.10254104e+01, -5.39832764e+00, 1.54413526e+01, -4.96687531e+00, 2.78309878e+00,
            1.23467007e+01, 1.92028979e+00, -1.41323425e+00, 4.02796229e+00, -3.88560930e+00, -2.28142204e+00,
            4.61370141e+00, 8.12409536e+00, -3.61143455e+00, 3.98316712e+00, 7.60734704e+00, 5.34849724e+00,
            4.19158467e+00, -4.10520517e-01, 2.94955389e+00, -7.41692842e-01, 8.81472898e+00, -2.66220800e+00,
            2.40452790e+00, 4.51325149e+00, 1.07705026e+01, -1.81316740e+00, 6.08851902e+00, -1.33960525e+00,
            -1.84972816e+00, 1.00338648e+01, 8.49841168e+00, 2.61640443e-01, 3.78532081e+00, 4.44110171e+00,
            -4.69829792e+00, 6.57603212e+00, -1.60824143e+00, 1.64857785e+00, -1.91777935e+00, 5.15432986e+00,
            1.27109157e+01, 6.65272154e+00, -1.62318213e-02, 1.25270924e+01, 6.85004340e+00, 4.89176311e+00,
            1.86134930e+00, 5.58235114e+00, 5.48535975e+00, 5.09437450e-01, -3.56841266e+00, 2.27601337e+00,
            -9.48658433e-01, -5.00438189e+00, 7.14556925e+00, 1.85377258e+00, 4.91136904e+00, 4.06844658e+00,
            -1.57649490e+00, 4.11562674e+00, 2.80097980e-01, -3.10266677e+00, 9.35727279e-01, -3.93324433e-01,
            1.73864701e-01, 9.75573781e+00, -5.04468346e+00, 3.77359120e+00, 7.69486773e+00, 2.04964933e+00,
            -8.52734230e+00, -2.22758235e+00, 2.98844473e+00, 2.85848731e+00, -4.34319907e+00, 1.10209710e+01,
            -7.42822594e-01, 5.88585513e+00, -9.01851958e-01, 8.23346556e+00, 1.34818971e+00, 9.15128001e+00,
            2.44022317e+00, -4.93212227e+00, 1.14495720e+01, -6.13120244e-01, 2.94087235e+00, 4.33600975e+00,
            4.21070683e+00, 4.19901030e+00, 3.61920035e+00, -4.85270114e-01, 6.74463326e+00, -2.39388024e+00,
            -4.45953969e+00, 6.43799500e+00, 4.81809499e+00, -8.38273273e-01, 3.23000309e+00, 7.04347640e+00,
            9.53262265e+00, 1.19614877e+01, 7.83728214e+00, 1.93434938e+00, 7.76834393e+00, 9.52625100e+00,
            3.54157349e+00, 4.65820074e+00, -1.34838661e+00, -8.11734077e+00, 5.36974205e+00, 3.87964646e+00,
            1.03797407e+01, 1.76463889e+00, 9.45948116e+00, 6.51933587e+00, -4.44066968e+00, 1.00779433e+01,
            5.60353547e+00, 7.72079491e-01, -4.17589179e+00, 9.15217921e-01, 1.02902097e+01, 7.86050701e+00,
            5.10126976e+00, 3.38572923e+00, -3.14122229e+00, -1.40419404e+00, 6.03142328e+00, 1.13392254e+01,
            -2.60501786e+00, -1.04865914e+00, 7.89850547e+00, 1.05717308e+01, -2.35355931e+00, 2.25757554e+00,
            -3.99736215e+00, 7.27040876e+00, 3.10925966e+00, 1.91034663e+00, -4.03804524e+00, 2.48465442e+00,
            3.11061363e+00, 4.05290090e+00, 5.73059255e+00, 5.76271349e+00, -4.13209347e+00, 2.27990264e+00,
            5.82856184e+00, 3.67222374e+00, -5.28997625e+00, 9.65160391e+00, 8.77355894e+00, 1.28957284e+00,
            1.70106357e+01, 4.64143563e+00, -3.69409262e+00, 8.13217906e+00, 2.30708051e-03, 4.14927765e+00,
            7.01400323e+00, 3.19955714e-01, -1.09349546e+00, 2.33793389e+00, 4.68280698e+00, -6.88882265e+00,
            8.17642874e-01, -2.34507105e+00, 1.39609050e+01, 4.00035479e+00, 3.66553375e+00, 5.13312005e+00,
            2.62869316e+00, 4.61096394e+00, 1.03550730e+00, 5.42593741e+00, 1.29105103e+00, -8.00039945e+00,
            6.41247147e+00, 1.02032209e+00, 7.44824474e+00, 4.37216809e+00, 8.38012920e+00, -5.11205365e-01,
            2.51928818e+00, -4.21311151e+00, 5.07920663e+00, 9.89359879e-01, -3.33011280e+00, 7.96810959e+00,
            4.04567851e+00, 1.39036450e+01, -1.44658958e+00, 2.21099403e+00, 1.88088627e+00, 6.95317295e+00,
            1.56174700e+00, 3.24068776e-01, 6.28997248e+00, 7.92304657e+00, -2.72523691e-02, 7.99670279e+00,
            7.60134710e+00, 6.86917011e+00, 4.09540991e+00, 7.60540489e+00, 8.10411588e+00, 6.80656632e+00,
            1.91468941e+00, -1.15203491e+00, 6.20309060e+00, -5.57687553e+00, 3.21032139e+00, -4.80500088e+00,
            7.60800563e+00, 1.60831531e+00, -7.22760323e-01, 7.54708512e+00, 8.41825743e+00, 1.58200046e+00,
            8.89292098e+00, 5.15681964e+00, -4.26088110e+00, -1.59383541e+00, 8.74308069e+00, 1.00428729e+01,
            9.47477023e+00, 1.25623342e+00, 6.69468683e+00, 1.15582301e+01, -4.56381450e+00, 5.68673493e+00,
            1.04810716e+01, 6.09080210e+00, 1.02040189e+01, 6.10268123e+00, 2.81750290e+00, 3.57536010e+00,
            -4.18879305e+00, 6.83238057e+00, 1.98220538e+00, 6.04372786e+00, 4.46170294e+00, 3.63795241e+00,
            1.02447111e+00, -5.98485908e+00, -2.08447591e+00, -5.46918579e+00, 5.15949208e+00, 4.51869099e+00,
            5.65796010e+00, 4.62272548e+00, 7.54507438e+00, 7.20822124e+00, -4.11464798e+00, 1.25727172e+00,
            1.75527515e+00, 3.98045217e+00, 4.07294929e+00, -4.26304003e+00, 3.96818304e+00, 4.08394781e+00,
            9.34669134e+00, 6.87316207e+00, 2.29925302e+00, -5.81575818e+00, 7.45397269e+00, 1.09893431e+01,
            1.90865608e+01, 5.62565536e+00, 1.62332804e+00, 8.95327149e+00, 8.47764813e+00, 9.38996490e+00,
            3.08381835e+00, 5.41692330e+00, 4.91297242e+00, -2.54617865e+00, 4.08893562e+00, 9.29398033e-02,
            -6.37895728e+00, -2.65855239e-01, 5.25757072e+00, 1.44806770e+00, 9.01393518e+00, 2.46044708e+00,
            -5.54319803e+00, -1.43120550e+00, -1.20822017e+00, 4.09809946e+00, 3.35379213e+00, 6.58866991e+00,
            3.81774909e+00, 9.33054558e+00, -2.08925988e-01, 6.77856538e+00, -1.80516186e+00, -5.54762447e+00,
            4.27611157e+00, 2.46851401e+01, 2.06404863e+02, 1.28848683e+03, 6.00935215e+03, 2.06554721e+04,
            5.61063910e+04, 1.19050161e+05, 1.96703852e+05, 2.35968502e+05, 2.36902863e+05, 1.99011045e+05,
            1.22827945e+05, 5.75069548e+04, 2.06008437e+04, 5.56651931e+03, 1.20803811e+03, 1.69911105e+02,
            1.35047705e+01, -7.77008480e-01, -5.23615926e+00, 3.07567909e+00, -1.00982202e+00, 2.32113458e+00,
            3.64824040e+00, -7.15696561e+00, -2.02497599e+00, 4.12568669e+00, 8.54850464e+00, 2.16511545e+00,
            -3.36655003e-01, 2.22994819e+00, -2.47211921e+00, 6.07615383e+00, 1.41107159e+01, -1.56016794e+00,
            7.94447268e+00, 7.11752824e+00, 2.06734624e+00, 8.37270969e+00, 5.35293436e+00, 3.76370973e+00,
            1.40210399e+01, 6.51314072e+00, 7.56407825e+00, 4.48677266e-02, 8.33032021e+00, 3.55418651e+00,
            4.13674387e+00, 3.14379306e+00, -6.15735643e+00, 3.46845112e+00, 7.95261190e+00, -1.97462402e+00,
            -1.16608798e+00, -6.44207009e-01, 7.47334275e+00, 1.23055222e+01, -2.13992263e+00, -1.86004267e+00,
            1.29208397e+01, 3.33701794e+00, 6.27342227e-01, 1.93298359e+00, 1.82177405e-01, -9.93584856e-02,
            -1.80692315e+00, 8.70430089e+00, 4.81331352e+00, 7.72258103e+00, 3.19708991e+00, -6.64177814e+00,
            3.21426641e+00, 1.98521319e+00, -9.99868841e-01, 3.11033877e+00, -3.07979681e-01, 1.12713663e+01,
            1.24589024e+01, -2.53090936e+00, -2.38140664e+00, 9.99995205e-03, 3.83518521e+00, 1.07903519e+01,
            1.96643444e+00, 3.96894448e+00, 5.41448821e+00, 4.39465404e+00, 8.19728319e+00, 3.53548105e+00,
            2.77162893e+00, -1.19316823e+00, 8.37188012e+00, 3.15071748e+00, 1.67988683e+00, -3.26070182e+00,
            8.15745739e+00, 1.36312272e-01, 6.92583117e+00, 1.04460441e+00, 8.04706564e+00, -1.93261158e-01,
            4.35998924e+00, -2.82246855e+00, 3.02100270e+00, 1.16343213e+01, 2.96477918e+00, 2.38401353e+00,
            2.88046059e+00, 6.61535315e-01, 1.03154597e+01, 3.53939450e+00, 7.27393390e+00, 8.69345349e+00,
            -2.69928239e+00, 9.70624250e+00, -6.82459209e-01, -6.88234473e+00, 8.46652087e-01, 7.05680610e+00,
            2.06780610e+00, 6.87266969e+00, 6.49995504e+00, -1.53008900e+00, 4.20416385e+00, 1.31443800e+00,
            3.94387595e+00, 3.06169857e+00, 1.02358758e+00, 7.20113343e+00, 8.53999872e+00, 8.40222785e+00,
            1.10574193e+01, -3.52489900e+00, 1.13967754e+00, 3.66442378e+00, 6.38564484e+00, 1.20002035e-01,
            9.38538104e+00, 1.58999227e+00, -5.94068587e+00, 6.70145106e+00, 9.15727227e+00, 7.21515512e+00,
            3.00680160e+00, 3.70791103e+00, 3.60120272e+00, 1.22214706e+01, -2.56022413e+00, 2.68336789e-01,
            1.11802192e+00, 9.99443084e+00, 2.07620256e+00, -2.43028809e-01, 5.62076557e+00, -8.76720105e+00,
            6.52450256e+00, 3.33041919e+00, 5.87249855e+00, 1.28661212e+01, -5.87899968e+00, 1.16637515e+01,
            -4.44774926e+00, -4.88524851e+00, 1.54299667e+00, -1.30451568e+00, -2.62224356e+00, 9.01120972e+00,
            3.50477497e+00, 8.50591282e+00, -2.83448062e+00, 5.39706104e+00, 1.31064450e+01, 6.08980198e+00,
            4.40133245e+00, 1.11284665e+01, 5.94048898e+00, -2.79675884e+00, 3.17168056e+00, 1.07099179e+01,
            1.67443018e+00, 5.79831367e+00, 5.60396254e+00, -8.68419879e+00, 3.54565141e+00, 5.09208475e+00,
            4.80571315e+00, -1.16795438e+00, 4.20067873e+00, 1.76075683e+00, 5.43082373e+00, 7.63343587e+00,
            7.66510752e+00, 5.53614454e+00, 5.91097561e+00, 4.45493021e+00, 3.04077162e+00, -7.67698390e+00,
            7.04839997e+00, 5.59076302e+00, -7.27658131e+00, 3.30331264e+00, 1.58234319e+00, 1.17555377e+00,
            7.92472647e+00, 3.26981731e+00, -5.69757277e+00, 4.95676559e+00, 2.92139033e+00, 1.26843864e+00,
            3.42694058e+00, 5.23042484e+00, 3.78747086e-01, -3.38449747e+00, 8.73865181e+00, -4.13861071e+00,
            2.06627568e-01, -9.30876589e+00, 1.12043132e+00, 2.85285905e+00, 6.60057685e+00, 3.97296058e+00,
            -1.05680692e+00, 4.30465334e-01, 4.26186068e+00, 1.26033968e+01, 1.68496964e+00, 1.42545583e+01,
            5.19691514e+00, 1.49206349e+00, -9.31648576e+00, 7.16269324e+00, -2.29206644e+00, -1.14607592e+00,
            6.93310824e+00, -4.65994833e+00, 3.38638476e+00, -9.62614734e+00, 3.59754840e+00, 1.18464347e+01,
            1.74067338e+01, 5.55406937e-01, -2.28537544e+00, 3.70990004e+00, 4.83679247e-01, 7.37429281e+00,
            7.36933938e+00, 3.97875823e+00, 8.50635374e+00, 1.69138552e+00, 1.26834070e+01, 5.66220888e+00,
            9.11749008e+00, -3.15090839e+00, -3.70206653e+00, 1.05856128e+01, 5.57279007e+00, 5.21669281e+00,
            -2.97787684e+00, 4.47356619e+00, 4.98670053e+00, 1.35716131e+01, -4.47937759e+00, 5.13784152e+00,
            3.63257293e-01, -1.34847251e+00, 8.48403166e+00, 1.98449135e+00, 7.13645559e+00, -2.66917546e+00,
            3.37714348e+00, 3.90296904e+00, 9.18459192e+00, -2.40090107e+00, 5.33692502e+00, -2.31173577e+00,
            1.02600386e+01, 1.59235778e+01, 1.14985164e+01, 8.31621403e+00, -4.47041030e+00, -3.02718905e+00,
            5.10210813e+00, 3.90595865e+00, 8.52083512e+00, 4.94919988e+00, -1.78010291e+00, 2.85041516e+00,
            1.20740561e-01, 6.33297331e+00, -1.03870804e+01, 6.34241433e+00, 1.83050009e+00, 8.33108128e+00,
            1.05230312e+01, 9.30211361e+00, -8.47373750e-03, 5.50445564e+00, 1.62557362e+01, 1.33655013e+00,
            -5.27132289e+00, 7.95187045e+00, -1.99940668e+00, 2.17870818e+00, 8.48751266e+00, 4.44875591e+00,
            2.98900127e+00, -2.07605721e+00, 2.80579658e+00, 5.62624881e-01, -6.77154965e+00, -2.98749637e+00,
            1.23287778e+00, -8.75840666e+00, 7.92840276e+00, 1.73519427e+00, 3.41674413e+00, 2.41110653e+00,
            3.79582543e+00, 5.44792949e+00, 1.23203165e+01, 8.35551663e+00, -1.18661347e+00, 4.77647255e+00,
            5.65321399e+00, 1.00308640e+01, 3.59291534e+00, 7.63533206e+00, -2.33206606e+00, 1.02937678e+01,
            4.67133135e-01, 6.06923178e+00, 4.16185144e+00, -1.08798773e+01, 8.88162919e+00, 4.45951777e+00,
            1.22113375e+01, 7.99653820e-01, 3.57785353e+00, 7.80278450e+00, 1.22343481e+01, 2.59866385e+00,
            -1.55371686e+00, 1.12812356e+00, 7.40208988e+00, 3.98917389e+00, 8.82137035e+00, 5.94948760e+00,
            6.40733828e+00, 2.28750399e+00, -4.58382933e+00, 3.73019138e+00, 3.41875363e+00, 6.48947941e+00,
            4.13979849e+00, 2.47316130e+00, 4.58882442e+00, 1.12287004e+01, 4.31571783e+00, 2.86369490e-01,
            5.46374471e+00, -1.19956636e+00, -4.54850625e+00, -1.55822634e+00, 6.38073593e+00, 6.12026685e+00,
            1.04506840e+01, 1.17448824e+01, 1.20845533e+01, 3.20182168e+00, 3.71342499e+00, 5.33592496e+00,
            4.88136473e+00, -4.82133879e+00, 3.39164302e+00, 3.30849108e-02, 1.72587367e+00, -1.30377507e+00,
            -1.51791137e+00, 5.67824223e+00, 1.71344163e+00, 4.90864932e+00, 1.57029788e+00, 3.89250280e+00,
            7.89102045e+00, 2.88391682e+00, -1.59193540e-01, -3.25219658e+00, 2.73056314e+00, -3.46373040e+00,
            7.74105330e+00, 8.99588120e+00, -7.22491669e-01, 4.46893661e+00, -3.11903887e+00, -1.68046951e+00,
            -4.47849053e+00, -5.51617505e+00, -7.06682011e-01, 8.52891861e-01, 9.13138543e+00, 5.82807980e+00,
            1.58448013e+00, -9.41361506e+00, 1.19170124e+00, -2.08558052e-01, 1.15089239e+00, 4.57770001e+00,
            2.78834211e+01, 1.82854919e+02, 1.19963444e+03, 5.71814180e+03, 2.10463927e+04, 5.76251009e+04,
            1.22441711e+05, 1.96423803e+05, 2.33011148e+05, 2.28078764e+05, 1.78086522e+05, 1.01893068e+05,
            4.46138518e+04, 1.51820278e+04, 4.06105648e+03, 8.02469370e+02, 1.17655508e+02, 7.65380331e+00,
            1.24556800e+01, 5.98065807e+00, 4.88221432e+00, 4.28655689e+00, 1.63723662e+00, 3.41103067e+00,
            4.28381837e+00, 9.53239052e-01, 8.04588838e+00, -3.75245553e-01, -6.86330842e+00, -1.13272659e-01,
            2.50576636e+00, 5.98867160e+00, 3.94938396e+00, 1.82037853e+00, 5.80404360e+00, 4.88735066e+00,
            6.06612586e+00, -5.59332312e+00, -1.96539714e+00, -6.36722223e+00, -6.20281230e+00, 2.27133277e+00,
            4.07033485e+00, 7.19247547e+00, 2.18805052e+00, -4.34847869e-01, 4.15363046e+00, 7.93156724e+00,
            3.82454970e+00, 1.49802468e+01, -1.49032085e+00, 6.72381403e+00, 1.42479093e+01, 6.32818907e+00,
            8.11462302e+00, 4.40367003e+00, 5.60623610e+00, 7.21717567e+00, 5.08623948e+00, 1.65746206e+00,
            2.90884180e+00, -5.17311755e+00, -3.54726248e+00, 1.92137787e+01, 5.58853651e+00, 7.81820709e+00,
            -6.21369794e+00, 5.00745780e+00, 5.37606608e+00, -6.00854278e+00, 5.83689682e+00, 2.87287472e+00,
            1.79528398e+01, 1.11644728e+00, 6.68690662e+00, 1.29822047e+01, 7.86712731e+00, 9.26455729e+00,
            4.99332089e+00, 4.08491728e+00, 1.20016677e+01, -9.19279474e-01, 7.67281376e+00, 7.53352695e+00,
            -1.00660112e+01, 5.06972482e+00, 1.28780534e+01, 5.40332524e+00, 5.71604316e+00, 7.24939260e+00,
            3.23908777e+00, 6.33462851e+00, 2.25081830e+00, 1.98054658e+00, 8.96537572e+00, -1.21067345e+00,
            -2.65975334e+00, 7.60361697e+00, 1.20753248e+01, 6.49195106e+00,
            7.75675150e+00, -5.78234086e+00, 3.50123791e+00, 9.87413637e+00, 7.76298031e-01, 9.26081868e+00,
            1.07010898e+01, 7.83019847e+00, 5.53884376e+00, 4.17203711e+00, -2.39426688e+00, 5.48126615e+00,
            3.92434561e+00, 7.56597995e+00, -2.71128058e+00, -5.15104119e-01, 1.39761635e+01, 8.55567796e+00,
            -4.83103059e+00, 1.07504273e+01, -1.91835931e+00, 7.47316936e+00, -1.14579918e-01, 4.92364290e-01,
            1.05564252e+01, 2.23286611e+00, 5.03219604e+00, 8.60842762e+00, 6.53956552e+00, 3.71735774e+00,
            3.64929586e-01, 1.02625202e+01, -9.31515935e+00, 3.04495930e+00, 6.43655805e+00, 6.25853537e-01,
            -8.41138590e-01, -3.12461778e+00, 5.50423172e+00, -1.09641169e+00, 4.25193427e+00, -1.79070054e-01,
            -2.87090530e+00, 2.87092503e+00, 1.58179461e+00, 4.59503187e-01, 6.92535321e+00, 6.61369152e+00,
            -4.70632774e+00, 4.20108113e-01, 4.50098451e+00, 1.16107413e+01, 2.01186398e-01, 3.51229840e+00,
            5.25271232e+00, 6.67538033e+00, -3.06768445e+00, 1.45849908e+00, 1.15122311e+00, 3.37520786e+00,
            1.84607433e+01, -6.50530474e+00, -2.75661117e-01, 5.54808950e+00, 6.06968187e+00, 9.05612398e+00,
            1.21547615e+01, -4.68331907e+00, 7.24817531e+00, 1.78932895e+00, 4.80174496e+00, 2.80171883e+00,
            4.69988455e+00, 3.41061984e+00, 2.82545019e+00, 7.39568106e+00, 9.87904558e+00, -3.22338170e-01,
            8.79338086e+00, 4.31503022e+00, 5.29431800e+00, -2.25198415e+00, -1.01688116e+00, -2.97051873e+00,
            -5.47996357e+00, 7.04578928e+00, 5.25838258e+00, 9.95139896e+00, -1.42477426e+00, -2.17276527e+00,
            -1.02654209e+00, 1.98557040e+00, 1.26064047e+00, -5.62265819e-01, 1.96292432e+00, 1.09351159e+01,
            1.71432896e+00, 3.07206656e+00, 3.89104000e+00, 2.56719036e+00, 4.86356446e+00, 5.29536194e+00,
            1.65323851e+01, -3.79642195e+00, -2.74518884e-01, 1.90470435e+00, 7.86812576e+00, 1.12677102e+00,
            2.66355597e+00, 7.27440951e+00, 2.65795890e+00, -4.89090106e+00, -5.20595245e-01, 3.66466110e+00,
            6.00648372e+00, -8.93324599e-01, 1.40730183e+01, 3.99178519e+00, 6.81242296e+00, 6.26746080e+00,
            1.29000345e+00, 6.64530824e+00, -3.67221594e+00, 6.32701242e-01, 9.16986438e+00, -6.75589418e+00,
            2.89146737e+00, 2.66035947e+00, 4.71836650e+00, -2.82489624e+00, 8.31735491e+00, 3.57785488e+00]

    return numpy.array(data)


def test_peak_finding(benchmark, spectrum):
    # Reference
    peakin = [478, 899, 1028, 1086, 2031, 2317, 2645, 2795, 3030, 3466, 3872]
    result = benchmark(peakdet.find_peaks_indexes, spectrum, 5, 278.040081456)

    assert numpy.allclose(peakin, result)


def test_peak_refine(benchmark, spectrum):
    y_results = [12173.29394431, 13836.87760174, 16195.39569704, 14005.5855603, 23886.19685264, 28189.30599361,
                 71913.00598368, 54225.26907113, 41031.12155186, 241241.71427945, 235431.61691958]
    resultado_nuevo = [477.85626527, 898.93671683, 1027.96507433, 1086.20599208, 2030.76951558, 2317.26107361,
                       2645.32659386, 2794.60128342, 3029.93485482, 3465.52062997, 3872.3456723]
    peakin = peakdet.find_peaks_indexes(spectrum, 5, 278.040081456)
    peakpos = benchmark(peakdet.refine_peaks, spectrum, peakin, 5)

    assert numpy.allclose(peakpos[1], y_results)
    assert numpy.allclose(peakpos[0], resultado_nuevo)


@pytest.mark.xfail(raises=ValueError)
def test_peak_finding_window_odd(spectrum):
    peakdet.find_peaks_indexes(spectrum, 4)


@pytest.mark.xfail(raises=ValueError)
def test_peak_finding_window_small(spectrum):
    peakdet.find_peaks_indexes(spectrum, 1)


@pytest.mark.parametrize("peaks", [[20, 10, 0, 0, 0, 0, 0, 21],[20, 0, 0, 0, 0, 0, 0, 20],[20, 10, 0, 0, 0, 0, 0, 0],[0, 0, 0, 0, 0, 0, 0, 20]])
def test_peak_index(benchmark, peaks):
    peaks_result = []
    result = benchmark(peakdet.find_peaks_indexes, numpy.array(peaks), 3, 0.0)
    assert numpy.allclose(peaks_result, result)


def test_peak_index_2_indexes(benchmark):
    peaks_result = [3,5]
    test = numpy.array([10, 0, 0, 10, 0, 10, 0, 0])
    result = benchmark(peakdet.find_peaks_indexes, test, 3, 0.0)
    assert numpy.allclose(peaks_result, result)


@pytest.mark.parametrize("peaks", [[0, 10, 0, 0, 10, 0, 0, 0],[0, 10, 0, 0, 10, 0, 0, 10]])
def test_peak_index_2_indexes_param(benchmark, peaks):
    peaks_result = [1,4]
    result = benchmark(peakdet.find_peaks_indexes, numpy.array(peaks), 3, 0.0)
    assert numpy.allclose(peaks_result, result)


def test_peak_index_1_index(benchmark):
    peaks_result = [2]
    test = numpy.array([0, 10, 20, 10, 0, 0, 0, 0])
    result = benchmark(peakdet.find_peaks_indexes, test, 3, 0.0)
    assert numpy.allclose(peaks_result, result)


@pytest.mark.xfail(raises=ValueError)
def test_exception_fpeak_negative(spectrum):
    peakdet.find_peaks_indexes(spectrum, window_width=3, fpeak=-100)


@pytest.mark.xfail(raises=ValueError)
def test_exception_fpeak_limit(spectrum):
    peakdet.find_peaks_indexes(spectrum, window_width=3, fpeak=2)


@pytest.mark.xfail(raises=ValueError)
def test_exception_fpeak_limit1(spectrum):
    peakdet.find_peaks_indexes(spectrum, window_width=3, fpeak=1)


@pytest.mark.xfail(raises=ValueError)
def test_exception_fpeak_limit2(spectrum):
    peakdet.find_peaks_indexes(spectrum, window_width=3, fpeak=4)


def test_fpeak_1():
    peaks_result = []
    test = numpy.array([0, 10, 20, 20, 0, 0, 0, 0])
    result = peakdet.find_peaks_indexes(test, 3, threshold=0.0, fpeak=0)
    assert numpy.allclose(peaks_result, result)

    peaks_result = [2, 3]
    test = numpy.array([0, 10, 20, 20, 0, 0, 0, 0])
    result = peakdet.find_peaks_indexes(test, 3, threshold=0.0, fpeak=1)
    assert numpy.allclose(peaks_result, result)
