
!> A program that runs unit tests on the global MHD equilbrium geometry specification.
!!
!! Current tests:
!!   1. Calculates several geometric coefficients from the global
!!      flux surface specification to verify that they have not changed.
!!
!!
!! This is free software released under the MIT license
!!   Written by: Edmund Highcock (edmundhighcock@users.sourceforge.net)
!!   Adapted by: Justin Ball (Justin.Ball@epfl.ch)
program test_asym_geo_global
  use unit_tests
  use theta_grid, only: init_theta_grid, finish_theta_grid
  use theta_grid, only: initialized, theta
  use theta_grid, only: bmag, grho, gbdrift, cvdrift, gds2, gds21
  use egrid
  use mp, only: init_mp, finish_mp, proc0
  use file_utils, only: init_file_utils, finish_file_utils
  use species, only: init_species, nspec, spec
  use constants, only: pi
  use kt_grids, only: naky, ntheta0, init_kt_grids
  use gs2_layouts, only: init_gs2_layouts, g_lo, ie_idx
  implicit none
  real :: eps
  logical :: dummy
  real, dimension(33, 7) :: new_rslt
  real, dimension(33, 7) :: prev_rslt


  eps = 1.0e-7

  if (precision(eps) .lt. 11) eps = eps * 1000.0

  call init_mp
  call announce_module_test('asym_geo_global')

  new_rslt = 0.

  prev_rslt = prev_result()
  if (proc0) call init_file_utils(dummy, .true., .true., .false., 'local')
  call init_theta_grid
  call get_new_results()
  if (proc0) call finish_file_utils
  call announce_test(' that global has not changed')
  call process_test( &
    test_results(new_rslt(:,:), prev_rslt, 2e-2, 3.0e-1), &
    ' that global has not changed')



  call close_module_test('asym_geo_global')
  call finish_mp

contains
  function test_results(rslt1, rslt2, epsin, bmagerror)
    real, dimension(:,:) :: rslt1, rslt2
    logical :: test_results
    real :: epsin, bmagerror
    test_results = .true.
    call announce_check('bmag')
    call process_check(test_results, &
      agrees_with(rslt1(:,1), rslt2(:,1), bmagerror), 'bmag')
    call announce_check('grho')
    call process_check(test_results, &
      agrees_with(rslt1(:,2), rslt2(:,2), bmagerror), 'grho')
    call announce_check('gbdrift')
    call process_check(test_results, &
      agrees_with(rslt1(:,3), rslt2(:,3), epsin), 'gbdrift')
    call announce_check('cvdrift')
    call process_check(test_results, &
      agrees_with(rslt1(:,4), rslt2(:,4), epsin), 'cvdrift')

  end function test_results
  subroutine get_new_results()
    call init_theta_grid
    new_rslt(:,1) = spl_fit(bmag)
    new_rslt(:,2) = spl_fit(grho)
    new_rslt(:,3) = spl_fit(gbdrift)
    new_rslt(:,4) = spl_fit(cvdrift)
    new_rslt(:,5) = spl_fit(gds2)
    new_rslt(:,6) = spl_fit(gds21)
    call finish_theta_grid
  end subroutine get_new_results

  function spl_fit(var)
    use splines
    type(spline) :: spl
    real, dimension(:), intent(in) :: var
    real, dimension(33) :: spl_fit
    integer :: i
    call new_spline(size(var), theta, var, spl)
    do i = 1,33
      spl_fit(i) = splint(prev_rslt(i,7), spl)
    end do
    call delete_spline(spl)
  end function spl_fit

  function prev_result()
    real, dimension (33,7) :: prev_result

prev_result = &
reshape([&
! Bmag
1.27658976420608 , &
1.27543059479085 , &
1.26441546371943 , &
1.23484406312188 , &
1.20274088169899 , &
1.15277539474225 , &
1.10295957813011 , &
1.05400330444693 , &
1.0036264790282 , &
0.962941772822837 , &
0.936270189937017 , &
0.915695687367451 , &
0.896791850239388 , &
0.876916936372997 , &
0.854123317068131 , &
0.830241304336681 , &
0.824547154531564 , &
0.830241304336681 , &
0.854123317068131 , &
0.876916936372997 , &
0.896791850239388 , &
0.915695687367451 , &
0.936270189937017 , &
0.962941772822837 , &
1.0036264790282 , &
1.05400330444693 , &
1.10295957813011 , &
1.15277539474225 , &
1.20274088169899 , &
1.23484406312188 , &
1.26441546371943 , &
1.27543059479085 , &
1.27658976420608 , &
! grho
0.603210285160155 , &
0.604167368310658 , &
0.605276542816884 , &
0.598354150102021 , &
0.581904925126082 , &
0.552593175300915 , &
0.543574024554079 , &
0.584932513598576 , &
0.719266975872997 , &
0.934669940497681 , &
1.11723244811437 , &
1.23817710683911 , &
1.30120339156853 , &
1.2996558473705 , &
1.20006992294854 , &
0.93361102625124 , &
0.777606351264851 , &
0.714410374678469 , &
0.774901347005355 , &
0.855788216629032 , &
0.922016295737529 , &
0.975600061346732 , &
1.02145932194407 , &
1.06025021454146 , &
1.07547289175806 , &
1.03444921848742 , &
0.949804053844579 , &
0.839999222507609 , &
0.727556969093593 , &
0.663397313431075 , &
0.616284055544682 , &
0.604094677638426 , &
0.603210285160074 , &
! gbdrift
-1.33248976741345 , &
-1.17466611780719 , &
-0.951105668165607 , &
-0.669013203442768 , &
-0.457563284404402 , &
-0.199374785082813 , &
0.0222025931628693 , &
0.206466938331983 , &
0.290591333407503 , &
0.278482033968678 , &
0.255876030817078 , &
0.24731164342606 , &
0.261176996614344 , &
0.30724271881571 , &
0.410231017870196 , &
0.578187819874322 , &
0.688561114966926 , &
0.800368391384352 , &
0.960688088251269 , &
1.06329388473032 , &
1.12413978556924 , &
1.14924238495048 , &
1.14246305195681 , &
1.09639686339545 , &
0.975041752373722 , &
0.782578171036883 , &
0.56651010525637 , &
0.30600226121939 , &
-0.0361750144811776 , &
-0.342701536283546 , &
-0.779225759370871 , &
-1.11106749258069 , &
-1.20716155977844 , &
! cvdrift
-1.29626021760933 , &
-1.13802849989306 , &
-0.913577669182128 , &
-0.629614033177262 , &
-0.416323336577055 , &
-0.155428799539355 , &
0.0693482838337908 , &
0.258219174656909 , &
0.349446605094527 , &
0.344625283637301 , &
0.326336035338745 , &
0.320658404210892 , &
0.337422988043997 , &
0.387114577049452 , &
0.494712267885322 , &
0.663110083750464 , &
0.770112138072069 , &
0.878926021664994 , &
1.03793877375137 , &
1.13871698215241 , &
1.19738453210071 , &
1.22033728871456 , &
1.21122144741783 , &
1.16199802694285 , &
1.03568778563527 , &
0.837579451792781 , &
0.616624908416168 , &
0.351555327492623 , &
0.00509087063060037 , &
-0.303960353286136 , &
-0.742415251132626 , &
-1.07478885422174 , &
-1.17093200997432 , &
! gds2
9.95826670726143 , &
9.10988520281037 , &
8.1812084791778 , &
7.35798267146448 , &
6.69849671738242 , &
5.50952332809903 , &
4.23833466121629 , &
3.03613192674247 , &
1.92732277361199 , &
1.22427267152132 , &
0.85998620104471 , &
0.649796185682891 , &
0.528974471427739 , &
0.482634999599694 , &
0.533734505850971 , &
0.756873317718338 , &
1.00873314166442 , &
1.33362558717305 , &
1.91896239735854 , &
2.4717261338051 , &
2.93557549300508 , &
3.30577032400869 , &
3.61251567204091 , &
3.85741677826495 , &
3.94955333727688 , &
3.8378748387185 , &
3.79255842997973 , &
4.11501636486512 , &
5.12791808193097 , &
6.39307164290032 , &
8.57153329728752 , &
10.3825011098032 , &
10.9209001886057 , &
! gds21
1.86286292894779 , &
1.70779898732936 , &
1.5407085980472 , &
1.36959867163635 , &
1.1924632000522 , &
0.84728063889947 , &
0.488789239039743 , &
0.173939850582263 , &
-0.151154110178759 , &
-0.413180908627265 , &
-0.495072036197634 , &
-0.441994089504594 , &
-0.306937935985994 , &
-0.126123787569995 , &
0.0271633660220536 , &
-0.0554712546415856 , &
-0.220606329252638 , &
-0.449686953285259 , &
-0.92939240544825 , &
-1.35979377716508 , &
-1.70813426774287 , &
-1.98405171155839 , &
-2.21277493074252 , &
-2.39559846753779 , &
-2.4452996716201 , &
-2.23900947275725 , &
-1.91674058823388 , &
-1.622084869483 , &
-1.47485074145661 , &
-1.50505640192924 , &
-1.70638867061648 , &
-1.93964477305629 , &
-2.01607080588923 , &
! theta
-3.14159265358979 , &
-3.01069295969022 , &
-2.81434341884086 , &
-2.55254403104171 , &
-2.35745693226546 , &
-2.12182938216258 , &
-1.92413941474038 , &
-1.73634131051826 , &
-1.51092689030096 , &
-1.24526972838728 , &
-0.98174770424681 , &
-0.719948316447661 , &
-0.458148928648511 , &
-0.196349540849362 , &
0.0654498469497874 , &
0.327249234748937 , &
0.47197356555882 , &
0.600493528375452 , &
0.760504225052732 , &
0.868382642268042 , &
0.95406205918831 , &
1.03451089886336 , &
1.12348995983655 , &
1.24354709204596 , &
1.43989663289532 , &
1.70169602069447 , &
1.96349540849362 , &
2.22529479629277 , &
2.48709418409192 , &
2.67280432895371 , &
2.9063077603848 , &
3.08766802498129 , &
3.14159265358979 &
], (/33,7/))
end function prev_result

end program test_asym_geo_global
