
!> A program that runs unit tests on the Fourier geometry specification.
!!
!! Current tests:
!!   1. Calculates several geometric coefficients from the Fourier
!!      flux surface specification to verify that they have not changed.
!!
!!
!! This is free software released under the MIT license
!!   Written by: Edmund Highcock (edmundhighcock@users.sourceforge.net)
!!   Adapted by: Justin Ball (Justin.Ball@epfl.ch)
program test_asym_geo_fourier
  use unit_tests
  use theta_grid, only: init_theta_grid, finish_theta_grid
  use theta_grid, only: initialized, theta
  use theta_grid, only: bmag, grho, gbdrift, cvdrift, gds2, gds21
  use egrid
  use mp, only: init_mp, finish_mp, proc0
  use file_utils, only: init_file_utils, finish_file_utils
  use species, only: init_species, nspec, spec
  use constants, only: pi
  use kt_grids, only: naky, ntheta0, init_kt_grids
  use gs2_layouts, only: init_gs2_layouts, g_lo, ie_idx
  implicit none
  real :: eps
  logical :: dummy
  real, dimension(33, 7) :: new_rslt
  real, dimension(33, 7) :: prev_rslt


  eps = 1.0e-7

  if (precision(eps) .lt. 11) eps = eps * 1000.0

  call init_mp
  call announce_module_test('asym_geo_fourier')

  new_rslt = 0.

  prev_rslt = prev_result()
  if (proc0) call init_file_utils(dummy, .true., .true., .false., 'local')
  call init_theta_grid
  call get_new_results()
  if (proc0) call finish_file_utils
  call announce_test(' that Fourier has not changed')
  call process_test( &
    test_results(new_rslt(:,:), prev_rslt, 2e-2, 3.0e-1), &
    ' that Fourier has not changed')



  call close_module_test('asym_geo_fourier')
  call finish_mp

contains
  function test_results(rslt1, rslt2, epsin, bmagerror)
    real, dimension(:,:) :: rslt1, rslt2
    logical :: test_results
    real :: epsin, bmagerror
    test_results = .true.
    call announce_check('bmag')
    call process_check(test_results, &
      agrees_with(rslt1(:,1), rslt2(:,1), bmagerror), 'bmag')
    call announce_check('grho')
    call process_check(test_results, &
      agrees_with(rslt1(:,2), rslt2(:,2), bmagerror), 'grho')
    call announce_check('gbdrift')
    call process_check(test_results, &
      agrees_with(rslt1(:,3), rslt2(:,3), epsin), 'gbdrift')
    call announce_check('cvdrift')
    call process_check(test_results, &
      agrees_with(rslt1(:,4), rslt2(:,4), epsin), 'cvdrift')

  end function test_results
  subroutine get_new_results()
    call init_theta_grid
    new_rslt(:,1) = spl_fit(bmag)
    new_rslt(:,2) = spl_fit(grho)
    new_rslt(:,3) = spl_fit(gbdrift)
    new_rslt(:,4) = spl_fit(cvdrift)
    new_rslt(:,5) = spl_fit(gds2)
    new_rslt(:,6) = spl_fit(gds21)
    call finish_theta_grid
  end subroutine get_new_results

  function spl_fit(var)
    use splines
    type(spline) :: spl
    real, dimension(:), intent(in) :: var
    real, dimension(33) :: spl_fit
    integer :: i
    call new_spline(size(var), theta, var, spl)
    do i = 1,33
      spl_fit(i) = splint(prev_rslt(i,7), spl)
    end do
    call delete_spline(spl)
  end function spl_fit

  function prev_result()
    real, dimension (33,7) :: prev_result

prev_result = &
reshape([&
! Bmag
1.33243312394724 , &
1.31609038353426 , &
1.29944581732185 , &
1.27865665577025 , &
1.22904201792353 , &
1.17531996543297 , &
1.1212472336655 , &
1.09384040948333 , &
1.04452336063811 , &
1.01749312268327 , &
0.976370713227221 , &
0.948526753229513 , &
0.925435118566163 , &
0.901347548257888 , &
0.876578418704114 , &
0.857978934867862 , &
0.853728419084156 , &
0.857978934867862 , &
0.876578418704114 , &
0.901347548257888 , &
0.925435118566163 , &
0.948526753229513 , &
0.976370713227221 , &
1.01749312268327 , &
1.04452336063811 , &
1.09384040948333 , &
1.1212472336655 , &
1.17531996543297 , &
1.22904201792353 , &
1.27865665577025 , &
1.29944581732185 , &
1.31609038353426 , &
1.33243312394724 , &
! grho
0.671476783814217 , &
0.655376205516751 , &
0.645474330113914 , &
0.635755286635991 , &
0.621030571063152 , &
0.619021950832028 , &
0.636083048443971 , &
0.654858594504839 , &
0.715714466573281 , &
0.771688700482797 , &
0.906600342256315 , &
1.02597090961678 , &
1.082828342861 , &
1.0746905130402 , &
1.00349468546693 , &
0.889504009645021 , &
0.819130116239692 , &
0.773807436271998 , &
0.753383817344089 , &
0.767126347171711 , &
0.791641503483525 , &
0.818410486514247 , &
0.850385090496886 , &
0.890164163611287 , &
0.90760647432831 , &
0.914685797109559 , &
0.903898726229265 , &
0.859137751377334 , &
0.799324691492278 , &
0.74340860227985 , &
0.721335240987457 , &
0.703937480088525 , &
0.671476783814217 , &
! gbdrift
-1.02617555770194 , &
-0.712969498574763 , &
-0.561130408637568 , &
-0.42910219126836 , &
-0.212236954155456 , &
-0.0378366665373612 , &
0.107464840410952 , &
0.170645312002224 , &
0.260715418207449 , &
0.294306342460963 , &
0.31942079872758 , &
0.308963075229353 , &
0.317646184154479 , &
0.411506012588182 , &
0.535249631719891 , &
0.632120231695397 , &
0.68677368851193 , &
0.728859615295146 , &
0.760976125502635 , &
0.770619607724319 , &
0.772569161208991 , &
0.770190777155894 , &
0.7582240733142 , &
0.713492291896416 , &
0.662075666983065 , &
0.526883041142097 , &
0.437229953670605 , &
0.247544442774311 , &
0.0246414289692345 , &
-0.267563533410638 , &
-0.444357542010986 , &
-0.640990660582648 , &
-1.30694265090338 , &
! cvdrift
-0.99304581578849 , &
-0.678616124963569 , &
-0.525961319078489 , &
-0.392888822003733 , &
-0.173239727644619 , &
0.00465794461185634 , &
0.154146189210009 , &
0.219761393918476 , &
0.314909796811417 , &
0.351773129722504 , &
0.38283551178222 , &
0.377303474963089 , &
0.389796984068148 , &
0.486641864466124 , &
0.613536606015087 , &
0.71269402756357 , &
0.767473129194199 , &
0.808360704079882 , &
0.837013911151268 , &
0.842709853511131 , &
0.841197738958925 , &
0.835778792319529 , &
0.820448798914196 , &
0.771275884923043 , &
0.717228017107965 , &
0.577634722507131 , &
0.485644150080945 , &
0.291448182038186 , &
0.0644069116604236 , &
-0.231118184710186 , &
-0.409151176013497 , &
-0.606693163409686 , &
-1.27381290898992 , &
! gds2
11.5624499413368 , &
9.10607609760297 , &
7.91860938834958 , &
6.89382945882611 , &
5.27737016918049 , &
4.0947664808342 , &
3.1658609400591 , &
2.75001991337884 , &
2.06849031866514 , &
1.73223480461451 , &
1.27522131589324 , &
0.987673532589969 , &
0.795036544307944 , &
0.729446155982084 , &
0.777709753405467 , &
0.910616600855998 , &
1.04894421182027 , &
1.20807001096086 , &
1.44438064609281 , &
1.66531655498877 , &
1.87070263934217 , &
2.07606641561136 , &
2.33396748864275 , &
2.71181597311874 , &
2.93417937606753 , &
3.24053724492585 , &
3.36745414485372 , &
3.69140370361368 , &
4.40475246086213 , &
5.75266873518745 , &
6.68750088717724 , &
7.78610225773405 , &
11.6460729613819 , &
! gds21
2.48260433013266 , &
1.96980889550023 , &
1.7003076750408 , &
1.44851731630675 , &
1.00883477033495 , &
0.658427244546478 , &
0.370065566272634 , &
0.235316007596034 , &
-0.00830824712186361 , &
-0.152028403670999 , &
-0.375833045101806 , &
-0.408259296824652 , &
-0.202640762423129 , &
0.052466633011864 , &
0.138146124877443 , &
0.040099348214095 , &
-0.0745691140125738 , &
-0.200760227898734 , &
-0.400206438627727 , &
-0.601352319637693 , &
-0.784834496580205 , &
-0.956102451798717 , &
-1.15266885933104 , &
-1.40274694759805 , &
-1.52290744280547 , &
-1.61580703877306 , &
-1.60043993778815 , &
-1.51920214691934 , &
-1.51156870568341 , &
-1.6590686420898 , &
-1.78921816429184 , &
-1.9484052219476 , &
-2.49591334712911 , &
! theta
-3.14159265358979 , &
-2.91727670215663 , &
-2.79597844038702 , &
-2.67945583098899 , &
-2.46152917748859 , &
-2.26207356592901 , &
-2.06935425297297 , &
-1.96827958767276 , &
-1.76714586764426 , &
-1.63624617374468 , &
-1.37444678594553 , &
-1.11264739814639 , &
-0.850848010347236 , &
-0.589048622548086 , &
-0.327249234748937 , &
-0.0654498469497874 , &
0.110664600390518 , &
0.2771702192506 , &
0.494417439935701 , &
0.667312568579739 , &
0.800558607925003 , &
0.913804260054291 , &
1.04137270948477 , &
1.22325635606895 , &
1.34350356539704 , &
1.5707963267949 , &
1.70169602069447 , &
1.96349540849362 , &
2.22529479629277 , &
2.48709418409192 , &
2.61799387799149 , &
2.74889357189107 , &
3.14159265358979 &
], (/33,7/))
end function prev_result

end program test_asym_geo_fourier
