"""
Written by Sean Gordon, Aleksandar Jelenak, and Ted Habermann.
Based on the NOAA rubrics Dr Habermann created, and his work
conceptualizing the documentation language so that rubrics using
recommendations from other earth science communities can be applied
to multiple metadata dialects as a part of the USGeo BEDI and
NSF DIBBs projects. This python module as an outcome of DIBBs allows
a user to initiate an evaluation of valid XML.

The basic workflow is to retrieve records, evaluate for concept and xpath
content, run concept/xpath counts and occurrence functions on csv output of
evaluation, create collectionspreadsheet with the outputs, if you want to
compare between collections, combine csv outputs with appropriate combination
functions, create organizationSpreadsheet. Finally run WriteGoogleSheets on any
xlsx outputs you want to share.
"""


import pandas as pd
import csv
import gzip
import os
import requests
import xlsxwriter
from pydrive.auth import GoogleAuth
from pydrive.drive import GoogleDrive
from lxml import etree
import sys
import logging
from IPython.core.display import display, HTML

import itertools
from plotly import tools
import plotly.plotly
import plotly.graph_objs as go
import plotly.io as pio
from plotly.offline import iplot, init_notebook_mode

init_notebook_mode(connected=True)
from PIL import Image
Image.MAX_IMAGE_PIXELS = None

lggr = logging.getLogger(__name__)
csv.field_size_limit(sys.maxsize)


# function to download metadata


def get_records(urls, xml_files, well_formed=True):
    """Download metadata records. Metadata records are download from the
    supplied ``urls`` and stored in files whose names are found on
    ``xml_files``. When ``well_formed`` is ``True`` downloaded XML will
    be saved to a file only if well-formed.
    """
    """ if we used a function
    like this to collect xml, it would be the root of any processing steps
    """
    if len(urls) != len(xml_files):
        raise ValueError('Different number of URLs and record file names')

    for url, fname in zip(urls, xml_files):
        try:
            r = requests.get(url)
            r.raise_for_status()
            
        except Exception:
            print('There was an error downloading from {}'.format(url))

        if well_formed:
            try:
                etree.fromstring(r.text)
            except Exception:
                print('Metadata record from {} not well-formed'.format(url))

        if fname[-4:] != '.xml':
            fname += '.xml'

        with open(fname, 'wt') as f:
            f.write(r.text)


def recordXpathContent(EvaluatedMetadataDF):
    """requires a dataframe with elements. Creates a vertical view of
    concept content for each record in the collection. Useful in the
    creation of json.
    """
    EvaluatedMetadataDF = EvaluatedMetadataDF.applymap(str)

    group_name = EvaluatedMetadataDF.groupby([
        'Collection', 'Record', 'XPath'], as_index=False)
    occurrenceMatrix = group_name['Content'].apply(
        lambda x: '%s' % ', '.join(x)).unstack().reset_index()

    occurrenceMatrix.columns.names = ['']

    #FILLvalues = 'No Content'
    #occurrenceMatrix = occurrenceMatrix.fillna(value=FILLvalues)
    occurrenceMatrix.reset_index()

    return(occurrenceMatrix)


def XpathCounts(EvaluatedMetadataDF,
                DataDestination, to_csv=True):
    """XpathCounts requires a dataframe with xpath.The DF
    can created be localAllNodesEval, XMLeval(not accurate), or
    a simpleXpath. It is required for combineXpathCounts"""
    group_name = EvaluatedMetadataDF.groupby(
        ['Collection', 'Record', 'XPath'], as_index=False)
    XpathCountsDF = group_name.size().unstack().reset_index()
    XpathCountsDF = XpathCountsDF.fillna(0)
    pd.options.display.float_format = '{:,.0f}'.format

    if to_csv:
        lggr.info('Saving Xpath counts report to %s' % DataDestination)
        XpathCountsDF.to_csv(DataDestination, mode='w', index=False)

    return XpathCountsDF


def XpathOccurrence(EvaluatedMetadataDF, Collection,
                    DataDestination, to_csv=True):
    # xpath occurrence data product
    """requires a list of xpathOccurrence csv.
    It is required for CombinationSpreadsheet
    """
    DataDestinationDirectory = DataDestination[:DataDestination.rfind('/') + 1]
    os.makedirs(DataDestinationDirectory, exist_ok=True)
    group_name = EvaluatedMetadataDF.groupby(
        ['Record', 'XPath'], as_index=False)
    occurrenceMatrix = group_name.size().unstack().reset_index()
    occurrenceMatrix = occurrenceMatrix.fillna(0)
    occurrenceSum = occurrenceMatrix.sum()
    occurrenceCount = occurrenceMatrix[occurrenceMatrix != 0].count()

    result = pd.concat([occurrenceSum, occurrenceCount], axis=1).reset_index()
    result.insert(
        1, 'Collection', Collection)
    result.insert(4, 'CollectionOccurrence%', Collection)
    result.insert(4, 'AverageOccurrencePerRecord', Collection)
    result.columns = [
        'XPath', 'Collection', 'XPathCount', 'RecordCount',
        'AverageOccurrencePerRecord', 'CollectionOccurrence%'
    ]
    NumberOfRecords = result.at[0, 'XPathCount'].count('.xml')
    result['CollectionOccurrence%'] = result['RecordCount'] / NumberOfRecords
    result.at[0, 'XPathCount'] = NumberOfRecords
    result.at[0, 'XPath'] = 'Number of Records'
    result.at[0, 'CollectionOccurrence%'] = NumberOfRecords
    result['AverageOccurrencePerRecord'] = (
        result['XPathCount'] / NumberOfRecords)
    result[['AverageOccurrencePerRecord', 'CollectionOccurrence%']] = (
        result[['AverageOccurrencePerRecord',
                'CollectionOccurrence%']].astype(float)
    )
    result[["XPathCount", "RecordCount"]] = (
        result[["XPathCount", "RecordCount"]].astype(int)
    )
    result['AverageOccurrencePerRecord'] = (pd.Series([
        "{0:.2f}".format(val) for val in result['AverageOccurrencePerRecord']
    ], index=result.index))
    result.at[0, 'AverageOccurrencePerRecord'] = NumberOfRecords

    if to_csv:
        lggr.info('Saving XPath occurrence report to %s' % DataDestination)
        result.to_csv(DataDestination, mode='w', index=False)

    return result


def CombineXPathOccurrence(CollectionComparisons,
                           DataDestination, to_csv=True):
    """Using xpath occurrence data products, combine them and produce a
    collection occurrence% table with collections for columns and
    concepts for rows requires a list of xpathOccurrence csv.
    It is required for CombinationSpreadsheet
    """
    DataDestinationDirectory = DataDestination[:DataDestination.rfind('/') + 1]
    os.makedirs(DataDestinationDirectory, exist_ok=True)
    CombinedDF = pd.concat((pd.read_csv(f) for f in CollectionComparisons))
    CombinedPivotDF = CombinedDF.pivot(
        index='XPath', columns='Collection', values='CollectionOccurrence%')

    ConceptCountsDF = CombinedPivotDF.fillna(0)
    ConceptCountsDF.columns.names = ['']
    ConceptCountsDF = ConceptCountsDF.reset_index()

    line = ConceptCountsDF[ConceptCountsDF['XPath'] == 'Number of Records']
    ConceptCountsDF = pd.concat(
        [ConceptCountsDF[0:0], line,
         ConceptCountsDF[0:]]).reset_index(drop=True)
    ConceptCountsDF.drop(
        ConceptCountsDF.tail(1).index, inplace=True)

    if to_csv:
        lggr.info('Saving concept count report to %s' % DataDestination)
        ConceptCountsDF.to_csv(DataDestination, mode='w', index=False)

    return ConceptCountsDF


def CombineXPathCounts(CollectionComparisons, DataDestination, to_csv=True):
    """Using xpath occurrence data products, combine them and produce a
    record count table with collections for columns and concepts for rows
    requires a list of xpath counts csv. It is required for
    CombinationSpreadsheet
    """
    DataDestinationDirectory = DataDestination[:DataDestination.rfind('/') + 1]
    os.makedirs(DataDestinationDirectory, exist_ok=True)
    XPathCountCombinedDF = pd.concat(
        (pd.read_csv(f) for f in CollectionComparisons),
        axis=0, ignore_index=True, sort=True)
    XPathCountCombinedDF = XPathCountCombinedDF.fillna(0)
    XPathCountCombinedDF.columns.names = ['']
    # get a list of columns
    cols = list(XPathCountCombinedDF)
    # move the column to head of list using index, pop and insert
    cols.insert(0, cols.pop(cols.index('Record')))
    # use ix to reorder
    CombinedXPathCountsDF = XPathCountCombinedDF.loc[:, cols]
    cols2 = list(CombinedXPathCountsDF)
    # move the column to head of list using index, pop and insert
    cols2.insert(0, cols2.pop(cols.index('Collection')))
    # use ix to reorder
    CombinedXPathCountsDF = CombinedXPathCountsDF.loc[:, cols2]
    CombinedXPathCountsDF

    if to_csv:
        lggr.info('Saving record count report to %s' % DataDestination)
        CombinedXPathCountsDF.to_csv(DataDestination, mode='w', index=False)

    return CombinedXPathCountsDF


def CombineAverageXPathOccurrencePerRecord(CollectionComparisons,
                                           DataDestination,
                                           to_csv=True):
    """Using concept occurrence data products, combine them
    and produce a record count table with collections for columns and
    concepts for rows. It is required for CombinationSpreadsheet.
    """
    DataDestinationDirectory = DataDestination[:DataDestination.rfind('/') + 1]
    os.makedirs(DataDestinationDirectory, exist_ok=True)
    CombinedDF = pd.concat((pd.read_csv(f) for f in CollectionComparisons))
    CombinedPivotDF = CombinedDF.pivot(
        index='XPath', columns='Collection',
        values='AverageOccurrencePerRecord')

    pd.options.display.float_format = '{:,.0f}'.format
    ConceptCountsDF = CombinedPivotDF.fillna(0)
    ConceptCountsDF.columns.names = ['']
    ConceptCountsDF = ConceptCountsDF.reset_index()
    line = ConceptCountsDF[ConceptCountsDF['XPath'] == 'Number of Records']
    ConceptCountsDF = pd.concat(
        [ConceptCountsDF[0:0], line,
         ConceptCountsDF[0:]]).reset_index(drop=True)
    ConceptCountsDF.drop(
        ConceptCountsDF.tail(1).index, inplace=True)

    if to_csv:
        lggr.info('Saving CombineAverageXPathOccurrencePerRecord to %s'
                  % DataDestination)
        ConceptCountsDF.to_csv(DataDestination, mode='w', index=False)
    return ConceptCountsDF


def collectionSpreadsheet(EvaluatedRecommendations, EvaluatedXpaths,
                          xpathOccurrence, xpathCounts,
                          recommendationOccurrence, recommendationCounts,
                          DataDestination):
    # create spreadsheet for an collection
    """requires xpath and concept occurrence,
    as well as the concept counts csv for a collection
    """

    workbook = xlsxwriter.Workbook(
        DataDestination,
        {'strings_to_numbers': True, 'strings_to_urls': False})
    cell_format11 = workbook.add_format()
    cell_format11.set_num_format('0%')
    cell_format04 = workbook.add_format()
    cell_format04.set_num_format('0')
    cell_format05 = workbook.add_format()
    cell_format05.set_num_format('0.00')
    formatGreen = workbook.add_format({'bg_color': '#C6EFCE',
                                       'font_color': '#006100'})
    formatRed = workbook.add_format({'bg_color': '#FFC7CE',
                                     'font_color': '#9C0006'})
    formatYellow = workbook.add_format({'bg_color': '#FFEB9C',
                                        'font_color': '#9C6500'})

    RecommendationOccurrence = workbook.add_worksheet(
        'RecommendationOccurrence')
    RecommendationOccurrence.set_column('A:A', 100)
    RecommendationOccurrence.set_column('C:D', 20)
    RecommendationOccurrence.set_column('E:E', 30)
    RecommendationOccurrence.set_column('F:F', 25, cell_format11)
    RecommendationOccurrence.set_column('B:B', 30)

    RecommendationCounts = workbook.add_worksheet('RecommendationCounts')
    RecommendationCounts.set_column('A:OD', 30)

    RecommendationContent = workbook.add_worksheet('RecommendationContent')
    RecommendationContent.set_column('A:B', 30)
    RecommendationContent.set_column('C:C', 100)
    RecommendationContent.set_column('D:D', 20)
    XpathOccurrence = workbook.add_worksheet('XpathOccurrence')
    XpathOccurrence.set_column('A:A', 100)
    XpathOccurrence.set_column('C:D', 20)
    XpathOccurrence.set_column('E:E', 30)
    XpathOccurrence.set_column('F:F', 25, cell_format11)
    XpathOccurrence.set_column('B:B', 30)

    XpathCounts = workbook.add_worksheet('XpathCounts')
    XpathCounts.set_column('A:OD', 30)

    XpathContent = workbook.add_worksheet('XpathContent')
    XpathContent.set_column('A:B', 30)
    XpathContent.set_column('C:C', 100)
    XpathContent.set_column('D:D', 20)

    Reader = csv.reader(gzip.open(
        EvaluatedXpaths, "rt"), delimiter=',', quotechar='"')
    row_count = 0

    for row in Reader:
        for col in range(len(row)):
            XpathContent.write(row_count, col, row[col])
        row_count += 1
    Reader = csv.reader(
        gzip.open(EvaluatedXpaths, "rt"), delimiter=',', quotechar='"')
    row_count = 0

    absRowCount = sum(1 for row in Reader)
    XpathContent.autofilter(0, 0, absRowCount - 1, 3)

    Reader = csv.reader(open(
        xpathOccurrence, 'r'), delimiter=',', quotechar='"')
    row_count = 0
    for row in Reader:
        for col in range(len(row)):
            XpathOccurrence.write(row_count, col, row[col])
        row_count += 1
    Reader = csv.reader(open(
        xpathOccurrence, 'r'), delimiter=',', quotechar='"')
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"'
    )))
    XpathOccurrence.autofilter(0, 0, absRowCount - 1, 5)
    XpathOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})

    XpathOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})

    XpathOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})

    Reader = csv.reader(open(
        xpathCounts, 'r'), delimiter=',', quotechar='"'
    )
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(xpathCounts, 'r'), delimiter=',', quotechar='"'
    )))
    XpathCounts.autofilter(0, 0, absRowCount - 1, absColCount - 1)
    XpathCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})

    XpathCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})
    XpathCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})

    Reader = csv.reader(open(
        xpathCounts, 'r'), delimiter=',', quotechar='"')
    row_count = 0

    for row in Reader:
        for col in range(len(row)):
            XpathCounts.write(row_count, col, row[col])

        row_count += 1

    # rec

    Reader = csv.reader(gzip.open(EvaluatedRecommendations, "rt"), delimiter=',', quotechar='"')
    row_count = 0

    for row in Reader:
        for col in range(len(row)):
            RecommendationContent.write(row_count, col, row[col])
        row_count += 1
    Reader = csv.reader(gzip.open(EvaluatedRecommendations, "rt"), delimiter=',', quotechar='"')
    row_count = 0

    absRowCount = sum(1 for row in Reader)
    RecommendationContent.autofilter(0, 0, absRowCount - 1, 3)

    Reader = csv.reader(open(
        recommendationOccurrence, 'r'), delimiter=',', quotechar='"')
    row_count = 0
    for row in Reader:
        for col in range(len(row)):
            RecommendationOccurrence.write(row_count, col, row[col])
        row_count += 1
    Reader = csv.reader(open(
        recommendationOccurrence, 'r'), delimiter=',', quotechar='"')
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"'
    )))
    RecommendationOccurrence.autofilter(0, 0, absRowCount - 1, 5)
    RecommendationOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})

    RecommendationOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})

    RecommendationOccurrence.conditional_format(
        2, 5, absRowCount - 1, 5,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})

    Reader = csv.reader(open(
        recommendationCounts, 'r'), delimiter=',', quotechar='"'
    )
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(recommendationCounts, 'r'), delimiter=',', quotechar='"'
    )))
    RecommendationCounts.autofilter(0, 0, absRowCount - 1, absColCount - 1)
    RecommendationCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})

    RecommendationCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})
    RecommendationCounts.conditional_format(
        1, 2, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})

    Reader = csv.reader(open(
        recommendationCounts, 'r'), delimiter=',', quotechar='"')
    row_count = 0

    for row in Reader:
        for col in range(len(row)):
            RecommendationCounts.write(row_count, col, row[col])

        row_count += 1
    # rec
    workbook.close()


def CombinationSpreadsheet(xpathOccurrence, recommendationOccurrence,
                           RecommendationConcept, RecommendationGraph,
                           RecGraphLink,
                           DataDestination, AVGxpathOccurrence=None,
                           AVGrecommendationOccurrence=None,
                           recommendationCounts=None, xpathCounts=None):
    # create spreadsheet for an organization
    """requires each xpath and concept occurrence,
    csv for a organization
    (or any group of collections you want to compare)
    """

    lggr.info('Saving spreadsheet %s' % DataDestination)
    workbook = xlsxwriter.Workbook(DataDestination,
                                   {'strings_to_numbers': True})
    workbook.use_zip64()
    cell_format11 = workbook.add_format()
    cell_format11.set_num_format('0%')

    cell_format04 = workbook.add_format()
    cell_format04.set_num_format('0')
    cell_format05 = workbook.add_format()
    cell_format05.set_num_format('0.00')

    formatGreen = workbook.add_format(
        {'bg_color': '#C6EFCE', 'font_color': '#006100'})
    formatRed = workbook.add_format(
        {'bg_color': '#FFC7CE', 'font_color': '#9C0006'})
    formatYellow = workbook.add_format(
        {'bg_color': '#FFEB9C', 'font_color': '#9C6500'})
    RecommendationConceptWS = workbook.add_worksheet(
        'RecommendationConcepts')
    # RecommendationGraphWS = workbook.add_worksheet(
    #    'RecommendationGraph')
    # Insert an image with scaling.
    RecommendationConceptWS.write('A29', "Full Image")
    RecommendationConceptWS.write('B29', RecGraphLink)
    RecommendationConceptWS.insert_image('A30', RecommendationGraph, {'x_scale': .07, 'y_scale': .07})

    Reader = csv.reader(
        open(RecommendationConcept, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    RecommendationConceptWS.set_row(0, None, cell_format04)
    RecommendationConceptWS.set_row(2, None, cell_format04)
    
    for row in Reader:
        for col in range(len(row)):
            RecommendationConceptWS.write(row_count, col, row[col])
            RecommendationConceptWS.set_column(col, col, 7, cell_format11)
        row_count += 1
    RecommendationConceptWS.set_column(0, 0, 20)
    RecommendationConceptWS.set_column(1, 1, 15)
    RecommendationConceptWS.set_column(2, 2, 20)
    RecommendationAnalysisWS = workbook.add_worksheet(
        'RecommendationElements')
    RecommendationAnalysisWS.set_column(2, 4, 12)
    recommendationoccurrenceWS = workbook.add_worksheet(
        'RecommendationOccurrence')
    avgRecommendationOccurWS = workbook.add_worksheet(
            'AVGrecommendationOccurrence')
    if recommendationCounts is not None:
        recommendationcounts = workbook.add_worksheet('RecommendationCounts')

    XpathAnalysisWS = workbook.add_worksheet('AllXpaths')
    xpathoccurrenceWS = workbook.add_worksheet('XpathOccurrence')
    avgXpathOccurWS = workbook.add_worksheet('AVGxpathOccurrence')
    if xpathCounts is not None:
        xpathcounts = workbook.add_worksheet('XpathCounts')
    XpathAnalysisWS.set_column('A:A', 70)
    XpathAnalysisWS.set_column('B:B', 20)
    recommendationoccurrenceWS.hide()
    xpathoccurrenceWS.hide()
    avgXpathOccurWS.hide()
    avgRecommendationOccurWS.hide()
    xpathoccurrenceWS.set_column('A:A', 70)

    Reader = csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    xpathoccurrenceWS.set_row(1, None, cell_format04)
    for row in Reader:
        for col in range(len(row)):
            xpathoccurrenceWS.write(row_count, col, row[col])
            xpathoccurrenceWS.set_column(col, col, 15, cell_format11)
        row_count += 1

    Reader = csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    for row in Reader:
        if Reader.line_num != 2:
            for col in range(1, len(row)):
                XpathAnalysisWS.write(
                    row_count + 9, col + 4, row[col], cell_format11
                )

            for col in range(0, 1):
                XpathAnalysisWS.write(row_count + 9, col, row[col], cell_format11)

                Xpathcell = xlsxwriter.utility.xl_rowcol_to_cell(row_count + 9, 0)
                formulaElementSimplifier = (
                    '=MID(' + Xpathcell +
                    ',1+FIND("|",SUBSTITUTE(' + Xpathcell +
                    ',"/","|",LEN(' + Xpathcell + ')-LEN(SUBSTITUTE(' +
                    Xpathcell + ',"/","")))),100)'
                )
                XpathAnalysisWS.write(
                    row_count + 9, col + 1, formulaElementSimplifier, cell_format11
                )
            row_count += 1
    if AVGxpathOccurrence is not None:
        avgXpathOccurWS.set_column('A:A', 70)

        Reader = csv.reader(
        open(AVGxpathOccurrence, 'r'), delimiter=',', quotechar='"')
    row_count = 0
    avgXpathOccurWS.set_row(1, None, cell_format04)
    for row in Reader:
        for col in range(len(row)):
            avgXpathOccurWS.write(row_count, col, row[col])
            avgXpathOccurWS.set_column(col, col, 15, cell_format05)

        for col in range(len(row) - 1):

            cell2 = xlsxwriter.utility.xl_rowcol_to_cell(0, col + 1)
            cell3 = xlsxwriter.utility.xl_rowcol_to_cell(2, col + 5)
            colRange = xlsxwriter.utility.xl_range(2, col + 1, 5000, col + 1)
            colRange2 = xlsxwriter.utility.xl_range(2, 5, 2, len(row) + 3)

            formula2 = '=COUNTIF(xpathOccurrence!' + colRange + ',">"&0)'
            XpathAnalysisWS.write(2, col + 5, formula2)

            formula3 = (
                '=' + cell3 + '/COUNTA(xpathOccurrence!' + colRange + ')')
            XpathAnalysisWS.write(3, col + 5, formula3, cell_format11)

            formula4 = '=SUM(xpathOccurrence!' + colRange + ')/' + '%s' % cell3
            XpathAnalysisWS.write(4, col + 5, formula4, cell_format11)

            formula5 = '=' + '%s' % cell3 + '/MAX(' + colRange2 + ')'
            XpathAnalysisWS.write(5, col + 5, formula5, cell_format11)

            formula6 = (
                '=COUNTIF(xpathOccurrence!' +
                colRange + ',">="&1)/' + '%s' % cell3
            )
            XpathAnalysisWS.write(6, col + 5, formula6, cell_format11)

            formula7 = (
                '=COUNTIFS(xpathOccurrence!' +
                colRange + ',">"&0,xpathOccurrence!' +
                colRange + ',"<"&1)/' + '%s' % cell3
            )
            XpathAnalysisWS.write(7, col + 5, formula7, cell_format11)

            formula1 = (
                '=VLOOKUP("Number of Records",xpathOccurrence!1:1048576,' +
                str(col + 2) + ', False)'
            )
            XpathAnalysisWS.write(1, col + 5, formula1, cell_format04)

            cell2 = xlsxwriter.utility.xl_rowcol_to_cell(0, col + 1)

            formula = '=xpathOccurrence!' + '%s' % cell2
            XpathAnalysisWS.write(0, col + 5, formula)
            dateFormula = (
                '=LEFT(RIGHT(xpathOccurrence!' + '%s' % cell2 +
                ',LEN(xpathOccurrence!' + '%s' % cell2 +
                ')-FIND("_", xpathOccurrence!' +
                '%s' % cell2 + ')-1),FIND("_",xpathOccurrence!' +
                '%s' % cell2 + ')+1)'
            )
            XpathAnalysisWS.write(8, col + 5, dateFormula)
            collectFormula = (
                '=LEFT(xpathOccurrence!' + '%s' % cell2 +
                ',FIND("_",xpathOccurrence!' + '%s' % cell2 + ')-1)'
            )

            XpathAnalysisWS.write(9, col + 5, collectFormula)

        row_count += 1
    #######################################################################

    if xpathCounts is not None:
        Reader = csv.reader(
            open(xpathCounts, 'r'), delimiter=',', quotechar='"')
        row_count = 0

        for row in Reader:
            for col in range(len(row)):
                xpathcounts.write(row_count, col, row[col], cell_format04)
            row_count += 1
        Reader = csv.reader(
            open(xpathCounts, 'r'), delimiter=',', quotechar='"')
        row_count = 0
        absRowCount = sum(1 for row in Reader)
        absColCount = len(next(csv.reader(
            open(xpathCounts, 'r'), delimiter=',', quotechar='"')))
        xpathcounts.autofilter(0, 0, absRowCount - 1, absColCount - 1)

    XpathAnalysisWS.write('A2', 'Number of Records')
    XpathAnalysisWS.write('A3', 'Number of Elements / Attributes')
    XpathAnalysisWS.write(
        'A4',
        'Coverage w/r to Repository (CR): \
     number of elements / total number of elements'
    )
    XpathAnalysisWS.write('A5', 'Average Occurrence Rate')
    XpathAnalysisWS.write('A6', 'Repository Completeness: Number of elements \
    / number of elements in most complete collection in repository')
    XpathAnalysisWS.write('A7', 'Homogeneity: Number >= 1 \
    / Total Number of elements in the collection')
    XpathAnalysisWS.write('A8', 'Partial Elements: Number < 0 and < 1')
    XpathAnalysisWS.write('A9', 'Retrieval Date')
    XpathAnalysisWS.write('B1', 'Formulas')
    XpathAnalysisWS.write('C1', 'MIN')
    XpathAnalysisWS.write('D1', 'MAX')
    XpathAnalysisWS.write('E1', 'AVG')
    XpathAnalysisWS.write('B10', 'Element Name')
    XpathAnalysisWS.write('C10', 'Collections')
    XpathAnalysisWS.write('D10', 'Complete')
    XpathAnalysisWS.write('E10', 'Partial')

    for row in range(1, 3):
        absColCount = len(next(csv.reader(
            open(xpathOccurrence, 'r'), delimiter=',', quotechar='"'
        )))
        colRange4 = xlsxwriter.utility.xl_range(row, 5, row, 3 + absColCount)
        miniFormula = '=MIN(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 2, miniFormula, cell_format04)
        maxiFormula = '=MAX(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 3, maxiFormula, cell_format04)
        avgFormula = '=AVERAGE(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 4, avgFormula, cell_format04)

    for row in range(3, 8):
        absColCount = len(next(csv.reader(
            open(xpathOccurrence, 'r'), delimiter=',', quotechar='"'
        )))
        colRange4 = xlsxwriter.utility.xl_range(row, 5, row, 3 + absColCount)
        miniFormula = '=MIN(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 2, miniFormula, cell_format11)
        maxiFormula = '=MAX(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 3, maxiFormula, cell_format11)
        avgFormula = '=AVERAGE(' + colRange4 + ')'
        XpathAnalysisWS.write(row, 4, avgFormula, cell_format11)

    Reader = csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"'
    )
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"'
    )))

    XpathAnalysisWS.autofilter(9, 0, absRowCount + 7, absColCount + 3)
    xpathoccurrenceWS.autofilter(0, 0, absRowCount - 2, absColCount - 1)
    avgXpathOccurWS.autofilter(0, 0, absRowCount - 2, absColCount - 1)

    XpathAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount +
        3,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    XpathAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount + 3,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    XpathAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount + 3,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )
    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )

    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    xpathoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )


    avgXpathOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})
    avgXpathOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})
    avgXpathOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})
    for row in range(10, absRowCount + 8):
        colRange5 = xlsxwriter.utility.xl_range(row, 5, row, absColCount + 3)
        numbCollectFormula = '=COUNTIF(' + colRange5 + ',">"&0)'
        CompleteCollectFormula = '=COUNTIF(' + colRange5 + ',"="&1)'
        GreatCollectFormula = '=COUNTIF(' + colRange5 + ',"<"&1)-COUNTIF('+ colRange5 + ',"=0")'
        XpathAnalysisWS.write(row, 2, numbCollectFormula)
        XpathAnalysisWS.write(row, 3, CompleteCollectFormula)
        XpathAnalysisWS.write(row, 4, GreatCollectFormula)

    #######################################################################
    Reader = csv.reader(
        open(xpathOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    for row in Reader:
        for col in range(len(row) - 1):

            cell2 = xlsxwriter.utility.xl_rowcol_to_cell(0, col + 1)
            cell3 = xlsxwriter.utility.xl_rowcol_to_cell(2, col + 5)
            colRange = xlsxwriter.utility.xl_range(2, col + 1, 5000, col + 1)
            colRange2 = xlsxwriter.utility.xl_range(2, 5, 2, len(row) + 3)

            formula2 = '=COUNTIF(xpathOccurrence!' + colRange + ',">"&0)'
            XpathAnalysisWS.write(2, col + 5, formula2)

            formula3 = (
                '=' + cell3 + '/COUNTA(xpathOccurrence!' + colRange + ')')
            XpathAnalysisWS.write(3, col + 5, formula3, cell_format11)

            formula4 = '=SUM(xpathOccurrence!' + colRange + ')/' + '%s' % cell3
            XpathAnalysisWS.write(4, col + 5, formula4, cell_format11)

            formula5 = '=' + '%s' % cell3 + '/MAX(' + colRange2 + ')'
            XpathAnalysisWS.write(5, col + 5, formula5, cell_format11)

            formula6 = (
                '=COUNTIF(xpathOccurrence!' +
                colRange + ',">="&1)/' + '%s' % cell3
            )
            XpathAnalysisWS.write(6, col + 5, formula6, cell_format11)

            formula7 = (
                '=COUNTIFS(xpathOccurrence!' +
                colRange + ',">"&0,xpathOccurrence!' +
                colRange + ',"<"&1)/' + '%s' % cell3
            )
            XpathAnalysisWS.write(7, col + 5, formula7, cell_format11)

            formula1 = (
                '=VLOOKUP("Number of Records",xpathOccurrence!1:1048576,' +
                str(col + 2) + ', False)'
            )
            XpathAnalysisWS.write(1, col + 5, formula1, cell_format04)

            cell2 = xlsxwriter.utility.xl_rowcol_to_cell(0, col + 1)

            formula = '=xpathOccurrence!' + '%s' % cell2
            XpathAnalysisWS.write(0, col + 5, formula)
            dateFormula = (
                '=LEFT(RIGHT(xpathOccurrence!' + '%s' % cell2 +
                ',LEN(xpathOccurrence!' + '%s' % cell2 +
                ')-FIND("_", xpathOccurrence!' +
                '%s' % cell2 + ')-1),FIND("__",xpathOccurrence!' +
                '%s' % cell2 + ')+1)'
            )
            XpathAnalysisWS.write(8, col + 5, dateFormula)
            collectFormula = (
                '=LEFT(xpathOccurrence!' + '%s' % cell2 +
                ',FIND("_",xpathOccurrence!' + '%s' % cell2 + ')-1)'
            )

            XpathAnalysisWS.write(9, col + 5, collectFormula)
    #######################################################################

    RecommendationAnalysisWS.set_column('A:A', 70)
    RecommendationAnalysisWS.set_column('B:B', 20)

    recommendationoccurrenceWS.set_column('A:A', 70)

    Reader = csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    recommendationoccurrenceWS.set_row(1, None, cell_format04)
    for row in Reader:
        for col in range(len(row)):
            recommendationoccurrenceWS.write(
                row_count, col, row[col])

            recommendationoccurrenceWS.set_column(
                col, col, 15, cell_format11)
        row_count += 1
    Reader = csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    for row in Reader:
        if Reader.line_num != 2:
            for col in range(1, len(row)):
                RecommendationAnalysisWS.write(
                    row_count + 9, col + 4, row[col], cell_format11
                )

            for col in range(0, 1):
                RecommendationAnalysisWS.write(
                    row_count + 9, col, row[col], cell_format11)

                Recommendationcell = xlsxwriter.utility.xl_rowcol_to_cell(
                    row_count + 9, 0)
                formulaElementSimplifier = (
                    '=MID(' + Recommendationcell +
                    ',1+FIND("|",SUBSTITUTE(' + Recommendationcell +
                    ',"/","|",LEN(' + Recommendationcell + ')-LEN(SUBSTITUTE(' +
                    Recommendationcell + ',"/","")))),100)'
                )
                RecommendationAnalysisWS.write(
                    row_count + 9, col + 1, formulaElementSimplifier, cell_format11
                )
            row_count += 1

    avgRecommendationOccurWS.set_column('A:A', 70)

    Reader = csv.reader(
        open(AVGrecommendationOccurrence, 'r'), delimiter=',', quotechar='"')
    row_count = 0
    avgRecommendationOccurWS.set_row(1, None, cell_format04)
    for row in Reader:
        for col in range(len(row)):
            avgRecommendationOccurWS.write(
                row_count, col, row[col])
            avgRecommendationOccurWS.set_column(col, col, 15, cell_format05)
    RecommendationAnalysisWS.write('A2', 'Number of records')
    RecommendationAnalysisWS.write('A3', 'Number of elements')
    RecommendationAnalysisWS.write(
        'A4',
        'Number of recommendation elements'
    )
    RecommendationAnalysisWS.write('A5', 'Recommendation focus')
    RecommendationAnalysisWS.write('A6', 'Complete elements in the collection')
    RecommendationAnalysisWS.write('A7', 'Complete recommendation elements in the collection')
    RecommendationAnalysisWS.write(
        'A8', 'Recommendation completeness focus')
    RecommendationAnalysisWS.write('A9', 'Retrieval Date')
    RecommendationAnalysisWS.write('B1', 'Formulas')
    RecommendationAnalysisWS.write('C1', 'MIN')
    RecommendationAnalysisWS.write('D1', 'MAX')
    RecommendationAnalysisWS.write('E1', 'AVG')
    RecommendationAnalysisWS.write('B10', 'Element Name')
    RecommendationAnalysisWS.write('C10', 'Collections')
    RecommendationAnalysisWS.write('D10', 'Complete')
    RecommendationAnalysisWS.write('E10', 'Partial')

    Reader = csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"')

    row_count = 0
    for row in Reader:
        for col in range(len(row) - 1):
            ElementTotal = xlsxwriter.utility.xl_rowcol_to_cell(5, col + 5)
            RecommendationElementTotal = xlsxwriter.utility.xl_rowcol_to_cell(6, col + 5)
            cell2 = xlsxwriter.utility.xl_rowcol_to_cell(0, col + 1)
            cell3 = xlsxwriter.utility.xl_rowcol_to_cell(2, col + 5)
            cell4 = xlsxwriter.utility.xl_rowcol_to_cell(3, col + 5)
            colRange = xlsxwriter.utility.xl_range(2, col + 1, 5000, col + 1)
            colRange2 = xlsxwriter.utility.xl_range(2, 5, 2, len(row) + 3)

            formula2 = '=COUNTIF(XpathOccurrence!' + colRange + ',">"&0)'
            RecommendationAnalysisWS.write(2, col + 5, formula2)

            formula3 = '=COUNTIF(recommendationOccurrence!' + colRange + ',">"&0)'
            RecommendationAnalysisWS.write(3, col + 5, formula3)

            formula4 = '='+cell4+'/'+cell3
            RecommendationAnalysisWS.write(4, col + 5, formula4, cell_format11)

            formula5 = '=COUNTIF(XpathOccurrence!' + colRange + ',"=1")/'+cell3 
            RecommendationAnalysisWS.write(5, col + 5, formula5, cell_format11)

            formula6 = '=COUNTIF(RecommendationOccurrence!' + colRange + ',"=1")/'+cell3
            RecommendationAnalysisWS.write(6, col + 5, formula6, cell_format11)

            formula7 = '='+RecommendationElementTotal+'/'+ElementTotal
            RecommendationAnalysisWS.write(7, col + 5, formula7, cell_format11)

            formula1 = (
                '=VLOOKUP("Number of Records",recommendationOccurrence!1:1048576,' +
                str(col + 2) + ', False)'
            )
            RecommendationAnalysisWS.write(1, col + 5, formula1, cell_format04)

            formula = '=recommendationOccurrence!' + '%s' % cell2
            RecommendationAnalysisWS.write(0, col + 5, formula)
            dateFormula = (
                '=LEFT(RIGHT(recommendationOccurrence!' + '%s' % cell2 +
                ',LEN(recommendationOccurrence!' + '%s' % cell2 +
                ')-FIND("_", recommendationOccurrence!' +
                '%s' % cell2 + ')-1),FIND("_",recommendationOccurrence!' +
                '%s' % cell2 + ')+1)'
            )
            RecommendationAnalysisWS.write(8, col + 5, dateFormula)
            collectFormula = (
                '=LEFT(recommendationOccurrence!' + '%s' % cell2 +
                ',FIND("_",recommendationOccurrence!' + '%s' % cell2 + ')-1)'
            )

            RecommendationAnalysisWS.write(9, col + 5, collectFormula)

        row_count += 1
    #######################################################################

    if recommendationCounts is not None:
        Reader = csv.reader(
            open(recommendationCounts, 'r'), delimiter=',', quotechar='"')
        row_count = 0

        for row in Reader:
            for col in range(len(row)):
                recommendationcounts.write(
                    row_count, col, row[col], cell_format04)
            row_count += 1
        Reader = csv.reader(
            open(recommendationCounts, 'r'), delimiter=',', quotechar='"')
        row_count = 0
        absRowCount = sum(1 for row in Reader)
        absColCount = len(next(csv.reader(
            open(recommendationCounts, 'r'), delimiter=',', quotechar='"')))
        recommendationcounts.autofilter(0, 0, absRowCount - 1, absColCount - 1)

    
    for row in range(1, 4):
        absColCount = len(next(csv.reader(
            open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"'
        )))
        colRange4 = xlsxwriter.utility.xl_range(row, 5, row, 3 + absColCount)
        miniFormula = '=MIN(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 2, miniFormula, cell_format04)
        maxiFormula = '=MAX(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 3, maxiFormula, cell_format04)
        avgFormula = '=AVERAGE(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 4, avgFormula, cell_format04)

    for row in range(4, 8):
        absColCount = len(next(csv.reader(
            open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"'
        )))
        colRange4 = xlsxwriter.utility.xl_range(row, 5, row, 3 + absColCount)
        miniFormula = '=MIN(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 2, miniFormula, cell_format11)
        maxiFormula = '=MAX(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 3, maxiFormula, cell_format11)
        avgFormula = '=AVERAGE(' + colRange4 + ')'
        RecommendationAnalysisWS.write(row, 4, avgFormula, cell_format11)

    Reader = csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"'
    )
    absRowCount = sum(1 for row in Reader)
    absColCount = len(next(csv.reader(
        open(recommendationOccurrence, 'r'), delimiter=',', quotechar='"'
    )))

    RecommendationAnalysisWS.autofilter(9, 0, absRowCount + 7, absColCount + 3)
    recommendationoccurrenceWS.autofilter(
        0, 0, absRowCount - 2, absColCount - 1)
    avgRecommendationOccurWS.autofilter(0, 0, absRowCount - 2, absColCount - 1)

    RecommendationAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount +
        3,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    RecommendationAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount + 3,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    RecommendationAnalysisWS.conditional_format(
        10, 5, absRowCount + 8, absColCount + 3,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )
    recommendationoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    recommendationoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    recommendationoccurrenceWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )
    avgRecommendationOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen})
    avgRecommendationOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow})
    avgRecommendationOccurWS.conditional_format(
        2, 1, absRowCount - 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed})

    RecommendationConceptWS.conditional_format(
        3, 3, 28, absColCount + 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    RecommendationConceptWS.conditional_format(
        3, 3, 28, absColCount + 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    RecommendationConceptWS.conditional_format(
        3, 3, 28, absColCount + 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )
    RecommendationConceptWS.conditional_format(
        1, 3, 1, absColCount - 1,
        {'type': 'cell', 'criteria': '>=', 'value': 1, 'format': formatGreen}
    )
    RecommendationConceptWS.conditional_format(
        1, 3, 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': 0, 'format': formatYellow}
    )
    RecommendationConceptWS.conditional_format(
        1, 3, 1, absColCount - 1,
        {'type': 'cell', 'criteria': '=', 'value': -1, 'format': formatRed}
    )
    for row in range(10, absRowCount + 8):
        colRange5 = xlsxwriter.utility.xl_range(row, 5, row, absColCount + 3)
        numbCollectFormula = '=COUNTIF(' + colRange5 + ',">"&0)'
        CompleteCollectFormula = '=COUNTIF(' + colRange5 + ',"="&1)'
        GreatCollectFormula = '=COUNTIF(' + colRange5 + ',"<"&1)-COUNTIF('+ colRange5 + ',"=0")'
        RecommendationAnalysisWS.write(row, 2, numbCollectFormula)
        RecommendationAnalysisWS.write(row, 3, CompleteCollectFormula)
        RecommendationAnalysisWS.write(row, 4, GreatCollectFormula)

    #######################################################################
    
    #######################################################################
    workbook.close()


def WriteToGoogle(SpreadsheetLocation, folderID=None, Convert=None, Link=None):
    """
    Upload files to Google Drive. Requires 
    """
    
    client_json = '../scripts/client_secrets.json'

    GoogleAuth.DEFAULT_SETTINGS['client_config_file'] = (client_json)

    gauth = GoogleAuth()
    # Try to load saved client credentials
    mycred_file = '../scripts/mycreds.txt'

    gauth.LoadCredentialsFile(mycred_file)
    # if not creds or creds.invalid:

    if gauth.credentials is None:
        # Authenticate if they're not there
        gauth.LocalWebserverAuth()
    elif gauth.access_token_expired:
        # Refresh them if expired
        gauth.Refresh()

    else:
        # Initialize the saved creds
        gauth.Authorize()
# Save the current credentials to a file
    gauth.SaveCredentialsFile(mycred_file)

    drive = GoogleDrive(gauth)

    SpreadsheetName = SpreadsheetLocation.rsplit('/', 1)[-1]
    SpreadsheetName = SpreadsheetName[:-5]
    if folderID is not None:
        test_file = drive.CreateFile({'title': SpreadsheetName,
            "parents": [{"kind": "drive#fileLink", "id": folderID}]})
    else:
        test_file = drive.CreateFile(
            {'title': SpreadsheetName, "parents": [{"kind": "drive#fileLink"}]})

    test_file.SetContentFile(SpreadsheetLocation)
    if Convert is None:
        test_file.Upload({'convert': False})
    else:
        test_file.Upload({'convert': True})

    # Insert the permission.
    permission = test_file.InsertPermission(
        {'type': 'anyone', 'value': 'anyone', 'role': 'reader'})

    hyperlink = (test_file['alternateLink'])  # Display the sharable link.

    if Link is True:
        return hyperlink

    else:
        ReportURLstring = '<a href="' + str(hyperlink) + '">' + SpreadsheetName + '</a>'
        display(HTML(ReportURLstring))  # Display the sharable link.


def crop(image_path, coords, saved_location):
    """
    @param image_path: The path to the image to edit
    @param coords: A tuple of x/y coordinates (x1, y1, x2, y2)
    @param saved_location: Path to save the cropped image
    """
    image_obj = Image.open(image_path)
    cropped_image = image_obj.crop(coords)
    cropped_image.save(saved_location)


def Site_ttConceptAnalysis(Site, RecDict, LevelOrder, ConceptOrder, ElementOrder, YearsInvestigated, folderID):
    recMD = ['RecConcept',
             'RecLevel',
             'RecElement']
     # use a sites recommendation elements occurrence table, and add some columns for metadata about the recommendation
    recOccurDF = pd.read_csv(os.path.join("..","data","LTER",Site.upper()+"_RecommendationOccurrence.csv"))
    recOccurDF.insert(0, "RecElement", 0, allow_duplicates=False)
    recOccurDF.insert(0, "RecLevel", 0, allow_duplicates=False)    
    recOccurDF.insert(0, "RecConcept", 0, allow_duplicates=False)
    ''' 
    use the RecDict to look at the XPath column and for each key that matches part of any cell,
    write the value into the same row in the recOccurDF
    '''
    recOccurDF['RecElement'] = recOccurDF['XPath'].apply(lambda x: [value for key, value in RecDict.items() if key in x][0] )
    # create a list to order the columns with
    columnOrder = list(recOccurDF)
    # don't need xpaths any more
    columnOrder.remove('XPath')

    ''' 
    create a pivot table that leverages the dataframe recOccurDF's column for recommendation elements 
    and assigns the highest percentage any of the xpaths of the child elements to that row for a particular year
    '''
    radarElements = pd.pivot_table(recOccurDF, index='RecElement', columns=None, aggfunc='max').reindex(ElementOrder).reset_index()
    radarElements = radarElements[columnOrder]
   
    # fill in the metadata about concepts and recommendation levels
    radarElements['RecConcept'] = pd.Series(ConceptOrder)
    radarElements['RecLevel'] = pd.Series(LevelOrder)
    radarElements = radarElements.fillna(value=0.0)
    lineConcepts = radarElements
     # remove the site name from the column
    radarElements = radarElements.rename(columns={col: col.split('__')[-1] for col in radarElements.columns})
    # create recommendation concept csv
    #lineConcepts = lineConcepts.drop(['RecElement','RecLevel'], axis=1)
    lineConcepts.loc[-1] = lineConcepts.iloc[1:,:].mean(axis=0, numeric_only=True)
    lineConcepts.index = lineConcepts.index + 1  # shifting index
    lineConcepts.fillna('Average Completeness', inplace=True)
    lineConcepts = lineConcepts.sort_index()
    lineConcepts.to_csv(os.path.join('..','data','LTER',Site.upper()+'_RecommendationComplete.csv'), index=False)
    
    # remove the site name from the column
    lineConcepts = lineConcepts.rename(columns={col: col.split('__')[-1] for col in lineConcepts.columns})
    lineConcepts.to_csv(os.path.join('..','data','LTER',Site.upper()+'_RecommendationCompleteness.csv'), index=False)

    # create new version of concept occurrence table
    radarList = list(radarElements)
    difference = list(set(YearsInvestigated) - set(radarList[3:]))
    for year in difference:
        radarElements.insert(0, year, 0, allow_duplicates=False) 
    RecOccurDFcols = recMD + YearsInvestigated
    radarElements = radarElements[RecOccurDFcols]
    '''
    Take the occurrence of the conceptual LTER elements from each site's pivot table and plot each years output
    on a radar chart of 0 to 1 with each RecElement as an axis and the occurrence of records the percentage of color along that axis.
    '''
    # create a structure to add data to.
    data = []
    fig = tools.make_subplots(rows=14, cols=1, print_grid=False)
    count = 0
    # add the data from each year to a subplot.
    for year in YearsInvestigated:
        count = count + 1
        data.append(go.Scatterpolar(
            name = year,  
            r = radarElements[year].tolist()[1:],
            theta = radarElements['RecElement'].tolist()[1:],
            fill = 'toself',
            subplot = 'polar'+year
        ))

        
    fig.add_traces(data)
    
    layout = {
    'polar2005': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2006': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2007': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2008': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2009': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2010': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2011': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2012': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2013': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2014': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2015': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2016': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2017': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2018': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'showlegend': False,
    "height": 1200, "width": 16800, "autosize": False, "title": Site.upper() + ' LTER Recommendation Completeness 2005-2018'    
}
    
# create a description of the placement of each subplot    
    layout2 = {
    'polar2005': dict(
      domain = dict(
        x = [0, 1],
        y = [.96, 1]
      ),        
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2006': dict(
      domain = dict(
        x = [0, 1],
        y = [0.89, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2007': dict(
      domain = dict(
        x = [0, 1],
        y = [0.818, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2008': dict(
      domain = dict(
        x = [0, 1],
        y = [0.746, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2009': dict(
      domain = dict(
        x = [0, 1],
        y = [0.675, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2010': dict(
      domain = dict(
        x = [0, 1],
        y = [0.603, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2011': dict(
      domain = dict(
        x = [0, 1],
        y = [0.531, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2012': dict(
      domain = dict(
        x = [0, 1],
        y = [0.460, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2013': dict(
      domain = dict(
        x = [0, 1],
        y = [0.388, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2014': dict(
      domain = dict(
        x = [0, 1],
        y = [0.317, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2015': dict(
      domain = dict(
        x = [0, 1],
        y = [0.245, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2016': dict(
      domain = dict(
        x = [0, 1],
        y = [0.174, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2017': dict(
      domain = dict(
        x = [0, 1],
        y = [0.103, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2018': dict(
      domain = dict(
        x = [0, 1],
        y = [0.029, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'showlegend': False,
    "height": 32700, "width": 1200, "autosize": False    
}
    fig2 = {'data':data,'layout':layout2}

    
    pio.write_image(fig2, os.path.join('..','data','LTER',Site.upper()+'_bigPict_.png'))

    fig = {'data':data,'layout':layout}
    
    pio.write_image(fig, os.path.join('..','data','LTER',Site.upper()+'_.png'))

    crop(os.path.join('..','data','LTER',Site.upper()+'_bigPict_.png'), (0, 0, 1200, 16600), os.path.join('..','data','LTER',Site.upper()+'_bigPicture_.png'))
    os.remove(os.path.join('..','data','LTER',Site.upper()+'_bigPict_.png'))


def Organization_ttConceptAnalysis(
        Sites, RecDict, LevelOrder, ConceptOrder, ElementOrder,
        YearsInvestigated, folderID):
    RecCompleteSites = [
    os.path.join("../data/LTER", name) for name in os.listdir("../data/LTER") if name.endswith('_RecommendationComplete.csv')]
    recMD = ['RecConcept',
             'RecLevel',
             'RecElement']
    RecCompleteSites.sort()
    RecCompleteSitesNoAND = RecCompleteSites[1:]
    ANDcomplete = pd.read_csv(RecCompleteSites[0])
    for csv in RecCompleteSitesNoAND:
        dftocombine = pd.read_csv(csv)
        dftocombine = dftocombine.iloc[:,3:]
        ANDcomplete = pd.concat([ANDcomplete, dftocombine], axis=1)
    ANDcomplete = ANDcomplete.iloc[1:,1:]
    ANDcomplete.index = ANDcomplete.index - 1 
    radarElements = ANDcomplete  

    # fill in the metadata about concepts and recommendation levels

    radarElements['RecLevel'] = pd.Series(LevelOrder)
    radarElements['RecElement'] = pd.Series(ElementOrder)
    radarElements['RecConcept'] = pd.Series(ConceptOrder)
    radarElements = radarElements.fillna(value=0.0)

    radarList = list(radarElements)[2:]

    SitesInvestigated = list(itertools.product([Site.upper() for Site in Sites], YearsInvestigated))

    new_list = []
    for SiteYear in SitesInvestigated:
        new_list.append('__'.join(SiteYear))

    SitesInvestigated = new_list

    difference = list(set(SitesInvestigated) - set(radarList))


    for collection in difference:
        radarElements.insert(0, collection, 0, allow_duplicates=False) 
    RecOccurDFcols = recMD + SitesInvestigated
    # reorder the Dataframe
    radarElements = radarElements[RecOccurDFcols]
    #radarElements = radarElements.iloc[:, 25:]
    radarElementsAllYears = radarElements
    # drop any rows that contain all zeros to create an accurate overall raw data for measuring the conceptual completeness of all LTER
    radarElements = radarElements.loc[:, (radarElements != 0).any(axis=0)]

    # multiply each occurrence by the number of records to get the number of records that have a concept

    multiplierdf = radarElements.iloc[:1,3:]
    dftomultiply = radarElements.iloc[1:,3:]
    radarElementsLTER = dftomultiply.mul(multiplierdf.iloc[0], axis=1)
    radarElementsLTER

    radarElementsLTER.loc[0] = multiplierdf.iloc[0]  # adding a row

    radarElementsLTER = radarElementsLTER.sort_index()  # sorting by index    

    radarElementsLTER = radarElements.iloc[:,:3].join(radarElementsLTER)
    radarElementsLTER

    LTERdf = radarElementsLTER.iloc[1:,:3]
    recordTotals = ['Number of Records','Number of Records','Number of Records']
    for year in YearsInvestigated:
        filterdf = radarElementsLTER.filter(regex=year)
        LTERdf[year] = filterdf.iloc[1:,:].sum(axis=1)
        filterdf = filterdf[0:1]
        recordTotals.append(int(filterdf.sum(axis=1)))
        LTERdf[year]
    LTERdf.loc[0] = recordTotals  # adding a row

    LTERdf = LTERdf.sort_index()  # sorting by index  
    # divide by Number of Records to get accurate occurrence for each years contributions

    dividerdf = LTERdf.iloc[:1,3:]
    dftodivide = LTERdf.iloc[1:,3:]
    radaroccurElementsLTER = dftodivide.div(dividerdf.iloc[0], axis=1)
    radaroccurElementsLTER

    radaroccurElementsLTER.loc[0] = dividerdf.iloc[0]  # adding a row
    radaroccurElementsLTER
    radaroccurElementsLTER = radaroccurElementsLTER.sort_index()  # sorting by index    
    radaroccurElementsLTER
    LTERttDF = LTERdf.iloc[:,:3].join(radaroccurElementsLTER)
    LTERttDF.loc[-1] = LTERttDF.iloc[1:,:].mean(axis=0, numeric_only=True)
    LTERttDF.index = LTERttDF.index + 1  # shifting index
    LTERttDF.fillna('Average Completeness', inplace=True)
    LTERttDF = LTERttDF.sort_index()
    LTERttDF.to_csv(os.path.join('..','data','LTER','LTER_RecommendationCompleteness.csv'), index=False)
    '''
    Take the occurrence of the conceptual LTER elements from each site's pivot table and plot each years output
    on a radar chart of 0 to 1 with each RecElement as an axis and the occurrence of records the percentage of color along that axis.
    '''
    # create a structure to add data to.

    data = []
    fig = tools.make_subplots(rows=14, cols=1, print_grid=False)
    count = 0
    # add the data from each year to a subplot.
    for year in YearsInvestigated:
        count = count + 1
        data.append(go.Scatterpolar(
            name = year,  
            r = LTERttDF[year].tolist()[2:],
            theta = LTERttDF['RecElement'].tolist()[2:],
            fill = 'toself',
            subplot = 'polar'+year
        ))

        
    fig.add_traces(data)
    
    layout = {
    'polar2005': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2006': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2007': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2008': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2009': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2010': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2011': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2012': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2013': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2014': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2015': dict(

      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2016': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2017': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2018': dict(

      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'showlegend': False,
    "height": 1200, "width": 16800, "autosize": False, "title": 'All Sites LTER Recommendation Completeness 2005-2018'    
}
    
# create a description of the placement of each subplot    
    layout2 = {
    'polar2005': dict(
      domain = dict(
        x = [0, 1],
        y = [.96, 1]
      ),        
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2006': dict(
      domain = dict(
        x = [0, 1],
        y = [0.89, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2007': dict(
      domain = dict(
        x = [0, 1],
        y = [0.818, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2008': dict(
      domain = dict(
        x = [0, 1],
        y = [0.746, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2009': dict(
      domain = dict(
        x = [0, 1],
        y = [0.675, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2010': dict(
      domain = dict(
        x = [0, 1],
        y = [0.603, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2011': dict(
      domain = dict(
        x = [0, 1],
        y = [0.531, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2012': dict(
      domain = dict(
        x = [0, 1],
        y = [0.460, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2013': dict(
      domain = dict(
        x = [0, 1],
        y = [0.388, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2014': dict(
      domain = dict(
        x = [0, 1],
        y = [0.317, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2015': dict(
      domain = dict(
        x = [0, 1],
        y = [0.245, 1]
      ),
      
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2016': dict(
      domain = dict(
        x = [0, 1],
        y = [0.174, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2017': dict(
      domain = dict(
        x = [0, 1],
        y = [0.103, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'polar2018': dict(
      domain = dict(
        x = [0, 1],
        y = [0.029, 1]
      ),
      radialaxis = dict(
        angle = 0
      ),
      angularaxis = dict(
        direction = "clockwise",
        period = 6
      )
    ),
    'showlegend': False,
    "height": 32700, "width": 1200, "autosize": False    
}
    fig2 = {'data':data,'layout':layout2}

    
    pio.write_image(fig2, os.path.join('..','data','LTER','LTER_bigPict_.png'))

    fig = {'data':data,'layout':layout}
    
    pio.write_image(fig, os.path.join('..','data','LTER','LTER_.png'))

    crop(os.path.join('..','data','LTER','LTER_bigPict_.png'), (0, 0, 1200, 16600), os.path.join('..','data','LTER','LTER_bigPicture_.png'))
    os.remove(os.path.join('..','data','LTER','LTER_bigPict_.png'))

    # delete helper files
    for file in RecCompleteSites:
        os.remove(file)    